/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/dataseriesmodel', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/model/attributedmodel',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/tokenarray'
], function (Utils, AttributedModel, SheetUtils, TokenArray) {

    'use strict';

    var SOURCE_LINK_TYPES = ['names', 'title', 'values', 'bubbles'];

    // class DataSeriesModel ==================================================

    /**
     * A class that implements a data series model to be inserted into a chart
     * model.
     *
     * Series can have formulas 'Title', 'Names', 'Values' and 'Bubbles' on
     * 'change:attributes' it refreshes all formulas to TokenArrays
     *
     * Additionally it can have 'fill' and 'line' for styling
     *
     * @constructor
     *
     * @extends AttributedModel
     *
     * @param {SheetModel} sheetModel
     *  The sheet model that contains the chart object with this data series.
     *
     * @param {Object} [initAttributes]
     *  Initial formatting attribute set for this data series model.
     */
    function DataSeriesModel(sheetModel, initAttributes) {

        var // self reference
            self = this,

            // the document model
            docModel = sheetModel.getDocModel(),

            // token arrays for dynamic source ranges, mapped by source link type
            tokenArrays = {};

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, docModel, initAttributes, { families: 'series fill line' });

        // private methods ----------------------------------------------------

        /**
         * Updates the token arrays, after the source link attributes of this
         * data series have been changed.
         */
        function changeAttributesHandler() {

            var // the merged series attributes
                allAttributes = self.getMergedAttributes() || {},
                attributes = allAttributes.series || {};

            _(SOURCE_LINK_TYPES).each(function (type) {

                var sourceLink = attributes[type];
                var tokenArray = tokenArrays[type];

                // create, update, or delete token array according to type of source data
                if (_.isString(sourceLink)) {
                    // string: parse link formula
                    if (!tokenArray) {
                        tokenArray = new TokenArray(sheetModel);
                        tokenArrays[type] = tokenArray;
                        tokenArray.on('triggered', function () {
                            self.setAttributes({ series: Utils.makeSimpleObject(type, tokenArray.getFormula('op')) });
                        });
                    }
                    tokenArray.invokeSilently('parseFormula', 'op', sourceLink);
                } else if (tokenArray) {
                    // else: constant source data, delete token array
                    tokenArray.destroy();
                    delete tokenArrays[type];
                }
            });
        }

        // public methods -----------------------------------------------------

        this.iterateTokenArrays = function (callback, context) {
            _.each(tokenArrays, function (tokenArray, name) {
                if (tokenArray) {
                    callback.call(context, tokenArray, name);
                }
            });
        };

        this.transformRanges = function (sheet, interval, insert, columns) {

            var changed = false;

            this.iterateTokenArrays(function (tokenArray) {
                if (tokenArray.transformRanges(sheet, interval, insert, columns)) {
                    changed = true;
                }
            });

            return changed;
        };

        this.getRange = function (name) {
            return tokenArrays[name].resolveRangeList({ resolveNames: true })[0];
        };

        /**
         * Returns whether any of the source links contained in this data
         * series overlaps with any of the passed cell ranges.
         *
         * @param {Range3DArray|Range3D} ranges
         *  The addresses of the cell ranges, or a single cell range address,
         *  to be checked. The cell range addresses MUST be instances of the
         *  class Range3D with sheet indexes.
         *
         * @returns {Boolean}
         *  Whether any of the passed ranges overlaps with the source links of
         *  this data series.
         */
        this.rangesOverlap = function (ranges) {
            return _.some(tokenArrays, function (tokenArray) {
                return tokenArray.resolveRangeList({ resolveNames: true }).overlaps(ranges);
            });
        };

        // initialization -----------------------------------------------------

        this.on('change:attributes', changeAttributesHandler);
        changeAttributesHandler();

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.invoke(tokenArrays, 'destroy');
            self = docModel = sheetModel = tokenArrays = null;
        });

    } // class DataSeriesModel

    // exports ================================================================

    // derive this class from class AttributedModel
    return AttributedModel.extend({ constructor: DataSeriesModel });

});
