/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/condformatcollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/editframework/model/attributedmodel',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/tokenarray'
], function (Utils, ModelObject, AttributedModel, SheetUtils, TokenArray) {

    'use strict';

    var // convenience shortcuts
        Address = SheetUtils.Address;

    // class RuleModel ========================================================

    /**
     * Stores settings for a single formatting rule of conditionally formatted
     * cell ranges.
     *
     * @constructor
     *
     * @extends AttributedModel
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this instance.
     *
     * @param {Object} initAttributes
     *  The initial rule and formatting attributes.
     */
    var RuleModel = AttributedModel.extend({ constructor: function (sheetModel, initAttributes) {

        var // self reference
            self = this,

            // the spreadsheet document model
            docModel = sheetModel.getDocModel(),

            // token array representing the first condition
            tokenArray1 = new TokenArray(sheetModel),

            // token array representing the second condition
            tokenArray2 = new TokenArray(sheetModel);

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, docModel, initAttributes, { styleFamily: 'cell', families: 'rule character' });

        // private methods ----------------------------------------------------

        /**
         * Updates internal settings after the rule attributes of this rule
         * model have been changed.
         */
        function changeAttributesHandler(event, newAttributes, oldAttributes) {

            // extract the rule attributes from the passed attribute sets
            newAttributes = (newAttributes || self.getMergedAttributes()).rule;
            oldAttributes = oldAttributes ? oldAttributes.rule : {};

            // parse a changed formula expression in attribute 'value1'
            if (newAttributes.value1 !== oldAttributes.value1) {
                tokenArray1.invokeSilently('parseFormula', 'op', _.isString(newAttributes.value1) ? newAttributes.value1 : '');
            }

            // parse a changed formula expression in attribute 'value2'
            if (newAttributes.value2 !== oldAttributes.value2) {
                tokenArray2.invokeSilently('parseFormula', 'op', _.isString(newAttributes.value2) ? newAttributes.value2 : '');
            }
        }

        // protected methods --------------------------------------------------

        /**
         * Creates and returns a cloned instance of this rule model for the
         * specified sheet.
         *
         * @internal
         *  Used by the class CondFormatCollection during clone construction.
         *  DO NOT CALL from external code!
         *
         * @param {SheetModel} targetModel
         *  The model instance of the new cloned sheet that will own the clone
         *  returned by this method.
         *
         * @returns {RuleModel}
         *  A clone of this rule model, initialized for ownership by the passed
         *  sheet model.
         */
        this.clone = function (targetModel) {
            // construct a new model instance and relocate the sheet references
            var ruleModel = new RuleModel(targetModel, this.getExplicitAttributes(true));
            ruleModel.relocateSheet(sheetModel.getIndex(), targetModel.getIndex());
            return ruleModel;
        };

        // public methods -----------------------------------------------------

        /**
         * Recalculates the formula expressions stored in the rule attributes
         * 'value1' and 'value2', after columns or rows have been inserted into
         * or deleted from any sheet in the document.
         */
        this.transformValues = function (sheet, interval, insert, columns) {
            tokenArray1.transformRanges(sheet, interval, insert, columns);
            tokenArray2.transformRanges(sheet, interval, insert, columns);
            return true;
        };

        /**
         * Changes all formula expressions stored in the rule attributes
         * 'value1' and 'value2' with sheet references containing the specified
         * old sheet index to the new sheet index.
         *
         * @returns {RuleModel}
         *  A reference to this instance.
         */
        this.relocateSheet = function (oldSheet, newSheet) {
            tokenArray1.relocateSheet(oldSheet, newSheet);
            tokenArray2.relocateSheet(oldSheet, newSheet);
            return this;
        };

        // initialization -----------------------------------------------------

        // parse the source formulas after changing value attributes
        this.on('change:attributes', changeAttributesHandler);
        changeAttributesHandler();

        // update rule attributes after one of the token array has changed (but prevent updating the token array again)
        tokenArray1.on('triggered', function () {
            self.off('change:attributes', changeAttributesHandler);
            self.setAttributes({ rule: { value1: tokenArray1.getFormula('op') } });
            self.on('change:attributes', changeAttributesHandler);
        });
        tokenArray2.on('triggered', function () {
            self.off('change:attributes', changeAttributesHandler);
            self.setAttributes({ rule: { value2: tokenArray2.getFormula('op') } });
            self.on('change:attributes', changeAttributesHandler);
        });

        // destroy all class members
        this.registerDestructor(function () {
            tokenArray1.destroy();
            tokenArray2.destroy();
            self = docModel = sheetModel = tokenArray1 = tokenArray2 = null;
        });

    }}); // class RuleModel

    // class CondFormatModel ==================================================

    /**
     * Stores multiple conditional formatting rules for the same target cell
     * ranges in a specific sheet.
     *
     * Triggers the following events:
     * - 'insert:rule'
     *      After a new rule has been inserted into this formatting model.
     *      Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number} index
     *          The array index of the new formatting rule.
     * - 'delete:rule'
     *      After an existing rule has been deleted from this formatting model.
     *      Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number} index
     *          The old array index of the deleted formatting rule.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this instance.
     *
     * @param {RangeArray} targetRanges
     *  The addresses of all cell ranges that will be covered by the rules of
     *  this formatting model.
     */
    var CondFormatModel = ModelObject.extend({ constructor: function (sheetModel, targetRanges) {

        var // the spreadsheet document model
            docModel = sheetModel.getDocModel(),

            // all formatting rules (array of RuleModel instances)
            ruleModels = [];

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // private methods ----------------------------------------------------

        /**
         * Returns the rule model addressed by the property 'ruleIndex' in the
         * passed document operation.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing a document operation for formatting rules,
         *  expected to contain an integer property 'ruleIndex'.
         *
         * @returns {Object}
         *  A descriptor for the the rule model and its index, in the following
         *  properties:
         *  - {Number} index
         *      The array index of the rule model (the value of the 'ruleIndex'
         *      property in the passed operation).
         *  - {RuleModel} model
         *      The rule model referenced by the document operation.
         *
         * @throws {OperationError}
         *  If the operation does not contain a valid 'ruleIndex' property, or
         *  if the rule model with such an index does not exist.
         */
        function getModelData(context) {

            // get the index from the operation
            var index = context.getInt('ruleIndex');

            // check existence of the rule model
            var ruleModel = ruleModels[index];
            context.ensure(ruleModel, 'missing formatting rule');

            return { index: index, model: ruleModel };
        }

        // protected methods --------------------------------------------------

        /**
         * Creates and returns a cloned instance of this formatting model for
         * the specified sheet.
         *
         * @internal
         *  Used by the class CondFormatCollection during clone construction.
         *  DO NOT CALL from external code!
         *
         * @param {SheetModel} targetModel
         *  The model instance of the new cloned sheet that will own the clone
         *  returned by this method.
         *
         * @returns {CondFormatModel}
         *  A clone of this formatting model, initialized for ownership by the
         *  passed sheet model.
         */
        this.clone = function (targetModel) {
            // construct a new model instance and relocate the sheet references
            var formatModel = new CondFormatModel(targetModel, targetRanges);
            formatModel.relocateSheet(sheetModel.getIndex(), targetModel.getIndex());
            return formatModel;
        };

        /**
         * Callback handler for the document operation 'insertCondFormatRule'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'insertCondFormatRule' operation.
         */
        this.applyInsertOperation = function (context) {

            // get and validate the insertion index
            var index = context.getInt('ruleIndex');
            context.ensure((index >= 0) && (index <= ruleModels.length), 'invalid index');

            // the formatting attributes of the rule
            var attributes = context.getOptObj('attrs', {});

            // add type and other settings of the new rule
            attributes.rule = {
                type: context.getOptStr('type', 'formula'),
                value1: context.getOpt('value1', ''),
                value2: context.getOpt('value2', ''),
                priority: context.getOptInt('priority'),
                stop: context.getOptBool('stop')
            };

            // create and insert the new rule model, notify listeners
            var ruleModel = new RuleModel(sheetModel, attributes);
            ruleModels.splice(index, 0, ruleModel);
            this.trigger('insert:rule', index);
        };

        /**
         * Callback handler for the document operation 'deleteCondFormatRule'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'deleteCondFormatRule' operation.
         */
        this.applyDeleteOperation = function (context) {

            // resolve the rule model addressed by the operation (throws on error)
            var modelData = getModelData(context);

            // destroy and remove the formatting model, notify listeners
            modelData.model.destroy();
            ruleModels.splice(modelData.index, 1);
            this.trigger('delete:rule', modelData.index);
        };

        /**
         * Callback handler for the document operation 'changeCondFormatRule'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'changeCondFormatRule' operation.
         */
        this.applyChangeOperation = function (context) {

            // resolve the formatting model addressed by the operation (throws on error)
            /*var modelData = */getModelData(context);

            // TODO

            // TODO: notify listeners
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the addresses of all cell ranges covered by the formatting
         * rules of this model.
         *
         * @returns {RangeArray}
         *  The addresses of all cell ranges covered by the formatting rules.
         */
        this.getRanges = function () {
            return targetRanges.clone(true);
        };

        /**
         * Returns whether the passed cell address is contained by the target
         * cell ranges of this formatting model.
         *
         * @param {Address} address
         *  The address of the cell to be checked.
         *
         * @returns {Boolean}
         *  Whether the specified cell is contained in this formatting model.
         */
        this.containsCell = function (address) {
            return targetRanges.containsAddress(address);
        };

        /**
         * Returns the reference addresses of this formatting model (the
         * top-left cell of the bounding range of all target cell ranges of
         * this formatting model).
         *
         * @returns {Address}
         *  The address of the reference cell of this formatting model.
         */
        this.getRefAddress = function () {
            return targetRanges.empty() ? Address.A1.clone() : targetRanges.boundary().start;
        };

        /**
         * Changes the target cell ranges of this formatting model.
         *
         * @param {RangeArray} newRanges
         *  The addresses of the new target cell ranges. The old target ranges
         *  of this formatting model will be replaced completely.
         *
         * @returns {CondFormatModel}
         *  A reference to this instance.
         */
        this.setRanges = function (newRanges) {
            targetRanges = newRanges.merge();
            return this;
        };

        /**
         * Transforms the target cell ranges of this formatting model, after
         * columns or rows have been inserted into or removed from the sheet.
         *
         * @returns {Boolean}
         *  Whether the transformed target cell ranges are still valid (not all
         *  deleted). If false is returned, this formatting model must be
         *  deleted from the conditional formatting collection.
         */
        this.transformRanges = function (interval, insert, columns) {
            targetRanges = docModel.transformRanges(targetRanges, interval, insert, columns, 'expandEnd');
            return !targetRanges.empty();
        };

        /**
         * Recalculates the formula expressions of all rules in this formatting
         * model, after columns or rows have been inserted into or deleted from
         * any sheet in the document.
         *
         * @returns {CondFormatModel}
         *  A reference to this instance.
         */
        this.transformValues = function (sheet, interval, insert, columns) {
            _.invoke(ruleModels, 'transformValues', sheet, interval, insert, columns);
            return this;
        };

        /**
         * Changes the sheet references in all formula expressions of this
         * formatting model, after a sheet has been inserted, deleted, or
         * moved.
         *
         * @returns {CondFormatModel}
         *  A reference to this instance.
         */
        this.relocateSheet = function (oldSheet, newSheet) {
            _.invoke(ruleModels, 'relocateSheet', oldSheet, newSheet);
            return this;
        };

        // initialization -----------------------------------------------------

        // merge the target range addresses
        targetRanges = targetRanges.merge();

        // destroy all class members
        this.registerDestructor(function () {
            _.invoke(ruleModels, 'destroy');
            docModel = sheetModel = ruleModels = null;
        });

    }}); // class CondFormatModel

    // class CondFormatCollection =============================================

    /**
     * Stores conditional formatting settings in a specific sheet. Conditional
     * formatting can be used to modify specific formatting attributes of one
     * or more cell ranges in a sheet automatically.
     *
     * Triggers the following events:
     * - 'insert:condformat'
     *      After new conditional formatting ranges have been inserted into
     *      this collection. Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number} index
     *          The array index of the new formatting model.
     * - 'delete:condformat'
     *      After existing conditional formatting ranges have been deleted from
     *      this collection. Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number} index
     *          The old array index of the deleted formatting model.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */
    function CondFormatCollection(sheetModel) {

        var // the spreadsheet document model
            docModel = sheetModel.getDocModel(),

            // all formatting models (array of CondFormatModel instances)
            formatModels = [];

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // private methods ----------------------------------------------------

        /**
         * Returns the formatting model addressed by the property 'index' in
         * the passed document operation.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing a document operation for conditional
         *  formatting, expected to contain an integer property 'index'.
         *
         * @returns {Object}
         *  A descriptor for the the formatting model and its index, in the
         *  following properties:
         *  - {Number} index
         *      The array index of the formatting model (the value of the
         *      'index' property in the passed operation).
         *  - {FormatModel} model
         *      The formatting model referenced by the document operation.
         *
         * @throws {OperationError}
         *  If the operation does not contain a valid 'index' property, or if
         *  the formatting model with such an index does not exist.
         */
        function getModelData(context) {

            // get the index from the operation
            var index = context.getInt('index');

            // check existence of the formatting model
            var formatModel = formatModels[index];
            context.ensure(formatModel, 'missing conditional formatting');

            return { index: index, model: formatModel };
        }

        /**
         * Recalculates the position of the target cell ranges of all
         * formatting models, after columns or rows have been inserted into or
         * deleted from the sheet.
         */
        function transformRanges(interval, insert, columns) {
            Utils.iterateArray(formatModels, function (formatModel, index) {
                if (!formatModel.transformRanges(interval, insert, columns)) {
                    formatModel.destroy();
                    formatModels.splice(index, 1);
                }
            }, { reverse: true }); // reverse to be able to delete in-place
        }

        /**
         * Recalculates the formula expressions stored in the rules of all
         * formatting model, after columns or rows have been inserted into or
         * deleted from any sheet in the document.
         */
        function transformValues(sheet, interval, insert, columns) {
            _.invoke(formatModels, 'transformValues', sheet, interval, insert, columns);
        }

        // protected methods --------------------------------------------------

        /**
         * Returns the internal contents of this collection, needed for cloning
         * into another collection.
         *
         * @internal
         *  Used during clone construction. DO NOT CALL from external code!
         *
         * @returns {Object}
         *  The internal contents of this collection.
         */
        this.getCloneData = function () {
            return { formatModels: formatModels };
        };

        /**
         * Clones all contents from the passed collection into this collection.
         *
         * @internal
         *  Used by the class SheetModel during clone construction. DO NOT CALL
         *  from external code!
         *
         * @param {CondFormatCollection} collection
         *  The source collection whose contents will be cloned into this
         *  collection.
         *
         * @returns {CondFormatCollection}
         *  A reference to this instance.
         */
        this.cloneFrom = function (collection) {

            var // the internal contents of the source collection
                cloneData = collection.getCloneData();

            // clone the contents of the source collection
            formatModels = _.invoke(cloneData.formatModels, 'clone', sheetModel);

            return this;
        };

        /**
         * Callback handler for the document operation 'insertCondFormat'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'insertCondFormat' document operation.
         */
        this.applyInsertOperation = function (context) {

            // get and validate the insertion index
            var index = context.getInt('index');
            context.ensure((index >= 0) && (index <= formatModels.length), 'invalid index');

            // the new formatting model
            var formatModel = new CondFormatModel(sheetModel, context.getRangeArray());

            // insert the new formatting model, notify listeners
            formatModels.splice(index, 0, formatModel);
            this.trigger('insert:condformat', index);
        };

        /**
         * Callback handler for the operation 'deleteCondFormat'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'deleteCondFormat' document operation.
         */
        this.applyDeleteOperation = function (context) {

            // resolve the formatting model addressed by the operation (throws on error)
            var modelData = getModelData(context);

            // destroy and remove the formatting model, notify listeners
            modelData.model.destroy();
            formatModels.splice(modelData.index, 1);
            this.trigger('delete:condformat', modelData.index);
        };

        /**
         * Callback handler for the operation 'changeCondFormat'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'changeCondFormat' document operation.
         */
        this.applyChangeOperation = function (context) {

            // resolve the formatting model addressed by the operation (throws on error)
            var modelData = getModelData(context);

            // change target ranges (ranges of different formatting models may overlap)
            modelData.model.setRanges(context.getRangeArray());

            // TODO: notify listeners
        };

        /**
         * Callback handler for the document operation 'insertCondFormatRule'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'insertCondFormatRule' operation.
         */
        this.applyInsertRuleOperation = function (context) {
            getModelData(context).model.applyInsertOperation(context);
        };

        /**
         * Callback handler for the document operation 'deleteCondFormatRule'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'deleteCondFormatRule' operation.
         */
        this.applyDeleteRuleOperation = function (context) {
            getModelData(context).model.applyDeleteOperation(context);
        };

        /**
         * Callback handler for the document operation 'changeCondFormatRule'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'changeCondFormatRule' operation.
         */
        this.applyChangeRuleOperation = function (context) {
            getModelData(context).model.applyChangeOperation(context);
        };

        // initialization -----------------------------------------------------

        // update target ranges after inserting/deleting columns or rows in the own sheet
        sheetModel.registerTransformationHandler(transformRanges);

        // update source formulas after inserting/deleting columns or rows in any sheet
        docModel.registerTransformationHandler(transformValues);

        // destroy all class members
        this.registerDestructor(function () {
            docModel.unregisterTransformationHandler(transformValues);
            _.invoke(formatModels, 'destroy');
            docModel = sheetModel = formatModels = null;
        });

    } // class CondFormatCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: CondFormatCollection });

});
