/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/app/drawingcontroller', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/drawinglayer/utils/drawingutils',
    'io.ox/office/drawinglayer/view/drawinglabels',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/tokenarray',
    'io.ox/office/spreadsheet/view/chartcreator',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, Color, DrawingUtils, DrawingLabels, Operations, SheetUtils, TokenArray, ChartCreator, gt) {

    'use strict';

    // class DrawingControllerMixin ===========================================

    /**
     * Drawing controller (will) encapsulate(s) all drawing features of the controller
     * For now Chart are part of Drawings!
     *
     * @constructor
     */
    function DrawingControllerMixin(app, docModel, docView) {

        var self = this,

            sourceSelector = false,

            chartSourceToken = null,

            // the token arrays containing the highlighted ranges of the selected drawing object
            tokenArrays = [],

            // the unique identifier of the range highlighting mode
            highlightUid = null;

        // private methods ----------------------------------------------------

        /**
         * Shows a warning alert box for the 'exchange direction' function for
         * chart series.
         */
        function yellChartDirection(type) {

            switch (type) {
            case 'sheets':
                docView.yell({
                    type: 'warning',
                    message:
                        //#. Warning text: A chart object in a spreadsheet contains complex data source (cells from multiple sheets).
                        gt('Data references are located on different sheets.')
                });
                break;

            case 'directions':
                docView.yell({
                    type: 'warning',
                    message:
                        //#. Warning text: A chart object in a spreadsheet contains complex data source (not a simple cell range).
                        gt('Data references are too complex for this operation.')
                });
                break;

            default:
                Utils.warn('DrawingController.yellChartDirection(): unknown warning type: "' + type + '"');
            }

            return $.Deferred().reject();
        }

        /**
         * changeSelection
         * if only one drawing is selected and this is a chart, the function highlights its source cells via token array
         */
        function changeSelectionHandler(evt, selection) {

            if (highlightUid) {
                docView.endRangeHighlighting(highlightUid);
                highlightUid = null;
            }

            tokenArrays = [];

            if (selection.drawings.length === 1) {

                var chartModel = docView.getDrawingCollection().findModel(selection.drawings[0], { type: 'chart' });

                if (chartModel) {
                    chartModel.getTokenArrays(tokenArrays);
                }
            }

            // highlight the source ranges of the chart in the active sheet
            if (tokenArrays.length > 0) {
                highlightUid = docView.startRangeHighlighting(tokenArrays, { priority: true, resolveNames: true });
            }
        }

        /**
         * Returns the position of the selected drawing object, if exactly one
         * drawing object is currently selected.
         *
         * @returns {Array<Number>|Null}
         *  The position of the selected drawing object in the active sheet
         *  (without leading sheet index), if available; otherwise null.
         */
        function getDrawingPosition() {
            var drawingPositions = docView.getSelectedDrawings();
            return (drawingPositions.length === 1) ? drawingPositions[0] : null;
        }

        /**
         * @return the current selected Drawing
         */
        function getDrawingModel(type) {
            var drawingIndices = getDrawingPosition();
            if (drawingIndices) {
                return docView.getDrawingCollection().findModel(drawingIndices, { type: type });
            }
        }

        /**
         * Generates and applies a drawing operation.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the drawing operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        function createAndApplyDrawingOperation(opName, properties, callback) {
            return docView.getSheetModel().createAndApplyOperations(function (generator) {
                var position = getDrawingPosition();
                if (callback) { callback(generator, position); }
                generator.generateDrawingOperation(opName, position, properties);
            });
        }

        /**
         * Returns the title text from the passed attribute set.
         *
         * @param {Object} [attributes]
         *  The merged attribute set for a chart title. If omitted, returns an
         *  empty string.
         *
         * @returns {String}
         *  The title text from the passed attribute set.
         */
        function getChartTitle(attributes) {
            return _.isObject(attributes) ? (attributes.text.link[0] || '') : '';
        }

        /**
         * Changes the attributes of the specified title in the selected chart
         * object.
         *
         * @param {String} titleId
         *  The identifier of the title object. Must be one of 'main', 'x',
         *  'y', or 'z'.
         *
         * @param {Object} attrs
         *  The attribute set to be set for the title.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the operation has been
         *  applied successfully, or that will be rejected on any error.
         */
        function setChartTitleAttributes(titleId, attrs) {
            return createAndApplyDrawingOperation(Operations.SET_CHART_TITLE_ATTRIBUTES, { axis: titleId, attrs: attrs });
        }

        /**
         * Changes the text of the specified title in the selected chart
         * object.
         *
         * @param {String} titleId
         *  The identifier of the title object. Must be one of 'main', 'x',
         *  'y', or 'z'.
         *
         * @param {String} title
         *  The new text contents of the title. An empty string will remove the
         *  title from the chart.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the operation has been
         *  applied successfully, or that will be rejected on any error.
         */
        function setChartTitle(titleId, title) {
            return setChartTitleAttributes(titleId, { text: { link: [title] }, character: ChartCreator.getHeadChar() });
        }

        function selectNewChartSource() {

            var // enter custom selection mode
                promise = docView.enterCustomSelectionMode('chartsource', {
                    statusLabel: /*#. change source data for a chart object in a spreadsheet */ gt('Select source data')
                });

            // set selected range as source data
            promise.done(function (selection) {

                var range = selection.ranges[0];
                var sheet = docView.getActiveSheet();

                if (!chartSourceToken) {
                    chartSourceToken = new TokenArray(docView.getSheetModel());
                }

                chartSourceToken.clear().appendRange(range, { sheet: sheet, abs: true });

                var chart = getDrawingModel('chart');

                var possSources = chart.getDataSourceInfo();
                if (possSources.warn) {
                    return ChartCreator.updateSeries(app, range, sheet, chart);
                }

                return ChartCreator.updateSeries(app, range, sheet, chart, possSources.axis);
            });
        }

        /**
         * changeChartType maps the assigned id to chart-data
         * all ids are mapped in DrawingLabels.CHART_TYPE_STYLES
         *
         * There is a special behavior for bubble-chart, change to bubble or from bubble.
         * all series-data will be removed an initialized complete new by the ChartCreator
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the operation has been
         *  applied successfully, or that will be rejected on any error.
         */
        function changeChartType(id) {

            var chart = getDrawingModel('chart');
            var data = DrawingLabels.CHART_TYPE_STYLES[id];
            var oldData = chart.getMergedAttributes().chart;

            if (data.type !== oldData.type && (data.type.indexOf('bubble') === 0 || oldData.type.indexOf('bubble') === 0)) {
                //special behavior for bubble charts!!!
                var possSources = chart.getDataSourceInfo();
                if (possSources.warn) {
                    return yellChartDirection(possSources.warn);
                }
                return ChartCreator.updateSeries(app, possSources.range, possSources.sheet, chart, possSources.axis, data);
            }

            return createAndApplyDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, { attrs: { chart: data } }, function (generator, position) {
                if ((data.type === 'pie2d' || data.type === 'donut2d')) {
                    var noShape = { line: ChartCreator.getNoneShape(), label: false };
                    generator.generateDrawingOperation(Operations.SET_CHART_AXIS_ATTRIBUTES,     position, { attrs: noShape, axis: 'x' });
                    generator.generateDrawingOperation(Operations.SET_CHART_AXIS_ATTRIBUTES,     position, { attrs: noShape, axis: 'y' });
                    generator.generateDrawingOperation(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, position, { attrs: noShape, axis: 'x' });
                    generator.generateDrawingOperation(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, position, { attrs: noShape, axis: 'y' });
                    generator.generateDrawingOperation(Operations.SET_CHART_TITLE_ATTRIBUTES,    position, { attrs: { text: { link: [] } }, axis: 'x' });
                    generator.generateDrawingOperation(Operations.SET_CHART_TITLE_ATTRIBUTES,    position, { attrs: { text: { link: [] } }, axis: 'y' });
                }
            });
        }

        function addAxisDef(axisId, defs) {

            var // the base path of axis items
                keyPath = 'drawing/chart/axis/' + axisId + '/';

            function getCorrectAxisId(chartModel) {
                return chartModel ? chartModel.getAxisIdForDrawing(axisId) : axisId;
            }

            function getStandardLineAttributes(visible) {
                return { line: visible ? ChartCreator.getStandardShape() : ChartCreator.getNoneShape() };
            }

            function setAxisAttributes(chartModel, attrs) {
                return createAndApplyDrawingOperation(Operations.SET_CHART_AXIS_ATTRIBUTES, { axis: getCorrectAxisId(chartModel), attrs: attrs });
            }

            function setGridLineAttributes(chartModel, attrs) {
                return createAndApplyDrawingOperation(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, { axis: getCorrectAxisId(chartModel), attrs: attrs });
            }

            // *** axis items ***

            // parent item providing access to a chart axis model
            defs[keyPath + 'model'] = {
                parent: ['drawing/chart', 'drawing/chart/axes/enabled'],
                get: function (chartModel) { return chartModel ? chartModel.getAxisModel(getCorrectAxisId(chartModel)) : null; }
            };

            // parent item providing access to the attributes of a chart axis model
            defs[keyPath + 'attributes'] = {
                parent: [keyPath + 'model', 'drawing/chart'],
                get: function (axisModel) { return axisModel ? axisModel.getMergedAttributes() : null; },
                set: function (attributes) { return setAxisAttributes(this.getParentValue(1), attributes); }
            };

            // return or modify the visibility of the axis caption labels
            defs[keyPath + 'labels/visible'] = {
                parent: [keyPath + 'attributes', 'drawing/chart'],
                get: function (attributes) { return _.isObject(attributes) && (attributes.axis.label === true) && this.areParentsEnabled(); },
                set: function (visible) { return setAxisAttributes(this.getParentValue(1), { axis: { label: visible } }); }
            };

            // return or modify the visibility of the axis line
            defs[keyPath + 'line/visible'] = {
                parent: [keyPath + 'attributes', 'drawing/chart'],
                get: function (attributes) { return _.isObject(attributes) && (attributes.line.type !== 'none'); },
                set: function (visible) { return setAxisAttributes(this.getParentValue(1), getStandardLineAttributes(visible)); }
            };

            // *** grid line items ***

            // parent item providing access to a chart axis grid model
            defs[keyPath + 'grid/model'] = {
                parent: keyPath + 'model',
                get: function (axisModel) { return axisModel ? axisModel.getGrid() : null; }
            };

            // parent item providing access to the attributes of a chart axis grid model
            defs[keyPath + 'grid/attributes'] = {
                parent: [keyPath + 'grid/model', 'drawing/chart'],
                get: function (gridModel) { return gridModel ? gridModel.getMergedAttributes() : null; },
                set: function (attributes) { return setGridLineAttributes(this.getParentValue(1), attributes); }
            };

            // return or modify the visibility of the axis grid lines
            defs[keyPath + 'grid/visible'] = {
                parent: [keyPath + 'grid/attributes', 'drawing/chart'],
                get: function (attributes) { return _.isObject(attributes) && (attributes.line.type !== 'none'); },
                set: function (visible) { return setGridLineAttributes(this.getParentValue(1), getStandardLineAttributes(visible)); }
            };

            // *** title items ***

            // parent item providing access to a chart axis title model
            defs[keyPath + 'title/model'] = {
                parent: keyPath + 'model',
                get: function (axisModel) { return axisModel ? axisModel.getTitle() : null; }
            };

            // parent item providing access to the attributes of a chart axis title model
            defs[keyPath + 'title/attributes'] = {
                parent: [keyPath + 'title/model', 'drawing/chart'],
                get: function (titleModel) { return titleModel ? titleModel.getMergedAttributes() : null; },
                set: function (attributes) { return setChartTitleAttributes(getCorrectAxisId(this.getParentValue(1)), attributes); }
            };

            // return or modify the text contents of a chart axis title
            defs[keyPath + 'title/text'] = {
                parent: [keyPath + 'title/attributes', 'drawing/chart'],
                get: function (attributes) { return getChartTitle(attributes); },
                set: function (title) { return setChartTitle(getCorrectAxisId(this.getParentValue(1)), title); }
            };

            // remove the main title from a chart (disabled if title does not exist)
            defs[keyPath + 'title/delete'] = {
                parent: [keyPath + 'title/text', 'drawing/chart'],
                enable: function () { return this.getValue().length > 0; },
                set: function () { return setChartTitle(getCorrectAxisId(this.getParentValue(1)), ''); }
            };
        }

        /**
         * private registerDefinitions handles all controller definitions for Drawings and Charts.
         * Including the Insert-Drawing button which is physical part of the original Sidepane but is better placed here
         */
        function registerDefinitions() {
            var defs = {};

            defs['image/insert'] = {
                parent: 'sheet/operation/unlocked',
                set: function (imageDesc) { return docView.insertImages(imageDesc); }
            };

            defs['image/insert/dialog'] = {
                parent: 'sheet/operation/unlocked',
                set: function () { return docView.showInsertImageDialog(); }
            };

            defs['chart/insert'] = {
                parent: 'sheet/operation/unlocked',
                enable: function () {
                    var ranges = docView.getSelectedRanges();
                    return (ranges.length === 1) && !docView.getSheetModel().isSingleCellInRange(ranges[0]);
                },
                set: function (id) { return ChartCreator.createChart(app, DrawingLabels.CHART_TYPE_STYLES[id]); }
            };

            defs['drawing/operation'] = {
                parent: 'sheet/operation/unlocked',
                enable: function () { return docView.hasDrawingSelection(); }
            };

            defs['drawing/model'] = {
                parent: 'drawing/operation',
                get: getDrawingModel
            };

            defs['drawing/type/label'] = {
                parent: 'drawing/model',
                get: function (drawingModel) { return DrawingLabels.getDrawingTypeLabel(drawingModel ? drawingModel.getType() : ''); }
            };

            defs['drawing/delete'] = {
                parent: 'drawing/operation',
                set: function () { return docView.deleteDrawings(); }
            };

            defs['drawing/image'] = {
                parent: 'drawing/operation',
                enable: function () { return _.isObject(this.getValue()); },
                get: function () { return getDrawingModel('image'); }
            };

            defs['drawing/chart'] = {
                parent: 'drawing/operation',
                enable: function () { return _.isObject(this.getValue()); },
                get: function () { return getDrawingModel('chart'); }
            };

            defs['drawing/chartlabels'] = {
                parent: 'drawing/chart',
                get: function () { return docView.getChartLabelsMenu().isVisible(); },
                set: function (state) { docView.getChartLabelsMenu().toggle(state); }
            };

            defs['drawing/chartexchange'] = {
                parent: 'drawing/chart',
                enable: function (chartModel) { return chartModel.getSeriesCount() > 0; },
                set: function () {
                    var chart = getDrawingModel('chart');
                    var possSources = chart.getDataSourceInfo();
                    if (possSources.warn) {
                        return yellChartDirection(possSources.warn);
                    }
                    return ChartCreator.updateSeries(app, possSources.range, possSources.sheet, chart, 1 - possSources.axis);
                }
            };

            defs['drawing/chartfirstcol'] = {
                parent: 'drawing/chart',
                enable: function (chartModel) { return chartModel.getSeriesCount() > 0; },
                get: function (chartModel) { return chartModel ? chartModel.isFirstColLabel() : null; },
                set: function (state) {
                    var chart = getDrawingModel('chart');
                    var possSources = chart.getDataSourceInfo();
                    if (possSources.warn) {
                        return yellChartDirection(possSources.warn);
                    }
                    return ChartCreator.updateSeries(app, possSources.range, possSources.sheet, chart, possSources.axis, undefined, state, chart.isFirstRowLabel());
                }
            };

            defs['drawing/chartfirstrow'] = {
                parent: 'drawing/chart',
                enable: function (chartModel) { return chartModel.getSeriesCount() > 0; },
                get: function (chartModel) { return chartModel ? chartModel.isFirstRowLabel() : null; },
                set: function (state) {
                    var chart = getDrawingModel('chart');
                    var possSources = chart.getDataSourceInfo();
                    if (possSources.warn) {
                        return yellChartDirection(possSources.warn);
                    }
                    return ChartCreator.updateSeries(app, possSources.range, possSources.sheet, chart, possSources.axis, undefined, chart.isFirstColLabel(), state);
                }
            };

            defs['drawing/charttype'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getChartTypeForGui() : null; },
                set: changeChartType
            };

            // parent item providing access to the attributes of a chart model
            defs['drawing/chart/attributes'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getMergedAttributes() : null; },
                set: function (attributes) { this.getParentValue().setAttributes(attributes); }
            };

            defs['drawing/chartvarycolor'] = {
                parent: ['drawing/chart', 'drawing/chart/attributes'],
                enable: function (chartModel) { return chartModel.isVaryColorEnabled(); },
                get: function (chartModel, attributes) { return attributes ? attributes.chart.varyColors : null; },
                set: function (state) { this.getParentValue().changeVaryColors(state); }
            };

            defs['drawing/chartdatalabel'] = {
                parent: ['drawing/chart', 'drawing/chart/attributes'],
                get: function (chartModel, attributes) { return attributes ? attributes.chart.dataLabel : false; },
                set: function (state) { return createAndApplyDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, { attrs: { chart: { dataLabel: state } } }); }
            };

            defs['drawing/chartmarkeronly'] = {
                parent: ['drawing/chart'],
                enable: function (chartModel) { var data = chartModel ? chartModel.getMergedAttributes().chart : {}; return data.type === 'line2d' || data.type === 'scatter2d'; },
                get: function (chartModel) { return chartModel ? chartModel.isMarkerOnly() : false; },
                set: function (state) { this.getParentValue().setMarkerOnly(state); }
            };

            defs['drawing/chartcolorset'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getColorSet() : null; },
                set: function (colorset) { this.getParentValue().changeColorSet(colorset); }
            };

            defs['drawing/chartstyleset'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getStyleSet() : null; },
                set: function (colorset) { this.getParentValue().changeStyleSet(colorset); }
            };

            defs['drawing/chartsource'] = {
                parent: 'drawing/chart',
                get: function () { return sourceSelector; },
                set: selectNewChartSource
            };

            defs['drawing/chartlegend/pos'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getLegendModel().getMergedAttributes().legend.pos : null; },
                set: function (pos) { return createAndApplyDrawingOperation(Operations.SET_CHART_LEGEND_ATTRIBUTES, { attrs: { legend: { pos: pos } } }); }
            };

            defs['drawing/chart/axes/enabled'] = {
                parent: 'drawing/chart',
                enable: function (chartModel) { return chartModel.isAxesEnabled(); }
            };

            addAxisDef('x', defs);
            addAxisDef('y', defs);

            // parent item providing access to the main title model of a chart
            defs['drawing/chart/title/model'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getTitleModel('main') : null; }
            };

            // parent item providing access to the attributes of the main title model of a chart
            defs['drawing/chart/title/attributes'] = {
                parent: 'drawing/chart/title/model',
                get: function (titleModel) { return titleModel ? titleModel.getMergedAttributes() : null; },
                set: function (attributes) { return setChartTitleAttributes('main', attributes); }
            };

            // return or modify the text contents of the main title of a chart
            defs['drawing/chart/title/text'] = {
                parent: 'drawing/chart/title/attributes',
                get: function (attributes) { return getChartTitle(attributes); },
                set: function (title) { return setChartTitle('main', title); }
            };

            // remove the main title from a chart (disabled if title does not exist)
            defs['drawing/chart/title/delete'] = {
                parent: 'drawing/chart/title/text',
                enable: function () { return this.getValue().length > 0; },
                set: function () { return setChartTitle('main', ''); }
            };

            // IMAGE

            defs['drawing/border/style'] = {
                parent: 'drawing/image',
                get: function (imageModel) { return DrawingUtils.getPresetBorder(imageModel ? imageModel.getMergedAttributes().line : null); },
                set: function (preset) {
                    var attributes = this.getParentValue().getMergedAttributes(),
                        options = { line: DrawingUtils.resolvePresetBorder(preset) };

                    if (attributes.line.color.type === 'auto') {
                        options.line.color = { type: 'rgb', value: '000000' };
                    }

                    return docView.setDrawingAttributes(options);
                }
            };

            defs['drawing/border/color'] = {
                parent: 'drawing/image',
                get: function (imageModel) { return imageModel ? imageModel.getMergedAttributes().line.color : Color.AUTO; },
                set: function (color) {
                    var attributes = this.getParentValue().getMergedAttributes(),
                        options = { line: { color: color }};

                    if (attributes.line.type !== 'solid') {
                        options.line.type = 'solid';
                    }

                    return docView.setDrawingAttributes(options);
                }
            };

            self.registerDefinitions(defs);
        }

        // initialization -----------------------------------------------------

        registerDefinitions();

        this.listenTo(docView, 'change:selection', changeSelectionHandler);

        this.registerDestructor(function () {
            self = app = docModel = docView = chartSourceToken = tokenArrays = null;
        });

    } // class DrawingControllerMixin

    // exports ================================================================

    return DrawingControllerMixin;

});
