/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/settings/templatefolders', [
    'io.ox/office/tk/dialogs',
    'io.ox/mail/detail/links',
    'io.ox/core/notifications',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/driveutils',
    'io.ox/office/settings/view/settingslabels'
], function (Dialogs, DeepLinks, Notifications, Utils, DriveUtils, Labels) {

    'use strict';

    // class TemplateFolders ==================================================

    /**
     * @param {TemplateModel} templateModel
     *  storage
     *
     * @param {Object} options
     *  @param {Boolean} [options.useMyFiles=false]
     *  @param {Boolean} [options.readOnly=false]
     *  @param {String} [options.title]
     *  @param {String} [options.folderPicker]
     *  @param {String} [options.newFolder]
     */
    function TemplateFolders(templateModel, options) {
        var readOnly = Utils.getBooleanOption(options, 'readOnly', false);
        var showReadAccess = Utils.getBooleanOption(options, 'showReadAccess', false);

        var myFiles = DriveUtils.getStandardTemplateFolderId();

        this.refresh = function (settings) {
            settings.appendTitle(Utils.getStringOption(options, 'title', ''));

            var folders = $('<ul class="list-unstyled publications list-group widget-list"></ul>');

            settings.append(folders);

            /**
             * calls DriveUtils.getTrashState() and if not
             * it calls DriveUtils.getPath(id)
             * and ssignes this to callback
             */
            function path(id, callback) {
                DriveUtils.getTrashState(id).done(function (state) {
                    if (!state) {
                        DriveUtils.getPath(id).done(callback);
                    }
                });
            }

            /**
             * removes folder from the Model & from the Gui
             */
            function remove(path, li) {
                var id = _.last(path).id;
                if (!templateModel.remove(id)) {
                    if (ox.debug) { console.warn('folder not in list', id); }
                } else {
                    li.remove();
                }
            }

            function handlePermissions(/*btn, id*/) {

            }

            /**
             * add new li-element with:
             *
             * folder-title
             * folder-link (opens Drive)
             * delete button
             */
            function add(path, noRemove, oldLi, click) {
                var folder = _.last(path);
                var id = folder.id;

                var li = $('<li class="widget-settings-view">');

                var ctrl = $('<div class="widget-controls">');
                li.append(ctrl);

                if (!noRemove && !readOnly) {

                    var rem = $('<a href="#" tabindex="1" role="button" data-action="remove" style="right: 0;">');
                    var icn = $('<i class="fa fa-trash-o">');
                    rem.append(icn);
                    rem.attr('title', title);
                    rem.click(function (evt) {
                        evt.preventDefault();
                        remove(path, li);
                    });

                    ctrl.append(rem);
                }

                if (showReadAccess) {
                    var changeListener = function () {
                        DriveUtils.showPermissionsDialog(id).always(function () {
                            handlePermissions(perm, id);
                        });
                    };

                    var perm = $('<a href="#" tabindex="1" role="button" style="right: 45px; position: absolute; margin-top: -5px; padding-right: 10px;">');
                    perm.text(Labels.PERMISSIONS_LABEL);
                    ctrl.append(perm);
                    perm.click(function (evt) {
                        evt.preventDefault();
                        changeListener();
                    });
                    handlePermissions(perm, id);

                    if (click) {
                        Notifications.yell({ type: 'info', message: Labels.PERMISSIONS_MESSAGE_LABEL });
                    }
                }

                var cnt = $('<span class="content">');
                li.append(cnt);

                var title = $('<span data-property="displayName" class="list-title pull-left">');
                title.html('<b>' + Utils.escapeHTML(folder.title) + '</b>');
                cnt.append(title);

                var link = $('<a tabindex="1" class="sectiontitle" target="_blank" style="text-align: left; display: block; max-width: 80%;" href="#"></a>');

                link.data({ folder: id });
                link.addClass('deep-link-files');

                var displayPath = '';
                for (var i = 1; i < path.length - 1; i += 1) {
                    displayPath += DriveUtils.getShortenedTitle(path[i].title, 30) + ' / ';
                }
                displayPath += DriveUtils.getShortenedTitle(folder.title, 30);

                link.text(_.noI18n(displayPath));
                link.attr('title', _.noI18n(displayPath));

                cnt.append(link);

                if (oldLi) {
                    oldLi.replaceWith(li);
                } else {
                    folders.append(li);
                }
            }

            if (Utils.getBooleanOption(options, 'useMyFiles', false)) {
                path(myFiles, function (path) {
                    add(path, true);
                });
            }

            //load all folders saved on account
            //generates its gui elements
            _.each(templateModel.getAll(), function (id) {
                DriveUtils.deleteFromCache(id);
                path(id, function (path) {
                    add(path);
                });
            });

            var busy = false;

            function clickAdd(evt, oldLi, oldId) {
                evt.preventDefault();

                if (busy) {
                    return;
                }
                busy = true;

                var dialog = new Dialogs.ModalFolderPicker({
                    button: Labels.OK_LABEL,
                    height: 350,
                    title: Utils.getStringOption(options, 'folderPicker', Labels.SELECT_TEMPLATE_FOLDER)
                });
                dialog.show().always(function () {
                    busy = false;
                }).done(function (folder_id) {
                    var folder_id = dialog.getSelectedFolderId();
                    if (folder_id && folder_id !== myFiles) {
                        path(folder_id, function (path) {
                            var id = _.last(path).id;
                            if (!templateModel.add(id)) {
                                if (ox.debug) { console.warn('folder already in list', id); }
                                return;
                            }
                            if (oldId) {
                                templateModel.remove(oldId);
                            }
                            add(path, false, oldLi, true);
                        });
                    }
                });
            }

            if (!readOnly) {
                /**
                 * standard button for opening FolderPicker
                 * if selected folder is new, it will be added to account & to gui
                 */
                settings.appendButton(Utils.getStringOption(options, 'newFolder', Labels.ADD_NEW_TEMPLATE_FOLDER), clickAdd);
            }

            settings.append($('<div style="display: block; height: 20px">'));

        };
    }

    // exports ================================================================

    return TemplateFolders;

});
