/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/toolbartabcollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/baseframework/view/viewobjectmixin'
], function (Utils, TriggerObject, ViewObjectMixin) {

    'use strict';

    // class ToolBarTabCollection =============================================

    /**
     * A collection containing all tool bar tabs. Provides information about
     * the visibility of the tool bars, according to the application state.
     *
     * Instances of this class trigger the following events:
     *  - 'tab:create'
     *      After a new tab has been created. Event handlers receive the
     *      following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {String} tabId
     *          The identifier of the new tab.
     *      (3) {Object} [options]
     *          Additional options passed while creating the new tab.
     *  - 'tab:show'
     *      After a tab has been shown (and was hidden before). Event handlers
     *      receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {String} tabId
     *          The identifier of the visible tab.
     *  - 'tab:hide'
     *      After a tab has been hidden (and was visible before). Event
     *      handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {String} tabId
     *          The identifier of the hidden tab.
     *  - 'tab:activate'
     *      After a new tab has been activated. Event handlers receive the
     *      following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {String} newTabId
     *          The identifier of the activated tab.
     *      (3) {String} oldTabId
     *          The identifier of the deactivated tab.
     *  - 'tab:label'
     *      After the label text of a tab has been changed dynamically due to
     *      the current value of a controller item. Event handlers receive the
     *      following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {String} tabId
     *          The identifier of the tab.
     *      (3) {String} label
     *          The new label text for the tab.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends ViewObjectMixin
     */
    function ToolBarTabCollection(docView) {

        var // self reference
            self = this,

            // all existing tool bar tabs
            tabs = [],

            // identifier of the active tab
            activeTabId = null,

            // identifier of the last active tab, while no tab is active (bug 39071)
            lastTabId = null;

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        ViewObjectMixin.call(this, docView);

        // private methods ----------------------------------------------------

        /**
         * Returns the descriptor of the active tab.
         */
        function getActiveTab() {
            return _.findWhere(tabs, { tabId: activeTabId });
        }

        /**
         * Activates the first visible tab.
         *
         * @param {Array} [filteredTabs]
         *  If specified, activates the first visible tab in the passed array.
         *  Otherwise, searches for a visible tab in all available tabs.
         *
         * @returns {Boolean}
         *  Whether a visible tab has been found and activated successfully.
         */
        function activateFirstTab(filteredTabs) {

            var // filter by visible tabs
                visibleTabs = _.where(filteredTabs || tabs, { visible: true }),
                // the active tab
                activeTab = getActiveTab();

            // filter by priority of the active tab
            if (activeTab && activeTab.visible) {
                visibleTabs = _.filter(visibleTabs, function (tab) {
                    return tab.priority <= activeTab.priority;
                });
            }

            // no tab available for activation
            if (visibleTabs.length === 0) {
                return false;
            }

            // activate tab with highest priority
            visibleTabs = _.sortBy(visibleTabs, _.property('priority'));
            self.activateTab(visibleTabs[0].tabId);
            return true;
        }

        /**
         * Updates the visibility of the tab buttons.
         */
        function controllerChangeHandler() {

            var // the application controller
                controller = docView.getApp().getController(),
                // all tabs currently visible
                visibleTabs = _.where(tabs, { visible: true }),
                // all tabs currently hidden
                hiddenTabs = _.where(tabs, { visible: false }),
                // the active tab
                activeTab = getActiveTab();

            // process all tabs
            _.each(tabs, function (tab) {

                var // tabs without visibility key are always visible
                    visible = _.isString(tab.visibleKey) ? controller.isItemEnabled(tab.visibleKey) : true,
                    // get dynamic label text
                    label = _.isString(tab.labelKey) ? controller.getItemValue(tab.labelKey) : true;

                // show or hide the tabs, if the state has changed
                if (tab.visible !== visible) {
                    tab.visible = visible;
                    self.trigger(visible ? 'tab:show' : 'tab:hide', tab.tabId);
                }

                // change the button text, if the item is contained in the event data
                if (_.isString(label) && (tab.label !== label)) {
                    tab.label = label;
                    self.trigger('tab:label', tab.tabId, label);
                }
            });

            // do nothing, if active tab is visible and sticky
            if (activeTab && activeTab.visible && activeTab.sticky) {
                return;
            }

            // do not activate a tab, if some are visible, but none is active
            if ((visibleTabs.length > 0) && !activeTab) { return; }

            // activate the first tab that changed from hidden to visible
            if (activateFirstTab(hiddenTabs)) { return; }

            // nothing to do, if the active tab is visible
            if (activeTab && activeTab.visible) { return; }

            // otherwise, if the active tab is now hidden, activate the first visible tab (except any sticky tab)
            activateFirstTab(_.filter(visibleTabs, function (tab) {
                return tab.sticky !== true;
            }));
        }

        // public methods -----------------------------------------------------

        /**
         * Registers a new tool bar tab.
         *
         * @param {String} tabId
         *  The unique identifier for the new tab.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.visibleKey]
         *      The key of the controller item that controls the visibility of
         *      the tool bar tab. The visibility will be bound to the 'enabled'
         *      state of the respective controller item. If omitted, the tab
         *      will always be visible.
         *  @param {String} [options.labelKey]
         *      If specified, the key of a controller item whose value will be
         *      used to update the label text of the tab dynamically.
         *  @param {Number} [options.priority=0]
         *      The priority of the tool bar tab. Used to decide which tool bar
         *      tab gets activated when the tool bar tab currently active has
         *      been hidden by a controller item. The higher the number, the
         *      lower the priority of the tab.
         *  @param {Number} [options.sticky=false]
         *      Whether the tool bar tab should stay active regardless of other
         *      tabs becoming visible.
         *
         * @returns {ToolBarTabCollection}
         *  A reference to this instance.
         */
        this.createTab = function (tabId, options) {

            var // the tab descriptor
                tab = {
                    tabId: tabId,
                    visibleKey: Utils.getStringOption(options, 'visibleKey'),
                    labelKey: Utils.getStringOption(options, 'labelKey'),
                    priority: Utils.getIntegerOption(options, 'priority', 0),
                    sticky: Utils.getBooleanOption(options, 'sticky', false),
                    visible: true,
                    label: null
                };

            // insert new tab and notify listeners
            tabs.push(tab);
            this.trigger('tab:create', tabId, options);

            // initially hide all tabs with existing visibility key
            if (_.isString(tab.visibleKey)) {
                tab.visible = false;
                this.trigger('tab:hide', tabId);
            }

            return this;
        };

        /**
         * Returns the identifier of the tab currently active.
         *
         * @returns {String|Null}
         *  The identifier of the tab that is currently activated.
         */
        this.getActiveTabId = function () {
            return activeTabId;
        };

        /**
         * Activates the specified tab. If another tab was currently active, a
         * 'tab:deactivate' event will be triggered for that tab. Afterwards,
         * if the passed tab identifier is valid, a 'tab:activate' event will
         * be triggered.
         *
         * @param {String|Null} tabId
         *  The identifier of the tab to be activated.
         *
         * @returns {ToolBarTabCollection}
         *  A reference to this instance.
         */
        this.activateTab = function (tabId) {
            if (activeTabId !== tabId) {
                var oldTabId = activeTabId;
                activeTabId = tabId;
                this.trigger('tab:activate', tabId, oldTabId);
                docView.getApp().getController().update();
                // bug 39071: rescue last tab identifier, to be able to restore later
                if (!tabId) { lastTabId = oldTabId; }
            }
            return this;
        };

        /**
         * Activates the first visible tab.
         *
         * @returns {ToolBarTabCollection}
         *  A reference to this instance.
         */
        this.activateFirstTab = function () {
            activateFirstTab();
            return this;
        };

        /**
         * Restores the last visible tab, if no tab is currently visible.
         *
         * @returns {ToolBarTabCollection}
         *  A reference to this instance.
         */
        this.restoreActivateTab = function () {
            if (!activeTabId) {
                if (lastTabId) {
                    this.activateTab(lastTabId);
                    lastTabId = null;
                } else {
                    activateFirstTab();
                }
            }
            return this;
        };

        // initialization -----------------------------------------------------

        // update visibility of the tabs
        this.handleChangedControllerItems(controllerChangeHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = tabs = null;
        });

    } // class ToolBarTabCollection

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: ToolBarTabCollection });

});
