/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.git.conflict;

import static com.openexchange.Definitions.DEBIAN_CHANGELOG_PATTERN;
import static com.openexchange.Definitions.DEBIAN_DF;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.regex.Matcher;
import com.openexchange.data.DebianChangeLogSorter;
import com.openexchange.data.DebianChangeLogEntry;
import com.openexchange.git.MergeException;
import com.openexchange.git.Strings;
import com.openexchange.git.data.Conflict;

/**
 * {@link DebianChangelog}
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public final class DebianChangelog implements Solver {

    public DebianChangelog() {
        super();
    }

    public boolean applies(Conflict conflict) {
        String here = Strings.append(conflict.getHere());
        String there = Strings.append(conflict.getThere());
        Matcher matcher = DEBIAN_CHANGELOG_PATTERN.matcher(here);
        StringBuilder sb = new StringBuilder();
        while (matcher.find()) {
            sb.append(matcher.group());
            sb.append('\n');
        }
        if (sb.length() > 0 && there.length() > 0) {
            sb.setLength(sb.length() - 1);
        }
        if (!sb.toString().equals(here)) {
            sb.append('\n');
            if (!sb.toString().equals(here)) {
                return false;
            }
        }
        sb.setLength(0);
        matcher = DEBIAN_CHANGELOG_PATTERN.matcher(there);
        while (matcher.find()) {
            sb.append(matcher.group());
            sb.append('\n');
        }
        // If lines are only added then we have to check if here is empty. Only then the last trailing line feed is part of there, too.
        if (sb.length() > 0 && here.length() > 0) {
            sb.setLength(sb.length() - 1);
        }
        if (!sb.toString().equals(there)) {
            sb.append('\n');
            if (!sb.toString().equals(there)) {
                return false;
            }
        }
        return true;
    }

    public List<String> solve(Conflict conflict) throws MergeException {
        List<DebianChangeLogEntry> entries = new ArrayList<DebianChangeLogEntry>();
        String here = Strings.append(conflict.getHere());
        Matcher matcher = DEBIAN_CHANGELOG_PATTERN.matcher(here);
        while (matcher.find()) {
            String datum = matcher.group(6);
            final Date date;
            try {
                date = new SimpleDateFormat(DEBIAN_DF, Locale.ENGLISH).parse(datum);
            } catch (ParseException e) {
                throw new MergeException("Unable to parse date for changelog entry: \"" + datum + "\"", e);
            }
            DebianChangeLogEntry entry = new DebianChangeLogEntry(matcher.group(1), matcher.group(2), matcher.group(3), matcher.group(4), matcher.group(5), date);
            entries.add(entry);
        }
        String there = Strings.append(conflict.getThere());
        matcher = DEBIAN_CHANGELOG_PATTERN.matcher(there);
        while (matcher.find()) {
            String datum = matcher.group(6);
            final Date date;
            try {
                date = new SimpleDateFormat(DEBIAN_DF, Locale.ENGLISH).parse(datum);
            } catch (ParseException e) {
                throw new MergeException("Unable to parse date for changelog entry: \"" + datum + "\"", e);
            }
            DebianChangeLogEntry entry = new DebianChangeLogEntry(matcher.group(1), matcher.group(2), matcher.group(3), matcher.group(4), matcher.group(5), date);
            entries.add(entry);
        }
        Collections.sort(entries, new DebianChangeLogSorter());
        List<String> replacement = new ArrayList<String>();
        for (DebianChangeLogEntry entry : entries) {
            for (String line : entry.toString().split("\n")) {
                replacement.add(line);
            }
            replacement.add("");
        }
        if (here.length() != 0 && there.length() != 0) {
            replacement.remove(replacement.size() - 1);
        }
        return replacement;
    }
}
