/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.obs.api;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

/**
 * Writes the XML necessary to create a project on the build service.
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public final class ProjectWriter {

    private static final DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();

    private ProjectWriter() {
        super();
    }

    public static Document write(final Project project) throws BuildServiceException {
        final Document document;
        try {
            final DocumentBuilder builder = dbf.newDocumentBuilder();
            document = builder.newDocument();
            final Element projectElem = document.createElement("project");
            document.appendChild(projectElem);
            writeProject(project, projectElem);
        } catch (ParserConfigurationException e) {
            throw new BuildServiceException(e.getMessage(), e);
        }
        return document;
    }

    private static void writeProject(final Project project, Element projectElem) {
        final Document document = projectElem.getOwnerDocument();
        projectElem.setAttribute("name", project.getName());
        final Element title = document.createElement("title");
        projectElem.appendChild(title);
        final Text titleText = document.createTextNode(project.getTitle());
        title.appendChild(titleText);
        final Element description = document.createElement("description");
        projectElem.appendChild(description);
        writePerson(project, projectElem);
        writeBuild(project, projectElem);
        writeRepositories(project.getRepositories(), projectElem);
    }

    private static void writePerson(Project project, Element projectElem) {
        final Document document = projectElem.getOwnerDocument();
        final Element person = document.createElement("person");
        projectElem.appendChild(person);
        person.setAttribute("userid", project.getPerson());
        person.setAttribute("role", "maintainer");
        // always add the administrator. this makes editing projects easier and faster. oxbuilderuser is somewhat slow.
        final Element person2 = document.createElement("person");
        projectElem.appendChild(person2);
        person2.setAttribute("userid", "Admin");
        person2.setAttribute("role", "maintainer");
    }

    private static void writeBuild(Project project, Element projectElem) {
        final Document document = projectElem.getOwnerDocument();
        if (project.getDisabledRepositories().length == 0) {
            return;
        }
        final Element build = document.createElement("build");
        projectElem.appendChild(build);
        if (project.isBuildDisabled()) {
            final Element disable = document.createElement("disable");
            build.appendChild(disable);
        }
        for (final String repository : project.getDisabledRepositories()) {
            final Element disable = document.createElement("disable");
            build.appendChild(disable);
            disable.setAttribute("repository", repository);
        }
    }

    private static void writeRepositories(final Repository[] repositories, final Element project) {
        final Document document = project.getOwnerDocument();
        for (final Repository repository : repositories) {
            final Element repositoryElement = document.createElement("repository");
            project.appendChild(repositoryElement);
            repositoryElement.setAttribute("name", repository.getName());
            writePaths(repository.getPaths(), repositoryElement);
            writeArchs(repository,repositoryElement);
        }
    }

    private static void writePaths(final Path[] paths, final Element repository) {
        final Document document = repository.getOwnerDocument();
        for (final Path path : paths){
            final Element pathElement = document.createElement("path");
            repository.appendChild(pathElement);
            pathElement.setAttribute("repository", path.getRepository().getName());
            pathElement.setAttribute("project", path.getProject().getName());
        }
    }

    private static void writeArchs(final Repository repository, final Element repositoryElement) {
        final Document document = repositoryElement.getOwnerDocument();
        for (Architecture arch : repository.getArch()) {
            final Element elementArch = document.createElement("arch");
            repositoryElement.appendChild(elementArch);
            elementArch.appendChild(document.createTextNode(arch.toString()));
        }
    }
}
