/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.*;
import java.util.UUID;
import javax.servlet.http.HttpServletRequest;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.exceptions.TemporaryDownOrBusyException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.ServletConstants;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.session.dataobject.DataObjectSet;

/**
 * Handler for the USM-Retrieve OX Ids command.
 * 
 * @author ldo
 *
 */
public class ResolveIDsHandler extends BaseCommandHandler {

	private static final String NOT_FOUND = "Not Found";
	private static final String[] REQUIRED_PARAMETERS = { USER, DEVICE, PASSWORD, OBJECT_UUIDS };
	private static final String[] OPTIONAL_PARAMETERS = { FOLDER_UUID };

    public ResolveIDsHandler(USMJSONServlet servlet, HttpServletRequest request) throws USMJSONAPIException {
        super(servlet, request);
    }

	@Override
	public ResponseObject handleRequest() throws USMJSONAPIException {
		try {
			String user = getStringParameter(USER);
			String password = getStringParameter(PASSWORD);
			String device = getStringParameter(DEVICE);
			
			boolean sessionAlreadyInDB = _servlet.getSessionManager().existsSessionInDB(user, password, ServletConstants.PROTOCOL, device);
			
			if (!sessionAlreadyInDB)
				throw new USMJSONAPIException(ConnectorBundleErrorCodes.RESOLVE_IDS_INVALID_DEVICE,
						ResponseStatusCode.WRONG_MISSING_PARAMETERS, "Unknown device ID");

			_session = _servlet.getSession(user, password, device, _currentHttpRequest.getRemoteAddr(), getXHeadersFromCurrentRequest());

			String folderUUID = getStringParameter(FOLDER_UUID);
			JSONArray objectUUIDs = getJSONArray(_parameters, OBJECT_UUIDS);
			JSONObject resultData = new JSONObject();
			JSONObject objectIds = new JSONObject();
			DataObject[] cachedObjects = null;
			if (folderUUID != null) {
				Folder folder = null;
				try {
					folder = getFolderByUUID(folderUUID);
				} catch (USMJSONAPIException ignored) {
					//do nothing
				}
				resultData.put(FOLDER_ID, (folder != null) ? folder.getID() : NOT_FOUND);
				if (folder != null)
					cachedObjects = _session.getCachedFolderElements(folder.getID(), folder.getElementsContentType());
			} else {
				cachedObjects = _session.getCachedFolders();
			}

			DataObjectSet objectSet = new DataObjectSet(cachedObjects);
			int length = objectUUIDs.length();
			for (int i = 0; i < length; i++) {
				String uuid = objectUUIDs.getString(i);
				DataObject object = objectSet.get(UUID.fromString(uuid));
				objectIds.put(uuid, (object != null) ? object.getID() : NOT_FOUND);
			}
			resultData.put(OBJECTIDS, objectIds);
			return new ResponseObject(ResponseStatusCode.SUCCESS, resultData);
		} catch (USMJSONAPIException e) {
			throw e;
		} catch (IllegalArgumentException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.RESOLVE_IDS_INVALID_UUID,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "Invalid UUID", e);
        } catch (TemporaryDownOrBusyException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.RESOLVE_IDS_TEMPORARY_DOWN_OR_BUSY, e);
        } catch (USMException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.RESOLVE_IDS_LOGIN_UNSUCCESSFUL,
					ResponseStatusCode.ACCESS_DENIED, "Can not login.", e);
		} catch (JSONException e) {
			throw USMJSONAPIException.createJSONError(
					ConnectorBundleErrorCodes.RESOLVE_IDS_JSON_ERROR_ON_CREATE_RESULT, e);
		}
	}

	@Override
	protected String[] getOptionalParameters() {
		return OPTIONAL_PARAMETERS;
	}

	@Override
	protected String[] getRequiredParameters() {
		return REQUIRED_PARAMETERS;
	}
}
