/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.json;

import static com.openexchange.usm.connector.commands.CommandConstants.EXTRA_PROPERTIES_PREFIX;
import java.util.BitSet;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.calendar.AppointmentContentType;
import com.openexchange.usm.api.contenttypes.common.CommonConstants;
import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.contenttypes.common.ContentTypeField;
import com.openexchange.usm.api.contenttypes.common.DefaultContentTypes;
import com.openexchange.usm.api.contenttypes.folder.FolderContentType;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.AuthenticationFailedException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.USMAccessDeniedException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.SessionInitializer;
import com.openexchange.usm.connector.sync.ContactSorter;
import com.openexchange.usm.connector.sync.MailSorter;
import com.openexchange.usm.connector.sync.TaskAndAppointmentSorter;
import com.openexchange.usm.mimemail.ExternalMailContentTypeFields;
import com.openexchange.usm.util.BitSetEncoder;
import com.openexchange.usm.util.JSONToolkit;

public class JSONSessionInitializer implements SessionInitializer {

    public static final String FOLDER_TREE = "tree";

    private static final String[] EXCLUDED_FIELDS_FROM_DEFAULT = { "creation_date", "last_modified", "last_modified_utc" };

    private static final String[] INVALID_TASK_FIELDS = { "recurrence_position", "recurrence_date_position", "notification",
    /* "occurrences", */
    /* "users" */
    };

    private static final String[] RELEVANT_MAIL_FIELDS = { "id", "folder_id", "flags", "color_label" };

    private static final String[] INVALID_FOLDER_FIELDS = { "total", "new", "unread", "deleted"};

    private static ContactSorter _contactSorter;

    private static MailSorter _mailSorter;

    private static TaskAndAppointmentSorter _taskSorter;

    private static TaskAndAppointmentSorter _appointmentSorter;

    public static boolean excludeInvalidFieldsFromFilter(ContentType type, BitSet fieldSet) {
        boolean excluded = false;
        if (DefaultContentTypes.TASK_ID.equals(type.getID())) {
            ContentTypeField[] fields = type.getFields();
            for (int i = fieldSet.nextSetBit(0); i >= 0; i = fieldSet.nextSetBit(i + 1)) {
                if (isExcludedField(fields[i].getFieldName(), INVALID_TASK_FIELDS)) {
                    excluded = true;
                    fieldSet.clear(i);
                }
            }
        }
        if (DefaultContentTypes.MAIL_ID.equals(type.getID())) {
            ContentTypeField[] fields = type.getFields();
            for (int i = fieldSet.nextSetBit(0); i >= 0; i = fieldSet.nextSetBit(i + 1)) {
                String name = fields[i].getFieldName();
                if (!isExcludedField(name, RELEVANT_MAIL_FIELDS)) {
                    excluded |= !isInternalMailField(name);
                    fieldSet.clear(i);
                }
            }
            activateInternalFields(fieldSet, fields);
        }
        if (DefaultContentTypes.FOLDER_ID.equals(type.getID())) {
            ContentTypeField[] fields = type.getFields();
            for (int i = fieldSet.nextSetBit(0); i >= 0; i = fieldSet.nextSetBit(i + 1)) {
                if (isExcludedField(fields[i].getFieldName(), INVALID_FOLDER_FIELDS)) {
                    excluded = true;
                    fieldSet.clear(i);
                }
            }
        }
        return excluded;
    }

    public static boolean isInternalMailField(String name) {
        return "size".equals(name) || "received_date".equals(name);
    }

    private static boolean activateInternalFields(BitSet fieldSet, ContentTypeField[] fields) {
        boolean result = false;
        for (int i = 0; i < fields.length; i++) {
            if (!fieldSet.get(i)) {
                String fieldName = fields[i].getFieldName();
                if (isInternalMailField(fieldName)) {
                    fieldSet.set(i);
                    result = true;
                }
            }
        }
        return result;
    }

    private static boolean isExcludedField(String fieldName, String[] invalidFields) {
        for (String s : invalidFields) {
            if (fieldName.equals(s))
                return true;
        }
        return false;
    }

    private final USMJSONServlet _servlet;

    public JSONSessionInitializer(USMJSONServlet connectorServlet) {
        _servlet = connectorServlet;
    }

    @Override
    public void initializeSession(Session session, JSONObject ajaxConfiguration) throws USMException {
        // set the folder tree identifier to the default value "1"
        session.setPersistentField(FolderContentType.FOLDER_TREE, "1");
        session.setPersistentField(CommonConstants.ERROR_ON_DUPLICATE_NAME_DB_FIELD, "1");
        for (ContentType type : _servlet.getContentTypeManager().getRegisteredContentTypes()) {
            BitSet defaultFields = new BitSet();
            ContentTypeField[] fields = type.getFields();
            for (int i = 0; i < fields.length; i++) {
                if (shouldIncludeFieldAsDefault(fields[i]))
                    defaultFields.set(i);
            }
            excludeInvalidFieldsFromFilter(type, defaultFields);
            session.setFieldFilter(type, defaultFields);

            if (DefaultContentTypes.MAIL_ID.equals(type.getID())) {
                BitSet extraFieldsBitSet = new BitSet();
                String[] externalFields = ExternalMailContentTypeFields.getFields();
                for (int i = 0; i < externalFields.length; i++) {
                    extraFieldsBitSet.set(i);
                }
                storeExtraFields(session, type, extraFieldsBitSet);
            }

            // special case for Resources and Groups where all fields although with negative indexes are included for synchronization
            if (DefaultContentTypes.GROUPS_ID.equals(type.getID()) || DefaultContentTypes.RESOURCES_ID.equals(type.getID())) {
                for (int i = 0; i < fields.length; i++) {
                    defaultFields.set(i);
                }
                session.setFieldFilter(type, defaultFields);
            }
        }
        reinitializeSession(session, ajaxConfiguration); // called to set the custom properties
    }

    public static void storeExtraFields(Session session, ContentType type, BitSet extraFields) throws StorageAccessException, USMStorageException {
        String extraFieldStorage = extraFields.isEmpty() ? "" : BitSetEncoder.toBase64(extraFields);
        session.setPersistentField(getTypePropertyKey(type), extraFieldStorage);
    }

    public static String getTypePropertyKey(ContentType type) {
        return EXTRA_PROPERTIES_PREFIX + type.getID();
    }

    private boolean shouldIncludeFieldAsDefault(ContentTypeField field) {
        if (field.getFieldID() < 0)
            return false;
        for (String f : EXCLUDED_FIELDS_FROM_DEFAULT) {
            if (f.equals(field.getFieldName()))
                return false;
        }
        return true;
    }

    @Override
    public void reinitializeSession(Session session, JSONObject ajaxConfiguration) throws USMException {
        checkAccess(session, ajaxConfiguration);
        ContentType mailType = _servlet.getContentTypeManager().getContentType(DefaultContentTypes.MAIL_ID);
        BitSet fields = session.getFieldFilter(mailType);
        if (fields == null)
            fields = new BitSet();
        if (mailType != null && activateInternalFields(fields, mailType.getFields()))
            session.setFieldFilter(mailType, fields);
        session.setCustomProperty(Session.MAIL_HARDDELETE_PROPERTY, Boolean.TRUE);
        session.setCustomProperty(AppointmentContentType.DECLINE_INSTEAD_OF_DELETE, Boolean.TRUE);
        setContentTypeSorters(session);
    }

    private void setContentTypeSorters(Session session) {
        for (ContentType ct : _servlet.getContentTypeManager().getRegisteredContentTypes()) {
            if (ct.getID().equals(DefaultContentTypes.CALENDAR_ID)) {
                if (_appointmentSorter == null)
                    _appointmentSorter = new TaskAndAppointmentSorter(ct);
                session.setContentTypeSorter(ct, _appointmentSorter);
            } else if (ct.getID().equals(DefaultContentTypes.CONTACTS_ID)) {
                if (_contactSorter == null)
                    _contactSorter = new ContactSorter(ct);
                session.setContentTypeSorter(ct, _contactSorter);
            } else if (ct.getID().equals(DefaultContentTypes.MAIL_ID)) {
                if (_mailSorter == null)
                    _mailSorter = new MailSorter(ct);
                session.setContentTypeSorter(ct, _mailSorter);
            } else if (ct.getID().equals(DefaultContentTypes.TASK_ID)) {
                if (_taskSorter == null)
                    _taskSorter = new TaskAndAppointmentSorter(ct);
                session.setContentTypeSorter(ct, _taskSorter);
            }
        }
    }

    @Override
    public void checkAccess(Session session) throws USMException {
        checkAccess(session, null);
    }

    private void checkAccess(Session session, JSONObject ajaxConfiguration) throws USMAccessDeniedException, AuthenticationFailedException, OXCommunicationException {
        JSONObject oxInterfaces = getUSMJSONModule(session, ajaxConfiguration);
        if (oxInterfaces == null || !oxInterfaces.optBoolean("active")) {
            throw new USMAccessDeniedException(
                ConnectorBundleErrorCodes.INITIALIZER_USM_JSON_ACCESS_DENIED_FOR_USER,
                "USM JSON access disabled for user");
        }
        oxInterfaces = getNewFolderTreeModule(session, ajaxConfiguration);
        if (oxInterfaces == null || !oxInterfaces.has(FOLDER_TREE)) {
            throw new USMAccessDeniedException(
                ConnectorBundleErrorCodes.INITIALIZER_NEW_FOLDER_TREE_MODULE_NOT_INSTALLED,
                "Folder Tree package: open-xchange-folder-json is not installed. Access not possible.");
        }
    }

    private JSONObject getUSMJSONModule(Session session, JSONObject ajaxConfiguration) throws AuthenticationFailedException, OXCommunicationException {
        if (ajaxConfiguration != null)
            return JSONToolkit.getJSONObject(ajaxConfiguration, "modules", "com.openexchange.usm.json");
        try {
            return session.getOXUserConfiguration("modules", "com.openexchange.usm.json");
        } catch (OXCommunicationException e) {
            return null;
        }
    }

    private JSONObject getNewFolderTreeModule(Session session, JSONObject ajaxConfiguration) throws AuthenticationFailedException, OXCommunicationException {
        if (ajaxConfiguration != null)
            return JSONToolkit.getJSONObject(ajaxConfiguration, "modules", "folder");
        try {
            return session.getOXUserConfiguration("modules", "folder");
        } catch (OXCommunicationException e) {
            return null;
        }
    }

    @Override
    public boolean allowsUnlimitedWaitForChangesOnModifiedFolders() {
        return true;
    }

    @Override
    public boolean objectsSynchronizedByUUID() {
        return true;
    }

    @Override
    public void updateDefaultFolders(Session session, JSONObject ajaxConfiguration) throws USMException {
        // TODO Auto-generated method stub
        
    }
}
