/*
 *
  *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.tools.cache;

import java.lang.ref.SoftReference;
import java.util.HashMap;

/**
 * A SoftReference cache which can be used to store data for fast access while
 * not burden the memory footprint with data that can be restored in others
 * ways.
 * 
 * @author Carsten Driesner
 *
 */
public class SoftReferenceCache<K, V> {

	private HashMap<K, SoftReference<V>> hashMap;
	
	/**
	 * Initializes a new soft cache.
	 */
	public SoftReferenceCache() {
		hashMap = new HashMap<K, SoftReference<V>>();
	}

	/**
	 * Provides the number of entries within the cache. As this is a
	 * soft cache, there is no guarantee that the entries are all
	 * accessible.
	 *
	 * @return
	 *  The number of entries in the cache.
	 */
	public int size() {
		return hashMap.size();
	}

	/**
	 * Stores a new entry in the cache.
	 *
	 * @param key
	 *  The key that is used to uniquely identify the value object.
	 *
	 * @param value
	 *  The value that is associated with the key.
	 */
	public void setCacheEntry(K key, V value) {
		SoftReference<V> softRef = new SoftReference<V>(value);
		
		hashMap.put(key,softRef);
	}

	/**
	 * Retrieves an entry from the cache. Due to the nature of a soft
	 * reference cache the value could be null although the association
	 * is still stored in the cache.
	 *
	 * @param key
	 *  The key that references the value in the cache.
	 *
	 * @return
	 *  The value or null if the value is not within the cache or the
	 *  garbage collector chose to remove the object.
	 */
	public V getCacheEntry(K key) {
		SoftReference<V> softRef = hashMap.remove(key);
		
		if (null == softRef)
			return null;

		V value = softRef.get();
	    return value;
	}
}
