/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.realtime.impl;

import java.util.Date;
import java.util.Enumeration;
import java.util.concurrent.ConcurrentHashMap;
import com.openexchange.office.realtime.impl.calc.AsyncLoadRequestQueue;
import com.openexchange.office.realtime.osgi.Services;
import com.openexchange.office.tools.ErrorCode;
import com.openexchange.office.tools.IDUtils;
import com.openexchange.realtime.Component;
import com.openexchange.realtime.ComponentHandle;
import com.openexchange.realtime.cleanup.AbstractRealtimeJanitor;
import com.openexchange.realtime.cleanup.GlobalRealtimeCleanup;
import com.openexchange.realtime.packet.ID;
import com.openexchange.server.ServiceLookup;
import com.openexchange.timer.ScheduledTimerTask;
import com.openexchange.timer.TimerService;

/**
 * {@link ConnectionComponent}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
public class ConnectionComponent extends AbstractRealtimeJanitor implements Component {

    private static class FailSafeRunnable implements Runnable {

        private final ConnectionComponent m_connectionPool;

        public FailSafeRunnable(final ConnectionComponent connectionPool) {
            super();
            m_connectionPool = connectionPool;
        }

        @Override
        public void run() {
            m_connectionPool.checkForFailSaveDocument();
        }
    }

    /**
     * Initializes a new {@link ConnectionComponent}.
     * @param services
     */
    public ConnectionComponent(ServiceLookup services) {
        super();
        m_services = services;
        m_saveDataTimer = m_services.getService(TimerService.class);
        m_requestQueue = new AsyncLoadRequestQueue(services);
    }

    /**
     * @return
     */
    ConcurrentHashMap<ID, Connection> getOpenConnections() {
        return m_openConnections;
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.realtime.Component#create(com.openexchange.realtime.packet.ID)
     */
    @Override
    public ComponentHandle create(ID id) {
        // Create operations queue with layout: synthetic.office://operations/folderId.fileId~fileVersion_fileName
        Connection connection = new Connection(m_services, id, m_requestQueue);
        m_openConnections.put(id, connection);

        if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: New document created with ID: " + id.toString() + ", connection count: " + m_openConnections.size());
        }

         if (m_openConnections.size() == 1) {
            this.startFailSafeTimer();
            m_requestQueue.startQueue();
        }

        return connection;
    }

    /* (non-Javadoc)
     * @see com.openexchange.realtime.Component#getId()
     */
    @Override
    public String getId() {
        // Address operations queue with layout: synthetic.office://operations/folderId.fileId~fileVersion_fileName
        return "office";
    }

    /* (non-Javadoc)
     * @see com.openexchange.realtime.Component#getEvictionPolicy()
     */
    @Override
    public EvictionPolicy getEvictionPolicy() {
        // No automatic shut down of operations queue
        return com.openexchange.realtime.Component.NONE;
    }

    /**
     * Tries to shutdown the connection component gracefully, including saving
     * the current modified documents to OX Drive and to close timer/threads.
     */
    public void shutdown() {
        for(ID id: m_openConnections.keySet()) {
            Connection connection = m_openConnections.get(id);
            if (null != connection) {
                connection.setShutdown();
            }
        }

        if (m_saveDataTimer != null) {
            m_saveDataTimer.purge();
        }

        GlobalRealtimeCleanup globalRealtimeCleanup = Services.optService(GlobalRealtimeCleanup.class);
        if(globalRealtimeCleanup == null) {
            LOG.error("Unable to issue cluster wide cleanup for Connection IDs due to missing GlobalRealtimeCleanup service.");
        }

        if (null != m_requestQueue) {
            m_requestQueue.destroyQueue();
        }

        for(ID id: m_openConnections.keySet()) {
            Connection connection = m_openConnections.get(id);
            if (null != connection) {
                connection.saveDocumentOnDispose();
            }

            if(globalRealtimeCleanup != null) {
                globalRealtimeCleanup.cleanForId(id);
            }
        }
    }

    /**
     * Checks if there are documents which need to be saved
     * for better fail safe handling.
     */
    void checkForFailSaveDocument() {
        Enumeration<Connection> connections = m_openConnections.elements();
        while (connections.hasMoreElements()) {
            try {
                Date currentDate = new Date();
                Connection connection = connections.nextElement();

                // Don't try to save, if the connection has already been disposed, we are
                // in a middle of a save or we were not able to save the document correctly.
                if ((connection != null) && !connection.isDisposed() &&
                    !connection.savingDocumentInProgress() &&
                    (connection.getLastFailSafeSaveError().getCode() == ErrorCode.NO_ERROR.getCode())) {
                    long diff = getTimeDifference(connection.getSaveTimeStamp(), currentDate);
                    long ops = connection.getNumberOfPendingOperations();

                    // check if we have to do a fail safe document save
                    if (((diff >= MIN_TIME_FOR_MANY_OPS_FAILSAFESAVE) && (ops >= MIN_OPERATIONS_FOR_FAILSAFESAVE)) ||
                        ((diff >= MAX_TIME_FOR_MODIFIED_FAILSAFESAVE) && (ops > 0))) {
                        connection.failSafeSaveDocument();
                    }
                }
            } catch (Exception e) {
                // Ignore if saving doesn't work. This is just an additional
                // periodically save in the background.
                LOG.error("RT connection: Exception catched during failSafeSaveDocument", e);
            }
        }
    }

    /**
     * Starts the fail safe save timer.
     */
    void startFailSafeTimer() {
        if (m_saveDataTimer != null) {
            m_failSafeTimerTask = m_saveDataTimer.scheduleAtFixedRate(new FailSafeRunnable(this), MIN_PERIODIC_TIMERVALUE, MIN_PERIODIC_TIMERVALUE);
        } else {
            m_failSafeTimerTask = null;
        }
    }

    /**
     * Stops a running fail safe save timer.
     */
    void stopFailSafeTimer() {
        if (m_failSafeTimerTask != null) {
            m_failSafeTimerTask.cancel();
            m_failSafeTimerTask = null;
        }
    }

    /**
     * Calculates the absolute difference between two Dates
     *
     * @param d1 Date1
     * @param d2 Date2
     * @return
     *  The difference in milliseconds.
     */
    private long getTimeDifference(Date d1, Date d2) {
        return Math.abs(d1.getTime() - d2.getTime());
    }

    @Override
    public void cleanupForId(ID id) {
        // We are only interested in IDs of office components for now
        if (OFFICE_COMPONENT.equals(id.getComponent())) {
            ConcurrentHashMap<ID, Connection> connections = getOpenConnections();
            Connection removedConnection = connections.remove(id);
            if (removedConnection == null) {
                LOG.debug("Couldn't remove connection for ID: " + id);
            } else {
            	m_requestQueue.purgeDocumentRequests(IDUtils.extractFolderAndFileId(id));
                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: Removed document with ID: " + id.toString() + ", connection count: " + connections.size());
                }
            }
            if (connections.size() == 0) {
                stopFailSafeTimer();
            }
        }
    }

    // - Members ---------------------------------------------------------------
    static protected final org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(ConnectionComponent.class);

    private final static String OFFICE_COMPONENT="office";

    private final ServiceLookup m_services;

    private final ConcurrentHashMap<ID, Connection> m_openConnections = new ConcurrentHashMap<ID, Connection>();

    private final TimerService m_saveDataTimer;

    private ScheduledTimerTask m_failSafeTimerTask = null;

    private AsyncLoadRequestQueue m_requestQueue = null;

    private static final int MIN_OPERATIONS_FOR_FAILSAFESAVE = 50;

    private static final int MIN_TIME_FOR_MANY_OPS_FAILSAFESAVE = 60 * 1000; // time in milliseconds

    private static final int MAX_TIME_FOR_MODIFIED_FAILSAFESAVE = 15 * 60 * 1000; // time in milliseconds

    private static final int MIN_PERIODIC_TIMERVALUE = 30 * 1000; // time in milliseconds

}
