/*
 *
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.docx.operations;

import java.math.BigInteger;
import java.net.URI;
import java.net.URISyntaxException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.bind.JAXBElement;

import org.docx4j.openpackaging.URIHelper;
import org.docx4j.openpackaging.packages.WordprocessingMLPackage;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.openpackaging.parts.WordprocessingML.FontTablePart;
import org.docx4j.openpackaging.parts.WordprocessingML.FooterPart;
import org.docx4j.openpackaging.parts.WordprocessingML.HeaderPart;
import org.docx4j.openpackaging.parts.WordprocessingML.NumberingDefinitionsPart;
import org.docx4j.openpackaging.parts.WordprocessingML.StyleDefinitionsPart;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart;
import org.docx4j.relationships.Relationship;
import org.docx4j.vml.CTImageData;
import org.docx4j.vml.CTShape;
import org.docx4j.wml.BooleanDefaultTrue;
import org.docx4j.wml.Br;
import org.docx4j.wml.CTRel;
import org.docx4j.wml.CTSettings;
import org.docx4j.wml.CTTabStop;
import org.docx4j.wml.CTTblPrBase;
import org.docx4j.wml.CTTblStylePr;
import org.docx4j.wml.CTTwipsMeasure;
import org.docx4j.wml.CTZoom;
import org.docx4j.wml.Color;
import org.docx4j.wml.ContentAccessor;
import org.docx4j.wml.DocDefaults;
import org.docx4j.wml.DocDefaults.PPrDefault;
import org.docx4j.wml.DocDefaults.RPrDefault;
import org.docx4j.wml.FontFamily;
import org.docx4j.wml.FontPanose;
import org.docx4j.wml.FontPitch;
import org.docx4j.wml.Fonts;
import org.docx4j.wml.Fonts.Font.AltName;
import org.docx4j.wml.FooterReference;
import org.docx4j.wml.Ftr;
import org.docx4j.wml.Hdr;
import org.docx4j.wml.HeaderReference;
import org.docx4j.wml.IText;
import org.docx4j.wml.Jc;
import org.docx4j.wml.Lvl;
import org.docx4j.wml.Lvl.LvlText;
import org.docx4j.wml.Lvl.Start;
import org.docx4j.wml.Lvl.Suff;
import org.docx4j.wml.NumFmt;
import org.docx4j.wml.NumberFormat;
import org.docx4j.wml.Numbering;
import org.docx4j.wml.Numbering.AbstractNum;
import org.docx4j.wml.Numbering.AbstractNum.NumStyleLink;
import org.docx4j.wml.Numbering.Num;
import org.docx4j.wml.Numbering.Num.AbstractNumId;
import org.docx4j.wml.Numbering.Num.LvlOverride;
import org.docx4j.wml.Numbering.Num.LvlOverride.StartOverride;
import org.docx4j.wml.Numbering.NumPicBullet;
import org.docx4j.wml.P;
import org.docx4j.wml.PPr;
import org.docx4j.wml.PPrBase.Ind;
import org.docx4j.wml.PPrBase.NumPr;
import org.docx4j.wml.PPrBase.NumPr.NumId;
import org.docx4j.wml.Pict;
import org.docx4j.wml.RFonts;
import org.docx4j.wml.RPr;
import org.docx4j.wml.STBrType;
import org.docx4j.wml.STPitch;
import org.docx4j.wml.STTblStyleOverrideType;
import org.docx4j.wml.SectPr;
import org.docx4j.wml.SectPr.PgMar;
import org.docx4j.wml.SectPr.PgSz;
import org.docx4j.wml.Style;
import org.docx4j.wml.Style.Name;
import org.docx4j.wml.Style.Next;
import org.docx4j.wml.Style.UiPriority;
import org.docx4j.wml.Styles;
import org.docx4j.wml.Tabs;
import org.docx4j.wml.Tc;
import org.docx4j.wml.TcPr;
import org.docx4j.wml.TrPr;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.office.FilterException;
import com.openexchange.office.ooxml.docx.OperationDocument;
import com.openexchange.office.ooxml.docx.tools.Character;
import com.openexchange.office.ooxml.docx.tools.Component;
import com.openexchange.office.ooxml.docx.tools.Component.FldSimpleComponent;
import com.openexchange.office.ooxml.docx.tools.Component.HardBreakComponent;
import com.openexchange.office.ooxml.docx.tools.Component.MultiComponent;
import com.openexchange.office.ooxml.docx.tools.Component.ParagraphComponent;
import com.openexchange.office.ooxml.docx.tools.Component.ShapeComponent;
import com.openexchange.office.ooxml.docx.tools.Component.ShapeComponent_Root;
import com.openexchange.office.ooxml.docx.tools.Component.ShapeType;
import com.openexchange.office.ooxml.docx.tools.Component.TabComponent;
import com.openexchange.office.ooxml.docx.tools.Component.TableComponent;
import com.openexchange.office.ooxml.docx.tools.Component.TextComponent;
import com.openexchange.office.ooxml.docx.tools.Paragraph;
import com.openexchange.office.ooxml.docx.tools.Table;
import com.openexchange.office.ooxml.docx.tools.Utils;
import com.openexchange.office.ooxml.tools.Commons;


public class CreateOperationHelper extends com.openexchange.office.ooxml.operations.CreateOperationHelper {

    private final WordprocessingMLPackage wordprocessingMLPackage;
    private final Map<String, String> conditionalTableAttrNames;

    public CreateOperationHelper(OperationDocument _operationDocument, JSONArray mOperationsArray)
    	throws FilterException {

    	super(_operationDocument, mOperationsArray);
        wordprocessingMLPackage = _operationDocument.getPackage();

        conditionalTableAttrNames = new HashMap<String, String>();
        conditionalTableAttrNames.put("firstRow", "firstRow");
        conditionalTableAttrNames.put("lastRow", "lastRow");
        conditionalTableAttrNames.put("firstCol", "firstCol");
        conditionalTableAttrNames.put("lastCol", "lastCol");
        conditionalTableAttrNames.put("band1Vert", "band1Vert");
        conditionalTableAttrNames.put("band2Vert", "band2Vert");
        conditionalTableAttrNames.put("band1Horz", "band1Hor");
        conditionalTableAttrNames.put("band2Horz", "band2Hor");
        conditionalTableAttrNames.put("neCell", "northEastCell");
        conditionalTableAttrNames.put("nwCell", "northWestCell");
        conditionalTableAttrNames.put("seCell", "southEastCell");
        conditionalTableAttrNames.put("swCell", "southWestCell");
    }

    @Override
    public OperationDocument getOperationDocument() {
        return (OperationDocument)operationDocument;
    }

    public void createOperations()
        throws JSONException, ParseException, FilterException {

        createOperations(wordprocessingMLPackage.getMainDocumentPart(), new ArrayList<Integer>(), null);
    }

    public void CreateHeaderFooterOperations()
    	throws JSONException, ParseException, FilterException {

    	final SectPr sectPr = Utils.getDocumentProperties(wordprocessingMLPackage.getMainDocumentPart());
        if(sectPr!=null) {
        	for(CTRel rel:sectPr.getEGHdrFtrReferences()) {
        		final String id = rel.getId();
        		if(id!=null&&!id.isEmpty()) {
        			final RelationshipsPart relationshipsPart = wordprocessingMLPackage.getMainDocumentPart().getRelationshipsPart(false);
        			if(relationshipsPart!=null) {
        				AddInsertHeaderFooterOperation(id, rel);
        				final Part part = relationshipsPart.getPart(id);
        				if(part instanceof HeaderPart) {
        					final Hdr hdr = ((HeaderPart)part).getJaxbElement();
        					operationDocument.setContextPart(part);
        			        createOperations(hdr, new ArrayList<Integer>(), id);
        				}
        				else if(part instanceof FooterPart) {
        					final Ftr ftr = ((FooterPart)part).getJaxbElement();
        					operationDocument.setContextPart(part);
        					createOperations(ftr, new ArrayList<Integer>(), id);
        				}
        			}
        		}
        	}
        	operationDocument.setContextPart(null);
        }
    }

    @Override
    public void createDocumentDefaults(String userLanguage)
        throws Exception {

        final JSONObject documentAttributes = new JSONObject();
        final CTSettings settings = getOperationDocument().getSettings(false);
        final SectPr sectPr = Utils.getDocumentProperties(wordprocessingMLPackage.getMainDocumentPart());
        // document defaults
        final JSONObject jsonDocumentSettings = new JSONObject();
        if (settings!=null) {
            final BooleanDefaultTrue trackRevisions = settings.getTrackRevisions();
            if(trackRevisions!=null&&trackRevisions.isVal()) {
            	jsonDocumentSettings.put("changeTracking", true);
            }
            final JSONArray authors = getOperationDocument().getChangeTrackAuthors();
            if(authors!=null) {
            	jsonDocumentSettings.put("changeTrackAuthors", authors);
            }
            final CTTwipsMeasure defaultTabStop = settings.getDefaultTabStop();
            if (defaultTabStop != null) {
                jsonDocumentSettings.put("defaultTabStop", Utils.mapTwipTo100THMM(defaultTabStop.getVal()));
            }
            final CTZoom zoom = settings.getZoom();
            if (zoom!=null) {
                boolean addZoomAttr = false;
                JSONObject zoomAttributes = new JSONObject();
                if (zoom.getVal() != null) {
                    zoomAttributes.put("type", zoom.getVal().toString());
                    addZoomAttr = true;
                }
                if (zoom.getPercent() != null) {
                    zoomAttributes.put("value", zoom.getPercent().intValue());
                    addZoomAttr = true;
                }
                if (addZoomAttr) {
                    jsonDocumentSettings.put("zoom", zoomAttributes);
                }
            }
        }
        if(!jsonDocumentSettings.isEmpty()) {
            documentAttributes.put("document", jsonDocumentSettings);
        }
        // page defaults
        JSONObject jsonPageSettings = new JSONObject();
        if(settings!=null) {
            final BooleanDefaultTrue evenAndOddHeaders = settings.getEvenAndOddHeaders();
            final boolean evenOddPages = evenAndOddHeaders!=null?evenAndOddHeaders.isVal():false;
            if(evenOddPages) {
            	jsonPageSettings.put("evenOddPages", evenOddPages);
            }
        }
        if(sectPr!=null) {
            final PgSz pgSz = sectPr.getPgSz();
            if(pgSz!=null) {
                jsonPageSettings.put("width", Utils.mapTwipTo100THMM(pgSz.getW()));
                jsonPageSettings.put("height", Utils.mapTwipTo100THMM(pgSz.getH()));
            }
            final PgMar pgMar = sectPr.getPgMar();
            if(pgMar!=null) {
                jsonPageSettings.put("marginLeft", Utils.mapTwipTo100THMM(pgMar.getLeft()));
                jsonPageSettings.put("marginTop", Utils.mapTwipTo100THMM(pgMar.getTop()));
                jsonPageSettings.put("marginRight", Utils.mapTwipTo100THMM(pgMar.getRight()));
                jsonPageSettings.put("marginBottom", Utils.mapTwipTo100THMM(pgMar.getBottom()));
                jsonPageSettings.put("marginHeader", Utils.mapTwipTo100THMM(pgMar.getHeader()));
                jsonPageSettings.put("marginFooter", Utils.mapTwipTo100THMM(pgMar.getFooter()));
            }
            final BooleanDefaultTrue titlePg = sectPr.getTitlePg();
            final boolean firstPage = titlePg!=null?titlePg.isVal():false;
            if(firstPage) {
            	jsonPageSettings.put("firstPage", firstPage);
            }
        }
        if(!jsonPageSettings.isEmpty()) {
            documentAttributes.put("page", jsonPageSettings);
        }

        final StyleDefinitionsPart styleDefinitionsPart = wordprocessingMLPackage.getMainDocumentPart().getStyleDefinitionsPart();
        if(styleDefinitionsPart!=null) {
            final Styles styles = styleDefinitionsPart.getJaxbElement();
            final DocDefaults docDefaults = styles.getDocDefaults();
            if(docDefaults!=null) {
                final RPrDefault rDefault = docDefaults.getRPrDefault();
                if(rDefault!=null) {
                    final RPr rPrDefault = rDefault.getRPr();
                    if(rPrDefault!=null) {
                        final JSONObject jsonCharacterSettings = Character.createCharacterProperties(themeFonts, rPrDefault);
                        if(jsonCharacterSettings!=null&&jsonCharacterSettings.length()>0) {

                            // set user language if it doesn't exists within the document
                            String value = jsonCharacterSettings.optString("language");
                            if (value == null || value.length() == 0) {
                                if (userLanguage != null && userLanguage.length() > 0)
                                    jsonCharacterSettings.put("language", userLanguage);
                                else
                                    jsonCharacterSettings.put("language", "en-US"); // set default
                            }
                            value = jsonCharacterSettings.optString("languageEa");
                            if (value == null || value.length() == 0) {
                                jsonCharacterSettings.put("languageEa", "en-US"); // set default
                            }
                            value = jsonCharacterSettings.optString("languageBidi");
                            if (value == null || value.length() == 0) {
                                jsonCharacterSettings.put("languageBidi", "ar-SA"); // set default
                            }
                            documentAttributes.put("character", jsonCharacterSettings);
                        }
                    }
                }
                final PPrDefault pDefault = docDefaults.getPPrDefault();
                if(pDefault!=null) {
                    final PPr pPrDefault = pDefault.getPPr();
                    if(pPrDefault!=null) {
                        final JSONObject jsonParagraphSettings = Paragraph.createParagraphProperties(pPrDefault);
                        if(jsonParagraphSettings!=null&&jsonParagraphSettings.length()>0)
                            documentAttributes.put("paragraph", jsonParagraphSettings);
                    }
                }
            }
        }
        if(documentAttributes.length()>0)
            addSetDocumentAttributesOperation(documentAttributes);
    }

    /**
     * @param aOperationsArray
     * @param nPara
     */
    public void AddInsertParagraphOperation(final List<Integer> start, String target, final JSONObject attrs)
        throws JSONException {

        final JSONObject insertParagraphObject = new JSONObject();
        insertParagraphObject.put("name", "insertParagraph");
        insertParagraphObject.put("start", start);
        if(target!=null) {
        	insertParagraphObject.put("target", target);
        }
        if(attrs!=null&&!attrs.isEmpty()) {
        	insertParagraphObject.put("attrs", attrs);
        }
        operationsArray.put(insertParagraphObject);
    }

    /**
     * @param aOperationsArray
     * @param nPara
     * @param nPos
     * @param aText
     */
    public void AddInsertTextOperation(List<Integer> paragraphPosition, int textPosition, String target, TextComponent textComponent)
        throws JSONException {

        final JSONObject insertTextObject = new JSONObject();
        List<Integer> startArray = new ArrayList<Integer>(paragraphPosition);
        startArray.add(textPosition);

        insertTextObject.put("name", "insertText");
        insertTextObject.put("start", startArray);
        if(target!=null) {
        	insertTextObject.put("target", target);
        }
        insertTextObject.put("text", ((IText)textComponent.getObject()).getValue());
        operationsArray.put(insertTextObject);
    }

    public void AddInsertHeaderFooterOperation(String id, CTRel rel)
    	throws JSONException {


    	final JSONObject headerFooterOperation = new JSONObject();
    	headerFooterOperation.put("name", "insertHeaderFooter");
    	headerFooterOperation.put("id", id);
    	if(rel instanceof HeaderReference) {
			headerFooterOperation.put("type", "header_" + ((HeaderReference)rel).getType().value());
    	}
		else if (rel instanceof FooterReference) {
			headerFooterOperation.put("type", "footer_" + ((FooterReference)rel).getType().value());
    	}
    	operationsArray.put(headerFooterOperation);
    }

    public void AddInsertTabOperation(final List<Integer> start, String target)
        throws JSONException {

        final JSONObject insertTabObject = new JSONObject();
        insertTabObject.put("name", "insertTab");
        insertTabObject.put("start", start);
        if(target!=null) {
        	insertTabObject.put("target", target);
        }
        operationsArray.put(insertTabObject);
    }

    public void AddInsertHardBreakOperation(List<Integer> start, String target, STBrType type)
        throws JSONException {

        final JSONObject insertHardBreakObject = new JSONObject();
        insertHardBreakObject.put("name", "insertHardBreak");
        insertHardBreakObject.put("start", start);
        if(target!=null) {
        	insertHardBreakObject.put("target", target);
        }
        if(type!=null) {
        	insertHardBreakObject.put("type", type.value());
        }
        operationsArray.put(insertHardBreakObject);
    }

    public boolean AddInsertTableOperation(final List<Integer> start, String target, final JSONObject attrs)
        throws JSONException {

    	boolean tableSizeAllowed = true;
    	final JSONObject insertTableObject = new JSONObject();
        insertTableObject.put("name", "insertTable");
        insertTableObject.put("start", start);
        if(target!=null) {
        	insertTableObject.put("target", target);
        }
        if(attrs!=null&&!attrs.isEmpty()) {
            if(attrs.has("sizeExceeded")) {
            	tableSizeAllowed = false;
            	insertTableObject.put("sizeExceeded", attrs.getJSONObject("sizeExceeded"));
            	attrs.remove("sizeExceeded");
            }
        	insertTableObject.put("attrs", attrs);
        }
        operationsArray.put(insertTableObject);
        return tableSizeAllowed;
    }

    public void AddInsertRowsOperation(List<Integer> start, String target, JSONObject attrs)
        throws JSONException {

        final JSONObject insertRowObject = new JSONObject();
        insertRowObject.put("name", "insertRows");
        insertRowObject.put("start", start);
        if(target!=null) {
        	insertRowObject.put("target", target);
        }
        if(attrs!=null&&!attrs.isEmpty()) {
            insertRowObject.put("attrs", attrs);
        }
        operationsArray.put(insertRowObject);
    }

    public void AddInsertCellsOperation(List<Integer> start, String target, JSONObject attrs)
        throws JSONException {

        final JSONObject insertCellObject = new JSONObject();
        insertCellObject.put("name", "insertCells");
        insertCellObject.put("start", start);
        if(target!=null) {
        	insertCellObject.put("target", target);
        }
        if(attrs!=null&&!attrs.isEmpty()) {
            insertCellObject.put("attrs", attrs);
        }
        operationsArray.put(insertCellObject);
    }

    public void AddInsertFieldOperation(final List<Integer> start, String target, String type, String representation)
        throws JSONException {

        final JSONObject insertFieldObject = new JSONObject();
        insertFieldObject.put("name", "insertField");
        insertFieldObject.put("start", start);
        if(target!=null) {
        	insertFieldObject.put("target", target);
        }
        if(type!=null) {
            insertFieldObject.put("type", type);
        }
        if(representation!=null) {
            insertFieldObject.put("representation", representation);
        }
        operationsArray.put(insertFieldObject);
    }

    public void AddInsertDrawingOperation(final List<Integer> start, String target, ShapeType graphicType, final JSONObject attrs)
        throws JSONException {

        final JSONObject insertDrawingObject = new JSONObject();
        insertDrawingObject.put("name", "insertDrawing");
        insertDrawingObject.put("start", start);
        insertDrawingObject.put("type", graphicType.name().toLowerCase());
        if(target!=null) {
        	insertDrawingObject.put("target", target);
        }
        if(attrs!=null&&!attrs.isEmpty()) {
            insertDrawingObject.put("attrs", attrs);
        }
        operationsArray.put(insertDrawingObject);
    }

    public void AddSetAttributesOperation(final JSONObject attrs, final List<Integer> startPosition, final List<Integer> endPosition, String target)
        throws JSONException {

        JSONObject setAttributeObject = new JSONObject();
        setAttributeObject.put("name", "setAttributes");
        setAttributeObject.put("attrs", attrs );
        setAttributeObject.put("start", startPosition);
        if(target!=null) {
        	setAttributeObject.put("target", target);
        }
        if(endPosition!=null) {
            setAttributeObject.put("end", endPosition);
        }
        operationsArray.put(setAttributeObject);
    }

    public void AddInsertFontDescription(String name, JSONObject attrs)
        throws JSONException {

        if(name!=null&&name.length()>0&&attrs!=null&&attrs.length()>0) {
            final JSONObject fontDescription = new JSONObject();
            fontDescription.put("name", "insertFontDescription");
            fontDescription.put("fontName", name);
            fontDescription.put("attrs", attrs);
            operationsArray.put(fontDescription);
        }
    }

    public void CreateFontDescriptions()
        throws JSONException {

        FontTablePart fontTablePart = wordprocessingMLPackage.getMainDocumentPart().getFontTablePart();
        if(fontTablePart!=null) {
            org.docx4j.wml.Fonts fonts = fontTablePart.getJaxbElement();
            List<Fonts.Font> fontList = fonts.getFont();
            for(Fonts.Font font: fontList) {
                JSONObject attrs = new JSONObject();
                AltName altNames = font.getAltName();
                if(altNames!=null) {
                    String names = altNames.getVal();
                    if(names!=null) {
                        JSONArray jsonAltNames = new JSONArray();
                        String[] tokens = names.split(",");
                        for(String token : tokens)
                            Commons.jsonPut(jsonAltNames, token);
                        Commons.jsonPut(attrs, "altNames", jsonAltNames);
                    }
                }
                FontFamily fontFamily = font.getFamily();
                if(fontFamily!=null)
                    Commons.jsonPut(attrs, "family", fontFamily.getVal());
                FontPitch fontPitch = font.getPitch();
                if(fontPitch!=null) {
                    STPitch stPitch = fontPitch.getVal();
                    if(stPitch!=STPitch.DEFAULT)
                        Commons.jsonPut(attrs, "pitch", stPitch.toString().toLowerCase());
                }
                FontPanose fontPanose = font.getPanose1();
                if(fontPanose!=null) {
                    byte[] panose = fontPanose.getVal();
                    if(panose.length==10) {
                        JSONArray jsonPanose = new JSONArray();
                        for(byte entry : panose)
                            jsonPanose.put(entry);
                        Commons.jsonPut(attrs, "panose1", jsonPanose);
                    }
                }
                AddInsertFontDescription(font.getName(), attrs);
            }
        }
    }

    private void CreateConditionalTableStyle(JSONObject attrs, String conditionalTableStyle, CTTblPrBase tblPr, TrPr trPr, TcPr tcPr, PPr pPr, RPr rPr)
        throws JSONException {

        final JSONObject tableStyle = new JSONObject();
        Table.createTableProperties(wordprocessingMLPackage, tblPr, null, true, 0, tableStyle);
        Table.createRowProperties(trPr, tableStyle);
        Table.createCellProperties(tcPr, null, tableStyle);
        final JSONObject paragraphProperties = Paragraph.createParagraphProperties(pPr);
        if(paragraphProperties!=null) {
        	tableStyle.put("paragraph", paragraphProperties);
        }
        final JSONObject characterProperties = Character.createCharacterProperties(themeFonts, rPr);
        if(characterProperties!=null&&!characterProperties.isEmpty()) {
        	tableStyle.put("character", characterProperties);
        }
        if(!tableStyle.isEmpty()) {
            attrs.put(conditionalTableStyle, tableStyle);
        }
    }

    public void CreateStyleOperations()
        throws JSONException {

        final StyleDefinitionsPart styleDefinitionsPart = wordprocessingMLPackage.getMainDocumentPart().getStyleDefinitionsPart();
        if(styleDefinitionsPart!=null) {
            final Styles styles = styleDefinitionsPart.getJaxbElement();
            final List< Style > lStyle = styles.getStyle();
            for(Style style : lStyle) {
                String styleId = null;
                String styleName = null;
                String type = style.getType();
                Name name = style.getName();
                if(name!=null)
                    styleName = name.getVal();
                styleId = style.getStyleId();
                if(type!=null&&styleId!=null&&styleId.length()>0) {
                    String parentId = null;
                    Style.BasedOn basedOn = style.getBasedOn();
                    if (basedOn!=null)
                        parentId = basedOn.getVal();

                    Integer uipriority = null;
                    Boolean hidden = null;
                    Boolean isDefault = style.isDefault();
                    BooleanDefaultTrue defaultTrueHidden = style.getHidden();
                    if (defaultTrueHidden!=null)
                        hidden = defaultTrueHidden.isVal();
                    if(hidden==null||hidden==false)
                    {
                        UiPriority prio = style.getUiPriority();
                        if(prio!=null)
                            uipriority = prio.getVal().intValue();
                    }

                    RPr rPr = null;
                    PPr pPr = null;
                    if(type.equals("character")) {
                        rPr = style.getRPr();
                        JSONObject attrs = new JSONObject();
                        if (rPr!=null) {
                            JSONObject character = Character.createCharacterProperties(themeFonts, rPr);
                            attrs.put("character", character);
                        }
                        operationsArray.put(createInsertStyleSheetOperation("character", styleId, styleName, attrs, parentId, hidden, uipriority, isDefault, false));
                    }
                    else if (type.equals("paragraph")) {
                        rPr = style.getRPr();
                        pPr = style.getPPr();
                        JSONObject attrs = new JSONObject();

                        if(rPr!=null) {
                            JSONObject character = Character.createCharacterProperties(themeFonts, rPr);
                            attrs.put("character", character);
                        }
                        JSONObject paragraph = null;
                        if(pPr!=null) {
                            paragraph = Paragraph.createParagraphProperties(pPr);
                        }
                        Next nextStyle = style.getNext();
                        if(nextStyle!=null) {
                            String nextStyleId = nextStyle.getVal();
                            if(nextStyleId!=null&&nextStyleId.length()>0) {
                                if (paragraph==null) {
                                    paragraph = new JSONObject();
                                }
                                paragraph.put("nextStyleId", nextStyleId);
                            }
                        }
                        if (paragraph!=null) {
                            attrs.put("paragraph", paragraph);
                        }
                        operationsArray.put(createInsertStyleSheetOperation("paragraph", styleId, styleName, attrs, parentId, hidden, uipriority, isDefault, false));
                    }
                    else if (type.equals("table")) {
                        JSONObject attrs = new JSONObject();
                        CreateConditionalTableStyle(attrs, "wholeTable", style.getTblPr(), style.getTrPr(), style.getTcPr(), style.getPPr(), style.getRPr());
                        List<CTTblStylePr> conditionalStyles = style.getTblStylePr();
                        for(CTTblStylePr conditionalStyle : conditionalStyles) {
                            STTblStyleOverrideType conditionalType = conditionalStyle.getType();
                            String attrName = conditionalType.value();
                            if(conditionalTableAttrNames.containsKey(attrName))
                                attrName = conditionalTableAttrNames.get(attrName);
                            CreateConditionalTableStyle(attrs, attrName, conditionalStyle.getTblPr(), conditionalStyle.getTrPr(), conditionalStyle.getTcPr(), conditionalStyle.getPPr(), conditionalStyle.getRPr());
                        }
                        operationsArray.put(createInsertStyleSheetOperation("table", styleId, styleName, attrs, parentId, hidden, uipriority, isDefault, false));
                    }
                }
            }
        }
    }

    private void createOperations(ContentAccessor parent, List<Integer> parentPosition, String target)
        throws JSONException, ParseException, FilterException {

        final Component.RootComponent rootComponent = new Component.RootComponent(parent);
        Component component = rootComponent.getNextChildComponent(null, null);
        while (component!=null) {

        	final Component nextComponent = component.getNextComponent();

        	final Object o = component.getObject();
        	if(component instanceof ParagraphComponent) {
                // in a cell we have to check if the last object within the content is
                // a placeholder paragraph that was added by us to fullfil the ooxml spec
        		if(nextComponent==null&&parent instanceof Tc) {
                    if(((P)o).getRsidP()!=null) {
                        if(((P)o).getRsidP().equals("47110815")&&(((P)o).getContent().size()==0)) {
                            return;
                        }
                    }
	        	}
                CreateParagraphOperations((ParagraphComponent)component, parentPosition, target);
            }
            else if(component instanceof TableComponent) {
                CreateTableOperations((TableComponent)component, parentPosition, target);
            }
        	component = nextComponent;
        }
    }

    private void CreateParagraphOperations(ParagraphComponent paragraphComponent, List<Integer> parentPosition, String target)
        throws JSONException, ParseException, FilterException {

        // insert paragraph and apply paragraph attributes
        final List<Integer> paragraphPosition = new ArrayList<Integer>(parentPosition);
        paragraphPosition.add(paragraphComponent.getComponentNumber());
        AddInsertParagraphOperation(paragraphPosition, target, paragraphComponent.createJSONAttrs(this, new JSONObject(4)));

        // insert components (text and or other drawing objects)
        Component component = paragraphComponent.getNextChildComponent(null, null);
        while(component!=null) {
            if (component instanceof TextComponent) {
                AddInsertTextOperation(paragraphPosition, component.getComponentNumber(), target, (TextComponent)component);
            }
            else {
                final List<Integer> startPosition = new ArrayList<Integer>(paragraphPosition);
                startPosition.add(component.getComponentNumber());
            	if (component instanceof ShapeComponent_Root){
                    CreateDrawingOperations((ShapeComponent_Root)component, target, paragraphPosition);
                }
                else if (component instanceof TabComponent) {
                    AddInsertTabOperation(startPosition, target);
                }
                else if (component instanceof HardBreakComponent) {
                    AddInsertHardBreakOperation(startPosition, target, ((Br)component.getObject()).getType());
                }
                else if (component instanceof FldSimpleComponent) {
                    AddInsertFieldOperation(startPosition, target, ((FldSimpleComponent)component).getInstr(), ((FldSimpleComponent)component).getRepresentation());
                }
                else {
                    AddInsertDrawingOperation(startPosition, target, ShapeType.UNDEFINED, null);
                }
            }
            component = component.getNextComponent();
        }
        component = paragraphComponent.getNextChildComponent(null, null);
        while(component!=null) {
            final JSONObject attrs = component.createJSONAttrs(this, new JSONObject());
            if (!attrs.isEmpty()) {
	            int startComponent = component.getComponentNumber();
	            int endComponent   = component.getNextComponentNumber()-1;
	            List<Integer> startPosition = new ArrayList<Integer>(paragraphPosition);
	            startPosition.add(startComponent);
	            if(startComponent==endComponent) {
	                AddSetAttributesOperation(attrs, startPosition, null, target);
	            }
	            else {
	                List<Integer> endPosition = new ArrayList<Integer>(paragraphPosition);
	                endPosition.add(endComponent);
	                AddSetAttributesOperation(attrs, startPosition, endPosition, target);
	            }
            }
            component = component.getNextComponent();
        }
    }

    private void CreateDrawingOperations(Component component, String target, List<Integer> parentPosition)
    	throws JSONException, ParseException, FilterException {

        final List<Integer> position = new ArrayList<Integer>(parentPosition);
        position.add(component.getComponentNumber());
        AddInsertDrawingOperation(position, target, ((Component.IShapeType)component).getType(),
        	component instanceof ShapeComponent_Root ? null : component.createJSONAttrs(this, new JSONObject()));

        Component childComponent = component.getNextChildComponent(null, null);
        while(childComponent!=null) {
            final Component c = childComponent instanceof MultiComponent ? ((MultiComponent)childComponent).getComponentList().get(0) : childComponent;
	    	if(c instanceof ShapeComponent) {
	    		CreateDrawingOperations(c, target, position);
	    	}
	    	else if(c instanceof ParagraphComponent) {
	  			CreateParagraphOperations((ParagraphComponent)c, position, target);
	    	}
			else if(c instanceof TableComponent) {
				CreateTableOperations((TableComponent)c, position, target);
			}
	    	childComponent = childComponent.getNextComponent();
        }
    }

    private void CreateTableOperations(TableComponent tableComponent, List<Integer> parentPosition, String target)
        throws JSONException, ParseException, FilterException {

        final List<Integer> tablePosition = new ArrayList<Integer>(parentPosition);
        tablePosition.add(tableComponent.getComponentNumber());
        final JSONObject attrs = tableComponent.createJSONAttrs(this, new JSONObject());
        if(AddInsertTableOperation(tablePosition, target, attrs)) {
        	Component rowComponent = tableComponent.getNextChildComponent(null, null);
            while(rowComponent!=null) {
                final List<Integer> rowPosition = new ArrayList<Integer>(tablePosition);
                rowPosition.add(rowComponent.getComponentNumber());
                AddInsertRowsOperation(rowPosition, target, rowComponent.createJSONAttrs(this, new JSONObject()));
                Component cellComponent = rowComponent.getNextChildComponent(null, null);
                while(cellComponent!=null) {
                    final List<Integer> cellPosition = new ArrayList<Integer>(rowPosition);
                    cellPosition.add(cellComponent.getComponentNumber());
                    AddInsertCellsOperation(cellPosition, target, cellComponent.createJSONAttrs(this, new JSONObject()));
                    createOperations((Tc)cellComponent.getNode().getData(), cellPosition, target);
                    cellComponent = cellComponent.getNextComponent();
                }
                rowComponent = rowComponent.getNextComponent();
            }
        }
    }

    public void AddInsertListOperation( long listIdentifier, JSONObject listDefinition )
        throws JSONException {

        final JSONObject insertListObject = new JSONObject();
        insertListObject.put("name", "insertListStyle");
        insertListObject.put("listStyleId", 'L' + String.valueOf(listIdentifier));
        insertListObject.put("listDefinition", listDefinition);
        operationsArray.put(insertListObject);
    }

    /**
     * Puts all list level attributes into the level's JSONObject
     * Is used to load the levels of the abstract numbering as well
     * as for the levels of the LvlOverride
     */
    private void createLevel( JSONObject levelObject, Lvl level, final HashMap<Long, String> picMap )
        throws JSONException {

        Jc jc = level.getLvlJc();//start, end, center, both, ....
        if(jc!=null&&jc.getVal()!=null)
            levelObject.put( "textAlign", jc.getVal().value() );
        Start start = level.getStart();
        if( start != null )
            levelObject.put("listStartValue", start.getVal().longValue());
        NumFmt numFmt = level.getNumFmt();
        // contains the numbering type like
        // decimal, upperRoman, lowerRoman, upperLetter, lowerLetter, ordinal, cardinalText, ordinalText
        // hex, ...
        if( numFmt != null && numFmt.getVal() != null){
            NumberFormat numberFormat = numFmt.getVal();
            levelObject.put("numberFormat", numberFormat.value());
        }
        LvlText lvlText = level.getLvlText();
        if( lvlText != null )
            levelObject.put("levelText", lvlText.getVal() );

        if( level.getLvlRestart() != null ) // determines whether a sub-level restarts on a 'higher' than the parent level
            levelObject.put("listStartValue", level.getLvlRestart().getVal().longValue());
        Lvl.PStyle pstyle = level.getPStyle();
        if( pstyle != null ){
            levelObject.put("paraStyle", pstyle.getVal());
        }
        Lvl.LvlPicBulletId lvlPicBulletId = level.getLvlPicBulletId();
        if( lvlPicBulletId != null ) {
            long bulletId = lvlPicBulletId.getVal().longValue();
            String picURI = picMap.get( new Long( bulletId ));
            levelObject.put( "levelPicBulletUri", picURI );
        }

        Suff suff = level.getSuff();
        if( suff != null){ //number suffix - could be 'tab', 'space' or 'nothing'
            levelObject.put( "labelFollowedBy", suff.getVal().equals("tab") ? "listtab" : suff.getVal());
        }
        PPr paraProperties = level.getPPr();//paragraph properties - indent, tab, ...
        if( paraProperties != null ){
            Ind ind = paraProperties.getInd();
            if( ind != null ){
                BigInteger hanging = ind.getHanging();
                if( hanging != null && hanging.longValue() > 0) //either hanging or firstLine can be set, if hanging is set then firstLine is ignored
                    levelObject.put("indentFirstLine", - Utils.mapTwipTo100THMM(hanging) );
                else{
                    BigInteger firstLine = ind.getFirstLine();
                    if( firstLine != null && firstLine.longValue() > 0 )
                        levelObject.put("indentFirstLine", Utils.mapTwipTo100THMM(firstLine));
                }
                levelObject.put("indentLeft", Utils.mapTwipTo100THMM(ind.getLeft()));
            }
            Tabs tabs = paraProperties.getTabs();
            if( tabs != null ){
                List<CTTabStop> tabList = tabs.getTab();
                if( tabList != null && !tabList.isEmpty()){
                    // there is only one tab stop
                    levelObject.put( "tabStopPosition", Utils.mapTwipTo100THMM(tabList.get(0).getPos()) );
                }
            }
        }
        RPr rPr = level.getRPr();
        if( rPr != null ){
            /*
            <w:rFonts w:ascii="Symbol" w:hAnsi="Symbol" w:hint="default"/><w:color w:val="auto"/>
            */
            RFonts rFonts = rPr.getRFonts();
            if( rFonts != null){
                levelObject.put( "fontName", rFonts.getAscii());
                //rFonts.getHAnsi();
                //rFonts.getHint();
            }
            Color color = rPr.getColor();
            if( color != null ){
                Commons.jsonPut(levelObject, "color", Utils.createColor(color));
            }
        }
    }

    /**
     * OOXML numbering parts contains two related arrays. One is the array of abstract numberings and on is the mapping from
     * numId to abstractNumId.
     * A numId of zero (assigned to a paragraph) switches the numbering off.
     *
     * The list operations create list definitions for a numId rather than the mapping numId->abstractNumId->definition
     *
     * The mapping also allows for overriding certain levels of the abstract numbering.
     *
     */
    public void CreateListOperations()
        throws JSONException {

        NumberingDefinitionsPart numDefPart = wordprocessingMLPackage.getMainDocumentPart().getNumberingDefinitionsPart();
        if( numDefPart == null )
            return;
        Numbering numbering = numDefPart.getJaxbElement();
        // load bullet picture mapping
        HashMap<Long, String> picMap = new HashMap<Long, String>();
        List<NumPicBullet> numPicBulletList = numbering.getNumPicBullet();
        if( numPicBulletList != null ){
            Iterator<NumPicBullet> numPicBulletIter = numPicBulletList.iterator();
            while( numPicBulletIter.hasNext() ){
                NumPicBullet numPicBullet = numPicBulletIter.next();
                Pict pict = numPicBullet.getPict();
                if( pict != null ){
                    //
                    CTShape shape = null;
                    Iterator<Object> anyAndAnyIter = pict.getContent().iterator();
                     while( anyAndAnyIter.hasNext() ){
                         Object element = anyAndAnyIter.next();
                         if( element instanceof JAXBElement ) {
                             if( ((JAXBElement<?>)element).getValue().getClass().getName() == "org.docx4j.vml.CTShape"){
                                 shape = (CTShape)((JAXBElement<?>)element).getValue();
                                 break;
                            }
                         }
                         else if( element instanceof CTShape ) {
                             //
                             shape = (CTShape)element;
                             break;
                         }
                     }
                     if( shape != null ){
                         List<JAXBElement<?>> shapeElements = shape.getEGShapeElements();
                         JAXBElement<?> imageData = Commons.findElement(shapeElements, "imagedata");
                         if( imageData != null){
                             CTImageData ctImageData = (CTImageData)imageData.getValue();
                             String pictRelId = ctImageData.getId();
                             RelationshipsPart relPart = numDefPart.getRelationshipsPart();
                             Relationship rel = numDefPart.getRelationshipsPart().getRelationshipByID(pictRelId);
                             String imageUrl;
                             try {
                                 imageUrl = URIHelper. resolvePartUri(relPart.getSourceURI(), new URI(rel.getTarget())).getPath();
                                 if(imageUrl.charAt(0)=='/') {
                                     imageUrl = imageUrl.substring(1);
                                 }
                                 picMap.put( new Long( numPicBullet.getNumPicBulletId().longValue() ), imageUrl );
                             } catch (URISyntaxException e) {
                                // TODO Auto-generated catch block
                             }
                         }
                     }
                }
            }
        }


        //abstract numbering container
        HashMap<Long, JSONObject> abstractNumMap = new HashMap<Long, JSONObject>();
        //iterator for abstract definitions
        Iterator<AbstractNum> abstractIter = numbering.getAbstractNum().iterator();
        while( abstractIter.hasNext() ){
            AbstractNum abstractNum = abstractIter.next();
            // abstractNum.getMultiLevelType();//hybridMultilevel, multilevel, singleLevel
            long abstractNumId = abstractNum.getAbstractNumId().longValue();
            NumStyleLink styleLink = abstractNum.getNumStyleLink();
            if(styleLink!=null) {
                String numberingStyleId = styleLink.getVal();
                if(numberingStyleId!=null) {
                    // now trying to get the numbering style ...
                    Styles styles = wordprocessingMLPackage.getMainDocumentPart().getStyleDefinitionsPart().getJaxbElement();
                    for(Style style : styles.getStyle()) {
                        if(style.getType().equals("numbering")) {
                            if(style.getStyleId().equals(numberingStyleId)) {
                                PPr paragraphProperties = style.getPPr();
                                if(paragraphProperties==null)
                                    break;
                                NumPr numPr = paragraphProperties.getNumPr();
                                if(numPr==null)
                                    break;
                                NumId numId = numPr.getNumId();
                                if(numId==null||numId.getVal()==null)
                                    break;
                                long id = numId.getVal().longValue();

                                // now trying to get the corresponding Numbering
                                for(Num num:numbering.getNum()) {
                                    if(num.getNumId()!=null&&num.getNumId().longValue()==id) {
                                        AbstractNumId abstractId = num.getAbstractNumId();
                                        if(abstractId==null||abstractId.getVal()==null)
                                            break;
                                        long abstractL = abstractId.getVal().longValue();
                                        // now searching for the correct AbstractNum to replace our
                                        // initial abstract numbering who was just a link
                                        for(AbstractNum an:numbering.getAbstractNum()) {
                                            if(an.getAbstractNumId()!=null&&an.getAbstractNumId().longValue()==abstractL) {
                                                abstractNum = an;
                                                break;
                                            }
                                        }
                                        break;
                                    }
                                }
                            }
                        }
                    }
                }
            }
            final JSONObject listObject = new JSONObject();
            //now create an operation
            Iterator<Lvl> levelIter = abstractNum.getLvl().iterator();
            for( int index = 0; index < 9; ++index ){
                final JSONObject levelObject = new JSONObject();
                if( levelIter.hasNext() ){
                    Lvl level = levelIter.next();
                    createLevel( levelObject, level, picMap );
                }
                listObject.put( "listLevel" + index, levelObject );
            }
            abstractNumMap.put( new Long(abstractNumId), listObject );
        }
        //
        Iterator<Num> numListIter = numbering.getNum().iterator();
        while( numListIter.hasNext() ){
            Num num = numListIter.next();

            JSONObject listObject = abstractNumMap.get(num.getAbstractNumId().getVal().longValue());
            List<LvlOverride> lvlOverride = num.getLvlOverride();

            if( lvlOverride != null && !lvlOverride.isEmpty()){
                listObject = new JSONObject(listObject.toString());
                Iterator<LvlOverride> lvlIter = lvlOverride.iterator();
                while( lvlIter.hasNext() ){
                    LvlOverride override = lvlIter.next();
                    long ilvlOverride = override.getIlvl().longValue();
                    JSONObject levelObject = listObject.getJSONObject("listLevel" + ilvlOverride);
                    StartOverride startOverride = override.getStartOverride();
                    if( startOverride != null )
                        levelObject.put("listStartValue", startOverride.getVal().longValue());
                    Lvl level = override.getLvl();
                    if( level != null ){
                        createLevel( levelObject, level, picMap );
                    }
                }
            }
            AddInsertListOperation( num.getNumId().longValue(), listObject);
        }
    }
}
