/**
 * OPEN-XCHANGE legal information
 *
 * All intellectual property rights in the Software are protected by
 * international copyright laws.
 *
 *
 * In some countries OX, OX Open-Xchange, open xchange and OXtender
 * as well as the corresponding Logos OX Open-Xchange and OX are registered
 * trademarks of the Open-Xchange, Inc. group of companies.
 * The use of the Logos is not covered by the GNU General Public License.
 * Instead, you are allowed to use these Logos according to the terms and
 * conditions of the Creative Commons License, Version 2.5, Attribution,
 * Non-commercial, ShareAlike, and the interpretation of the term
 * Non-commercial applicable to the aforementioned license is published
 * on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 * Please make sure that third-party modules and libraries are used
 * according to their respective licenses.
 *
 * Any modifications to this package must retain all copyright notices
 * of the original copyright holder(s) for the original code used.
 *
 * After any such modifications, the original and derivative code shall remain
 * under the copyright of the copyright holder(s) and/or original author(s)per
 * the Attribution and Assignment Agreement that can be located at
 * http://www.open-xchange.com/EN/developer/. The contributing author shall be
 * given Attribution for the derivative code and a license granting use.
 *
 *  Copyright (C) 2004-2014 Open-Xchange, Inc.
 *  Mail: info@open-xchange.com
 *
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License, Version 2 as published
 *  by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc., 59
 *  Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package com.openxchange.office_communication.tools.persistence;

import java.io.IOException;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Vector;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;

//=============================================================================
public class SimplePersistenceBeanBase
{
	//-------------------------------------------------------------------------
	public static final String SEPARATOR = ".";
	
	//-------------------------------------------------------------------------
	public SimplePersistenceBeanBase ()
		throws Exception
	{}

	//-------------------------------------------------------------------------
	public synchronized void defineSubset (final String sSet)
		throws Exception
	{
		m_sSubset = sSet;
	}

	//-------------------------------------------------------------------------
	/** bind this bean to the persistence layer where it can read/write values from/to
	 * 
	 *  @param	iLayer [IN]
	 *  		the persistence layer.
	 */
	public synchronized void bindToPeristenceLayer (final ISimplePersistenceTransacted iLayer)
		throws Exception
	{
		m_iPersistenceLayer = iLayer;
	}
	
	//-------------------------------------------------------------------------
	public synchronized List< String > list ()
		throws Exception
	{
		final List< String >        lList    = new Vector< String > ();
		final Map< String, Object > lChanges = mem_Changes ();

		final Iterator< String > rChangedKeys = lChanges.keySet().iterator();
		while (rChangedKeys.hasNext())
		{
			final String sFullKey = rChangedKeys.next();
			if ( ! impl_isKeyInCurrentSubset (sFullKey))
				continue;
			
			final String sRelKey = impl_makeKeyRelative (sFullKey);
			lList.add (sRelKey);
		}
		
		final Iterator< String > rPersistentKeys = m_iPersistenceLayer.listKeys().iterator();
		while (rPersistentKeys.hasNext())
		{
			final String sFullKey = rPersistentKeys.next();
			if ( ! impl_isKeyInCurrentSubset (sFullKey))
				continue;
			
			final String sRelKey = impl_makeKeyRelative (sFullKey);
			lList.add (sRelKey);
		}
		
		return lList;
	}
	
	//-------------------------------------------------------------------------
	public synchronized < T extends Serializable > void set (final String sKey  ,
			               					                 final T      aValue)
	    throws Exception
	{
		final Map< String, Object > lChanges = mem_Changes ();
		final String                sFullKey = impl_makeKeyAbsolute (sKey);
		lChanges.put (sFullKey, aValue);
	}

	//-------------------------------------------------------------------------
	@SuppressWarnings("unchecked")
	public synchronized < T extends Serializable > T get (final String sKey)
	    throws Exception
	{
		      T      aValue   = null;
		final String sFullKey = impl_makeKeyAbsolute (sKey);
		
		final Map< String, Object > lChanges = mem_Changes ();
		if (lChanges.containsKey(sFullKey))
			aValue = (T) lChanges.get (sFullKey);
		else
			aValue = (T) m_iPersistenceLayer.get(sFullKey);

		return aValue;
	}

	//-------------------------------------------------------------------------
	public synchronized void flush ()
		throws Exception
	{
		final Map< String, Object > lChanges = mem_Changes ();
		if (lChanges.isEmpty())
			return;
		
		try
		{
			m_iPersistenceLayer.begin();
			
			final Iterator< Entry< String, Object > > rChanges = lChanges.entrySet().iterator();
			while (rChanges.hasNext())
			{
				final Entry< String, Object > rChange  = rChanges.next     ();
				final String                  sKey     = rChange .getKey   ();
				final Object                  aValue   = rChange .getValue ();

				m_iPersistenceLayer.set(sKey, (Serializable)aValue);
			}
			
			m_iPersistenceLayer.commit();
		}
		catch (final Throwable ex)
		{
			m_iPersistenceLayer.rollback();
			throw new IOException (ex);
		}
		
		m_lChanges = null;
	}

	//-------------------------------------------------------------------------
	@SuppressWarnings("unchecked")
	public synchronized <T extends SimplePersistenceBeanBase> T getSubset (final String     sSubset,
																		   final Class< T > aType  )
		throws Exception
	{
		Validate.isTrue(m_lChanges == null, "Cant create sub set if changes exists.");
		
		final SimplePersistenceBeanBase aSubset = aType.newInstance();
		aSubset.bindToPeristenceLayer( m_iPersistenceLayer);
		aSubset.defineSubset         ( sSubset            );
		aSubset.m_lChanges           = m_lChanges          ;
	
		return (T) aSubset;
	}

	//-------------------------------------------------------------------------
	public synchronized String dump ()
		throws Exception
	{
		final StringBuffer sDump = new StringBuffer (256);
		
		sDump.append (super.toString ());
		sDump.append ("\n"             );

		// a) dump 'real persistent' key-value pairs first
		
		sDump.append ("commited values :\n");
		
		final Iterator< String > rKeys = m_iPersistenceLayer.listKeys().iterator();
		while (rKeys.hasNext())
		{
			final String sKey   = rKeys.next();
			if ( ! impl_isKeyInCurrentSubset (sKey))
				continue;
			
			final Object aValue = m_iPersistenceLayer .get (sKey);
			sDump.append ("['"+sKey+"'] = '"+aValue+"'\n");
		}

		// b) dump 'non flushed' changes then

		sDump.append ("actual changes :\n");
		
		final Map< String, Object >               lChanges = mem_Changes ();
		final Iterator< Entry< String, Object > > rChanges = lChanges.entrySet().iterator();
		while (rChanges.hasNext())
		{
			final Entry< String, Object > rChange  = rChanges.next     ();
			final String                  sKey     = rChange .getKey   ();
			final Object                  aValue   = rChange .getValue ();

			if ( ! impl_isKeyInCurrentSubset (sKey))
				continue;

			sDump.append ("['"+sKey+"'] = '"+aValue+"'\n");
		}
		
		return sDump.toString ();
	}
	
	//-------------------------------------------------------------------------
	private String impl_makeKeyAbsolute (final String sRelKey)
		throws Exception
	{
		if (StringUtils.isEmpty(m_sSubset))
			return sRelKey;
		
		final StringBuffer sFullKey = new StringBuffer (256);
		sFullKey.append (m_sSubset);
		sFullKey.append (SEPARATOR);
		sFullKey.append (sRelKey  );
		return sFullKey.toString ();
	}

	//-------------------------------------------------------------------------
	private String impl_makeKeyRelative (final String sAbsKey)
		throws Exception
	{
		if (StringUtils.isEmpty(m_sSubset))
			return sAbsKey;
		
		final String sRelKey = StringUtils.substringAfter(sAbsKey, m_sSubset+SEPARATOR);
		return sRelKey;
	}

	//-------------------------------------------------------------------------
	private boolean impl_isKeyInCurrentSubset (final String sKey)
		throws Exception
	{
		final boolean bIs = (
							 (StringUtils.isEmpty   (m_sSubset          )) ||
							 (StringUtils.startsWith(sKey    , m_sSubset))
							);
		return bIs;
	}
	
	//-------------------------------------------------------------------------
	private Map< String, Object > mem_Changes ()
	    throws Exception
	{
		if (m_lChanges == null)
			m_lChanges = new HashMap< String, Object > ();
		return m_lChanges;
	}

	//-------------------------------------------------------------------------
	private ISimplePersistenceTransacted m_iPersistenceLayer = null;

	//-------------------------------------------------------------------------
	private String m_sSubset = null;
	
	//-------------------------------------------------------------------------
	private Map< String, Object > m_lChanges = null;
}
