/**
 * OPEN-XCHANGE legal information
 *
 * All intellectual property rights in the Software are protected by
 * international copyright laws.
 *
 *
 * In some countries OX, OX Open-Xchange, open xchange and OXtender
 * as well as the corresponding Logos OX Open-Xchange and OX are registered
 * trademarks of the Open-Xchange, Inc. group of companies.
 * The use of the Logos is not covered by the GNU General Public License.
 * Instead, you are allowed to use these Logos according to the terms and
 * conditions of the Creative Commons License, Version 2.5, Attribution,
 * Non-commercial, ShareAlike, and the interpretation of the term
 * Non-commercial applicable to the aforementioned license is published
 * on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 * Please make sure that third-party modules and libraries are used
 * according to their respective licenses.
 *
 * Any modifications to this package must retain all copyright notices
 * of the original copyright holder(s) for the original code used.
 *
 * After any such modifications, the original and derivative code shall remain
 * under the copyright of the copyright holder(s) and/or original author(s)per
 * the Attribution and Assignment Agreement that can be located at
 * http://www.open-xchange.com/EN/developer/. The contributing author shall be
 * given Attribution for the derivative code and a license granting use.
 *
 *  Copyright (C) 2004-2014 Open-Xchange, Inc.
 *  Mail: info@open-xchange.com
 *
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License, Version 2 as published
 *  by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc., 59
 *  Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package com.openxchange.office_communication.cluster_management.core.impl;

import java.util.List;

import org.jolokia.jvmagent.client.AgentLauncher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openxchange.office_communication.cluster_management.core.ClusterAppCmdLineBase;
import com.openxchange.office_communication.cluster_management.core.IClusterApp;
import com.openxchange.office_communication.configuration.configitems.cluster_management.AppDescriptor;
import com.openxchange.office_communication.configuration.configitems.cluster_management.EAppType;
import com.openxchange.office_communication.tools.exec.Executable;
import com.openxchange.office_communication.tools.exec.impl.GlobalPidProcessHandler;

//=============================================================================
public class SimpleApp implements IClusterApp
{
	//-------------------------------------------------------------------------
	private static final Logger LOG = LoggerFactory.getLogger(SimpleApp.class);
	
	//-------------------------------------------------------------------------
	public SimpleApp ()
		throws Exception
	{}

	//-------------------------------------------------------------------------
	public void setUCRID (final String sId)
		throws Exception
	{
		m_sUCRID = sId;
	}
	
	//-------------------------------------------------------------------------
	public String getUCRID ()
		throws Exception
	{
		return m_sUCRID;
	}

	//-------------------------------------------------------------------------
	public void setInstanceId (final String sId)
		throws Exception
	{
		m_sInstanceId = sId;
	}
	
	//-------------------------------------------------------------------------
	public String getInstanceId ()
		throws Exception
	{
		return m_sInstanceId;
	}

	//-------------------------------------------------------------------------
	public void setJolokiaPort (final int nPort)
		throws Exception
	{
		m_nJolokiaPort = nPort;
	}

	//-------------------------------------------------------------------------
	@Override
	public void setDescriptor (final AppDescriptor aDescriptor)
	    throws Exception
	{
		m_aDescriptor = aDescriptor;
	}
	
	//-------------------------------------------------------------------------
	public AppDescriptor getDescriptor ()
	    throws Exception
	{
		return m_aDescriptor;
	}

	//-------------------------------------------------------------------------
	@Override
	public boolean isRunning ()
		throws Exception
	{
		LOG.trace("watchdog : is running ... ?");
		
		if (m_aProcess == null)
		{
			LOG.trace("watchdog : ... no process -> not running");
			return false;
		}
		
		if ( ! m_aProcess.isAlive())
		{
			LOG.trace("watchdog : ... not alive -> not running");
			return false;
		}
	
		LOG.trace("watchdog : ... yes -> its running");
		return true;
	}
	
	//-------------------------------------------------------------------------
	@Override
	public void start ()
		throws Exception
	{
		if (m_aProcess != null)
			return;
		
		Executable aExe = impl_defineExe ();
		aExe.runAsync (null);

		synchronized (this)
		{
			wait (500);
		}
		
		m_aProcess = aExe;
		
		synchronized(this)
		{
			++m_nStartCount;
		}
	}
	
	//-------------------------------------------------------------------------
	@Override
	public void stop ()
		throws Exception
	{
		if (m_aProcess == null)
			return;
		
		m_aProcess.kill();
		m_aProcess = null;
	}

	//-------------------------------------------------------------------------
	public int getStartCount ()
		throws Exception
	{
		return m_nStartCount;
	}

	//-------------------------------------------------------------------------
	@Override
	public String toString ()
	{
		final StringBuffer sString = new StringBuffer (256);
		sString.append (super.toString ());
		sString.append (" [descriptor:"  );
		sString.append (m_aDescriptor    );
		sString.append (" ]"             );
		return sString.toString ();
	}
	
	//-------------------------------------------------------------------------
	public int getPid ()
		throws Exception
	{
		if (m_aProcess == null)
			return GlobalPidProcessHandler.INVALID_PID;
		return m_aProcess.getPid();
	}

	//-------------------------------------------------------------------------
	private Executable impl_defineExe ()
		throws Exception
	{
		      Executable aExe  = null;
		final EAppType   eType = m_aDescriptor.getType();

		if (eType.equals(EAppType.E_SHELL))
			aExe = impl_defineExe4Shell ();
		else
		if (eType.equals(EAppType.E_JAR))
			aExe = impl_defineExe4Jar ();
		else
		if (eType.equals(EAppType.E_JAVA_MAIN))
			aExe = impl_defineExe4JavaMain ();
		else
			throw new UnsupportedOperationException ("No support for app type '"+eType+"' implemented yet.");
		
		return aExe;
	}

	//-------------------------------------------------------------------------
	private Executable impl_defineExe4Shell ()
		throws Exception
	{
		final Executable aExe = new Executable ();
		aExe.setExecutable                   (m_aDescriptor.getExecutable());
		impl_addClusterEnvArgumentsIfNeeded  (aExe, m_aDescriptor          );
		impl_addManagmentEnvArgumentsIfNeeded(aExe, m_aDescriptor          );
		impl_addArguments                    (aExe, m_aDescriptor          );
		return aExe;
	}

	//-------------------------------------------------------------------------
	private Executable impl_defineExe4Jar ()
		throws Exception
	{
		final Executable aExe = new Executable ();
		
		aExe.setExecutable                   ("java"                       );
		impl_enableJmxIfConfigured           (aExe, m_aDescriptor          );
		aExe.addArgument                     ("-jar"                       );
		aExe.addArgument                     (m_aDescriptor.getExecutable());
		impl_addClusterEnvArgumentsIfNeeded  (aExe, m_aDescriptor          );
		impl_addManagmentEnvArgumentsIfNeeded(aExe, m_aDescriptor          );
		impl_addArguments                    (aExe, m_aDescriptor          );
		
		return aExe;
	}

	//-------------------------------------------------------------------------
	private Executable impl_defineExe4JavaMain ()
		throws Exception
	{
		final Executable aExe       = new Executable ();
		final String     sClasspath = System.getProperty("java.class.path");
		
//		System.err.println
//		(
//				CollectionUtils.toString(
//						StringUtils.splitByWholeSeparatorPreserveAllTokens(sClasspath, ":"),
//						'\n')
//		);

		aExe.setExecutable                   ("java"                       );
		impl_enableJmxIfConfigured           (aExe, m_aDescriptor          );
		aExe.addArgument                     ("-classpath"                 );
		aExe.addArgument                     (sClasspath                   );
		aExe.addArgument                     (m_aDescriptor.getExecutable());
		impl_addClusterEnvArgumentsIfNeeded  (aExe, m_aDescriptor          );
		impl_addManagmentEnvArgumentsIfNeeded(aExe, m_aDescriptor          );
		impl_addArguments                    (aExe, m_aDescriptor          );
		
		return aExe;
	}

	//-------------------------------------------------------------------------
	private void impl_enableJmxIfConfigured (final Executable    aExe ,
											 final AppDescriptor aDesc)
		throws Exception
	{
		if ( ! aDesc.isJmxEnabled())
			return;

		final int nJmxPort = aDesc.getJmxPort();
		
		aExe.addArgument("-Dcom.sun.management.jmxremote"                   );
		aExe.addArgument("-Dcom.sun.management.jmxremote.port="+nJmxPort    );
		aExe.addArgument("-Dcom.sun.management.jmxremote.authenticate=false");
		aExe.addArgument("-Dcom.sun.management.jmxremote.ssl=false"         );
	}
	
	//-------------------------------------------------------------------------
	private void impl_addClusterEnvArgumentsIfNeeded (final Executable    aExe ,
											          final AppDescriptor aDesc)
	    throws Exception
	{
		if ( ! aDesc.isClusterEnvAware())
			return;

		aExe.addArgument ("-"+ClusterAppCmdLineBase.OPT_SHORT_UNIQUE_CLUSTER_RESID, m_sUCRID);
	}
	
	//-------------------------------------------------------------------------
	private void impl_addManagmentEnvArgumentsIfNeeded (final Executable    aExe ,
											            final AppDescriptor aDesc)
	    throws Exception
	{
		if (m_nJolokiaPort == AppDescriptor.INVALID_PORT)
			return;
		
		final String sPort = Integer.toString(m_nJolokiaPort);
		aExe.addArgument ("-"+ClusterAppCmdLineBase.OPT_SHORT_MANAGEMENT_PORT, sPort);
	}

	//-------------------------------------------------------------------------
	private void impl_addArguments (final Executable    aExe       ,
								    final AppDescriptor aDescriptor)
	     throws Exception
	{
		final List< String > lArgs = aDescriptor.getArguments();
		for (final String sArg : lArgs)
			aExe.addArgument(sArg);
	}
	
	//-------------------------------------------------------------------------
	private String m_sInstanceId = null;
	
	//-------------------------------------------------------------------------
	private String m_sUCRID = null;

	//-------------------------------------------------------------------------
	private int m_nJolokiaPort = AppDescriptor.INVALID_PORT;

	//-------------------------------------------------------------------------
	private AppDescriptor m_aDescriptor = null;

	//-------------------------------------------------------------------------
	private Executable m_aProcess = null;

	//-------------------------------------------------------------------------
	private int m_nStartCount = 0;
}
