/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office;

import org.json.JSONObject;


public class PreviewData {

    public static final int MINIMAL_MAX_CELLS = 100;
    public static final int MAXIMAL_MAX_CELLS = 1000;

    private JSONObject m_previewOperations;
    private int m_activeSheetIndex = 0;
    private long m_maxMemoryUsage = 0;
    private long m_minFreeMemoryBarrier = 0;

    private int m_maxCells;

    /**
     * Initializes a new {@link PreviewData}.
     */
    public PreviewData() {
        m_maxCells = MINIMAL_MAX_CELLS;
    }

    /**
     * Provides the maximal cell count to be used by the preview process.
     *
     * @return
     *  The maximal number of cells to be used by the preview process.
     */
    public int getMaxCellsCount() {
        return m_maxCells;
    }

    /**
     * Sets the maximal cell count to be used by the preview process.
     *
     * @param newMaxCells
     *  The maximal cell count. A value below MINIMAL_MAX_CELLS is round to
     *  MINIMAL_MAX_CELLS. A value higher than MAXIMAL_MAX_CELLS is round to
     *  MAXIMAL_MAX_CELLS.
     */
    public void setMaxCellsCount(int newMaxCells) {
        m_maxCells = Math.max(Math.min(MAXIMAL_MAX_CELLS, newMaxCells), MINIMAL_MAX_CELLS);
    }

    /**
     * Provides the operations JSON object to be used for the preview.
     *
     * @return
     *  A JSON object containing the operations or null if no operations JSON
     *  object is available.
     */
    public final JSONObject getPreviewOperations() {
        return m_previewOperations;
    }

    /**
     * Sets the operations JSON object to be used for the preview.
     *
     * @param previewOperations
     *  A JSON object containing the property "operations" which is a JSON
     *  array.
     */
    public void setPreviewOperations(final JSONObject previewOperations) {
        m_previewOperations = previewOperations;
    }

    /**
     * Provides the active sheet index;
     *
     * @return
     *  The active sheet index.
     */
    public int getActiveSheetIndex() {
        return m_activeSheetIndex;
    }

    /**
     * Sets the active sheet index.
     *
     * @param index
     *  The active sheet index (must not be negative).
     */
    public void setActiveSheetIndex(final int index) {
        if (index >= 0) {
            m_activeSheetIndex = index;
        }
    }

    /**
     * Provides the maximal memory to be used by the filter. It must throw
     * a FilterException if it detects that the document cannot be loaded
     * with this memory.
     *
     * @return
     *  The maximal memory to be used by the filter.
     */
    public long getMaxMemoryUsage() {
        return m_maxMemoryUsage;
    }

    /**
     * Sets the maximal memory to be used by the filter.
     *
     * @param maxMemoryUsage
     *  The maximal memory to be used by the filter. The filter must throw
     *  a FilterException, if it detects that the document cannot be loaded
     *  with this memory.
     */
    public void setMaxMemoryUsage(final long maxMemoryUsage) {
        if (maxMemoryUsage > 0) {
            this.m_maxMemoryUsage = maxMemoryUsage;
        }
    }

    /**
     * Provides the free memory barrier value, which must never be exceeded.
     *
     * @return
     *  The minimal memory which must always be left free for the system.
     */
    public long getMinFreeMemoryBarrier() {
        return m_minFreeMemoryBarrier;
    }

    /**
     * Sets the free memory barrier value, which must never be exceeded.
     *
     * @param minFreeMemory
     *  The minimal memory which must always be left free for the system.
     */
    public void setMinFreeMemoryBarrier(final long minFreeMemory) {
        if (minFreeMemory > 0) {
            m_minFreeMemoryBarrier = minFreeMemory;
        }
    }
}
