/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.docx.tools;

import org.docx4j.IndexedNode;
import org.docx4j.IndexedNodeList;
import org.docx4j.mce.AlternateContent;
import org.docx4j.wml.Br;
import org.docx4j.wml.CTObject;
import org.docx4j.wml.CTSimpleField;
import org.docx4j.wml.ContentAccessor;
import org.docx4j.wml.Drawing;
import org.docx4j.wml.IText;
import org.docx4j.wml.P;
import org.docx4j.wml.Pict;
import org.docx4j.wml.R;
import org.docx4j.wml.R.Tab;
import org.docx4j.wml.RunDel;
import org.docx4j.wml.RunIns;
import org.docx4j.wml.Tbl;

import com.openexchange.office.ooxml.docx.tools.Component.AlternateContentComponent;
import com.openexchange.office.ooxml.docx.tools.Component.DrawingComponent;
import com.openexchange.office.ooxml.docx.tools.Component.FldSimpleComponent;
import com.openexchange.office.ooxml.docx.tools.Component.HardBreakComponent;
import com.openexchange.office.ooxml.docx.tools.Component.ParagraphComponent;
import com.openexchange.office.ooxml.docx.tools.Component.PictVMLComponent;
import com.openexchange.office.ooxml.docx.tools.Component.TabComponent;
import com.openexchange.office.ooxml.docx.tools.Component.TableComponent;
import com.openexchange.office.ooxml.docx.tools.Component.TextComponent;
import com.openexchange.office.ooxml.docx.tools.Component.TextRun_Base;

public abstract class ComponentContext {

    private ComponentContext parentContext;
    private IndexedNode<Object> node;

    public ComponentContext(ComponentContext p, IndexedNode<Object> n) {
        parentContext = p;
        node = n;
    }
    public ComponentContext getParentContext() {
        return parentContext;
    }
    public void setParentContext(ComponentContext p) {
        parentContext = p;
    }
    public IndexedNode<Object> getNode() {
        return node;
    }
    public void setNode(IndexedNode<Object> n) {
        node = n;
    }

    public abstract Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent);

    public static class TextRunContext extends ComponentContext {

        final private IndexedNode<Object> paragraphNode;

        public TextRunContext(ComponentContext p, IndexedNode<Object> n, IndexedNode<Object> _paragraphNode) {
            super(p, n);
            paragraphNode = _paragraphNode;
        }
        @Override
        public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
            final int index = previousChildContext!=null?previousChildContext.getNode().getIndex()+1:0;
            final int nextComponentNumber = previousChildComponent!=null?previousChildComponent.getNextComponentNumber():0;

            TextRun_Base nextComponent = null;
            final IndexedNode<Object> textRunNode = getNode();
            final IndexedNodeList<Object> nodeList = (IndexedNodeList<Object>)((ContentAccessor)textRunNode.getData()).getContent();
            for(int i=index; nextComponent==null&&i<nodeList.size(); i++) {
                final IndexedNode<Object> childNode = nodeList.getNode(i);
                final Object o = Component.getContentModel(childNode, textRunNode.getData());
                if(o instanceof IText) {
                    if(((IText)o).getValue().length()>0) {
                        nextComponent = new TextComponent(this, childNode, paragraphNode, nextComponentNumber);
                    }
                }
                else if(o instanceof AlternateContent)
                    nextComponent = new AlternateContentComponent(this, childNode, paragraphNode, nextComponentNumber);
                else if(o instanceof Drawing)
                    nextComponent = new DrawingComponent(this, childNode, paragraphNode, nextComponentNumber);
                else if(o instanceof Pict||o instanceof CTObject)
                    nextComponent = new PictVMLComponent(this, childNode, paragraphNode, nextComponentNumber);
                else if(o instanceof Tab)
                    nextComponent = new TabComponent(this, childNode, paragraphNode, nextComponentNumber);
                else if(o instanceof Br)
                    nextComponent = new HardBreakComponent(this, childNode, paragraphNode, nextComponentNumber);
            }
            return nextComponent;
        }
    }

    public static class HyperlinkContext extends ComponentContext {

        final private IndexedNode<Object> paragraphNode;

        public HyperlinkContext(ComponentContext p, IndexedNode<Object> n, IndexedNode<Object> _paragraphNode) {
            super(p, n);
            paragraphNode = _paragraphNode;
        }
        @Override
        public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
            final int index = previousChildContext!=null?previousChildContext.getNode().getIndex()+1:0;

            Component nextComponent = null;
            final IndexedNode<Object> hyperlinkNode = getNode();
            final IndexedNodeList<Object> nodeList = (IndexedNodeList<Object>)((ContentAccessor)hyperlinkNode.getData()).getContent();
            for(int i=index; nextComponent==null&&i<nodeList.size(); i++) {
                final IndexedNode<Object> childNode = nodeList.getNode(i);
                final Object o = Component.getContentModel(childNode, hyperlinkNode.getData());
                if(o instanceof R) {
                    final TextRunContext textRunContext = new TextRunContext(this, childNode, paragraphNode);
                    nextComponent = textRunContext.getNextChildComponent(null, previousChildComponent);
                }
                else if(o instanceof RunIns) {
                    final ComponentContext.RunInsContext runInsContext = new RunInsContext(this, childNode, paragraphNode);
                    nextComponent = runInsContext.getNextChildComponent(null, previousChildComponent);
                }
                else if(o instanceof RunDel) {
                    final ComponentContext.RunDelContext runDelContext = new RunDelContext(this, childNode, paragraphNode);
                    nextComponent = runDelContext.getNextChildComponent(null, previousChildComponent);
                }
            }
            return nextComponent;
        }
    }

    public static class RunInsContext extends ComponentContext {

        final private IndexedNode<Object> paragraphNode;

        public RunInsContext(ComponentContext p, IndexedNode<Object> n, IndexedNode<Object> _paragraphNode) {
            super(p, n);
            paragraphNode = _paragraphNode;
        }
        @Override
        public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
            final int index = previousChildContext!=null?previousChildContext.getNode().getIndex()+1:0;

            Component nextComponent = null;
            final IndexedNode<Object> runInsNode = getNode();
            final IndexedNodeList<Object> nodeList = (IndexedNodeList<Object>)((ContentAccessor)runInsNode.getData()).getContent();
            for(int i=index; nextComponent==null&&i<nodeList.size(); i++) {
                final IndexedNode<Object> childNode = nodeList.getNode(i);
                final Object o = Component.getContentModel(childNode, runInsNode.getData());
                if(o instanceof R) {
                    final TextRunContext textRunContext = new TextRunContext(this, childNode, paragraphNode);
                    nextComponent = textRunContext.getNextChildComponent(null, previousChildComponent);
                }
                else if(o instanceof RunDel) {
                    final RunDelContext runDelContext = new RunDelContext(this, childNode, paragraphNode);
                    nextComponent = runDelContext.getNextChildComponent(null, previousChildComponent);
                }
            }
            return nextComponent;
        }
    }

    public static class RunDelContext extends ComponentContext {

        final private IndexedNode<Object> paragraphNode;

        public RunDelContext(ComponentContext p, IndexedNode<Object> n, IndexedNode<Object> _paragraphNode) {
            super(p, n);
            paragraphNode = _paragraphNode;
        }
        @Override
        public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
            final int index = previousChildContext!=null?previousChildContext.getNode().getIndex()+1:0;

            Component nextComponent = null;
            final IndexedNode<Object> runDelNode = getNode();
            final IndexedNodeList<Object> nodeList = (IndexedNodeList<Object>)((ContentAccessor)runDelNode.getData()).getContent();
            for(int i=index; nextComponent==null&&i<nodeList.size(); i++) {
                final IndexedNode<Object> childNode = nodeList.getNode(i);
                final Object o = Component.getContentModel(childNode, runDelNode.getData());
                if(o instanceof R) {
                    final TextRunContext textRunContext = new TextRunContext(this, childNode, paragraphNode);
                    nextComponent = textRunContext.getNextChildComponent(null, previousChildComponent);
                }
            }
            return nextComponent;
        }
    }

    public static class SdtRootContext extends ComponentContext {

        final private IndexedNode<Object> rootNode;

        public SdtRootContext(ComponentContext p, IndexedNode<Object> n, IndexedNode<Object> _rootNode) {
            super(p, n);
            rootNode = _rootNode;
        }
        @Override
        public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
            final int index = previousChildContext!=null?previousChildContext.getNode().getIndex()+1:0;
            final int nextComponentNumber = previousChildComponent!=null?previousChildComponent.getNextComponentNumber():0;

            Component nextComponent = null;
            final IndexedNode<Object> runDelNode = getNode();
            final IndexedNodeList<Object> nodeList = (IndexedNodeList<Object>)((ContentAccessor)runDelNode.getData()).getContent();
            for(int i=index; nextComponent==null&&i<nodeList.size(); i++) {
                final IndexedNode<Object> childNode = nodeList.getNode(i);
                final Object o = Component.getContentModel(childNode, runDelNode.getData());
                if(o instanceof P) {
                    nextComponent = new ParagraphComponent(this, childNode, rootNode, nextComponentNumber);
                }
                else if(o instanceof Tbl) {
                    nextComponent = new TableComponent(this, childNode, rootNode, nextComponentNumber);
                }
            }
            return nextComponent;
        }
    }

    public static class SdtParagraphContext extends ComponentContext {

        final private IndexedNode<Object> paragraphNode;

        public SdtParagraphContext(ComponentContext p, IndexedNode<Object> n, IndexedNode<Object> _paragraphNode) {
            super(p, n);
            paragraphNode = _paragraphNode;
        }
        @Override
        public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
            final int index = previousChildContext!=null?previousChildContext.getNode().getIndex()+1:0;
            final int nextComponentNumber = previousChildComponent!=null?previousChildComponent.getNextComponentNumber():0;

            Component nextComponent = null;
            final IndexedNode<Object> runDelNode = getNode();
            final IndexedNodeList<Object> nodeList = (IndexedNodeList<Object>)((ContentAccessor)runDelNode.getData()).getContent();
            for(int i=index; nextComponent==null&&i<nodeList.size(); i++) {
                final IndexedNode<Object> childNode = nodeList.getNode(i);
                final Object o = Component.getContentModel(childNode, runDelNode.getData());
                if(o instanceof R) {
                    final ComponentContext.TextRunContext textRunContext = new ComponentContext.TextRunContext(this, childNode, paragraphNode);
                    nextComponent = textRunContext.getNextChildComponent(null, previousChildComponent);
                }
                else if(o instanceof P.Hyperlink) {
                    final ComponentContext.HyperlinkContext hyperlinkContext = new ComponentContext.HyperlinkContext(this, childNode, paragraphNode);
                    nextComponent = hyperlinkContext.getNextChildComponent(null, previousChildComponent);
                }
                else if(o instanceof RunIns) {
                    final ComponentContext.RunInsContext runInsContext = new ComponentContext.RunInsContext(this, childNode, paragraphNode);
                    nextComponent = runInsContext.getNextChildComponent(null, previousChildComponent);
                }
                else if(o instanceof RunDel) {
                    final ComponentContext.RunDelContext runDelContext = new ComponentContext.RunDelContext(this, childNode, paragraphNode);
                    nextComponent = runDelContext.getNextChildComponent(null, previousChildComponent);
                }
                else if(o instanceof CTSimpleField) {
                    nextComponent = new FldSimpleComponent(this, childNode, paragraphNode, nextComponentNumber);
                }
            }
            return nextComponent;
        }
    }
}
