/**
 * **********************************************************************
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER
 *
 * Copyright 2008, 2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Use is subject to license terms.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at http://www.apache.org/licenses/LICENSE-2.0. You can also
 * obtain a copy of the License at http://odftoolkit.org/docs/license.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************************
 */
package org.odftoolkit.odfdom.component;

import static org.odftoolkit.odfdom.component.JsonOperationProducer.OX_DEFAULT_LIST;
import static org.odftoolkit.odfdom.component.OdfFileSaxHandler.PageArea.FOOTER_DEFAULT;
import static org.odftoolkit.odfdom.component.OdfFileSaxHandler.PageArea.FOOTER_EVEN;
import static org.odftoolkit.odfdom.component.OdfFileSaxHandler.PageArea.FOOTER_FIRST;
import static org.odftoolkit.odfdom.component.OdfFileSaxHandler.PageArea.HEADER_DEFAULT;
import static org.odftoolkit.odfdom.component.OdfFileSaxHandler.PageArea.HEADER_EVEN;
import static org.odftoolkit.odfdom.component.OdfFileSaxHandler.PageArea.HEADER_FIRST;
import static org.odftoolkit.odfdom.component.OdfOperationDocument.MAX_SHEETS;
import static org.odftoolkit.odfdom.component.OdfOperationDocument.MAX_TABLE_CELLS;
import static org.odftoolkit.odfdom.component.OdfOperationDocument.MAX_TABLE_COLUMNS;
import static org.odftoolkit.odfdom.component.OdfOperationDocument.MAX_TABLE_ROWS;

import java.io.IOException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.Stack;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.doc.OdfDocument;
import org.odftoolkit.odfdom.doc.OdfSpreadsheetDocument;
import org.odftoolkit.odfdom.doc.OdfTextDocument;
import org.odftoolkit.odfdom.dom.OdfContentDom;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.dom.OdfMetaDom;
import org.odftoolkit.odfdom.dom.OdfSchemaDocument;
import org.odftoolkit.odfdom.dom.OdfSettingsDom;
import org.odftoolkit.odfdom.dom.OdfStylesDom;
import org.odftoolkit.odfdom.dom.element.OdfStylableElement;
import org.odftoolkit.odfdom.dom.element.OdfStyleableShapeElement;
import org.odftoolkit.odfdom.dom.element.draw.DrawConnectorElement;
import org.odftoolkit.odfdom.dom.element.draw.DrawFrameElement;
import org.odftoolkit.odfdom.dom.element.draw.DrawGElement;
import org.odftoolkit.odfdom.dom.element.draw.DrawImageElement;
import org.odftoolkit.odfdom.dom.element.draw.DrawLineElement;
import org.odftoolkit.odfdom.dom.element.draw.DrawMeasureElement;
import org.odftoolkit.odfdom.dom.element.draw.DrawTextBoxElement;
import org.odftoolkit.odfdom.dom.element.office.OfficeSpreadsheetElement;
import org.odftoolkit.odfdom.dom.element.style.StyleFontFaceElement;
import org.odftoolkit.odfdom.dom.element.style.StyleMasterPageElement;
import org.odftoolkit.odfdom.dom.element.style.StyleTableColumnPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleTablePropertiesElement;
import org.odftoolkit.odfdom.dom.element.svg.SvgDescElement;
import org.odftoolkit.odfdom.dom.element.table.TableDatabaseRangeElement;
import org.odftoolkit.odfdom.dom.element.table.TableFilterConditionElement;
import org.odftoolkit.odfdom.dom.element.table.TableFilterElement;
import org.odftoolkit.odfdom.dom.element.table.TableFilterOrElement;
import org.odftoolkit.odfdom.dom.element.table.TableFilterSetItemElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableCellElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableColumnElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableColumnGroupElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableColumnsElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableHeaderColumnsElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableHeaderRowsElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableRowElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableRowGroupElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableRowsElement;
import org.odftoolkit.odfdom.dom.element.text.TextAElement;
import org.odftoolkit.odfdom.dom.element.text.TextHElement;
import org.odftoolkit.odfdom.dom.element.text.TextLineBreakElement;
import org.odftoolkit.odfdom.dom.element.text.TextListElement;
import org.odftoolkit.odfdom.dom.element.text.TextListHeaderElement;
import org.odftoolkit.odfdom.dom.element.text.TextListItemElement;
import org.odftoolkit.odfdom.dom.element.text.TextListStyleElement;
import org.odftoolkit.odfdom.dom.element.text.TextNoteCitationElement;
import org.odftoolkit.odfdom.dom.element.text.TextPElement;
import org.odftoolkit.odfdom.dom.element.text.TextParagraphElementBase;
import org.odftoolkit.odfdom.dom.element.text.TextSElement;
import org.odftoolkit.odfdom.dom.element.text.TextSpanElement;
import org.odftoolkit.odfdom.dom.element.text.TextTabElement;
import org.odftoolkit.odfdom.dom.element.text.TextUserFieldDeclElement;
import org.odftoolkit.odfdom.dom.style.OdfStyleFamily;
import org.odftoolkit.odfdom.dom.style.props.OdfStyleProperty;
import org.odftoolkit.odfdom.incubator.doc.office.OdfOfficeAutomaticStyles;
import org.odftoolkit.odfdom.incubator.doc.office.OdfOfficeStyles;
import org.odftoolkit.odfdom.incubator.doc.style.OdfStyle;
import org.odftoolkit.odfdom.incubator.doc.style.OdfStylePageLayout;
import org.odftoolkit.odfdom.incubator.doc.text.OdfTextListStyle;
import org.odftoolkit.odfdom.pkg.OdfAttribute;
import org.odftoolkit.odfdom.pkg.OdfElement;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.odftoolkit.odfdom.pkg.OdfNamespace;
import org.odftoolkit.odfdom.type.Length;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * @author svante.schubertATgmail.com
 */
public class OdfFileSaxHandler extends DefaultHandler {

    private static final Logger LOG = Logger.getLogger(OdfFileSaxHandler.class.getName());
    private static final String ROW_SPAN = "rowSpan";
    // ToDo: Fix API with its 'ugly' property name
    private static final String COLUMN_SPAN = "gridSpan";
    // ODF value types used for cell content
    private static final String VALUE_TYPE_BOOLEAN = "boolean";
    private static final String VALUE_TYPE_TIME = "time";
    private static final String VALUE_TYPE_DATE = "date";
    private static final String VALUE_TYPE_CURRENCY = "currency";
    private static final String VALUE_TYPE_FLOAT = "float";
    private static final String VALUE_TYPE_PERCENTAGE = "percentage";
    private static final String VALUE_TYPE_STRING = "string";
    private static final String VALUE_TYPE_VOID = "void";
    static final String VISIBLE = "visible";
    static final String COLLAPSE = "collapse";
    static final String APOSTROPHE = "'";
    static final String APOSTROPHE_AND_EQUATION = "'=";
    static final String EMPTY_STRING = "";
    static final String SPACE_CHAR = " ";
    static final String NONE = "none";
    static final String DOT_CHAR = ".";
    static final String DOT = "dot";
    static final String EQUATION = "=";
    static final String HYPHEN_CHAR = "-";
    static final String HYPHEN = "hyphen";
    static final String UNDERSCORE_CHAR = "_";
    static final String UNDERSCORE = "underscore";
    private static final String COLUMN_CELL_DEFAULT_STYLE = "Default";
    private static final String LIBRE_OFFICE_MS_INTEROP_NAMESPACE = "urn:openoffice:names:experimental:ooo-ms-interop:xmlns:field:1.0";
    private static final String LIBRE_OFFICE_MS_INTEROP_TYPE_CHECKBOX = "vnd.oasis.opendocument.field.FORMCHECKBOX";
    private static final String LIBRE_OFFICE_MS_INTEROP_CHECKBOX_UNICODE = "\u25A1";
    private static final Integer ONE = 1;
    // the empty XML file to which nodes will be added
    private OdfFileDom mFileDom;
    private JsonOperationProducer mJsonOperationProducer;
    private Map<String, TextListStyleElement> mAutoListStyles = null;
    private Map<String, TextUserFieldDeclElement> mUserFieldDecls = null;
    /**
     * Represents a stack of TextSpanElement. The text span will be added during
     * startElement(..) with the start address of text span And during
     * endElement(..) the correct span will be returned and the end address can
     * be provided as well.
     *
     *
     */
    private final ArrayDeque<TextSelection> mTextSelectionStack;
    // the context node
    private Node mCurrentNode;
    private final StringBuilder mCharsForElement = new StringBuilder();
    // Text for operations will be collected separately to allow to push not at every new delimiter (e.g. span).
    // In addition will be <text:s/> in the output string exchange to spaces
    private final StringBuilder mCharsForOperation = new StringBuilder();
    // Gatheres the start text position for operations
    private boolean mIsCharsBeginning = true;
    List<Integer> mCharsStartPosition = null;
    private int mComponentDepth = -1;       // as component depth starts with zero
    // the actual component. Linking each other building the tree view of the document
    private Component mCurrentComponent;
    // the postion of the component, being updated for the operations being generated
    private final LinkedList<Integer> mLastComponentPositions = new LinkedList<Integer>();
    /**
     * DOM is created by default, but is in general not needed
     */
    private final boolean domCreationEnabled = true;
//    private final ArrayDeque<ShapeProperties> mShapePropertiesStack;
    // *** TABLE PROPERTIES ***
    // ToDo: Move this table member variables to a special Table related parser/evaluator (Strategy Pattern?)
    // name of the table or spreadsheet
    private String mTableName;
    private TableTableElement mTableElement;
    private List<TableTableColumnElement> mColumns;
    // The relative widths of the columns of a table
    private List<Integer> mColumnRelWidths;

    private int mColumnCount;
    private boolean mIsBigColumnCount = false; // 16384 columns
    // Required as the component table/draw can only be delayed created,
    // After the first child has been parsed.. And should ONLY ONCE created!
    private boolean isTableNew = false;
    Map<String, Object> mTableHardFormatting = null;
    // *** LIST Properties ***
    // @text:start-value is provided to the first paragraph only
    private int mListStartValue = -1;
    private final ArrayDeque<ParagraphListProperties> mListStyleStack;
    // used to track in a text:h/text:p if currently whitespace is being deleted/trimmed
    private final ArrayDeque<WhitespaceStatus> mWhitespaceStatusStack;
    /**
     * Quick cache to get the correct linked list. Key is the xml:id of the
     * first list. The sequence of all continued lists and usability functions
     * are provided by ContinuedList
     */
    private final Map<String, ContinuedList> mLinkedLists = new HashMap<String, ContinuedList>();
    //*** FOR BLOCKING OPERATIONS
    // the number of elements above the current element during parsing.
    // Required to find out if the correct blocking element for the UI was found
    int mElementDepth = 0;
    // The depth of the element responsible of blocking further operations
    int mBlockingElementDepth = 0;
    boolean mNoOperationsAllowed = false;
    // All following blocking modes have different behavior
    boolean mIsBlockingFrame = false; // itself and children are allowed
    boolean mIsIgnoredElement = false; // not even itself allwed
    boolean mIsBlockingShape = false; // itself allowed
    // RunTimeConfiguration given by the caller of the ODF Adapter
    private int mMaxAllowedColumnCount;
    private int mMaxAllowedRowCount;
    private int mMaxAllowedCellCount;
    private int mMaxAllowedSheetCount;
    // temporary during start and end tag of cell the string will be collected
    private StringBuilder mSpreadsheetCellContent;
    private boolean mIsSpreadsheet = true;

    private static final int MAX_REPEATING_EMPTY_CELLS = 3;
    private static final int MIN_REPEATING_CONTENT_CELLS = 2;
    /**
     * LO/AOO/Calligra are applying to Hyperlinks the "Internet_20_link" style,
     * without writing out the dependency into XML. Therefore whenever a
     * Hyperlink existists without character style properties, the reference
     * will be set. see as well OX issue #29658#
     */
    private static final String HYERLINK_DEFAULT_STYLE = "Internet_20_link";
    private boolean mHasHyperlinkTemplateStyle = false;

    /**
     * Properties for the HEADER_DEFAULT and FOOTER_DEFAULT page area. Defining the page layout
     */
    private String mMasterPageStyleName = null;
    private String mPageLayoutName = null;
    /**
     * ODF attribute on pageLayout
     */
    private String mPageStyleUsage = null;
    /** indication of being a first page */
    private boolean mHasNextMasterPage = false;
    /**
     * In the beginning it is only the styleId of the masterPage plus
     * "HeaderDefault" or "FooterDefault"
     */
    private String mContextName = null;
    public static final String CONTEXT_DELIMITER = "_";
    PageArea mPageArea = null;

    /**
     * The parser is always in one of the three areas header, footer and body
     */
    public enum PageArea {
        HEADER_DEFAULT("header_default", "header"),
        HEADER_FIRST("header_first", "header-first"),
        HEADER_EVEN("header_even", "header-left"),
        BODY("body", null),
        FOOTER_DEFAULT("footer_default", "footer"),
        FOOTER_FIRST("footer_first", "footer-first"),
        FOOTER_EVEN("footer_even", "footer-left");

        private String areaName;
        private String localName;

        private PageArea(String areaName, String localName) {
            this.areaName = areaName;
            this.localName = localName;
        }

        public String getPageAreaName(){
            return areaName;
        }

        /** @return the local name of the XML element */
        public String getLocalName(){
            return localName;
        }
    };

    /**
     * "footer_default_" "footer_even_" "footer_first_" "header_default_"
     * "header_even_" "header_first_"
     */
    /**
     * The document might be of different types
     */
    enum DOCUMENT_TYPE {

        TEXT, SPREADSHEET, OTHER
    }
    DOCUMENT_TYPE mDocumentType = null;

    /**
     * Required as the order of linked-list is important! All xml:ids of a
     * connected/linked lists are put into a single list. This collection is
     * used to get the correct reference to the xml:id of the preceding list and
     * have to be updated, when linked lists are created, deleted or moved. Only
     * the text:continue-list of a new list will be evaluated
     */
    class ContinuedList {

        private String mListId;
        private List<String> mSortedIds = null;

        public ContinuedList(String precedingListId, String currentListId) {
            if (precedingListId != null && !precedingListId.isEmpty()) {
                mListId = precedingListId;
            } else {
                if (currentListId != null && !currentListId.isEmpty()) {
                    mListId = currentListId;
                }
            }
            mSortedIds = new LinkedList<String>();
        }

        public void add(String listId) {
            mSortedIds.add(listId);
        }

        public List<String> getListIds() {
            return mSortedIds;
        }

        public String getListId() {
            return mListId;
        }
    }

    /**
     * Checks if the preceding list is already part of a continued list,
     * otherwise creates a new continued list and adds both ids to it
     */
    ContinuedList newContinuedList(String precedingListId, String currentListId) {
        ContinuedList continuedList;
        if (!mLinkedLists.containsKey(precedingListId)) {
            continuedList = new ContinuedList(precedingListId, currentListId);
            continuedList.add(precedingListId);
            mLinkedLists.put(precedingListId, continuedList);
        } else {
            continuedList = mLinkedLists.get(precedingListId);
        }
        if (currentListId != null && !currentListId.isEmpty()) {
            continuedList.add(currentListId);
            mLinkedLists.put(currentListId, continuedList);
        }
        return continuedList;
    }

    /**
     * Checks if the preceding list is already part of a continued list,
     * otherwise creates a new continued list and adds both id to it
     */
    ContinuedList newContinuedList(String currentListId) {
        ContinuedList continuedList = null;
        if (currentListId != null && !currentListId.isEmpty()) {
            if (!mLinkedLists.containsKey(currentListId)) {
                continuedList = new ContinuedList(null, currentListId);
                mLinkedLists.put(currentListId, continuedList);
            } else {
                continuedList = mLinkedLists.get(currentListId);
            }
        }
        return continuedList;
    }
    enum ShapeType{
        NormalShape,
        ImageShape,
        GroupShape
    };

    /**
     * The status of frames
     */
    class ShapeProperties extends CachedComponent {

        private static final long serialVersionUID = 1L;
        // *** FRAME PROPERTIES ***
        // Required as the component draw frame can only be delayed created,
        // After the first child has been parsed.. And should ONLY ONCE created!
        DrawFrameElement mDrawFrameElement = null;
        List<Integer> mShapePosition = null;
        Map<String, Object> mShapeHardFormatations = null;
        List<OdfElement> mFrameChildren = null;
        int mFrameChildrenNumber = 0;
        boolean mIsImageFrame = false;
        boolean mIsGroup = false;
        Integer mVertOffsetMin = null;
        Integer mHoriOffsetMin = null;
        Integer mVertOffsetMax = null;
        Integer mHoriOffsetMax = null;

        public DrawFrameElement getDrawFrameElement() {
            return mDrawFrameElement;
        }

//        private ShapeProperties() {
//        }

        public ShapeProperties(List<Integer> start, Map<String, Object> hardFormatations) {
            // Maps are being reused, for upcoming components, therefore the collections have to be cloned
            mShapePosition = new LinkedList<Integer>(start);
            if (hardFormatations != null) {
                mShapeHardFormatations = new HashMap<String, Object>();
                mShapeHardFormatations.putAll(hardFormatations);
                JSONObject originalDrawingProps = (JSONObject) hardFormatations.get("drawing");
                // Unfortunately the JSON lib being used, does not support deep cloning
                JSONObject newDrawingProps = new JSONObject();
                if (originalDrawingProps != null) {
                    // copying hard
                    for (String key : originalDrawingProps.keySet()) {
                        try {
                            newDrawingProps.put(key, originalDrawingProps.get(key));
                        } catch (JSONException ex) {
                            Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                        }
                    }
                    mShapeHardFormatations.put("drawing", newDrawingProps);
                }
                JSONObject originalImageProps = (JSONObject) hardFormatations.get("image");
                // Unfortunately the JSON lib being used, does not support deep cloning
                JSONObject newImageProps = new JSONObject();
                if (originalImageProps != null) {
                    // copying hard
                    for (String key : originalImageProps.keySet()) {
                        try {
                            newImageProps.put(key, originalImageProps.get(key));
                        } catch (JSONException ex) {
                            Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                        }
                    }
                    mShapeHardFormatations.put("image", newImageProps);
                }
            }
        }

        /**
         * If the frame has one or more image elements this method dispatches an
         * operation for the first image
         *
         * @param desc description of the image
         */
        private void createShapeOperation(String desc, ShapeType shapeType, String context) {
            if (desc != null && !desc.isEmpty()) {
                JSONObject drawingProps = (JSONObject) this.mShapeHardFormatations.get("drawing");
                try {
                    drawingProps.put("description", desc);
                } catch (JSONException ex) {
                    Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                }
            }
            CachedComponent parentComponent;
            if(isGroupShape()) {
                try{
                    JSONObject drawingProps = (JSONObject) mShapeHardFormatations.get("drawing");
                    if(mVertOffsetMin != null) {
                        drawingProps.put("anchorVertOffset", mVertOffsetMin);
                        if(mVertOffsetMax != null) {
                            drawingProps.put("height", mVertOffsetMax - mVertOffsetMin);
                        }
                    }
                    if(mHoriOffsetMin != null) {
                        drawingProps.put("anchorHorOffset", mHoriOffsetMin);
                        if(mHoriOffsetMax != null) {
                            drawingProps.put("width", mHoriOffsetMax - mHoriOffsetMin);
                        }
                    }
                } catch (JSONException ex) {
                    Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                }
            }
            if(!mComponentStack.empty() && (parentComponent = mComponentStack.peek()) instanceof ShapeProperties && ((ShapeProperties)parentComponent).isGroupShape()) {
                JSONObject drawingProps = (JSONObject) mShapeHardFormatations.get("drawing");
                ((ShapeProperties)parentComponent).addMemberPosSize( drawingProps.opt("anchorHorOffset"), drawingProps.opt("anchorVertOffset"),
                                                                drawingProps.opt("width"), drawingProps.opt("height"));
            }
            switch(shapeType) {
                case ImageShape:
                    cacheOperation(false, OperationConstants.IMAGE, this.mShapePosition, this.mShapeHardFormatations, context);
                break;
                case NormalShape:
                {
                    if(mShapeHardFormatations.get("styleId") == null){
                        Object lineObject = mShapeHardFormatations.get("line");
                        // shapes require a default line setting
                        try {
                            if( lineObject == null )  {
                                lineObject = new JSONObject();
                                ((JSONObject)lineObject).put("type", "solid");
                                ((JSONObject)lineObject).put("width", 1);
                                ((JSONObject)lineObject).put("style", "solid");
                                mShapeHardFormatations.put("line", lineObject);
                            }
                        } catch (JSONException e) {
                            //no handling required
                        }
                    }
                    cacheOperation(false, OperationConstants.SHAPE, this.mShapePosition, this.mShapeHardFormatations, context);
                }
                break;
                case GroupShape:
                    cacheOperation(false, OperationConstants.GROUP, this.mShapePosition, this.mShapeHardFormatations, context);
                break;
            }
        }

        public void setDrawFrameElement(DrawFrameElement drawFrameElement) {
            mDrawFrameElement = drawFrameElement;
        }

        public Map<String, Object> getShapeHardFormatting() {
            return mShapeHardFormatations;
        }

        public boolean hasImageSibling() {
            return mIsImageFrame;
        }

        public void declareImage() {
            mIsImageFrame = true;
        }
        public void setGroupShape() {
            mIsGroup = true;
        }
        public boolean isGroupShape() {
            return mIsGroup;
        }
        public void addMemberPosSize(Object horiOffset, Object vertOffset, Object width, Object height) {
            if(vertOffset != null) {
                if(mVertOffsetMin == null) {
                    mVertOffsetMin = (Integer)vertOffset;
                } else {
                    mVertOffsetMin = Math.min(mVertOffsetMin, ((Integer)vertOffset).intValue());
                }
            }
            if(horiOffset != null) {
                if(mHoriOffsetMin == null) {
                    mHoriOffsetMin = (Integer)horiOffset;
                } else {
                    mHoriOffsetMin = Math.min(mHoriOffsetMin, ((Integer)horiOffset).intValue());
                }
            }
            if(width != null ){
                Integer maxHori = (Integer)width;
                if(horiOffset != null){
                    maxHori += (Integer)horiOffset;
                }
                if(mHoriOffsetMax == null) {
                    mHoriOffsetMax = maxHori;
                } else {
                    mHoriOffsetMax = Math.max(mHoriOffsetMax, maxHori);
                }
            }
            if(height != null ){
                Integer maxVert = (Integer)height;
                if(vertOffset != null){
                    maxVert += (Integer)vertOffset;
                }
                if(mVertOffsetMax == null) {
                    mVertOffsetMax = maxVert;
                } else {
                    mVertOffsetMax = Math.max(mVertOffsetMax, maxVert);
                }
            }
        }
        /**
         * In a frame there might be multiple objects, but only the first
         * applicable is shown. For instance, there are replacement images after
         * an OLE object *
         */
        public int incrementChildNumber() {
            return ++mFrameChildrenNumber;
        }

        public int decrementChildNumber() {
            return --mFrameChildrenNumber;
        }

        public int getChildNumber() {
            return mFrameChildrenNumber;
        }
        public Node mOwnNode = null;
        public String mDescription = null;
        public String mContext = null;
    }


    /**
     * The status of the list style
     */
    class ParagraphListProperties {

        // on each list
        String mStyleName = null;
        // on each list item
        String mStyleOverride = null;
        boolean mShowListLabel = false;
        boolean mIsListStart = false;
        String mListId;
        String mListXmlId;
        String mListItemXmlId;

        public String getListId() {
            return mListId;
        }

        public void setListId(String listId) {
            mListId = listId;
        }

        public String getListXmlId() {
            return mListXmlId;
        }

        public void setListXmlId(String listXmlId) {
            mListXmlId = listXmlId;
        }

        public String getListItemXmlId() {
            return mListItemXmlId;
        }

        public void setListItemXmlId(String listItemXmlId) {
            mListItemXmlId = listItemXmlId;
        }

        public void setListStart(boolean isListStart) {
            mIsListStart = isListStart;
        }

        public ParagraphListProperties() {
        }

        public void showListLabel(boolean showListLabel) {
            mShowListLabel = showListLabel;
        }

        public boolean hasListLabel() {
            return mShowListLabel;
        }

        public boolean isListStart() {
            return mIsListStart;
        }

        /**
         * Overrides the list style given by a <text:list> element.
         *
         * @param styleName the new list style, or null to unset a previous
         * override
         */
        public void overrideListStyle(String styleName) {
            mStyleOverride = styleName;
        }

        void setListStyleName(String styleName) {
            mStyleName = styleName;
        }

        String getListStyleName() {
            String styleName = null;
            if (mStyleOverride != null && !mStyleOverride.isEmpty()) {
                styleName = mStyleOverride;
            } else if (mStyleName != null && !mStyleName.isEmpty()) {
                styleName = mStyleName;
            }
            return styleName;
        }
    }

    /**
     * The whitespace status of a text container (ie. paragraph or heading).
     * Required for whitespace handling
     */
    class WhitespaceStatus {

        WhitespaceStatus(boolean isParagraphIgnored, int depth) {
            mDepth = depth;
//            mIsParagraphIgnored = isParagraphIgnored;
        }
        int mDepth = -1;

        public int getParagraphDepth() {
            return mDepth;
        }
        boolean mOnlyWhiteSpaceSoFar = true;
        int mFirstSpaceCharPosition = -1;

        public boolean hasOnlyWhiteSpace() {
            return mOnlyWhiteSpaceSoFar;
        }

        public void setOnlyWhiteSpace(boolean onlyWhiteSpace) {
            mOnlyWhiteSpaceSoFar = onlyWhiteSpace;
        }

        /**
         * During parsing the first character of space siblings. -1 if there is
         * no space sibling
         */
        public int getFirstSpaceCharPosition() {
            return mFirstSpaceCharPosition;
        }

        /**
         * During parsing the first character of space siblings. -1 if there is
         * no space sibling
         */
        public void setFirstSpaceCharPosition(int currentSpaceCharPosition) {
            mFirstSpaceCharPosition = currentSpaceCharPosition;
        }

        /**
         * @return true if the previous character was a white space character
         */
        public boolean hasSpaceBefore() {
            return mFirstSpaceCharPosition > -1;
        }
    }
    OdfSchemaDocument mSchemaDoc = null;

    // Candidate Component Mode
    // Some components consist of multiple XML elements.
    // Even some ODF components start with the same
    // 2DO - DRAGON BOOK - Parser Look-ahead funzt net bei SAX? ;)
    //private boolean isCandidateComponentMode = true;
    public OdfFileSaxHandler(Node rootNode) throws SAXException {
        // Initialize starting DOM node
        if (rootNode instanceof OdfFileDom) {
            mFileDom = (OdfFileDom) rootNode;
        } else {
            mFileDom = (OdfFileDom) rootNode.getOwnerDocument();
        }
        mCurrentNode = rootNode;

        // *** COMPONENT HANDLING ***
        // Initialize starting Component
        // Make the root of component tree (to be created) accessible via the ODF schema document
        mSchemaDoc = (OdfSchemaDocument) mFileDom.getDocument();
        if (mSchemaDoc != null) {
            if (mFileDom instanceof OdfContentDom) {
                mSchemaDoc.setContentDom((OdfContentDom) mFileDom);
            } else if (mFileDom instanceof OdfStylesDom) {
                mSchemaDoc.setStylesDom((OdfStylesDom) mFileDom);
            } else if (mFileDom instanceof OdfMetaDom) {
                mSchemaDoc.setMetaDom((OdfMetaDom) mFileDom);
            } else if (mFileDom instanceof OdfSettingsDom) {
                mSchemaDoc.setSettingsDom((OdfSettingsDom) mFileDom);
            }
        }

        // The current component is the root component
        mCurrentComponent = null;

        // Getting Configuration
        Map configuration = mSchemaDoc.getPackage().getRunTimeConfiguration();
        if (mSchemaDoc instanceof OdfTextDocument) {
            mMaxAllowedColumnCount = 15; // Test default for text documents
            mMaxAllowedRowCount = 1500;
            mMaxAllowedCellCount = 1500;
            mIsSpreadsheet = false;
            mDocumentType = DOCUMENT_TYPE.TEXT;
            mMaxAllowedSheetCount = 200;
        } else if (mSchemaDoc instanceof OdfSpreadsheetDocument) {
            mMaxAllowedColumnCount = Integer.MAX_VALUE; // Test default for spreadsheet documents
            mMaxAllowedRowCount = Integer.MAX_VALUE;
            mMaxAllowedCellCount = Integer.MAX_VALUE;
            mIsSpreadsheet = true;
            mMaxAllowedSheetCount = 200;
        } else {
            mDocumentType = DOCUMENT_TYPE.OTHER;
        }
        if (configuration != null ) {
            if (!mIsSpreadsheet) {
                if (configuration.containsKey(MAX_TABLE_COLUMNS)) {
                    mMaxAllowedColumnCount = (Integer) configuration.get(MAX_TABLE_COLUMNS);
                }
                if (configuration.containsKey(MAX_TABLE_ROWS)) {
                    mMaxAllowedRowCount = (Integer) configuration.get(MAX_TABLE_ROWS);
                }
                if (configuration.containsKey(MAX_TABLE_CELLS)) {
                    mMaxAllowedCellCount = (Integer) configuration.get(MAX_TABLE_CELLS);
                }
            } else {
                if (configuration.containsKey(MAX_SHEETS)) {
                    mMaxAllowedSheetCount = (Integer) configuration.get(MAX_SHEETS);
                }
            }
        }
        LOG.log(Level.FINEST, "mMaxTableColumnCount{0}", mMaxAllowedColumnCount);
        LOG.log(Level.FINEST, "mMaxTableRowCount{0}", mMaxAllowedRowCount);
        LOG.log(Level.FINEST, "mMaxTableCellCount{0}", mMaxAllowedCellCount);

        // Make the Operation Queue to be created accessible via the Schema Document
        mJsonOperationProducer = mSchemaDoc.getJsonOperationQueue();
        if (mJsonOperationProducer == null) {
            // temporary initated here as all the tests are not using the OperationTextDocument
            mJsonOperationProducer = new JsonOperationProducer();
            mSchemaDoc.setJsonOperationQueue(mJsonOperationProducer);
        }

        mAutoListStyles = new HashMap<String, TextListStyleElement>();
        mUserFieldDecls = new HashMap<String, TextUserFieldDeclElement>();
        // initalize all template styles
        OdfStylesDom stylesDom;
        // the content will be parsed after the styles, therefore all style properties are available at calling time
        if (mFileDom instanceof OdfContentDom) {
            Integer defaultTabStopWidth = null;
            JSONObject defaultPageStyles = null;
            stylesDom = mSchemaDoc.getStylesDom();
            // reset the position context used for header/footer
            mContextName = null;
            OdfOfficeStyles officeStyles = stylesDom.getOfficeStyles();
            if (officeStyles != null) {
                // check if the default hyperlinkstyle do exist (more info in test for OX issue #29658#)
                mHasHyperlinkTemplateStyle = officeStyles.getStyle(HYERLINK_DEFAULT_STYLE, OdfStyleFamily.Text) != null;

                List<OdfStyle> paragraphStyles = mapIterableToList(officeStyles.getStylesForFamily(OdfStyleFamily.Paragraph));
                // The sort is for testing purpose to receive across different JDK an equal result
                Collections.sort(paragraphStyles, new AlphanumComparator());
                Integer _defaultTabStopWidth = null;
                for (OdfStyle style : paragraphStyles) {
                    // defaulTableWidth is part of the paragraph default style (optional)
                    _defaultTabStopWidth = mJsonOperationProducer.triggerStyleHierarchyOps(officeStyles, OdfStyleFamily.Paragraph, style);
                    if (_defaultTabStopWidth != null) {
                        defaultTabStopWidth = _defaultTabStopWidth;
                    }
                }
                List<OdfStyle> textStyles = mapIterableToList(officeStyles.getStylesForFamily(OdfStyleFamily.Text));
                // The sort is for testing purpose to receive across different JDK an equal result
                Collections.sort(textStyles, new AlphanumComparator());
                for (OdfStyle style : textStyles) {
                    mJsonOperationProducer.triggerStyleHierarchyOps(officeStyles, OdfStyleFamily.Text, style);
                }
                List<OdfStyle> graphicStyles = mapIterableToList(officeStyles.getStylesForFamily(OdfStyleFamily.Graphic));
                Collections.sort(graphicStyles, new AlphanumComparator());
                for (OdfStyle style : officeStyles.getStylesForFamily(OdfStyleFamily.Graphic)) {
                    mJsonOperationProducer.triggerStyleHierarchyOps(officeStyles, OdfStyleFamily.Graphic, style);
                }
                //always generate graphic default style
                mJsonOperationProducer.triggerDefaultStyleOp(OdfStyleFamily.Graphic, officeStyles.getDefaultStyle(OdfStyleFamily.Graphic));

//			for(OdfStyle style : officeStyles.getStylesForFamily(OdfStyleFamily.Table)){
//				mJsonOperationProducer.triggerStyleHierarchyOps(officeStyles, OdfStyleFamily.Table, style);
//			}
//			for(OdfStyle style : officeStyles.getStylesForFamily(OdfStyleFamily.TableRow)){
//				mJsonOperationProducer.triggerStyleHierarchyOps(officeStyles, OdfStyleFamily.TableRow, style);
//			}
//			for(OdfStyle style : officeStyles.getStylesForFamily(OdfStyleFamily.TableColumn)){
//				mJsonOperationProducer.triggerStyleHierarchyOps(officeStyles, OdfStyleFamily.TableColumn, style);
//			}
                for (OdfStyle style : officeStyles.getStylesForFamily(OdfStyleFamily.TableCell)) {
                    mJsonOperationProducer.triggerStyleHierarchyOps(officeStyles, OdfStyleFamily.TableCell, style);
//				mJsonOperationProducer.triggerStyleHierarchyOps(officeStyles, OdfStyleFamily.TableCell, (OdfStyleBase) officeStyles.getDefaultStyle(OdfStyleFamily.TableCell));
                }
//			for(OdfStyle style : officeStyles.getStylesForFamily(OdfStyleFamily.Section)){
//				mJsonOperationProducer.triggerStyleHierarchyOps(officeStyles, OdfStyleFamily.Section, style);
//			}
//			for(OdfStyle style : officeStyles.getStylesForFamily(OdfStyleFamily.List)){
//				mJsonOperationProducer.triggerStyleHierarchyOps(officeStyles, OdfStyleFamily.List, style);
//			}
                List<OdfTextListStyle> listStyles = mapIterableToList(officeStyles.getListStyles());
                Collections.sort(listStyles, new AlphanumComparator());
                for (OdfTextListStyle listStyle : listStyles) {
                    mJsonOperationProducer.addListStyle(listStyle);
                }

                // maps page properties, but returns the default page properties
                defaultPageStyles = mJsonOperationProducer.insertPageProperties(stylesDom);
                // dispatches default document attributes
                mJsonOperationProducer.insertDocumentProperties(stylesDom, defaultTabStopWidth, defaultPageStyles);
            } else {
                mJsonOperationProducer.insertDocumentProperties(stylesDom, null, null);
            }
        }

        // Stack to remember/track the nested delimiters not being componenets (spans) open-up by SAX events
        mTextSelectionStack = new ArrayDeque<TextSelection>();
        mListStyleStack = new ArrayDeque<ParagraphListProperties>();
//        mShapePropertiesStack = new ArrayDeque<ShapeProperties>();
        mWhitespaceStatusStack = new ArrayDeque<WhitespaceStatus>();
    }

    /**
     * Maps an iterable object to a list.
     */
    private static <T> List<T> mapIterableToList(Iterable<T> iterable) {
        List<T> copy = null;
        if (!(iterable instanceof List)) {
            Iterator iter = iterable.iterator();
            copy = new ArrayList<T>();
            while (iter.hasNext()) {
                copy.add((T) iter.next());
            }
        } else {
            copy = (List) iterable;
        }
        return copy;
    }

    @Override
    public void startDocument() throws SAXException {
    }

    @Override
    public void endDocument() throws SAXException {
    }

    /**
     * There are areas that are not allowed to addChild further components
     * beyond. All further operations have to be blocked, but the creation of
     * the DOM tree must not be disturbed.
     */
    private boolean isBlockedSubTree() {
        return mNoOperationsAllowed;
    }

    /**
     * There are areas that are not allowed to addChild further components
     * beyond. All further operations have to be blocked, but the creation of
     * the DOM tree must not be disturbed.
     */
    private boolean checkEndOfBlockedSubTree(String uri, String localName) {

        boolean isBlocked = mNoOperationsAllowed;
        if (mNoOperationsAllowed) {
            isBlocked = isBlockedSubTree(uri, localName, false);
        }
        mElementDepth--;
        return isBlocked;
    }

    private boolean checkStartOfBlockedSubTree(String uri, String localName) {
        mElementDepth++;
        boolean isBlocked = mNoOperationsAllowed;
        if (!mNoOperationsAllowed) {
            isBlocked = isBlockedSubTree(uri, localName, true);
        } else if (mIsBlockingFrame) {
            if (mBlockingElementDepth == mElementDepth - 1 && !localName.equals("table")) {
                isBlocked = false;
            } else {
                isBlocked = true;
            }
        }
        return isBlocked;
    }

    //ToDo: Differentiate  if there is a shapeBlock, ImageBlock or ParagraphBlock
    private boolean isBlockedSubTree(String uri, String localName, boolean isStart) {
        // within a paragraph within a paragraph
        boolean isBlocked = mNoOperationsAllowed;
        boolean isMasterPage = uri != null && uri.equals(StyleMasterPageElement.ELEMENT_NAME.getUri()) && localName.equals(StyleMasterPageElement.ELEMENT_NAME.getLocalName());
        if (isStart) {
            // if it is a second text component (ie. text:p or text:h element)
            if (/*!mWhitespaceStatusStack.isEmpty() && Component.isTextComponentRoot(uri, localName) || */OdfElement.isIgnoredElement(uri, localName)
                || ((isMasterPage || Component.isHeaderRoot(uri, localName) || Component.isFooterRoot(uri, localName)) && mDocumentType != DOCUMENT_TYPE.TEXT)) {
                isBlocked = true;
                mNoOperationsAllowed = true;
                mIsIgnoredElement = true;
                mBlockingElementDepth = mElementDepth;
                // if it is a <draw:frame>
            }
        } else { // if this is the closing event of an element
            if (mNoOperationsAllowed) {
                if (mBlockingElementDepth == mElementDepth) {
                    if (mIsIgnoredElement && (/*!mWhitespaceStatusStack.isEmpty() && Component.isTextComponentRoot(uri, localName) || */OdfElement.isIgnoredElement(uri, localName))
                        || ((isMasterPage || Component.isHeaderRoot(uri, localName) || Component.isFooterRoot(uri, localName)) && mDocumentType != DOCUMENT_TYPE.TEXT)) {
                        mIsIgnoredElement = false;
                        mBlockingElementDepth = -1;
                        mNoOperationsAllowed = false;
                        isBlocked = true;
                        // if it is a <draw:frame>
                    }
                } else if (mIsBlockingFrame && mBlockingElementDepth == mElementDepth - 1 && !localName.equals("table")) {
                    isBlocked = false;
                }
            } else { // closing will never enabled a blocking
                if (mIsIgnoredElement || mIsBlockingShape) {
                    // close this element, but afterwards
                    mNoOperationsAllowed = true;
                    isBlocked = false;
                } else if (mIsBlockingFrame) {
                    if (mBlockingElementDepth == mElementDepth - 1 && !localName.equals("table")) {
                        isBlocked = false;
                    } else {
                        isBlocked = true;
                    }
                }

            }
        }
        return isBlocked;
    }

    @Override
    public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException {
        flushTextAtStart(uri, localName, qName);
        // if there is a specilized handler on the stack, dispatch the event
        OdfElement element = null;
        // ToDo: Should be able to create operations without creating the DOM Tree
        // ToDo: Are there use-cases that text:s still resides in the DOM? if(isWhiteSpaceElement) ??
        // If Paragraph is not being edited, it will be saved as it is..

        if (domCreationEnabled) {
            if (uri.equals(EMPTY_STRING) || qName.equals(EMPTY_STRING)) {
                element = mFileDom.createElement(localName);
            } else {
                element = mFileDom.createElementNS(uri, qName);
            }
            addAttributes(element, attributes);
        }
        //if it is the last page bound object then move all the nodes to a temporary location
        if(mComponentDepth < 0 && m_cachedPageShapes.size() > 0 && (localName.equals("p") || localName.equals("h") || localName.equals("table"))) {
            //move nodes
            Node bodyNode = mCurrentNode.getParentNode();
            Iterator<ShapeProperties> it = m_cachedPageShapes.iterator();
            while(it.hasNext()){
                ShapeProperties component = it.next();
                bodyNode.insertBefore(component.mOwnNode, bodyNode.getFirstChild());
            }
            mLastComponentPositions.clear();
        }
        // Font declarations are before the component
        if (element instanceof StyleFontFaceElement) {
            String fontName = element.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "name");
            if (fontName != null && !fontName.isEmpty()) {
                Set fontNames = ((OdfDocument) mSchemaDoc).getFontNames();
                if (!fontNames.contains(fontName)) {
                    mJsonOperationProducer.insertFontDescription(fontName, null, element.getAttributeNS(OdfDocumentNamespace.SVG.getUri(), "font-family"), element.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "font-family-generic"), element.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "font-pitch"), element.getAttributeNS(OdfDocumentNamespace.SVG.getUri(), "panose-1"));
                    fontNames.add(fontName);
                }
            }
        }
        if (element instanceof TextListStyleElement) {
            // We need the reference for later gettin the list styles
            TextListStyleElement listStyle = (TextListStyleElement) element;
            String styleName = listStyle.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "name");
            if (styleName != null && !styleName.isEmpty()) {
                mAutoListStyles.put(styleName, listStyle);
            }
        } else if( element instanceof TextUserFieldDeclElement ) {
            TextUserFieldDeclElement fieldDecl = (TextUserFieldDeclElement)element;
            mUserFieldDecls.put(fieldDecl.getAttribute("text:name"), fieldDecl);
        }
        if (!checkStartOfBlockedSubTree(uri, localName)) {
            if (Component.isComponentRoot(uri, localName)) { // || Component.isCoveredComponentRoot(uri, localName)) {

                // It is not allowed to addChild further components..,
                // within a paragraph within a paragraph
                // ToDo ? -- HashMap with KEY - URL+localname, VALUE - ComponentName
                if (element instanceof TextPElement || element instanceof TextHElement) {
                    // Paragraphs that are not child of a known component should be ignored, otherwise the client gets into trouble with nested paragraphs
                    boolean isNestedParagraph = false;

                    if (!isNestedParagraph) {
                        mComponentDepth++;
                        TextParagraphElementBase p = (TextParagraphElementBase) element;
                        Map<String, Object> hardFormatting = mJsonOperationProducer.getHardStyles(p);
                        if (hardFormatting == null) {
                            hardFormatting = new HashMap<String, Object>();
                        }
                        if (element instanceof TextHElement || !mListStyleStack.isEmpty()) {
                            if (!hardFormatting.containsKey("paragraph")) {
                                // if there are absolute styles, but not the main property set, where the templateStyleId should be placed in
                                hardFormatting.put("paragraph", new JSONObject());
                            }
                            JSONObject paraProps = (JSONObject) hardFormatting.get("paragraph");

                            try {
                                if (!mListStyleStack.isEmpty()) {
                                    paraProps.put("listLevel", mListStyleStack.size() - 1);
                                    // Only the first paragraph within a list item should show a label!
                                    ParagraphListProperties listProps = mListStyleStack.getLast();
                                    if (listProps.hasListLabel()) {
                                        listProps.showListLabel(Boolean.FALSE);
                                    } else {
                                        paraProps.put("listLabelHidden", Boolean.TRUE);
                                    }
                                    String listId = listProps.getListId();
                                    if (listId != null && !listId.isEmpty()) {
                                        paraProps.put("listId", listId);
                                    }
                                    boolean foundListXmlId = false;
                                    boolean foundListItemXmlId = false;
                                    Iterator<ParagraphListProperties> listPropsIter = mListStyleStack.descendingIterator();
                                    while ((!foundListXmlId || !foundListItemXmlId) && listPropsIter.hasNext()) {
                                        ParagraphListProperties currentListProp = listPropsIter.next();

                                        String listXmlId = currentListProp.getListXmlId();
                                        if (!foundListXmlId && listXmlId != null && !listXmlId.isEmpty()) {
                                            foundListXmlId = true;
                                            paraProps.put("listXmlId", listXmlId);
                                        }
                                        String listItemXmlId = currentListProp.getListItemXmlId();
                                        if (!foundListItemXmlId && listItemXmlId != null && !listItemXmlId.isEmpty()) {
                                            foundListItemXmlId = true;
                                            paraProps.put("listItemXmlId", listItemXmlId);
                                        }
                                    }

                                    if (listProps.isListStart()) {
                                        paraProps.put("listStart", Boolean.TRUE);
                                    }
                                    String listStyleId = mJsonOperationProducer.getListStyle(mListStyleStack, p);
                                    if (listStyleId != null && !listStyleId.isEmpty()) {
                                        mJsonOperationProducer.addListStyle(mSchemaDoc, mAutoListStyles, listStyleId);
                                        paraProps.put("listStyleId", listStyleId);
                                    } else {
                                        paraProps.put("listStyleId", OX_DEFAULT_LIST);
                                    }
                                    if (mListStartValue != -1) {
                                        paraProps.put("listStartValue", mListStartValue);
                                        mListStartValue = -1;
                                    }
                                }
                                // Add heading outline numbering
                                if (element instanceof TextHElement) {
                                    Integer outlineLevel = ((TextHElement) p).getTextOutlineLevelAttribute();
                                    if (outlineLevel != null) {
                                        outlineLevel--;
                                        paraProps.put("outlineLevel", outlineLevel);
                                    }
                                }
                            } catch (JSONException ex) {
                                Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                            }
                        }

                        List<Integer> position = updateComponentPosition();
                        OdfStyle templateStyle = p.getDocumentStyle();
                        String styleId = null;
                        if (templateStyle != null) {
                            styleId = templateStyle.getStyleNameAttribute();
                            if (styleId != null && !styleId.isEmpty()) {
                                hardFormatting.put("styleId", styleId);
                            }
                        }
                        mCurrentComponent = mCurrentComponent.createChildComponent(p);
                        boolean paragraphOpCreated = false;
                        if(!mPageBoundObjectsRelocated && !m_cachedPageShapes.isEmpty()) {
                            //first document paragraph might be inside of a table
                            boolean isFirstDocumentParagraph = mComponentStack.empty() || mComponentStack.peek() instanceof CachedTable;
                            if(isFirstDocumentParagraph && m_cachedPageShapes.size() > 0){
                                cacheOperation(false, OperationConstants.PARAGRAPH, position, hardFormatting, mContextName);
                                paragraphOpCreated = true;
                                Iterator<ShapeProperties> it = m_cachedPageShapes.iterator();
                                int frameIndex = 0;
                                while(it.hasNext()){
                                    ShapeProperties component = it.next();
                                    Component frameComponent = component.getDrawFrameElement().getComponent();
                                    Component frameComponentParent = frameComponent.getParent();
                                    int framePosition = frameComponentParent.indexOf(frameComponent);
                                    frameComponentParent.remove(framePosition);
                                    element.getComponent().addChild(frameIndex, frameComponent);
                                    element.appendChild(component.mOwnNode);
                                    component.mShapePosition.addAll(0, position);
                                    component.createShapeOperation(component.mDescription,
                                            component.hasImageSibling() ? ShapeType.ImageShape: component.isGroupShape() ? ShapeType.GroupShape : ShapeType.NormalShape,
                                            component.mContext);
                                    Iterator<CachedOperation> opIter = component.iterator();
                                    while(opIter.hasNext())
                                    {
                                        CachedOperation op = opIter.next();
                                        List<Integer> start = op.mStart;
                                        if(op.mComponentType.equals(OperationConstants.ATTRIBUTES)) {
                                            List<Integer> end = (List<Integer>)op.mComponentProperties[0];
                                            //TODO: add _real_ position of the current paragraph (could be in a table...)
                                            end.addAll(0, position);
                                        }
                                        //TODO: add _real_ position of the current paragraph (could be in a table...)

                                        start.addAll(0, position);
                                        cacheOperation(false, op.mComponentType, start, op.mHardFormattingProperties, op.mComponentProperties);
                                    }
                                    ++frameIndex;
                                }
                                m_cachedPageShapes.clear();
                            }
                            mPageBoundObjectsRelocated |= isFirstDocumentParagraph;
                        }
                        if(!paragraphOpCreated) {
                            cacheOperation(false, OperationConstants.PARAGRAPH, position, hardFormatting, mContextName);
                        }


                        // For each new paragraph/heading addChild a new context information for their whitespace, required for normalization
                        mWhitespaceStatusStack.add(new WhitespaceStatus(false, mComponentDepth));
                        element.markAsComponentRoot(true);
// ToDo: NEW COMPONENTS - SECTION
//			} else if (element instanceof TextSectionElement) {
//                mJsonOperationProducer.addChild("Section", position);
//                mCurrentComponent = mCurrentComponent.addChild((TextSectionElement) element);
                    } else {
                        // a nested text component without known component in-between
                        // ignore nested paragraph content
                        mWhitespaceStatusStack.add(new WhitespaceStatus(true, mComponentDepth));
                        element.ignoredComponent(true);
                    }
                } else if (element instanceof DrawFrameElement || Component.isShapeElement(uri, localName)) {
                    OdfElement shape = element;
                    Map<String, Object> hardFormatting = null;
                    if (element instanceof OdfStyleableShapeElement) {
                        hardFormatting = mJsonOperationProducer.getHardStyles((OdfStyleableShapeElement) shape);
                    }
                    if (hardFormatting == null || !hardFormatting.containsKey("drawing")) {
                        // if there are absolute styles, but not the main property set, where the templateStyleId should be placed in
                        if (hardFormatting == null) {
                            hardFormatting = new HashMap<String, Object>();
                        }
                        hardFormatting.put("drawing", new JSONObject());
                    }
                    JSONObject drawingProps = (JSONObject) hardFormatting.get("drawing");
                    if (hardFormatting == null || !hardFormatting.containsKey("image")) {
                        // if there are absolute styles, but not the main property set, where the templateStyleId should be placed in
                        if (hardFormatting == null) {
                            hardFormatting = new HashMap<String, Object>();
                        }
                        hardFormatting.put("image", new JSONObject());
                    }
                    JSONObject imageProps = (JSONObject) hardFormatting.get("image");
                    if (shape.hasAttributeNS(OdfDocumentNamespace.DRAW.getUri(), "name")) {
                        try {
                            drawingProps.put("name", shape.getAttributeNS(OdfDocumentNamespace.DRAW.getUri(), "name"));
                        } catch (JSONException ex) {
                            Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                        }
// Might be confusing for the user..
//					} else {
//						if (element instanceof DrawFrameElement) {
//							// if there is no name, the type will be added
//							drawingProps.put("name", "image");
//						}else{
//							// if there is no name, the type will be added
//							drawingProps.put("name", localName);
//						}
                    }
                    int anchorHorOffset = 0;
                    int anchorVertOffset = 0;
                    int width = 0;
                    int height = 0;
                    if (element instanceof DrawLineElement || element instanceof DrawConnectorElement || element instanceof DrawMeasureElement) {
                        if (shape.hasAttributeNS(OdfDocumentNamespace.SVG.getUri(), "y1" ) &&
                             shape.hasAttributeNS(OdfDocumentNamespace.SVG.getUri(), "x1") &&
                             shape.hasAttributeNS(OdfDocumentNamespace.SVG.getUri(), "y2" ) &&
                             shape.hasAttributeNS(OdfDocumentNamespace.SVG.getUri(), "x2") ){
                            int x1 = MapHelper.normalizeLength(shape.getAttributeNS(OdfDocumentNamespace.SVG.getUri(), "x1"));
                            int x2 = MapHelper.normalizeLength(shape.getAttributeNS(OdfDocumentNamespace.SVG.getUri(), "x2"));
                            int y1 = MapHelper.normalizeLength(shape.getAttributeNS(OdfDocumentNamespace.SVG.getUri(), "y1"));
                            int y2 = MapHelper.normalizeLength(shape.getAttributeNS(OdfDocumentNamespace.SVG.getUri(), "y2"));
                            anchorHorOffset = Math.min(x1, x2);
                            width = Math.abs(x2 -x1) + 1;
                            anchorVertOffset = Math.min(y1, y2);
                            height = Math.abs(y2 -y1) + 1;
                        }

                    } else {

                        if (shape.hasAttributeNS(OdfDocumentNamespace.SVG.getUri(), "width")) {
                            width = MapHelper.normalizeLength(shape.getAttributeNS(OdfDocumentNamespace.SVG.getUri(), "width"));
                        }
                        if (shape.hasAttributeNS(OdfDocumentNamespace.SVG.getUri(), "height")) {
                            height = MapHelper.normalizeLength(shape.getAttributeNS(OdfDocumentNamespace.SVG.getUri(), "height"));
                        }
                                             if (shape.hasAttributeNS(OdfDocumentNamespace.SVG.getUri(), "x")) {
                            anchorHorOffset = MapHelper.normalizeLength(shape.getAttributeNS(OdfDocumentNamespace.SVG.getUri(), "x"));
                        }
                        if (shape.hasAttributeNS(OdfDocumentNamespace.SVG.getUri(), "y")) {
                            anchorVertOffset = MapHelper.normalizeLength(shape.getAttributeNS(OdfDocumentNamespace.SVG.getUri(), "y"));
                        }
                    }
                    try {
                        if( height != 0) {
                            drawingProps.put("height", height);
                        }
                        if( width != 0) {
                            drawingProps.put("width", width);
                        }
                        if (anchorHorOffset != 0) {
                            drawingProps.put("anchorHorOffset", anchorHorOffset);
                            drawingProps.put("left", anchorHorOffset);
                        }
                        if (anchorVertOffset != 0) {
                            drawingProps.put("anchorVertOffset", anchorVertOffset);
                            drawingProps.put("top", anchorVertOffset);
                        }
                    } catch (JSONException ex) {
                        Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                    }
                    if (shape.hasAttributeNS(OdfDocumentNamespace.DRAW.getUri(), "transform")) {
                        try {
                            String transform = shape.getAttributeNS(OdfDocumentNamespace.DRAW.getUri(), "transform");
                            int index = transform.indexOf("translate");
                            if( index >= 0) {
                                index = transform.indexOf('(', index);
                                transform = transform.substring(index, transform.length());
                                int separator = transform.indexOf(' ');
                                String leftValue = transform.substring(1, separator);
                                index = transform.indexOf(')', separator);
                                String rightValue = transform.substring(separator + 1, index);
                                anchorHorOffset += MapHelper.normalizeLength(leftValue);
                                anchorVertOffset += MapHelper.normalizeLength(rightValue);
                            }
                            if (anchorVertOffset != 0) {
                                drawingProps.put("anchorVertOffset", anchorVertOffset);
                            }
                            if (anchorHorOffset != 0) {
                                drawingProps.put("anchorHorOffset", anchorHorOffset);
                            }
                        } catch (IndexOutOfBoundsException ex) {
                            Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                        } catch (JSONException ex) {
                            Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                        }
                    }
                    //<attribute name="text:anchor-type">
                    //	<choice>
                    //		<value>page</value>
                    //		<value>frame</value>
                    //		<value>paragraph</value>
                    //		<value>char</value>
                    //		<value>as-char</value>
                    //	</choice>
                    //</attribute>
				/*	API:
                     anchorHorBase:  Horizontal anchor mode:		One of 'margin', 'page', 'column', 'character', 'leftMargin', 'rightMargin', 'insideMargin', or 'outsideMargin'.
                     /*
                     @text:anchor-type: h=anchorHorBase & v=anchorVerBase
                     page		=> h=page	v=page
                     frame		=> h=column v=margin
                     paragraph	=> h=column v=paragraph
                     char		=> h=character v=paragraph
                     as-char		=> inline & h & v weglassen*/
                    if (shape.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "anchor-type")) {
                        try {
                            String anchorVertBase = null;
                            String anchorHorBase = null;
                            String anchorType = shape.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "anchor-type");
                            if (anchorType.equals("page")) {
                                // OX API: true: image as character, false: floating mode
                                drawingProps.put("inline", Boolean.FALSE);
                                //page anchor requires page relation
                                drawingProps.put("anchorHorBase", "page");
                                drawingProps.put("anchorVertBase", "page");
                            } else if (anchorType.equals("frame")) {
                                // OX API: true: image as character, false: floating mode
                                drawingProps.put("inline", Boolean.FALSE);
                                anchorVertBase = "column";
                                anchorVertBase= "margin";
                            } else if (anchorType.equals("paragraph")) {
                                // OX API: true: image as character, false: floating mode
                                drawingProps.put("inline", Boolean.FALSE);
                                anchorHorBase = "column";
                                anchorVertBase = "paragraph";
                            } else if (anchorType.equals("char")) {
                                // OX API: true: image as character, true: floating mode
                                drawingProps.put("inline", Boolean.FALSE);
                                anchorHorBase = "character";
                                anchorVertBase = "paragraph";
                            } else if (anchorType.equals("as-char")) {
                                // OX API: true: image as character, false: floating mode
                                drawingProps.put("inline", Boolean.TRUE);
                            }
                            if(anchorVertBase != null && !drawingProps.has("anchorVertBase")) {
                                drawingProps.put("anchorVertBase", anchorVertBase);
                            }
                            if(anchorHorBase != null && !drawingProps.has("anchorHorBase")) {
                                drawingProps.put("anchorHorBase", anchorHorBase);
                            }
                        } catch (JSONException ex) {
                            Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                        }
                    }
                    hardFormatting.put("drawing", drawingProps);

                    mComponentDepth++;
                    List<Integer> pos = updateComponentPosition();
                    // the delay of the operation was not the solution, as the children would be added fist
                    // instead a setAttribute would be more appropriate
                    // even if there is a automatic style, only the template style is required
                    if (element instanceof OdfStyleableShapeElement) {
                        String styleId = ((OdfStyleableShapeElement) shape).getDocumentStyleName();
                        if (styleId != null && !styleId.isEmpty()) {
                            hardFormatting.put("styleId", styleId);
                        }
                    }
                    ShapeProperties shapeProps = new ShapeProperties(pos, hardFormatting);
                    // special handling for frames as together with the image child they are a single component
                    if (element instanceof DrawFrameElement) {
                        shapeProps.setDrawFrameElement((DrawFrameElement) shape);
                    } else if(element instanceof DrawGElement){
                        shapeProps.setGroupShape();
                    }
                    element.markAsComponentRoot(true);
                    mCurrentComponent = mCurrentComponent.createChildComponent(element);
                    mComponentStack.push(shapeProps);
//                    mShapePropertiesStack.push(shapeProps);

                    // table component (table within a text document or a spreadsheet)
                } else if (element instanceof TableTableElement) {
                    mComponentDepth++;
                    // The table will be created with column width, after columns are parsed (just before first row!)
                    updateComponentPosition();
                    // tables are not written out directly, but its operation collected and only flushed
                    // if they are not exceeding a maximum size

                    isTableNew = true;
                    mTableElement = (TableTableElement) element;
                    mCurrentComponent = mCurrentComponent.createChildComponent(mTableElement);
                    // initialize a new list for the relative column widths
                    //ToDo: Receive the styles from the root component
                    //ToDo: If I do not want a DOM, do I have to parse the styles and addChild them to component?
                    //		Do I have to parse the styles.xml first to get the props as maps (hashmaps)?
                    if (mTableElement.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "style-name")) {
                        mTableHardFormatting = mJsonOperationProducer.getHardStyles(mTableElement);
                        String styleId = mTableElement.getDocumentStyleName();
                        if (styleId != null && !styleId.isEmpty()) {
                            if (mTableHardFormatting == null) {
                                mTableHardFormatting = new HashMap<String, Object>();
                            }
                            mTableHardFormatting.put("styleId", styleId);
                            //	All ODF styles are hard formatted
                            //	JSONObject tableProps = mTableHardFormatting.get("table");
                            //	mTableHardFormatting.put("templateStyleId", table.getDocumentStyle().getStyleNameAttribute());
                            //	OdfStyle tableStyle = table.getDocumentStyle();
                            //	if(tableStyle != null){
                            //	mTableDisplayName = tableStyle.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "display-name");
                        }
                    }
                    mTableName = mTableElement.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "name");
                    if (mIsSpreadsheet) {
                        // The grid is known after columns had been parsed, updating later to row positino
                        List<Integer> tablePosition = new LinkedList<Integer>(mLastComponentPositions);
                        cacheTableOperation( OperationConstants.TABLE, tablePosition, mTableHardFormatting, null, mTableName);
                    } else {
                        mColumnRelWidths = new LinkedList<Integer>();
                    }
                    element.markAsComponentRoot(true);
                } else if (element instanceof TableTableRowElement) {
                    mComponentDepth++;
                    if (isTableNew) {
                        if (!mIsSpreadsheet) {
                            // In case neiter relative nor absolute table column width were given, all column are equal sized (given rel size of '1')
                            mColumnRelWidths = collectColumnWidths(mTableElement, mColumns);
                            mColumns.clear();
                            if (mColumnRelWidths != null && mColumnRelWidths.isEmpty()) {
                                for (int i = 0; i < mColumnCount; i++) {
                                    mColumnRelWidths.add(ONE);
                                }
                            }

                            // The grid is known after columns had been parsed, updating later to row positino
                            List<Integer> tablePosition = new LinkedList<Integer>(mLastComponentPositions);
                            cacheTableOperation( OperationConstants.TABLE, tablePosition, mTableHardFormatting, mColumnRelWidths, mTableName);
                        }
                        mTableHardFormatting = null;
                        isTableNew = false;
                        mTableName = null;
                        if(mColumnCount > 1024) {
                            mIsBigColumnCount = true;
                        }
                        mColumnCount = 0;
                        mColumnRelWidths = null;
                    }
                    List<Integer> position = updateComponentPosition();
                    TableTableRowElement row = (TableTableRowElement) element;
                    mCurrentComponent = mCurrentComponent.createChildComponent(row);
                    // repeatition can cause a different positioning
                    int repeatedRows = 1;
                    if (row.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-rows-repeated")) {
                        repeatedRows = Integer.parseInt(row.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-rows-repeated"));
                        mCurrentComponent.hasRepeated(true);
                    }
                    boolean isVisible = Boolean.TRUE;
                    if (row.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "visibility")) {
                        isVisible = VISIBLE.equals(row.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "visibility"));
                    }
                    Map<String, Object> hardFormatting = mJsonOperationProducer.getHardStyles(row);
                    OdfStyle templateStyle = row.getDocumentStyle();
                    String styleId = null;
                    if (templateStyle != null) {
                        styleId = templateStyle.getStyleNameAttribute();
                        if (styleId != null && !styleId.isEmpty()) {
                            hardFormatting.put("styleId", styleId);
                        }
                    }
                    if (!isVisible) {
                        JSONObject rowProps;
                        if (hardFormatting == null) {
                            // if there are absolute styles, but not the main property set, where the templateStyleId should be placed in
                            if (hardFormatting == null) {
                                hardFormatting = new HashMap<String, Object>();
                            }
                        }
                        if (!hardFormatting.containsKey("row")) {
                            rowProps = new JSONObject();
                            hardFormatting.put("row", rowProps);
                        } else {
                            rowProps = (JSONObject) hardFormatting.get("row");
                            if (rowProps == null) {
                                rowProps = new JSONObject();
                            }
                        }
                        try {
                            rowProps.put("visible", Boolean.FALSE);
                        } catch (JSONException ex) {
                            Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                        }
                    }
                    if (mIsSpreadsheet) {
                        int firstRow = position.get(position.size() - 1);
                        CachedTable cachedTableOps = (CachedTable)mComponentStack.peek();
                        cachedTableOps.mFirstRow = firstRow;
                        cachedTableOps.mLastRow = firstRow + repeatedRows - 1;
                        cachedTableOps.mRepeatedRowsOnly = repeatedRows - 1;
                        cachedTableOps.rowCount = cachedTableOps.mLastRow + repeatedRows - 1;
                        String styleName = null;
                        if (row.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "style-name")) {
                            styleName = row.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "style-name");
                        }
                        cacheTableOperation( OperationConstants.ROWS, position, hardFormatting, firstRow, firstRow + repeatedRows - 1,
                        		cachedTableOps.mPreviousRepeatedRows, isVisible, styleName, repeatedRows);
                    } else {
                        cacheTableOperation( OperationConstants.ROWS, position, hardFormatting, repeatedRows);
                    }
                    element.markAsComponentRoot(true);
                } else if (element instanceof TableTableCellElement) {
                    mComponentDepth++;
                    TableTableCellElement cell = (TableTableCellElement) element;
                    mCurrentComponent = mCurrentComponent.createChildComponent(cell);
                    CachedTable cachedTableOps = (CachedTable)mComponentStack.peek();
                    cachedTableOps.mCellRepetition = 1;
                    int repetition = 1;
                    Map<String, Object> hardFormatting = mJsonOperationProducer.getHardStyles(cell);
                    if (cell.hasAttributes()) {
                        try {
                            // if there are absolute styles, but not the main property set, where the templateStyleId should be placed in
                            if (hardFormatting == null || !hardFormatting.containsKey("cell")) {
                                if (hardFormatting == null) {
                                    hardFormatting = new HashMap<String, Object>();
                                }
                            }
                            JSONObject cellProps = (JSONObject) hardFormatting.get("cell");
                            if (cellProps == null) {
                                cellProps = new JSONObject();
                            }
                            // repeatition and covering can cause a different positioning
                            // ToDo: To make DOM optional, work on the component instead of the element. Check directly SAX attributes parameter!
                            if (cell.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-repeated")) {
                                // cellProps.put("repeatedColumns", cell.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-repeatedColumns"));
                            	cachedTableOps.mCellRepetition = Integer.parseInt(cell.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-repeated"));
                                repetition = Integer.parseInt(cell.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-repeated"));
                                mCurrentComponent.hasRepeated(true);
                            }
                            if (cell.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-spanned")) {
                                cellProps.put(COLUMN_SPAN, Integer.parseInt(cell.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-spanned")));
                            }
                            if (cell.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-rows-spanned")) {
                                cellProps.put(ROW_SPAN, Integer.parseInt(cell.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-rows-spanned")));
                            }
                            if (cell.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "formula") || cell.hasAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "value-type")) {
                                String valueType = cell.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "formula");
                                Object cellValue = getCellValue(cell, valueType);
                                if (cellValue != null) {
                                    cellProps.put("value", cellValue);
                                }
                                // API MISSING: cellProps.put("numberFormat", mJsonOperationProducer.getCellNumberFormat(valueType));
                            }
                            if (cellProps.length() != 0) {
                                hardFormatting.put("cell", cellProps);
                            }
                        } catch (JSONException ex) {
                            Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                        }
                    }
                    List<Integer> position = updateComponentPosition();
                    OdfStyle templateStyle = cell.getDocumentStyle();
                    if (templateStyle != null) {
                        String styleId = templateStyle.getStyleNameAttribute();
                        if (styleId != null && !styleId.isEmpty()) {
                            hardFormatting.put("styleId", styleId);
                        }
                    }
                    if (mIsSpreadsheet) { // caching the formats, when traversing through possible sub tables
                        cacheTableOperation( OperationConstants.CELLS, mLastComponentPositions, hardFormatting, mCurrentComponent, repetition, cell.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "style-name"));
                        // all content of all paragraphs had been added to this builder, as our spreadsheet cell has only ONE format at this time.
                        if (mSpreadsheetCellContent == null) {
                            mSpreadsheetCellContent = new StringBuilder();
                        } else {
                            // reset and reuse buffer
                            mSpreadsheetCellContent.setLength(0);
                        }

                    } else {
                        cacheTableOperation( OperationConstants.CELLS, position, hardFormatting, mCurrentComponent, repetition );
                    }
                    element.markAsComponentRoot(true);
                } else if (element instanceof TextLineBreakElement) {
                    mComponentDepth++;
                    TextLineBreakElement lineBreak = (TextLineBreakElement) element;
                    List<Integer> position = updateComponentPosition();
                    mCurrentComponent = mCurrentComponent.createChildComponent(lineBreak);
                    cacheOperation(false, OperationConstants.HARD_BREAK, position, null, null, null);
                    element.markAsComponentRoot(true);
                } else if (element instanceof TextTabElement) {
                    mComponentDepth++;
                    TextTabElement tab = (TextTabElement) element;
                    List<Integer> position = updateComponentPosition();
                    mCurrentComponent = mCurrentComponent.createChildComponent(tab);
                    cacheOperation(false, OperationConstants.TAB, position, null, null, null);
                    element.markAsComponentRoot(true);
                } else if (Component.isField(uri, localName)) {
                    mComponentDepth++;
                    List<Integer> position = updateComponentPosition();
                    mCurrentComponent = mCurrentComponent.createChildComponent(element);
                    TextFieldSelection selection = null;
                    if (element.hasAttributeNS(LIBRE_OFFICE_MS_INTEROP_NAMESPACE, "type") && element.getAttributeNS(LIBRE_OFFICE_MS_INTEROP_NAMESPACE, "type").equals(LIBRE_OFFICE_MS_INTEROP_TYPE_CHECKBOX)) {
                        selection = new TextFieldSelection(element, position, LIBRE_OFFICE_MS_INTEROP_CHECKBOX_UNICODE);
                    } else {
                        if (mFileDom instanceof OdfContentDom) {
                            selection = new TextFieldSelection(element, position, ((OdfContentDom) mFileDom).getAutomaticStyles(), mUserFieldDecls);
                        } else {
                            selection = new TextFieldSelection(element, position, ((OdfStylesDom) mFileDom).getAutomaticStyles(), mUserFieldDecls);
                        }

                        // kann auch (OdfStylesDom) sein!
//                        element.getParentNode();
//                       TextTimeElement telem = (TextTimeElement)element;
//                       Map<String, Object> hardFormatting = mJsonOperationProducer.getHardStyles(telem);
                    }
                    mTextSelectionStack.add(selection);
                } else {
                    mComponentDepth++;
                    element.markAsComponentRoot(true);
                }

            } else if (element instanceof TextSpanElement) {
                // Span <text:span> will be triggering an operation after the text content is parsed
                TextSpanSelection selection = new TextSpanSelection((TextSpanElement) element, getTextPosition());
                mTextSelectionStack.add(selection);
            } else if (element instanceof TextAElement) {
                TextHyperlinkSelection selection = new TextHyperlinkSelection((TextAElement) element, getTextPosition());
                mTextSelectionStack.add(selection);
            } else if (element instanceof TextSElement) { // IMPROVABLE: Currently no component, as it will be removed anyway and would burden removal from automatic path counting
                mComponentDepth++;
                List<Integer> position = updateComponentPosition();
                if (mIsCharsBeginning) {
                    mCharsStartPosition = position;
                    mIsCharsBeginning = false;
                }
                // No operation triggering as client knows only space characters. We keep the parsing/mapping to the more performant server
                TextSElement spaces = (TextSElement) element;
                mCurrentComponent = mCurrentComponent.createChildComponent(spaces);
                Integer quantity = spaces.getTextCAttribute();
                if (quantity == null) {
                    addText(/*mCachedTableOps, */"\u0020");
                    // mCharsForOperation.append('\u0020');
                } else {
                    for (int i = 0; i < quantity; i++) {
                        mCharsForOperation.append('\u0020');
                    }
                    addText(/*mCachedTableOps, */mCharsForOperation);
                }

            } else if (element instanceof TableTableColumnElement) {
                // Columns can be grouped by <table:table-columns> and <table:table-column-group>, these would addChild metadata to the following columns
                // Column command should be triggered when one of the grouping starts or closes or if the first row arrives
                TableTableColumnElement column = (TableTableColumnElement) element;
                // Adjust Column Count
                mColumnCount++;
                int repeatedColumns = 1;
                int firstColumn = mColumnCount - 1;
                if (column.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-repeated")) {
                    repeatedColumns = Integer.parseInt(column.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-repeated"));
                    if (repeatedColumns > 1) {
                        mColumnCount += (repeatedColumns - 1);
                    }
                }
                String styleName = null;
                if (column.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "style-name")) {
                    styleName = column.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "style-name");
                }
                Map<String, Object> mappedDefaultCellStyleProperties = null;
                String defaultCellStyleId = null;
                if (column.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "default-cell-style-name")) {
                    defaultCellStyleId = column.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "default-cell-style-name");
                    mappedDefaultCellStyleProperties = getMappedStyleProperties(defaultCellStyleId, OdfStyleFamily.TableCell);
                }
                String templateStyleId = null;
                if (mIsSpreadsheet) { // Text are handling columns only as an array of widths
                    Map<String, Object> hardFormatting = mJsonOperationProducer.getHardStyles(column);
                    OdfStyle templateStyle = column.getDocumentStyle();
                    if (templateStyle != null) {
                        templateStyleId = templateStyle.getStyleNameAttribute();
                    }
                    boolean isVisible = Boolean.TRUE;
                    if (column.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "visibility")) {
                        isVisible = VISIBLE.equals(column.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "visibility"));
                    }
                    // if I need to created column attribute set to store one of the following..
                    if (templateStyleId != null || mappedDefaultCellStyleProperties != null || !isVisible) {
                        JSONObject columnProps;
                        if (hardFormatting == null) {
                            // if there are absolute styles, but not the main property set, where the templateStyleId should be placed in
                            if (hardFormatting == null) {
                                hardFormatting = new HashMap<String, Object>();
                            }
                        }
                        if (!isVisible) {
                            if (!hardFormatting.containsKey("column")) {
                                columnProps = new JSONObject();
                                hardFormatting.put("column", columnProps);
                            } else {
                                columnProps = (JSONObject) hardFormatting.get("column");
                            }
                            try {
                                columnProps.put("visible", Boolean.FALSE);
                            } catch (JSONException ex) {
                                Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                            }
                        }
                        if (templateStyleId != null && !templateStyleId.isEmpty()) {
                            hardFormatting.put("styleId", templateStyleId);
                        }
                    }
                    cacheTableOperation( OperationConstants.COLUMNS, mLastComponentPositions, hardFormatting, firstColumn, mColumnCount - 1, mappedDefaultCellStyleProperties, repeatedColumns, isVisible, styleName, defaultCellStyleId);
                } else { // if text
                    if (mColumns == null) {
                        mColumns = new ArrayList();
                    }
                    mColumns.add(column);
                }
            } else if (element instanceof TextListElement) {
                TextListElement list = (TextListElement) element;
                // in case it is a new list
                if (mListStyleStack.isEmpty()) {
                    // Add always a style, so it can be popped of the stack EVERY time a list element ends
                    ParagraphListProperties paragraphListProps = new ParagraphListProperties();
                    paragraphListProps.setListStart(true);

                    // There are two continuation mechanisms for lists in ODF.
                    // ODF 1.0/1.1 uses @text:continue-numbering using true/false
                    // ODF 1.2 added @text:continue-list using an IDRef to an xml:id of another list.
                    String continuedListId = list.getTextContinueListAttribute();
                    String listXmlId = list.getXmlIdAttribute();
                    if (continuedListId != null && !continuedListId.isEmpty()) {
                        paragraphListProps.setListId(newContinuedList(continuedListId, listXmlId).getListId());
                    } else if (listXmlId != null && !listXmlId.isEmpty()) {
                        paragraphListProps.setListId(newContinuedList(listXmlId).getListId());
                    }
                    if (listXmlId != null && !listXmlId.isEmpty()) {
                        paragraphListProps.setListXmlId(listXmlId);
                    } else {
                        paragraphListProps.setListXmlId(null);
                    }
                    mListStyleStack.add(paragraphListProps);
                } else {
                    // Add always a style, so it can be popped of the stack EVERY time a list element ends
                    mListStyleStack.add(new ParagraphListProperties());
                }

                // @text:continue-numbering LATER
                // @text:continue-list LATER
                // @xml-id - LATER
                // @text:style-name is the given list style unless overwritten by a decendent list
                //	Check if the list style was used already in the document,
                //	if not, map the list properties. Check first in auto than in template.
                //	(Due to MSO issue the style might be even in auto in styles.xml - shall I move them back to content?)
                if (list.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "style-name")) {
                    String listStyle = list.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "style-name");
                    mListStyleStack.getLast().setListStyleName(listStyle);
                }
            } else if (element instanceof TextListItemElement || element instanceof TextListHeaderElement) {
                ParagraphListProperties paragraphListStyle = mListStyleStack.getLast();
                OdfElement listItem = element;
                if (listItem instanceof TextListHeaderElement) {
                    // list header never show a label
                    paragraphListStyle.showListLabel(false);
                } else {
                    // As a new list item starts, the next paragraph needs to provide the list label
                    paragraphListStyle.showListLabel(true);
                }
                //	@text:start-value is provided to the first paragraph only
                if (listItem.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "start-value")) {
                    mListStartValue = Integer.parseInt(listItem.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "start-value"));
                }
                //	@text:style-override overrides within this list item the list style
                if (listItem.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "style-override")) {
                    String styleOverride = listItem.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "style-override");
                    if (styleOverride != null && !styleOverride.isEmpty()) {
                        paragraphListStyle.overrideListStyle(styleOverride);
                    } else {
                        paragraphListStyle.overrideListStyle(null);
                    }
                } else {
                    paragraphListStyle.overrideListStyle(null);
                }
                //  @xml-id
                String listXmlId = null;
                if (listItem instanceof TextListItemElement) {
                    listXmlId = ((TextListItemElement) listItem).getXmlIdAttribute();
                } else if (listItem instanceof TextListHeaderElement) {
                    listXmlId = ((TextListHeaderElement) listItem).getXmlIdAttribute();
                }
                if (listXmlId != null && !listXmlId.isEmpty()) {
                    mListStyleStack.getLast().setListItemXmlId(listXmlId);
                } else {
                    mListStyleStack.getLast().setListItemXmlId(null);
                }
                //		<style:master-page style:name="Standard" style:page-layout-name="Mpm1">
            } else if (element instanceof StyleMasterPageElement) {
                StyleMasterPageElement masterPage = (StyleMasterPageElement) element;
                mMasterPageStyleName = masterPage.getStyleNameAttribute();
                mPageLayoutName = masterPage.getStylePageLayoutNameAttribute();
                if (mPageLayoutName != null) {
                    OdfOfficeAutomaticStyles autoStyles = mSchemaDoc.getStylesDom().getAutomaticStyles();
                    OdfStylePageLayout pageLayout = autoStyles.getPageLayout(mPageLayoutName);
                    mPageStyleUsage = pageLayout.getStylePageUsageAttribute();
                }
                String nextMasterPageStyle = masterPage.getStyleNextStyleNameAttribute();
                if(nextMasterPageStyle != null && !nextMasterPageStyle.isEmpty()){
                    mHasNextMasterPage = true;
                }else{
                    mHasNextMasterPage = false;
                }

            } else if (Component.isHeaderRoot(uri, localName)) {
                    PageArea pageArea = null;
                    if (localName.equals("header")) {
                        /*
                         <value>all</value>
                         <value>left</value>
                         <value>right</value>
                         <value>mirrored</value>*/
                        if (mPageStyleUsage != null && mPageStyleUsage.equals("right")) {
                            pageArea = HEADER_EVEN;
                        } else {
                            if(mHasNextMasterPage){
                                pageArea = HEADER_FIRST;
                            }else{
                                pageArea = HEADER_DEFAULT;
                            }
                        }
                    } else if (localName.equals("header-left")) {
                        pageArea = HEADER_EVEN;

                    } else {
                        pageArea = HEADER_FIRST;
                    }
                mContextName = pageArea.getPageAreaName() + CONTEXT_DELIMITER + mMasterPageStyleName;
                //insert the Header style
                // {"name":"insertHeaderFooter","id":"Standard_header_default","type":"header_default"}
                mJsonOperationProducer.insertHeaderFooter(mContextName, pageArea);
            } else if (Component.isFooterRoot(uri, localName)) {
                //insert the Footer style
                    PageArea pageArea = null;
                    if (localName.equals("footer")) {
                        /*
                         <value>all</value>
                         <value>left</value>
                         <value>right</value>
                         <value>mirrored</value>*/
                        if (mPageStyleUsage != null && mPageStyleUsage.equals("right")) {
                            pageArea = FOOTER_EVEN;
                        } else {
                            if(mHasNextMasterPage){
                                pageArea = FOOTER_FIRST;
                            }else{
                                pageArea = FOOTER_DEFAULT;
                            }
                        }
                    } else if (localName.equals("footer-left")) {
                        pageArea = FOOTER_EVEN;
                    } else {
                        pageArea = FOOTER_FIRST;
                    }
                mContextName = pageArea.getPageAreaName() + CONTEXT_DELIMITER + mMasterPageStyleName;
                mJsonOperationProducer.insertHeaderFooter(mContextName, pageArea);
            } else if (element instanceof DrawImageElement) {
                DrawImageElement image = (DrawImageElement) element;
                ShapeProperties frameProps = (ShapeProperties)mComponentStack.peek();
//                ShapeProperties frameProps = mShapePropertiesStack.peekFirst();
                int childNo = frameProps.incrementChildNumber();
                if (childNo == 1) {

                    Map<String, Object> hardFormatting = new HashMap<String, Object>();
                    hardFormatting.putAll(frameProps.getShapeHardFormatting());
                    JSONObject drawingProps = (JSONObject) hardFormatting.get("drawing");
                    JSONObject imageProps = (JSONObject) hardFormatting.get("image");
                    if (image.hasAttributeNS(OdfDocumentNamespace.XLINK.getUri(), "href")) {
                        try {
                            String href = image.getAttributeNS(OdfDocumentNamespace.XLINK.getUri(), "href");
                            imageProps.put("imageUrl", href);
                            // if there is cropping from the frame, we need to do further calculation based on real graphic size
                            if (imageProps.has("cropRight") && (imageProps.has("height") || imageProps.has("width"))) {
                                JsonOperationProducer.calculateCrops(image, href, imageProps);
                            }
                        } catch (JSONException ex) {
                            Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                        }
                    }
                    if (image.hasAttributeNS(OdfDocumentNamespace.XML.getUri(), "id")) {
                        try {
                            drawingProps.put("imageXmlId", image.getAttributeNS(OdfDocumentNamespace.XML.getUri(), "id"));
                        } catch (JSONException ex) {
                            Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                        }
                    }
                    //ToDo: Need test document with child element having office:binary-data with base64 content

                    DrawFrameElement frameElement = frameProps.getDrawFrameElement();
                    frameElement.markAsComponentRoot(true);
                    mComponentStack.pop();
//                    mShapePropertiesStack.pollFirst();
                    mComponentStack.push(frameProps);
//                    mShapePropertiesStack.addFirst(frameProps);
                    frameProps.declareImage();
                    hardFormatting.put("drawing", drawingProps);
//					}else {
//						drawingProps.put("viewAlternative", childNo - 1);
                }

//				if (!frameProps.hasImageSibling()) {
//					mComponentDepth++;
//					frameProps.setFramePosition(updateComponentPosition());
//					mCurrentComponent = mCurrentComponent.createChildComponent(frameElement);
//					//			position.set(position.size() - 1, position.get(position.size() - 1) +1);
//				}
//				if (childNo == 1) { // DISABLING REPLACEMENT IMAGE FEATURE AS LONG CLIENT DOES NOT SUPPORT IT
//					 ToDo: Dependencies for frame replacement feature has to be updated in OdfElement.raiseComponentSize() for every Frame child/feature enabled
//					frameProps.saveShapeProps(frameProps.getShapePosition(), hardFormatting);
//				}
            } else if( element instanceof DrawTextBoxElement) {
//            	element.getAttributeNodeNS(namespaceURI, localName);
//                Map<String, Object> hardFormatting = null;
//                if (element instanceof OdfStyleableShapeElement) {
//                    hardFormatting = mJsonOperationProducer.getHardStyles((OdfStyleableShapeElement) element);
//                }
//                JSONObject drawingProps = (JSONObject) hardFormatting.get("drawing");
//            	JSONObject drawingProps = new JSONObject();
                ShapeProperties parentShapeProps = (ShapeProperties)mComponentStack.peek();
                JSONObject originalDrawingProps = (JSONObject) parentShapeProps.mShapeHardFormatations.get("drawing");
            	if (originalDrawingProps != null && !originalDrawingProps.has("height") ) {
                    try {
                        if(!parentShapeProps.mShapeHardFormatations.containsKey("shape")) {
                            parentShapeProps.mShapeHardFormatations.put("shape", new JSONObject());
                        }
                        JSONObject originalShapeProps = (JSONObject) parentShapeProps.mShapeHardFormatations.get("shape");
                        originalShapeProps.put("autoResizeHeight", "true");
                    } catch (JSONException ex) {
                        Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                    }
                }
            } else if( element instanceof TableDatabaseRangeElement) {
                CachedDBRange dbRange = new CachedDBRange(mTopLevelTables);
                dbRange.name = attributes.getValue("table:name");
                dbRange.setRangeAddress( attributes.getValue("table:target-range-address") );
                String display = attributes.getValue("table:display-filter-buttons");
                dbRange.displayFilterButtons = display != null && display.equals("true");
                mComponentStack.push(dbRange);
            } else if( element instanceof TableFilterElement) {
            } else if( element instanceof TableFilterOrElement) {
            } else if( element instanceof TableFilterConditionElement) {
                CachedDBRange dbRange = (CachedDBRange)mComponentStack.peek();
                String column = attributes.getValue("table:field-number");
                String value = attributes.getValue("table:value");
                String operator = attributes.getValue("table:operator");
                dbRange.createFilterCondition(Integer.parseInt(column), operator, value);
            } else if( element instanceof TableFilterSetItemElement) {
                CachedDBRange dbRange = (CachedDBRange)mComponentStack.peek();
                String value = attributes.getValue("table:value");
                dbRange.addFilterConditionItem(value);
            }

            // but within a shape being a frame, the child elements are still of interest (currently only <draw:image> supported)
//		} else if (!mShapePropertiesStack.isEmpty() && mShapePropertiesStack.peekLast().mDrawFrameElement != null) {

            if (Component.isDocumentRoot(uri, localName) || Component.isHeaderRoot(uri, localName) || Component.isFooterRoot(uri, localName)) {
                // temporary initated here as all the tests are not using the OperationTextDocument
                mCurrentComponent = new Component(element);
                mSchemaDoc.setRootComponent(mCurrentComponent);
                // for every header and footer restart counting
                if (Component.isHeaderRoot(uri, localName) || Component.isFooterRoot(uri, localName)) {
                    mLastComponentPositions.clear();
                } else {
                    mPageArea = PageArea.BODY;
                }
            }
        } else {
            if (element instanceof OdfElement) {
                element.ignoredComponent(true);
            }
        }

        // push the new element as the context node...
        //mCurrentNode = element;
        // addChild the new element as a child of the current context node
        mCurrentNode = mCurrentNode.appendChild(element);
    }

    private Map<String, Object> getMappedStyleProperties(String styleName, OdfStyleFamily family) {
        Map<String, Object> mappedStyleProperties = null;
        if (styleName != null && !styleName.isEmpty()) {
            // first check for automatic style of default cell style
            // the automatic styles are in a spreadsheet always in the content.xml, only when a table is in a header/footer it would be in the styles.xml (latter we do not support)
            OdfOfficeAutomaticStyles autoStyles;
            if (mFileDom instanceof OdfContentDom) {
                autoStyles = ((OdfContentDom) this.mFileDom).getAutomaticStyles();
            } else {
                autoStyles = ((OdfStylesDom) this.mFileDom).getAutomaticStyles();
            }

            OdfStyle templateStyle = null;
            OdfStyle autoStyle = autoStyles.getStyle(styleName, family);
            // found an automatic style
            if (autoStyle != null) {
                mappedStyleProperties = MapHelper.getMappedStyleProperties(autoStyle);
            } else {
                // if there is no automatic style, check if the style exist in the templates
                OdfOfficeStyles documentStyles = mSchemaDoc.getDocumentStyles();
                templateStyle = documentStyles.getStyle(styleName, family);
            }
            // found a template style
            if (templateStyle != null) {
                if (mappedStyleProperties == null) {
                    mappedStyleProperties = new HashMap<String, Object>();
                }
                mappedStyleProperties.put("styleId", styleName);
            }
        }
        return mappedStyleProperties;
    }

    private static String getFormulaValue(TableTableCellElement cell) {
        String value = null;
        if (cell.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "formula")) {
            String formula = cell.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "formula");
            // official ODF 1.2 formula prefix, unknown to OX System (calc backend)
            if (formula.startsWith("of:")) {
                value = formula.substring(3);
                // ancient/deprecated OpenOffice 2.x formula prefix
            } else if (formula.startsWith("oooc:")) {
                value = formula.substring(5);
            } else {
                value = formula;
            }
        }
        return value;
    }

    private static Object getCellValue(TableTableCellElement cell, String valueType) {
        Object value = null;
        if (valueType != null && !valueType.isEmpty()) {
            if (cell.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "formula")) {
                //table:formula
                value = getFormulaValue(cell);
            } else if (valueType.equals("boolean")) {
                //office:boolean-value
                value = Boolean.parseBoolean(cell.getAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "boolean-value"));
            } else if (valueType.equals("currency")) {
                //office:value and office:currency
                // 2Do: Missing Mapping for currency!!
                value = Long.parseLong(cell.getAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "value"));
            } else if (valueType.equals("date")) {
                // office:date-value
                value = cell.getAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "date-value");
            } else if (valueType.equals("float")) {
                //office:value
                value = Double.parseDouble(cell.getAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "value"));
            } else if (valueType.equals("percentage")) {
                //office:value
                value = Long.parseLong(cell.getAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "value"));
            } else if (valueType.equals("string")) {
                //office:string-value
                value = cell.getAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "string-value");
            } else if (valueType.equals("time")) {
                //office:time-value
                value = cell.getAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "time-value");
            }
            // do not allow empty strings.
            if (value != null && value instanceof String && ((String) value).isEmpty()) {
                value = null;
            }
        } else if (cell.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "formula")) {
            value = getFormulaValue(cell);
        } else if (cell.hasAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "value")) {
            value = Double.parseDouble(cell.getAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "value"));
        }

        return value;
    }

    @Override
    public void endElement(String uri, String localName, String qName) throws SAXException {
        flushTextAtEnd(uri, localName, qName);
        // at the end of a table check if it can be flushed
        if (uri != null && localName != null && localName.equals(TableTableElement.ELEMENT_NAME.getLocalName()) && uri.equals(OdfDocumentNamespace.TABLE.getUri())) {
            endTableSizeEvaluation();
        }
        // if we remove the current element, the current node shall not be changed in the end
        boolean selectionNormalization = false;
        // SPECIAL HANDLING FOR DESCRIPTION OF SHAPES: draw:frame as the shape is one of the children, e.g a draw:image child
        if (!checkEndOfBlockedSubTree(uri, localName)) {
            boolean isImageComponent = false;
            if ((uri != null && uri.equals(DrawFrameElement.ELEMENT_NAME.getUri()) && localName.equals(DrawFrameElement.ELEMENT_NAME.getLocalName()) || Component.isShapeElement(uri, localName))) {
                mComponentDepth--;
                // Check for description of shape/frame about to be closed
                ShapeProperties shapeProps = (ShapeProperties)mComponentStack.pop();
//                ShapeProperties shapeProps = mShapePropertiesStack.removeLast();
                isImageComponent = shapeProps.hasImageSibling();
                NodeList descList = mCurrentComponent.mRootElement.getElementsByTagNameNS(OdfDocumentNamespace.SVG.getUri(), SvgDescElement.ELEMENT_NAME.getLocalName());
                String description = null;
                if (descList.getLength() > 0) {
                    SvgDescElement desc = (SvgDescElement) descList.item(0);
                    Node descText = desc.getFirstChild();
                    if (descText != null && descText instanceof Text) {
                        description = ((Text) descText).getTextContent();
                    }
                }
                //TODO: Detection by node name might be wrong
                boolean isTopLevelAndPageBound = mCurrentNode.getParentNode().getNodeName().equals("office:text");
                if(isTopLevelAndPageBound)
                {
                    shapeProps.mDescription = description;
                    shapeProps.mOwnNode = mCurrentNode;
                    shapeProps.mContext = mContextName;
                    m_cachedPageShapes.add(shapeProps);
                } else {
                    shapeProps.createShapeOperation(description,
                            isImageComponent ? ShapeType.ImageShape : shapeProps.isGroupShape() ? ShapeType.GroupShape : ShapeType.NormalShape,
                                    mContextName);
                    if(shapeProps.isGroupShape()) {
                        mCurrentNode.setUserData("groupWidth", new Integer(shapeProps.mHoriOffsetMax - shapeProps.mHoriOffsetMin), null);
                        mCurrentNode.setUserData("groupHeight", new Integer(shapeProps.mVertOffsetMax - shapeProps.mVertOffsetMin), null);
                    }
                    //flush the inner operations of the shape
                    Iterator<CachedOperation> opIter = shapeProps.iterator();
                    while(opIter.hasNext())
                    {
                        CachedOperation op = opIter.next();
                        cacheOperation(true, op.mComponentType, op.mStart, op.mHardFormattingProperties, op.mComponentProperties);
                    }
                }
                mCurrentComponent = mCurrentComponent.getParent();
//			} else if (Component.isCoveredComponentRoot(uri, localName)) { // adjust counting for table cells without numbering
//				//ToDO: Instead to count the covered someone should count the spanning (BUT this is against OOXML cell numbering!)
//				mComponentDepth--;
            } else if (isSpaceElement(uri, localName)) {
                mComponentDepth--;
                mCurrentComponent = mCurrentComponent.getParent();
//		} else if (uri != null && uri.equals(DrawImageElement.ELEMENT_NAME.getUri()) && localName.equals(DrawImageElement.ELEMENT_NAME.getLocalName())) {
//			mFramePropertiesStack.getFirst().decrementChildNumber();
            } else if (uri != null && Component.isComponentRoot(uri, localName)) {
//                if (Component.isTextComponentRoot(uri, localName) && mWhitespaceStatusStack.size() > 0 && mWhitespaceStatusStack.getLast().mIsParagraphIgnored) {
                /* no ignored paragraphs anymore
            	if (Component.isTextComponentRoot(uri, localName) && mWhitespaceStatusStack.size() > 0 && mWhitespaceStatusStack.getLast().mIsParagraphIgnored) {) {
                    // do nothing for a ignored paragraph
                    mWhitespaceStatusStack.removeLast();
                    // SPECIAL HANDLING FOR IMAGE: draw:image (not a component root T- replacement for draw:frame)
                } else */
            	if (localName.equals(DrawFrameElement.ELEMENT_NAME.getLocalName()) && uri.equals(DrawFrameElement.ELEMENT_NAME.getUri()) && isImageComponent
                    || !(localName.equals(DrawFrameElement.ELEMENT_NAME.getLocalName()) && uri.equals(DrawFrameElement.ELEMENT_NAME.getUri()))) {
                    // if the current component is a text container flush spans
                    if (Component.isTextComponentRoot(mCurrentNode)) {
                        Collection<TextSelection> selections = ((TextParagraphElementBase) mCurrentNode).getTextSelections();
                        if (selections != null) {
                            for (TextSelection s : selections) {
                                OdfStylableElement selectionElement = (OdfStylableElement) s.getSelectionElement();
                                Map<String, Object> hardFormatting = mJsonOperationProducer.getHardStyles(selectionElement);
                                String styleId = null;
                                OdfStyle templateStyle = selectionElement.getDocumentStyle();
                                if (templateStyle != null) {
                                    styleId = templateStyle.getStyleNameAttribute();
                                }
                                if (s.hasUrl() || styleId != null) {
                                    try {
                                        JSONObject charProps;
                                        if (hardFormatting == null) {
                                            // if there are absolute styles, but not the main property set, where the templateStyleId should be placed in
                                            if (hardFormatting == null) {
                                                hardFormatting = new HashMap<String, Object>();
                                            }
                                        }
                                        if (s.hasUrl()) {
                                            if (!hardFormatting.containsKey("character")) {
                                                charProps = new JSONObject();
                                                hardFormatting.put("character", charProps);
                                            } else {
                                                charProps = (JSONObject) hardFormatting.get("character");
                                            }
                                            charProps.put("url", s.getURL());
                                        }
                                        if (styleId != null && !styleId.isEmpty()) {
                                            hardFormatting.put("styleId", styleId);
                                        } else {
                                            // add the implicit by LO/AOO used hyperlink style
                                            if (mHasHyperlinkTemplateStyle) {
                                                hardFormatting.put("styleId", HYERLINK_DEFAULT_STYLE);

                                            }
                                        }
                                    } catch (JSONException ex) {
                                        Logger.getLogger(OdfFileSaxHandler.class
                                            .getName()).log(Level.SEVERE, null, ex);
                                    }
                                }
                                if (hardFormatting != null) {
//                                    if (mWithinTable) {
                                    CachedComponent cachedTableOps = mComponentStack.empty() ? null : mComponentStack.peek();
                                        if (mIsSpreadsheet && cachedTableOps != null && cachedTableOps instanceof CachedTable) {
                                            // Currently all hardformatting text styles are being mapped to the cell
                                            CachedInnerTableOperation cellOperation = null;
                                            int lastOperation = cachedTableOps.size() - 1;
                                            cellOperation = (CachedInnerTableOperation) cachedTableOps.get(lastOperation);
                                            if (cellOperation.mComponentType.equals(OperationConstants.CELLS)) {
                                                if (cellOperation.mHardFormattingProperties == null) {
                                                    cellOperation.mHardFormattingProperties = new HashMap<String, Object>();

                                                }
                                                if (hardFormatting.containsKey("character")) {
                                                    if (!cellOperation.mHardFormattingProperties.containsKey("character")) {
                                                        // addChild map directly
                                                        cellOperation.mHardFormattingProperties.put("character", hardFormatting.get("character"));
                                                    } else {
                                                        // merge map items
                                                        JSONObject existingFormatting = (JSONObject) cellOperation.mHardFormattingProperties.get("character");
                                                        // all text styles are mapped to the cell
                                                        JSONObject newFormatting = (JSONObject) hardFormatting.get("character");
                                                        if (existingFormatting != null && newFormatting != null) {
                                                            Set<String> keys = existingFormatting.keySet();
                                                            for (String key : keys) {
                                                                try {
                                                                    newFormatting.put(key, existingFormatting.get(key));

                                                                } catch (JSONException ex) {
                                                                    Logger.getLogger(OdfFileSaxHandler.class
                                                                        .getName()).log(Level.SEVERE, null, ex);
                                                                }
                                                            }
                                                        }
                                                        cellOperation.mHardFormattingProperties.put("character", newFormatting);
                                                    }
                                                    // cache the style, in case the content is being removed, the style will be kept
                                                    // Cell cellComponent = (Cell) cellOperation.mComponentProperties[0];
                                                    //cellComponent.setInternalCellStyle(cellOperation.mHardFormattingProperties);
                                                }
                                                if (hardFormatting.containsKey("cell")) {
                                                    if (!cellOperation.mHardFormattingProperties.containsKey("cell")) {
                                                        // addChild map directly
                                                        cellOperation.mHardFormattingProperties.put("cell", hardFormatting.get("cell"));
                                                    } else {
                                                        // merge map items
                                                        JSONObject existingStyles = (JSONObject) cellOperation.mHardFormattingProperties.get("cell");
                                                        JSONObject newStyles = (JSONObject) hardFormatting.get("cell");
                                                        if (existingStyles != null && newStyles != null) {
                                                            Set<String> keys = existingStyles.keySet();
                                                            for (String key : keys) {
                                                                try {
                                                                    newStyles.put("cell", existingStyles.get(key));

                                                                } catch (JSONException ex) {
                                                                    Logger.getLogger(OdfFileSaxHandler.class
                                                                        .getName()).log(Level.SEVERE, null, ex);
                                                                }
                                                            }
                                                        }
                                                    }
                                                }
                                            }
                                        } else {
                                            cacheOperation(false, OperationConstants.ATTRIBUTES, s.getStartPosition(), hardFormatting, s.getEndPosition(), mContextName);
                                      }
                                }
                            }
                        }
//					// in this case check if the closing descendent (this element)
//					// had any none whitespace text and apply if necessary the change
//					// remove the current whitespace properties from the stack
//					int depth = mWhitespaceStatusStack.size();
//					boolean childHasWhiteSpace = true;
//					boolean parentHasOnlyWhiteSpace;
                        // if there is a parent text container
                        if (mWhitespaceStatusStack.size() > 0) {
                            mWhitespaceStatusStack.removeLast();
// BEFORE WE DID NOT ALLOWED TO FOLLOWING PARAGRAPHS
//						// see if the child only had whitespaces
//						childHasWhiteSpace = mWhitespaceStatusStack.getLast().hasOnlyWhiteSpace();
//						// switch to parent
//						mWhitespaceStatusStack.pop();
//						// see if the parent had only whitespaces
//						WhitespaceStatus parentWhiteSpaceStatus = mWhitespaceStatusStack.getLast();
//						parentHasOnlyWhiteSpace = parentWhiteSpaceStatus.hasOnlyWhiteSpace();
//						// if the parent had only whitespaces, but not the child
//						if (parentHasOnlyWhiteSpace && !childHasWhiteSpace) {
//							// remove the only whitespace modus from the parent
//							parentWhiteSpaceStatus.setOnlyWhiteSpace(childHasWhiteSpace);
//						}
//					} else {
//						// otherwise just end the state collection of this paragraph/heading
//						mWhitespaceStatusStack.pop();
                        }
                    }
                    // removing the last in the list of positions, when a component is closed
                    if (mIsSpreadsheet && localName.equals("table-cell") && mComponentStack.peek() instanceof CachedTable /*mCachedTableOps != null*/) {
                        CachedInnerTableOperation cellOperation = null;
                        // Caching between previous start-element method and now (end-element method), as we required the content string of the cell!
                        CachedTable cachedTableOps = (CachedTable)mComponentStack.peek();
                        int operationPos = cachedTableOps.mCachedTableContentOps.size() - 1;
                        cellOperation = cachedTableOps.mCachedTableContentOps.get(operationPos);
                        if (cellOperation != null && cellOperation.mComponentType.equals(OperationConstants.CELLS)) {
                            // adding the cell characters to the cell operation
                            cellOperation.mCellContentString = mSpreadsheetCellContent.toString();
                            //mCachedTableOps.add(operationPos)
                            // Adjust position for cell mColumnRepetition

                        }
                    }
                    if (mCurrentComponent.hasRepeated()) {
                        // if it is a cell or row not in spreadsheets all operations back from insertCell/insertRows
                        // need to be repeated with incremented positions (TODO: needs to be recursive! )
                        boolean isCell = localName.equals("table-cell");
                        boolean isRow = localName.equals("table-row");
                        if(!mIsSpreadsheet && (isRow || isCell)) {
                            CachedTable currentTable = (CachedTable)mComponentStack.peek();
                            int opSize = currentTable.size();
                            int pos = 0;
                            for(pos = opSize - 1; pos >= 0; --pos) {
                                CachedOperation op = currentTable.get(pos);
                                if(op.mComponentType.equals(isRow ? OperationConstants.ROWS : OperationConstants.CELLS) ){
                                    break;
                                }
                            }
                            //now we have a start index - add all ops from pos to opSize- 1 again repetition-times with modified position
                            if(pos > 0) {
                                CachedOperation cellInsertOp = currentTable.get(pos);
                                int incrementPos = cellInsertOp.mStart.size() - 1;
                                int repetition = (Integer)cellInsertOp.mComponentProperties[isRow ? 0 : 1];
                                for( int rep = 0; rep < repetition - 1; ++rep ) {
                                    for(int opPos = pos; opPos < opSize; ++opPos) {
                                        CachedOperation newOp = currentTable.get(opPos).clone();
                                        if(newOp.mStart != null) {
                                            int oldIndex = newOp.mStart.get(incrementPos);
                                            newOp.mStart.set( incrementPos, oldIndex  + rep + 1 );
                                        }
                                        cacheOperation(false, newOp.mComponentType, newOp.mStart, newOp.mHardFormattingProperties, newOp.mComponentProperties);
                                    }

                                }
                            }

                        }
                        mLastComponentPositions.set(mComponentDepth, mLastComponentPositions.get(mComponentDepth) + mCurrentComponent.repetition() - 1);
                    }
                    mComponentDepth--;
                    mCurrentComponent = mCurrentComponent.getParent();
                }
            } // if text delimiter - addChild text
            else if (Component.isTextSelection(mCurrentNode)) {
                // dropping the last (most inner) selection from the stack
                TextSelection textSelection = mTextSelectionStack.pollLast();
                if (textSelection != null) {
                    textSelection.setEndPosition(getTextPosition());
                    OdfElement root = ((OdfElement) mCurrentNode).getComponentRoot();
                    if (Component.isTextComponentRoot(root)) {
                        // sometimes when spans share the same text area, they are condensed to one. The remaining one is being returned, or in case of an empty element the parent
                        mCurrentNode = ((TextContainingElement) root).appendTextSelection(textSelection);
                        // selectionNormalization might delete an element in this case the change of the current node would be an error!
                        selectionNormalization = true;
                    }
                }
            } else if (uri.equals(OdfDocumentNamespace.TEXT.getUri()) && localName.equals("list-item")) {
                mListStyleStack.getLast().overrideListStyle(null);
            } else if (uri.equals(OdfDocumentNamespace.TEXT.getUri()) && localName.equals("list")) {
                // POP UP NEW LIST STYLE
                mListStyleStack.removeLast();
            } else if (uri.equals(OdfDocumentNamespace.TABLE.getUri()) && localName.equals("database-range")) {
                CachedDBRange dbRange = (CachedDBRange)mComponentStack.peek();
                mComponentStack.pop();
                dbRange.createOperations();
                Iterator<CachedOperation> opIter = dbRange.iterator();
                while(opIter.hasNext())
                {
                    CachedOperation op = opIter.next();
                    cacheOperation(true, op.mComponentType, op.mStart, op.mHardFormattingProperties, op.mComponentProperties);
                }
            } else if(localName.equals("spreadsheet") && uri.equals( OfficeSpreadsheetElement.ELEMENT_NAME.getUri())){
                if( mIsBigColumnCount ) {
                    mJsonOperationProducer.appendColumnDocumentProperties();
                    mCurrentNode.setUserData("maxColumns", "true", null);
                }

            }

        }
        // selectionNormalization might delete an element in this case the change of the current node would be an error!
        if (mCurrentNode != null && !selectionNormalization) {
            // pop to the parent node
            mCurrentNode = mCurrentNode.getParentNode();
        }
    }

    /**
     * Removes the cell and row insert operation after mapping the info set
     * (content and attributes) to range operations
     */
    private CachedTable mapCellsAndRow(CachedInnerTableOperation operation, CachedTable currentTable) {

        if (operation.mComponentType.equals(OperationConstants.CELLS)) {
            JSONObject cellJSON;
            Cell cellComponent = (Cell) operation.mComponentProperties[0];
            currentTable.mCellRepetition = cellComponent.repetition();

            // the content of a cell might be in the office:value attribute alone, while the <table:table-cell> only contains an empty paragraph
            boolean hasFormulaValue = operation.mHardFormattingProperties != null && !operation.mHardFormattingProperties.isEmpty() && operation.mHardFormattingProperties.containsKey("cell") && ((JSONObject) operation.mHardFormattingProperties.get("cell")).hasAndNotNull("value");
            // if there is either the formula and/or value attribute or the a text content string from the paragraph
            boolean hasContent = hasFormulaValue || operation.mCellContentString != null && operation.mCellContentString.length() > 0;
            // the existence of the @table:style-name attribute
            boolean hasFormat = (Boolean) operation.mComponentProperties[2]; // has a style name;

            // if there are existing styles these take precedence over the default cell style of the column
            boolean hasNamedDefault = currentTable.mHasOnlyDefaultColumnCellStyles != null && currentTable.mHasOnlyDefaultColumnCellStyles && (currentTable.mColumnCount == Table.MAX_COLUMN_NUMBER_CALC || currentTable.mColumnCount == Table.MAX_COLUMN_NUMBER_EXCEL);
            boolean hasFullRowSize = currentTable.mRowCount == Table.MAX_ROW_NUMBER;

            Map<String, Object> mappedDefaultCellStyle = getCurrentDefaultCellStyle(currentTable);
            // The column defaultCellStyle is sometimes named "default"., which will be mapped to the table style (the max number of columns exist and there are no gaps without default cell style)
            boolean isBlockedDefault = mappedDefaultCellStyle != null && hasNamedDefault && mappedDefaultCellStyle.size() == 1 && mappedDefaultCellStyle.containsKey("styleId") && mappedDefaultCellStyle.get("styleId").equals(COLUMN_CELL_DEFAULT_STYLE);
            boolean applyDefaultStyle = mappedDefaultCellStyle != null && !mappedDefaultCellStyle.isEmpty() && (!hasFullRowSize && !hasFormat && !isBlockedDefault || hasContent && !hasFormat);

            // Any mColumnRepetition have to split (as operation evaluation) in case different default cell style exist for the columns covering the repeated cell(s)
            // If it is a repeated cell and there are default cell column styles, which might overlap...
            if (currentTable.mCellRepetition > 1 && currentTable.mColumnDefaultCells != null && currentTable.mCurrentDefaultCellProps != null && (hasFormat || hasContent || (!hasFullRowSize && currentTable.mCurrentDefaultCellProps.mDefaultCellStyleId != null))) {

                int remainingCellRepetition = cellComponent.repetition();
                int cellRangeStart = currentTable.mCurrentColumnNo;
                int columnRangeStart = currentTable.mCurrentDefaultCellProps.mColumnStartPos;
                int columnRangeEnd = currentTable.mCurrentDefaultCellProps.mColumnStartPos + currentTable.mCurrentDefaultCellProps.mColumnRepetition - 1;

                // while the cell repetition is overlapping with default cells columns, in the beginning or end
                while (remainingCellRepetition >= 1) {
                    // One cell pseudo-operation --> for the GAP of cell range BEFORE the column range
                    int distanceToColumnStart = columnRangeStart - cellRangeStart;
                    if (distanceToColumnStart > 0) {
                        currentTable.mCellRepetition = distanceToColumnStart;
                        remainingCellRepetition -= distanceToColumnStart;
                        cellRangeStart += distanceToColumnStart;
                        if (hasFormat || hasContent || applyDefaultStyle) {
                            cellJSON = getCellJSON(operation, mappedDefaultCellStyle, applyDefaultStyle);
                        } else {
                            cellJSON = null;
                        }
                        // start position will only be used for sheet position
                        currentTable = evaluateSimilarCells(currentTable, operation, cellJSON, false);
                    }
                    // One cell pseudo-operation --> for the cell range from beginning of column range to the end
                    int distanceToColumnEnd = columnRangeEnd - currentTable.mCurrentColumnNo + 1;
                    if (distanceToColumnEnd >= 0) {
                        if (remainingCellRepetition >= distanceToColumnEnd) {
                            currentTable.mCellRepetition = distanceToColumnEnd;
                            remainingCellRepetition -= distanceToColumnEnd;
                        } else {
                            currentTable.mCellRepetition = remainingCellRepetition;
                            remainingCellRepetition = 0;
                        }
                    } else {
                        currentTable.mCellRepetition = remainingCellRepetition;
                        remainingCellRepetition = 0;
                    }
                    if (hasFormat || hasContent || applyDefaultStyle) {
                        cellJSON = getCellJSON(operation, mappedDefaultCellStyle, applyDefaultStyle);
                    } else {
                        cellJSON = null;
                    }
                    // start position will only be used for sheet position
                    currentTable = evaluateSimilarCells(currentTable, operation, cellJSON, false);
                    if (remainingCellRepetition > 0) {
                        getCurrentDefaultCellStyle(currentTable);
                        // if there are repeated cells left, but no further column cell style
                        if (currentTable.mCurrentDefaultCellProps == null) {
                            if (hasFormat || hasContent) {
                                cellJSON = getCellJSON(operation, mappedDefaultCellStyle, false);
                            } else {
                                cellJSON = null;
                            }
                            currentTable.mCellRepetition = remainingCellRepetition;
                            // start position will only be used for sheet position
                            currentTable = evaluateSimilarCells(currentTable, operation, cellJSON, false);
                            break;
                        }
                    }
                    cellRangeStart = currentTable.mCurrentColumnNo;
                    columnRangeStart = currentTable.mCurrentDefaultCellProps.mColumnStartPos;
                    columnRangeEnd = currentTable.mCurrentDefaultCellProps.mColumnStartPos + currentTable.mCurrentDefaultCellProps.mColumnRepetition - 1;
                    mappedDefaultCellStyle = getCurrentDefaultCellStyle(currentTable);
                }
            } else {
                if (hasFormat || hasContent || applyDefaultStyle) {
                    cellJSON = getCellJSON(operation, mappedDefaultCellStyle, applyDefaultStyle);
                } else {
                    cellJSON = null;
                }
                // start position will only be used for sheet position
                currentTable = evaluateSimilarCells(currentTable, operation, cellJSON, false);
            }
        } else if (operation.mComponentType.equals(OperationConstants.ROWS)) {
            currentTable.nextColumnDefaultCellListPos = 0;
            currentTable.mCurrentDefaultCellProps = null;
            // flush cells not being written out yet
            currentTable = evaluateSimilarCells(currentTable, null, null, true);
        }
        return currentTable;
    }

    /**
     * @return the current default cell styles
     */
    private static Map<String, Object> getCurrentDefaultCellStyle(CachedTable tableOps) {
        int currentPos = tableOps.mCurrentColumnNo;
        Map<String, Object> mappedDefaultCellStyle = null;
        // if there is no default cell settings
        if (tableOps.mCurrentDefaultCellProps == null) {
            tableOps = nextDefaultCellProps(tableOps);
        }
        if (tableOps.mCurrentDefaultCellProps != null) {
            int columnRangeStart = tableOps.mCurrentDefaultCellProps.mColumnStartPos;
            int columnRangeEnd = tableOps.mCurrentDefaultCellProps.mColumnStartPos + tableOps.mCurrentDefaultCellProps.mColumnRepetition - 1;
            if (currentPos > columnRangeEnd) {
                tableOps = nextDefaultCellProps(tableOps);
            }
            if (tableOps.mCurrentDefaultCellProps != null) {
                columnRangeStart = tableOps.mCurrentDefaultCellProps.mColumnStartPos;
                columnRangeEnd = tableOps.mCurrentDefaultCellProps.mColumnStartPos + tableOps.mCurrentDefaultCellProps.mColumnRepetition - 1;
                if (columnRangeStart <= currentPos && currentPos <= columnRangeEnd) {
                    mappedDefaultCellStyle = tableOps.mCurrentDefaultCellProps.mMappedDefaultCellStyles;
                }
            }
        }
        return mappedDefaultCellStyle;
    }

    private static CachedTable nextDefaultCellProps(CachedTable tableOps) {
        // check if there is another property set left
        if (tableOps.mColumnDefaultCells != null && tableOps.nextColumnDefaultCellListPos <= (tableOps.mColumnDefaultCells.size() - 1)) {
            // Get the Default Column Cell Style from the column - in ODF only applied on an existing XML cell element without style
            tableOps.mCurrentDefaultCellProps = tableOps.mColumnDefaultCells.get(tableOps.nextColumnDefaultCellListPos);
            tableOps.nextColumnDefaultCellListPos++;
        } else {
            tableOps.mCurrentDefaultCellProps = null;
        }
        return tableOps;
    }

    /**
     * Map the content and the style to a JSONObject containing two JSONObject
     * "value" and "attrs" If there are existing cell styles these take
     * precedence over the default cell style of the column Otherwise if there
     * are no styles and there is a default cell style, apply that..
     */
    private JSONObject getCellJSON(CachedInnerTableOperation cellOperation, Map<String, Object> columnDefaultCellStyle, boolean applyDefaultCellStyle) {

        JSONObject cellRepresentation = null;
        try {
            // if there are existing styles these take precedence over the default cell style of the column
            if (cellOperation.mHardFormattingProperties != null && !cellOperation.mHardFormattingProperties.isEmpty()) {
                if (cellOperation.mHardFormattingProperties.containsKey("cell")) {
                    JSONObject cellProps = (JSONObject) cellOperation.mHardFormattingProperties.get("cell");
                    if (cellProps.hasAndNotNull("value")) {
                        if (cellRepresentation == null) {
                            cellRepresentation = new JSONObject();
                        }
                        cellRepresentation.put("value", cellProps.opt("value"));
                        cellProps.remove("value");
                        if (cellProps.length() == 0) {
                            cellOperation.mHardFormattingProperties.remove("cell");
                        }
                    }
                }
                if (!cellOperation.mHardFormattingProperties.isEmpty()) {
                    JSONObject attrs = new JSONObject();
                    for (String arg : cellOperation.mHardFormattingProperties.keySet()) {
                        attrs.put(arg, cellOperation.mHardFormattingProperties.get(arg));
                    }
                    if (cellRepresentation == null) {
                        cellRepresentation = new JSONObject();
                    }
                    cellRepresentation.put("attrs", attrs);
                } else if (applyDefaultCellStyle) {
                    JSONObject attrs = new JSONObject();
                    for (String arg : columnDefaultCellStyle.keySet()) {
                        attrs.put(arg, columnDefaultCellStyle.get(arg));
                    }
                    if (cellRepresentation == null) {
                        cellRepresentation = new JSONObject();
                    }
                    cellRepresentation.put("attrs", attrs);
                }
                // if there are no styles and there is a default cell style, apply that..
            } else if (applyDefaultCellStyle) {
                JSONObject attrs = new JSONObject();
                for (String arg : columnDefaultCellStyle.keySet()) {
                    attrs.put(arg, columnDefaultCellStyle.get(arg));
                }
                if (cellRepresentation == null) {
                    cellRepresentation = new JSONObject();
                }
                cellRepresentation.put("attrs", attrs);
            }
            // if there is content within the cell element, use this as value, unless there is already a none-null value (given by one of the value attributes, e.g. formula)
            if (cellOperation.mCellContentString != null && cellOperation.mCellContentString.length() > 0 && (cellRepresentation == null || !cellRepresentation.hasAndNotNull("value"))) {
                // SPREADSHEET SVANTE 2DO DATA FORMATS
                //cellRepresentation.put("value", castSpreadsheetCellContent(mSpreadsheetCellContent.toString(), ((TableTableCellElement) root).getOfficeValueTypeAttribute()));
                String cellValue = cellOperation.mCellContentString.toString();
                // if there is a content starting with = we have to mask it to distinguish it from formula
                if (cellRepresentation == null) {
                    cellRepresentation = new JSONObject();
                }
                if (cellValue.startsWith(EQUATION)) {
                    cellRepresentation.put("value", APOSTROPHE + cellValue);
                } else {
                    cellRepresentation.put("value", cellValue);

                }

            }
        } catch (JSONException ex) {
            Logger.getLogger(OdfFileSaxHandler.class
                .getName()).log(Level.SEVERE, null, ex);
        }
        return cellRepresentation;
    }

    /**
     * Supported types by API are: Empty Number String Boolean Error Formula
     */
    // Used later for data formats
    private Object castSpreadsheetCellContent(String cellContent, String cellType) {
        Object castedCellContent = cellContent;
        if (cellType != null && !cellType.equals(VALUE_TYPE_STRING) || cellType.equals(VALUE_TYPE_CURRENCY)) {
            if (cellType.equals(VALUE_TYPE_BOOLEAN)) {
                castedCellContent = Boolean.parseBoolean(cellContent);
            } else if (cellType.equals(VALUE_TYPE_FLOAT)
                || cellType.equals(VALUE_TYPE_PERCENTAGE)) {
                castedCellContent = Double.parseDouble(cellContent);
            } else if (cellType.equals(VALUE_TYPE_TIME)
                || cellType.equals(VALUE_TYPE_DATE)) {
                // we need to convert the values to an Integer
                //ToDo
            } else if (cellType.equals(VALUE_TYPE_VOID)) {
                castedCellContent = null;
            }
        }
        return castedCellContent;
    }

    private void addAttributes(Element element, Attributes attributes) {
        String attrQname;
        String attrURL;
        OdfAttribute attr;
        for (int i = 0; i < attributes.getLength(); i++) {
            attrURL = attributes.getURI(i);
            attrQname = attributes.getQName(i);
            // if no namespace exists
            if (attrURL.equals(EMPTY_STRING) || attrQname.equals(EMPTY_STRING)) {
                // create attribute without prefix
                attr = mFileDom.createAttribute(attributes.getLocalName(i));
            } else {
                if (attrQname.startsWith("xmlns:")) {
                    // in case of xmlns prefix we have to create a new OdfNamespace
                    OdfNamespace namespace = mFileDom.setNamespace(attributes.getLocalName(i), attributes.getValue(i));
                    // if the file Dom is already associated to parsed XML addChild the new namespace to the root element
                    Element root = mFileDom.getRootElement();
                    if (root == null) {
                        root = element;
                    }
                    root.setAttributeNS("http://www.w3.org/2000/xmlns/", "xmlns:" + namespace.getPrefix(), namespace.getUri());
                }
                // create all attributes, even namespace attributes
                attr = mFileDom.createAttributeNS(attrURL, attrQname);
            }

            // namespace attributes will not be created and return null
            if (attr != null) {
                element.setAttributeNodeNS(attr);
                // don't exit because of invalid attribute values
                try {
                    // set Value in the attribute to allow validation in the attribute
                    attr.setValue(attributes.getValue(i));
                } // if we detect an attribute with invalid value: remove attribute node
                catch (IllegalArgumentException e) {
                    element.removeAttributeNode(attr);
                }
            }
        }
    }

    private void flushTextAtStart(String uri, String localName, String qName) {
        flushText(uri, localName, qName, false);
    }

    private void flushTextAtEnd(String uri, String localName, String qName) {
        flushText(uri, localName, qName, true);
    }

    /**
     * Consumers shall collapse white space characters that occur in <ul> <li>a
     * <text:p> or <text:h> element (so called paragraph elements), and </li>
     * <li>in their descendant elements, if the OpenDocument schema permits the
     * inclusion of character data for the element itself and all its ancestor
     * elements up to the paragraph element. </li></ul> Collapsing white space
     * characters is defined by the following algorithm: 1)The following
     * [UNICODE] characters are replaced by a " " (U+0020, SPACE) character:
     * \ue570HORIZONTAL TABULATION (U+0009) \ue570CARRIAGE RETURN (U+000D)
     * \ue570LINE FEED (U+000A) 2)The character data of the paragraph element
     * and of all descendant elements for which the OpenDocument schema permits
     * the inclusion of character data for the element itself and all its
     * ancestor elements up to the paragraph element, is concatenated in
     * document order. 3)Leading " " (U+0020, SPACE) characters at the start of
     * the resulting text and trailing SPACE characters at the end of the
     * resulting text are removed. 4)Sequences of " " (U+0020, SPACE) characters
     * are replaced by a single " " (U+0020, SPACE) character.
     */
    private void flushText(String uri, String localName, String qName, boolean isEndOfElement) {
        // check if there is was text found to be added to the element
        if (mCharsForElement.length() > 0) {
            // every text will be kept from the XML file (e.g. indent)
            String newString = mCharsForElement.toString();
            mCharsForElement.setLength(0);
            Text text = mFileDom.createTextNode(newString);
            if (isEndOfElement && Component.isField(uri, localName)) {
                TextSelection textSelection = mTextSelectionStack.pollLast();
                if (!isBlockedSubTree()) {
                    // Currently only check-box have an UTF-8 square as replacementText
                    TextFieldSelection textFieldSelection = (TextFieldSelection) textSelection;
                    String replacementText = textFieldSelection.getReplacementText();
                    Map<String, Object> attrMap = textFieldSelection.getAttributes();

                    cacheOperation(false, OperationConstants.FIELD, textSelection.getStartPosition(), null, localName, replacementText != null ? replacementText : newString, attrMap, mContextName);
                }
                mCurrentNode.appendChild(text);
            } else {
                // if the text is within a text aware component
                if (mCurrentNode instanceof OdfElement) {
                    // ToDo: Uncertain what with text should happen that is not within a text component? Neglectable by
//                    if ((Component.isTextComponentRoot(mCurrentNode) || Component.isTextComponentRoot(((OdfElement) mCurrentNode).getComponentRoot())) && !isBlockedSubTree() && mWhitespaceStatusStack.size() > 0 && !mWhitespaceStatusStack.getLast().isParagraphIgnored() && !(mCurrentNode instanceof TextNoteCitationElement)) {
                    if ((Component.isTextComponentRoot(mCurrentNode) || Component.isTextComponentRoot(((OdfElement) mCurrentNode).getComponentRoot())) && !isBlockedSubTree()
                            && mWhitespaceStatusStack.size() > 0 &&  !(mCurrentNode instanceof TextNoteCitationElement)) {
                        mComponentDepth++;
                        if (mIsCharsBeginning) {
                            mCharsStartPosition = updateTextPosition();
                            mIsCharsBeginning = false;
                        }
                        mComponentDepth--;
                        // The new charPosition adds the text lenght, but inserted will be without
                        // 1) insertion
                        addText(/*mCachedTableOps, */newString);
                        // the following would cumulate the text of a paragraph to a single large string
                        // mCharsForOperation.append(newString);

                        // muss ich rekursiv die gr\u00f6sse nach oben reichen? f\u00fcr jeden none component descendant? K\u00f6nnte ich in OdfElement implemenentieren!
                        // \u00fcberschreibe alle addChild/delete Funktionalit\u00e4t!
                        // Merge/split/delete Text Funktionalit\u00e4t f\u00fcr alle ELEMENTE? Komponenten m\u00fcssen mit einbezogen werden! Reuse of Recursion -- ACTION KLASSE.operate() aufruf!?!?
                        //					OdfElement element = (OdfElement) mCurrentNode;
                        //					element.appendChild(text);
                    }
                    if (isSpaceElement(mCurrentNode)) {
                        mCurrentNode.getParentNode().appendChild(text);
                    } else {
                        mCurrentNode.appendChild(text);
                    }
                }
            }
        } else {
            if (isEndOfElement && Component.isField(uri, localName)) {
                TextSelection textSelection = mTextSelectionStack.pollLast();
                if (!isBlockedSubTree()) {
                    // Currently only check-box have an UTF-8 square as replacementText
                    TextFieldSelection textFieldSelection = (TextFieldSelection) textSelection;
                    String replacementText = textFieldSelection.getReplacementText();
                    Map<String, Object> attrMap = textFieldSelection.getAttributes();
                    if(replacementText == null) {
                        replacementText = new String();
                    }
                    cacheOperation(false, OperationConstants.FIELD, textSelection.getStartPosition(), null, localName, replacementText, attrMap, mContextName);
                }
            }
        }
        if (mCharsForOperation.length() > 0 && Component.isComponentRoot(uri, localName) && !isSpaceElement(uri, localName)) {
            addText(/*mCachedTableOps, */mCharsForOperation);
        }
    }

    private void addText(CharSequence newText) {
        if (mIsSpreadsheet) {
            if (newText instanceof String) {
                mSpreadsheetCellContent.append(newText);
            } else {
                mSpreadsheetCellContent.append(newText.toString());
            }
        } else {
          cacheOperation(false, OperationConstants.TEXT, mCharsStartPosition, null, newText.toString(), mContextName);
        }
        mCharsForOperation.setLength(0);
        mIsCharsBeginning = true;
    }

    static boolean isSpaceElement(Node node) {
        return node instanceof TextSElement;
    }

    static boolean isSpaceElement(String uri, String localName) {
        return uri != null && uri.equals(TextSElement.ELEMENT_NAME.getUri()) && localName.equals(TextSElement.ELEMENT_NAME.getLocalName());
    }

    @Override
    /**
     * http://xerces.apache.org/xerces2-j/faq-sax.html#faq-2 : SAX may deliver
     * contiguous text as multiple calls to characters, for reasons having to do
     * with parser efficiency and input buffering. It is the programmer's
     * responsibility to deal with that appropriately, e.g. by accumulating text
     * until the next non-characters event. This method will finalize the text
     * of an element, by flushing/appending it to the element node. It is called
     * at the beginning of startElement/endElement. In case of startElement the
     * text will be referred to the previous element node (before the new
     * started). In case of endElement the text will be referred to the current
     * element node.
     */
    public void characters(char[] ch, int startPosition, int length) throws SAXException {
        if (mCurrentComponent instanceof TextContainer) {
            // ODF Whitespacehandling
            WhitespaceStatus currentWhiteSpaceStatus = mWhitespaceStatusStack.getLast();
            // Note: The delta between startPosition and endPosition marks the text to be written out
            // startPosition will only be raised to endposition, when characters have to be skipped!
            int endPosition = startPosition;
            int lastPos = startPosition + length;
            boolean previousContentWritten = false;
            char c;
            // Go through all characters found by the parser..
            for (int i = startPosition; i < lastPos; i++) {
                c = ch[i];
                // first part is trimming in the beginning of the element
                if (currentWhiteSpaceStatus.hasOnlyWhiteSpace()) {
                    // \t (tabulator = 0x09)
                    if (c == '\u0020' // space
                        || c == '\t'
                        // \r (carriage return = 0x0D)
                        || c == '\r'
                        // \n (line feed = 0x0A)
                        || c == '\n') {
                        // skipt this character, keeping the difference between start & end (length) equal
                        startPosition++;
                        endPosition++;
                    } else {
                        // first character being found worth to be written
                        currentWhiteSpaceStatus.setOnlyWhiteSpace(false);
                        endPosition++;
                    }
                    // second part is about collapsing multiple whitespaces
                } else {
                    if (c == '\u0020' // space
                        || c == '\t' // \t (tabulator = 0x09)
                        // \r (carriage return = 0x0D)
                        || c == '\r'
                        // \n (line feed = 0x0A)
                        || c == '\n') {
                        // if we have aleady a preceding whitespace character
                        if (currentWhiteSpaceStatus.hasSpaceBefore()) {
                            if (!previousContentWritten) {
                                // as we have to skip a character in the array, write what we have
                                if (endPosition - startPosition > 0) {
                                    mCharsForElement.append(ch, startPosition, endPosition - startPosition);
                                }
                                previousContentWritten = true;
                            }
                            // NOT including this character
                            endPosition++;
                            startPosition = endPosition;
                        } else {
                            currentWhiteSpaceStatus.setFirstSpaceCharPosition(i);
                            ch[i] = '\u0020'; // overwrite all
                            endPosition++;
                        }
                    } else {
                        if (currentWhiteSpaceStatus.hasSpaceBefore()) {
                            currentWhiteSpaceStatus.setFirstSpaceCharPosition(-1);
                        }
                        endPosition++; // including this character
                    }
                }
            }
            if (endPosition - startPosition > 0) {
                mCharsForElement.append(ch, startPosition, endPosition - startPosition);
            }
        } else {
            /*
             * ToDo: The following will be ignored for now:
             In addition, OpenDocument Consumers shall ignore all element children ([RNG] section 5,
             Data Model) of elements defined in this specification that are strings consisting entirely of whitespace characters and
             which do not satisfy a pattern of the OpenDocument schema definition for the element.
             */
            // See http://docs.oasis-open.org/office/v1.2/os/OpenDocument-v1.2-os-part1.html#a3_18White_Space_Processing_and_EOL_Handling
            mCharsForElement.append(ch, startPosition, length);
        }
    }

    @Override
    public InputSource resolveEntity(String publicId, String systemId) throws IOException, SAXException {
        return super.resolveEntity(publicId, systemId);
    }

    /**
     * This method is being called whenever an element being the root element is
     * found, within startElement function. It considers mComponentDepth and the
     * size of the list mLastComponentPositions. It updates the position (ie.
     * list mLastComponentPositions) by adding a child or a sibling. This ONLY
     * works if mComponentDepth has already been updated before this method is
     * being called. Currently at the beginning of startElement, when realizing
     * it is a component root element.
     *
     * The only reason to update the position (mLastComponentPositions) outside
     * of this function is to handle attributes representing repeatedColumns
     * components	* (i.e. for cells & rows).
     *
     * @return the path of the current found component as Integer array
     */
    private List<Integer> updateComponentPosition() {
        List<Integer> pos = updatePosition(true);
        return pos;
    }

    /**
     * This method is being called whenever an element being the root element is
     * found, within startElement function. It considers mComponentDepth and the
     * size of the list mLastComponentPositions. It updates the position (ie.
     * list mLastComponentPositions) by adding a child or a sibling. This ONLY
     * works if mComponentDepth has already been updated before this method is
     * being called. Currently at the beginning of startElement, when realizing
     * it is a component root element.
     *
     * The only reason to update the position (mLastComponentPositions) outside
     * of this function is to handle attributes representing repeatedColumns
     * components	* (i.e. for cells & rows).
     *
     * @return the path of the current found component as Integer array
     */
    private List<Integer> updateTextPosition() {
        return updatePosition(false);
    }

    /**
     * This method is being called whenever an element being the root element is
     * found, within startElement function. It considers mComponentDepth and the
     * size of the list mLastComponentPositions. It updates the position (ie.
     * list mLastComponentPositions) by adding a child or a sibling. This ONLY
     * works if mComponentDepth has already been updated before this method is
     * being called. Currently at the beginning of startElement, when realizing
     * it is a component root element.
     *
     * The only reason to update the position (mLastComponentPositions) outside
     * of this function is to handle attributes representing repeatedColumns
     * components	* (i.e. for cells & rows).
     *
     * @return the path of the current found component as Integer array
     */
    private List<Integer> updatePosition(boolean isComponent) {

        /**
         * Used components being siblings of text, where the position is
         * determined by the text count, not by the previous component number
         */
        // take care of position handling
        // NEW COMPONENT CHILD: If the component Level is deeper than the last position
        // Actually at the first component: componentDepth start with 0, but the size with 1
        if (mComponentDepth == mLastComponentPositions.size()) {
            // addChild a new level
            mLastComponentPositions.add(mComponentDepth, 0);

            // NEW SIBLING: If the depth is "equal" (ie. lower) than addChild a sibling
        } else if (mComponentDepth == mLastComponentPositions.size() - 1) {
            int positionUpdate;
            if (mCurrentComponent instanceof TextContainer) {
                // increment the last position number as a sibling was added
                positionUpdate = mCurrentComponent.size();
            } else if (mCurrentComponent instanceof Cell) {
                positionUpdate = mLastComponentPositions.get(mComponentDepth) + 1;
//            } else if (mCurrentComponent instanceof Table) {
//                positionUpdate =mCurrentComponent.size();
            } else {
                positionUpdate = mLastComponentPositions.get(mComponentDepth) + 1;
            }
            mLastComponentPositions.set(mComponentDepth, positionUpdate);
// FINISHED COMPONENT - REMOVE DEPTH - If a component was closed and the position needds to be added
        } else if (mComponentDepth < mLastComponentPositions.size() - 1) {
            // remove the last position and addChild a new one
            mLastComponentPositions.removeLast();
            updatePosition(isComponent);
        } else {
            LOG.warning("Houston, we have a problem..");
        }
        //ToDo: Do I need a new LIST for every component? Or may I addChild a position to the component?
        return new LinkedList<Integer>(mLastComponentPositions);
        //ToDo: Below a bad idea?
        //return Collections.unmodifiableList(mLastComponentPositions);
    }

    /**
     * @return the path of the current found component as Integer array
     */
    private List<Integer> getTextPosition() {
        // The text is one level further down
        mComponentDepth++;
        List<Integer> position = updateTextPosition();
        mComponentDepth--;
        return position;
    }

    /**
     *
     * @return a property value.
     */
//	private String getPropertyValue(OdfStyleProperty prop) {
//		String value = null;
//
//		OdfStylePropertiesBase properties = getPropertiesElement(prop.getPropertySet());
//		if (properties != null) {
//			if (properties.hasAttributeNS(prop.getName().getUri(), prop.getName().getLocalName())) {
//				return properties.getOdfAttribute(prop.getName()).getValue();
//			}
//		}
//
//		OdfStyleBase parent = getParentStyle();
//		if (parent != null) {
//			return parent.getProperty(prop);
//		}
//
//		return value;
//	}
    public static String getProperty(OdfStyleProperty prop, OdfStylableElement element) {
        String value = null;
        OdfStyle style = element.getAutomaticStyle();
        if (style == null) {
            element.getOrCreateUnqiueAutomaticStyle();
            style = element.getAutomaticStyle();
            if (style == null) {
                style = element.getDocumentStyle();
            }
        }
        if (style != null) {
            value = style.getProperty(prop);
        }
        return value;
    }

    private static Length getPropertyLength(OdfStyleProperty prop, OdfStylableElement element) {
        Length length = null;
        String propValue = getProperty(prop, element);
        if (propValue != null && !propValue.isEmpty()) {
            length = new Length(propValue);
        }
        return length;

    }

    /**
     * This class compares alphanumeric Strings consistent across multiple OS
     * and JDK. Required to sort Styles within a List to have a consistent order
     * of createStyle operations in regression tests.
     */
    private static class AlphanumComparator implements Comparator {

        @Override
        public int compare(Object o1, Object o2) {
            if (!(o1 instanceof String)) {
                o1 = o1.toString();
            }

            if (!(o2 instanceof String)) {
                o2 = o2.toString();
            }
            String s1 = (String) o1;
            String s2 = (String) o2;

            int thisMarker = 0;
            int thatMarker = 0;
            int s1Length = s1.length();
            int s2Length = s2.length();

            while (thisMarker < s1Length && thatMarker < s2Length) {
                String thisPart = getPart(s1, s1Length, thisMarker);
                thisMarker += thisPart.length();

                String thatPart = getPart(s2, s2Length, thatMarker);
                thatMarker += thatPart.length();

                // If both parts contain numeric characters, sort them numerically
                int result = 0;
                if (isDigit(thisPart.charAt(0)) && isDigit(thatPart.charAt(0))) {
                    // Simple parts comparison by length.
                    int thisPartLength = thisPart.length();
                    result = thisPartLength - thatPart.length();
                    // If equal, the first different number counts
                    if (result == 0) {
                        for (int i = 0; i < thisPartLength; i++) {
                            result = thisPart.charAt(i) - thatPart.charAt(i);
                            if (result != 0) {
                                return result;
                            }
                        }
                    }
                } else {
                    result = thisPart.compareTo(thatPart);
                }

                if (result != 0) {
                    return result;
                }
            }
            return s1Length - s2Length;
        }

        /**
         * Length of string for improved efficiency (only calculated once) *
         */
        private String getPart(String s, int slength, int marker) {
            StringBuilder part = new StringBuilder();
            char c = s.charAt(marker);
            part.append(c);
            marker++;
            if (isDigit(c)) {
                while (marker < slength) {
                    c = s.charAt(marker);
                    if (!isDigit(c)) {
                        break;
                    }
                    part.append(c);
                    marker++;
                }
            } else {
                while (marker < slength) {
                    c = s.charAt(marker);
                    if (isDigit(c)) {
                        break;
                    }
                    part.append(c);
                    marker++;
                }
            }
            return part.toString();
        }

        private boolean isDigit(char ch) {
            return ch >= 48 && ch <= 57;
        }
    }
    //CachedTable mCachedTableOps = null;

    // based on document position the tables and subtables are being flushed after each end
//    HashMap<List<Integer>, CachedTable> mAllTables = null;
//    private ArrayList<CachedTable> mTableStack;
//    boolean mWithinTable = false; //TODO: Detect via type of top element of mComponentStack
    //private int mNumberOfNestedTables = 0;

    	class CachedTable extends CachedComponent {
			private static final long serialVersionUID = 1L;
		private ArrayList<CachedTable> mTableStack = new ArrayList<CachedTable>();
	    HashMap<List<Integer>, CachedTable> mAllTables = new HashMap<List<Integer>, CachedTable>();
        // START *** SPREADSHEET PROPERTIES
        // required to track repeatedColumns rows for spreadsheets..
        int mPreviousRepeatedRows = 0;
        int mRepeatedRowsOnly = 0;
        // spreadsheet range, representing a subset of a single row.
        // added cell data (content & format) neglecting starting/ending empty cells and within the row any adjacent empty cells above MAX_REPEATING_EMPTY_CELLS
        JSONArray mCurrentRange;
        // only cells with content should trigger an operatoin
        int mFirstContentCellNo = -1;
        // equal cell contents can be given out compressed
        int mFirstEqualCellNo = -1;
        int mCurrentColumnNo = 0;
        // after there had been content, three following empty cells are being counted, before a new insert operation is being triggered
        int mEmptyCellCount = 0;
        int mFirstEmptyCell = -1;
        // the JSONArray does not allow to delete empty cells, therefore they have to be counted
        JSONObject mPreviousCell = null;
        private int mCellRepetition = 1;
        // get the sheetNo
        Integer mSheetNo;
        Integer mFirstRow = 0;
        Integer mLastRow = 0;
        ArrayList<CachedInnerTableOperation> mCachedTableContentOps = null;
        // 1) Mapping the column's default cell style to empty cells for none max size OR
        // adding column's default cell style to column style
        // 2) Mapping the largest range of rows/columns to the table in case of MAX sheet
        Integer rowCount = null;
        Integer columnCount = null;

        // We want to compress two identical following row format operations
        Boolean lastRowIsVisible = Boolean.TRUE;
        String lastRowStyleName = null;
        int lastRowRepetition = 0;
        CachedInnerTableOperation lastRowFormatOperation = null;

        // List of the column properties with default cell styles
        List<ColumnDefaultCellProps> mColumnDefaultCells = null;
        ColumnDefaultCellProps mCurrentDefaultCellProps = null;
        int nextColumnDefaultCellListPos = 0;
        String mPreviousColumnCellStyleID = null;
        // Boolean object as three states: There is a default style and all columns covered with dfault style (TRUE), there is a default style, but sometimes no style (FALSE) and no "Default" style (NULL)
        Boolean mHasOnlyDefaultColumnCellStyles = null;

        // Temporary map to count the most used column style, to move this in case of full column count to the table style
        Map<String, Integer> columnStyleOccurrence = null;
        // the name of the column style being used most in the table
        String mMostUsedColumnStyle = null;

        // Temporary map to count the most used row style, to move this in case of full row count to the table style
        Map<String, Integer> rowStyleOccurrence = null;
        // the name of the column style being used most in the table
        String mMostUsedRowStyle = null;

        // END *** SPREADSHEET PROPERTIES
        int mColumnCount = 0;
        int mRowCount = 0;
        boolean mWhiteSpaceOnly = true;
        int mCellCount = 0;
        boolean mIsTooLarge = false;
        List<Integer> mTableGrid;
        List<Integer> mStart;
        String tableName;

        CachedTable() {
            super();
        }
        public int getSubTableCount() {
            return mTableStack.size();
        }
        public void addSubTable(CachedTable subTable, List<Integer> position) {
            //TODO: Do we need two containers?
            mTableStack.add(subTable);
            mAllTables.put(position, subTable);
        }
        public CachedTable  getSubTable(List<Integer> position) {
            return mAllTables.get(position);
        }
        public void removeSubTable() {
            mTableStack.remove(mTableStack.size() - 1);
        }
    }

    Stack<CachedComponent> mComponentStack = new Stack<CachedComponent>();
    //cache objects bound to page that are on top level of the document
    ArrayDeque<ShapeProperties> m_cachedPageShapes = new ArrayDeque<ShapeProperties>();
    boolean mPageBoundObjectsRelocated = false;
    HashMap<String, Integer> mTopLevelTables = new HashMap<String, Integer>();//mapping of spreadsheet index to their names

    /**
     * The column ranges with a default cell style will be memorized. Within a
     * sheet with maximum numer of rows, the default cell style can be kept as
     * cell style on the column insertion. As within the operational table model
     * the table, row and column styles are only affecting cells that have
     * neither content nor style, all default cell styles have to applied to
     * cells with content without style. Repeated cells might get be split due
     * to changing default cell styles. Instead of altering the XML model the
     * method to collect the cell range operation will be triggered for every
     * change of default cell style.
     */
    class ColumnDefaultCellProps {

        /**
         * position starts counting with 0. It becomes necessary as there might
         * be gaps in between columns due to missing default cell styles at the
         * column.
         */
        int mColumnStartPos = 0;
        /**
         * the default mColumnRepetition is 1
         */
        int mColumnRepetition = 1;
        String mDefaultCellStyleId = null;
        Map<String, Object> mMappedDefaultCellStyles = null;
    }

    private void cacheOperation(boolean fillCacheOnly, String componentType, List<Integer> start, Map<String, Object> hardFormattingProperties, Object... componentProperties) {
    	if(mComponentStack.empty()) {
    		//send to producer
    		if( componentType.equals(OperationConstants.TEXT)) {
                String text = (String) componentProperties[0];
                String context = (String) componentProperties[1];
                mJsonOperationProducer.addText(start, text, context);
            } else if( componentType.equals(OperationConstants.PARAGRAPH)){
                String context = (String) componentProperties[0];
                mJsonOperationProducer.add(componentType, start, hardFormattingProperties, context);
    		} else if( componentType.equals(OperationConstants.TABLE)){
    		    List<Integer> tableGrid = (List<Integer>) componentProperties[0];
    		    String tableName = (String)componentProperties[1];
    		    boolean isSpreadsheet = (Boolean)componentProperties[2];
    		    String context = (String)componentProperties[3];
    		    mJsonOperationProducer.addTable(start, hardFormattingProperties, tableGrid, tableName, isSpreadsheet, context);
    		    if(isSpreadsheet) {
    		        mTopLevelTables.put(tableName, start.get(0));
    		    }
            } else if( componentType.equals(OperationConstants.EXCEEDEDTABLE)){
                int columns = (Integer)componentProperties[0];
                int rows = (Integer)componentProperties[1];
                List<Integer> tableGrid = (List<Integer>) componentProperties[2];
                //addExceededTable(final List<Integer> start, int columns, int rows, final List<Integer> tableGrid) {
                mJsonOperationProducer.addExceededTable(start, columns, rows, tableGrid);
            } else if( componentType.equals(OperationConstants.ATTRIBUTES)){
                List<Integer> end = (List<Integer> ) componentProperties[0];
                String context = (String) componentProperties[1];
                mJsonOperationProducer.setAttributes(start, end, hardFormattingProperties, context);
            } else if( componentType.equals(OperationConstants.FORMATROWS)){
                Integer firstRow = (Integer)componentProperties[0];
                Integer lastRow = (Integer)componentProperties[1];
                Integer repeatedRowOffset = (Integer)componentProperties[2];
                String context = (String) componentProperties[3];
                mJsonOperationProducer.formatRows(start, hardFormattingProperties, firstRow, lastRow, repeatedRowOffset, context);
            } else if( componentType.equals(OperationConstants.REMOVEFORMAT)){
                mJsonOperationProducer.removeFormat(start);
            } else if( componentType.equals(OperationConstants.FORMATCOLUMNS)){
                Integer firstColumn = (Integer)componentProperties[0];
                Integer lastColumn = (Integer)componentProperties[1];
                String context = (String) componentProperties[2];
                mJsonOperationProducer.formatColumns(start, hardFormattingProperties, firstColumn, lastColumn, context);
            } else if( componentType.equals(OperationConstants.SHAPE)||componentType.equals(OperationConstants.GROUP)){
                String context = (String) componentProperties[0];
                mJsonOperationProducer.addShape(start, hardFormattingProperties, context, componentType.equals(OperationConstants.GROUP));
            } else if( componentType.equals(OperationConstants.IMAGE)){
                String context = (String) componentProperties[0];
                mJsonOperationProducer.addImage(start, hardFormattingProperties, context);
            } else if( componentType.equals(OperationConstants.FIELD)){
                String fieldType = (String) componentProperties[0];
                String fieldContent = (String) componentProperties[1];
                Map<String, Object> fieldAttributes = (Map<String, Object>) componentProperties[2];
                String context = (String) componentProperties[3];
                mJsonOperationProducer.insertField(start, fieldType, fieldContent, fieldAttributes, context);
            } else if( componentType.equals(OperationConstants.AUTOFILTER)){
                List<Integer> end = (List<Integer>) componentProperties[0];
                Integer sheet = (Integer)componentProperties[1];
                mJsonOperationProducer.insertAutoFilter(start, end, sheet, hardFormattingProperties);
            } else if( componentType.equals(OperationConstants.AUTOFILTERCOLUMN)){
                Integer sheet = (Integer)componentProperties[0];
                List<String> entries = (List<String>) componentProperties[1];
                mJsonOperationProducer.insertAutoFilterColumn(start, sheet, entries);
            } else {
        		mJsonOperationProducer.add(componentType, start, hardFormattingProperties, mContextName);
    		}
    	} else {
	    	CachedComponent topComponent = mComponentStack.peek();
	    	if(!fillCacheOnly && topComponent instanceof CachedTable) {
	    		cacheTableOperation(componentType, start, hardFormattingProperties, componentProperties);
	    	} else {
	    		//collect the operations at the CachedComponent
	            LinkedList<Integer> position = null;
	            if (start != null) {
	                position = new LinkedList<Integer>(start);
	            }

	            topComponent.add(new CachedOperation(componentType, position, hardFormattingProperties, componentProperties));
	    	}
    	}
    }

    	/**
     * Table operation are being cached in one of two caches
     */
    private void /*CachedTable */cacheTableOperation(/*CachedTable currentTable,*/ String componentType, List<Integer> start, Map<String, Object> hardFormattingProperties, Object... componentProperties) {
        LinkedList<Integer> position = null;
        CachedTable currentTable = mComponentStack.empty() ? null : (mComponentStack.peek() instanceof CachedTable) ? (CachedTable)mComponentStack.peek() : null;
        if (start != null) {
            position = new LinkedList<Integer>(start);
        }
        if (!mIsSpreadsheet && componentType.equals(OperationConstants.CELLS)) { // does not effect a spreadsheet
            currentTable.mCellCount++;
            if (currentTable.mCellCount > mMaxAllowedCellCount) {
                currentTable.mIsTooLarge = true;
            }
        } else if (componentType.equals(OperationConstants.ROWS)) { // Counting Rows
            if (mIsSpreadsheet) {
                // getting the last end position (plus one as counting starts with 0)
                currentTable.mRowCount += currentTable.mRepeatedRowsOnly + 1;
                if (currentTable.mCachedTableContentOps == null) {
                    currentTable.mCachedTableContentOps = new ArrayList<CachedInnerTableOperation>();
                }
                // adds the cached style name to the table cache
                if (currentTable.rowStyleOccurrence == null) {
                    currentTable.rowStyleOccurrence = new HashMap();
                }
                countStyle((String) componentProperties[4], currentTable.rowStyleOccurrence, (Integer) componentProperties[5]);
            } else {
                currentTable.mRowCount++; // no repeated in writer
                if (currentTable.mRowCount > mMaxAllowedRowCount) {
                    currentTable.mIsTooLarge = true;
                }
            }
        } else if (mIsSpreadsheet && componentType.equals(OperationConstants.COLUMNS)) { // in a text document the columns are instead defined by an array of interger (widths)
            if (componentProperties != null) {
                if (currentTable.mColumnDefaultCells == null) {
                    //  firstColumn, mColumnCount - 1, mappedDefaultCellStyleProperties, repeatedColumns, isVisible, styleName
                    currentTable.mColumnDefaultCells = new ArrayList<ColumnDefaultCellProps>();
                }
                String defaultCellStyleId = (String) componentProperties[6];
                if (defaultCellStyleId != null) {
                    if (currentTable.mPreviousColumnCellStyleID != null && defaultCellStyleId.equals(currentTable.mPreviousColumnCellStyleID)) {
                        ColumnDefaultCellProps columnDefaultCellProps = currentTable.mColumnDefaultCells.get(currentTable.mColumnDefaultCells.size() - 1);
                        columnDefaultCellProps.mColumnRepetition += (Integer) componentProperties[3];
                    } else {
                        ColumnDefaultCellProps columnDefaultCellProps = new ColumnDefaultCellProps();
                        columnDefaultCellProps.mColumnRepetition = (Integer) componentProperties[3];
                        columnDefaultCellProps.mMappedDefaultCellStyles = ((Map<String, Object>) componentProperties[2]);
                        columnDefaultCellProps.mColumnStartPos = (Integer) componentProperties[0];
                        columnDefaultCellProps.mDefaultCellStyleId = defaultCellStyleId;
                        currentTable.mColumnDefaultCells.add(columnDefaultCellProps);
                        // If all columns have a style, we can move the column default cell style to the table
                        if (defaultCellStyleId.equals(COLUMN_CELL_DEFAULT_STYLE) && currentTable.mHasOnlyDefaultColumnCellStyles == null) {
                            currentTable.mHasOnlyDefaultColumnCellStyles = Boolean.TRUE;
                        }
                        currentTable.mPreviousColumnCellStyleID = defaultCellStyleId;
                    }

                    // adds the row style name to the table cache to find out the most repeated row style and move that to the table
                    if (currentTable.columnStyleOccurrence == null) {
                        currentTable.columnStyleOccurrence = new HashMap();
                    }
                    countStyle((String) componentProperties[5], currentTable.columnStyleOccurrence, (Integer) componentProperties[3]);
                } else {
                    // If there one column without style, we are not able to map the DEFAULT to the table, otherwise there might be two defaults (from application and from "Default" style)
                    currentTable.mHasOnlyDefaultColumnCellStyles = Boolean.FALSE;
                }
            }
            // the end position will be kept
            currentTable.mColumnCount = (Integer) componentProperties[1] + 1;
            currentTable.add(new CachedInnerTableOperation(componentType, position, hardFormattingProperties, componentProperties));
        } else if (componentType.equals(OperationConstants.RANGE)) { // spreadsheet only
            currentTable.mCachedTableContentOps.add(new CachedInnerTableOperation(componentType, position, hardFormattingProperties, componentProperties));
        } else if (componentType.equals(OperationConstants.TABLE)) { // all formats (Text & Spreadsheet atm)
            List<Integer> tableGrid = (List<Integer>) componentProperties[0];
            CachedTable newCachedTable = startTableSizeEvaluation(position, tableGrid);
            if (newCachedTable.mTableGrid != null) {
                if (newCachedTable.mTableGrid.size() > mMaxAllowedColumnCount) {
                    newCachedTable.mIsTooLarge = true;
                    // adding the table now, as it would not be below as already too large..
                } else {
                    newCachedTable.mColumnCount = newCachedTable.mTableGrid.size();
                }
            }
            // if it is the first root table
            if (currentTable == null) {
                // only for the root table the table itself will be added at the beginning
                currentTable = newCachedTable;
//                ++currentTable.mNumberOfNestedTables;
                mComponentStack.push(currentTable);
                currentTable.add(new CachedInnerTableOperation(componentType, position, hardFormattingProperties, componentProperties));
            } else { // as subtable the table will be added twice:
                // once for the parent as notifier
                currentTable.addSubTable(newCachedTable, start);
                currentTable.add(new CachedInnerTableOperation(componentType, position, hardFormattingProperties, componentProperties));
                currentTable = newCachedTable;
                // once for the child to create
                currentTable.add(new CachedInnerTableOperation(componentType, position, hardFormattingProperties, componentProperties));
            }
        }

        if (!currentTable.mIsTooLarge && !componentType.equals(OperationConstants.TABLE) && !componentType.equals(OperationConstants.COLUMNS) && !componentType.equals(OperationConstants.RANGE)) {
            if (mIsSpreadsheet && (componentType.equals(OperationConstants.CELLS) || componentType.equals(OperationConstants.ROWS))) {
                // All operations are being added to the queue responsible to create the content (ranges)
                currentTable.mCachedTableContentOps.add(new CachedInnerTableOperation(componentType, position, hardFormattingProperties, componentProperties));
                // Row operation are being added twice to an additional queue to be evaluated prior the content (formatRows)
                if (componentType.equals(OperationConstants.ROWS)) {
                    // if this row style is equal to the previous row style
                    Boolean isVisible = (Boolean) componentProperties[3];
                    String styleName = (String) componentProperties[4];
                    if (styleName != null && currentTable.lastRowStyleName != null && styleName.equals(currentTable.lastRowStyleName) && currentTable.lastRowIsVisible.equals(isVisible)) {
                        currentTable.lastRowRepetition += (Integer) componentProperties[5];
                        currentTable.lastRowFormatOperation.mComponentProperties[1] = (Integer) currentTable.lastRowFormatOperation.mComponentProperties[0] + currentTable.lastRowRepetition - 1;
                    } else {
                        currentTable.lastRowRepetition = (Integer) componentProperties[5];
                        currentTable.lastRowStyleName = styleName;
                        currentTable.lastRowIsVisible = isVisible;
                        // save this row style (clone the componentProperties otherwise the altering will have affect on the other queue!)
                        currentTable.lastRowFormatOperation = new CachedInnerTableOperation(componentType, position, hardFormattingProperties, componentProperties.clone());
                        currentTable.add(currentTable.lastRowFormatOperation);
                    }
                }
            } else if( !mIsSpreadsheet ||
                    (!componentType.equals(OperationConstants.HARD_BREAK) && !componentType.equals(OperationConstants.TAB)
                            && !componentType.equals(OperationConstants.PARAGRAPH) )){ // in spreadsheet paragraphs, text content, format and text breaks are given via cell range
                currentTable.add(new CachedInnerTableOperation(componentType, position, hardFormattingProperties, componentProperties));
                // HERE I could reference to the start operation (might go till the same componenttype (or higher) is reached? What about repeated Subtables?`
//                if(mCachedTableOps.mCellRepetition  > 1){
//                    // START REPEATED RECORDING MODE!!
//                    System.out.println("START" + currentTable.size());
//                }
            }
        }
//        return currentTable;
    }

    /**
     * Caches the most style to the table cache to identify laster the most used
     * style
     */
    private static void countStyle(String styleName, Map<String, Integer> styleOccurrences, Integer repetition) {
        if (styleName != null) {
            if (styleOccurrences == null) {
                styleOccurrences = new HashMap<String, Integer>();
            }
            Integer styleCount = styleOccurrences.get(styleName);
            if (styleCount == null) {
                styleOccurrences.put(styleName, repetition);
            } else {
                styleOccurrences.put(styleName, styleCount + repetition);
            }
        }
    }

    /**
     * According to user run-time configuration only tables of a certain size
     * are allowed to be created. Tables exceeding the limit are being shown by
     * a replacement object, otherwise the client performance might not be
     * sufficient.
     */
    private CachedTable startTableSizeEvaluation(List<Integer> position, List<Integer> tableGrid) {
        CachedTable cachedTable = null;

        cachedTable = new CachedTable();

        cachedTable.mTableGrid = tableGrid;
        return cachedTable;
    }

    /**
     * According to user run-time configuration only tables of a certain size
     * are allowed to be created. Tables exceeding the limit are being shown by
     * a replacement object, otherwise the client performance might not be
     * sufficient.
     *
     * As the limit is being checked on sub table level, the complete table have
     * to be parsed before giving green light for any table. On the opposite, if
     * a subtable is already too large, it can be neglected collecting
     * operations for that subtable.
     * @throws SAXException
     */
    private void endTableSizeEvaluation() throws SAXException {

        CachedTable cachedTableOps = (CachedTable) mComponentStack.peek();
        if (cachedTableOps.getSubTableCount() == 0) {
            cachedTableOps.mMostUsedColumnStyle = getMostUsedStyle(cachedTableOps.columnStyleOccurrence);
            cachedTableOps.mMostUsedRowStyle = getMostUsedStyle(cachedTableOps.rowStyleOccurrence);
            mComponentStack.pop();
            flushTableOperations(cachedTableOps, true);
            if (cachedTableOps != null && cachedTableOps.mCachedTableContentOps != null) {
                cachedTableOps.mCachedTableContentOps = null;
                cachedTableOps.lastRowFormatOperation = null;
            }
        } else if (cachedTableOps.getSubTableCount() > 0) {
            // when leaving a table, continue with the parent table
            cachedTableOps.removeSubTable();
        } else { // below zero might appear, when table had started in blocked area
            //TODO: is it really possible to reach this point?
            if (cachedTableOps != null && cachedTableOps.mCachedTableContentOps != null) {
                cachedTableOps.mCachedTableContentOps = null;
                cachedTableOps.lastRowFormatOperation = null;
            }
            mComponentStack.pop();
        }
    }

    private void flushTableOperations(CachedTable currentTable, boolean isStartOfTable) throws SAXException {

        boolean putPageBreak = false;
        boolean isBreakBefore = true;
        ListIterator<CachedOperation> cachedOperationIterator = currentTable.listIterator();
        while (cachedOperationIterator.hasNext()) {
            CachedOperation operation = cachedOperationIterator.next();
            if ( operation instanceof CachedInnerTableOperation && operation.mComponentType.equals(OperationConstants.TABLE)) {
                if (isStartOfTable) {
                    isStartOfTable = false;
                    if (currentTable.mIsTooLarge) {
                        // replacement table
                        cacheOperation(false, OperationConstants.EXCEEDEDTABLE, operation.mStart, null, ((List) operation.mComponentProperties[0]).size(), currentTable.mRowCount, operation.mComponentProperties[0]);
                        break;
                    } else {
                        if (currentTable.mRowCount > mMaxAllowedRowCount || currentTable.mColumnCount > mMaxAllowedColumnCount || currentTable.mCellCount > mMaxAllowedCellCount) {
                            //TODO: Exceeded table operation name
                            cacheOperation(false, OperationConstants.EXCEEDEDTABLE, operation.mStart, null, ((List) operation.mComponentProperties[0]).size(), currentTable.mRowCount, operation.mComponentProperties[0]);
                            break;
                        } else {
                            if (mIsSpreadsheet) {
                                if( mTopLevelTables.size() >= mMaxAllowedSheetCount ) {
                                    throw new SAXException("Too many sheets");
                                }

                                if (currentTable.mColumnCount == Table.MAX_COLUMN_NUMBER_CALC || currentTable.mColumnCount == Table.MAX_COLUMN_NUMBER_EXCEL) {
                                    if (currentTable.mRowCount == Table.MAX_ROW_NUMBER) {
                                        Map mappedColumnStyleProperties = getMappedStyleProperties(currentTable.mMostUsedColumnStyle, OdfStyleFamily.TableColumn);
                                        operation.mHardFormattingProperties = addTableStyles(mappedColumnStyleProperties, operation.mHardFormattingProperties);
                                    }
                                    // if there is a DEFAULT Cell style
                                    if (currentTable.mHasOnlyDefaultColumnCellStyles != null && currentTable.mHasOnlyDefaultColumnCellStyles) {
                                        // Add the default style to the column
                                        operation.mHardFormattingProperties.put(("styleId"), COLUMN_CELL_DEFAULT_STYLE);
                                    }
                                }
                                if (currentTable.mRowCount == Table.MAX_ROW_NUMBER) {
                                    Map mappedRowStyleProperties = getMappedStyleProperties(currentTable.mMostUsedRowStyle, OdfStyleFamily.TableRow);
                                    operation.mHardFormattingProperties = addTableStyles(mappedRowStyleProperties, operation.mHardFormattingProperties);
                                }
                            }
                            // the last parameter are: mColumnRelWidths, mTableName, mIsTableVisible);
                            JSONObject tableAttr = null;
                            if( operation.mHardFormattingProperties.containsKey("table") && (((tableAttr = (JSONObject)operation.mHardFormattingProperties.get("table")).has("pageBreakBefore")) || tableAttr.has("pageBreakAfter"))) {
                                try {
                                    isBreakBefore = tableAttr.has("pageBreakBefore");
                                    String breakString = isBreakBefore ? "pageBreakBefore" : "pageBreakAfter";
                                    boolean breakAttr = tableAttr.getBoolean( breakString );
                                    if(breakAttr){
                                        putPageBreak = true;
                                    }
                                    tableAttr.remove(breakString);
                                } catch (JSONException e) {
                                    //no handling required
                                }
                            }
                            cacheOperation(false, OperationConstants.TABLE, operation.mStart, operation.mHardFormattingProperties, operation.mComponentProperties[0], operation.mComponentProperties[1], mIsSpreadsheet, mContextName);
                        }
                    }
                } else {
                    flushTableOperations(currentTable.getSubTable(operation.mStart), true);
                }
            } else if (operation.mComponentType.equals(OperationConstants.TEXT)) {
                cacheOperation(false, operation.mComponentType, operation.mStart, null, operation.mComponentProperties[0], mContextName);
            } else if (operation.mComponentType.equals(OperationConstants.ATTRIBUTES)) {
                cacheOperation(false, OperationConstants.ATTRIBUTES, operation.mStart, operation.mHardFormattingProperties, operation.mComponentProperties[0], mContextName);
            } else if (mIsSpreadsheet && operation.mComponentType.equals(OperationConstants.ROWS)) { // spreadsheet function
                // instead of counting the correct rows, the repeatedColumns offset will be added to the cell position
                int firstRow = (Integer) operation.mComponentProperties[0];
                int lastRow = (Integer) operation.mComponentProperties[1];
                int repeatedRowOffset = (Integer) operation.mComponentProperties[2];
                boolean isVisibleRow = (Boolean) operation.mComponentProperties[3];
                // if the maximum vertical table size is reached...
                if (currentTable.mRowCount == Table.MAX_ROW_NUMBER && operation.mHardFormattingProperties != null && currentTable.mMostUsedRowStyle != null) {
                    // remove if possible the most used style from the column and attach it to table style
                    if (operation.mComponentProperties[4] == null) {
                        // if there was no style, we need to a add a "removeFormat" operation, otherwise the default style will be triggered
                        cacheOperation(false, OperationConstants.REMOVEFORMAT, operation.mStart, null, null, null);

                        // If the current style name is the most used style
                    } else if (isVisibleRow && currentTable.mMostUsedRowStyle.equals(operation.mComponentProperties[4]) && operation.mHardFormattingProperties.containsKey("row")) {
                        operation.mHardFormattingProperties.remove("row");
                    }
                }
                // if there is no style left, avoid the operation at all
                if (operation.mHardFormattingProperties != null && !operation.mHardFormattingProperties.isEmpty()) {
                    cacheOperation(false, OperationConstants.FORMATROWS, operation.mStart, operation.mHardFormattingProperties, firstRow, lastRow, repeatedRowOffset, mContextName);
                }

            } else if (mIsSpreadsheet && operation.mComponentType.equals(OperationConstants.COLUMNS)) { // spreadsheet function
                // if the table row have not the maximium size (Excel dimension)
                if (currentTable.mRowCount == Table.MAX_ROW_NUMBER && operation.mHardFormattingProperties != null) {
                    Map defaultCellProperties = (Map) operation.mComponentProperties[2];
                    // add all default properites to the column styles
                    if (defaultCellProperties != null) {
                        if (defaultCellProperties.containsKey("cell")) {
                            operation.mHardFormattingProperties.put(("cell"), defaultCellProperties.get("cell"));
                        }
                        if (defaultCellProperties.containsKey("character")) {
                            operation.mHardFormattingProperties.put(("character"), defaultCellProperties.get("character"));
                        }
                        if (defaultCellProperties.containsKey("paragraph")) {
                            operation.mHardFormattingProperties.put(("paragraph"), defaultCellProperties.get("paragraph"));
                        }
                        if (defaultCellProperties.containsKey("styleId")) {
                            operation.mHardFormattingProperties.put(("styleId"), defaultCellProperties.get("styleId"));
                        }
                    }
                }

                // if the maximum vertical table size is reached...
                if (operation.mHardFormattingProperties != null) {
                    boolean isVisibleColumn = (Boolean) operation.mComponentProperties[4];
                    if ((currentTable.mColumnCount == Table.MAX_COLUMN_NUMBER_CALC || currentTable.mColumnCount == Table.MAX_COLUMN_NUMBER_EXCEL)) {

                        if (currentTable.mMostUsedColumnStyle != null) {
                            // remove if possible the most used style from the column and attach it to table style
                            if (operation.mComponentProperties[5] == null) {
                                // if there was no style, we need to add some neglect style otherwise the default style will be triggered
                                cacheOperation(false, OperationConstants.REMOVEFORMAT, operation.mStart, null, null, null);

                                // If the current style name is the most used style
                            } else if (isVisibleColumn && currentTable.mMostUsedColumnStyle.equals(operation.mComponentProperties[5]) && operation.mHardFormattingProperties.containsKey("column")) {
                                // remove the column attribute (as it will be at the table)
                                operation.mHardFormattingProperties.remove("column");
                            }
                        }

                        // if there is a DEFAULT Cell style at the Columns and it fullfils all requirements (no gap of default cell styles and full column number in sheet)
                        if (currentTable.mHasOnlyDefaultColumnCellStyles != null && currentTable.mHasOnlyDefaultColumnCellStyles) {
                            // Remove the default style to the column
                            String styleID = (String) operation.mHardFormattingProperties.get("styleId");
                            if (styleID != null && styleID.equals(COLUMN_CELL_DEFAULT_STYLE)) {
                                operation.mHardFormattingProperties.remove("styleId");
                            }
                        }
                    }
                    // if there is no style left, avoid the operation at all
                    if (operation.mHardFormattingProperties != null && !operation.mHardFormattingProperties.isEmpty()) {
                        cacheOperation(false, OperationConstants.FORMATCOLUMNS, operation.mStart, operation.mHardFormattingProperties, operation.mComponentProperties[0], operation.mComponentProperties[1], mContextName);
                    }
                }
            } else if (operation.mComponentType.equals(OperationConstants.SHAPE) || operation.mComponentType.equals(OperationConstants.IMAGE) || operation.mComponentType.equals(OperationConstants.GROUP)) {
                cacheOperation(false, operation.mComponentType, operation.mStart, operation.mHardFormattingProperties, mContextName);
            } else if (operation.mComponentType.equals(OperationConstants.FIELD)) {
                //TODO: Why do I have to check for map<> casts but not with String casts?
                @SuppressWarnings("unchecked")
                Map<String, Object> attrMap = (Map<String, Object>) operation.mComponentProperties[2];
                cacheOperation(false, operation.mComponentType, operation.mStart, null, operation.mComponentProperties[0], operation.mComponentProperties[1], attrMap, mContextName);
            } else if(operation.mComponentType.equals(OperationConstants.TABLE)) {
                cacheOperation(false, operation.mComponentType, operation.mStart, operation.mHardFormattingProperties, operation.mComponentProperties);
            } else if (!(mIsSpreadsheet && (operation.mComponentType.equals(OperationConstants.CELLS) || operation.mComponentType.equals(OperationConstants.ROWS)))) {
                if(putPageBreak && operation.mComponentType.equals(OperationConstants.PARAGRAPH)) {
                    JSONObject paraProps = null;
                    if( operation.mHardFormattingProperties == null) {
                        operation.mHardFormattingProperties = new HashMap<String, Object>();
                    }
                    if(!operation.mHardFormattingProperties.containsKey("paragraph")) {
                        paraProps = new JSONObject();
                    } else {
                        paraProps = (JSONObject)operation.mHardFormattingProperties.get("paragraph");
                    }
                    try {
                        paraProps.put( isBreakBefore ? "pageBreakBefore" : "pageBreakAfter", true);
                    } catch (JSONException e) {
                        //no handling required
                    }
                    operation.mHardFormattingProperties.put("paragraph", paraProps);
                    putPageBreak = false;
                }

                cacheOperation(false, operation.mComponentType, operation.mStart, operation.mHardFormattingProperties, mContextName);
            }
        }
        if (mIsSpreadsheet) {
            // there is a second table operation list to be able to sort the commands.
        	ListIterator<CachedInnerTableOperation>  tableOperationIterator = currentTable.mCachedTableContentOps.listIterator();
            // initialize the new position
            currentTable.mCurrentColumnNo = 0;
            currentTable.mFirstRow = 0;
            CachedInnerTableOperation lastRowOp = null;
            // Get the Default Column Cell Style from the column - in ODF only applied on an existing XML cell element without style
            // only rows and cell insertion ops are within this iterator --> to be mapped to range insertion ops
            while (tableOperationIterator.hasNext()) {
                CachedInnerTableOperation operation = tableOperationIterator.next();
                if (operation.mComponentType.equals(OperationConstants.ROWS)) {
                    currentTable.mPreviousRepeatedRows = (Integer) operation.mComponentProperties[2];
                    currentTable = mapCellsAndRow(operation, currentTable);
                    currentTable.mFirstRow = (Integer) operation.mComponentProperties[0];
                    currentTable.mLastRow = (Integer) operation.mComponentProperties[1];
                    currentTable.mCurrentColumnNo = 0;
                    lastRowOp = operation;
                } else if (operation.mComponentType.equals(OperationConstants.CELLS)) {
                    currentTable = mapCellsAndRow(operation, currentTable);
                }
            }
            // flush the cached cells (as the row ops came before the cells, we trigger the last row ops again to collect cashed cells)
            mapCellsAndRow(lastRowOp, currentTable);
        }
    }

    private static Map addTableStyles(Map origin, Map destiny) {
        if (origin != null && destiny != null) {
            if (origin.containsKey("cell")) {
                destiny.put(("cell"), origin.get("cell"));
            }
            if (origin.containsKey("character")) {
                destiny.put(("character"), origin.get("character"));
            }
            if (origin.containsKey("paragraph")) {
                destiny.put(("paragraph"), origin.get("paragraph"));
            }
            if (origin.containsKey("row")) {
                destiny.put(("row"), origin.get("row"));
            }
            if (origin.containsKey("column")) {
                destiny.put(("column"), origin.get("column"));
            }
            if (origin.containsKey("styleId")) {
                destiny.put(("styleId"), origin.get("styleId"));
            }
        }
        return destiny;
    }

    private static String getMostUsedStyle(Map<String, Integer> styleOccurrances) {
        String mostUsedStyleName = null;
        if (styleOccurrances != null) {
            Set entrySet = styleOccurrances.entrySet();
            Iterator<Entry<String, Integer>> iter = entrySet.iterator();
            Integer styleOccurance = null;
            Integer styleOccuranceMax = null;
            while (iter.hasNext()) {
                Entry<String, Integer> entry = iter.next();
                styleOccurance = entry.getValue();
                if (styleOccuranceMax == null || styleOccuranceMax < styleOccurance) {
                    styleOccuranceMax = styleOccurance;
                    mostUsedStyleName = entry.getKey();
                }
            }
            // if there is a most used style
            if (mostUsedStyleName != null) {
                // make sure it is not by coincidence the most single used one..
                if (styleOccurrances.get(mostUsedStyleName) == 1) {
                    mostUsedStyleName = null;
                }
            }
        }

        return mostUsedStyleName;
    }

    /**
     * Optimizes the operations of spreadsheet cells neglecting
     * starting/trailing empty cells and for cells with content or style
     * bundling similar cells to single operations.
     *
     * Repeated rows are automatically a range.
     *
     * There are three pointer (variables), that are updated during parsing the
     * spreadsheet: mCurrentCellNo is the actual column number
     * mFirstContentCellNo is mFirstEqualCellNo is set to the first cell to be
     * written, after a cell was written out or an empty precessor
     *
     * ToDo: Refactoring - As soon every component got its own parser, the
     * tableOps. have to be replaced by the Context of the component
     */
    private CachedTable evaluateSimilarCells(CachedTable tableOps, CachedInnerTableOperation cellOperation, JSONObject currentCell, boolean isRow) {
        // An Operation will always be triggered in the end of the function
        boolean triggerOperation = false;

        // every repeatedColumns row will result into a fillRange operation
        int previousContentRepetition = 1;

        // if the previous cells are equal
        if (tableOps.mFirstEqualCellNo > -1) {
            previousContentRepetition = tableOps.mCurrentColumnNo - tableOps.mFirstEqualCellNo;
        }

        boolean isRepeatedRow = tableOps.mLastRow != null && !tableOps.mFirstRow.equals(tableOps.mLastRow);

        // do not trigger the operation if the spreadsheetRow is null and its only member is null
        if (tableOps.mSheetNo == null && cellOperation != null && cellOperation.mStart != null) {
            // we have a cell position and require the two above (first parent row, afterwards cell) => already -2
            // and an additional - 1 as size of 1 would result in zero position ==> finally -3
            tableOps.mSheetNo = cellOperation.mStart.get(cellOperation.mStart.size() - 3);
        }
        // ** There are four variations for previous/current cell we have to check:
        // 1) Current Content Cell, Previous Content empty
        if (currentCell != null && tableOps.mPreviousCell != null) {
            // if the two cells are NOT the same
            if (!currentCell.equals(tableOps.mPreviousCell)) {
                if (previousContentRepetition > MIN_REPEATING_CONTENT_CELLS) {
                    triggerOperation = true;
                } else {
                    if (tableOps.mCurrentRange == null) {
                        tableOps.mCurrentRange = new JSONArray();
                    }
                    // Resolving mColumnRepetition, explicitly adding cells to the range
                    for (int i = 0; tableOps.mCurrentColumnNo - tableOps.mFirstEqualCellNo > i; i++) {
                        tableOps.mCurrentRange.put(tableOps.mPreviousCell);
                    }
                    // if the row is being repeated, there are always vertical spans (fill same content multiple times
                    if (tableOps.mLastRow - tableOps.mFirstRow > 0) {
                        triggerOperation = true;
                    }
                    // there is an upcoming fill operation, the previous content has to be flushed
                    if (tableOps.mCellRepetition > MIN_REPEATING_CONTENT_CELLS) {
                        triggerOperation = true;
                    }
                    tableOps.mFirstEqualCellNo = tableOps.mCurrentColumnNo;
                }
            }
            // 2) Current Content Cell, Previous Empty Cell (never have saved anything)
        } else if (currentCell != null && tableOps.mPreviousCell == null) { // && tableOps.mCurrentRange == null
            tableOps.mFirstEqualCellNo = tableOps.mCurrentColumnNo;
            if (tableOps.mFirstContentCellNo == -1) {
                // reset the empty cell counter - if previous was empty
                tableOps.mFirstContentCellNo = tableOps.mCurrentColumnNo;
                tableOps.mEmptyCellCount = 0;
            } else {
                if (tableOps.mCellRepetition > MIN_REPEATING_CONTENT_CELLS) {
                    triggerOperation = true;
                } else {
                    if (tableOps.mCurrentRange == null) {
                        tableOps.mCurrentRange = new JSONArray();
                    }
                    for (int i = 0; tableOps.mEmptyCellCount > i; i++) {
                        tableOps.mCurrentRange.put(JSONObject.NULL);
                    }
                    tableOps.mEmptyCellCount = 0;
                }
            }
            // 3) Content Cell empty, Previo Cell full
        } else if (currentCell == null && tableOps.mPreviousCell != null) {
            tableOps.mEmptyCellCount += tableOps.mCellRepetition;
            // as there had been previously content
            // check if it was repeating content
            if (previousContentRepetition > MIN_REPEATING_CONTENT_CELLS) {
                triggerOperation = true;
            } else {
                if (tableOps.mCurrentRange == null) {
                    tableOps.mCurrentRange = new JSONArray();
                }
                // save the previous cell for later compressed output
                for (int i = 0; tableOps.mCurrentColumnNo - tableOps.mFirstEqualCellNo > i; i++) {
                    tableOps.mCurrentRange.put(tableOps.mPreviousCell);
                }
                // if the row is being repeated, there are always vertical spans (fill same content multiple times
                if (tableOps.mLastRow - tableOps.mFirstRow > 0) {
                    triggerOperation = true;
                }
            }
            tableOps.mFirstEqualCellNo = -1; // if there was previously repeating content cells
            // 4) Both are null
        } else if (currentCell == null && tableOps.mPreviousCell == null & !isRow) {
            // note that an empty cell was passed
            tableOps.mEmptyCellCount += tableOps.mCellRepetition;
            // if this is the first empty cell
            if (tableOps.mFirstEmptyCell == -1) {
                // remember when it started
                tableOps.mFirstEmptyCell = tableOps.mCurrentColumnNo;

                // else check if the maximum repeated empty cells was reached and existing content has to be dispatched as an operation
            } else if (tableOps.mFirstContentCellNo != -1 && MAX_REPEATING_EMPTY_CELLS > tableOps.mEmptyCellCount) {
                triggerOperation = true;
            }
        }

        // RANGE CREATION: for every row we flush previous content OR if we want to flush for other reasons
        if (isRow && tableOps.mFirstContentCellNo > -1 || triggerOperation) {
            // WRITING WHITESPACE TO ROW
            // if the last cell used content, but there was previous whitespace, the whitespace has to be explicitly set
            if (tableOps.mEmptyCellCount > 0 && currentCell != null) {
                for (int i = 0; tableOps.mEmptyCellCount > i; i++) {
                    if (tableOps.mCurrentRange == null) {
                        tableOps.mCurrentRange = new JSONArray();
                    }
                    tableOps.mCurrentRange.put(JSONObject.NULL);
                }
                tableOps.mEmptyCellCount = 0;
            }
            // WRITING CELL TO ROW
            // if content to flush exist and the operation was triggered
            //  OR there is horizontal repeated content
            //  OR there is vertical repeated content
            if (tableOps.mCurrentRange != null || previousContentRepetition > MIN_REPEATING_CONTENT_CELLS || isRepeatedRow) {
                mJsonOperationProducer.addRange(tableOps.mSheetNo, tableOps.mFirstRow, tableOps.mLastRow, tableOps.mPreviousRepeatedRows, tableOps.mFirstContentCellNo, previousContentRepetition, tableOps.mPreviousCell, tableOps.mCurrentRange, previousContentRepetition > MIN_REPEATING_CONTENT_CELLS);
            }
            // if a fill sufficent repeating is now after a content, the previous content was flushed
            if (tableOps.mFirstEqualCellNo == tableOps.mCurrentColumnNo) {
                // but still a content and repeating content exits
                tableOps.mFirstContentCellNo = tableOps.mFirstEqualCellNo;
            } else {
                if (currentCell != null) {
                    tableOps.mFirstContentCellNo = tableOps.mCurrentColumnNo;
                    tableOps.mFirstEqualCellNo = tableOps.mCurrentColumnNo;
                } else {
                    tableOps.mFirstContentCellNo = -1;
                    tableOps.mFirstEqualCellNo = -1;
                }
            }
            tableOps.mCurrentRange = null;
        }
        if (!isRow) {
            // Making the current cell the previous for next round
            tableOps.mPreviousCell = currentCell;
            tableOps.mCurrentColumnNo += tableOps.mCellRepetition;
            tableOps.mCellRepetition = 1;
        } else {
            // after the end of a row reset all values
            tableOps.mSheetNo = null;
            tableOps.mCurrentRange = null;
            tableOps.mPreviousCell = null;
            tableOps.mFirstContentCellNo = -1;
            tableOps.mFirstEqualCellNo = -1;
            tableOps.mCurrentColumnNo = 0;
            tableOps.mEmptyCellCount = 0;
            tableOps.mFirstEmptyCell = -1;
            tableOps.mCellRepetition = 1;
        }
        return tableOps;
    }

    static void stashColumnWidths(TableTableElement tableElement) {
        List<TableTableColumnElement> existingColumnList = getTableColumnElements(tableElement, new LinkedList<TableTableColumnElement>());
        List<Integer> tableColumWidths = OdfFileSaxHandler.collectColumnWidths(tableElement, existingColumnList);
        tableElement.pushTableGrid(tableColumWidths);
    }

    static List<Integer> collectColumnWidths(TableTableElement tableElement, List<TableTableColumnElement> columns) {
        boolean hasRelColumnWidth = false;
        boolean hasAbsColumnWidth = false;
        boolean hasColumnWithoutWidth = false;
        List<Integer> columnRelWidths = new ArrayList();
        for (TableTableColumnElement column : columns) {
            if (column.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "style-name")) {
                Length tableWidth = getPropertyLength(StyleTablePropertiesElement.Width, tableElement);

                int repeatedColumns = 1;
                if (column.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-repeated")) {
                    repeatedColumns = Integer.parseInt(column.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-repeated"));
                }

                String columnRelWidth = getProperty(StyleTableColumnPropertiesElement.RelColumnWidth, column);

                // it is being assumed, when the columnRelWidth is once set, it is always set
                if (columnRelWidth != null && !columnRelWidth.isEmpty()) {
                    hasRelColumnWidth = true;
                    if (hasAbsColumnWidth) {
                        LOG.warning("******* BEWARE: Absolute and relative width are not supposed to be mixed!! ***********");
                    }
                    columnRelWidth = columnRelWidth.substring(0, columnRelWidth.indexOf('*'));
                    Integer relWidth = Integer.parseInt(columnRelWidth);
                    for (int i = 0; i < repeatedColumns; i++) {
                        columnRelWidths.add(relWidth);
                    }
                } else { // if there is no relative column width
                    if (hasRelColumnWidth) {
                        LOG.warning("******* BEWARE: Absolute and relative width are not supposed to be mixed!! ***********");
                    }

                    Length columnWidth = getPropertyLength(StyleTableColumnPropertiesElement.ColumnWidth, column);
                    // there can be only table width and ..
                    if (tableWidth != null) {
                        // columnwidth, with a single one missing
                        if (columnWidth != null) {
                            hasAbsColumnWidth = true;
                            int widthFactor = (int) Math.round((columnWidth.getMillimeters() * 100) / tableWidth.getMillimeters());
                            for (int i = 0; i < repeatedColumns; i++) {
                                columnRelWidths.add(widthFactor);
                            }
                        } else {
                            if (hasColumnWithoutWidth) {
                                LOG.warning("******* BEWARE: Two columns without width and no column width are not expected!! ***********");
                            }
                            hasColumnWithoutWidth = true;
                        }
                        // if the table is not set, it will always be unset..
                    } else {
                        if (columnWidth != null) {
                            hasAbsColumnWidth = true;
                            int widthFactor = (int) Math.round((columnWidth.getMicrometer() * 10));
                            for (int i = 0; i < repeatedColumns; i++) {
                                columnRelWidths.add(widthFactor);
                            }
                        } else {
                            LOG.warning("******* BEWARE: Two columns without width and no column width are not expected!! ***********");
                        }
                    }
                }
            }
        }
        return columnRelWidths;
    }

    /**
     * Returns all TableTableColumn descendants that exist within the
     * tableElement, even within groups, columns and header elements
     */
    static List<TableTableColumnElement> getTableColumnElements(Element parent, List columns) {
        NodeList children = parent.getChildNodes();
        for (int i = 0; i < children.getLength(); i++) {
            Node child = children.item(i);
            if (child instanceof Element) {
                if (child instanceof TableTableColumnElement) {
                    columns.add(child);
                } else if (child instanceof TableTableColumnGroupElement
                    || child instanceof TableTableHeaderColumnsElement
                    || child instanceof TableTableColumnsElement) {
                    columns = getTableColumnElements((Element) child, columns);
                } else if (child instanceof TableTableRowGroupElement
                    || child instanceof TableTableHeaderRowsElement
                    || child instanceof TableTableRowElement
                    || child instanceof TableTableRowsElement) {
                    break;
                }
            }
        }
        return columns;
    }
}
