/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.calcengine.client;

import org.apache.commons.logging.Log;

import com.openexchange.office.calcengine.client.impl.CalcEngineClientBound2ClipBoard;
import com.openexchange.office.calcengine.client.impl.CalcEngineClientHTTP;
import com.openexchange.office.calcengine.client.impl.CalcEngineClientJMS;
import com.openexchange.office.calcengine.client.impl.CalcEngineClientLocal;
import com.openexchange.office.calcengine.client.impl.CalcEngineClientMonitoring;
import com.openexchange.office.calcengine.client.impl.CalcEngineClientNative;
import com.openexchange.office.calcengine.client.impl.CalcEngineClientPerformanceMeasurement;
import com.openexchange.office.calcengine.client.impl.CalcEngineErrorSimulator;
import com.openexchange.office.tools.logging.LogFactory;

//=============================================================================
/** factory where you can retrieve instances of type ICalcEngine from.
 */
public class CalcEngineClientFactory
{
    //-------------------------------------------------------------------------
	private static final Log LOG = LogFactory.getJclLog(CalcEngineClientFactory.class);

    //-------------------------------------------------------------------------
    private CalcEngineClientFactory ()
        throws Exception
    {}

    //-------------------------------------------------------------------------
    private static CalcEngineClientFactory m_gSingleton = null;

    //-------------------------------------------------------------------------
    public static synchronized CalcEngineClientFactory getFactory ()
        throws Exception
    {
        if (m_gSingleton == null)
            m_gSingleton = new CalcEngineClientFactory ();
        return m_gSingleton;
    }

    //-------------------------------------------------------------------------
    /** @return an instance of type ICalcEngine.
     *
     *  Those instance is full initialized and operational.
     *  Difference between embedded and remote implementation is
     *  handled by the right set of configuration data.
     *
     *  It's not guaranteed nor expected to get the same instance
     *  of calc engine on two different requests.
     *
     *  @throws an exception in case operation failed by any reason.
     */
    public ICalcEngineClient get ()
        throws Exception
    {
        final CalcEngineConfig     aConfig       = CalcEngineConfig.get ();
        final CalcEngineDescriptor aDesc         = new CalcEngineDescriptor ();
                                   aDesc.m_eMode = aConfig.getMode ();
        final ICalcEngineClient    iClient       = getDirect (aDesc);
        return iClient;
    }

    //-------------------------------------------------------------------------
    /** INTERNAL API ! DO NOT USE IT FROM OUTSIDE !
     */
    public static ICalcEngineClient getDirect (final CalcEngineDescriptor aDesc)
        throws Exception
    {
        ICalcEngineClient iClient = null;

        if (aDesc.m_eMode == ECalcEngineMode.E_NATIVE)
            iClient = impl_getImplNATIVE (aDesc);
        else
        if (aDesc.m_eMode == ECalcEngineMode.E_INLINE)
            iClient = impl_getImplINLINE (aDesc);
        else
        if (aDesc.m_eMode == ECalcEngineMode.E_LOCAL)
            iClient = impl_getImplLOCAL (aDesc);
        else
        if (aDesc.m_eMode == ECalcEngineMode.E_HTTP)
            iClient = impl_getImplHTTP (aDesc);
        else
        if (aDesc.m_eMode == ECalcEngineMode.E_HTTP_CLIENT)
            iClient = impl_getImplHTTPCLIENT (aDesc);
        else
        if (aDesc.m_eMode == ECalcEngineMode.E_HTTP_CLIENT_SIMPLE)
            iClient = impl_getImplHTTPCLIENTSIMPLE (aDesc);
        else
        if (aDesc.m_eMode == ECalcEngineMode.E_HTTP_SERVER)
            iClient = impl_getImplHTTPSERVER (aDesc);
        else
        if (aDesc.m_eMode == ECalcEngineMode.E_JMS)
            iClient = impl_getImplJMS (aDesc);
        else
        if (aDesc.m_eMode == ECalcEngineMode.E_JMS_CLIENT)
            iClient = impl_getImplJMSCLIENT (aDesc);
        else
        if (aDesc.m_eMode == ECalcEngineMode.E_JMS_SERVER)
            throw new UnsupportedOperationException ("JMS server can be created so easy ... look for our cal engine JMS worker registration !");
        else
        if (aDesc.m_eMode == ECalcEngineMode.E_SIMULATOR)
            iClient = impl_getImplSIMULATOR (aDesc);
        else
            throw new UnsupportedOperationException ("no support for '"+aDesc.m_eMode+"' implemented yet.");

        LOG.info("created calc engine client instance => ["+iClient+"]");
        return iClient;
    }

    //-------------------------------------------------------------------------
    private static ICalcEngineClient impl_getImplNATIVE (final CalcEngineDescriptor aDesc)
        throws Exception
    {
        LOG.info("create calc engine client : NATIVE ...");
        final CalcEngineClientNative aClient = CalcEngineClientNative.create ();
        return aClient;
    }

    //-------------------------------------------------------------------------
    private static ICalcEngineClient impl_getImplINLINE (final CalcEngineDescriptor aDesc)
        throws Exception
    {
        LOG.info("create calc engine client : INLINE ...");
        final CalcEngineClientNative           aClient            = CalcEngineClientNative         .create (                 );
        final CalcEngineClientBound2ClipBoard  aClipBoardWrapper  = CalcEngineClientBound2ClipBoard.create (aClient          );
        final CalcEngineClientMonitoring       aMonitoringWrapper = CalcEngineClientMonitoring     .create (aClipBoardWrapper);
        
        if (CalcEngineConfig.get ().isPerformanceMeasurementOn())
        {
        	final CalcEngineClientPerformanceMeasurement aRecorder = CalcEngineClientPerformanceMeasurement.create(aMonitoringWrapper);
        	return aRecorder;
        }
        
        return aMonitoringWrapper;
    }

    //-------------------------------------------------------------------------
    private static ICalcEngineClient impl_getImplLOCAL (final CalcEngineDescriptor aDesc)
        throws Exception
    {
        LOG.info("create calc engine client : LOCAL ...");
        final CalcEngineClientLocal           aClient            = CalcEngineClientLocal          .create (                 );
        final CalcEngineClientBound2ClipBoard aClipBoardWrapper  = CalcEngineClientBound2ClipBoard.create (aClient          );
        final CalcEngineClientMonitoring      aMonitoringWrapper = CalcEngineClientMonitoring     .create (aClipBoardWrapper);
        aClient.describeEnvironment(aDesc);
        
        if (CalcEngineConfig.get ().isPerformanceMeasurementOn())
        {
        	final CalcEngineClientPerformanceMeasurement aRecorder = CalcEngineClientPerformanceMeasurement.create(aMonitoringWrapper);
        	return aRecorder;
        }
        
        return aMonitoringWrapper;
    }

    //-------------------------------------------------------------------------
    private static ICalcEngineClient impl_getImplHTTP (final CalcEngineDescriptor aDesc)
        throws Exception
    {
        final CalcEngineConfig aConfig       = CalcEngineConfig.get ();
        final int              nTHISServerID = aConfig.getServerNumberForTHISHost();
        final boolean          bServerMode   = nTHISServerID != CalcEngineConfig.INVALID_SERVER_NUMBER;

        if (bServerMode == true)
        {
            aDesc.m_eMode = ECalcEngineMode.E_HTTP_SERVER;
            return CalcEngineClientFactory.getDirect(aDesc);
        }
        else
        {
            aDesc.m_eMode = ECalcEngineMode.E_HTTP_CLIENT;
            return CalcEngineClientFactory.getDirect(aDesc);
        }
    }

    //-------------------------------------------------------------------------
    private static ICalcEngineClient impl_getImplHTTPCLIENT (final CalcEngineDescriptor aDesc)
        throws Exception
    {
        LOG.info("create calc engine client : HTTP-CLIENT ...");
        final CalcEngineClientHTTP            aHTTPClient       = CalcEngineClientHTTP.create ();
        final CalcEngineClientBound2ClipBoard aClipBoardWrapper = CalcEngineClientBound2ClipBoard.create (aHTTPClient);
        aHTTPClient.describeEnvironment(aDesc);
        
        if (CalcEngineConfig.get ().isPerformanceMeasurementOn())
        {
        	final CalcEngineClientPerformanceMeasurement aRecorder = CalcEngineClientPerformanceMeasurement.create(aClipBoardWrapper);
        	return aRecorder;
        }
        
        return aClipBoardWrapper;
    }

    //-------------------------------------------------------------------------
    private static ICalcEngineClient impl_getImplHTTPCLIENTSIMPLE (final CalcEngineDescriptor aDesc)
        throws Exception
    {
        LOG.info("create calc engine client : HTTP-CLIENT-SIMPLE ...");
        final CalcEngineClientHTTP aClient = CalcEngineClientHTTP.create ();
        aClient.describeEnvironment(aDesc);
        return aClient;
    }

    //-------------------------------------------------------------------------
    private static ICalcEngineClient impl_getImplHTTPSERVER (final CalcEngineDescriptor aDesc)
        throws Exception
    {
        LOG.info("create calc engine client : HTTP-SERVER ...");
        // note : clip board handling is done on HTTP client side ... not on server side !
        final CalcEngineClientLocal aClient = CalcEngineClientLocal.create ();
        aClient.describeEnvironment(aDesc);
        return aClient;
    }

    //-------------------------------------------------------------------------
    private static ICalcEngineClient impl_getImplSIMULATOR (final CalcEngineDescriptor aDesc)
        throws Exception
    {
        LOG.info("create calc engine client : SIMULATOR ...");
        //final CalcEngineClientSimulator aClient = CalcEngineClientSimulator.create ();
        final CalcEngineErrorSimulator aClient = CalcEngineErrorSimulator.create ();
        return aClient;
    }

    //-------------------------------------------------------------------------
    private static ICalcEngineClient impl_getImplJMS (final CalcEngineDescriptor aDesc)
        throws Exception
    {
        final CalcEngineConfig aConfig       = CalcEngineConfig.get ();
        final int              nTHISServerID = aConfig.getServerNumberForTHISHost();
        final boolean          bServerMode   = nTHISServerID != CalcEngineConfig.INVALID_SERVER_NUMBER;

        if (bServerMode == true)
        {
            aDesc.m_eMode = ECalcEngineMode.E_JMS_SERVER;
            return CalcEngineClientFactory.getDirect(aDesc);
        }
        else
        {
            aDesc.m_eMode = ECalcEngineMode.E_JMS_CLIENT;
            return CalcEngineClientFactory.getDirect(aDesc);
        }
    }

    //-------------------------------------------------------------------------
    private static ICalcEngineClient impl_getImplJMSCLIENT (final CalcEngineDescriptor aDesc)
        throws Exception
    {
		LOG.info("create calc engine client : JMS-CLIENT ...");
		final com.openxchange.office_communication.configuration.configitems.CalcEngineConfig aCEConfig = com.openxchange.office_communication.configuration.configitems.CalcEngineConfig.access();
		final CalcEngineClientJMS             aJMSClient        = CalcEngineClientJMS.create (aCEConfig.getInQueue());
        final CalcEngineClientBound2ClipBoard aClipBoardWrapper = CalcEngineClientBound2ClipBoard.create (aJMSClient);
        final CalcEngineClientMonitoring      aMonitoringWrapper = CalcEngineClientMonitoring    .create (aClipBoardWrapper);
        aJMSClient.start ();
        
        if (CalcEngineConfig.get ().isPerformanceMeasurementOn())
        {
        	final CalcEngineClientPerformanceMeasurement aRecorder = CalcEngineClientPerformanceMeasurement.create(aMonitoringWrapper);
        	return aRecorder;
        }
        
		return aMonitoringWrapper;
    }

    //-------------------------------------------------------------------------
    public String m_sConfiguredServerURI = null;
}
