/**
 * OPEN-XCHANGE legal information
 *
 * All intellectual property rights in the Software are protected by
 * international copyright laws.
 *
 *
 * In some countries OX, OX Open-Xchange, open xchange and OXtender
 * as well as the corresponding Logos OX Open-Xchange and OX are registered
 * trademarks of the Open-Xchange, Inc. group of companies.
 * The use of the Logos is not covered by the GNU General Public License.
 * Instead, you are allowed to use these Logos according to the terms and
 * conditions of the Creative Commons License, Version 2.5, Attribution,
 * Non-commercial, ShareAlike, and the interpretation of the term
 * Non-commercial applicable to the aforementioned license is published
 * on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 * Please make sure that third-party modules and libraries are used
 * according to their respective licenses.
 *
 * Any modifications to this package must retain all copyright notices
 * of the original copyright holder(s) for the original code used.
 *
 * After any such modifications, the original and derivative code shall remain
 * under the copyright of the copyright holder(s) and/or original author(s)per
 * the Attribution and Assignment Agreement that can be located at
 * http://www.open-xchange.com/EN/developer/. The contributing author shall be
 * given Attribution for the derivative code and a license granting use.
 *
 *  Copyright (C) 2004-2014 Open-Xchange, Inc.
 *  Mail: info@open-xchange.com
 *
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License, Version 2 as published
 *  by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc., 59
 *  Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package com.openxchange.office_communication.tools.exec.impl;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.mutable.Mutable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jezhumble.javasysmon.JavaSysMon;
import com.jezhumble.javasysmon.OsProcess;

//==============================================================================
public class GlobalPidProcessHandler
{
	//--------------------------------------------------------------------------
	private static final Logger LOG = LoggerFactory.getLogger(GlobalPidProcessHandler.class);

	//--------------------------------------------------------------------------
	public static final int INVALID_PID = -1;

	//--------------------------------------------------------------------------
	/// force using of factory method to ensure singleton instance !
	private GlobalPidProcessHandler ()
	{}

	//--------------------------------------------------------------------------
	public static synchronized GlobalPidProcessHandler get ()
		throws Exception
	{
		if (m_gSingleton == null)
			m_gSingleton = new GlobalPidProcessHandler ();
		return m_gSingleton;
	}

	//--------------------------------------------------------------------------
    public synchronized void startChildProcess (final ProcessBuilder     aBuilder,
    								            final Mutable< Process > aProcess,
    								            final Mutable< Integer > nPid    )
    	throws Exception
    {
        final List< Integer > lChildPidsBefore = impl_getChildPidsOfThisProcess ();
        final Process         aNewProcess      = aBuilder.start();
        final List< Integer > lChildPidsAfter  = impl_getChildPidsOfThisProcess ();
        final int             nNewPid          = impl_getPidOfNewStartedProcess (lChildPidsBefore, lChildPidsAfter);
        final List< Integer > lAllChildPids    = mem_AllChildPids ();

        lAllChildPids.clear ();
        lAllChildPids.addAll(lChildPidsAfter);
        
    	aProcess.setValue (aNewProcess);
    	nPid    .setValue (nNewPid    );
    }

    //--------------------------------------------------------------------------
    public synchronized void killChildProcess (final int nPid)
    	throws Exception
    {
    	Validate.isTrue(nPid != INVALID_PID, "invalid argument 'pid'");
    	
        JavaSysMon aMonitor = new JavaSysMon();
        aMonitor.killProcess(nPid);
        impl_forgetChildPid (nPid);
    }
	
    //--------------------------------------------------------------------------
    public synchronized void killAllChilds ()
    	throws Exception
    {
    	final List< Integer > lChilds = new ArrayList< Integer > (mem_AllChildPids ());
    	for (final Integer nPid : lChilds)
    		killChildProcess (nPid);
    }

    //--------------------------------------------------------------------------
    public synchronized boolean isChildProcessAlive (final int nPid)
    	throws Exception
    {
    	if (nPid == GlobalPidProcessHandler.INVALID_PID)
    		return false;
    	
    	final JavaSysMon aMonitor  = new JavaSysMon();
    	final OsProcess  aTreeRoot = aMonitor.processTree();
    	final OsProcess  aProcess  = aTreeRoot.find(nPid);
    	
    	if (aProcess == null)
    	{
            impl_forgetChildPid (nPid);
    		return false;
    	}
    	return true;
    }

	//--------------------------------------------------------------------------
	public synchronized List< Integer > getAllChildPids ()
	     throws Exception
	{
		return mem_AllChildPids ();
	}

	//--------------------------------------------------------------------------
    private List< Integer > impl_getChildPidsOfThisProcess ()
    	throws Exception
    {
	    final JavaSysMon          aMonitor = new JavaSysMon();
	    final ProcessChildVisitor aVisitor = new ProcessChildVisitor();
	    final int                 nThisPid = aMonitor.currentPid();

	    aVisitor.setRootPid      (nThisPid);
	    aMonitor.visitProcessTree(nThisPid, aVisitor);
	    
	    final List< Integer > lChildPids = new ArrayList< Integer >(aVisitor.getPids());
	    return lChildPids;
    }

    //--------------------------------------------------------------------------
    private int impl_getPidOfNewStartedProcess (final List< Integer > lPidsBefore,
			   									final List< Integer > lPidsAfter)
    	throws Exception
    {
    	final List< Integer > aDiff = impl_diffChildPids (lPidsBefore, lPidsAfter);

    	if (aDiff.size() < 1)
    	{
    		LOG.trace ("no pid in diff ...");
    		return INVALID_PID;
    	}
    	else
    	if (aDiff.size() > 1)
    	{
    		LOG.trace ("to many pids in diff ["+aDiff+"] ...");
    		return INVALID_PID;
    	}
    	
    	final int nPid = aDiff.get(0);
    	if (nPid < 1)
    	{
    		LOG.trace ("invalid pid '"+nPid+"' in diff ...");
    		return INVALID_PID;
    	}

    	return nPid;
    }
    
    //--------------------------------------------------------------------------
	private List< Integer > impl_diffChildPids (final List< Integer > lPidsBefore,
    										    final List< Integer > lPidsAfter )
    	throws Exception
    {
    	
    	final List< Integer > lDiff = new ArrayList< Integer > (lPidsAfter);
    	lDiff.removeAll(lPidsBefore);

		LOG.trace ("diff (before) : "+lPidsBefore);
		LOG.trace ("diff (after ) : "+lPidsAfter);
		LOG.trace ("diff (result) : "+lDiff  );

    	return lDiff;
    }
	
	//--------------------------------------------------------------------------
    private void impl_forgetChildPid (final int nPid)
    	throws Exception
    {
        final List< Integer > lAllChildPids = mem_AllChildPids ();
        final Integer         nPidItem      = new Integer (nPid); // if you use nPid directly you call container.remove(index) instead of container.remove(item) ! .-)

        if (lAllChildPids.contains(nPidItem))
        	lAllChildPids.remove(nPidItem);
    }
    
	//--------------------------------------------------------------------------
	private List< Integer > mem_AllChildPids ()
	     throws Exception
	{
		if (m_lAllChildPids == null)
			m_lAllChildPids = new ArrayList< Integer > ();
		return m_lAllChildPids;
	}

	//--------------------------------------------------------------------------
	private static GlobalPidProcessHandler m_gSingleton = null;

	//--------------------------------------------------------------------------
	private List< Integer > m_lAllChildPids = null;
}
