/**
 * OPEN-XCHANGE legal information
 *
 * All intellectual property rights in the Software are protected by
 * international copyright laws.
 *
 *
 * In some countries OX, OX Open-Xchange, open xchange and OXtender
 * as well as the corresponding Logos OX Open-Xchange and OX are registered
 * trademarks of the Open-Xchange, Inc. group of companies.
 * The use of the Logos is not covered by the GNU General Public License.
 * Instead, you are allowed to use these Logos according to the terms and
 * conditions of the Creative Commons License, Version 2.5, Attribution,
 * Non-commercial, ShareAlike, and the interpretation of the term
 * Non-commercial applicable to the aforementioned license is published
 * on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 * Please make sure that third-party modules and libraries are used
 * according to their respective licenses.
 *
 * Any modifications to this package must retain all copyright notices
 * of the original copyright holder(s) for the original code used.
 *
 * After any such modifications, the original and derivative code shall remain
 * under the copyright of the copyright holder(s) and/or original author(s)per
 * the Attribution and Assignment Agreement that can be located at
 * http://www.open-xchange.com/EN/developer/. The contributing author shall be
 * given Attribution for the derivative code and a license granting use.
 *
 *  Copyright (C) 2004-2014 Open-Xchange, Inc.
 *  Mail: info@open-xchange.com
 *
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License, Version 2 as published
 *  by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc., 59
 *  Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package com.openxchange.office_communication.cluster_management.core.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.CountDownLatch;

import org.apache.camel.Exchange;
import org.apache.camel.Message;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openxchange.office_communication.cluster_management.core.ClusterState;
import com.openxchange.office_communication.cluster_management.core.EClusterState;
import com.openxchange.office_communication.cluster_management.core.IClusterManager;
import com.openxchange.office_communication.cluster_management.core.IClusterStateNotification;
import com.openxchange.office_communication.cluster_management.core.impl.messages.AppControlMessage;
import com.openxchange.office_communication.cluster_management.core.impl.messages.EAppOperation;
import com.openxchange.office_communication.configuration.configitems.cluster_management.AppDescriptor;
import com.openxchange.office_communication.configuration.configitems.cluster_management.ClusterNodeConfig;
import com.openxchange.office_communication.configuration.configitems.cluster_management.EStartMode;
import com.openxchange.office_communication.configuration.configitems.cluster_management.WatchdogDescriptor;
import com.openxchange.office_communication.jms.core.plain.beans.JMSMessageUtils;
import com.openxchange.office_communication.tools.logging.LogUtils;

//=============================================================================
public class ClusterManager implements IClusterManager
{
	//-------------------------------------------------------------------------
	private static final Logger LOG = LoggerFactory.getLogger(ClusterManager.class);

	//-------------------------------------------------------------------------
	private static final boolean RUNORDER_ASCENDING  = false;
	private static final boolean RUNORDER_DESCENDING = true ;

	//-------------------------------------------------------------------------
	private static final int FIRST_START = 1;
	private static final int RESTART     = 2;
	
	//-------------------------------------------------------------------------
	public ClusterManager ()
		throws Exception
	{}
	
	//-------------------------------------------------------------------------
	@Override
	public void start ()
		throws Exception
	{
		LOG.info("start ...");
		if (m_bStarted)
		{
			LOG.info("... already started.");
			return;
		}

		impl_updateClusterState   (EClusterState.E_STARTING);

		impl_defineApps           ();
		impl_ensureAppsAreRunning (FIRST_START);
		
		m_bStarted = true;
		impl_updateClusterState   (EClusterState.E_RUNNING);

		impl_registerShutdownHook ();
		impl_startWatchdog        ();

		LOG.info("started.");
	}

	//-------------------------------------------------------------------------
	@Override
	public void stop ()
		throws Exception
	{
		LOG.info("stop ...");
		if ( ! m_bStarted)
		{
			LOG.info("... already stopped.");
			return;
		}

		impl_updateClusterState (EClusterState.E_STOPPING);

		impl_stopWatchdog ();
		impl_shutdown     ();
		
		m_bStarted = false;
		impl_updateClusterState (EClusterState.E_STOPPED);

		LOG.info("stopped.");
	}

	//-------------------------------------------------------------------------
	@Override
	public void waitForShutdown ()
		throws Exception
	{
		// TODO check if shutdown was not already done before !
		mem_StopSync ().await();
	}

	//-------------------------------------------------------------------------
	@Override
	public void addStateListener (final IClusterStateNotification iListener)
		throws Exception
	{
		final List< IClusterStateNotification > aRegistry = mem_StateListenerRegistry ();
		aRegistry.add (iListener);
		
		// listener is 'to late' all notifications send before are missing
		// ensure new listener 'is up to date' ;-)
		impl_notifyListener (iListener, mem_State ());
	}

	//-------------------------------------------------------------------------
	// called from camel-MQ context for incoming messages
	public synchronized void handleMessage (final Exchange aExchange)
		throws Exception
	{
		final Message                 aRequest           = aExchange.getIn();
		final AppControlMessage       aAppControlRequest = JMSMessageUtils.mapMessageToBean(aRequest);
		final String                  sAppId             = aAppControlRequest.getAppId       ();
		final EAppOperation           eOp                = aAppControlRequest.getAppOperation();
		final Map< String, AppGroup > aRegistry          = mem_AppRegistry ();
		final AppGroup                aAppGroup          = aRegistry.get(sAppId);

		LOG.info("... handle remote control request"+
				LogUtils.formatContextVariables
				(
					new String[] {"appid", "operation"},
					new Object[] {sAppId , eOp        }
				));
		
		if (aAppGroup == null)
			return;
		
		final AppDescriptor     aAppConfig  = aAppGroup.aAppDescriptor;
		final List< SimpleApp > lInstances  = aAppGroup.lInstances    ;
		final List< SimpleApp > lRunning    = new ArrayList< SimpleApp > ();
		final List< SimpleApp > lNotRunning = new ArrayList< SimpleApp > ();
		
		impl_filterRunningInstances (lInstances, lRunning, lNotRunning);
		
		if (eOp == EAppOperation.E_START)
		{
			impl_startAppMinimum (aAppConfig, lRunning, lNotRunning);
		}
		else
			throw new UnsupportedOperationException ("no support for '"+eOp+"' implemented yet");
	}

	//-------------------------------------------------------------------------
	private void impl_startAppMinimum (final AppDescriptor     aApp       ,
									   final List< SimpleApp > lRunning   ,
									   final List< SimpleApp > lNotRunning)
	    throws Exception
	{
		final int nMin     = aApp.getMinInstances ();
		final int nMax     = aApp.getMaxInstances ();
		final int nRunning = lRunning.size ();
		      int nMissing = nMin - nRunning;
		final int nReserve = lNotRunning.size ();
		
		if (nRunning >= nMin)
			return;

		if (nMissing > nReserve)
			nMissing = nReserve;
		
		for (int i=0; i<nMissing; ++i)
		{
			final SimpleApp aInstance = lNotRunning.get(i);
			impl_startApp (aInstance);
		}
	}
	
	//-------------------------------------------------------------------------
	private void impl_filterRunningInstances (final List< SimpleApp > lAll       ,
											  final List< SimpleApp > lRunning   ,
											  final List< SimpleApp > lNotRunning)
	    throws Exception
	{
		for (final SimpleApp aInst : lAll)
		{
			if (aInst.isRunning())
				lRunning.add (aInst);
			else
				lNotRunning.add (aInst);
		}
	}
	
	//-------------------------------------------------------------------------
	private void impl_defineApps ()
		throws Exception
	{
		LOG.info("define apps ...");
		final ClusterNodeConfig       aConfig      = mem_Config ();
		final Set< AppDescriptor >    lAppConfigs  = aConfig.accesAppConfig();
		final Map< String, AppGroup > aAppRegistry = mem_AppRegistry ();
		
		for (final AppDescriptor aAppConfig : lAppConfigs)
		{
			final String   sAppId            = aAppConfig.getId();
			final int      nInstances        = aAppConfig.getMaxInstances();
				  AppGroup aAppGroup         = aAppRegistry.get(sAppId);

			final boolean  bIsJolokiaEnabled = aAppConfig.isJolokiaEnabled();
			final int      nJolokiaPortMin   = aAppConfig.getJolokiaPortRangeMin();
			final int      nJolokiaPortMax   = aAppConfig.getJolokiaPortRangeMax();
			      int      nJolokiaPort      = nJolokiaPortMin;

		    if (aAppGroup == null)
			{
				aAppGroup                = new AppGroup ();
				aAppGroup.aAppDescriptor = aAppConfig;
				aAppGroup.lInstances     = new ArrayList< SimpleApp > ();
				aAppRegistry.put(sAppId, aAppGroup);
			}
			
			for (int i=0; i<nInstances; ++i)
			{
				final String sInstId = sAppId + "-inst-" + i;
				final String sUCRID  = impl_newUCRID ();
				
				LOG.info("... define app"+
						LogUtils.formatContextVariables
						(
							new String[] {"appid", "instance", "ucrid"},
							new Object[] {sAppId , i         , sUCRID }
						));

				final SimpleApp aApp = new SimpleApp ();
				
				aApp.setUCRID     (sUCRID    );
				aApp.setInstanceId(sInstId   );
				aApp.setDescriptor(aAppConfig);
				
				if (bIsJolokiaEnabled)
				{
					LOG.info("... enable jolokia"+
							LogUtils.formatContextVariables
							(
								new String[] {"appid", "instance", "ucrid", "jolokia-port"},
								new Object[] {sAppId , i         , sUCRID , nJolokiaPort  }
							));
					Validate.isTrue(nJolokiaPort <= nJolokiaPortMax, "Jolokia port out of range. Did you configured port range of app '"+sAppId+"' right ?\nport-range-min="+nJolokiaPortMin+", port-range-max="+nJolokiaPortMax+", port="+nJolokiaPort);
					aApp.setJolokiaPort(nJolokiaPort);
					nJolokiaPort++;
				}
				
				aAppGroup.lInstances.add(aApp);
				
				impl_updateAppState (sInstId, "ready for start ...", false);
			}
		}
		
		LOG.info("apps defined.");
	}

	//-------------------------------------------------------------------------
	private void impl_ensureAppsAreRunning (final int nStartMode)
		throws Exception
	{
		LOG.info("ensure apps are running ...");
		final Iterator< AppGroup > rGroups  = impl_listAppsByRunLevel (RUNORDER_ASCENDING).iterator();
		while (rGroups.hasNext())
		{
			try
			{
				final AppGroup          aGroup     = rGroups.next();
				final AppDescriptor     aDesc      = aGroup.aAppDescriptor;
				final List< SimpleApp > lInstances = aGroup.lInstances;
				final ClusterState      aState     = mem_State();
				final int               nMin       = aDesc.getMinInstances();
				final int               nMax       = aDesc.getMaxInstances();
				
				int nInst = 0;
				for (final SimpleApp aInstance : lInstances)
				{
					nInst++;
					final String sId = aInstance.getInstanceId();
					
					LOG.info("... is app running ?");
					if (aInstance.isRunning())
					{
						LOG.info("... app already running"+
								LogUtils.formatContextVariables
								(
									new String[] {"id", "run-level"        , "start-mode"        , "start-count"            },
									new Object[] {sId , aDesc.getRunLevel(), aDesc.getStartMode(), aInstance.getStartCount()}
								));
						continue;
					}
	
					if (nStartMode == RESTART)
					{
						// restart apps only if they was running before !
						if (aState.getAppStartCount(sId) < 1)
							continue;
						
						LOG.info("... kill app");
						impl_stopApp (aInstance);
						LOG.info("... restart app");
						impl_startApp (aInstance);
						continue;
					}
					
					LOG.info("... is app start mode = auto ?");
					if (aDesc.getStartMode() == EStartMode.E_AUTO)
					{
						if (nInst > nMin)
						{
							LOG.info("... min amount of instances already started ! No further instances will be started.");
							continue;
						}
						
						impl_startApp (aInstance);
						continue;
					}
					
					LOG.info("... app wont be started automatically - it's not configured so"+
							LogUtils.formatContextVariables
							(
								new String[] {"id", "run-level"        , "start-mode"         },
								new Object[] {sId , aDesc.getRunLevel(), aDesc.getStartMode() }
							));
				}
			}
			catch (final Throwable ex)
			{
				LOG.error (ex.getMessage (), ex);
			}
		}
		LOG.info("apps are running.");
	}

	//-------------------------------------------------------------------------
	private void impl_startApp (final SimpleApp aApp)
		throws Exception
	{
		final String        sId   = aApp.getInstanceId();
		final AppDescriptor aDesc = aApp.getDescriptor();

		LOG.info("... start app"+
				LogUtils.formatContextVariables
				(
					new String[] {"id", "run-level"        , "start-mode"        , "start-count"       },
					new Object[] {sId , aDesc.getRunLevel(), aDesc.getStartMode(), aApp.getStartCount()}
				));
		
		impl_updateAppState(sId, "starting ...", false);
		aApp.start();
		impl_updateAppState(sId, "started [pid="+aApp.getPid()+"]", true);

		if (aDesc.isJmxEnabled())
			impl_establishJmxBinding (aApp);

		LOG.info("... OK");
	}
	
	//-------------------------------------------------------------------------
	private void impl_stopApp (final SimpleApp aApp)
		throws Exception
	{
		final String        sId   = aApp.getInstanceId();
		final AppDescriptor aDesc = aApp.getDescriptor();

		LOG.info("... stop app"+
				LogUtils.formatContextVariables
				(
					new String[] {"id", "run-level"        },
					new Object[] {sId , aDesc.getRunLevel()}
				));
		
		impl_updateAppState(sId, "stopping [pid="+aApp.getPid()+"] ...", false);
		aApp.stop();
		impl_updateAppState(sId, "stopped", false);

		LOG.info("... OK");
	}

	//-------------------------------------------------------------------------
	private void impl_stopApps ()
		throws Exception
	{
		LOG.info("stop apps ...");
		final Iterator< AppGroup > rGroups = impl_listAppsByRunLevel (RUNORDER_DESCENDING).iterator();
		while (rGroups.hasNext())
		{
			try
			{
				final AppGroup          aGroup     = rGroups.next();
				final List< SimpleApp > lInstances = aGroup.lInstances;
				
				for (final SimpleApp aInstance : lInstances)
				{
					final String sId = aInstance.getInstanceId();

					if ( ! aInstance.isRunning())
					{
						LOG.info("... app '"+sId+"' is already stopped");
						continue;
					}
					
					LOG.info("... stop app '"+sId+"'");
					impl_updateAppState(sId, "stopping ...", false);
					aInstance.stop();
					impl_updateAppState(sId, "stopped", false);
					LOG.info("... OK");
				}
			}
			catch (final Throwable ex)
			{
				ex.printStackTrace();
				/// TODO handle errors
			}
		}
		LOG.info("apps stopped.");
	}

	//-------------------------------------------------------------------------
	private void impl_establishJmxBinding (final SimpleApp aApp)
		throws Exception
	{
//		final AppDescriptor aDesc = aApp.getDescriptor ();
//		final JmxClient     aJmx  = new JmxClient ();
//		final int           nPort = aDesc.getJmxPort();
//		
//		aJmx.setJmxPort(nPort);
//		aJmx.connect   (     );
//		
//		m_aJmx = aJmx;
	}
	
	//-------------------------------------------------------------------------
	private class AppGroupRunLevelComparator implements Comparator< AppGroup >
	{
		@Override
		public int compare(final AppGroup aGroup1,
						   final AppGroup aGroup2)
		{
			try
			{
				final AppDescriptor aDesc1     = aGroup1.aAppDescriptor;
				final AppDescriptor aDesc2     = aGroup2.aAppDescriptor;
				final int           nRunLevel1 = aDesc1 .getRunLevel();
				final int           nRunLevel2 = aDesc2 .getRunLevel();

				if (nRunLevel1 > nRunLevel2)
					return 1;
				
				if (nRunLevel1 < nRunLevel2)
					return -1;

				return 0;
			}
			catch (Throwable ex)
			{
				throw new RuntimeException (ex);
			}
		}
	}
	
	//-------------------------------------------------------------------------
	private List< AppGroup > impl_listAppsByRunLevel (final boolean bReverseOrder)
		throws Exception
	{
		final List< AppGroup >       lOrderedApps = new ArrayList< AppGroup > (mem_AppRegistry ().values());
		      Comparator< AppGroup > aComparator  = new AppGroupRunLevelComparator ();
		
		if (bReverseOrder)
			aComparator = Collections.reverseOrder(aComparator);
		
		Collections.sort(lOrderedApps, aComparator);
		return lOrderedApps;
	}
	
	//-------------------------------------------------------------------------
	/**
	 * not called within eclipse : https://bugs.eclipse.org/bugs/show_bug.cgi?id=38016
	 */
	private void impl_registerShutdownHook ()
	    throws Exception
	{
		LOG.info("register shutdown hook ...");
		final ClusterManager aNodeContext = this;
		Runtime.getRuntime().addShutdownHook(new Thread ()
		{
			@Override
			public void run ()
			{
				try
				{
					aNodeContext.stop ();
				}
				catch (Throwable ex)
				{
					LOG.error (ex.getMessage (), ex);
				}
			}
		});
	}

	//-------------------------------------------------------------------------
	private void impl_startWatchdog ()
		throws Exception
	{
		if (m_aWatchDog != null)
			return;
		
		final WatchdogDescriptor aConfig       = mem_Config ().accessWatchdogConfig();
		final int                nPollTimeInMS = aConfig.getPollTimeInMS();
		
		LOG.trace("watchdog : start ...");
		m_aWatchDog = new Thread (new Runnable ()
		{
			@Override
			public void run ()
			{
				try
				{
					while (true)
					{
						LOG.trace("watchdog : ensure apps are running ...");
						impl_ensureAppsAreRunning (RESTART);
						
						LOG.trace("watchdog : sleep ["+nPollTimeInMS+"] ...");
						synchronized (this)
						{
							wait (nPollTimeInMS);
						}
						LOG.trace("watchdog : wake up ...");
					}
				}
				catch (Throwable ex)
				{
					LOG.error("watchdog : died - reason was :\n"+ex.getMessage (), ex);
				}
			}
		});
		
		m_aWatchDog.start ();
	}
	
	//-------------------------------------------------------------------------
	private void impl_stopWatchdog ()
		throws Exception
	{
		if (m_aWatchDog == null)
			return;
		
		try
		{
			Thread aWatchDog = m_aWatchDog;
			m_aWatchDog = null;
			aWatchDog.stop ();
		}
		catch (final Throwable ex)
		{
			// ignore !
		}
	}
	
	//-------------------------------------------------------------------------
	private void impl_shutdown ()
		throws Exception
	{
		LOG.info("shutdown triggered ...");
		impl_stopApps   ();
		impl_notifyStop ();
		LOG.info("... shutdown finished.");
	}
	
	//-------------------------------------------------------------------------
	private void impl_notifyStop ()
		throws Exception
	{
		mem_StopSync ().countDown();
	}
	
	//-------------------------------------------------------------------------
	private void impl_updateClusterState (final EClusterState eState)
		throws Exception
	{
		final ClusterState aState = mem_State ();
		aState.setClusterState(eState);
		impl_notifyListener (aState);
	}

	//-------------------------------------------------------------------------
	private void impl_updateAppState (final String  sAppId       ,
									  final String  sState       ,
									  final boolean bCountRestart)
		throws Exception
	{
		final ClusterState aState = mem_State ();
		aState.setAppState(sAppId, sState);
		if (bCountRestart)
			aState.countAppStart(sAppId);
		impl_notifyListener (aState);
	}

	//-------------------------------------------------------------------------
	private void impl_notifyListener (final ClusterState aState)
		throws Exception
	{
		final List< IClusterStateNotification > aRegistry = mem_StateListenerRegistry ();
		for (final IClusterStateNotification iListener : aRegistry)
			impl_notifyListener (iListener, aState);
	}
	
	//-------------------------------------------------------------------------
	private void impl_notifyListener (final IClusterStateNotification iListener,
									  final ClusterState              aState   )
	{
		try
		{
			LOG.trace("notify listener ["+iListener+"] about state : ["+aState+"]...");
			iListener.notifyState(aState);
		}
		catch (Throwable ex)
		{
			// by intention !
			// listener has not to disturb our workflow ;-)
		}
	}

	//-------------------------------------------------------------------------
	private String impl_newUCRID ()
		throws Exception
	{
		final String sUCRID = UUID.randomUUID().toString();
		return sUCRID;
	}

	//-------------------------------------------------------------------------
	private CountDownLatch mem_StopSync ()
	    throws Exception
	{
		if (m_aStopSync == null)
			m_aStopSync = new CountDownLatch (1);
		return m_aStopSync;
	}

	//-------------------------------------------------------------------------
	private ClusterNodeConfig mem_Config ()
		throws Exception
	{
		if (m_aConfig == null)
			m_aConfig = ClusterNodeConfig.access();
		return m_aConfig;
	}

	//-------------------------------------------------------------------------
	private Map< String, AppGroup > mem_AppRegistry ()
	    throws Exception
	{
		if (m_aAppRegistry == null)
			m_aAppRegistry = new HashMap< String, AppGroup > ();
		return m_aAppRegistry;
	}

	//-------------------------------------------------------------------------
	private List< IClusterStateNotification > mem_StateListenerRegistry ()
	    throws Exception
	{
		if (m_aStateListenerRegistry == null)
			m_aStateListenerRegistry = new ArrayList< IClusterStateNotification > ();
		return m_aStateListenerRegistry;
	}

	//-------------------------------------------------------------------------
	private ClusterState mem_State ()
	    throws Exception
	{
		if (m_aState == null)
			m_aState = new ClusterState ();
		return m_aState;
	}

	//-------------------------------------------------------------------------
	public static class AppGroup
	{
		public AppDescriptor     aAppDescriptor = null;
		public List< SimpleApp > lInstances     = null;
	}
	
	//-------------------------------------------------------------------------
	private CountDownLatch m_aStopSync = null;
	
	//-------------------------------------------------------------------------
	private boolean m_bStarted = false;
	
	//-------------------------------------------------------------------------
	private ClusterNodeConfig m_aConfig = null;

	//-------------------------------------------------------------------------
	private Map< String, AppGroup > m_aAppRegistry = null;

	//-------------------------------------------------------------------------
	private List< IClusterStateNotification > m_aStateListenerRegistry = null;

	//-------------------------------------------------------------------------
	private ClusterState m_aState = null;

	//-------------------------------------------------------------------------
	private Thread m_aWatchDog = null;
}
