/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define(['io.ox/office/tk/object/baseobject'], function (BaseObject) {

    'use strict';

    describe('OX Documents Base Framework', function () {
        describe('class BaseObject', function () {

            it('should exist', function () {
                expect(BaseObject).to.be.a('function');
            });

            // methods ========================================================

            describe('method "registerDestructor"', function () {
                var obj = new BaseObject(), spy = sinon.spy();
                it('should exist', function () {
                    expect(obj).to.respondTo('registerDestructor');
                });
                it('should register a destructor callback', function () {
                    obj.registerDestructor(spy);
                    expect(spy.called).to.be['false'];
                });
            });

            // ----------------------------------------------------------------

            describe('method "getUid"', function () {
                var obj1 = new BaseObject(), obj2 = new BaseObject();
                it('should exist', function () {
                    expect(obj1).to.respondTo('getUid');
                });
                it('should return a UID', function () {
                    expect(obj1.getUid()).to.be.a('string');
                });
                it('should return the same UID for the same object', function () {
                    expect(obj1.getUid()).to.equal(obj1.getUid());
                });
                it('should return different UIDs for different objects', function () {
                    expect(obj1.getUid()).to.not.equal(obj2.getUid());
                });
            });

            // ----------------------------------------------------------------

            describe('method "listenTo"', function () {

                var obj = new BaseObject();

                it('should exist', function () {
                    expect(obj).to.respondTo('listenTo');
                });
                it('should call the event handler', function () {
                    var src = $({}), spy = sinon.spy();
                    obj.listenTo(src, 'test', spy);
                    expect(spy.called).to.be['false'];
                    src.trigger('wrong');
                    expect(spy.called).to.be['false'];
                    src.trigger('test');
                    expect(spy.called).to.be['true'];
                });
                it('should call the event handler for multiple event types', function () {
                    var src = $({}), spy = sinon.spy();
                    obj.listenTo(src, 'test1 test2', spy);
                    expect(spy.called).to.be['false'];
                    src.trigger('wrong');
                    expect(spy.called).to.be['false'];
                    src.trigger('test1');
                    expect(spy.calledOnce).to.be['true'];
                    src.trigger('test2');
                    expect(spy.calledTwice).to.be['true'];
                });
                it('should call the event handlers of an event map', function () {
                    var src = $({}), spy1 = sinon.spy(), spy2 = sinon.spy();
                    obj.listenTo(src, { test1: spy1, 'test1 test2': spy2 });
                    expect(spy1.called).to.be['false'];
                    expect(spy2.called).to.be['false'];
                    src.trigger('wrong');
                    expect(spy1.called).to.be['false'];
                    expect(spy2.called).to.be['false'];
                    src.trigger('test1');
                    expect(spy1.calledOnce).to.be['true'];
                    expect(spy2.calledOnce).to.be['true'];
                    src.trigger('test2');
                    expect(spy1.calledOnce).to.be['true'];
                    expect(spy2.calledTwice).to.be['true'];
                });
                it('should call the done handler of a promise listener', function () {
                    var def = $.Deferred(), doneSpy = sinon.spy(), failSpy = sinon.spy();
                    obj.listenTo(def, 'done', doneSpy);
                    obj.listenTo(def, 'fail', failSpy);
                    expect(doneSpy.called).to.be['false'];
                    expect(failSpy.called).to.be['false'];
                    def.resolve();
                    expect(doneSpy.called).to.be['true'];
                    expect(failSpy.called).to.be['false'];
                });
                it('should call the fail handler of a promise listener', function () {
                    var def = $.Deferred(), doneSpy = sinon.spy(), failSpy = sinon.spy();
                    obj.listenTo(def, 'done', doneSpy);
                    obj.listenTo(def, 'fail', failSpy);
                    expect(doneSpy.called).to.be['false'];
                    expect(failSpy.called).to.be['false'];
                    def.reject();
                    expect(doneSpy.called).to.be['false'];
                    expect(failSpy.called).to.be['true'];
                });
                it('should call the always handler of a promise listener', function () {
                    var def1 = $.Deferred(), def2 = $.Deferred(), spy = sinon.spy();
                    obj.listenTo(def1, 'always', spy);
                    obj.listenTo(def2, 'always', spy);
                    expect(spy.called).to.be['false'];
                    def1.resolve();
                    expect(spy.calledOnce).to.be['true'];
                    def2.reject();
                    expect(spy.calledTwice).to.be['true'];
                });
                it('should call the handlers of an event map', function () {
                    var def = $.Deferred(), doneSpy = sinon.spy(), progSpy = sinon.spy();
                    obj.listenTo(def, { done: doneSpy, progress: progSpy });
                    expect(doneSpy.called).to.be['false'];
                    expect(progSpy.called).to.be['false'];
                    def.notify(1);
                    expect(doneSpy.called).to.be['false'];
                    expect(progSpy.calledOnce).to.be['true'];
                    def.resolve();
                    expect(doneSpy.calledOnce).to.be['true'];
                    expect(progSpy.calledOnce).to.be['true'];
                });
            });

            // ----------------------------------------------------------------

            describe('method "stopListeningTo"', function () {

                it('should exist', function () {
                    var obj = new BaseObject();
                    expect(obj).to.respondTo('stopListeningTo');
                });
                it('should unregister all event listeners', function () {
                    var obj = new BaseObject(), src = $({}), spy1 = sinon.spy(), spy2 = sinon.spy();
                    obj.listenTo(src, 'test1 test2', spy1).listenTo(src, 'test1 test2', spy2);
                    expect(spy1.called).to.be['false'];
                    expect(spy2.called).to.be['false'];
                    src.trigger('test1');
                    expect(spy1.calledOnce).to.be['true'];
                    expect(spy2.calledOnce).to.be['true'];
                    obj.stopListeningTo(src);
                    src.trigger('test1').trigger('test2');
                    expect(spy1.calledOnce).to.be['true'];
                    expect(spy2.calledOnce).to.be['true'];
                });
                it('should unregister a specific event type', function () {
                    var obj = new BaseObject(), src = $({}), spy1 = sinon.spy(), spy2 = sinon.spy();
                    obj.listenTo(src, 'test1 test2 test3', spy1).listenTo(src, 'test1 test2 test3', spy2);
                    expect(spy1.called).to.be['false'];
                    expect(spy2.called).to.be['false'];
                    src.trigger('test1');
                    expect(spy1.calledOnce).to.be['true'];
                    expect(spy2.calledOnce).to.be['true'];
                    obj.stopListeningTo(src, 'test2 test3');
                    src.trigger('test1').trigger('test2').trigger('test3');
                    expect(spy1.calledTwice).to.be['true'];
                    expect(spy2.calledTwice).to.be['true'];
                });
                it('should unregister a specific event listener', function () {
                    var obj = new BaseObject(), src = $({}), spy1 = sinon.spy(), spy2 = sinon.spy();
                    obj.listenTo(src, 'test1 test2', spy1).listenTo(src, 'test1 test2', spy2);
                    expect(spy1.called).to.be['false'];
                    expect(spy2.called).to.be['false'];
                    src.trigger('test1');
                    expect(spy1.calledOnce).to.be['true'];
                    expect(spy2.calledOnce).to.be['true'];
                    obj.stopListeningTo(src, spy2);
                    src.trigger('test1').trigger('test2');
                    expect(spy1.calledThrice).to.be['true'];
                    expect(spy2.calledOnce).to.be['true'];
                });
                it('should unregister a specific event type and listener', function () {
                    var obj = new BaseObject(), src = $({}), spy1 = sinon.spy(), spy2 = sinon.spy();
                    obj.listenTo(src, 'test1 test2 test3', spy1).listenTo(src, 'test1 test2 test3', spy2);
                    expect(spy1.called).to.be['false'];
                    expect(spy2.called).to.be['false'];
                    src.trigger('test1');
                    expect(spy1.calledOnce).to.be['true'];
                    expect(spy2.calledOnce).to.be['true'];
                    obj.stopListeningTo(src, 'test2 test3', spy2);
                    src.trigger('test1').trigger('test2').trigger('test3');
                    expect(spy1.callCount).to.equal(4);
                    expect(spy2.calledTwice).to.be['true'];
                });
                it('should unregister listeners from an event map', function () {
                    var obj = new BaseObject(), src = $({}), spy1 = sinon.spy(), spy2 = sinon.spy();
                    obj.listenTo(src, 'test1 test2 test3', spy1).listenTo(src, 'test1 test2 test3', spy2);
                    expect(spy1.called).to.be['false'];
                    expect(spy2.called).to.be['false'];
                    src.trigger('test1');
                    expect(spy1.calledOnce).to.be['true'];
                    expect(spy2.calledOnce).to.be['true'];
                    obj.stopListeningTo(src, { test1: spy1, 'test2 test3': spy2 });
                    src.trigger('test1').trigger('test2').trigger('test3');
                    expect(spy1.calledThrice).to.be['true'];
                    expect(spy2.calledTwice).to.be['true'];
                });
            });

            // ----------------------------------------------------------------

            describe('method "destroy"', function () {

                var obj = new BaseObject(),
                    src = $({}),
                    def = $.Deferred(),
                    spy1 = sinon.spy(),
                    spy2 = sinon.spy(),
                    spy3 = sinon.spy();

                obj.registerDestructor(spy1);
                obj.registerDestructor(spy2);
                obj.listenTo(src, 'test', spy3);
                obj.listenTo(def, 'done', spy3);

                it('should exist', function () {
                    expect(obj).to.respondTo('destroy');
                });
                it('should call the registered destructors in reversed order', function () {
                    expect(obj.destroyed).to.be['undefined'];
                    obj.destroy();
                    expect(obj.destroyed).to.be['true'];
                    expect(spy1.called).to.be['true'];
                    expect(spy2.called).to.be['true'];
                    expect(spy2.calledBefore(spy1)).to.be['true'];
                });
                it('should delete all public methods', function () {
                    expect(obj).not.to.respondTo('registerDestructor');
                    expect(obj).not.to.respondTo('getUid');
                    expect(obj).not.to.respondTo('listenTo');
                    expect(obj).not.to.respondTo('stopListeningTo');
                    expect(obj).not.to.respondTo('destroy');
                });
                it('should release event handlers', function () {
                    src.trigger('test');
                    expect(spy3.called).to.be['false'];
                });
                it('should release promise handlers', function () {
                    def.resolve();
                    expect(spy3.called).to.be['false'];
                });
            });

            // ----------------------------------------------------------------

        });
    });
});
