/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/object/triggerobject', [
    'io.ox/core/event',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/baseobject'
], function (Events, Utils, BaseObject) {

    'use strict';

    // class TriggerObject ====================================================

    /**
     * An abstract base class for all classes that want to trigger events to
     * listeners.
     *
     * @constructor
     *
     * @extends BaseObject
     * @extends Events
     */
    var TriggerObject = BaseObject.extend({ constructor: function () {

        // base constructors --------------------------------------------------

        BaseObject.call(this);
        Events.extend(this);

        // public methods -----------------------------------------------------

        /**
         * Executes the specified callback function. As long as the callback is
         * running, this instance will be silent, i.e. it will not notify any
         * listeners when its method TriggerObject.trigger() will be called.
         *
         * @param {Function} callback
         *  The callback function to be invoked in silent mode.
         *
         * @param {Object} [context]
         *  The context to be bound to the symbol 'this' inside the callback
         *  function.
         *
         * @returns {Any}
         *  The result of the callback function.
         */
        this.executeSilently = function (callback, context) {

            var // the original trigger() method
                triggerMethod = this.trigger;

            try {
                // replace the trigger() method with a no-op version
                this.trigger = Utils.NOOP;
                // invoke the callback function
                return callback.call(context);
            } finally {
                // restore the trigger() method afterwards
                this.trigger = triggerMethod;
            }
        };

        /**
         * Invokes the specified public method of this instance. As long as the
         * method is running, this instance will be silent, i.e. it will not
         * notify any listeners when its method TriggerObject.trigger() will be
         * called.
         *
         * @param {String} methodName
         *  The name of the public method of this instance to be invoked.
         *
         * @param {Any} [...]
         *  All additional parameters will be passed to the method.
         *
         * @returns {Any}
         *  The result of the invoked method.
         */
        this.invokeSilently = function (methodName) {
            var args = _.toArray(arguments).slice(1);
            return this.executeSilently(function () {
                return this[methodName].apply(this, args);
            }, this);
        };

        // initialization -----------------------------------------------------

        // destroy the core events mix-in
        this.registerDestructor(_.bind(this.events.destroy, this.events));

    }}); // class TriggerObject

    // exports ================================================================

    return TriggerObject;

});
