/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/dialogs', [
    'io.ox/core/tk/dialogs',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/keycodes',
    'gettext!io.ox/office/main'
], function (CoreDialogs, Utils, Forms, KeyCodes, gt) {

    'use strict';

    var // the number of visible modal dialogs
        modalDialogs = 0;

    // static class Dialogs ===================================================

    var Dialogs = {};

    // methods ----------------------------------------------------------------

    /**
     * Returns whether a modal dialog is currently open.
     *
     * @returns {Boolean}
     *  Whether a modal dialog is currently open.
     */
    Dialogs.isModalDialogOpen = function () {
        return modalDialogs > 0;
    };

    // class ModalDialog ======================================================

    /**
     * Creates an empty modal dialog.
     *
     * @constructor
     *
     * @extends CoreDialogs.ModalDialog
     *
     * @param {Object} [initOptions]
     *  Optional parameters that control the appearance and behavior of the
     *  dialog. The following options are supported:
     *  @param {String} [initOptions.title]
     *      The title of the dialog window that will be shown in a larger font.
     *  @param {Number} [initOptions.width=400]
     *      The width of the dialog, in pixels.
     *  @param {String} [initOptions.okLabel=gt('OK')]
     *      The label of the primary button that triggers the default action of
     *      the dialog. If omitted, the button will be created with the
     *      translated default label 'OK'.
     *  @param {String} [initOptions.cancelLabel=gt('Cancel')]
     *      The label of the cancel button that closes the dialog without
     *      performing any action. If omitted, the button will be created with
     *      the translated default label 'Cancel'.
     *  @param {Boolean} [initOptions.async=false]
     *      Whether the dialog will be opened in asynchronous mode.
     */
    Dialogs.ModalDialog = _.makeExtendable(CoreDialogs.ModalDialog).extend({ constructor: function (initOptions) {

        var // self reference
            self = this,

            // the title text
            title = Utils.getStringOption(initOptions, 'title'),

            // the text label of the 'OK' button
            okLabel = Utils.getStringOption(initOptions, 'okLabel', gt('OK')),

            // the text label of the 'Cancel' button
            cancelLabel = Utils.getStringOption(initOptions, 'cancelLabel', gt('Cancel'));

        // base constructor ---------------------------------------------------

        CoreDialogs.ModalDialog.call(this, {
            width: Utils.getIntegerOption(initOptions, 'width', 400),
            async: Utils.getBooleanOption(initOptions, 'async', false),
            enter: inputEnterHandler
        });

        // private methods ----------------------------------------------------

        /**
         * Handler that triggers the primary button of the dialog, after the
         * ENTER key has been pressed in a text input control.
         */
        function inputEnterHandler() {

            var // the primary button bound to the ENTER key
                okButton = self.getOkButton(),
                // whether all criteria are met to trigger the default action
                triggerAction = (okButton.length > 0) && (okButton.attr('disabled') !== 'disabled');

            if (triggerAction) { self.invoke('ok'); }
            return !triggerAction;
        }

        function keyDownHandler(event) {
            if (event.keyCode === KeyCodes.ENTER) {
                inputEnterHandler();
                return false;
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the button element from the footer of this dialog bound to
         * the specified action.
         *
         * @param {String} action
         *  The identifier of the action the button is bound to.
         *
         * @returns {jQuery}
         *  The specified button element, as jQuery object.
         */
        this.getButton = function (action) {
            return this.getFooter().find('.btn[data-action="' + action + '"]');
        };

        /**
         * Returns the 'OK' button of this dialog, as jQuery object.
         *
         * @returns {jQuery}
         *  The button element of the 'OK' button.
         */
        this.getOkButton = function () {
            return this.getButton('ok');
        };

        /**
         * Returns the 'Cancel' button of this dialog, as jQuery object.
         *
         * @returns {jQuery}
         *  The button element of the 'Cancel' button.
         */
        this.getCancelButton = function () {
            return this.getButton('cancel');
        };

        /**
         * Enables or disables the button element from the footer of this
         * dialog bound to the specified action.
         *
         * @param {String} action
         *  The identifier of the action the button is bound to.
         *
         * @param {Boolean} state
         *  Whether to enable (true) or disable (false) the button.
         *
         * @returns {ModalDialog}
         *  A reference to this instance.
         */
        this.enableButton = function (action, state) {
            var buttonNode = this.getButton(action);
            if (state) {
                buttonNode.removeAttr('disabled').removeAttr('aria-disabled');
            } else {
                buttonNode.attr({ disabled: 'disabled', 'aria-disabled': true });
            }
            return this;
        };

        /**
         * Enables or disables the 'OK' button element of this dialog.
         *
         * @param {Boolean} state
         *  Whether to enable (true) or disable (false) the button.
         *
         * @returns {ModalDialog}
         *  A reference to this instance.
         */
        this.enableOkButton = function (state) {
            return this.enableButton('ok', state);
        };

        /**
         * Enables or disables the 'Cancel' button element of this dialog.
         *
         * @param {Boolean} state
         *  Whether to enable (true) or disable (false) the button.
         *
         * @returns {ModalDialog}
         *  A reference to this instance.
         */
        this.enableCancelButton = function (state) {
            return this.enableButton('cancel', state);
        };

        /**
         * Shows this dialog.
         *
         * @attention
         *  The dialog can only be shown once. It will destroy itself after it
         *  has been closed.
         *
         * @param {Function} [callback]
         *  A callback function that will be invoked after the dialog became
         *  visible.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved with the
         *  action identifier, if any action button ('OK' button, alternative
         *  buttons) has been pressed; or rejected, if the 'Cancel' button has
         *  been pressed.
         */
        this.show = (function () {
            var show = self.show;
            return function () {
                // work-around the weird behavior of the core dialog in asynchronous mode:
                // - the promise returned by show() will not resolve/reject when triggering an action
                // - the generic 'action' event does not send the action identifier
                // solution is to process all dialog events and look for an action button to resolve/reject the own promise
                var def = $.Deferred();
                function actionHandler(event, action) {
                    if (action === 'close') {
                        // bug 32244: handle explicit "dialog.close()" calls
                        self.close = $.noop; // prevent recursive calls from promise callbacks
                        def.reject(action, self);
                    } else if (self.getButton(action).length > 0) {
                        def[(action === 'cancel') ? 'reject' : 'resolve'](action, self);
                    }
                }
                modalDialogs += 1;
                self.on('triggered', actionHandler);
                def.always(function () { modalDialogs -= 1; self.off('triggered', actionHandler); self = null; });
                // show the dialog, ignore the promise it returns
                show.apply(self, arguments);
                // return the own promise that will always be resolved/rejected
                // (dialog remains still open in asynchronous mode)
                return def.promise();
            };
        }());

        // initialization -----------------------------------------------------

        // add own CSS marker class
        this.getPopup().addClass('io-ox-office-dialog');
        Utils.addDeviceMarkers(this.getPopup());

        // add title
        if (_.isString(title)) {
            this.header($('<h4>', { id: 'dialog-title' }).text(title));
        }

        // add 'OK' button
        if (_.isString(okLabel)) {
            this.addPrimaryButton('ok', okLabel, 'ok', { tabIndex: 1 });
        }

        // add 'Cancel' button
        if (_.isString(cancelLabel)) {
            this.addButton('cancel', cancelLabel, 'cancel', { tabIndex: 1 });
        }

        // listen to key events for additional keyboard handling
        this.getBody().on('keydown', keyDownHandler);

    }}); // class ModalDialog

    // class ModalTabDialog ===================================================

    /**
     * Creates an empty modal dialog that contains a tab bar and tab panels.
     *
     * Additionally to the events triggered by the base class ModalDialog, an
     * instance of this class triggers the following events:
     * - 'tab:hide'
     *      Before the active tab panel will be deactivated. Event handlers
     *      receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {String} tabId
     *          The identifier of the active tab panel.
     *      (3) {jQuery} tabPanel
     *          The root node of the active tab panel.
     *      (4) {jQuery} tabButton
     *          The button element of the active tab button.
     * - 'tab:show'
     *      After a tab panel has been activated. Event handlers receive the
     *      following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {String} tabId
     *          The identifier of the active tab panel.
     *      (3) {jQuery} tabPanel
     *          The root node of the active tab panel.
     *      (4) {jQuery} tabButton
     *          The button element of the active tab button.
     *
     * @constructor
     *
     * @extends ModalDialog
     *
     * @param {Object} [initOptions]
     *  Optional parameters that control the appearance and behavior of the
     *  dialog. Supports all options also supported by the base class
     *  ModalDialog.
     */
    Dialogs.ModalTabDialog = Dialogs.ModalDialog.extend({ constructor: function (initOptions) {

        var // self reference
            self = this,

            // the container for the tab buttons
            tabBarNode = $('<ul>', { role: 'tablist' }).addClass('nav nav-tabs'),

            // the container for the tab panel elements
            tabPanelNode = $('<div>').addClass('tab-content').css('margin-top', '13px'),

            // maps tab panels by user identifiers
            tabPanelsMap = {},

            // all tab panel identifiers in insertion order
            tabPanelIds = [],

            // the identifier of the visible tab panel
            activeId = null;

        // base constructor ---------------------------------------------------

        Dialogs.ModalDialog.call(this, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Sets the browser focus into the button element of the active tab
         * panel.
         */
        function focusActiveTabButton() {
           tabBarNode.find('li.active a').first().focus();
        }

        /**
         * Handles keyboard events for browser focus navigation.
         */
        function keyDownHandler(event) {

            var // index of the active tab panel
                activeIndex = _.indexOf(tabPanelIds, activeId),
                // identifier of the new tab panel to be activated
                newTabId = null;

            switch (event.keyCode) {
            case KeyCodes.RIGHT_ARROW:
                newTabId = tabPanelIds[activeIndex + 1];
                break;
            case KeyCodes.LEFT_ARROW:
                newTabId = tabPanelIds[activeIndex - 1];
                break;
            case KeyCodes.SPACE:
                newTabId = $(event.target).attr('data-user-id');
                break;
            }

            if (newTabId) {
                // focus active tab button first to allow the active tab panel
                // to finally focus a control inside the panel
                focusActiveTabButton();
                self.activateTabPanel(newTabId);
                return false;
            }

            //we need to call preventDefault() because of focus travelling back to the document when this dialog has no input-fields
            event.preventDefault();
        }

        // methods ------------------------------------------------------------

        /**
         * Creates a new tab panel container element associated to the
         * specified identifier.
         *
         * @param {String} tabId
         *  The identifier of the tab panel. The identifiers of all tab panels
         *  created for this dialog must be different.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.label]
         *      The text label to be shown in the tab button activating the new
         *      tab panel. If omitted, the tab button will not contain a text
         *      label.
         *  @param {String} [options.icon]
         *      The CSS class name of the icon to be shown in the tab button
         *      activating the new tab panel. If omitted, the tab button will
         *      not contain an icon. Must be an icon provided by the Font
         *      Awesome icon set.
         *
         * @returns {jQuery}
         *  The container element of the specified tab panel, as jQuery object.
         */
        this.addTabPanel = function (tabId, options) {

            var // the unique DOM identifier for the tab button
                buttonId = _.uniqueId('tab'),
                // the unique DOM identifier for the tab panel container
                panelId = _.uniqueId('panel'),

                // the button element in the tab bar
                tabButton = $('<a href="#' + panelId + '" id="' + buttonId + '" tabindex="1" role="tab" data-toggle="tab" data-user-id="' + tabId + '" aria-selected="false" aria-controls="' + panelId + '">'),
                // the container element for the tab panel
                tabPanel = $('<div class="tab-pane row-fluid" id="' + panelId + '" role="tabpanel" data-user-id="' + tabId + '" aria-labelledby="' + buttonId + '">');

            // set button caption
            tabButton.append(Forms.createCaptionMarkup(options));

            // insert the tab button and tab panel
            tabBarNode.append($('<li>').append(tabButton));
            tabPanelNode.append(tabPanel);
            tabPanelsMap[tabId] = tabPanel;
            tabPanelIds.push(tabId);

            // return the tab panel for further initialization
            return tabPanel;
        };

        /**
         * Returns the tab panel container element associated to the specified
         * identifier.
         *
         * @param {String} tabId
         *  The identifier of the tab panel, as passed to the method
         *  ModalTabDialog.addTabPanel().
         *
         * @returns {jQuery}
         *  The container element of the specified tab panel. If the passed
         *  identifier is invalid, returns an empty jQuery collection.
         */
        this.getTabPanel = function (tabId) {
            return (tabId in tabPanelsMap) ? tabPanelsMap[tabId] : $();
        };

        /**
         * Returns the identifier of the tab panel that is currently visible.
         *
         * @returns {String|Null}
         *  The identifier of the tab panel that is currently visible, or null
         *  if no panel is visible yet.
         */
        this.getActiveTabId = function () {
            return activeId;
        };

        /**
         * Activates (displays) the specified tab panel.
         *
         * @param {String} tabId
         *  The identifier of the tab panel, as passed to the method
         *  ModalTabDialog.addTabPanel().
         *
         * @returns {ModalTabDialog}
         *  A reference to this instance.
         */
        this.activateTabPanel = function (tabId) {
            tabBarNode.find('[data-user-id="' + tabId + '"]').tab('show');
            return this;
        };

        // initialization -----------------------------------------------------

        // insert the nodes into the dialog body
        this.append(tabBarNode, tabPanelNode);

        // initially focus the active tab button
        this.on('show', focusActiveTabButton);

        // trigger 'tab:hide' event before the active tab panel will be hidden
        tabBarNode.on('show.bs.tab', function (event) {
            var tabButton = $(event.relatedTarget).closest('a'),
                tabId = tabButton.attr('data-user-id');
            if (tabButton.length > 0) {
                tabButton.attr('aria-selected', false);
                self.trigger('tab:hide', tabId, tabPanelsMap[tabId], tabButton);
            }
        });

        // trigger 'tab:show' event after a tab panel has been shown
        tabBarNode.on('shown.bs.tab', function (event) {
            var tabButton = $(event.target).closest('a');
            activeId = tabButton.attr('data-user-id');
            tabButton.attr('aria-selected', true);
            self.trigger('tab:show', activeId, tabPanelsMap[activeId], tabButton);
        });

        // handle keyboard events for focus navigation
        tabBarNode.on('keydown', keyDownHandler);

        // destroy members
        this.on('close', function () {
            tabBarNode.remove();
            tabPanelNode.remove();
            self = tabBarNode = tabPanelNode = tabPanelsMap = null;
        });

    }}); // class ModalTabDialog

    // class ModalQueryDialog =================================================

    /**
     * Creates a simple dialog with a message text and 'Yes'/'No' buttons.
     *
     * @constructor
     *
     * @extends ModalDialog
     *
     * @param {String} message
     *  The message text shown in the dialog body element.
     *
     * @param {Object} [initOptions]
     *  Optional parameters that control the appearance and behavior of the
     *  dialog. Supports all options also supported by the base class
     *  ModalDialog. The default value of the option 'okLabel' will be changed
     *  to the translated string 'Yes', and the default value of the option
     *  'cancelLabel' will be changed to the translated string 'No'.
     */
    Dialogs.ModalQueryDialog = Dialogs.ModalDialog.extend({ constructor: function (message, initOptions) {

        // base constructor ---------------------------------------------------

        Dialogs.ModalDialog.call(this, Utils.extendOptions({
            okLabel: gt('Yes'),
            cancelLabel: gt('No')
        }, initOptions));

        // initialization -----------------------------------------------------

        this.append($('<label>').text(message));

    }}); // class ModalQueryDialog

    // class ModalInputDialog =================================================

    /**
     * Creates a simple text input dialog.
     *
     * @constructor
     *
     * @extends ModalDialog
     *
     * @param {Object} [initOptions]
     *  Optional parameters that control the appearance and behavior of the
     *  dialog. Supports all options also supported by the base class
     *  ModalDialog. Additionally, the following options are supported:
     *  @param {String} [initOptions.value='']
     *      The initial value of the text field.
     *  @param {String} [initOptions.placeholder='']
     *      The place-holder text that will be shown in the empty text field.
     *  @param {Boolean} [initOptions.allowEmpty=false]
     *      If set to true, empty text will be considered valid. Otherwise, the
     *      primary button will be disabled as long as the text input field is
     *      empty.
     */
    Dialogs.ModalInputDialog = Dialogs.ModalDialog.extend({ constructor: function (initOptions) {

        var // self reference
            self = this,

            // the text field control
            inputNode = $('<input type="text" class="form-control" tabindex="1">').attr({
                placeholder: Utils.getStringOption(initOptions, 'placeholder', ''),
                value: Utils.getStringOption(initOptions, 'value', '')
            }),

            // whether to allow empty text
            allowEmpty = Utils.getBooleanOption(initOptions, 'allowEmpty', false);

        // base constructor ---------------------------------------------------

        Dialogs.ModalDialog.call(this, initOptions);

        // private functions --------------------------------------------------

        /**
         * Enables or disables the primary button according to the contents of
         * the text field.
         */
        function updateOkButton() {
            self.enableOkButton(allowEmpty || (inputNode.val().length > 0));
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the text that is contained in the text input element.
         *
         * @returns {String}
         *  The text that is contained in the text input element.
         */
        this.getText = function () {
            return inputNode.val();
        };

        /**
         * Changes the text contained in the text input element.
         *
         * @param {String} text
         *  The text to be inserted into the text input element.
         *
         * @returns {ModalInputDialog}
         *  A reference to this instance.
         */
        this.setText = function (text) {
            inputNode.val(text);
            return this;
        };

        // initialization -----------------------------------------------------

        // add the text field
        this.append(inputNode);

        // enable/disable the primary button
        inputNode.on('input', updateOkButton);
        updateOkButton();

        // focus and select the text in the input field
        this.on({
            show: function () { inputNode.focus().select(); },
            close: function () { inputNode.remove(); }
        });

    }}); // class ModalInputDialog

    // exports ================================================================

    return Dialogs;

});
