/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/control/widthmixin', [
    'io.ox/office/tk/utils'
], function (Utils) {

    'use strict';

    // mix-in class WidthMixin ================================================

    /**
     * Extends a Group object with additional methods to manipulate the width
     * of one of its child control elements.
     *
     * Note: This is a mix-in class supposed to extend an existing instance of
     * the class Group or one of its derived classes. Expects the symbol 'this'
     * to be bound to an instance of Group.
     *
     * @constructor
     *
     * @param {HTMLElement|jQuery} controlNode
     *  The form control element to be manipulated.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {Number|String} [initOptions.width]
     *      A fixed width for the control element, as number in pixels, or as
     *      CSS length string. If omitted, the control element will not have a
     *      fixed width, but will size itself according to its contents and
     *      element type.
     *  @param {Number|String} [initOptions.minWidth]
     *      A fixed minimum width for the control element, as number in pixels,
     *      or as CSS length string.
     *  @param {Number|String} [initOptions.maxWidth]
     *      A fixed maximum width for the control element, as number in pixels,
     *      or as CSS length string.
     */
    function WidthMixin(controlNode, initOptions) {

        // methods ------------------------------------------------------------

        /**
         * Returns the effective outer width of the wrapped control element.
         *
         * @return {Number}
         *  The effective outer width of the wrapped control element.
         */
        this.getWidth = function () {
            return controlNode.outerWidth();
        };

        /**
         * Sets the width of the wrapped control element.
         *
         * @param {Number|String} width
         *  The width to be set for the control element, as number in pixels,
         *  or as CSS length string. The empty string will remove the fixed
         *  width from the control element.
         *
         * @returns {WidthMixin}
         *  A reference to this instance.
         */
        this.setWidth = function (width) {
            controlNode.css({ width: width });
            this.layout();
            return this;
        };

        /**
         * Sets the minimum width of the wrapped control element.
         *
         * @param {Number} width
         *  The minimum width to be set for the control element, as number in
         *  pixels, or as CSS length string. The empty string will remove the
         *  fixed width from the control element.
         *
         * @returns {WidthMixin}
         *  A reference to this instance.
         */
        this.setMinWidth = function (width) {
            controlNode.css({ minWidth: width });
            this.layout();
            return this;
        };

        /**
         * Sets the maximum width of the wrapped control element.
         *
         * @param {Number} width
         *  The maximum width to be set for the control element, as number in
         *  pixels, or as CSS length string. The empty string will remove the
         *  fixed width from the control element.
         *
         * @returns {WidthMixin}
         *  A reference to this instance.
         */
        this.setMaxWidth = function (width) {
            controlNode.css({ maxWidth: width });
            this.layout();
            return this;
        };

        // initialization -----------------------------------------------------

        // convert to jQuery object
        controlNode = $(controlNode).first();

        // set the initial width of the control element
        _.each(['width', 'minWidth', 'maxWidth'], function (name) {
            var value = Utils.getOption(initOptions, name);
            if (_.isNumber(value)) { value += 'px'; }
            if (_.isString(value)) { controlNode.css(name, value); }
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            controlNode = initOptions = null;
        });

    } // class WidthMixin

    // exports ================================================================

    return WidthMixin;

});
