/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/view/controls', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/lineheight',
    'io.ox/office/editframework/view/editcontrols',
    'io.ox/office/text/format/listcollection',
    'io.ox/office/text/view/labels',
    'io.ox/office/text/view/control/paragraphstylepicker',
    'io.ox/office/text/view/control/tablestylepicker',
    'io.ox/office/text/view/control/liststylepicker',
    'io.ox/office/text/view/control/changetrackbadge',
    'io.ox/office/text/view/control/drawingpositionpicker',
    'gettext!io.ox/office/text'
], function (Utils, LineHeight, EditControls, ListCollection, Labels, ParagraphStylePicker, TableStylePicker, ListStylePicker, ChangeTrackBadge, DrawingPositionPicker, gt) {

    'use strict';

    // static class TextControls ==============================================

    /**
     * Additional classes defining specialized GUI controls for the OX Text
     * application.
     *
     * @extends EditControls
     */
    var TextControls = _.extend({}, EditControls, {
        ParagraphStylePicker: ParagraphStylePicker,
        TableStylePicker: TableStylePicker,
        ListStylePicker: ListStylePicker,
        ChangeTrackBadge: ChangeTrackBadge,
        DrawingPositionPicker: DrawingPositionPicker
    });

    // class ParagraphAlignmentGroup ==========================================

    /**
     * A picker control for horizontal text alignment in paragraphs.
     *
     * @constructor
     *
     * @extends RadioGroup
     */
    TextControls.ParagraphAlignmentGroup = EditControls.RadioGroup.extend({ constructor: function (initOptions) {

        EditControls.RadioGroup.call(this, Utils.extendOptions({
            tooltip: Labels.PARA_ALIGNMENT_LABEL
        }, initOptions));

        // set accessible name - WCAG 2.0 Success Criterion 4.1.2
        this.getNode().attr({ 'aria-label': Labels.PARA_ALIGNMENT_LABEL, 'aria-labelledby': 'dummy' });

        _.each(Labels.PARA_ALIGNMENT_STYLES, function (style) {
            this.createOptionButton(style.value, { icon: style.icon, tooltip: style.label });
        }, this);

    }}); // class ParagraphAlignmentGroup

    // class ParagraphAlignmentPicker =========================================

    /**
     * A picker control for horizontal text alignment in paragraphs.
     *
     * @constructor
     *
     * @extends RadioGroup
     */
    TextControls.ParagraphAlignmentPicker = EditControls.RadioList.extend({ constructor: function (initOptions) {

        EditControls.RadioList.call(this, Utils.extendOptions({
            icon: 'docs-para-align-left',
            tooltip: Labels.PARA_ALIGNMENT_LABEL,
            updateCaptionMode: 'icon'
        }, initOptions));

        _.each(Labels.PARA_ALIGNMENT_STYLES, function (style) {
            this.createOptionButton(style.value, { icon: style.icon, label: style.label });
        }, this);

    }}); // class ParagraphAlignmentPicker

    // class TextPositionGroup ================================================

    /**
     * A button group control for text position (subscript/superscript).
     *
     * @constructor
     *
     * @extends RadioGroup
     */
    TextControls.TextPositionGroup = EditControls.RadioGroup.extend({ constructor: function (initOptions) {

        EditControls.RadioGroup.call(this, Utils.extendOptions({
            tooltip: Labels.TEXT_POSITION_LABEL,
            toggleValue: 'baseline'
        }, initOptions));

        // set accessible name - WCAG 2.0 Success Criterion 4.1.2
        this.getNode().attr({ 'aria-label': Labels.TEXT_POSITION_LABEL, 'aria-labelledby': 'dummy' });

        this.createOptionButton('sub',   { icon: 'docs-font-subscript',   tooltip: gt('Subscript'),     dropDownVersion: { label: gt('Subscript')} })
            .createOptionButton('super', { icon: 'docs-font-superscript', tooltip: gt('Superscript'),   dropDownVersion: { label: gt('Superscript')} });

    }}); // class TextPositionGroup

    // class LineHeightPicker =================================================

    /**
     * A picker control for the line height in paragraphs.
     *
     * @constructor
     *
     * @extends RadioList
     */
    TextControls.LineHeightPicker = EditControls.RadioList.extend({ constructor: function () {

        EditControls.RadioList.call(this, {
            icon: 'docs-para-line-spacing-100',
            tooltip: Labels.LINE_SPACING_LABEL,
            updateCaptionMode: 'icon',
            dropDownVersion: { label: Labels.LINE_SPACING_LABEL }
        });

        this.createOptionButton(LineHeight.SINGLE,   { icon: 'docs-para-line-spacing-100', label: /*#. text line spacing in paragraphs */ gt('100%') })
            .createOptionButton(LineHeight._115,     { icon: 'docs-para-line-spacing-115', label: /*#. text line spacing in paragraphs */ gt('115%') })
            .createOptionButton(LineHeight.ONE_HALF, { icon: 'docs-para-line-spacing-150', label: /*#. text line spacing in paragraphs */ gt('150%') })
            .createOptionButton(LineHeight.DOUBLE,   { icon: 'docs-para-line-spacing-200', label: /*#. text line spacing in paragraphs */ gt('200%') });

    }}); // class LineHeightPicker

    // class BulletListStylePicker ============================================

    TextControls.BulletListStylePicker = ListStylePicker.extend({ constructor: function (app, initOptions) {

        ListStylePicker.call(this, app, ListCollection.getPredefinedBulletListStyles(), Utils.extendOptions({
            icon: 'docs-list-bullet',
            tooltip: gt('Bullet list'),
            gridColumns: 4,
            dropDownVersion: { label: gt('Bullet list') }
        }, initOptions));

    }}); // class BulletListStylePicker


    // class DrawingPositionPicker ============================================

    /**
     *
     * Drop-down menu control for position and text floating of drawing
     * objects in text documents.
     *
     * @constructor
     *
     * @extends RadioList
     */
    TextControls.AnchoragePicker = EditControls.RadioList.extend({ constructor: function () {

        // base constructor ---------------------------------------------------

        EditControls.RadioList.call(this, {
            icon: 'docs-position-anchor',
            label: /*#. alignment and text floating of drawing objects in text documents */ gt.pgettext('drawing-pos', 'Position'),
            tooltip: /*#. alignment and text floating of drawing objects in text documents */ gt.pgettext('drawing-pos', 'Position'),
            updateCaptionMode: 'icon',
            smallerVersion: {
                hideLabel: true
            }
        });

        // initialization -----------------------------------------------------

        this.createOptionButton('inline',     { icon: 'docs-position-on-text',         label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Move with text') })
            .createOptionButton('paragraph',  { icon: 'docs-position-on-paragraph',    label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Move with paragraph') })
            .createOptionButton('page',       { icon: 'docs-position-on-page',         label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Move with page') });

    }}); // class DrawingPositionPicker

    // class NumberedListStylePicker ==========================================

    TextControls.NumberedListStylePicker = ListStylePicker.extend({ constructor: function (app, initOptions) {

        ListStylePicker.call(this, app, ListCollection.getPredefinedNumberedListStyles(), Utils.extendOptions({
            icon: 'docs-list-numbered',
            tooltip: gt('Numbered list'),
            gridColumns: 4,
            dropDownVersion: { label: gt('Numbered list') }
        }, initOptions));

    }}); // class NumberedListStylePicker

    // class BorderPresetStylePicker ==========================================

    /**
     * A drop-down list control for border line styles used in OOXML files.
     *
     * @constructor
     *
     * @extends EditControls.BorderStylePicker
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options supported by the base class
     *  EditControls.BorderStylePicker.
     */
    TextControls.BorderPresetStylePicker = EditControls.BorderStylePicker.extend({ constructor: function (app, listEntries) {

        // base constructor ---------------------------------------------------

        EditControls.BorderStylePicker.call(this, listEntries, {
            label: Labels.BORDER_STYLE_LABEL,
            smallerVersion:{ hideLabel: true}
        });

    }}); // class BorderPresetStylePicker

    // class DrawingPositionPicker ============================================

    /**
     *
     * Drop-down menu control for position and text floating of drawing
     * objects in text documents.
     *
     * @constructor
     *
     * @extends RadioList
     */
    TextControls.DrawingPositionPicker = EditControls.RadioList.extend({ constructor: function () {

        // base constructor ---------------------------------------------------

        EditControls.RadioList.call(this, {
            icon: 'docs-drawing-inline',
            label: /*#. alignment and text floating of drawing objects in text documents */ gt.pgettext('drawing-pos', 'Alignment'),
            tooltip: /*#. alignment and text floating of drawing objects in text documents */ gt.pgettext('drawing-pos', 'Drawing alignment'),
            updateCaptionMode: 'icon',
            smallerVersion: {
                hideLabel: true
            }
        });

        // initialization -----------------------------------------------------

        this//.createMenuSection('inline')
            // .createOptionButton('inline',       { section: 'inline', icon: 'docs-drawing-inline',       label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Inline with text') })
            .createMenuSection('left')
            .createOptionButton('left:none',    { section: 'left',   icon: 'docs-drawing-left-none',    label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Left aligned, no text wrapping') })
            .createOptionButton('left:right',   { section: 'left',   icon: 'docs-drawing-left-right',   label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Left aligned, text wraps at right side') })
            .createMenuSection('right')
            .createOptionButton('right:none',   { section: 'right',  icon: 'docs-drawing-right-none',   label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Right aligned, no text wrapping') })
            .createOptionButton('right:left',   { section: 'right',  icon: 'docs-drawing-right-left',   label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Right aligned, text wraps at left side') })
            .createMenuSection('center')
            .createOptionButton('center:none',  { section: 'center', icon: 'docs-drawing-center-none',  label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Centered, no text wrapping') })
            .createOptionButton('center:left',  { section: 'center', icon: 'docs-drawing-center-left',  label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Centered, text wraps at left side') })
            .createOptionButton('center:right', { section: 'center', icon: 'docs-drawing-center-right', label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Centered, text wraps at right side') });

    }}); // class DrawingPositionPicker
    // exports ================================================================

    return TextControls;

});
