/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/view/control/changetrackbadge', [
    'io.ox/contacts/api',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/errorcode',
    'io.ox/office/tk/io',
    'io.ox/office/tk/control/picture',
    'io.ox/office/text/view/labels',
    'gettext!io.ox/office/text'
], function (ContactsAPI, Utils, ErrorCode, IO, Picture, Labels, gt) {

    'use strict';

    // class ChangeTrackBadge =================================================

    /**
     * Drop-down menu control for position and text floating of drawing
     * objects in text documents.
     *
     * @constructor
     *
     * @extends Picture
     */
    function ChangeTrackBadge(app, initOptions) {

        var authorLinkNode = $('<a>').addClass('change-track-author').text(gt('Unknown')),
            dateNode = $('<div>').addClass('change-track-date').text(gt('Unknown')),
            actionNode = $('<div>').addClass('change-track-action').text(gt('Unknown')),
            descriptionNode = $('<div>').addClass('change-track-description'),
            pictureSize = Utils.RETINA ? 96 : 48;

        // base constructor ---------------------------------------------------

        Picture.call(this, Labels.DEFAULT_PROFILE_PICTURE, Utils.extendOptions({
            tooltip: gt('Author picture'),
            alt: gt('Author picture')
        }, initOptions));

        // public methods -----------------------------------------------------

        // updates properties of the badge
        this.update = function (data) {
            if (!data) { return; }
            var self = this;
            // set placeholder picture as default and reset author halo link
            self.setSource(Labels.DEFAULT_PROFILE_PICTURE);
            authorLinkNode.removeClass('halo-link').removeData();
            // fetch user photo and enable halo view only if author OX User ID is available
            if (data.authorUserId) {
                // try to get the authors user data on the local AppSuite
                IO.sendRequest(IO.FILTER_MODULE_NAME, {
                    action: 'getuserinfo',
                    userId: data.authorUserId
                }).done(function (userInfo) {
                    // quit if user data can not be found
                    var errorCode = new ErrorCode(userInfo);
                    if (errorCode.isError()) {
                        Utils.warn('TextControls.ChangeTrackBadge.update(): ' + errorCode.getErrorText());
                        return;
                    }
                    // if we found the author in the local user DB, try getting his picture
                    if (data.author && (userInfo.userData.displayName === data.author)) {
                        // enable author halo view
                        authorLinkNode.addClass('halo-link').data({ internal_userid : data.authorUserId });
                        // build user photo url
                        var pictureParams = { internal_userid: data.authorUserId, width: pictureSize, height: pictureSize, scaleType: 'cover' },
                            userPhotoSource = ContactsAPI.pictureHalo(null, pictureParams),
                            userPhotoPromise = app.createImageNode(userPhotoSource);
                        // Check if the image width is 1x1 pixel. This means this particular user does not have any profile pic uploaded yet.
                        // Fall back to dummy placeholder in this case, otherwise set the profile pic.
                        userPhotoPromise.done(function (imgNode) {
                            if (imgNode[0].width === 1) {
                                self.setSource(Labels.DEFAULT_PROFILE_PICTURE);
                                return;
                            }
                            self.setSource(userPhotoSource);
                        });
                        userPhotoPromise.fail(function () {
                            Utils.warn('TextControls.ChangeTrackBadge.initProfilePicture(): Failed to load user profile picture');
                        });
                    }
                });
            }
            authorLinkNode.text(data.author);
            dateNode.text(data.date);
            actionNode.text(data.action);
        };

        // initialization -----------------------------------------------------

        // show placeholder image initially
        this.setSource(Labels.DEFAULT_PROFILE_PICTURE);

        // insert nodes to group node
        descriptionNode.append(authorLinkNode, actionNode);
        this.addChildNodes(descriptionNode, dateNode);

        // ARIA role
        this.getNode().attr({ role: 'dialog' });

    } // class ChangeTrackBadge

    // exports ================================================================

    // derive this class from class Picture
    return Picture.extend({ constructor: ChangeTrackBadge });

});
