/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 */

define('io.ox/office/text/hyperlink', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/dialogs',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/utils/hyperlinkutils',
    'io.ox/office/editframework/view/hyperlinkutil',
    'io.ox/office/text/utils/operations',
    'io.ox/office/text/dom',
    'io.ox/office/text/position'
], function (Utils, KeyCodes, Dialogs, AttributeUtils, HyperlinkUtils, HyperlinkUtil, Operations, DOM, Position) {

    'use strict';

    // static private functions ===============================================

    /**
     * Shows the hyperlink pop-up menu.
     *
     * @param {TextApplication} app
     *  The text application instance.
     */
    function showPopup(app, hyperlinkPopup) {
        hyperlinkPopup.show();
        app.getView().scrollToChildNode(hyperlinkPopup);
    }

    /**
     * Hides the hyperlink pop-up menu.
     *
     * @param {jQuery} hyperlinkPopup
     *  The hyperlink pop-up which should be hidden.
     */
    function hidePopup(hyperlinkPopup) {
        hyperlinkPopup.hide();
    }

    function createResultFromHyperlinkSelection(position, hyperlinkSelection) {

        var result = null,
            start = _.clone(position),
            end = _.clone(position);

        // create result with correct Position objects
        start[start.length - 1] = hyperlinkSelection.start;
        end[end.length - 1] = hyperlinkSelection.end;
        result = { start: start, end: end, text: hyperlinkSelection.text, url: null };

        return result;
    }

    // static class Hyperlink =================================================

    /**
     * Provides static helper methods for manipulation and calculation
     * of a hyperlink.
     */
    var Hyperlink = {
        Separators: [ '!', '?', '.', ' ', '-', ':', ',', '\xa0'],
        Protocols: [ 'http://', 'https://', 'ftp://', 'mailto:' ],

        // templates for web/ftp identifications
        TextTemplates: [ 'www.', 'ftp.' ],

        /**
         * Predefined character attributes needed to remove a hyperlink style
         * from a piece of text.
         */
        CLEAR_ATTRIBUTES: { styleId: null, character: { url: null } }
    };

    // methods ----------------------------------------------------------------

    /**
     * Shows a hyperlink input dialog.
     *
     * @param {String} text
     *  The optional text which represents the URL
     *
     * @param {String} url
     *  An optional URL which is set for the supplied text
     *
     * @param {BaseApplication} app
     *  The calling app
     *
     * @returns {jQuery.Promise}
     *  The promise of a deferred object that will be resolved if the primary
     *  button has been activated, or rejected if the dialog has been canceled.
     *  The done handlers registered at the promise object will receive the
     *  entered text.
     */
    Hyperlink.showDialog = function (text, url, app) {
        return HyperlinkUtil.showDialog(text, url, app);
    };

    /**
     * Provides the URL of a selection without range.
     *
     * @param {Editor} editor
     *  The editor object.
     *
     * @param {Selection} selection
     *  A selection object which describes the current selection.
     *
     * @param {Object} [options]
     *  Optional options, introduced for performance reasons.
     *
     * @returns {Object}
     *  - {String|Null} url
     *      The URL or null if no URL character attribute is set at the
     *      selection.
     *  - {Boolean} beforeHyperlink
     *      True if we provide the URL for the hyperlink which is located at
     *      the next position.
     *  - {Boolean} behindHyperlink
     *      True if the current selection is exactly behind a hyperlink
     *  - {Boolean} clearAttributes
     *      True if the preselected attributes of the text model have to be
     *      extended by character attributes that remove the current URL style.
     */
    Hyperlink.getURLFromPosition = function (editor, selection, options) {
        var result = { url: null, beforeHyperlink: false, clearAttributes: false },
            splitOperation = Utils.getBooleanOption(options, 'splitOperation', false);

        if (!selection.hasRange()) {
            // find out a possible URL set for the current position
            var obj = null;

            if (splitOperation && selection.getInsertTextPoint()) {
                obj = selection.getInsertTextPoint();  // Performance: Using previously cached node point
            } else {
                obj = Position.getDOMPosition(editor.getNode(), selection.getStartPosition());
            }

            if (obj && obj.node && DOM.isTextSpan(obj.node.parentNode)) {
                var attributes = AttributeUtils.getExplicitAttributes(obj.node.parentNode, { family: 'character', direct: true });
                if (attributes.url) {
                    // Now we have to check some edge cases to prevent to show
                    // the popup for a paragraph which contains only an empty span
                    // having set the url attribute.
                    var span = $(obj.node.parentNode);
                    if ((span.text().length > 0) ||
                        (span.prev().length) ||
                        (DOM.isTextSpan(span.next())))
                        result.url = attributes.url;
                    else {
                        result.clearAttributes = true;
                    }
                }

                // Check next special case: Before a hyperlink we always want to show the popup
                if (splitOperation && selection.getParagraphCache()) {
                    // Performance: Using previously cached paragraph and text position
                    obj = Position.getDOMPosition(selection.getParagraphCache().node, [selection.getParagraphCache().offset + 1]);
                } else {
                    var nextPosition = selection.getStartPosition();
                    nextPosition[nextPosition.length - 1] += 1;
                    obj = Position.getDOMPosition(editor.getNode(), nextPosition);
                }

                if (obj && obj.node && DOM.isTextSpan(obj.node.parentNode)) {
                    var nextAttributes = AttributeUtils.getExplicitAttributes(obj.node.parentNode, { family: 'character', direct: true });
                    if (nextAttributes.url && (attributes.url !== nextAttributes.url)) {
                        result.url = nextAttributes.url;
                        result.beforeHyperlink = true;
                    }
                }
            }
        }

        return result;
    };

    /**
     * Tries to find a selection range based on the current text cursor
     * position. The URL character style is used to find a consecutive
     * range.
     *
     * @param {Editor} editor
     *  The editor object.
     *
     * @param {Selection} selection
     *  The current selection.
     *
     * @returns {Object}
     *  The new selection properties which includes a range with the same URL
     *  character style or null if no URL character style could be found.
     */
    Hyperlink.findSelectionRange = function (editor, selection) {
        var newSelection = null;

        if (!selection.hasRange() && selection.getEnclosingParagraph()) {
            var paragraph = selection.getEnclosingParagraph(),
                startPosition = selection.getStartPosition(),
                pos = null, url = null, result = null;

            pos = startPosition[startPosition.length - 1];

            if (!selection.hasRange()) {
                result = Hyperlink.getURLFromPosition(editor, selection);
                url = result.url;
            }

            if (url) {
                if (result.beforeHyperlink)
                    pos += 1;
                startPosition[startPosition.length - 1] = pos;
                newSelection = Hyperlink.findURLSelection(editor, startPosition, url);
            }
            else {
                newSelection = Hyperlink.findTextSelection(paragraph, pos);
            }
        }

        return newSelection;
    };

    /**
     * Tries to find a selection based on the provided position which includes
     * the provided url as character style.
     *
     * @param {Editor} editor
     *  The editor instance.
     *
     * @param {Number[]} startPosition
     *  The startPosition in the paragraph
     *
     * @param url
     *  The hyperlink URL which is set as character style at pos
     *
     * @returns {Object}
     *  Contains start and end position of the selection where both could be
     *  null which means that there is no selection but the hyperlink should be
     *  inserted at the position.
     */
    Hyperlink.findURLSelection = function (editor, startPosition, url) {
        var startPos,
            endPos,
            startNode = null,
            endNode = null,
            attributes = null,
            obj = null,
            characterStyles = editor.getStyleCollection('character'),
            result = { start: null, end: null };

        obj = Position.getDOMPosition(editor.getNode(), startPosition);
        if (obj && obj.node && DOM.isTextSpan(obj.node.parentNode)) {

            startNode = obj.node.parentNode;
            endNode = obj.node.parentNode;

            while (endNode && endNode.nextSibling && DOM.isTextSpan(endNode.nextSibling)) {
                attributes = characterStyles.getElementAttributes(endNode.nextSibling).character;
                if (attributes.url !== url)
                    break;
                endNode = endNode.nextSibling;
            }

            while (startNode && startNode.previousSibling && DOM.isTextSpan(startNode.previousSibling)) {
                attributes = characterStyles.getElementAttributes(startNode.previousSibling).character;
                if (attributes.url !== url)
                    break;
                startNode = startNode.previousSibling;
            }

            startPos = Position.getPositionRangeForNode(editor.getNode(), startNode, true);
            if (startNode !== endNode) {
                endPos = Position.getPositionRangeForNode(editor.getNode(), endNode, true);
            }
            else {
                endPos = startPos;
            }

            result = { start: startPos.start[startPos.start.length - 1], end: endPos.end[endPos.end.length - 1] };
        }

        return result;
    };

    /**
     * Find a text selection based on the provided position which is limited
     * by separator characters.
     *
     * @param {HTMLElement|jQuery} paragraph
     *  The paragraph which contains the position provided as the second
     *  argument.
     *
     * @param {Number[]} pos
     *  The position relative inside the paragraph
     *
     * @returns {Object}
     *  An object which contains the start and end position relative to the
     *  provided paragraph. Both can be null if there is no selection and the
     *  hyperlink should be inserted at pos.
     */
    Hyperlink.findTextSelection = function (paragraph, pos) {
        var text = '',
            startFound = false,
            startPos = -1,
            endPos = -1,
            selection = { start: null, end: null, text: null };

        Position.iterateParagraphChildNodes(paragraph, function (node, nodeStart, nodeLength, nodeOffset, offsetLength) {

            if (DOM.isTextSpan(node)) {
                var str = $(node).text(), mustConcat = true;

                if (nodeStart <= pos) {
                    if (startPos === -1)
                        startPos = nodeStart;
                    text = text.concat(str.slice(nodeOffset, nodeOffset + offsetLength));
                    mustConcat = false;
                }
                if ((nodeStart + nodeLength) > pos) {
                    if (!startFound) {
                        var leftPos = startPos;

                        startFound = true;
                        startPos = Hyperlink.findLeftWordPosition(text, leftPos, pos);
                        if (startPos === -1)
                            return Utils.BREAK;
                        else if (leftPos < startPos)
                            text = text.slice(startPos - leftPos, text.length);
                    }
                    if (mustConcat)
                        text = text.concat(str.slice(nodeOffset, nodeOffset + offsetLength));
                    endPos = Hyperlink.findRightWordPosition(text, startPos, pos);
                    if (endPos < (nodeStart + nodeLength)) {
                        text = text.slice(0, endPos - startPos);
                        return Utils.BREAK;
                    }
                }
            }
            else {
                if (startFound)
                    return Utils.BREAK;
                else {
                    text = '';
                    startPos = -1;
                }
            }
        });

        if ((startPos >= 0) && (endPos >= startPos))
            selection = { start: startPos, end: endPos, text: text };

        return selection;
    };

    /**
     * Tries to find the left position of a word using a predefined separator
     * array.
     *
     * @param {String} text
     *  The text which should be parsed to find the left bound of a selection.
     *
     * @param {Number} offset
     *  An offset to be used to provide correct position data.
     *
     * @param {Number} pos
     *  The absolute position to start with (pos - offset) is the relative
     *  position in the provided text.
     *
     * @param optSeparators {optional, Array}
     *  An array filled with separator characters which are a
     *  border for the left work position search.
     *
     * @returns {Number}
     *  The absolute position of the left boundary or -1 if the current
     *  position is the boundary.
     */
    Hyperlink.findLeftWordPosition = function (text, offset, pos, optSeparators) {
        var i = pos - offset,
            separators = (optSeparators === undefined) ? Hyperlink.Separators : optSeparators;

        if (_.contains(separators, text[i]))
            return -1;

        while (i >= 0 && !_.contains(separators, text[i])) {
            i--;
        }
        return offset + i + 1;
    };

    /**
     * Tries to find the right position of a word using a
     * predefined separator array.
     *
     * @param {String} text
     *  The text which should be parsed to find the right bound of a selection.
     *
     * @param {Number} offset
     *  An offset to be used to provide correct position data.
     *
     * @param {Number} pos
     *  The absolute position to start with (pos - offset) is the relative
     *  position in the provided text.
     *
     * @returns {Number}
     *  The absolute position of the right boundary or -1 if the current
     *  position is the boundary.
     */
    Hyperlink.findRightWordPosition = function (text, offset, pos) {
        var i = pos - offset, length = text.length;

        if (_.contains(Hyperlink.Separators, text[i]))
            return -1;

        while (i < length && !_.contains(Hyperlink.Separators, text[i])) {
            i++;
        }
        return offset + i;
    };

    /**
     * Find a text selection based on the provided position to the left
     * which is limited by the provided separator characters.
     *
     * @param {Editor} editor
     *  The editor instance.
     *
     * @param paragraph {HTMLElement|jQuery}
     *  The paragraph which contains the position provided as the second
     *  argument.
     *
     * @param pos {Number}
     *  The position relative inside the paragraph
     *
     * @param separators {Array}
     *  An array with separator character to look for the left border.
     *
     * @returns {Object}
     *  An object which contains the Object.start and Object.end position
     *  relative to the provided paragraph. Both can be null if there is
     *  no selection and the hyperlink should inserted at pos. Object.text
     *  contains the text within the selection.
     */
    Hyperlink.findLeftText = function (editor, paragraph, pos, separators) {
        var text = '',
            startFound = false,
            startPos = -1,
            endPos = -1,
            textFound = false,
            url = null,
            characterStyles = editor.getStyleCollection('character'),
            selection = { start: null, end: null, text: null, url: null };

        Position.iterateParagraphChildNodes(paragraph, function (node, nodeStart, nodeLength, nodeOffset, offsetLength) {
            var // containing the text of the current text span
                str,
                // left position used for extracting the text
                leftPos,
                // character attributes of the current node
                attributes;

            if (textFound)
                return Utils.BREAK;

            if (DOM.isTextSpan(node)) {
                str = $(node).text();
                attributes = characterStyles.getElementAttributes(node).character;

                if (nodeStart <= pos) {
                    if (startPos === -1) {
                        startPos = nodeStart;
                    }
                    text = text.concat(str.slice(nodeOffset, nodeOffset + offsetLength));
                    url = attributes.url;
                }
                if ((nodeStart + nodeLength) >= pos) {
                    if (!startFound) {
                        leftPos = startPos;

                        // we just need all text left inclusive pos
                        text = text.slice(0, pos);

                        startFound = true;
                        startPos = Hyperlink.findLeftWordPosition(text, leftPos, pos, separators);
                        endPos = Math.max(startPos, pos - 1);

                        // make the final slice using startPos and endPos
                        text = text.slice(startPos, endPos + 1);
                        textFound = true;
                        url = attributes.url;
                        return Utils.Break;
                    }
                }
            } else {
                if (startFound)
                    return Utils.BREAK;
                else {
                    if (DOM.isDrawingFrame(node) || DOM.isTabNode(node) || DOM.isFieldNode(node)) {
                        if (startPos === -1) {
                            startPos = nodeStart;
                        }
                        text = text + ' ';  // <- using an empty space for a drawings, fields or tabs (Fix for 28095)
                    } else {
                        text = '';
                        startPos = -1;
                    }
                }
            }
        });

        if ((startPos >= 0) && (endPos >= startPos))
            selection = { start: startPos, end: endPos, text: text, url: url };

        return selection;
    };

    /**
     * Checks for a text in a paragraph which defines a hyperlink
     * e.g. http://www.open-xchange.com
     *
     * @param {Editor} editor
     *  The editor instance.
     *
     * @param paragraph {HTMLElement/jQuery}
     *  The paragraph node to check
     *
     * @param position {Position}
     *  The rightmost position to check the text to the left for
     *  a hyperlink text.
     *
     * @returns {Object}
     *  Returns an object containing the start and end position
     *  to set the hyperlink style/url or null if no hyperlink
     *  has been found.
     */
    Hyperlink.checkForHyperlinkText = function (editor, paragraph, position) {
        if (position !== null) {
            var pos = position[position.length - 1],
                hyperlinkSelection = Hyperlink.findLeftText(editor, paragraph, pos, [' ', '\xa0']),
                url = null;

            if (hyperlinkSelection.url) {
                url = hyperlinkSelection.url;
            } else if(hyperlinkSelection.text) {
                url = HyperlinkUtils.checkForHyperlink(hyperlinkSelection.text);
            }
            if (url) {
                var result = createResultFromHyperlinkSelection(position, hyperlinkSelection);
                result.url = url;
                return result;
            }
        }

        return null;
    };

    /**
     * 'Inserts' a hyperlink at the provided selection using the
     * provided url.
     *
     * @param editor {Editor}
     *  The editor instance to use.
     *
     * @param start {Position}
     *  The start position of the selection
     *
     * @param end {Position}
     *  The end position of the selection.
     *
     * @param url {String}
     *  The url of the hyperlink to set at the selection
     */
    Hyperlink.insertHyperlink = function (editor, start, end, url) {
        var hyperlinkStyleId = editor.getDefaultUIHyperlinkStylesheet(),
            characterStyles = editor.getStyleCollection('character'),
            generator = editor.createOperationsGenerator();

        if (characterStyles.isDirty(hyperlinkStyleId)) {
            // insert hyperlink style to document
            generator.generateOperation(Operations.INSERT_STYLESHEET, {
                attrs: characterStyles.getStyleSheetAttributeMap(hyperlinkStyleId),
                type: 'character',
                styleId: hyperlinkStyleId,
                styleName: characterStyles.getName(hyperlinkStyleId),
                parent: characterStyles.getParentId(hyperlinkStyleId),
                uiPriority: characterStyles.getUIPriority(hyperlinkStyleId)
            });
            characterStyles.setDirty(hyperlinkStyleId, false);
        }

        generator.generateOperation(Operations.SET_ATTRIBUTES, {
            attrs: { styleId: hyperlinkStyleId, character: { url: url } },
            start: _.clone(start),
            end: _.clone(end)
        });

        editor.applyOperations(generator.getOperations());
    };

    /**
     * 'Removes' the hyperlink at the provided selection.
     *
     * @param editor {Editor}
     *  The editor instance to use.
     *
     * @param start {Position}
     *  The start position of the selection
     *
     * @param end {Position}
     *  The end position of the selection.
     */
    Hyperlink.removeHyperlink = function (editor, start, end) {

        var generator = editor.createOperationsGenerator();

        generator.generateOperation(Operations.SET_ATTRIBUTES, {
            attrs: Hyperlink.CLEAR_ATTRIBUTES,
            start: _.clone(start),
            end: _.clone(end)
        });

        editor.applyOperations(generator.getOperations());
    };

    /**
     * Checks if element is a clickable part of the hyperlink popup
     * return true if yes otherwise false.
     *
     * @param element {HTMLElement/jQuery}
     */
    Hyperlink.isClickableNode = function (element) {

        var node = $(element);

        if (Hyperlink.isPopupNode(node.parent())) {
            return node.is('a');
        }

        return false;
    };

    /**
     * Checks if element is the topmost div of a popup
     * returns true if yes otherwise false.
     *
     * @param element {HTMLElement/jQuery}
     *
     * @returns {Boolean}
     */
    Hyperlink.isPopupNode = function (element) {
        return $(element).is('.inline-popup');
    };

    /**
     * Check if element is the hyperlink popup or a child
     * within the popup.
     *
     * @param element {HTMLElement/jQuery}
     *
     * @returns {Boolean}
     */
    Hyperlink.isPopupOrChildNode = function (element) {
        var node = $(element);

        return (Hyperlink.isClickableNode(node, false) ||
                 Hyperlink.isPopupNode(node));
    };

    /**
     * Creates a hyperlink popup and appends it to the page DOM if it doesn't
     * exist, otherwise nothing is done.
     *
     * @param {Editor} editor
     *  The editor object.
     *
     * @param {TextApplication} app
     *  The application object.
     *
     * @param {Selection} selection
     *  The selection object of the editor.
     */
    Hyperlink.createPopup = function (editor, app, selection) {
        var page = editor.getNode();

        if (Hyperlink.getPopupNode(editor).length === 0) {
            var hyperlinkPopup = HyperlinkUtil.createPopupNode(app);

            page.append(hyperlinkPopup);

            var updateEditMode = function (options) {
                Hyperlink.updatePopup(editor, app, selection, hyperlinkPopup, options);
            };

            selection.on('change', function (event, options) {
                if (Utils.getBooleanOption(options, 'insertOperation', false) && (hyperlinkPopup[0].style.display === 'none')) {
                    // Performance: Use 'hyperlinkPopup[0].style.display' instead of jQuery, because of performance problems in Firefox
                    return false;  // Performance: Do nothing for insertText operations, if no hyperlink popup is visible
                }
                updateEditMode(options);
            });

            HyperlinkUtil.addEditModeListener(editor, hyperlinkPopup, updateEditMode);
        }
    };

    /**
     * Internal method to update the hyperlink popup
     *
     * @param {Editor} editor
     *  The editor object.
     *
     * @param {TextApplication} app
     *  The application object.
     *
     * @param {Selection} selection
     *  The current selection.
     *
     * @param {jQuery} hyperlinkPopup
     *  The hyperlink popup node as jQuery.
     *
     * @param {Object} [options]
     *  Optional options, introduced for performance reasons.
     */
    Hyperlink.updatePopup = function (editor, app, selection, hyperlinkPopup, options) {

        var result = Hyperlink.getURLFromPosition(editor, selection, options),
            page = editor.getNode();

        if (result.url) {
            var urlSelection = selection.getStartPosition(),
                obj = null;

            if (result.beforeHyperlink) {
                urlSelection[urlSelection.length - 1] += 1;
            }
            obj = Position.getDOMPosition(page, urlSelection);
            if (obj && obj.node && DOM.isTextSpan(obj.node.parentNode)) {
                var pos = urlSelection[urlSelection.length - 1],
                    startEndPos = Hyperlink.findURLSelection(editor, urlSelection, result.url),
                    left, top, height, outerWidth, width, diffWidth;

                if (pos !== startEndPos.end || result.beforeHyperlink) {
                    // find out position of the first span of our selection
                    urlSelection[urlSelection.length - 1] = startEndPos.start;
                    obj = Position.getDOMPosition(page, urlSelection, true);
                    left = $(obj.node).offset().left;

                    // find out position of the last span of our selection
                    urlSelection[urlSelection.length - 1] = startEndPos.end - 1;
                    obj = Position.getDOMPosition(page, urlSelection, true);
                    top = $(obj.node).offset().top;
                    height = $(obj.node).height();

                    // calculate position relative to the application pane
                    var parent = page,
                        parentLeft = parent.offset().left,
                        parentTop = parent.offset().top,
                        parentWidth = parent.innerWidth(),
                        zoomFactor = app.getView().getZoomFactor();

                    left = (left - parentLeft) / zoomFactor * 100;
                    top = (top - parentTop) / zoomFactor * 100 + height;

                    HyperlinkUtil.updatePopup(result.url, hyperlinkPopup);

                    hyperlinkPopup.css({left: left, top: top, width: '', height: ''});
                    outerWidth = hyperlinkPopup.outerWidth();
                    width = hyperlinkPopup.width();
                    diffWidth = outerWidth - width;
                    if ((left + outerWidth) > parentWidth) {
                        left = Math.max(Utils.getElementCssLength(page, 'padding-left'), parentWidth - outerWidth);
                        if (!_.browser.IE) {
                            outerWidth = Math.min(outerWidth, parentWidth);
                            width = outerWidth - diffWidth;
                        }
                        hyperlinkPopup.css({left: left, width: width});
                    }
                    if (pos === startEndPos.start || result.beforeHyperlink) {
                        // special case: at the start of a hyperlink we want to
                        // write with normal style
                        editor.addPreselectedAttributes(Hyperlink.CLEAR_ATTRIBUTES);
                    }
                    if (HyperlinkUtils.hasSupportedProtocol(result.url)) {
                        if (hyperlinkPopup.css('display') === 'none') {
                            showPopup(app, hyperlinkPopup);
                        }
                    } else {
                        hidePopup(hyperlinkPopup);
                    }
                }
                else {
                    // special case: at the end of a hyperlink we want to
                    // write with normal style and we don't show the popup
                    editor.addPreselectedAttributes(Hyperlink.CLEAR_ATTRIBUTES);
                    hidePopup(hyperlinkPopup);
                }
            }
        }
        else {
            hidePopup(hyperlinkPopup);
            if (result.clearAttributes) {
                editor.addPreselectedAttributes(Hyperlink.CLEAR_ATTRIBUTES);
            }
        }
    };

    /**
     * Provides the text from a selection.
     *
     * @param {Selection} selection
     *  A selection object which is used to extract the text from.
     */
    Hyperlink.getSelectedText = function (selection) {
        var text = '';

        selection.iterateNodes(function (node, pos, start, length) {
            if ((start >= 0) && (length >= 0) && DOM.isTextSpan(node)) {
                var nodeText = $(node).text();
                if (nodeText) {
                    text = text.concat(nodeText.slice(start, start + length));
                }
            }
        });

        return text;
    };

    /**
     * Returns the hyperlink pop-up menu, as jQuery object.
     *
     * @param {Editor} editor
     *  An editor instance.
     *
     * @returns {jQuery}
     *  The hyperlink pop-up menu node, as jQuery object.
     */
    Hyperlink.getPopupNode = function (editor) {
        return editor.getNode().children('.inline-popup.hyperlink');
    };

    /**
     * Returns whether the hyperlink pop-up menu is visible.
     *
     * @param {Editor} editor
     *  An editor instance.
     *
     * @returns {Boolean}
     *  Whether the hyperlink pop-up menu is currently open.
     */
    Hyperlink.hasPopup = function (editor) {
        return Hyperlink.getPopupNode(editor).css('display') !== 'none';
    };

    /**
     * Closes an open hyperlink pop-up menu.
     *
     * @param {Editor} editor
     *  An editor instance.
     *
     * @returns {Boolean}
     *  Whether the pop-up menu was actually open.
     */
    Hyperlink.closePopup = function (editor) {
        var visible = Hyperlink.hasPopup(editor);
        hidePopup(Hyperlink.getPopupNode(editor));
        return visible;
    };

    // exports ================================================================

    return Hyperlink;

});
