/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/dom',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/drawinglayer/view/drawingframe',
     'io.ox/office/tk/forms',
     'gettext!io.ox/office/text'
    ], function (Utils, KeyCodes, DrawingFrame, Forms, gt) {

    'use strict';

    var // key codes of keys that change the text cursor position backwards
        ARROW_CURSOR_KEYS = _([ KeyCodes.UP_ARROW, KeyCodes.RIGHT_ARROW, KeyCodes.DOWN_ARROW, KeyCodes.LEFT_ARROW ]),
        // key codes of keys that change the text cursor position backwards
        BACKWARD_CURSOR_KEYS = _([ KeyCodes.PAGE_UP, KeyCodes.HOME, KeyCodes.LEFT_ARROW, KeyCodes.UP_ARROW ]),

        // key codes of keys that change the text cursor position forwards
        FORWARD_CURSOR_KEYS = _([ KeyCodes.PAGE_DOWN, KeyCodes.END, KeyCodes.RIGHT_ARROW, KeyCodes.DOWN_ARROW ]),

        // key codes of keys that will be passed directly to the browser
        IGNORABLE_KEYS = _([
            KeyCodes.SHIFT, KeyCodes.CONTROL, KeyCodes.ALT,
            KeyCodes.CAPS_LOCK, KeyCodes.NUM_LOCK, KeyCodes.SCROLL_LOCK,
            KeyCodes.BREAK, KeyCodes.PRINT, KeyCodes.SELECT,
            KeyCodes.LEFT_WINDOWS, KeyCodes.RIGHT_WINDOWS,
            KeyCodes.F5
        ]);

    // static class DOM =======================================================

    /**
     * Provides classes representing DOM points (DOM.Point) and ranges
     * (DOM.Range), and static helper methods for basic editor DOM
     * manipulation, and access the browser selection.
     */
    var DOM = {};

    // class DOM.Point ========================================================

    /**
     * A DOM text point contains a 'node' attribute referring to a DOM node,
     * and an 'offset' attribute containing an integer offset specifying the
     * position in the contents of the node.
     *
     * @constructor
     *
     * @param {Node|jQuery} node
     *  The DOM node selected by this DOM.Point instance. If this object is a
     *  jQuery collection, uses the first DOM node it contains.
     *
     * @param {Number} [offset]
     *  An integer offset relative to the DOM node specifying the position in
     *  the node's contents. If the node is a text node, the offset represents
     *  the character position in the node text. If the node is an element
     *  node, the offset specifies the index of a child node of this node. The
     *  value of the offset may be equal to the text length respectively the
     *  number of child nodes, in this case the DOM point refers to the
     *  position directly after the node's contents. If omitted, this DOM.Point
     *  instance refers to the start of the entire node, instead of specific
     *  contents.
     */
    DOM.Point = function (node, offset) {
        this.node = Utils.getDomNode(node);
        this.offset = offset;
    };

    // methods ----------------------------------------------------------------

    /**
     * Returns a new clone of this DOM point.
     */
    DOM.Point.prototype.clone = function () {
        return new DOM.Point(this.node, this.offset);
    };

    /**
     * Validates the offset of this DOM point. Restricts the offset to the
     * available index range according to the node's contents, or
     * initializes the offset, if it is missing.
     *
     * If this instance points to a text node, the offset will be
     * restricted to the text in the node, or set to zero if missing.
     *
     * If this instance points to an element node, the offset will be
     * restricted to the number of child nodes in the node. If the offset
     * is missing, it will be set to the index of the node in its siblings,
     * and the node will be replaced by its parent node.
     *
     * @returns {DOM.Point}
     *  A reference to this instance.
     */
    DOM.Point.prototype.validate = function () {

        // element: if offset is missing, take own index and refer to the parent node
        if (this.node.nodeType === 1) {
            if (_.isNumber(this.offset)) {
                this.offset = Utils.minMax(this.offset, 0, this.node.childNodes.length);
            } else {
                this.offset = $(this.node).index();
                this.node = this.node.parentNode;
            }

        // text node: if offset is missing, use zero
        } else if (this.node.nodeType === 3) {
            if (_.isNumber(this.offset)) {
                this.offset = Utils.minMax(this.offset, 0, this.node.nodeValue.length);
            } else {
                this.offset = 0;
            }
        }

        return this;
    };

    /**
     * Converts this DOM point to a human readable string representation.
     */
    DOM.Point.prototype.toString = function () {

        // generates a readable description of the passed node and offset
        function getNodeName(node, offset) {

            var // full string representation of this DOM Point
                result = node.nodeName.toLowerCase();

            if ((node.nodeType === 1) && (node.className.length > 0)) {
                // add class names of an element
                result += '.' + node.className.replace(/ /g, '.');
            } else if (node.nodeType === 3) {
                // add some text of a text node
                result += '"' + node.nodeValue.substr(0, 10) + ((node.nodeValue.length > 10) ? Utils.ELLIPSIS_CHAR : '') + '"';
            }

            if (_.isNumber(offset)) {
                result += ':' + offset;
            }

            return result;
        }

        return getNodeName(this.node.parentNode, $(this.node).index()) + '>' + getNodeName(this.node, this.offset);
    };

    // static methods ---------------------------------------------------------

    /**
     * Creates and returns a valid DOM.Point instance for the passed DOM node.
     * If the passed node is a text node, the DOM point will refer to its first
     * character, otherwise the DOM point will contain the parent node and the
     * child index of the passed node as offset.
     *
     * @param {Node|jQuery} node
     *  The DOM node selected by the created DOM.Point instance. If this object
     *  is a jQuery collection, uses the first DOM node it contains.
     *
     * @returns {DOM.Point}
     *  A new DOM.Point instance referring to the passed node.
     */
    DOM.Point.createPointForNode = function (node) {
        return new DOM.Point(node).validate();
    };

    /**
     * Returns whether the two passed DOM points are equal.
     *
     * @param {DOM.Point} point1
     *  The first DOM point. Must be valid (see DOM.Point.validate() method for
     *  details).
     *
     * @param {DOM.Point} point2
     *  The second DOM point. Must be valid (see DOM.Point.validate() method
     *  for details).
     *
     * @returns {Boolean}
     *  Whether the DOM points are equal.
     */
    DOM.Point.equalPoints = function (point1, point2) {
        return (point1.node === point2.node) && (point1.offset === point2.offset);
    };

    /**
     * Returns an integer indicating how the two DOM points are located to each
     * other.
     *
     * @param {DOM.Point} point1
     *  The first DOM point. Must be valid (see DOM.Point.validate() method for
     *  details).
     *
     * @param {DOM.Point} point2
     *  The second DOM point. Must be valid (see DOM.Point.validate() method
     *  for details).
     *
     * @returns {Number}
     *  The value zero, if the DOM points are equal, a negative number, if
     *  point1 precedes point2, or a positive number, if point1 follows point2.
     */
    DOM.Point.comparePoints = function (point1, point2) {

        // Returns the index of the inner node's ancestor in the outer node's
        // children list. 'outerNode' MUST contain 'innerNode'.
        function calculateOffsetInOuterNode(outerNode, innerNode) {
            while (innerNode.parentNode !== outerNode) {
                innerNode = innerNode.parentNode;
            }
            return $(innerNode).index();
        }

        // equal nodes: compare by offset
        if (point1.node === point2.node) {
            return point1.offset - point2.offset;
        }

        // Node in point1 contains the node in point2: point1 is before point2,
        // if offset of point1 (index of its child node) is less than or equal
        // to the offset of point2's ancestor node in the children of point1's
        // node. If offsets are equal, point2 is a descendant of the child node
        // pointed to by point1 and therefore located after point1.
        if (Utils.containsNode(point1.node, point2.node)) {
            return (point1.offset <= calculateOffsetInOuterNode(point1.node, point2.node)) ? -1 : 1;
        }

        // Node in point2 contains the node in point1: see above, reversed.
        if (Utils.containsNode(point2.node, point1.node)) {
            return (calculateOffsetInOuterNode(point2.node, point1.node) < point2.offset) ? -1 : 1;
        }

        // Neither node contains the other: compare nodes regardless of offset.
        return Utils.compareNodes(point1.node, point2.node);
    };

    // class DOM.Range ========================================================

    /**
     * A DOM text range represents a half-open range in the DOM tree. It
     * contains 'start' and 'end' attributes referring to DOM point objects.
     *
     * @constructor
     *
     * @param {DOM.Point} start
     *  The DOM point where the range starts.
     *
     * @param {DOM.Point} [end]
     *  The DOM point where the range ends. If omitted, uses the start position
     *  to construct a collapsed range (a simple 'cursor').
     */
    DOM.Range = function (start, end) {
        this.start = start;
        this.end = _.isObject(end) ? end : _.clone(start);
    };

    // methods ----------------------------------------------------------------

    /**
     * Returns a new clone of this DOM range.
     */
    DOM.Range.prototype.clone = function () {
        return new DOM.Range(this.start.clone(), this.end.clone());
    };

    /**
     * Validates the start and end position of this DOM range. See method
     * DOM.Point.validate() for details.
     */
    DOM.Range.prototype.validate = function () {
        this.start.validate();
        this.end.validate();
        return this;
    };

    /**
     * Swaps start and end position, if the start position is located after
     * the end position in the DOM tree.
     */
    DOM.Range.prototype.adjust = function () {
        if (DOM.Point.comparePoints(this.start, this.end) > 0) {
            var tmp = this.start;
            this.start = this.end;
            this.end = tmp;
        }
        return this;
    };

    /**
     * Returns whether the two passed DOM ranges are equal.
     *
     * @param {DOM.Range} range1
     *  The first DOM range. Must be valid (see DOM.Range.validate() method for
     *  details).
     *
     * @param {DOM.Range} range2
     *  The second DOM range. Must be valid (see DOM.Range.validate() method
     *  for details).
     *
     * @returns {Boolean}
     *  Whether the DOM ranges are equal.
     */
    DOM.Range.equalRanges = function (range1, range2) {
        return (DOM.Point.equalPoints(range1.start, range2.start) && DOM.Point.equalPoints(range1.end, range2.end));
    };

    /**
     * Returns whether the two passed DOM ranges have the same offsets (the nodes might be different).
     * This can be used for column selections, where the row might be different, but the offset are
     * constant inside a column.
     *
     * @param {DOM.Range} range1
     *  The first DOM range. Must be valid (see DOM.Range.validate() method for
     *  details).
     *
     * @param {DOM.Range} range2
     *  The second DOM range. Must be valid (see DOM.Range.validate() method
     *  for details).
     *
     * @returns {Boolean}
     *  Whether the DOM ranges have the same offsets.
     */
    DOM.Range.equalRangeOffsets = function (range1, range2) {
        return (range1.start.offset === range2.start.offset) && (range1.end.offset === range2.end.offset);
    };

    /**
     * Returns whether the DOM range is collapsed, i.e. start position and
     * end position are equal.
     *
     * @returns {Boolean}
     *  Whether this DOM range is collapsed.
     */
    DOM.Range.prototype.isCollapsed = function () {
        return DOM.Point.equalPoints(this.start, this.end);
    };

    /**
     * Converts this DOM range to a human readable string representation.
     */
    DOM.Range.prototype.toString = function () {
        return '[start=' + this.start + ', end=' + this.end + ']';
    };

    // static methods ---------------------------------------------------------

    /**
     * Creates a new DOM.Range instance from the passed nodes and offsets.
     *
     * @param {Node|jQuery} startNode
     *  The DOM node used for the start point of the created range. If this
     *  object is a jQuery collection, uses the first DOM node it contains.
     *
     * @param {Number} [startOffset]
     *  The offset for the start point of the created range.
     *
     * @param {Node|jQuery} [endNode]
     *  The DOM node used for the end point of the created range. If this
     *  object is a jQuery collection, uses the first DOM node it contains. If
     *  omitted, creates a collapsed range by cloning the start position.
     *
     * @param {Number} [endOffset]
     *  The offset for the end point of the created range. Not used, if endNode
     *  has been omitted.
     *
     * @returns {DOM.Range}
     *  The new DOM range object.
     */
    DOM.Range.createRange = function (startNode, startOffset, endNode, endOffset) {
        return new DOM.Range(new DOM.Point(startNode, startOffset), _.isObject(endNode) ? new DOM.Point(endNode, endOffset) : undefined);
    };

    /**
     * Creates and returns a valid DOM.Range instance for the passed DOM node.
     * If the passed node is a text node, the DOM range will select its entire
     * text, otherwise the DOM range will contain the parent node and the
     * child index of the passed node as start offset, and the next child index
     * as end offset, effectively selecting the entire node.
     *
     * @param {Node|jQuery} node
     *  The DOM node selected by the created DOM.Range instance. If this object
     *  is a jQuery collection, uses the first DOM node it contains.
     *
     * @returns {DOM.Range}
     *  A new DOM.Range instance referring to the passed node.
     */
    DOM.Range.createRangeForNode = function (node) {
        var range = new DOM.Range(DOM.Point.createPointForNode(node));
        if (range.end.node.nodeType === 1) {
            range.end.offset += 1;
        } else if (range.end.node.nodeType === 3) {
            range.end.offset = range.end.node.nodeValue.length;
        }
        return range;
    };

    // key codes ==============================================================

    /**
     * Returns whether the passed key code is a cursor navigation key that
     * moves the cursor backwards in the document.
     *
     * @param {Number} keyCode
     *  The key code from a 'keydown' browser event.
     */
    DOM.isBackwardCursorKey = function (keyCode) {
        return BACKWARD_CURSOR_KEYS.contains(keyCode);
    };

    /**
     * Returns whether the passed key code is a cursor navigation key that
     * moves the cursor forwards in the document.
     *
     * @param {Number} keyCode
     *  The key code from a 'keydown' browser event.
     */
    DOM.isForwardCursorKey = function (keyCode) {
        return FORWARD_CURSOR_KEYS.contains(keyCode);
    };

    /**
     * Returns whether the passed key code is a cursor navigation key.
     *
     * @param {Number} keyCode
     *  The key code from a 'keydown' browser event.
     */
    DOM.isCursorKey = function (keyCode) {
        return DOM.isBackwardCursorKey(keyCode) || DOM.isForwardCursorKey(keyCode);
    };

    /**
     * Returns whether the passed key code is an 'arrow' cursor navigation key.
     *
     * @param {Number} keyCode
     *  The key code from a 'keydown' browser event.
     */
    DOM.isArrowCursorKey = function (keyCode) {
        return ARROW_CURSOR_KEYS.contains(keyCode);
    };

    /**
     * Returns whether the passed key code has to be ignored silently.
     *
     * @param {Number} keyCode
     *  The key code from a 'keydown' browser event.
     */
    DOM.isIgnorableKey = function (keyCode) {
        return IGNORABLE_KEYS.contains(keyCode);
    };

    // pages ==================================================================

    /**
     * A jQuery selector that matches elements representing a page.
     */
    DOM.PAGE_NODE_SELECTOR = 'div.page';

    /**
     * Creates a new page element.
     *
     * @returns {jQuery}
     *  A page element, as jQuery object.
     */
    DOM.createPageNode = function () {
        return $('<div>').addClass('page').append($('<div>').addClass('pagecontent'));
    };

    /**
     * Returns whether the passed node is a page element.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether the passed node is a page element.
     */
    DOM.isPageNode = function (node) {
        return $(node).is(DOM.PAGE_NODE_SELECTOR);
    };

    /**
     * A jQuery selector that matches elements representing a page content node.
     */
    DOM.PAGECONTENT_NODE_SELECTOR = 'div.pagecontent';

    /**
     * Returns whether the passed node is a page content node that contains all
     * top-level content nodes (paragraphs and tables).
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a page content node.
     */
    DOM.isPageContentNode = function (node) {
        return $(node).is(DOM.PAGECONTENT_NODE_SELECTOR);
    };

    /**
     * Returns the container node of a page element that contains all top-level
     * content nodes (paragraphs and tables).
     *
     * @param {HTMLElement|jQuery} pageNode
     *  The page DOM node. If this object is a jQuery collection, uses the
     *  first DOM node it contains.
     *
     * @returns {jQuery}
     *  The container DOM node from the passed page that contains all top-level
     *  content nodes (paragraphs and tables).
     */
    DOM.getPageContentNode = function (pageNode) {
        return $(pageNode).children(DOM.PAGECONTENT_NODE_SELECTOR);
    };

    /**
     * A jQuery selector that matches elements representing an application content node.
     */
    DOM.APPCONTENT_NODE_SELECTOR = 'div.app-content';

    /**
     * Returns whether the passed node is an application content node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an application content node.
     */
    DOM.isAppContentNode = function (node) {
        return $(node).is(DOM.APPCONTENT_NODE_SELECTOR);
    };

    /**
     * A jQuery selector that matches elements representing a page break node.
     */
    DOM.PAGEBREAK_NODE_SELECTOR = 'div.page-break';

    /**
     * Returns whether the passed node is a page break node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a page break node.
     */
    DOM.isPageBreakNode = function (node) {
        return $(node).is(DOM.PAGEBREAK_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node contains at least one page break node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node contains a page break node.
     */
    DOM.hasPageBreakNode = function (node) {
        return $(node).has(DOM.PAGEBREAK_NODE_SELECTOR);
    };


    // paragraphs and tables ==================================================

    /**
     * A jQuery selector that matches elements representing a paragraph.
     */
    DOM.PARAGRAPH_NODE_SELECTOR = 'div.p';

    /**
     * A jQuery selector that matches elements representing a paragraph (also in header and footer).
     */
    DOM.PARAGRAPH_NODE_SELECTOR_COMPLETE = 'div.p, div.par';

    /**
     * Returns whether the passed node is a paragraph element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a paragraph element.
     */
    DOM.isParagraphNode = function (node) {
        return $(node).is(DOM.PARAGRAPH_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is an empty paragraph element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an empty paragraph element. It is empty, if
     *  it only contains an empty text span and a 'br' element.
     */
    DOM.isEmptyParagraph = function (node) {
        return ($(node).children().length === 2 && DOM.isEmptySpan($(node).children().first()) && $(node).children().last().is('br'));
    };

    /**
     * Returns whether the passed node is a paragraph element in a header or footer.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a paragraph element in a header or footer.
     */
    DOM.isMarginParagraphNode = function (node) {
        return $(node).is('div.par');
    };

    /**
     * Returns whether the passed node is a paragraph element. This can be a div.p
     * element or a div.par element located in a header or footer.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a paragraph element.
     */
    DOM.isParagraphNodeComplete = function (node) {
        return DOM.isParagraphNode(node) || DOM.isMarginParagraphNode(node);
    };

    /**
     * Creates a new paragraph element.
     *
     * @returns {jQuery}
     *  A paragraph element, as jQuery object.
     */
    DOM.createParagraphNode = function () {
        return $('<div>').addClass('p');
    };

    /**
     * Creates a new paragraph marked as 'implicit'. This is used in
     * empty documents or table cells to allow text insertion in browser.
     *
     * @returns {jQuery}
     *  An implicit paragraph node, as jQuery object.
     */
    DOM.createImplicitParagraphNode = function () {
        return DOM.createParagraphNode().data('implicit', true);
    };

    /**
     * Flag a paragraph node with data flag 'implicit'. This paragraphs are
     * used in empty documents or table cells to allow text insertion in browser.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be flagged.
     */
    DOM.makeParagraphNodeImplicit = function (node) {
        $(node).data('implicit', true);
    };

    /**
     * Returns whether the passed paragraph node is an implicit paragraph
     * node that is used to allow text insertion into document (into empty
     * document or cells). This paragraph nodes were created without sending
     * an operation to the server.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is an implicit paragraph node.
     */
    DOM.isImplicitParagraphNode = function (node) {
        return $(node).data('implicit') === true;
    };

    /**
     * Returns whether the passed paragraph node is a paragraph without
     * neighbours (other paragraphs or tables).
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is a paragraph without neighbours.
     */
    DOM.isParagraphWithoutNeighbour = function (node) {
        return $(node).parent().children(DOM.CONTENT_NODE_SELECTOR).length === 1;
    };

    /**
     * Returns whether the passed paragraph node is a paragraph that is a last
     * paragraph directly behind a table.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is a paragraph following a table and with no
     *  following sibling.
     */
    DOM.isFinalParagraphBehindTable = function (node) {
        return ((DOM.isTableNode($(node).prev())) && ($(node).next().length === 0));
    };

    /**
     * Returns whether the passed paragraph node is a paragraph that is a last
     * paragraph directly behind a table and that is empty. This paragraph contains
     * only an empty text span and a 'br' element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is an empty paragraph following a table and with no
     *  following sibling.
     */
    DOM.isFinalEmptyParagraphBehindTable = function (node) {
        return DOM.isFinalParagraphBehindTable(node) && DOM.isEmptyParagraph(node);
    };

    /**
     * Returns whether the passed paragraph node is an empty paragraph that is a last
     * paragraph directly behind a table and that is empty and that is positioned inside
     * a table cell.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is an empty paragraph following a table and with no
     *  following sibling. Furthermore the paragraph is positioned inside a table cell.
     */
    DOM.isFinalEmptyParagraphBehindTableInCell = function (node) {
        return DOM.isCellContentNode($(node).parent()) && DOM.isFinalEmptyParagraphBehindTable(node);
    };

    /**
     * Returns whether the height of the passed paragraph node can be increased and decreased
     * dynamically. This can happen for implicit paragraph nodes that are behind tables or for
     * non-implicit paragraph nodes, that are empty, behind a table, with no following sibling
     * and inside a table cell.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is a paragraph, whose height need to be set dynamically.
     */
    DOM.isIncreasableParagraph = function (node) {
        return ((DOM.isImplicitParagraphNode(node)) && (DOM.isFinalParagraphBehindTable(node))) || DOM.isFinalEmptyParagraphBehindTableInCell(node);
    };

    /**
     * Returns whether the passed paragraph node is a paragraph that can be merged
     * with a following paragraph.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is a paragraph that is followed by a further paragraph.
     */
    DOM.isMergeableParagraph = function (node) {
        if ($(node).next().hasClass('page-break')) {
            return (($(node).next().next().length !== 0) && (DOM.isParagraphNode($(node).next().next())));
        } else {
            return (($(node).next().length !== 0) && (DOM.isParagraphNode($(node).next())));
        }
    };

    /**
     * A jQuery selector that matches elements representing a table.
     */
    DOM.TABLE_NODE_SELECTOR = 'table';

    /**
     * A jQuery selector that matches elements representing a table with exceeded size.
     */
    DOM.TABLE_SIZE_EXCEEDED_NODE_SELECTOR = 'table.size-exceeded';

    /**
     * A jQuery selector that matches placeholder div elements included into an exceeded size table.
     */
    DOM.PLACEHOLDER_SELECTOR = 'div.placeholder';

    /**
     * Returns whether the passed node is a table element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table element.
     */
    DOM.isTableNode = function (node) {
        return $(node).is(DOM.TABLE_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed table node is a replacement table for
     * non-editable oversized tables. This table node is created without
     * further operations and takes care of its presentation.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is a replacement table for oversized tables.
     */
    DOM.isExceededSizeTableNode = function (node) {
        return DOM.isTableNode(node) && $(node).hasClass('size-exceeded');
    };

    /**
     * Makes a exceeded size table placeholder from a provided empty table node.
     *
     * @paaram {Node|jQuery|Null} [tableNode]
     */
    DOM.makeExceededSizeTable = function (tableNode, overflowElement, value, maxValue) {
        var placeHolder,
            cell = $('<td>').append(
                       placeHolder = $('<div>').addClass('placeholder').append(
                        $('<div>').addClass('abs background-icon').append(Forms.createIconMarkup('fa-table')),
                        $('<p>').text(gt('Table')),
                        $('<p>').text(gt('This table is too large to be displayed here.'))
                    )),
            labels = DOM.getTableSizeExceededTextLabels(overflowElement, value, maxValue);

        if (labels.countLabel && labels.limitLabel) {
            placeHolder.append($('<p>').text(labels.countLabel + ' ' + labels.limitLabel));
        }

        $(tableNode).addClass('size-exceeded')
            .attr('contenteditable', false)
            .append($('<tr>').append(cell));
    };

    /**
    * Provides the correct localized labels for a table exceeds size case.
    *
    * @param {String} [type]
    *  Specifies which part of a table exceeded the size. Currently the parts
    *  "rows", "cols", "cells".
    *
    * @param {Integer} [value]
    *  Specifies the current value of the table part which exceeds the limit.
    *
    * @param {Integer} [maxValue]
    *  Specifies the maximal allowed value of the table part which exceeds the
    *  limit.
    */
    DOM.getTableSizeExceededTextLabels = function (type, value, maxValue) {
        var result = {countLabel: null, limitLabel: null};

        switch (type) {
        case 'rows':
            result.countLabel = gt.format(
                   //#. %1$d is the number of rows in an oversized text table
                   //#, c-format
                   gt.ngettext('The table contains %1$d row.', 'This table contains %1$d rows.', value),
                   _.noI18n(value)
            );
            result.limitLabel = gt.format(
                   //#. %1$d is the maximum allowed number of rows in a text table
                   //#, c-format
                   gt.ngettext('Tables are limited to %1$d row.', 'Tables are limited to %1$d rows.', maxValue),
                   _.noI18n(maxValue)
           );
            break;
        case 'cols':
            result.countLabel = gt.format(
                   //#. %1$d is the number of columns in an oversized text table
                   //#, c-format
                   gt.ngettext('The table contains %1$d column.', 'This table contains %1$d columns.', value),
                   _.noI18n(value)
            );
            result.limitLabel = gt.format(
                   //#. %1$d is the maximum allowed number of columns in a text table
                   //#, c-format
                   gt.ngettext('Tables are limited to %1$d column.', 'Tables are limited to %1$d columns.', maxValue),
                   _.noI18n(maxValue)
            );
            break;
        case 'cells':
            result.countLabel = gt.format(
                   //#. %1$d is the number of cells in an oversized text table
                   //#, c-format
                   gt.ngettext('The table contains %1$d cell.', 'This table contains %1$d cells.', value),
                   _.noI18n(value)
            );
            result.limitLabel = gt.format(
                   //#. %1$d is the maximum allowed number of cells in a text table
                   //#, c-format
                   gt.ngettext('Tables are limited to %1$d cell.', 'Tables are limited to %1$d cells.', maxValue),
                   _.noI18n(maxValue)
            );
        }

        return result;
    };

    /**
     * Returns whether the passed node is a table node containing the splitting
     * class.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is a replacement table for oversized tables.
     */
    DOM.isSplittedTableNode = function (node) {
        return DOM.isTableNode(node) && $(node).hasClass('tb-split-nb');
    };

    /**
     * Returns whether the passed node is a div node with class 'placeholder' inside a table with
     * an exceeded size.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is an div node with class 'placeholder' inside a replacement table
     *  for oversized tables.
     */
    DOM.isPlaceholderNode = function (node) {
        return $(node).is(DOM.PLACEHOLDER_SELECTOR);
    };

    /**
     * Returns whether the passed node is a table node that is located in
     * another table.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table node inside another table.
     */
    DOM.isTableInTableNode = function (node) {
        return DOM.isTableNode(node) && DOM.isCellContentNode($(node).parent());
    };

    /**
     * Returns whether the passed node is a table node that is located in
     * another node (table, text frame, ...) so that it is NOT a top level
     * node inside the page content node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table node located inside another node
     *  that is not the page content.
     */
    DOM.isChildTableNode = function (node) {
        return DOM.isTableNode(node) && !$(node).parent().is(DOM.PAGECONTENT_NODE_SELECTOR);
    };

    /**
     * A jQuery selector that matches elements representing a table row.
     */
    DOM.TABLE_ROWNODE_SELECTOR = 'tr';

    /**
     * Returns whether the passed node is a table row element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table row element.
     */
    DOM.isTableRowNode = function (node) {
        return $(node).is(DOM.TABLE_ROWNODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a table row element inside a first
     * level table element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table row element in a first level table
     *  element.
     */
    DOM.isTopLevelRowNode = function (node) {
        return DOM.isTableRowNode(node) && $(node).parents(DOM.TABLE_ROWNODE_SELECTOR).length < 1;
    };

    /**
     * A jQuery selector that matches elements representing a table page break row.
     */
    DOM.TABLE_PAGEBREAK_ROWNODE_SELECTOR = 'tr.pb-row';

    /**
     * Returns whether the passed node is a table page break row element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table page break row element.
     */
    DOM.isTablePageBreakRowNode = function (node) {
        return $(node).is(DOM.TABLE_PAGEBREAK_ROWNODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a table node containing a page break
     * row element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table node containing a page break row element.
     */
    DOM.isTableWithPageBreakRowNode = function (node) {
        return DOM.isTableNode(node) && ($(node).find(DOM.TABLE_PAGEBREAK_ROWNODE_SELECTOR).length > 0);
    };

    /**
     * Returns the collection of table rows from the passed table element.
     *
     * @param {HTMLTableElement|jQuery} tableNode
     *  The table DOM node. If this object is a jQuery collection, uses the
     *  first DOM node it contains.
     *
     * @returns {jQuery}
     *  A jQuery collection containing all rows of the specified table.
     */
    DOM.getTableRows = function (tableNode) {
        return $(tableNode).find('> tbody > tr').not('.pb-row');
    };

    /**
     * A jQuery selector that matches elements representing a table cell.
     */
    DOM.TABLE_CELLNODE_SELECTOR = 'td';

    /**
     * Returns whether the passed node is a table cell element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table cell element.
     */
    DOM.isTableCellNode = function (node) {
        return $(node).is(DOM.TABLE_CELLNODE_SELECTOR);
    };

    /**
     * Creates a new table cell element.
     *
     * @param {jQuery} paragraph
     *  A paragraph node that is inserted into the cellcontent div of the new
     *  table cell.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.plain=false]
     *      If set to true, no resize nodes will be inserted into the cell, and
     *      other special formatting will be omitted (for example, the
     *      'contenteditable' attributes).
     *
     * @returns {jQuery}
     *  A table cell element, as jQuery object.
     */
    DOM.createTableCellNode = function (paragraph, options) {

        var cellContentNode = $('<div>').addClass('cellcontent'),
            cellChildNode = $('<div>').addClass('cell'),
            cellNode = $('<td>'),
            plain = Utils.getBooleanOption(options, 'plain', false),
            // contenteditable must be false in IE and true for all other browsers
            contentEditable = !_.browser.IE;

        if (!plain) {
            cellChildNode.append($('<div>').addClass('resize bottom'), $('<div>').addClass('resize right'));
            cellChildNode.children().toggleClass('touch', Modernizr.touch);
        }
        cellNode.append(cellChildNode.append(cellContentNode.append(paragraph)));

        if (!plain) {
            // -> setting attribute only for IE.
            //    this leads to a yellow border around div.p in Chrome
            // -> Chrome Bug is fixed with: "outline: none !important;"
            cellChildNode.attr('contenteditable', contentEditable);
            cellContentNode.attr({ contenteditable: true, 'data-focus-role': 'cell' });
        }

        return cellNode;
    };

    /**
     * A jQuery selector that matches elements representing a cell content node.
     */
    DOM.CELLCONTENT_NODE_SELECTOR = 'div.cellcontent';

    /**
     * A jQuery selector that matches elements representing a cell node inside a table cell.
     */
    DOM.CELL_NODE_SELECTOR = 'div.cell';

    /**
     * Returns whether the passed node is a table cell content element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table cell content element.
     */
    DOM.isCellContentNode = function (node) {
        return $(node).is(DOM.CELLCONTENT_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a table cell node element inside a table cell.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table cell node element.
     */
    DOM.isCellNode = function (node) {
        return $(node).is(DOM.CELL_NODE_SELECTOR);
    };

    /**
     * Returns the container node of a table cell that contains all top-level
     * content nodes (paragraphs and tables).
     *
     * @param {HTMLTableCellElement|jQuery} cellNode
     *  The table cell DOM node. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The container DOM node from the passed table cell that contains all
     *  top-level content nodes (paragraphs and tables).
     */
    DOM.getCellContentNode = function (cellNode) {
        return $(cellNode).find('> * > ' + DOM.CELLCONTENT_NODE_SELECTOR);
    };

    /**
     * A jQuery selector that matches elements representing a resize node.
     */
    DOM.RESIZE_NODE_SELECTOR = 'div.resize';

    /**
     * A jQuery selector that matches elements representing an active resize
     * node. This means the tracking is enabled on the node after a mouse down
     * or touchstart event.
     */
    DOM.ACTIVE_RESIZE_NODE_SELECTOR = 'div.resizeActive';

    /**
     * Returns whether the passed node is a table resize element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table resize element.
     */
    DOM.isResizeNode = function (node) {
        return $(node).is(DOM.RESIZE_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is an active table resize element. This
     * means the tracking is enabled on the node after a mouse down or
     * touchstart event.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an active table resize element.
     */
    DOM.isActiveResizeNode = function (node) {
        return (DOM.isResizeNode(node) && $(node).is(DOM.ACTIVE_RESIZE_NODE_SELECTOR));
    };

    /**
     * Returns all cell nodes of the passed table. Does not include cells from
     * tables embedded in the passed table.
     *
     * @param {HTMLTableElement|jQuery} tableNode
     *  The table DOM node. If this object is a jQuery collection, uses the
     *  first DOM node it contains.
     *
     * @returns {jQuery}
     *  A jQuery collection containing all cells of the passed table.
     */
    DOM.getTableCells = function (tableNode) {
        return $(tableNode).find('> tbody > tr > td');
    };

    /**
     * A jQuery selector that matches elements representing a table cell
     * or a table row.
     */
    DOM.TABLE_CELL_ROW_NODE_SELECTOR = 'tr, td';

    /**
     * Returns whether the passed node is a table row or a table cell element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table row or a table cell element.
     */
    DOM.isTableCellOrRowNode = function (node) {
        return $(node).is(DOM.TABLE_CELL_ROW_NODE_SELECTOR);
    };

    /**
     * Check, whether the browser selection containing a focus node and an
     * anchor node describes a cell selection.
     *
     * @param {Node|jQuery|Null} focusNode
     *  The focus node of the browser selection.
     *
     * @param {Node|jQuery|Null} anchorNode
     *  The anchor node of the browser selection.
     *
     * @returns {Boolean}
     *  Whether the browser selection containing a focus node and an
     *  anchor node describes a cell selection.
     */
    DOM.isCellRangeSelected = function (focusNode, anchorNode) {

        var // the row containing the anchor node
            anchorRowNode = null;

        // focus node must be a row
        if (DOM.isTableRowNode(focusNode)) {
            // both nodes are rows in the same table
            if (DOM.isTableRowNode(anchorNode) && (Utils.getDomNode(anchorNode).parentNode === Utils.getDomNode(focusNode).parentNode)) { return true; }

            // anchor node can also be a cell content node (only two neighboring cells selected)
            if (DOM.isCellContentNode(anchorNode)) {
                anchorRowNode = $(anchorNode).closest(DOM.TABLE_ROWNODE_SELECTOR);
                if (anchorRowNode.length > 0 && Utils.getDomNode(anchorRowNode).parentNode === Utils.getDomNode(focusNode).parentNode) { return true; }
            }
        }

        return false;
    };


    // text spans, text nodes, text components ================================

    // text spans -------------------------------------------------------------

    /**
     * Returns a new empty text span element with a single child text node.
     *
     * @returns {jQuery}
     *  The empty text span element, as jQuery object.
     */
    DOM.createTextSpan = function () {
        var textSpan = $('<span>');
        textSpan[0].appendChild(document.createTextNode(''));
        return textSpan;
    };

    /**
     * This function checks, if a specified node is a text span node. If the text node is missing
     * it adds an empty text node into the span. This is necessary, because jQuerys text-function
     * creates no text node, if it is given an empty string.
     *
     * @param {Node|jQuery|Null} node
     *  The text span element, as jQuery object or html node.
     */
    DOM.ensureExistingTextNode = function (node) {
        var domNode = node && Utils.getDomNode(node);
        if (DOM.isTextSpanWithoutTextNode(domNode)) { domNode.appendChild(document.createTextNode('')); }
    };

    /**
     * Returns whether the passed node is a <span> element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a span element.
     */
    DOM.isSpan = function (node) {
        return ($(node).is('span'));
    };

    /**
     * Returns whether the passed node is a <span> element containing a single
     * text node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a span element with a text node.
     */
    DOM.isTextSpan = function (node) {
        var contents = $(node).contents();
        return ($(node).is('span') && (contents.length === 1) && (contents[0].nodeType === 3));
    };

    /**
     * Returns whether the passed node is a text span element that extends to more
     * than one line in the containing paragraph.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a span element with a text node that extends
     *  to more than one line.
     */
    DOM.isMultiLineTextSpan = function (node) {
        return DOM.isTextSpan(node) && $(node).height() > Utils.getElementCssLength(node, 'line-height');
    };

    /**
     * Returns whether the passed node is a <span> whose parent is a
     * div with class 'hardbreak'
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a hard break span element.
     */
    DOM.isHardBreakSpan = function (node) {
        var domNode = node && Utils.getDomNode(node);
        return $(domNode).is('span') && domNode.parentNode && DOM.isHardBreakNode(domNode.parentNode);
    };

    /**
     * Returns whether the passed node is a <span> element containing an empty
     * text node
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a span element with an empty text node.
     */
    DOM.isEmptySpan = function (node) {
        return (DOM.isTextSpan(node) && ($(node).text().length === 0));
    };

    /**
     * Returns whether the passed node is a <span> element representing a text
     * portion (a child element of a paragraph node).
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a text portion span element.
     */
    DOM.isPortionSpan = function (node) {
        return DOM.isTextSpan(node) && DOM.isParagraphNode(Utils.getDomNode(node).parentNode);
    };

    /**
     * Internet Explorer removes empty text nodes from text spans inside
     * table cells.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a span element without a text node.
     */
    DOM.isTextSpanWithoutTextNode = function (node) {
        var domNode = node && Utils.getDomNode(node);
        return $(domNode).is('span') && ($(domNode).contents().length === 0) && DOM.isParagraphNode(domNode.parentNode);
    };

    /**
     * Returns whether the passed node is a text node embedded in a text
     * portion span (see DOM.isPortionSpan() method).
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a text node in a portion span element.
     */
    DOM.isTextNodeInPortionSpan = function (node) {
        var domNode = node && Utils.getDomNode(node);
        return domNode && (domNode.nodeType === 3) && DOM.isPortionSpan(domNode.parentNode);
    };

    /**
     * Returns the first text portion span of the specified paragraph node.
     *
     * @param {HTMLElement|jQuery} paragraph
     *  The paragraph node whose first text portion span will be returned. If
     *  this object is a jQuery collection, uses the first DOM node it
     *  contains.
     *
     * @returns {HTMLElement}
     *  The first text portion span of the paragraph.
     */
    DOM.findFirstPortionSpan = function (paragraph) {
        return Utils.findDescendantNode(paragraph, function () { return DOM.isTextSpan(this); }, { children: true });
    };

    /**
     * Returns the last text portion span of the specified paragraph node.
     *
     * @param {HTMLElement|jQuery} paragraph
     *  The paragraph node whose last text portion span will be returned. If
     *  this object is a jQuery collection, uses the first DOM node it
     *  contains.
     *
     * @returns {HTMLElement}
     *  The last text portion span of the paragraph.
     */
    DOM.findLastPortionSpan = function (paragraph) {
        return Utils.findDescendantNode(paragraph, function () { return DOM.isTextSpan(this); }, { children: true, reverse: true });
    };

    // change tracking ------------------------------------------

    /**
     * A jQuery selector that matches elements representing a node with
     * a changes attribute.
     */
    DOM.CHANGETRACK_NODE_SELECTOR = '[data-change-track-inserted="true"], [data-change-track-removed="true"], [data-change-track-modified="true"]';

    /**
     * A jQuery selector that matches elements representing a node with
     * the changes attribute 'inserted'.
     */
    DOM.CHANGETRACK_INSERTED_NODE_SELECTOR = '[data-change-track-inserted="true"]';

    /**
     * A jQuery selector that matches elements representing a node with
     * the changes attribute 'removed'.
     */
    DOM.CHANGETRACK_REMOVED_NODE_SELECTOR = '[data-change-track-removed="true"]';

    /**
     * A jQuery selector that matches elements representing a node with
     * the changes attribute 'modified'.
     */
    DOM.CHANGETRACK_MODIFIED_NODE_SELECTOR = '[data-change-track-modified="true"]';

    /**
     * A jQuery selector that matches table rows with the changes attribute 'inserted'.
     */
    DOM.CHANGETRACK_INSERTED_ROW_SELECTOR = 'tr[data-change-track-inserted="true"]';

    /**
     * A jQuery selector that matches table rows with the changes attribute 'removed'.
     */
    DOM.CHANGETRACK_REMOVED_ROW_SELECTOR = 'tr[data-change-track-removed="true"]';

    /**
     * Returns whether the passed node is a node with a changes
     * attribute.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a node with change tracking attribute.
     */
    DOM.isChangeTrackNode = function (node) {
        return $(node).is(DOM.CHANGETRACK_NODE_SELECTOR) || DOM.isChangeTrackInlineNode(node);
    };

    /**
     * Returns whether the passed node is a node with changes 'inserted'
     * attribute.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a node with changes 'inserted'
     *  attribute.
     */
    DOM.isChangeTrackInsertNode = function (node) {
        return $(node).is(DOM.CHANGETRACK_INSERTED_NODE_SELECTOR) || DOM.isChangeTrackInsertInlineNode(node);
    };

    /**
     * Returns whether the passed node is a node with changes 'removed'
     * attribute.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a node  with changes 'removed'
     *  attribute.
     */
    DOM.isChangeTrackRemoveNode = function (node) {
        return $(node).is(DOM.CHANGETRACK_REMOVED_NODE_SELECTOR) || DOM.isChangeTrackRemoveInlineNode(node);
    };

    /**
     * Returns whether the passed node is a node with changes 'modified'
     * attribute.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a node with changes 'modified' attribute.
     */
    DOM.isChangeTrackModifyNode = function (node) {
        return $(node).is(DOM.CHANGETRACK_MODIFIED_NODE_SELECTOR) || DOM.isChangeTrackModifyInlineNode(node);
    };

    /**
     * Returns whether the passed node is an inline node with changes attribute
     * at its span (tab, hard break, field, ...).
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an inline node with changes attribute set to its span.
     */
    DOM.isChangeTrackInlineNode = function (node) {
        return DOM.isInlineComponentNode(node) && $(Utils.getDomNode(node).firstChild).is(DOM.CHANGETRACK_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is an inline node with changes 'inserted'
     * attribute at its span (tab, hard break, field, ...).
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an inline node with changes 'inserted' attribute
     *  set to its span.
     */
    DOM.isChangeTrackInsertInlineNode = function (node) {
        return DOM.isInlineComponentNode(node) && $(Utils.getDomNode(node).firstChild).is(DOM.CHANGETRACK_INSERTED_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is an inline node with changes 'removed'
     * attribute at its span (tab, hard break, field, ...).
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an inline node with changes 'removed' attribute
     *  set to its span.
     */
    DOM.isChangeTrackRemoveInlineNode = function (node) {
        return DOM.isInlineComponentNode(node) && $(Utils.getDomNode(node).firstChild).is(DOM.CHANGETRACK_REMOVED_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is an inline node with changes 'modified'
     * attribute at its span (tab, hard break, field, ...).
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an inline node with changes 'modified' attribute
     *  set to its span.
     */
    DOM.isChangeTrackModifyInlineNode = function (node) {
        return DOM.isInlineComponentNode(node) && $(Utils.getDomNode(node).firstChild).is(DOM.CHANGETRACK_MODIFIED_NODE_SELECTOR);
    };

    // text components: fields, tabs ------------------------------------------

    /**
     * A jQuery selector that matches elements representing an inline component
     * inside a paragraph (text components or inline drawing objects).
     */
    DOM.INLINE_COMPONENT_NODE_SELECTOR = 'div.inline';

    /**
     * Returns whether the passed node is an editable inline component node in
     * a paragraph. Inline components include text components (e.g. fields,
     * tabs) and inline drawing objects.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an inline component node in a paragraph.
     */
    DOM.isInlineComponentNode = function (node) {
        return $(node).is(DOM.INLINE_COMPONENT_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a <div> container with embedded text
     * spans, used as root elements for special text components in a paragraph.
     * Does NOT return true for helper nodes that do not represent editable
     * contents of a paragraph (e.g. numbering labels). To check for all helper
     * nodes that contain text spans (also non-editable elements such as
     * numbering labels), use the method DOM.isTextSpanContainerNode() instead.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a <div> element representing an editable
     *  text component in a paragraph.
     */
    DOM.isTextComponentNode = function (node) {
        // text component nodes are all inline nodes, except drawings.
        // This includes tabs, fields and hard breaks
        return DOM.isInlineComponentNode(node) && !DOM.isDrawingFrame(node);
    };

    /**
     * Returns whether the passed node is a <div> container with embedded text
     * spans, used as root elements for special text components in a paragraph.
     * Does NOT return true for text nodes contained in helper nodes that do
     * not represent editable contents of a paragraph (e.g. numbering labels).
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a DOM text node contained in a <div> element
     *  representing an editable text component in a paragraph.
     */
    DOM.isTextNodeInTextComponent = function (node) {
        node = node ? Utils.getDomNode(node) : null;
        return node && node.parentNode && DOM.isTextComponentNode(node.parentNode.parentNode);
    };

    /**
     * A jQuery selector that matches elements representing a text field.
     */
    DOM.FIELD_NODE_SELECTOR = 'div.field';

    /**
     * A jQuery selector that matches elements representing a text field with type page-count.
     */
    DOM.PAGECOUNT_FIELD_SELECTOR = '.field.field-page-count, .field.field-NUMPAGES';

    /**
     * A jQuery selector that matches elements representing a text field with type page-number.
     */
    DOM.PAGENUMBER_FIELD_SELECTOR = '.field.field-page-number';

    /**
     * Returns whether the passed node is an element representing a text field.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an element representing a text field.
     */
    DOM.isFieldNode = function (node) {
        return $(node).is(DOM.FIELD_NODE_SELECTOR);
    };

    /**
     * Returns a new empty text field element.
     *
     * @returns {jQuery}
     *  A new empty text field element, as jQuery object.
     */
    DOM.createFieldNode = function () {
        return $('<div>', { contenteditable: false }).addClass('inline field');
    };

    /**
     * A jQuery selector that matches elements representing a tab.
     */
    DOM.TAB_NODE_SELECTOR = 'div.tab';

    /**
     * Returns whether the passed node is a tab element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a is a tab element.
     */
    DOM.isTabNode = function (node) {
        return $(node).is(DOM.TAB_NODE_SELECTOR);
    };

    /**
     * Returns a new tab element.
     *
     * @returns {jQuery}
     *  A new tab element, as jQuery object.
     */
    DOM.createTabNode = function () {
        return $('<div>', { contenteditable: false}).addClass('inline tab');
    };

    /**
     * Returns whether the passed node is a <span> whose parent is a
     * div with class 'tab'
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a tab span element.
     */
    DOM.isTabSpan = function (node) {
        var domNode = node && Utils.getDomNode(node);
        return $(domNode).is('span') && domNode.parentNode && DOM.isTabNode(domNode.parentNode);
    };

    /**
     * A jQuery selector that matches elements representing a hard-break.
     */
    DOM.HARDBREAK_NODE_SELECTOR = 'div.hardbreak';

    /**
     * Returns whether the passed node is a hard-break element
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a text node in a text field element.
     */
    DOM.isHardBreakNode = function (node) {
        return $(node).is(DOM.HARDBREAK_NODE_SELECTOR);
    };

    /**
     * Returns a new hard-break element.
     *
     * @param {String} type
     *  The type of the hard-break element. Supported values are
     *  'page', 'column' or 'textWrapping'.
     *
     * @returns {jQuery}
     *  A new hard-break element, as jQuery object.
     */
    DOM.createHardBreakNode = function (type) {
        return $('<div>', { contenteditable: false }).addClass('inline hardbreak').data('type', type);
    };

    // paragraph helper nodes -------------------------------------------------

    /**
     * Returns whether the passed node is a <div> container with embedded text
     * spans, used as root elements for special text elements in a paragraph.
     * Returns also true for helper nodes that do NOT represent editable
     * contents of a paragraph (e.g. numbering labels). To check for container
     * nodes that represent editable components in a paragraph only, use the
     * method DOM.isTextComponentNode() instead.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a <div> element containing text spans.
     */
    DOM.isTextSpanContainerNode = function (node) {
        node = node ? Utils.getDomNode(node) : null;
        return $(node).is('div') && DOM.isParagraphNode(node.parentNode) && DOM.isTextSpan(node.firstChild);
    };

    /**
     * A jQuery selector that matches elements representing a list label.
     */
    DOM.LIST_LABEL_NODE_SELECTOR = 'div.list-label';

    /**
     * Returns whether the passed node is an element representing a list label.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a list label element.
     */
    DOM.isListLabelNode = function (node) {
        return $(node).is(DOM.LIST_LABEL_NODE_SELECTOR);
    };

    /**
     * Creates a new element representing a list label.
     *
     * @param {String} [text]
     *  The text contents of the list label node.
     *
     * @returns {jQuery}
     *  A new list label node, as jQuery object.
     */
    DOM.createListLabelNode = function (text) {
        var node = $('<div>').addClass('helper list-label').append(text ? DOM.createTextSpan().text(text) : DOM.createTextSpan());

        // Bug 27902: prevent placing text cursor into list label node
        // Bug 26568: IE9 and IE10 show a drawing selection frame when 'contenteditable' is set to false
        // Bug 29546 (still open): this leads to a wrong mouse pointer (move) for the list labels
        if (!_.browser.IE || (_.browser.IE >= 11)) { node.attr('contenteditable', false); }
        return node;
    };

    /**
     * Returns whether the passed node is a dummy text node that is used in
     * empty paragraphs to preserve an initial element height according to the
     * current font size.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a dummy text node.
     */
    DOM.isDummyTextNode = function (node) {
        return $(node).data('dummy') === true;
    };

    /**
     * Creates a dummy text node that is used in empty paragraphs to preserve
     * an initial element height according to the current font size.
     *
     * @returns {jQuery}
     *  A dummy text node, as jQuery object.
     */
    DOM.createDummyTextNode = function () {
        // TODO: create correct element for current browser
        return $('<br>').data('dummy', true);
    };

    // drawing nodes ----------------------------------------------------------

    /**
     * Returns whether the passed node is the root node of a drawing frame.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is the root node of a drawing frame.
     */
    DOM.isDrawingFrame = DrawingFrame.isDrawingFrame;

    /**
     * A jQuery selector that matches nodes representing a text frame content
     * element inside a drawing node. This is reassigned here for convenience
     * reasons to reduce required imports.
     *
     * @constant
     */
    DOM.TEXTFRAMECONTENT_NODE_SELECTOR = DrawingFrame.TEXTFRAMECONTENT_NODE_SELECTOR;

    /**
     * A jQuery selector that matches elements representing a floating node
     * inside a paragraph (editable component nodes and helper nodes).
     */
    DOM.FLOATING_NODE_SELECTOR = 'div.float';

    /**
     * Returns whether the passed node is a floating node in a paragraph.
     * Floating nodes include floating drawing objects, and their associated
     * helper nodes.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a floating node in a paragraph.
     */
    DOM.isFloatingNode = function (node) {
        return $(node).is(DOM.FLOATING_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a <div> element wrapping a drawing
     * in inline mode.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a div element wrapping a drawing and is
     *  rendered inlined.
     */
    DOM.isInlineDrawingNode = function (node) {
        return DrawingFrame.isDrawingFrame(node) && DOM.isInlineComponentNode(node);
    };

    /**
     * Returns whether the passed node is a <div> element wrapping a drawing
     * in floating mode.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a div element wrapping a drawing and is
     *  rendered floated.
     */
    DOM.isFloatingDrawingNode = function (node) {
        return DrawingFrame.isDrawingFrame(node) && DOM.isFloatingNode(node);
    };

    /**
     * Returns whether the passed node (typically a paragraph) contains a floated
     * drawing node as direct child.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node contains a floated drawing node as direct child.
     */
    DOM.containsFloatingDrawingNode = function (node) {
        return $(node).children(DOM.FLOATING_NODE_SELECTOR).length > 0;
    };

    /**
     * Returns whether the passed node is a <div> element wrapping a drawing
     * of type 'image'.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a div element wrapping a drawing of type
     *  'image'.
     */
    DOM.isImageDrawingNode = function (node) {
        return $(node).is(DrawingFrame.NODE_SELECTOR) && ($(node).data('type') === 'image');
    };

    /**
     * Returns whether the passed node is a <div> element wrapping a drawing
     * that is not of type 'image'.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a div element wrapping a drawing that is not
     *  of type 'image'.
     */
    DOM.isNonImageDrawingNode = function (node) {
        return $(node).is(DrawingFrame.NODE_SELECTOR) && ($(node).data('type') !== 'image');
    };

    /**
     * Returns whether the passed node is a drawing node that cannot be restored
     * after deletion.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing node that cannot be restored after
     *  deletion.
     */
    DOM.isUnrestorableDrawingNode = function (node) {

        if (!DrawingFrame.isDrawingFrame(node)) { return false; }

        // unrestorable are all drawing nodes, that are not images, not text frames. Groups are restorable,
        // if they contain only images or text frames.
        if (DrawingFrame.isGroupDrawingFrame(node)) {
            // checking the children of the group
            return _.find($(node).find(DrawingFrame.NODE_SELECTOR), function (drawingNode) {
                return DOM.isNonImageDrawingNode(drawingNode) && !DrawingFrame.isTextFrameShapeDrawingFrame(drawingNode) && !DrawingFrame.isGroupDrawingFrame(drawingNode);
            });
        } else {
            return DOM.isNonImageDrawingNode(node) && !DrawingFrame.isTextFrameShapeDrawingFrame(node);
        }
    };

    /**
     * Returns whether the passed node is inside a drawing text frame node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is inside a text frame drawing node.
     */
    DOM.isNodeInsideTextFrame = function (node) {
        return node && $(node).length > 0 && $(node).closest(DrawingFrame.TEXTFRAME_NODE_SELECTOR).length > 0;
    };

    /**
     * Returns whether the passed node is inside an auto resizing drawing text frame node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is inside an auto resizing text frame drawing node.
     */
    DOM.isNodeInsideAutoResizeHightTextFrame = function (node) {
        return node && $(node).length > 0 && $(node).closest(DrawingFrame.AUTORESIZEHEIGHT_SELECTOR).length > 0;
    };

    /**
     * Returns whether the passed node is the first direct child inside a
     * text frame. This can be a paragraph or a table.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is the first child inside a text frame.
     */
    DOM.isFirstContentNodeInTextframe = function (node) {

        var // the dom node of the specified node
            domNode = Utils.getDomNode(node);

        return domNode && DOM.isContentNode(domNode) && !domNode.previousSibling && DrawingFrame.isTextFrameNode(domNode.parentNode);
    };

    /**
     * Returns whether the passed node is the first text span inside a text
     * frame.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is the first text span inside a text frame.
     */
    DOM.isFirstTextSpanInTextframe = function (node) {

        var // a text frame containing the specified node
            textFrame = null,
            // the first paragraph in the text frame
            firstParagraph = null;

        // Check: node is text span and node is inside text frame and node is the first text span in the text frame
        if (DOM.isTextSpan(node)) {
            textFrame = DrawingFrame.getClosestTextFrameDrawingNode(node);

            if (textFrame) {
                firstParagraph = textFrame.find(DOM.PARAGRAPH_NODE_SELECTOR_COMPLETE).first();
                return DOM.findFirstPortionSpan(firstParagraph) === Utils.getDomNode(node);
            }
        }

        return false;
    };

    /**
     * Returns whether the passed node is a <div> element wrapping an image.
     * Simple 'find' for 'img' does not work, because drawing groups can contain several
     * image drawings and images, but are no image drawings.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a div element wrapping an image.
     */
    DOM.isImageNode = function (node) {
        // drawing div contains another div (class content) that contains an image
        return DrawingFrame.isDrawingFrame(node) && ($(node).data('type') === 'image');
    };

    /**
     * Returns true if the source of the passed image node points to a server document.
     * The node can be a <div> wrapping an image or an <img> itself.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed image node points to a server document.
     */
    DOM.isDocumentImageNode = function (node) {

        var img = $(node).is('img') ? $(node) : $(node).find('img'),
            src = img.attr('src');

        if (!src) { return false; }  // replacement data is shown

        // check for base64 data-URL first
        if (src.substring(0, 10) === 'data:image') {
            return false;
        }

        // check for URL parameters
        return _.every(['session', 'uid', 'folder_id', 'action', 'get_filename'], function (param) {
            return src.indexOf(param) !== -1;
        });
    };

    /**
     * Returns true if the source of the passed image node contains a base64 data-URL.
     * The node can be a <div> wrapping an image or an <img> itself.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed image node contains a base64 data-URL.
     */
    DOM.isBase64ImageNode = function (node) {

        var img = $(node).is('img') ? $(node) : $(node).find('img'),
            src = img.attr('src');

        return src.substring(0, 10) === 'data:image';
    };

    /**
     * Returns the value of the given URL param of the image src attribute
     *
     * @param {Node|jQuery} node
     *  The image node to get the URL param from.
     *  If this object is a jQuery collection, uses the first DOM node it contains.
     *
     * @param param
     *  The URL param
     *
     * @returns
     *  The value of the given URL param or null
     */
    DOM.getUrlParamFromImageNode = function (node, param) {

        var img = $(node).is('img') ? $(node) : $(node).find('img'),
            src = img.attr('src'),
            reg,
            match = null;

        if (src && param && param.length) {
            reg = new RegExp('.*[?&]' + param + '=([^&]+)(&|$)');
            match = src.match(reg);
            return (_.isArray(match) ? match[1] : null);
        }

        return null;
    };

    /**
     * Returns the base64 encoded data-URL of the given image node
     *
     * @param {Node|jQuery} node
     *  The image node to get the base64 data-URL from.
     *  If this object is a jQuery collection, uses the first DOM node it contains.
     *
     *  @param {String} [mimeType='image/png']
     *   The mime type of the base64 encoded data-URL.
     *   If not given or not supported by the browser 'image/png' is used.
     *
     *  @returns {String}
     *  The base64 encoded data-URL
     */
    DOM.getBase64FromImageNode = function (node, mimeType) {

        var // the image DOM node
            img = Utils.getDomNode($(node).is('img') ? $(node) : $(node).find('img')),
            // the mime type
            mime = (_.isString(mimeType) && mimeType.length) ? mimeType : 'image/png',
            // create an empty canvas element
            canvas = document.createElement('canvas'),
            // the drawing context
            ctx,
            // the base64 data URL
            base64;


        try {
            // set canvas to natural image size
            canvas.width = img.naturalWidth || img.width;
            canvas.height = img.naturalHeight || img.height;

            // copy the image contents to the canvas
            ctx = canvas.getContext('2d');
            ctx.drawImage(img, 0, 0);

            // return the data-URL formatted image
            base64 = canvas.toDataURL(mime);

        } catch (ex) {
            Utils.warn('DOM.getBase64FromImageNode(): ' + ((ex && ex.message) ? ex.message : 'failed'));
            base64 = 'data:,';
        }

        return base64;
    };

    /**
     * A jQuery selector that matches elements representing a drawing offset
     * helper node.
     */
    DOM.OFFSET_NODE_SELECTOR = 'div.float.offset';

    /**
     * Returns whether the passed node is a <div> element for positioning
     * a drawing with a vertical or horizontal offset.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a div element wrapping a drawing.
     */
    DOM.isOffsetNode = function (node) {
        return $(node).is(DOM.OFFSET_NODE_SELECTOR);
    };

    /**
     * The CSS class used to mark drawing placeholder nodes. This place holder
     * nodes are located in the page content node and safe the position for
     * an absolutely positioned drawing in the drawing layer.
     *
     * @constant
     */
    DOM.DRAWINGPLACEHOLDER_CLASS = 'drawingplaceholder';

    /**
     * The data property name for registering the drawing place holder at the absolutely
     * positioned drawing. This works in both directions, because the drawing place
     * holder has a data attribute with this name to the absolutely positioned drawing
     * and vice versa.
     *
     * @constant
     */
    DOM.DRAWINGPLACEHOLDER_LINK = 'drawingplaceholderlink';

    /**
     * A jQuery selector that matches elements representing an image place holder.
     */
    DOM.DRAWINGPLACEHOLDER_NODE_SELECTOR = 'div.' + DOM.DRAWINGPLACEHOLDER_CLASS;

    /**
     * Returns whether the passed node is an element representing an image
     * place holder node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an element representing an image placeholder
     *  node.
     */
    DOM.isDrawingPlaceHolderNode = function (node) {
        return $(node).is(DOM.DRAWINGPLACEHOLDER_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node (a paragraph) has at least one child
     * image place holder node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node has at least one child representing an image
     *  place holder node.
     */
    DOM.hasDrawingPlaceHolderNode = function (node) {
        return $(node).children(DOM.DRAWINGPLACEHOLDER_NODE_SELECTOR).length > 0;
    };

    /**
     * Searching the place holder node for a specific drawing node. This place
     * holder node exists in the page content for all drawings inside the
     * drawing layer node. This also works vice versa: Searching the drawing
     * node in the drawing layer for a specified place holder. In both cases
     * the 'data' attribute is named DOM.DRAWINGPLACEHOLDER_LINK.
     *
     * @param {Node|jQuery|Null} [node]
     *  The drawing node.
     *
     * @returns {Node}
     *  The place holder node for the drawing node in the drawing layer node
     *  or the drawing in the drawing layer node itself.
     */
    DOM.getDrawingPlaceHolderNode = function (node) {
        return $(node).data(DOM.DRAWINGPLACEHOLDER_LINK);
    };

    /**
     * The CSS class used to mark drawing space maker nodes.
     *
     * @constant
     */
    DOM.DRAWING_SPACEMAKER_CLASS = 'drawingspacemaker';

    /**
     * The data property name for registering the drawing space maker at the absolutely
     * positioned drawing.
     *
     * @constant
     */
    DOM.DRAWING_SPACEMAKER_LINK = 'drawingspacemakerlink';

    /**
     * A jQuery selector that matches elements representing a drawing space maker node.
     */
    DOM.DRAWING_SPACEMAKER_NODE_SELECTOR = 'div.' + DOM.DRAWING_SPACEMAKER_CLASS;

    /**
     * Returns whether the passed node is an element representing a drawing space
     * maker node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an element representing a drawing space maker
     *  node.
     */
    DOM.isDrawingSpaceMakerNode = function (node) {
        return $(node).is(DOM.DRAWING_SPACEMAKER_NODE_SELECTOR);
    };

    /**
     * Searching the space maker node for a specific drawing node. This space
     * maker node exists in the page content for all drawings inside the
     * drawing layer node, that require space 'below' the drawing.
     *
     * @param {Node|jQuery|Null} [node]
     *  The drawing node.
     *
     * @returns {Node}
     *  The space maker node for the drawing node in the drawing layer node.
     */
    DOM.getDrawingSpaceMakerNode = function (node) {
        return $(node).data(DOM.DRAWING_SPACEMAKER_LINK);
    };

    /**
     * Returns whether the passed node (a paragraph) has at least one child
     * representing a drawing space maker node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node has at least one child representing a drawing
     *  space maker node.
     */
    DOM.hasDrawingSpaceMakerNode = function (node) {
        return $(node).children(DOM.DRAWING_SPACEMAKER_NODE_SELECTOR).length > 0;
    };

    /**
     * The CSS class used to mark the drawing layer node.
     *
     * @constant
     */
    DOM.DRAWINGLAYER_CLASS = 'textdrawinglayer';

    /**
     * A jQuery selector that matches the drawing layer node.
     */
    DOM.DRAWINGLAYER_NODE_SELECTOR = 'div.' + DOM.DRAWINGLAYER_CLASS;

    /**
     * Returns whether the passed node is the drawing layer node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is the drawing layer node.
     */
    DOM.isDrawingLayerNode = function (node) {
        return $(node).is(DOM.DRAWINGLAYER_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is inside the drawing layer node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is located inside the drawing layer node.
     */
    DOM.isInsideDrawingLayerNode = function (node) {
        return $(node).parents(DOM.DRAWINGLAYER_NODE_SELECTOR).length > 0;
    };

    /**
     * Returns the drawing node that is directly located below the drawing
     * layer. The specified node needs to be located inside this drawing.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Returns the top level drawing inside the drawing layer.
     */
    DOM.getTopLevelDrawingInDrawingLayerNode = function (node) {
        return $(node).parentsUntil(DOM.DRAWINGLAYER_NODE_SELECTOR, DrawingFrame.NODE_SELECTOR).last();
    };

    // manipulate and iterate text spans --------------------------------------

    /**
     * Splits the passed text span element into two text span elements. Clones
     * all formatting to the new span element.
     *
     * @param {HTMLSpanElement|jQuery} span
     *  The text span to be split. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @param {Number} offset
     *  The character position where the text span will be split. If this
     *  position is at the start or end of the text in the span, an empty text
     *  span will be inserted.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.append=false]
     *      If set to true, the right part of the text will be inserted after
     *      the passed text span; otherwise the left part of the text will be
     *      inserted before the passed text span. The position of the new text
     *      span may be important when iterating and manipulating a range of
     *      DOM nodes.
     *
     * @returns {jQuery}
     *  The newly created text span element, as jQuery object. Will be located
     *  before or after the passed text span, depending on the 'options.append'
     *  option.
     */
    DOM.splitTextSpan = function (span, offset, options) {

        var // the new span for the split text portion, as jQuery object
            newSpan = $(span).clone(true),
            // the existing text node (must not be invalidated, e.g. by using jQuery.text())
            textNode = Utils.getDomNode(span).firstChild,
            // text for the left span
            leftText = textNode.nodeValue.substr(0, offset),
            // text for the right span
            rightText = textNode.nodeValue.substr(offset);

        // insert the span and update the text nodes
        if (Utils.getBooleanOption(options, 'append', false)) {
            newSpan.insertAfter(span);
            textNode.nodeValue = leftText;
            newSpan.text(rightText);
            if (! rightText) { DOM.ensureExistingTextNode(newSpan); }
        } else {
            newSpan.insertBefore(span);
            newSpan.text(leftText);
            if (! leftText) { DOM.ensureExistingTextNode(newSpan); }
            textNode.nodeValue = rightText;
        }

        // return the new text span
        return newSpan;
    };

    /**
     * Calls the passed iterator function for all descendant text span elements
     * in a the passed node. As a special case, if the passed node is a text
     * span by itself, it will be visited directly. Text spans can be direct
     * children of a paragraph node (regular editable text portions), or
     * children of other nodes such as text fields or list label nodes.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM node whose descendant text spans will be visited (or which will
     *  be visited by itself if it is a text span). If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {Function} iterator
     *  The iterator function that will be called for every text span. Receives
     *  the DOM span element as first parameter. If the iterator returns the
     *  Utils.BREAK object, the iteration process will be stopped immediately.
     *
     * @param {Object} [context]
     *  If specified, the iterator will be called with this context (the symbol
     *  'this' will be bound to the context inside the iterator function).
     *
     * @returns {Utils.BREAK|Undefined}
     *  A reference to the Utils.BREAK object, if the iterator has returned
     *  Utils.BREAK to stop the iteration process, otherwise undefined.
     */
    DOM.iterateTextSpans = function (node, iterator, context) {

        // visit passed text span directly
        if (DOM.isTextSpan(node)) {
            return iterator.call(context, Utils.getDomNode(node));
        }

        // do not iterate into drawings, they may contain their own paragraphs
        // Bug 30794: also omit list-label nodes as they are especially handled
        // in updateList function in the editor.
        if (!DrawingFrame.isDrawingFrame(node) && !DOM.isListLabelNode(node)) {
            return Utils.iterateSelectedDescendantNodes(node, function () { return DOM.isTextSpan(this); }, iterator, context);
        }
    };

    // generic container and content nodes ====================================

    /**
     * A jQuery selector that matches elements representing a top-level content
     * node (e.g. paragraphs or tables).
     */
    DOM.CONTENT_NODE_SELECTOR = DOM.PARAGRAPH_NODE_SELECTOR + ', ' + DOM.TABLE_NODE_SELECTOR;

    /**
     * Returns whether the passed node is a top-level content node (e.g.
     * paragraphs or tables).
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a top-level content node.
     */
    DOM.isContentNode = function (node) {
        return $(node).is(DOM.CONTENT_NODE_SELECTOR);
    };

    /**
     * Returns the correct node of a container node that contains its content
     * nodes as direct children. Container nodes include the document page,
     * table cells, or drawing objects containing text.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The element that contains the child components of the passed nodes as
     *  direct children, as jQuery object.
     */
    DOM.getChildContainerNode = function (node) {

        // convert to a jQuery object
        node = $(node);

        // page nodes
        if (DOM.isPageNode(node)) {
            return DOM.getPageContentNode(node);
        }

        // table cell nodes
        if (node.is('td')) {
            return DOM.getCellContentNode(node);
        }

        // drawing nodes
        if (DrawingFrame.isDrawingFrame(node)) {

            if (DrawingFrame.isTextFrameShapeDrawingFrame(node)) {
                return DrawingFrame.getTextFrameNode(node);
            } else {
                return DrawingFrame.getContentNode(node);
            }

        }

        return $();
    };

    /**
     * Finding the allowed neighboring node taking into account all nodes
     * that are used for calculating the logical positions. For example as
     * top level nodes only paragraphs and tables are allowed, page-break
     * nodes need to be skipped.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @param {Object} options
     *  A map with additional options controlling the search of the neighbor.
     *  The following options are supported:
     *  @param {Boolean} [options.next=true]
     *      The search direction. If not specified, the next neighboring node
     *      will be searched.
     *
     * @returns {Node|null}
     *  The search element of the specified node. If it cannot be determined
     *  null is returned.
     */
    DOM.getAllowedNeighboringNode = function (node, options) {

        var // determining the direction for searching the neighbor
            next = Utils.getBooleanOption(options, 'next', true),
            // the dom node of the specified node
            domNode = Utils.getDomNode(node),
            // the iterator function
            iterator = next ? 'nextSibling' : 'previousSibling',
            // the neighboring node
            neighbor = domNode[iterator] || null;

        while (neighbor && DOM.isInvalidNode(neighbor)) {
            neighbor = neighbor[iterator];
        }

        return neighbor;
    };

    /**
     * Checking, whether a specified node is part of the calculation of
     * logical positions.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether the specified node is part of the calculation of
     *  logical positions.
     */
    DOM.isInvalidNode = function (node) {
        return DOM.isPageBreakNode(node) || DOM.isListLabelNode(node);  // TODO: List needs to be expanded
    };

    /**
     * Creates a white list of allowed DOM nodes (paragraphs and tables currently),
     * and ignores pagebreaks and clearfloating divs
     *
     * @param {HTMLElement|jQuery} childNodes
     *  List of nodes that we want to filter, to contain only allowed ones, such as top level nodes paragraphs and tables
     *
     * @returns {HTMLElement|jQuery} filteredNodes - list of nodes after filtering
     */
    DOM.allowedChildNodes = function (childNodes) {
        return $(childNodes).filter(DOM.PARAGRAPH_NODE_SELECTOR + ',' + DOM.TABLE_NODE_SELECTOR);
    };


    /**
     * jQuery selector for skipping drawing and page break nodes.
     *
     */
    DOM.SKIP_DRAWINGS_AND_P_BREAKS_SELECTOR = '.drawing, .page-break';

    // page break nodes ========================================================

    /**
     * jQuery selector that matches page break div, used to separate pages in page layout.
     */
    DOM.PAGE_BREAK_SELECTOR = '.page-break';

    /**
     * CSS class name that matches page break div, used to separate pages in page layout.
     */
    DOM.PAGE_BREAK_CLASS_NAME = 'page-break';

    /**
     * jQuery selector that matches manualy inserted page break by user
     */
    DOM.MANUAL_PAGE_BREAK_SELECTOR = '.manual-page-break';

    /**
     * CSS class name that matches manualy inserted page break by user
     */
    DOM.MANUAL_PAGE_BREAK_CLASS_NAME = 'manual-page-break';

    /**
     * jQuery selector that matches inner line in page break that marks begining of new page
     */
    DOM.PAGE_BREAK_LINE_SELECTOR = '.inner-pb-line';

    /**
     * CSS class name that matches line inside page break, that marks begining of new page
     */
    DOM.PAGE_BREAK_LINE_CLASS_NAME = 'inner-pb-line';

    /**
     * jQuery selector name that matches hardbreak with type page, from Microsoft only
     */
    DOM.MS_HARDBREAK_TYPE_PAGE_SELECTOR = '.ms-hardbreak-page';

    /**
     * CSS class name that matches hardbreak with type page, from Microsoft only
     */
    DOM.MS_HARDBREAK_TYPE_PAGE_CLASS_NAME = 'ms-hardbreak-page';

    /**
     * Returns whether the passed node is a page break element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a page break element.
     */
    DOM.isPageBreakNode = function (node) {
        return $(node).hasClass(DOM.PAGE_BREAK_CLASS_NAME);
    };

    /**
     * Returns whether the passed node is a page break element inserted by user.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a manual page break element.
     */
    DOM.isManualPageBreakNode = function (node) {
        return $(node).hasClass(DOM.MANUAL_PAGE_BREAK_CLASS_NAME);
    };

    /**
     * Returns whether the passed node is a hardbreak element with type page,
     * inserted in MS Word.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a MS Word page hardbreak element.
     */
    DOM.isMSPageHardbreakNode = function (node) {
        return $(node).hasClass(DOM.MS_HARDBREAK_TYPE_PAGE_CLASS_NAME);
    };

    /**
     * Returns whether the passed node contains a hardbreak element with type page,
     * inserted in MS Word.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node contains at least one MS Word page hardbreak element.
     */
    DOM.hasMSPageHardbreakNode = function (node) {
        return $(node).find(DOM.MS_HARDBREAK_TYPE_PAGE_SELECTOR).length > 0;
    };

    /**
     * Returns whether the passed node is a table element that contains a page break
     * in its first paragraph in the upper left cell.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table node with manual page break element.
     */
    DOM.isTableWithManualPageBreak = function (node) {
        return DOM.isTableNode(node) && $(node).find('.p').first().is(DOM.MANUAL_PAGE_BREAK_SELECTOR);
    };

    /**
     * Returns whether node contains manual page break after element attribute,
     * set as a class name.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table node with manual page break after element.
     * */
    DOM.isNodeWithPageBreakAfterAttribute = function (node) {
        return $(node).hasClass('manual-pb-after');
    };

    // header/footer ==========================================================

    DOM.HEADER_FOOTER_PLACEHOLDER_SELECTOR = '.header-footer-placeholder';

    DOM.HEADER_FOOTER_PLACEHOLDER_CLASSNAME = 'header-footer-placeholder';

    DOM.HEADER_WRAPPER_SELECTOR = '.header-wrapper';

    DOM.HEADER_WRAPPER_CLASSNAME = 'header-wrapper';

    DOM.FOOTER_WRAPPER_SELECTOR = '.footer-wrapper';

    DOM.FOOTER_WRAPPER_CLASSNAME = 'footer-wrapper';

    /**
     * The class name used to specify elements (drawings) positioned in the
     * headers or footers.
     */
    DOM.TARGET_NODE_CLASSNAME = 'has_target';

    /**
     * Checks if passed node is header or footer element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a header or footer element.
     */
    DOM.isHeaderOrFooter = function (node) {
        return $(node).hasClass('header') || $(node).hasClass('footer');
    };

    /**
     * Checks if passed node is positioned relative.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is 'relative' positioned.
     */
    DOM.isRelativePositionedNode = function (node) {
        return $(node).css('position') === 'relative';
    };

    /**
     * Returning the header or footer node that is wrapping a specified node.
     *
     * @param {Node|jQuery} rootNode
     *  The root node until which the search of header or footer is executed.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     *  Optional parameters:
     *  @param {Boolean} [options.footer=false]
     *    If set to true, the specified node is searched in the footer.
     *  @param {Boolean} [options.header=false]
     *    If set to true, the specified node is searched in the header.
     *
     * @returns {jQuery}
     *  A jQuery element, that contains the header or footer node or no node
     *  if the specified node is not inside a header or footer.
     */
    DOM.getMarginalTargetNode = function (rootNode, node, options) {

        var // finding the current node inside a footer
            findFooter = Utils.getBooleanOption(options, 'footer', false),
            // finding the current node inside a footer
            findHeader = Utils.getBooleanOption(options, 'header', false),
            // the selector to find the marginal node
            marginSelector = findFooter ? 'div.footer' : (findHeader ? 'div.header' : 'div.footer, div.header');

        return $(node).parentsUntil(rootNode, marginSelector);
    };

    /**
     * Returning whether a specified node is inside the header/footer
     * template folder (div.header-footer-placeholder).
     *
     * @param {Node|jQuery} rootNode
     *  The root node until which the search of header or footer is executed.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {jQuery}
     *  Whether the specified node is located inside header/footer
     *  template folder.
     */
    DOM.isInsideHeaderFooterTemplateNode = function (rootNode, node) {
        return $(node).parentsUntil(rootNode, DOM.HEADER_FOOTER_PLACEHOLDER_SELECTOR).length > 0;
    };

    // exports ================================================================

    return DOM;

});
