/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Malte Timmermann <malte.timmermann@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/text/app/application',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/errorcode',
     'io.ox/office/editframework/app/editapplication',
     'io.ox/office/text/editor',
     'io.ox/office/text/view/view',
     'io.ox/office/text/app/controller',
     'gettext!io.ox/office/text'
    ], function (Utils, ErrorCode, EditApplication, TextModel, TextView, TextController, gt) {

    'use strict';

    // class TextApplication ==================================================

    /**
     * The OX Text application.
     *
     * @constructor
     *
     * @extends EditApplication
     *
     * @param {Object} launchOptions
     *  All options passed to the core launcher (the ox.launch() method).
     *
     * @param {Object} [appOptions]
     *  Static application options that have been passed to the static method
     *  BaseApplication.createLauncher().
     */
    var TextApplication = EditApplication.extend({ constructor: function (launchOptions, appOptions) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        EditApplication.call(this, TextModel, TextView, TextController, launchOptions, appOptions, {
            applyActionsDetached: true,
            postProcessHandler: postProcessDocument,
            postProcessHandlerStorage: postProcessHandlerStorage,
            fastEmptyLoadHandler: fastEmptyLoadHandler,
            optimizeOperationsHandler: optimizeOperations,
            operationFilter: operationFilter,
            importFailedHandler: importFailedHandler,
            sendActionsDelay: 1000,
            localStorageApp: true,
            requiredStorageVersion: 2,
            supportedStorageVersion: 2,
            supportsOnlineSync: true
        });

        // private methods ----------------------------------------------------

        /**
         * Post-processes the document contents and formatting, after all its
         * import operations have been applied.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  document has been post-processed successfully, or rejected when an
         *  error has occurred.
         */
        function postProcessDocument() {
            return self.getModel().updateDocumentFormatting();
        }

        /**
         * Load performance: Post-processes the document contents and formatting,
         * after the document was loaded from local storage.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  document has been post-processed successfully, or rejected when an
         *  error has occurred.
         */
        function postProcessHandlerStorage() {
            return self.getModel().updateDocumentFormattingStorage();
        }

        /**
         * Handler will be called by base class if importing the document
         * failed. The functions handles Text specific errors.
         */
        function importFailedHandler(response) {

            var // specific error code sent by the server
                error = new ErrorCode(response);

            switch (error.getCodeAsConstant()) {
            case 'LOADDOCUMENT_COMPLEXITY_TOO_HIGH_ERROR':
                response.message = gt('This document exceeds the complexity limits.');
                break;
            }
        }

        /**
         * Optimizing the actions (operations) before sending them to the server.
         *
         * @param {Object[]} actionsBuffer
         *  An array with actions.
         *
         * @returns {Object[]}
         *  The optimized array with actions.
         */
        function optimizeOperations(actionsBuffer) {
            return self.getModel().optimizeActions(actionsBuffer);
        }

        /**
         * Load performance: Only selected operations need to be executed, if document
         * can be loaded from local storage.
         *
         * @param {Object[]} operation
         *  An operation object.
         *
         * @returns {Boolean}
         *  Whether the specified operation needs to be executed, although the document
         *  is loaded from the local storage.
         */
        function operationFilter(operation) {
            return self.getModel().operationFilter(operation);
        }

        /**
         * Handler for fast load of empty documents
         *
         * @param {String} markup
         *  The HTML mark-up to be shown as initial document contents.
         *
         * @param {Array} actions
         *  The operation actions to be applied to finalize the fast import.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  actions have been applied.
         */
        function fastEmptyLoadHandler(markup, actions) {
            return self.getModel().fastEmptyLoadHandler(markup, actions);
        }

        // initialization -----------------------------------------------------

        // destroy all class members
        this.registerDestructor(function () {
            launchOptions = appOptions = null;
        });

    }}); // class TextApplication

    // static methods ---------------------------------------------------------

    /**
     * Replacement for the generic method EditApplication.createLauncher()
     * without parameters, to launch text applications.
     */
    TextApplication.createLauncher = function () {
        return EditApplication.createLauncher('io.ox/office/text', TextApplication, { icon: 'fa-align-left' });
    };

    // exports ================================================================

    return TextApplication;

});
