/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/render/drawingrendermixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (Utils, Forms, DrawingFrame, SheetUtils) {

    'use strict';

    // mix-in class DrawingRenderMixin ========================================

    /**
     * Mix-in class for the class GridPane which renders the drawing objects of
     * the active sheet into the DOM drawing layer of a single grid pane.
     *
     * @constructor
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this renderer.
     *
     * @param {jQuery} drawingLayerNode
     *  The drawing layer DOM node of the grid pane extended with this mix-in.
     */
    function DrawingRenderMixin(app, drawingLayerNode) {

        var // self reference (the GridPane instance)
            self = this,

            // the spreadsheet model and view
            docModel = null,
            docView = null,

            // the cell range covered by the layer nodes in the sheet area
            layerRange = null,

            // all existing drawing frames, mapped by UID (faster access than DOM lookups)
            drawingFrames = {};

        // private methods ----------------------------------------------------

        /**
         * Returns the DOM drawing frame node that represents the passed
         * drawing model.
         *
         * @param {DrawingModel} drawingModel
         *  The drawing model whose DOM drawing frame will be returned.
         *
         * @returns {jQuery}
         *  The DOM node of the drawing frame, as jQuery object; or an empty
         *  jQuery collection, if the drawing frame does not exist.
         */
        function getDrawingFrame(drawingModel) {
            return drawingFrames[drawingModel.getUid()] || $();
        }

        /**
         * Returns all selected DOM drawing frame nodes.
         *
         * @returns {jQuery}
         *  The selected drawing frames, as jQuery object.
         */
        function getSelectedDrawingFrames() {
            return drawingLayerNode.find(DrawingFrame.NODE_SELECTOR + Forms.SELECTED_SELECTOR);
        }

        /**
         * Updates the position of the passed drawing frame according to its
         * anchor attributes.
         */
        function updateDrawingFramePosition(drawingModel, drawingFrame) {

            var // the effective drawing rectangle (null if the drawing is hidden)
                rectangle = drawingModel.getRectangle();

            if (rectangle) {
                // show the drawing frame, and set its position and size
                drawingFrame.show().css(self.convertToLayerRectangle(rectangle));
            } else {
                // hide the drawing frame if it is located in hidden columns or rows
                drawingFrame.hide();
            }
        }

        /**
         * Creates and inserts a new DOM drawing frame node that represents the
         * passed drawing model.
         */
        function createDrawingFrame(drawingModel, position) {

            var // the new drawing frame, as jQuery object
                drawingFrame = DrawingFrame.createDrawingFrame(drawingModel),
                // the merged drawing attributes
                attributes = drawingModel.getMergedAttributes(),
                // all existing drawing frames, as plain DOM collection
                drawingNodes = drawingLayerNode[0].childNodes,
                // the DOM child index (Z order) specified by the position
                index = position[0];

            // store the drawing frame in a local map (faster access than DOM lookup)
            drawingFrames[drawingModel.getUid()] = drawingFrame;

            // insert the drawing frame with the correct Z order
            if (drawingNodes.length === index) {
                drawingLayerNode.append(drawingFrame);
            } else {
                drawingFrame.insertBefore(drawingNodes[index]);
            }

            // update position and formatting
            updateDrawingFramePosition(drawingModel, drawingFrame);
            DrawingFrame.updateFormatting(app, drawingFrame, attributes);
        }

        /**
         * Updates the position and formatting of the drawing frame that
         * represents the passed drawing model.
         */
        function updateDrawingFrame(drawingModel) {

            var // the existing DOM drawing frame
                drawingFrame = getDrawingFrame(drawingModel);

            // update position and formatting of frames inside the visible area
            if (SheetUtils.rangeOverlapsRange(layerRange, drawingModel.getRange())) {
                updateDrawingFramePosition(drawingModel, drawingFrame);
                DrawingFrame.updateFormatting(app, drawingFrame, drawingModel.getMergedAttributes());
            } else {
                drawingFrame.hide();
            }
        }

        /**
         * Updates position and formatting of all drawing frames in the current
         * visible area.
         */
        function updateDrawingFrames() {
            docView.getDrawingCollection().iterateModelsByPosition(updateDrawingFrame);
        }

        /**
         * Updates the selection of all drawing frames, and triggers a
         * 'render:drawingselection' event, if the drawing selection has
         * changed.
         */
        function renderDrawingSelection() {

            var // all drawing frames selected in the DOM (before running this method)
                oldSelectedFrames = getSelectedDrawingFrames(),
                // all drawing frames selected in the DOM (after running this method)
                newSelectedFrames = $(),
                // all existing drawing frames, as plain DOM collection
                drawingNodes = drawingLayerNode[0].childNodes,
                // disable tracking in read-only mode, or if sheet is protected
                editMode = docModel.getEditMode() && !docView.isSheetLocked();

            // process all drawings to be selected
            _.each(docView.getSelectedDrawings(), function (position) {

                var // the root drawing frame addressed by the current position
                    drawingFrame = drawingNodes[position[0]];

                // drawing frame may be missing, e.g. while remotely inserting new drawings
                // TODO: find embedded drawing frames
                if (drawingFrame && (position.length === 1)) {
                    DrawingFrame.drawSelection(drawingFrame, { movable: editMode, resizable: editMode });
                    newSelectedFrames = newSelectedFrames.add(drawingFrame);
                }
            });

            // remove selection border from all drawing frames not selected anymore
            oldSelectedFrames.not(newSelectedFrames).each(function () { DrawingFrame.clearSelection(this); });

            // notify listeners
            if (!Utils.equalArrays(oldSelectedFrames.get(), newSelectedFrames.get())) {
                self.trigger('render:drawingselection', newSelectedFrames);
            }
        }

        /**
         * Updates the drawing layer according to the passed layer range.
         */
        function changeLayerRangeHandler(event, newLayerRange) {

            var // the old layer range
                oldLayerRange = layerRange;

            // store new layer range for convenience
            layerRange = newLayerRange;

            // repaint all visible drawing frames; or create all drawing frames if
            // not done yet (e.g. active sheet changed, enabled split in current sheet)
            if (oldLayerRange) {
                updateDrawingFrames();
            } else {
                docView.getDrawingCollection().iterateModelsByPosition(createDrawingFrame);
                renderDrawingSelection();
            }
        }

        /**
         * Resets this renderer. Clears the drawing layer element.
         */
        function hideLayerRangeHandler() {

            layerRange = null;

            // safely destroy all image nodes to prevent memory problems on iPad
            app.destroyImageNodes(drawingLayerNode.find('img'));
            drawingLayerNode.empty();
        }

        /**
         * Updates the drawing selection after specific sheet view attributes
         * have been changed.
         */
        function changeSheetViewAttributesHandler(event, attributes) {

            // render drawing selection (triggers a 'render:drawingselection' event)
            if ('selection' in attributes) {
                renderDrawingSelection();
            }

            // trigger a 'render:drawingselection' event if active pane changes
            // (no actual rendering needed, border colors will change via CSS)
            if ('activePane' in attributes) {
                self.trigger('render:drawingselection', getSelectedDrawingFrames());
            }
        }

        /**
         * Processes a new drawing object inserted into the drawing collection.
         */
        function insertDrawingHandler(event, drawingModel, position) {
            createDrawingFrame(drawingModel, position, true);
            renderDrawingSelection();
        }

        /**
         * Processes a drawing object removed from the drawing collection.
         */
        function deleteDrawingHandler(event, drawingModel) {
            getDrawingFrame(drawingModel).remove();
            delete drawingFrames[drawingModel.getUid()];
            renderDrawingSelection();
        }

        /**
         * Processes a drawing object that has been changed in any way.
         */
        function changeDrawingHandler(event, drawingModel) {
            updateDrawingFrame(drawingModel);
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the DOM drawing frame node at the passed document position.
         *
         * @param {Number[]} position
         *  The document position of the drawing model. May specify a position
         *  inside another drawing object.
         *
         * @returns {jQuery}
         *  The DOM node of the drawing frame, as jQuery object; or an empty
         *  jQuery collection, if the drawing frame does not exist.
         */
        this.getDrawingFrame = function (position) {
            var drawingModel = docView.getDrawingCollection().findModel(position);
            return drawingModel ? getDrawingFrame(drawingModel) : $();
        };

        // initialization -----------------------------------------------------

        // initialize class members
        app.onInit(function () {

            // the spreadsheet model and view
            docModel = app.getModel();
            docView = app.getView();

            // update grid pane when document edit mode changes
            self.listenToWhenVisible(docModel, 'change:editmode', renderDrawingSelection);

            // listen to updates of the layer range covered by the grid pane
            self.on('change:layerrange', changeLayerRangeHandler);
            self.on('hide:layerrange', hideLayerRangeHandler);

            // update drawing layer node (only, if this grid pane is visible)
            self.listenToWhenVisible(docView, 'change:sheet:viewattributes', changeSheetViewAttributesHandler);
            self.listenToWhenVisible(docView, 'insert:drawing', insertDrawingHandler);
            self.listenToWhenVisible(docView, 'delete:drawing', deleteDrawingHandler);
            self.listenToWhenVisible(docView, 'change:drawing', changeDrawingHandler);

            // bug 31479: suppress double-clicks for drawings
            drawingLayerNode.on('dblclick', false);
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app.destroyImageNodes(drawingLayerNode);
            app = self = docModel = docView = drawingLayerNode = drawingFrames = null;
        });

    } // class DrawingRenderMixin

    // exports ================================================================

    return DrawingRenderMixin;

});
