/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/popup/tablecolumnmenu', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/view/popup/compoundmenu',
    'io.ox/office/spreadsheet/view/labels',
    'io.ox/office/spreadsheet/view/control/discretefiltergroup',
    'gettext!io.ox/office/spreadsheet'
], function (Utils, CompoundMenu, Labels, DiscreteFilterGroup, gt) {

    'use strict';

    // class TableColumnMenu ==================================================

    /**
     * A drop-down menu for a header cell in a table range, containing options
     * for sorting and filtering.
     *
     * @constructor
     *
     * @extends CompoundMenu
     */
    function TableColumnMenu(app) {

        var // the check list control for discrete filters
            filterGroup = null;

        // base constructor ---------------------------------------------------

        CompoundMenu.call(this, app, {
            anchorAlign: 'trailing',
            actionKey: 'table/column/filter/discrete',
            actionValue: getDiscreteEntries
        });

        // private methods ----------------------------------------------------

        /**
         * Returns the selected discrete filter entries.
         *
         * @returns {Array|Undefined}
         *  An array with all selected discrete filter entries; or undefined,
         *  if all entries are selected (filter disabled).
         */
        function getDiscreteEntries() {
            var entries = filterGroup.getValue();
            if (entries.length < filterGroup.getOptionButtons().length) { return entries; }
        }

        // public methods -----------------------------------------------------

        /**
         * Collects and inserts the cell entries of a table column into this
         * menu, to be used to define discrete filters. Called by grid pane
         * instances from the click handler of cell drop-down buttons.
         *
         * @param {Number[]} address
         *  The address of the header cell of the table column.
         *
         * @param {Object} userData
         *  The user data stored during creation of the drop-down button
         *  associated to this menu. Contains the properties 'tableModel' (the
         *  table model), and 'tableCol' (zero-based column index relative to
         *  the table range).
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  menu has been initialized.
         */
        this.initialize = function (address, userData) {
            app.getView().setSheetViewAttribute('activeTableData', userData);
            return filterGroup.initialize(userData.tableModel, userData.tableCol);
        };

        // initialization -----------------------------------------------------

        this.addSectionLabel(Labels.FILTER_HEADER_LABEL)
            .addGroup(null, filterGroup = new DiscreteFilterGroup(app));

        // reset the 'activeTableData' view attribute when the menu is closed
        this.on('popup:hide', function () {
            app.getView().setSheetViewAttribute('activeTableData', null);
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = filterGroup = null;
        });

    } // class TableColumnMenu

    // constants --------------------------------------------------------------

    /**
     * Tool tip for cell drop-down buttons attached to an instance of the class
     * TableColumnMenu.
     *
     * @constant
     */
    TableColumnMenu.BUTTON_TOOLTIP = gt.pgettext('filter', 'Select values for filtering');

    // exports ================================================================

    // derive this class from class CompoundMenu
    return CompoundMenu.extend({ constructor: TableColumnMenu });

});
