/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/gridpane', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/drawinglayer/view/imageutil',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/utils/clipboard',
    'io.ox/office/spreadsheet/view/popup/tablecolumnmenu',
    'io.ox/office/spreadsheet/view/popup/validationlistmenu',
    'io.ox/office/spreadsheet/view/render/renderutils',
    'io.ox/office/spreadsheet/view/render/cellrendermixin',
    'io.ox/office/spreadsheet/view/render/drawingrendermixin',
    'io.ox/office/spreadsheet/view/mixin/gridtrackingmixin',
    'gettext!io.ox/office/spreadsheet'
], function (Utils, KeyCodes, Forms, TriggerObject, TimerMixin, ImageUtil, Config, SheetUtils, PaneUtils, Clipboard, TableColumnMenu, ValidationListMenu, RenderUtils, CellRenderMixin, DrawingRenderMixin, GridTrackingMixin, gt) {

    'use strict';

    // class GridPane =========================================================

    /**
     * Represents a single scrollable grid pane in the spreadsheet view. If the
     * view has been split or frozen at a specific cell position, the view will
     * consist of up to four panes (top-left, top-right, bottom-left, and
     * bottom-right pane).
     *
     * Instances of this class trigger the following events:
     * - 'change:layerrange'
     *      After the sheet cell range covered by this grid pane has been
     *      changed. Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Object} layerRange
     *          The address of the new cell range covered by this grid pane.
     *      (3) {Object} layerRectangle
     *          The absolute position of the new layer cell range in the sheet,
     *          in pixels.
     *      (3) {Object} refresh
     *          The refresh strategy for the for the columns/rows in the new
     *          layer cell range, with the following properties:
     *          @param {Number} [refresh.col]
     *              Index of the first column that has been changed due to a
     *              column operation in the active sheet (e.g. insert, delete).
     *          @param {Number} [refresh.row]
     *              Index of the first row that has been changed due to a row
     *              operation in the active sheet (e.g. insert, delete).
     *          @param {Boolean} [refresh.geometry=false]
     *              If set to true, the geometry of all columns, rows, and
     *              cells (position, size, and text clipping) needs to be
     *              refreshed completely (e.g. while zooming).
     * - 'hide:layerrange'
     *      After this grid pane has been hidden, and the layer cell range has
     *      been reset to nothing. Event listeners are expected to reset or
     *      update additional settings according to the hidden grid pane.
     * - 'change:scrollpos':
     *      After the scroll position of the grid pane has been changed. Event
     *      listeners may want to update the position of additional DOM content
     *      inserted into the grid pane.
     * - 'render:cellselection'
     *      After the current cell selection has actually been rendered into
     *      the DOM selection layer node (rendering may happen debounced after
     *      several 'change:selection' view events).
     * - 'render:drawingselection'
     *      After the current drawing selection has actually been rendered into
     *      the DOM drawing layer (rendering may happen debounced after several
     *      'change:selection' view events). Event handlers receive the DOM
     *      drawing frames currently selected in this grid pane, as jQuery
     *      collection.
     * - 'select:start'
     *      After selection tracking has been started. Event handlers receive
     *      the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number[]} address
     *          The address of the start cell.
     *      (3) {String} mode
     *          The selection mode according to the keyboard modifier keys:
     *          - 'select': Standard selection without modifier keys.
     *          - 'append': Append new range to current selection (CTRL/META).
     *          - 'extend': Extend current active range (SHIFT).
     * - 'select:move'
     *      After the address of the tracked cell has changed while selection
     *      tracking is active. Event handlers receive the following
     *      parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number[]} address
     *          The address of the current cell.
     * - 'select:end'
     *      After selection tracking has been finished (either successfully or
     *      not). Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number[]} address
     *          The address of the last cell.
     *      (3) {Boolean} success
     *          True if selecting has finished successfully, false if selecting
     *          has been canceled.
     * - 'select:change'
     *      While selection tracking on touch devices is active, and a
     *      selection range has been modified by tracking the resizer handles.
     *      Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number} rangeIndex
     *          The array index of the modified range in the current selection.
     *      (3) {Object} range
     *          The new address of the specified range.
     *      (4) {Number[]} [activeCell]
     *          The address of the active cell. If omitted, the current
     *          active cell will not be changed, unless the new active range
     *          does not contain it anymore.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     * @extends CellRenderMixin
     * @extends DrawingRenderMixin
     * @extends GridTrackingMixin
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this grid pane.
     *
     * @param {String} panePos
     *  The position of this grid pane.
     */
    function GridPane(app, panePos) {

        var // self reference
            self = this,

            // column and row pane side identifiers
            colPaneSide = PaneUtils.getColPaneSide(panePos),
            rowPaneSide = PaneUtils.getRowPaneSide(panePos),

            // the spreadsheet model, and related collections
            docModel = null,
            documentStyles = null,

            // the spreadsheet view, and the header panes associated to the grid pane
            docView = null,
            colHeaderPane = null,
            rowHeaderPane = null,

            // the container node of this grid pane
            rootNode = $('<div class="abs grid-pane f6-target" data-pos="' + panePos + '">'),

            // the scrollable node of this pane (embed into rootNode to allow to hide scroll bars if needed)
            // (attribute unselectable='on' is needed for IE to prevent focusing the node when clicking on a scroll bar)
            scrollNode = $('<div class="grid-scroll" unselectable="on">').appendTo(rootNode),

            // the node controlling the size and position of the scroll bars
            scrollSizeNode = $('<div class="grid-scroll-size">').appendTo(scrollNode),

            // the root node for all layer nodes, positioned dynamically in the scrollable area
            layerRootNode = $('<div class="grid-layer-root noI18n">').appendTo(scrollNode),

            // the cell layer (cell contents and formatting, and grid lines)
            cellLayerNode = $('<div class="grid-layer cell-layer">').appendTo(layerRootNode),

            // the remote layer node for displaying selections of other users
            remoteLayerNode = $('<div class="grid-layer remote-layer">').appendTo(layerRootNode),

            // the selection layer (container for the selected ranges)
            selectionLayerNode = $('<div class="grid-layer selection-layer">').appendTo(layerRootNode),

            // the form control layer (container for drop-down buttons and other controls)
            formLayerNode = $('<div class="grid-layer form-layer">').appendTo(layerRootNode),

            // the drawing layer (container for drawing frames)
            drawingLayerNode = $('<div class="grid-layer drawing-layer">').appendTo(layerRootNode),

            // the range highlight layer (container for the highlighted ranges of formulas)
            highlightLayerNode = $('<div class="grid-layer highlight-layer">').appendTo(layerRootNode),

            // the active range layer (container for the active range selected in cell edit mode)
            activeRangeLayerNode = $('<div class="grid-layer active-layer">').appendTo(layerRootNode),

            // the clipboard node that carries the browser focus for copy/paste events
            clipboardNode = null,
            clipboardFocusMethod = null,

            // the target node for the browser focus (either clipboard node, or the text area in cell edit mode)
            focusTargetNode = null,

            // the ratio between (public) absolute scroll position, and internal scroll bars
            scrollLeftRatio = 1,
            scrollTopRatio = 1,

            // distance between cell A1 and the top-left visible cell of this pane
            hiddenWidth = 0,
            hiddenHeight = 0,

            // the cell range and absolute position covered by the layer nodes in the sheet area
            layerRange = null,
            layerRectangle = null,

            // all registered embedded cell pop-up menus, mapped by type
            cellMenuRegistry = { menus: {}, order: [] },

            // the cell button element caused to open the active cell menu
            activeCellButton = null,

            // the embedded cell menu instance currently activated (visible)
            activeCellMenu = null;

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);
        CellRenderMixin.call(this, app, cellLayerNode);
        DrawingRenderMixin.call(this, app, drawingLayerNode);
        GridTrackingMixin.call(this, app, layerRootNode);

        // private methods ----------------------------------------------------

        /**
         * Refreshes the appearance of this grid pane according to the position
         * of the grid pane that is currently active.
         */
        function updateFocusDisplay() {

            var // which header pane is currently focused (while selection tracking is active)
                focusPaneSide = docView.getSheetViewAttribute('activePaneSide'),
                // which grid pane is currently focused
                focusPanePos = docView.getSheetViewAttribute('activePane'),
                // whether this grid pane appears focused
                focused = false;

            if (docView.isCellEditMode()) {
                // never highlight selection in cell in-place edit mode
                focused = false;
            } else if (docView.hasFrozenSplit()) {
                // always highlight all panes in frozen split mode
                focused = true;
            } else if (_.isString(focusPaneSide)) {
                // a header pane is active (selection tracking): focus this
                // grid pane if it is related to the active header pane
                focused = (colPaneSide === focusPaneSide) || (rowPaneSide === focusPaneSide);
            } else {
                // a single grid pane is focused
                focused = focusPanePos === panePos;
            }

            rootNode.toggleClass(Forms.FOCUSED_CLASS, focused);
        }

        /**
         * Updates the position of the auto-fill handle and the resize handles
         * for cell selection, when entire columns or rows are selected.
         */
        function updateResizeHandlePositions() {

            var // the auto-fill resizer handle node
                autoFillHandleNode = selectionLayerNode.find('.autofill.resizers>[data-pos]'),
                // the range node containing the auto-fill handle
                autoFillRangeNode = autoFillHandleNode.closest('.range');

            // adjust position of the auto-fill handle node for entire column or row selection
            switch (autoFillHandleNode.attr('data-pos')) {
            case 'r':
                autoFillHandleNode.css({ top: -(layerRootNode.position().top + autoFillRangeNode.position().top) });
                break;
            case 'b':
                autoFillHandleNode.css({ left: -(layerRootNode.position().left + autoFillRangeNode.position().left) });
                break;
            }

            // adjust position of the selection handle node for entire column or row selection
            selectionLayerNode.find('.select.resizers>[data-pos]').each(function () {

                var handleNode = $(this),
                    rangeNode = handleNode.closest('.range');

                switch (handleNode.attr('data-pos')) {
                case 'l':
                case 'r':
                    handleNode.css({ top: -(layerRootNode.position().top + rangeNode.position().top) + (scrollNode[0].clientHeight / 2) });
                    break;
                case 't':
                case 'b':
                    handleNode.css({ left: -(layerRootNode.position().left + rangeNode.position().left) + (scrollNode[0].clientWidth / 2) });
                    break;
                }
            });
        }

        /**
         * Handles changed size of the scroll area received from the column or
         * row header pane.
         */
        function updateScrollAreaSize() {

            var // the available size according to existence of scroll bars
                clientWidth = scrollNode[0].clientWidth,
                clientHeight = scrollNode[0].clientHeight,
                // difference of scroll area size between header pane and this grid pane
                widthCorrection = rootNode.width() - clientWidth,
                heightCorrection = rootNode.height() - clientHeight,
                // the new absolute width and height of the scrollable area
                scrollWidth = Math.max(0, colHeaderPane.getScrollSize() - widthCorrection),
                scrollHeight = Math.max(0, rowHeaderPane.getScrollSize() - heightCorrection),
                // the real size of the scroll size node (may be smaller due to browser limits)
                effectiveSize = null;

            // initialize the size of the scroll sizer node, the resulting size may be smaller
            effectiveSize = Utils.setContainerNodeSize(scrollSizeNode, scrollWidth, scrollHeight);

            // recalculate ratio between absolute and effective scroll area size
            scrollLeftRatio = (effectiveSize.width > clientWidth) ? ((scrollWidth - clientWidth) / (effectiveSize.width - clientWidth)) : 0;
            scrollTopRatio = (effectiveSize.height > clientHeight) ? ((scrollHeight - clientHeight) / (effectiveSize.height - clientHeight)) : 0;
        }

        /**
         * Handles changed scroll position received from the column or row
         * header pane.
         */
        function updateScrollPosition() {

            var // the new DOM scroll position
                scrollLeft = (scrollLeftRatio > 0) ? Math.round(colHeaderPane.getScrollPos() / scrollLeftRatio) : 0,
                scrollTop = (scrollTopRatio > 0) ? Math.round(rowHeaderPane.getScrollPos() / scrollTopRatio) : 0,
                // the effective DOM node offsets according to the scroll position
                offsetLeft = 0,
                offsetTop = 0;

            // Only set new DOM scroll position if it differs from current position. This test
            // MUST be done, otherwise Firefox scrolls extremely slow when using the mouse wheel!
            if (scrollLeft !== scrollNode[0].scrollLeft) { scrollNode[0].scrollLeft = scrollLeft; }
            if (scrollTop !== scrollNode[0].scrollTop) { scrollNode[0].scrollTop = scrollTop; }

            // calculate the effective offsets according to current scroll position
            offsetLeft = Math.max(-layerRectangle.width * 2, layerRectangle.left - colHeaderPane.getScrollPos() + scrollLeft - hiddenWidth);
            offsetTop = Math.max(-layerRectangle.height * 2, layerRectangle.top - rowHeaderPane.getScrollPos() + scrollTop - hiddenHeight);

            // workaround IE positioning limitations (not possible to set an
            // absolute left/top position of more than Utils.MAX_NODE_SIZE pixels)
            Utils.setPositionInContainerNode(scrollNode, layerRootNode, offsetLeft, offsetTop);

            // set size of the layer root node (should always be inside the node size limits)
            layerRootNode.css({ width: layerRectangle.width, height: layerRectangle.height });

            // update the position of resizer handles (auto fill, touch selection) for column/row selection
            updateResizeHandlePositions();

            // notify listeners
            self.trigger('change:scrollpos');
        }

        /**
         * Removes all cell drop-down button elements of the specified type.
         *
         * @param {String} menuType
         *  The type specifier of the button elements, as passed to the method
         *  GridPane.createCellButton().
         */
        function removeCellButtons(menuType) {
            formLayerNode.find('>.cell-dropdown-button[data-type="' + menuType + '"]').remove();
        }

        /**
         * Creates a drop-down button element intended to be attached to the
         * trailing border of the specified cell, and inserts it into the form
         * control layer of this grid pane.
         *
         * @param {String} menuType
         *  The type specifier of a pop-up menu that has been registered with
         *  the method GridPane.registerCellMenu(). If the created button will
         *  be clicked, the registered menu will be shown.
         *
         * @param {Number[]} address
         *  The address of the cell the button will be associated to.
         *
         * @param {Object} rectangle
         *  The position of the cell range the button is attached to (pixels).
         *  This rectangle may be larger than the area covered by the specified
         *  cell, e.g. if the cell is part of a merged range, and the drop-down
         *  button has to be displayed in the bottom-right corner of the range.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.icon]
         *      A specific icon to be shown in the button, instead of the
         *      default drop-down caret icon.
         *  @param {String} [options.tooltip]
         *      A tool tip text to be shown when the mouse hovers the drop-down
         *      button.
         *  @param {Boolean} [options.outside=false]
         *      If set to true, the button will be positioned outside the cell
         *      rectangle. By default, the button will be positioned inside.
         *  @param {Number} [options.distance=0]
         *      Distance of the button to the border of the cell rectangle.
         *  @param {Any} [options.userData]
         *      Any additional data that will be added to the jQuery data map
         *      of the button element, and that can be used during the
         *      initialization phase or event handling of pop-up menus
         *      associated to the drop-down button.
         *
         * @returns {jQuery}
         *  The created and inserted button element, as jQuery object. The
         *  jQuery data map of the button element will contain the following
         *  properties:
         *  - {Number[]} address
         *      The passed cell address.
         *  - {Number} anchorWidth
         *      The effective total width of the passed rectangle and the
         *      button rectangle.
         *  - {Object} [userData]
         *      The user data passed with the option 'userData'.
         */
        function createCellButton(menuType, address, rectangle, options) {

            var // whether to place the button element outside the cell rectangle
                outside = Utils.getBooleanOption(options, 'outside', false),
                // distance of the button element to the cell rectangle
                distance = Utils.getIntegerOption(options, 'distance', 0, 0),
                // additional properties for the jQuery data map
                userData = Utils.getOption(options, 'userData'),
                // outer size of the button element (according to cell height)
                size = Utils.minMax(rectangle.height - 1, 18, 30),
                // position of the button in the sheet area
                buttonRect = {
                    left: Math.max(rectangle.left + rectangle.width + (outside ? distance : -(size + 1)), hiddenWidth),
                    top: Math.max(rectangle.top + rectangle.height - size - 1, hiddenHeight),
                    width: size,
                    height: size
                },
                // the anchor rectangle to be used for attached drop-down menu
                anchorRect = Utils.getBoundingRectangle(rectangle, buttonRect),
                // the HTML mark-up of the resulting button element
                markup = null,
                // the button element, as jQuery object
                buttonNode = null;

            // create button mark-up
            markup = '<div class="cell-dropdown-button skip-tracking" tabindex="-1" data-type="' + menuType + '" data-address="' + SheetUtils.getCellKey(address) + '"';
            markup += ' style="' + self.getLayerRectangleStyleMarkup(buttonRect) + 'line-height:' + (buttonRect.height - 2) + 'px;">';
            markup += Forms.createIconMarkup(Utils.getStringOption(options, 'icon', 'fa-caret-down')) + '</div>';

            // create and return the button element
            buttonNode = $(markup).data({ address: address, anchorWidth: anchorRect.width, userData: userData });
            Forms.setToolTip(buttonNode, Utils.getStringOption(options, 'tooltip'));
            formLayerNode.append(buttonNode);
            return buttonNode;
        }

        /**
         * Creates the form controls (cell drop-down buttons) currently visible
         * in this grid pane.
         */
        function renderFormControls() {

            // nothing to do, if all columns/rows are hidden (method may be called debounced)
            if (!self.isVisible()) {
                formLayerNode.empty();
                return;
            }

            var // the collections of the active sheet
                colCollection = docView.getColCollection(),
                rowCollection = docView.getRowCollection();

            // create drop-down buttons for all visible table ranges with activated filter/sorting
            removeCellButtons('table');
            _.each(docView.getTableCollection().findTables(layerRange, { active: true }), function (tableModel) {

                var // the cell range covered by the table
                    tableRange = tableModel.getRange(),
                    // the settings of the header row
                    rowEntry = rowCollection.getEntry(tableRange.start[1]);

                // do nothing if the header row is outside the layer range, or hidden
                if ((tableRange.start[1] < layerRange.start[1]) || (rowEntry.size === 0)) { return; }

                // visit all visible columns of the table in the layer range
                var colInterval = SheetUtils.getIntersectionInterval(SheetUtils.getColInterval(tableRange), SheetUtils.getColInterval(layerRange));
                colCollection.iterateEntries(colInterval, function (colEntry) {

                    var // relative table column index
                        tableCol = colEntry.index - tableRange.start[0];

                    // bug 36152: do not render buttons covered by merged ranges
                    if (tableModel.getColumnAttributes(tableCol).filter.hideMergedButton) { return; }

                    // bug 36152: visible drop-down button in a merged range is placed in the last column of the
                    // merged range, but is associated to (shows cell data of) the first column in the merged range
                    while ((tableCol > 0) && tableModel.getColumnAttributes(tableCol - 1).filter.hideMergedButton) {
                        tableCol -= 1;
                    }

                    var address = [colEntry.index, rowEntry.index],
                        rectangle = { left: colEntry.offset, top: rowEntry.offset, width: colEntry.size, height: rowEntry.size };

                    createCellButton('table', address, rectangle, {
                        tooltip: TableColumnMenu.BUTTON_TOOLTIP,
                        icon: tableModel.isColumnFiltered(tableCol) ? 'fa-filter' : null,
                        userData: { tableModel: tableModel, tableCol: tableCol }
                    });
                });
            });
        }

        /**
         * Enlarges the passed rectangle in-place to the left.
         */
        function enlargeLeft(rectangle, size) {
            rectangle.left -= size;
            rectangle.width += size;
        }

        /**
         * Enlarges the passed rectangle in-place to the top.
         */
        function enlargeTop(rectangle, size) {
            rectangle.top -= size;
            rectangle.height += size;
        }

        /**
         * Renders all visible selection ranges according to the current pane
         * layout data received from a view layout update notification.
         */
        function renderCellSelection() {

            // nothing to do, if all columns/rows are hidden (method may be called debounced)
            if (!self.isVisible()) {
                selectionLayerNode.empty();
                return;
            }

            var // the entire cell selection
                selection = docView.getSelection(),
                // the range occupied by the active cell (will not be filled)
                activeRange = null,
                // the position of the active cell, extended to the merged range
                activeRectangle = null,
                // additional data for active auto-fill tracking
                autoFillData = docView.getSheetViewAttribute('autoFillData'),
                // whether the active cell touches the borders of selection ranges
                activeBorders = { left: false, top: false, right: false, bottom: false },
                // hide left/top border of selection ranges, if first column/row is hidden
                firstColHidden = !docView.getColCollection().isEntryVisible(0),
                firstRowHidden = !docView.getRowCollection().isEntryVisible(0),
                // the HTML mark-up for all selection ranges but the active range
                rangesMarkup = '',
                // the HTML mark-up for the active selection range
                activeMarkup = '';

            // returns the mark-up of a resizer handler according to the type of the passed range
            function createResizerMarkup(range, leadingCorner) {
                var hPos = leadingCorner ? 'l' : 'r', vPos = leadingCorner ? 't' : 'b';
                return '<div data-pos="' + (docModel.isColRange(range) ? hPos : docModel.isRowRange(range) ? vPos : (vPos + hPos)) + '"></div>';
            }

            // show drop-down button for list validation
            function createValidationButton() {

                // remove the old button element
                removeCellButtons('validation');

                // do not show in read-only mode, or when auto-fill is active
                if (!activeRange || !docModel.getEditMode() || _.isObject(autoFillData)) { return; }

                // check that the active range is inside the row interval covered by this grid pane (needed to prevent
                // painting the upper part of the button, if cell is located in the top row of the lower part of a frozen split)
                if (activeRange.end[1] < rowHeaderPane.getAvailableInterval().first) { return; }

                // check that the active cell provides a validation drop-down list
                var validationSettings = docView.getValidationCollection().getValidationSettings(selection.activeCell);
                if (!_.isObject(validationSettings) || !/^(source|list)$/.test(validationSettings.attributes.type) || !validationSettings.attributes.showDropDown) { return; }

                // create and insert the button node
                createCellButton('validation', selection.activeCell, docView.getRangeRectangle(activeRange), {
                    tooltip: ValidationListMenu.BUTTON_TOOLTIP,
                    outside: true,
                    distance: activeBorders.right ? 2 : 0
                });
            }

            // add active flag to range object
            selection.ranges[selection.activeRange].active = true;

            // adjust single range for auto-fill tracking
            if ((selection.ranges.length === 1) && _.isObject(autoFillData)) {

                var // the selected cell range
                    firstRange = selection.ranges[0],
                    // whether to expand/shrink the leading or trailing border
                    leading = /^(left|top)$/.test(autoFillData.border),
                    // whether to expand/shrink columns or rows
                    columns = /^(left|right)$/.test(autoFillData.border),
                    // the array index in the cell address
                    addrIndex = columns ? 0 : 1;

                activeRange = _.copy(firstRange, true);
                if (autoFillData.count >= 0) {
                    // adjust range for auto-fill mode
                    if (leading) {
                        firstRange.start[addrIndex] -= autoFillData.count;
                    } else {
                        firstRange.end[addrIndex] += autoFillData.count;
                    }
                } else {
                    // adjust range for deletion mode
                    firstRange.collapsed = true;
                    if (-autoFillData.count < SheetUtils.getIndexCount(firstRange, columns)) {
                        // range partly covered
                        if (leading) {
                            activeRange.start[addrIndex] -= autoFillData.count;
                        } else {
                            activeRange.end[addrIndex] += autoFillData.count;
                        }
                    } else {
                        // remove active range for deletion of the entire range
                        activeRange = {};
                    }
                }

                // add tracking style effect
                firstRange.tracking = true;
            }

            // the range covered by the active cell (or any other range in auto-fill tracking mode)
            activeRange = _.isObject(activeRange) ? (_.isEmpty(activeRange) ? null : activeRange) :
                docView.getMergeCollection().expandRangeToMergedRanges({ start: selection.activeCell, end: selection.activeCell });

            // convert active range to a rectangle relative to the layer root node
            if (_.isObject(activeRange)) {
                self.iterateRangesForRendering(activeRange, function (range, index, rectangle) {
                    activeRectangle = rectangle;
                }, { alignToGrid: true });
            }

            // render all ranges that are visible in this grid pane
            self.iterateRangesForRendering(selection.ranges, function (range, index, rectangle) {

                var // position of active range, relative to current range
                    relActiveRect = null,
                    // mark-up for the current range
                    rangeMarkup = '';

                // hide left/top border of the range, if the first column/row is hidden
                if (firstColHidden && (range.start[0] === 0)) {
                    enlargeLeft(rectangle, 5);
                }
                if (firstRowHidden && (range.start[1] === 0)) {
                    enlargeTop(rectangle, 5);
                }

                // generate the HTML mark-up for the range
                rangeMarkup = '<div class="range';
                if (range.active) { rangeMarkup += ' active'; }
                if (range.tracking) { rangeMarkup += ' tracking-active'; }
                if (range.collapsed) { rangeMarkup += ' collapsed'; }
                rangeMarkup += '" style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '" data-index="' + index + '">';

                // insert the semi-transparent fill elements (never cover the active range with the fill elements)
                if (activeRectangle && SheetUtils.rangeOverlapsRange(range, activeRange)) {

                    // initialize position of active cell, relative to selection range
                    relActiveRect = _.clone(activeRectangle);
                    relActiveRect.left -= rectangle.left;
                    relActiveRect.top -= rectangle.top;
                    relActiveRect.right = rectangle.width - relActiveRect.left - relActiveRect.width;
                    relActiveRect.bottom = rectangle.height - relActiveRect.top - relActiveRect.height;

                    // insert fill element above active cell
                    if (relActiveRect.top > 0) {
                        rangeMarkup += '<div class="fill" style="left:0;right:0;top:0;height:' + relActiveRect.top + 'px;"></div>';
                    }

                    // insert fill element left of active cell
                    if (relActiveRect.left > 0) {
                        rangeMarkup += '<div class="fill" style="left:0;width:' + relActiveRect.left + 'px;top:' + relActiveRect.top + 'px;height:' + relActiveRect.height + 'px;"></div>';
                    }

                    // insert fill element right of active cell
                    if (relActiveRect.right > 0) {
                        rangeMarkup += '<div class="fill" style="left:' + (relActiveRect.left + relActiveRect.width) + 'px;right:0;top:' + relActiveRect.top + 'px;height:' + relActiveRect.height + 'px;"></div>';
                    }

                    // insert fill element below active cell
                    if (relActiveRect.bottom > 0) {
                        rangeMarkup += '<div class="fill" style="left:0;right:0;top:' + (relActiveRect.top + relActiveRect.height) + 'px;bottom:0;"></div>';
                    }

                    // update border flags for the active cell
                    activeBorders.left = activeBorders.left || (range.start[0] === activeRange.start[0]);
                    activeBorders.top = activeBorders.top || (range.start[1] === activeRange.start[1]);
                    activeBorders.right = activeBorders.right || (range.end[0] === activeRange.end[0]);
                    activeBorders.bottom = activeBorders.bottom || (range.end[1] === activeRange.end[1]);

                } else {
                    rangeMarkup += '<div class="abs fill"></div>';
                }

                // generate the HTML mark-up for the selection range
                rangeMarkup += '<div class="border"></div>';

                // additional mark-up for single-range selection
                if (selection.ranges.length === 1) {
                    if (Modernizr.touch && !_.isObject(autoFillData)) {
                        // add resize handlers for selection on touch devices
                        rangeMarkup += '<div class="select resizers">' + createResizerMarkup(range, true) + createResizerMarkup(range, false) + '</div>';
                    } else {
                        // add the auto-fill handler in the bottom right corner of the selection
                        rangeMarkup += '<div class="autofill resizers">' + createResizerMarkup(range, false) + '</div>';
                    }
                }

                // close the range node
                rangeMarkup += '</div>';

                // and the generated mark-up to the appropriate mark-up list
                if (range.active) { activeMarkup += rangeMarkup; } else { rangesMarkup += rangeMarkup; }
            }, { alignToGrid: true });

            // additions for the active cell (or active range in auto-fill)
            if (activeRectangle) {

                // add thin border for active cell on top of the selection
                rangesMarkup += '<div class="active-cell';
                _.each(activeBorders, function (isBorder, borderName) { if (!isBorder) { rangesMarkup += ' ' + borderName; } });
                rangesMarkup += '" style="' + PaneUtils.getRectangleStyleMarkup(activeRectangle) + '"></div>';
            }

            // insert entire HTML mark-up into the selection container node
            selectionLayerNode[0].innerHTML = rangesMarkup + activeMarkup;

            // update the visibility/position of additional DOM elements
            createValidationButton();
            updateResizeHandlePositions();
            docView.updateHyperlinkPopup();

            // notify listeners
            self.trigger('render:cellselection');
        }

        /**
         * Renders all visible highlighted cell ranges.
         */
        function renderHighlightedRanges() {

            var // all ranges currently highlighted
                highlightRanges = docView.getSheetViewAttribute('highlightRanges'),
                // unified range identifiers
                rangeIds = _.chain(highlightRanges).pluck('id').unique().value(),
                // array index of the range currently tracked
                highlightIndex = docView.getSheetViewAttribute('highlightIndex'),
                // the HTML mark-up for all highlight ranges
                markup = '';

            // render all ranges that are visible in this grid pane
            if (self.isVisible() && _.isArray(highlightRanges)) {
                self.iterateRangesForRendering(highlightRanges, function (range, index, rectangle) {

                    var // array index of the range identifier (multiple ranges may origin from the same defined name)
                        idIndex = rangeIds.indexOf(range.id),
                        // CSS class name for range color style
                        styleId = (idIndex % Utils.SCHEME_COLOR_COUNT) + 1;

                    // generate the HTML mark-up for the highlighted range
                    markup += '<div class="range ' + ((highlightIndex === index) ? ' tracking-active' : '') + '" style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '"';
                    markup += ' data-style="' + styleId + '" data-index="' + index + '"><div class="fill"></div>';
                    if (docModel.getEditMode() && Utils.getBooleanOption(range, 'draggable', false)) {
                        markup += '<div class="borders">';
                        markup += _.map('lrtb', function (pos) { return '<div data-pos="' + pos + '"></div>'; }).join('');
                        markup += '</div><div class="resizers">';
                        markup += _.map(['tl', 'tr', 'bl', 'br'], function (pos) { return '<div data-pos="' + pos + '"></div>'; }).join('');
                        markup += '</div>';
                    } else {
                        markup += '<div class="static-border"></div>';
                    }
                    markup += '</div>';
                });
            }

            // insert entire HTML mark-up into the container node
            highlightLayerNode[0].innerHTML = markup;
        }

        /**
         * Renders the activated cell ranges. These ranges are used for example
         * while selecting new ranges for a formula while in cell edit mode.
         */
        var renderActiveSelection = (function () {

            var // the timer for the wireframe animation
                timer = null;

            return function () {

                var // the active ranges
                    selection = docView.getSheetViewAttribute('activeSelection'),
                    // the HTML mark-up for the range
                    markup = '';

                // generate the HTML mark-up for the ranges
                if (self.isVisible() && _.isObject(selection)) {
                    self.iterateRangesForRendering(selection.ranges, function (range, index, rectangle) {
                        markup += '<div class="range" style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '"><div class="borders">';
                        markup += _.map('lrtb', function (pos) { return '<div data-pos="' + pos + '"></div>'; }).join('');
                        markup += '</div></div>';
                    });
                    if (!timer) {
                        timer = self.repeatDelayed(function (index) {
                            activeRangeLayerNode.attr('data-frame', index % 6);
                        }, 100);
                    }
                } else if (timer) {
                    timer.abort();
                    timer = null;
                }

                // insert entire HTML mark-up into the container node
                activeRangeLayerNode[0].innerHTML = markup;
            };
        }());

        /**
         * Renders selections of all remote users which have the same sheet
         * activated.
         */
        var renderRemoteSelections = Config.SHOW_REMOTE_SELECTIONS ? function () {

            // nothing to do, if all columns/rows are hidden (method may be called debounced)
            if (!self.isVisible()) {
                remoteLayerNode.empty();
                return;
            }

            var // prepare equal mark-up in all selection ranges
                borderMarkup = '<div class="borders">' + _.map('lrtb', function (pos) { return '<div data-pos="' + pos + '"></div>'; }).join('') + '</div>',
                // topLeft visible address of this pane, to detect if the name-layer outside of the view
                firstRow = self.getTopLeftAddress()[1],
                // the HTML mark-up for all user selections
                markup = '';

            // generate selections of all remote users on the same sheet
            _.each(docModel.getViewAttribute('remoteClients'), function (client) {

                var // the selection of the remote client
                    selection = client.userData.selection;

                // skip users that have another sheet activated
                if (client.userData.sheet !== docModel.getActiveSheet()) { return; }

                var // the escaped user name, ready to be inserted into HTML mark-up
                    userName = Utils.escapeHTML(_.noI18n(client.userName));

                // render all ranges of the user
                self.iterateRangesForRendering(selection.ranges, function (range, index, rectangle) {

                    var // the CSS classes to be set at the range (bug 35604: inside range if on top of the grid pane)
                        classes = 'range' + ((range.start[1] === firstRow) ? ' badge-inside' : '');

                    // create the mark-up for the range
                    markup += '<div class="' + classes + '" data-style="' + client.colorIndex + '" data-username="' + userName + '"';
                    markup += ' style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '">' + borderMarkup + '</div>';
                }, { alignToGrid: true });
            });

            // insert entire HTML mark-up into the container node
            remoteLayerNode[0].innerHTML = markup;
        } : $.noop;

        /**
         * Registers a cell menu instance to be shown for specific drop-down
         * buttons embedded in the active sheet.
         *
         * @param {String} menuType
         *  The type specifier for the button elements in the form layer, that
         *  are actually triggering the menu.
         *
         * @param {BaseMenu} menu
         *  The menu instance that will be shown when clicking the specified
         *  cell drop-down buttons. This instance takes ownership of the passed
         *  menu (the menu does not have to be destroyed manually)! The passed
         *  menu instance may provide a public method initialize() that will be
         *  invoked before opening the menu. The method will receive the
         *  following parameters:
         *  (1) {Number[]} address
         *      The address of the cell containing the button node that caused
         *      opening the menu.
         *  (2) {Any} userData
         *      Additional user data that have been passed to the method
         *      GridPane.createCellButton().
         *  The method may return a promise indicating that initialization of
         *  the drop-down menu has been finished. The promise returned by the
         *  method may be abortable; in this case, when opening the drop-down
         *  menu repeatedly, or opening another cell drop-down menu quickly,
         *  old initialization requests will be aborted, before calling the
         *  initialization callback function again.
         */
        function registerCellMenu(menuType, menu) {

            // add the menu to the registry map
            cellMenuRegistry.menus[menuType] = menu;
            cellMenuRegistry.order.push(menuType);

            // only the active button must be included in the focus handling of the menu
            menu.registerFocusableNodes(function () { return activeCellButton; });

            // dynamic anchor position according to the clicked drop-down button
            menu.setAnchor(function () {

                var // current page position of the button node
                    buttonRect = Utils.getNodePositionInPage(activeCellButton),
                    // the width of cell and button rectangle, used for anchor of the drop-down menu node
                    anchorWidth = activeCellButton.data('anchorWidth');

                // create the anchor rectangle according to the current position of the button node
                return {
                    left: buttonRect.left + buttonRect.width - anchorWidth,
                    top: buttonRect.top,
                    width: anchorWidth,
                    height: buttonRect.height
                };
            });

            // hide the menu automatically, if the anchor leaves the visible area of the grid pane
            menu.setAnchorBox(scrollNode);

            // remember active menu for usage in other methods
            self.listenTo(menu, 'popup:show', function () {
                activeCellButton.addClass('dropdown-open');
                activeCellMenu = menu;
            });

            // abort initialization if pop-up menu will be closed quickly
            self.listenTo(menu, 'popup:hide', function () {
                activeCellButton.removeClass('dropdown-open');
                activeCellMenu = null;
            });
        }

        /**
         * Opens or closes the embedded cell pop-up menu associuated to the
         * passed cell drop-down button.
         *
         * @param {jQuery} buttonNode
         *  The cell button node whose menu will be opened or closed.
         */
        var toggleCellMenu = (function () {

            var // current asynchronous initialization of the cell menu contents
                initPromise = null;

            // aborts current menu initialization
            function abortInitialization() {
                if (_.isObject(initPromise) && _.isFunction(initPromise.abort)) {
                    initPromise.abort();
                }
            }

            // the actual toggleCellMenu() method to be returned from local scope
            function toggleCellMenu(buttonNode) {

                var // the pop-up menu instance
                    menu = Utils.getObjectOption(cellMenuRegistry.menus, buttonNode.attr('data-type')),
                    // the result of the initialization callback function
                    initResult = null;

                // check existence of menu registration
                if (!menu) {
                    Utils.warn('GridPane.toggleCellMenu(): missing menu registration for clicked cell drop-down button');
                    return;
                }

                // abort another initialization of a cell menu currently running
                abortInitialization();

                // always leave cell edit mode (do not open the menu, if formula validation fails)
                if (!docView.leaveCellEditMode('auto', { validate: true })) { return; }

                // if the menu is currently visible, close it and return
                if (menu.isVisible()) {
                    menu.hide();
                    docView.grabFocus();
                    return;
                }

                // remember current button node, will be returned as focusable node to prevent
                // closing the menu too fast (the menu may be associated to different buttons)
                activeCellButton = buttonNode;

                // invoke initialization callback handler
                if (_.isFunction(menu.initialize)) {
                    initResult = menu.initialize(buttonNode.data('address'), buttonNode.data('userData'));
                }

                // show the menu (unless the promise has already been rejected synchronously)
                initPromise = $.when(initResult);
                if (initPromise.state() !== 'rejected') { menu.busy().show(); }

                // show warning alerts if available
                docView.yellOnPromise(initPromise);

                // further initialization for the open menu
                if (menu.isVisible()) {
                    self.listenTo(initPromise, 'fail', function () { menu.hide(); });
                    menu.one('popup:hide', abortInitialization);
                }

                // final clean-up
                self.listenTo(initPromise, 'always', function () {
                    menu.idle();
                    initPromise = null;
                });
            }

            return toggleCellMenu;
        }());

        /**
         * Hides the cell pop-up menu currently shown.
         *
         * @returns {Boolean}
         *  Whether a cell list menu was actually open and has been hidden.
         */
        function hideActiveCellMenu() {
            if (activeCellMenu && activeCellMenu.isVisible()) {
                activeCellMenu.hide();
                return true;
            }
            return false;
        }

        /**
         * Tries to activate an embedded cell pop-up menu attached to the
         * specified cell. If the cell contains multiple drop-down menus, the
         * method searches for a cell menu currently activated for that cell,
         * and activates the next registered menu. The order of the menus is
         * equal to the registration order via the method registerCellMenu().
         *
         * @param {Number[]} address
         *  The address of a cell to open an embedded pop-up menu for.
         *
         * @returns {Boolean}
         *  Whether a drop-down menu has been found and activated for the cell.
         */
        function activateCellMenu(address) {

            var // all cell button nodes attached to the specified cell
                cellButtons = formLayerNode.find('>.cell-dropdown-button[data-address="' + SheetUtils.getCellKey(address) + '"]'),
                // all menu types occuring in cellButtons, in registration order
                menuTypes = null,
                // type of the pop-up menu currently open (needed to select the next menu)
                activeMenuType = null;

            // no cell drop-down button found: nothing to do
            if (cellButtons.length === 0) { return false; }

            // always scroll to the cell containing the pop-up menu
            self.scrollToCell(address);

            // filter the menu types of the existing cell buttons in registration order
            menuTypes = _.filter(cellMenuRegistry.order, function (menuType) {
                return cellButtons.filter('[data-type="' + menuType + '"]').length > 0;
            });

            // get type of the pop-up menu currently opened and attached to the specified cell
            if (activeCellButton && activeCellMenu && _.isEqual(activeCellButton.data('address'), address)) {
                activeMenuType = activeCellButton.attr('data-type');
            }

            // do nothing if the active menu is the only menu for the cell
            if (_.isString(activeMenuType) && (menuTypes.length === 1)) { return true; }

            // always hide the active cell menu before opening the next menu
            hideActiveCellMenu();

            // pick the next menu type from the list of available menu types
            activeMenuType = menuTypes[(_.indexOf(menuTypes, activeMenuType) + 1) % menuTypes.length];

            // open the menu
            toggleCellMenu(cellButtons.filter('[data-type="' + activeMenuType + '"]'));
            return true;
        }

        /**
         * Handles a click on an embedded cell drop-down button, and opens the
         * associated registered menu.
         *
         * @param {jQuery.Event} event
         *  The jQuery click event.
         */
        function cellButtonClickHandler(event) {
            toggleCellMenu($(event.currentTarget));
        }

        /**
         * Updates the layer range and the DOM layer nodes. Removes DOM cell
         * nodes of all cells not visible anymore, and renders new cell nodes
         * now visible in this grid pane.
         *
         * @param {Object|Null} interval
         *  The new column/row interval to be inserted into the layer range. If
         *  set to null, the grid pane will be hidden, and all layer nodes will
         *  be cleared.
         *
         * @param {Boolean} columns
         *  Whether the modified layer range was caused by a change in the
         *  column header pane (true) or row header pane (false).
         *
         * @param {Object} [refresh]
         *  The refresh strategy for the columns/rows in the new interval, as
         *  passed in the respective 'change:interval' event of a header pane.
         */
        var updateLayerInterval = (function () {

            var // the cached interval/refresh settings (needed to build the layer range from several events)
                colSettings = { interval: null, refresh: {} },
                rowSettings = { interval: null, refresh: {} };

            // direct callback: collect interval settings from column/row header pane
            function registerInterval(interval, columns, refresh) {

                var // store the passed column/row interval in the internal variable
                    settings = columns ? colSettings : rowSettings,
                    // the index of the first column/row that needs full refresh
                    index = Utils.getIntegerOption(refresh, 'index');

                RenderUtils.log('GridPane.registerInterval(): pos=' + panePos + ' interval=' + (interval ? SheetUtils.getIntervalName(interval, columns) : 'null') + ' refresh=' + JSON.stringify(refresh));

                // store the new interval, update refresh settings
                if (_.isObject(interval)) {
                    settings.interval = _.clone(interval);
                    if (_.isNumber(index)) {
                        settings.refresh.index = _.isNumber(settings.refresh.index) ? Math.min(settings.refresh.index, index) : index;
                    }
                    settings.refresh.geometry = settings.refresh.geometry || Utils.getBooleanOption(refresh, 'geometry', false);
                } else {
                    settings.interval = null;
                    settings.refresh = {};
                }
            }

            // deferred callback: calculate the new layer range
            function calculateLayerRange() {

                var // the old (current) layer range
                    oldLayerRange = layerRange;

                // calculate the new layer range
                if (colSettings.interval && rowSettings.interval) {
                    layerRange = SheetUtils.makeRangeFromIntervals(colSettings.interval, rowSettings.interval);
                    // shrink to visible columns/rows (range may be set to null)
                    layerRange = docView.getSheetModel().shrinkToVisibleRange(layerRange);
                } else {
                    layerRange = null;
                }
                RenderUtils.log('GridPane.calculateLayerRange(): pos=' + panePos + ' layerRange=' + (layerRange ? SheetUtils.getRangeName(layerRange) : 'null') + ' colRefresh=' + JSON.stringify(colSettings.refresh) + ' rowRefresh=' + JSON.stringify(rowSettings.refresh));

                // update visible sheet contents according to the new layer range
                if (layerRange) {

                    // get the new layer rectangle in pixels
                    layerRectangle = docView.getRangeRectangle(layerRange);

                    // update the scroll position (exact position of the layer root node in the DOM)
                    scrollNode.show();
                    updateScrollPosition();

                    // notify listeners (the various renderers)
                    self.trigger('change:layerrange', layerRange, layerRectangle, {
                        col: colSettings.refresh.index,
                        row: rowSettings.refresh.index,
                        geometry: colSettings.refresh.geometry || rowSettings.refresh.geometry
                    });

                    // update all own DOM layer nodes
                    renderCellSelection();
                    renderHighlightedRanges();
                    renderActiveSelection();
                    renderRemoteSelections();
                    renderFormControls();
                }

                // do nothing, if the layer range was invalid before
                else if (oldLayerRange) {

                    // clear the layer rectangle
                    layerRectangle = null;

                    // notify listeners (the various renderers)
                    self.trigger('hide:layerrange');

                    // clear all own DOM layers
                    scrollNode.hide();
                    remoteLayerNode.empty();
                    selectionLayerNode.empty();
                    formLayerNode.empty();
                    highlightLayerNode.empty();
                    activeRangeLayerNode.empty();
                }

                // reset cached refresh data (but keep the current interval)
                colSettings.refresh = {};
                rowSettings.refresh = {};
            }

            // debounced interval handling
            var calculateLayerRangeDebounced = self.createDebouncedMethod(registerInterval, calculateLayerRange);

            // the actual updateLayerInterval() method to be returned from the local scope
            function updateLayerInterval(interval, columns, refresh, options) {
                // refresh layer range directly depending on the passed option, but always if the interval is deleted
                if (!interval || Utils.getBooleanOption(options, 'direct', false)) {
                    registerInterval(interval, columns, refresh);
                    calculateLayerRange();
                } else {
                    calculateLayerRangeDebounced(interval, columns, refresh);
                }
            }

            return updateLayerInterval;
        }());

        /**
         * Updates the grid pane after the document edit mode has changed.
         */
        function editModeHandler(event, editMode) {

            // hide the active pop-up menu, if the document changes to read-only mode
            if (!editMode && app.isImportFinished()) {
                hideActiveCellMenu();
            }

            // redraw selection according to edit mode
            renderCellSelection();
            renderHighlightedRanges();
        }

        /**
         * Updates the layer nodes after specific document view attributes have
         * been changed.
         */
        function changeViewAttributesHandler(event, attributes) {

            // display remote selections
            if ('remoteClients' in attributes) {
                renderRemoteSelections();
            }
        }

        /**
         * Updates the layer nodes after specific sheet view attributes have
         * been changed.
         */
        function changeSheetViewAttributesHandler(event, attributes) {

            // render cell selection (triggers a 'render:cellselection' event)
            if (Utils.hasProperty(attributes, /^(selection$|split|autoFillData$)/)) {
                renderCellSelection();
            }

            // refresh focus display ('activePane', and 'activePaneSide')
            if (Utils.hasProperty(attributes, /^activePane/)) {
                updateFocusDisplay();
            }

            // refresh embedded form controls after split changes
            if (Utils.hasProperty(attributes, /^split/)) {
                renderFormControls();
            }

            // trigger a 'render:cellselection' event if active pane changes
            // (no actual rendering needed, border colors will change via CSS)
            if ('activePane' in attributes) {
                self.trigger('render:cellselection');
            }

            // changed highlight ranges
            if (Utils.hasProperty(attributes, /^highlight/)) {
                renderHighlightedRanges();
            }

            // changed active ranges (used while selecting ranges in cell edit mode)
            if ('activeSelection' in attributes) {
                renderActiveSelection();
            }

            // hide the active cell pop-up menu after specific attributes have been changed
            if (Utils.hasProperty(attributes, /^(selection$|split|activePane|autoFillData$)/)) {
                hideActiveCellMenu();
            }
        }

        /**
         * Prepares this grid pane for entering the in-place cell edit mode.
         */
        function cellEditEnterHandler() {
            // disable global F6 focus traveling into this grid pane while in-place cell edit mode is active in another grid pane
            rootNode.toggleClass('f6-target', self === docView.getActiveGridPane());
            updateFocusDisplay();
            // always hide the active cell pop-up menu when cell edit mode starts
            hideActiveCellMenu();
        }

        /**
         * Prepares this grid pane for leaving the in-place cell edit mode.
         */
        function cellEditLeaveHandler() {
            rootNode.addClass('f6-target');
            updateFocusDisplay();
        }

        /**
         * Renders the cell grid, the selection, and the highlighted ranges,
         * after the cell geometry has been changed.
         */
        var changeCellLayoutHandler = this.createDebouncedMethod($.noop, function () {
            renderCellSelection();
            renderHighlightedRanges();
            renderFormControls();
        });

        /**
         * Handles DOM 'scroll' events of the scrollable node.
         */
        function scrollHandler() {
            colHeaderPane.scrollTo(Math.round(scrollNode[0].scrollLeft * scrollLeftRatio));
            rowHeaderPane.scrollTo(Math.round(scrollNode[0].scrollTop * scrollTopRatio));
        }

        /**
         * Handles 'keydown' events from the root node.
         */
        function keyDownHandler(event) {

            // ESCAPE key (only one action per key event)
            if (event.keyCode === KeyCodes.ESCAPE) {

                // cancel current tracking, but do nothing else
                if ($.cancelTracking()) {
                    return false;
                }

                // cancel custom selection mode (before deselecting drawings)
                if (docView.cancelCustomSelectionMode()) {
                    return false;
                }

                // close cell pop-up menu currently open
                if (hideActiveCellMenu()) {
                    return false;
                }

                // drawing selection: back to cell selection
                if (docView.hasDrawingSelection()) {
                    docView.removeDrawingSelection();
                    return false;
                }

                // otherwise: let ESCAPE key bubble up
                return;
            }

            // show cell drop-down menu of active cell on ALT+DOWN
            if (KeyCodes.matchKeyCode(event, 'DOWN_ARROW', { alt: true })) {
                activateCellMenu(docView.getActiveCell());
                return false;
            }

            // handle keyboard shortcuts for the cell pop-up menu currently visible
            if (activeCellMenu && _.isFunction(activeCellMenu.handleKeyDownEvent) && activeCellMenu.handleKeyDownEvent(event)) {
                return;
            }
        }

        /**
         * Handles 'drop' events from the root node.
         */
        function dropHandler(event) {

            function insertImage(imageDescriptor) {
                return docView.insertImage(imageDescriptor);
            }

            function yellUnsupported() {
                docView.yell({ type: 'info', message: gt('File type not supported.') });
            }

            // image comes from local hard drive
            if (event.originalEvent.dataTransfer.files.length > 0) {

                _.each(event.originalEvent.dataTransfer.files, function (eventFile) {
                    if (ImageUtil.hasFileImageExtension(eventFile.name)) {
                        docView.enterBusy();
                        ImageUtil.insertFile(app, eventFile).then(insertImage).always(function () { docView.leaveBusy(); });
                    } else {
                        yellUnsupported();
                    }
                });

            // image comes from another browser-window
            } else if (event.originalEvent.dataTransfer.getData('Url').length > 0) {

                var url = event.originalEvent.dataTransfer.getData('Url');
                if (ImageUtil.hasUrlImageExtension(url)) {
                    docView.enterBusy();
                    ImageUtil.insertURL(url).then(insertImage).always(function () { docView.leaveBusy(); });
                } else {
                    yellUnsupported();
                }
            }
        }

        /**
         * Sets the browser focus to the clipboard node, and selects all its
         * contents.
         */
        function grabClipboardFocus() {

            var // the browser selection
                selection = window.getSelection(),
                // a browser selection range object
                docRange = null;

            // calls the native focus() method at the clipboard node
            function grabNativeFocus() {
                clipboardFocusMethod.call(clipboardNode[0]);
            }

            // bug 29948: prevent grabbing to clipboard node while in cell in-place edit mode
            if (docView.isCellEditMode()) { return; }

            // on touch devices, selecting text nodes in a content-editable node will
            // always show the virtual keyboard
            // TODO: need to find a way to provide clipboard support
            if (Modernizr.touch) {
                grabNativeFocus();
                return;
            }

            // Bug 26283: Browsers are picky how to correctly focus and select text
            // in a node that is content-editable: Chrome and IE will change the
            // scroll position of the editable node, if it has been focused
            // explicitly while not having an existing browser selection (or even
            // if the selection is not completely visible). Furthermore, the node
            // will be focused automatically when setting the browser selection.
            // Firefox, on the other hand, wants to have the focus already set
            // before the browser selection can be changed, otherwise it may throw
            // exceptions. Additionally, changing the browser selection does NOT
            // automatically set the focus into the editable node.
            if (_.browser.Firefox) {
                grabNativeFocus();
            }

            // Clear the old browser selection.
            // Bug 28515, bug 28711: IE fails to clear the selection (and to modify
            // it afterwards), if it currently points to a DOM node that is not
            // visible anymore (e.g. the 'Show/hide side panel' button). Workaround
            // is to move focus to an editable DOM node which will cause IE to update
            // the browser selection object. The target container node cannot be used
            // for that, see comments above for bug 26283. Using another focusable
            // node (e.g. the body element) is not sufficient either. Interestingly,
            // even using the (editable) clipboard node does not work here. Setting
            // the new browser selection below will move the browser focus back to
            // the application pane.
            Utils.clearBrowserSelection();

            // set the browser selection
            try {
                docRange = window.document.createRange();
                docRange.setStart(clipboardNode[0], 0);
                docRange.setEnd(clipboardNode[0], clipboardNode[0].childNodes.length);
                selection.addRange(docRange);
            } catch (ex) {
                Utils.error('GridPane.grabClipboardFocus(): failed to select clipboard node: ' + ex);
            }
        }

        /**
         * Handles 'cut' events of the clipboard node.
         * Invokes the copyHandler, then deletes the values and clears
         * the attributes of the selected cells.
         */
        function cutHandler(event) {

            var // the clipboard event data
                clipboardData = event && event.originalEvent && event.originalEvent.clipboardData;

            // check for cell protection
            if (!docView.requireEditableSelection()) {
                event.preventDefault();
                return false;
            }

            // invoke clipboard copy handler first
            if (copyHandler(event) === false) { return false; }

            if (clipboardData) {
                // prevent default cut handling for desktop browsers, but not for touch devices
                if (!Modernizr.touch) {
                    event.preventDefault();
                }

                // delete values and clear attributes of the selected cells
                docView.clearCellRanges();

            } else {

                // delete values and clear attributes of the selected cells
                self.executeDelayed(function () { docView.clearCellRanges(); });
            }
        }

        /**
         * Handles 'copy' events of the clipboard node.
         * Creates a HTML table of the active range of the current selection
         * and generates the client clipboard id. Adds both to the system clipboard.
         * And invokes the Calcengine copy operation.
         */
        function copyHandler(event) {

            var // model of the active sheet
                sheetModel = docView.getSheetModel(),
                // current selection in the active sheet
                selection = docView.getSelection(),
                // the active cell range in the selection
                activeRange = selection.ranges[selection.activeRange],
                // the clipboard event data
                clipboardData = event && event.originalEvent && event.originalEvent.clipboardData,
                // the client clipboard id to identify the copy data when pasting
                clientClipboardId = docView.createClientClipboardId(),
                // the html table string to export
                htmlTable,
                // the plain text to export
                plainText;

            // restrict the number of cells to copy at the same time
            if (SheetUtils.getCellCount(activeRange) > SheetUtils.MAX_FILL_CELL_COUNT) {
                docView.yell({ type: 'info', message: gt('It is not possible to copy more than %1$d cells at the same time.', _.noI18n(SheetUtils.MAX_FILL_CELL_COUNT)) });
                event.preventDefault();
                return false;
            }

            // generate HTML table and plain text representation from the active range of the current selection
            htmlTable = Clipboard.getHTMLStringFromRange(app, sheetModel, activeRange, clientClipboardId);
            plainText = Clipboard.getPlainTextFromRange(sheetModel, activeRange);

            // initiate clipboard server copy call
            docView.clipboardServerCopy(selection);
            Utils.log('clipboard - copy - client id: ' + clientClipboardId);

            // set clipboard debug pane content
            self.trigger('debug:clipboard', htmlTable);

            // if browser supports clipboard api add data to the event
            if (clipboardData) {
                // add plain text and html of the current browser selection
                clipboardData.setData('text/plain', plainText);
                clipboardData.setData('text/html', htmlTable);

                // prevent default copy handling for desktop browsers, but not for touch devices
                if (!Modernizr.touch) {
                    event.preventDefault();
                }
            } else {
                // add Excel XML name spaces for custom tag support
                $('html').attr({
                    'xmlns': 'http://www.w3.org/TR/REC-html40',
                    'xmlns:o': 'urn:schemas-microsoft-com:office:office',
                    'xmlns:x': 'urn:schemas-microsoft-com:office:excel'
                });

                // add content and focus
                app.destroyImageNodes(clipboardNode);
                clipboardNode.empty().append(htmlTable);
                grabClipboardFocus();

                // remove XML name spaces again
                self.executeDelayed(function () {
                    $('html').removeAttr('xmlns xmlns:o xmlns:x');
                });
            }
        }

        /**
         * Handles 'paste' events of the clipboard node.
         */
        function pasteHandler(event) {

            var // current selection in the active sheet
                selection = docView.getSelection(),
                // the clipboard event data
                clipboardData = event && event.originalEvent && event.originalEvent.clipboardData,
                // the cleaned up html data
                htmlData,
                // the html data attached to the event
                htmlRawData,
                // the plain text data attached to the event
                textData;


            function pasteHtmlClipboard(html) {
                var // the client clipboard id to identify the copy data when pasting
                    clientClipboardId = Clipboard.getClientClipboardId(html),
                    // the server clipboard id to identify the copy data when pasting
                    serverClipboardId = docView.getServerClipboardId(clientClipboardId);

                if (serverClipboardId) {
                    // set clipboard debug pane content
                    self.trigger('debug:clipboard', 'Server side paste with clipboard id: ' + serverClipboardId);
                    // server paste call
                    docView.clipboardServerPaste(selection, serverClipboardId);
                    Utils.log('clipboard - server side paste');
                } else {
                    // parse HTML
                    handleParserResult(Clipboard.parseHTMLData(html, documentStyles));
                    Utils.log('clipboard - client side html paste');
                }
            }

            function pasteTextClipboard(text) {
                // set clipboard debug pane content
                self.trigger('debug:clipboard', text);
                // parse text
                handleParserResult(Clipboard.parseTextData(text));
                Utils.log('clipboard - client side text paste');
            }

            function createRangeFromParserResult(contents) {
                var range = { start: _.clone(selection.activeCell), end: _.clone(selection.activeCell) };
                range.end[0] += ((contents.length > 0) && (contents[0].length > 0)) ? (contents[0].length - 1) : 0;
                range.end[1] += (contents.length > 0) ? (contents.length - 1) : 0;
                return range;
            }

            function handleParserResult(result) {

                var // the target range of the paste
                    pasteRange = createRangeFromParserResult(result.contents),
                    // the number of cells to paste
                    pasteCellCount = SheetUtils.getCellCount(pasteRange),
                    // the resulting promise
                    promise = null;

                // restrict the number of cells to paste at the same time, check for protected cells
                if (pasteCellCount > SheetUtils.MAX_FILL_CELL_COUNT) {
                    promise = $.Deferred().reject('paste:overflow');
                } else {
                    promise = docView.areRangesEditable(pasteRange, { error: 'paste:locked' });
                }

                // paste the contents contained in the passed result object
                promise.done(function () {

                    // create a single undo action for all affected operations
                    docModel.getUndoManager().enterUndoGroup(function () {

                        var // model of the active sheet
                            sheetModel = docView.getSheetModel(),
                            // the ranges to be merged before pasting
                            mergedRanges = [];

                        // unmerge cells if necessary
                        if ((pasteCellCount > 1) && sheetModel.getMergeCollection().rangesOverlapAnyMergedRange(pasteRange)) {
                            sheetModel.mergeRanges(pasteRange, 'unmerge');
                        }

                        // merge cells
                        _.each(result.mergedRanges, function (range) {
                            range = docModel.getCroppedMovedRange(range, selection.activeCell[0], selection.activeCell[1]);
                            if (range) { mergedRanges.push(range); }
                        });
                        if (mergedRanges.length > 0) {
                            sheetModel.mergeRanges(mergedRanges, 'merge');
                        }

                        // set cell contents
                        sheetModel.setCellContents(docView.getActiveCell(), result.contents, { parse: true });
                    });
                });

                // show alert messages on error
                docView.yellOnPromise(promise);
            }

            // check if we have edit rigths before pasting
            if (!docModel.getEditMode()) {
                event.preventDefault();
                app.rejectEditAttempt();
                return false;
            }

            htmlRawData = clipboardData && clipboardData.getData('text/html');
            htmlData = _.isString(htmlRawData) ? Utils.parseAndSanitizeHTML(htmlRawData) : $();

            // check if clipboard event data contains html
            if (Clipboard.containsHtmlTable(htmlData)) {
                // prevent default paste handling for desktop browsers, but not for touch devices
                if (!Modernizr.touch) {
                    event.preventDefault();
                }

                // set clipboard debug pane content
                self.trigger('debug:clipboard', htmlRawData);
                // render the html to apply css styles provided by Excel or Calc
                app.destroyImageNodes(clipboardNode);
                clipboardNode.empty().append(htmlData);
                // parse and create operations
                pasteHtmlClipboard(clipboardNode);
                // clear clipboard
                clipboardNode.text('\xa0');

                return false;
            }

            // focus and select the clipboard node
            grabClipboardFocus();

            // check if clipboard event data contains plain text
            // to be used in case the clipboard div contains no html table to parse
            textData = (_.browser.IE) ? window.clipboardData && window.clipboardData.getData('text') : clipboardData && clipboardData.getData('text/plain');

            // read and parse pasted data
            self.executeDelayed(function () {
                if (Clipboard.containsHtmlTable(clipboardNode)) {
                    // set clipboard debug pane content
                    self.trigger('debug:clipboard', clipboardNode);
                    pasteHtmlClipboard(clipboardNode);
                } else if (textData) {
                    pasteTextClipboard(textData);
                }
                // clear clipboard
                clipboardNode.text('\xa0');
            });
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the root DOM node of this grid pane.
         *
         * @returns {jQuery}
         *  The root node of this grid pane, as jQuery object.
         */
        this.getNode = function () {
            return rootNode;
        };

        /**
         * Returns the scrollable container node of all content layers in this
         * grid pane.
         *
         * @returns {jQuery}
         *  The scrollable container node of this grid pane, as jQuery object.
         */
        this.getScrollNode = function () {
            return scrollNode;
        };

        /**
         * Registers an event handler at the specified event source object that
         * will only be invoked when this grid pane is visible. See method
         * BaseObject.listenTo() for details about the parameters.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.listenToWhenVisible = function (source, type, handler) {
            this.listenTo(source, type, function () {
                if (layerRange) {
                    return handler.apply(self, arguments);
                }
            });
            return this;
        };

        /**
         * Returns whether the root node of this grid pane is currently focused
         * or contains a DOM node that is currently focused (e.g. the text area
         * node while edit mode is active).
         *
         * @returns {Boolean}
         *  Whether this grid pane is currently focused.
         */
        this.hasFocus = function () {
            return focusTargetNode.is(window.document.activeElement);
        };

        /**
         * Sets the browser focus into this grid pane.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.grabFocus = function () {
            focusTargetNode.focus();
            return this;
        };

        /**
         * Initializes the settings and layout of this grid pane.
         *
         * @param {Object} colSettings
         *  The view settings of the horizontal pane side (left or right).
         *  Supports the following properties:
         *  @param {Boolean} colSettings.visible
         *      Whether the column header pane is visible.
         *  @param {Number} colSettings.offset
         *      The absolute horizontal offset of the grid pane in the view
         *      root node, in pixels.
         *  @param {Number} colSettings.size
         *      The total outer width of the grid pane, in pixels.
         *  @param {Boolean} colSettings.frozen
         *      Whether the grid pane is frozen (not scrollable) horizontally.
         *  @param {Boolean} colSettings.showOppositeScroll
         *      Whether the vertical scroll bar will be visible or hidden
         *      outside the pane root node.
         *  @param {Number} colSettings.hiddenSize
         *      The total width of the hidden columns in front of the sheet in
         *      frozen view mode.
         *
         * @param {Object} rowSettings
         *  The view settings of the vertical pane side (top or bottom).
         *  Supports the following properties:
         *  @param {Boolean} rowSettings.visible
         *      Whether the row header pane is visible.
         *  @param {Number} rowSettings.offset
         *      The absolute vertical offset of the grid pane in the view root
         *      node, in pixels.
         *  @param {Number} rowSettings.size
         *      The total outer height of the grid pane, in pixels.
         *  @param {Boolean} rowSettings.frozen
         *      Whether the grid pane is frozen (not scrollable) vertically.
         *  @param {Boolean} rowSettings.showOppositeScroll
         *      Whether the horizontal scroll bar will be visible or hidden
         *      outside the pane root node.
         *  @param {Number} rowSettings.hiddenSize
         *      The total height of the hidden rows in front of the sheet in
         *      frozen view mode.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.initializePaneLayout = function (colSettings, rowSettings) {

            // initialize according to visibility
            if (colSettings.visible && rowSettings.visible) {

                // show pane root node and initialize auto-scrolling
                rootNode.show();
                layerRootNode.enableTracking(Utils.extendOptions(PaneUtils.DEFAULT_TRACKING_OPTIONS, {
                    autoScroll: colSettings.frozen ? (rowSettings.frozen ? false : 'vertical') : (rowSettings.frozen ? 'horizontal' : true),
                    borderNode: rootNode
                }));

                // position and size
                rootNode.css({ left: colSettings.offset, top: rowSettings.offset, width: colSettings.size, height: rowSettings.size });

                // initialize the scroll node containing the scroll bars
                scrollNode.css({
                    overflowX: colSettings.frozen ? 'hidden' : '',
                    overflowY: rowSettings.frozen ? 'hidden' : '',
                    bottom: (!colSettings.frozen && !rowSettings.showOppositeScroll) ? -Math.max(10, Utils.SCROLLBAR_HEIGHT) : 0,
                    right: (!rowSettings.frozen && !colSettings.showOppositeScroll) ? -Math.max(10, Utils.SCROLLBAR_WIDTH) : 0
                });

                // additional layout information for frozen splits
                hiddenWidth = colSettings.hiddenSize;
                hiddenHeight = rowSettings.hiddenSize;

                // update scroll area size annd CSS classes at root node for focus display
                updateScrollAreaSize();
                updateFocusDisplay();

            } else {

                // hide pane root node and deinitialize auto-scrolling
                rootNode.hide();
                layerRootNode.disableTracking();
            }

            return this;
        };

        /**
         * Returns whether this grid pane is currently visible (depending on
         * its position, and the view split settings of the active sheet).
         *
         * @returns {Boolean}
         *  Whether this grid pane is currently visible.
         */
        this.isVisible = function () {
            return _.isObject(layerRange);
        };

        /**
         * Returns the size of the leading hidden area that cannot be shown in
         * this grid pane in a frozen split view.
         *
         * @returns {Object}
         *  The size of the of the leading hidden area that cannot be shown in
         *  this grid pane, in the following properties:
         *  - {Number} width
         *      The width of the hidden area left of this grid pane. Will be 0,
         *      if the view does not contain a frozen split.
         *  - {Number} height
         *      The height of the hidden area left of this grid pane. Will be
         *      0, if the view does not contain a frozen split.
         */
        this.getHiddenSize = function () {
            return { width: hiddenWidth, height: hiddenHeight };
        };

        /**
         * Returns the address of the layer cell range.
         *
         * @returns {Object|Null}
         *  The address of the layer cell range, if this grid pane is visible;
         *  otherwise null.
         */
        this.getLayerRange = function () {
            return _.copy(layerRange, true);
        };

        /**
         * Returns the offset and size of the entire sheet rectangle covered by
         * the layer nodes of this grid pane, including the ranges around the
         * visible area.
         *
         * @returns {Object|Null}
         *  The offset and size of the sheet area covered by this grid pane, in
         *  pixels, if this grid pane is visible; otherwise null.
         */
        this.getLayerRectangle = function () {
            return _.copy(layerRectangle, true);
        };

        /**
         * Converts the passed absolute sheet rectangle to a rectangle relative
         * to the current layer nodes.
         *
         * @param {Object} rectangle
         *  An absolute sheet rectangle, in pixels.
         *
         * @returns {Object}
         *  The resulting layer rectangle, relative to the current layer nodes.
         */
        this.convertToLayerRectangle = function (rectangle) {
            return {
                left: rectangle.left - (layerRectangle ? layerRectangle.left : 0),
                top: rectangle.top - (layerRectangle ? layerRectangle.top : 0),
                width: rectangle.width,
                height: rectangle.height
            };
        };

        /**
         * Returns the CSS position and size attributes for the passed absolute
         * sheet rectangle in pixels, as HTML mark-up value for the 'style'
         * element attribute, adjusted to the current position of the layer
         * nodes in this grid pane.
         *
         * @param {Object} rectangle
         *  The absolute sheet rectangle, in pixels.
         *
         * @returns {String}
         *  The value for the 'style' element attribute in HTML mark-up text.
         */
        this.getLayerRectangleStyleMarkup = function (rectangle) {
            return PaneUtils.getRectangleStyleMarkup(this.convertToLayerRectangle(rectangle));
        };

        /**
         * Invokes the passed iterator function for all cell ranges that are
         * located inside the visible area of this grid pane. The iterator
         * function will receive additional data needed for rendering those
         * ranges in some way.
         *
         * @param {Object|Array}
         *  The address of a single cell range, or an array of cell range
         *  addresses.
         *
         * @param {Function} iterator
         *  The iterator function that will be invoked for each range that is
         *  inside the visible area of this grid pane. Receives
         *  the following parameters:
         *  (1) {Object} range
         *      The address of the cell range to be rendered.
         *  (2) {Number} index
         *      The array index of the current cell range.
         *  (3) {Object} rectangle
         *      The location of the current range, relative to the layer root
         *      node of this grid pane.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.alignToGrid=false]
         *      If set to true, the rectangles passed to the iterator callback
         *      function will be expanded by one pixel to the left and to the
         *      top, so that all its outer pixels are located on the grid lines
         *      surrounding the range.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateRangesForRendering = function (ranges, iterator, options) {

            // do nothing, if all columns or rows are hidden
            if (!this.isVisible()) { return; }

            var // the bounding rectangle for range nodes (prevent oversized DOM nodes)
                boundRectangle = { width: Utils.MAX_NODE_SIZE, height: Utils.MAX_NODE_SIZE },
                // whether to enlarge rectangle by one pixel to the left and top, to place it on grid lines
                alignToGrid = Utils.getBooleanOption(options, 'alignToGrid', false);

            // get start position of the bounding rectangle for range nodes, to prevent oversized DOM nodes
            boundRectangle.left = Math.max(0, layerRectangle.left - (boundRectangle.width - layerRectangle.width) / 2);
            boundRectangle.top = Math.max(0, layerRectangle.top - (boundRectangle.height - layerRectangle.height) / 2);

            // restrict to ranges in the current cell area to prevent oversized DOM nodes
            // that would collapse to zero size, e.g. when entire columns or rows are selected
            // (for details, see the comments for the Utils.MAX_NODE_SIZE constant)
            return Utils.iterateArray(_.getArray(ranges), function (range, index) {

                var // position and size of the selection range, restricted to the bounding rectangle
                    rectangle = Utils.getIntersectionRectangle(docView.getRangeRectangle(range), boundRectangle);

                // skip ranges located completely outside the bounding rectangle
                if (!rectangle) { return; }

                // enlarge rectangle to match the grid lines between the cells
                if (alignToGrid) {
                    enlargeLeft(rectangle, 1);
                    enlargeTop(rectangle, 1);
                }

                return iterator.call(self, range, index, self.convertToLayerRectangle(rectangle));
            });
        };

        /**
         * Returns the effective offset and size of the sheet area that is
         * really visible in this grid pane.
         *
         * @returns {Object}
         *  The offset and size of the sheet area visible in this grid pane, in
         *  pixels.
         */
        this.getVisibleRectangle = function () {
            return {
                left: colHeaderPane.getVisiblePosition().offset,
                top: rowHeaderPane.getVisiblePosition().offset,
                width: scrollNode[0].clientWidth,
                height: scrollNode[0].clientHeight
            };
        };

        /**
         * Returns the address of the top-left cell visible in this grid pane.
         * The cell is considered visible, it its column and row are at least
         * half visible.
         *
         * @returns {Number[]}
         *  The address of the top-left cell in this grid pane.
         */
        this.getTopLeftAddress = function () {
            return [colHeaderPane.getFirstVisibleIndex(), rowHeaderPane.getFirstVisibleIndex()];
        };

        /**
         * Changes the scroll position of this grid pane relative to the
         * current scroll position.
         *
         * @param {Number} leftDiff
         *  The difference for the horizontal scroll position, in pixels.
         *
         * @param {Number} topDiff
         *  The difference for the vertical scroll position, in pixels.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollRelative = function (leftDiff, topDiff) {
            colHeaderPane.scrollRelative(leftDiff);
            rowHeaderPane.scrollRelative(topDiff);
            return this;
        };

        /**
         * Scrolls this grid pane to make the specified cell visible.
         *
         * @param {Number[]} address
         *  The cell position to be made visible.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.forceTop=false]
         *      If set to true, the cell will always be scrolled into the upper
         *      border of the grid pane, also if it is already visible.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollToCell = function (address, options) {
            var position = docView.getCellRectangle(address, { expandMerged: true }),
                forceTop = Utils.getBooleanOption(options, 'forceTop', false);
            colHeaderPane.scrollToPosition(position.left, position.width);
            rowHeaderPane.scrollToPosition(position.top, position.height, { forceLeading: forceTop });
            return this;
        };

        /**
         * Scrolls this grid pane to make the specified DOM node visible.
         *
         * @param {HTMLElement|jQuery} node
         *  The DOM node to be made visible. If this object is a jQuery
         *  collection, uses the first node it contains. Must be a descendant
         *  node of the layer root node.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollToNode = function (node) {

            var // the position of the node in the layer root node
                position = Utils.getChildNodePositionInNode(layerRootNode, node);

            colHeaderPane.scrollToPosition(position.left + layerRectangle.left, position.width);
            rowHeaderPane.scrollToPosition(position.top + layerRectangle.top, position.height);
            return this;
        };

        /**
         * Scrolls this grid pane to make the specified drawing frame visible.
         *
         * @param {Number[]} position
         *  The document position of the drawing frame.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollToDrawingFrame = function (position) {
            var drawingFrame = this.getDrawingFrame(position);
            if (drawingFrame.length > 0) { this.scrollToNode(drawingFrame); }
            return this;
        };

        // in-place cell edit mode --------------------------------------------

        /**
         * Prepares the cell in-place edit mode for this grid pane.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.initializeCellEditMode = function () {
            focusTargetNode = rootNode.find('textarea').focus();
            clipboardNode.detach();
            return this;
        };

        /**
         * Cleans up after cell in-place edit mode has ended in this grid pane.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.cleanupCellEditMode = function () {
            rootNode.append(clipboardNode);
            focusTargetNode = clipboardNode;
            grabClipboardFocus();
            return this;
        };

        // initialization -----------------------------------------------------

        // marker for touch devices and browser types
        Utils.addDeviceMarkers(rootNode);

        // initialize class members
        app.onInit(function () {

            // document model, and other collections of the document
            docModel = app.getModel();
            documentStyles = docModel.getDocumentStyles();

            // the spreadsheet view, and other view object
            docView = app.getView();
            colHeaderPane = docView.getColHeaderPane(panePos);
            rowHeaderPane = docView.getRowHeaderPane(panePos);

            // the clipboard node that carries the browser focus for copy/paste events
            clipboardNode = docView.createClipboardNode().attr('tabindex', 1).text('\xa0').appendTo(rootNode);
            // by default, focus the clipboard node
            focusTargetNode = clipboardNode;

            // handle DOM focus requests at the clipboard node (select clipboard contents)
            clipboardFocusMethod = clipboardNode[0].focus;
            clipboardNode[0].focus = grabClipboardFocus;

            // on touch devices, disable the content-editable mode of the clipboard node,
            // to prevent showing the virtual keyboard all the time
            if (Modernizr.touch) { clipboardNode.removeAttr('contenteditable'); }

            // update grid pane when document edit mode changes
            self.listenToWhenVisible(docModel, 'change:editmode', editModeHandler);

            // listen to events from the view
            self.listenTo(docView, 'celledit:enter', cellEditEnterHandler);
            self.listenTo(docView, 'celledit:leave', cellEditLeaveHandler);

            // update DOM after interval changes of the column and row header pane
            self.listenTo(colHeaderPane, 'change:interval', function (event, interval, refresh, options) { updateLayerInterval(interval, true, refresh, options); });
            self.listenTo(colHeaderPane, 'hide:interval', function () { updateLayerInterval(null, true); });
            self.listenTo(rowHeaderPane, 'change:interval', function (event, interval, refresh, options) { updateLayerInterval(interval, false, refresh, options); });
            self.listenTo(rowHeaderPane, 'hide:interval', function () { updateLayerInterval(null, false); });

            // update selection after the column, row, or cell geometry has changed
            self.listenToWhenVisible(docView, 'insert:columns delete:columns change:columns insert:rows delete:rows change:rows insert:merged delete:merged insert:table change:table delete:table', changeCellLayoutHandler);

            // update layers according to changed view attributes (only, if this grid pane is visible)
            self.listenToWhenVisible(docModel, 'change:viewattributes', changeViewAttributesHandler);
            self.listenToWhenVisible(docView, 'change:sheet:viewattributes', changeSheetViewAttributesHandler);

            // listen to changed scroll position/size from header panes
            self.listenToWhenVisible(colHeaderPane, 'change:scrollsize', updateScrollAreaSize);
            self.listenToWhenVisible(colHeaderPane, 'change:scrollpos', updateScrollPosition);
            self.listenToWhenVisible(rowHeaderPane, 'change:scrollsize', updateScrollAreaSize);
            self.listenToWhenVisible(rowHeaderPane, 'change:scrollpos', updateScrollPosition);

            // listen to DOM scroll events
            scrollNode.on('scroll', scrollHandler);

            // activate this pane on focus change
            rootNode.on('focusin', function () {
                // do not change active pane while in cell edit mode (range selection mode in formulas)
                if (!docView.isCellEditMode()) {
                    docView.setSheetViewAttribute('activePane', panePos);
                }
            });

            // handle key and drop events
            rootNode.on({ keydown: keyDownHandler, drop: dropHandler });

            // handle system clipboard events
            clipboardNode.on({ cut: cutHandler, copy: copyHandler, paste: pasteHandler });

            // show/hide a registered menu on click of a cell drop-down button,
            // suppress double-clicks on drop-down buttons
            formLayerNode.on({ click: cellButtonClickHandler, dblclick: false }, '.cell-dropdown-button');

            // immediately hide the active pop-up menu on selection tracking
            self.on('select:start', function () { hideActiveCellMenu(); });

            // drop-down menu for table columns (sorting and filtering)
            registerCellMenu('table', new TableColumnMenu(app));

            // drop-down menu for data validation of active cell
            registerCellMenu('validation', new ValidationListMenu(app));
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {

            clipboardNode[0].focus = clipboardFocusMethod;
            _.invoke(cellMenuRegistry.menus, 'destroy');

            app.destroyImageNodes(clipboardNode);

            docModel = documentStyles = null;
            docView = colHeaderPane = rowHeaderPane = null;
            rootNode = scrollNode = scrollSizeNode = layerRootNode = null;
            cellLayerNode = remoteLayerNode = selectionLayerNode = null;
            formLayerNode = drawingLayerNode = highlightLayerNode = activeRangeLayerNode = null;
            clipboardNode = clipboardFocusMethod = null;
            cellMenuRegistry = activeCellButton = activeCellMenu = null;
        });

    } // class GridPane

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: GridPane });

});
