/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/control/activesheetlist', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/control/radiolist',
    'gettext!io.ox/office/spreadsheet'
], function (Utils, RadioList, gt) {

    'use strict';

    // class ActiveSheetList ==================================================

    /**
     * The selector for the current active sheet, as radio drop-down list.
     *
     * @constructor
     *
     * @extends RadioList
     */
    function ActiveSheetList(app, initOptions) {

        var // self reference
            self = this,

            // whether to show the sheet name as label text
            showNames = Utils.getBooleanOption(initOptions, 'showNames', false);

        // base constructor ---------------------------------------------------

        RadioList.call(this, {
            icon: showNames ? null : 'fa-bars',
            iconClasses: showNames ? 'small-icon' : null,
            tooltip: gt('Select sheet'),
            caret: showNames,
            updateCaptionMode: showNames ? 'label icon' : 'none'
        });

        // private methods ----------------------------------------------------

        /**
         * Recreates all sheet entries in the drop-down menu.
         */
        var fillList = this.createDebouncedMethod($.noop, function () {
            self.clearMenu();
            app.getView().iterateVisibleSheets(function (sheetModel, index, sheetName) {
                self.createOptionButton(index, {
                    icon: sheetModel.isLocked() ? 'fa-lock' : null,
                    iconClasses: 'small-icon',
                    label: _.noI18n(sheetName),
                    labelStyle: 'white-space:pre;',
                    tooltip: _.noI18n(sheetName)
                });
            });
        });

        // initialization -----------------------------------------------------

        // registering handler at the view
        app.onInit(function () {
            self.listenTo(app.getView(), 'change:sheets', fillList);
        });

        this.listenTo(app.getImportPromise(), 'done', function () {
            fillList();
            self.listenTo(app.getModel(), 'change:sheet:attributes', fillList); // e.g. sheet protection
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = initOptions = self = null;
        });

    } // class ActiveSheetList

    // exports ================================================================

    // derive this class from class RadioList
    return RadioList.extend({ constructor: ActiveSheetList });

});
