/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/numberformatter', [
    'io.ox/core/date',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/io',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (CoreDate, Utils, IO, BaseObject, SheetUtils) {

    'use strict';

    var // the exponent character
        EXP = 'E',

        // zero characters, used for formatting numbers with leading zeros
        ZEROS = '0000000000000000',

        // number of milliseconds per day
        MSEC_PER_DAY = 86400000;

    // class ParsedSection ====================================================

    /**
     * Contains the parsed information of a single section in a format code.
     * Sections will be separated with semicolons in the format code and can be
     * used to define different number formats for specific intervals of
     * numbers, e.g. for positive and negative numbers.
     *
     * @constructor
     */
    function ParsedSection() {

        /**
         * An array containing the format code tokens of this code section.
         */
        this.tokens = [];

        /**
         * The number of percent tokens contained in the code section.
         */
        this.percent = 0;

        /**
         * Whether the format code contains any date/time tokens.
         */
        this.dateTime = false;

        /**
         * Whether the format code contains an AM/PM token that switches the
         * time to 12-hours format.
         */
        this.isAmPm = false;

        /**
         * Scientific numbers set exponent on true,
         * so mantissa and exponent will be calculated before formating
         */
        this.expo = false;

        /**
         * if the format is a fraction or a significant number,
         * its integer part must not be rounded
         */
        this.notInteger = false;

        /**
         * if this section is only text
         */
        this.onlyText = false;

    } // class ParsedSection

    // methods ----------------------------------------------------------------

    /**
     * Parses the first section of the passed format code snippet, until
     * reaching the next semicolon character, or to the end of the string. All
     * parsed format tokens will be appended to the 'tokens' property of this
     * instance, and other information about the code section will be updated.
     *
     * @param {String} formatCode
     *  The format code snippet to be parsed.
     *
     * @returns {String}
     *  The remainder of the passed format code following the semicolon
     *  character that separates different code sections.
     */
    ParsedSection.prototype.parseSection = function (formatCode) {

        var // self reference
            self = this,
            // the matches of the regular expressions
            matches = null;

        // add a new token to the current section
        function pushToken(token) {
            self.tokens.push(token);
        }

        // add a new date/time token
        function pushDateTimeToken(type, length, total) {
            self.tokens.push({ type: type, length: length, total: !!total });
            self.dateTime = true;
        }

        // add a new string literal token, or extends the last existing token
        function pushLiteralToken(text) {
            pushCommonToken(text, 'lit');
        }

        function pushNumberToken(text) {
            pushCommonToken(text, 'nr');
        }

        function pushFractionToken(text) {
            pushCommonToken(text, 'frac');
            self.notInteger = true;
        }

        function pushSignToken(text) {
            pushCommonToken(text, 'sign');
            self.notInteger = true;
        }

        function pushScienceToken(text) {
            pushCommonToken(text.substring(2, text.length), 'scien');
            self.expo = true;
        }

        function pushCommonToken(text, type) {
            var token = _.last(self.tokens);
            if (_.isObject(token) && (token.type === type)) {
                token.text += text;
            } else {
                self.tokens.push({ type: type, text: text });
            }
        }

        //flag if last DateTime was an hour or not
        var lastWasHour = false;

        // stop if the entire format code has been consumed
        while (formatCode.length > 0) {

            // try/finally to be able to continue the while-loop at any place
            try {

                // section separator: break while-loop (finally block will remove the semicolon)
                if ((matches = /^;/.exec(formatCode))) {
                    break;
                }

                // 'General' format token
                if ((matches = /^GENERAL/i.exec(formatCode))) {
                    pushToken({ type: 'general' });
                    continue;
                }

                // date/time: AM/PM marker
                if ((matches = /^AM\/PM/i.exec(formatCode))) {
                    pushToken({ type: 'ampm' });
                    this.dateTime = this.isAmPm = true;
                    continue;
                }

                // date/time category: year
                if ((matches = /^Y+/i.exec(formatCode))) {
                    pushDateTimeToken('year', Math.min(4, Utils.roundUp(matches[0].length, 2))); // only YY and YYYY allowed
                    lastWasHour = false;
                    continue;
                }

                // date/time category: month
                if ((matches = /^M+/.exec(formatCode))) {
                    pushDateTimeToken('month', Math.min(5, matches[0].length)); // M to MMMMM allowed
                    lastWasHour = false;
                    continue;
                }

                // date/time category: day/weekday
                if ((matches = /^D+/i.exec(formatCode))) {
                    pushDateTimeToken('day', Math.min(4, matches[0].length)); // D to DDDD allowed
                    lastWasHour = false;
                    continue;
                }

                // date/time category: total number of hours
                if ((matches = /^\[(h+)\]/i.exec(formatCode))) {
                    pushDateTimeToken('hour', Math.min(2, matches[1].length), true); // [h] or [hh] allowed
                    lastWasHour = true;
                    continue;
                }

                // date/time category: total number of minutes
                if ((matches = /^\[(m+)\]/.exec(formatCode))) {
                    pushDateTimeToken('minute', Math.min(2, matches[1].length), true); // [m] or [mm] allowed
                    continue;
                }

                // date/time category: total number of seconds
                if ((matches = /^\[(s+)\]/i.exec(formatCode))) {
                    pushDateTimeToken('second', Math.min(2, matches[1].length), true); // [s] or [ss] allowed
                    lastWasHour = false;
                    continue;
                }

                // date/time category: hour
                if ((matches = /^h+/i.exec(formatCode))) {
                    pushDateTimeToken('hour', Math.min(2, matches[0].length)); // h or hh allowed
                    lastWasHour = true;
                    continue;
                }

                // date/time category: minute together with seconds
                if ((matches = /^(m+)\W+s/i.exec(formatCode))) {
                    pushDateTimeToken('minute', Math.min(2, matches[1].length)); // m or mm allowed
                    matches[0] = matches[1]; //small hack because we only want the minutes
                    lastWasHour = false;
                    continue;
                }

                // date/time category: minute
                if ((matches = /^m+/.exec(formatCode))) {
                    if(lastWasHour){
                        pushDateTimeToken('minute', Math.min(2, matches[0].length)); // m or mm allowed
                    }else{
                        pushDateTimeToken('month', Math.min(5, matches[0].length)); // M to MMMMM allowed
                    }
                    lastWasHour = false;
                    continue;
                }

                // date/time category: second
                if ((matches = /^s+/i.exec(formatCode))) {
                    pushDateTimeToken('second', Math.min(matches[0].length, 2)); // s or ss allowed
                    lastWasHour = false;
                    continue;
                }

                // percent category
                if ((matches = /^%/.exec(formatCode))) {
                    pushLiteralToken('%');
                    this.percent += 1;
                    continue;
                }

                // string literal, enclosed in double-quotes
                if ((matches = /^"([^"]*)"/.exec(formatCode))) {
                    pushLiteralToken(matches[1]);
                    continue;
                }

                // fraction format token
                if ((matches = /^[0-9?#]+\/[0-9?#]+/i.exec(formatCode))) {
                    pushFractionToken(matches[0]);
                    continue;
                }

                // significant format token
                if ((matches = /^\.([0-9?#]+)/i.exec(formatCode))) {
                    pushSignToken(matches[0]);
                    continue;
                }

                // nr category
                if ((matches = /^([0-9#,?])/.exec(formatCode))) {
                    pushNumberToken(matches[0]);
                    continue;
                }
                if ((matches =  /^(E-|E\+[0-9?#]+)/i.exec(formatCode))) {
                    pushScienceToken(matches[0]);
                    continue;
                }

                // whitespace
                if ((matches = /^_./.exec(formatCode))) {
                    pushLiteralToken(' ');
                    continue;
                }

                // repeated fill character
                if ((matches = /^\*./.exec(formatCode))) {
                    // bug 36439: not supported by CalcEngine, ignore silently
                    continue;
                }

                // character literal, preceded by backspace
                if ((matches = /^\\(.)/.exec(formatCode))) {
                    pushLiteralToken(matches[1]);
                    continue;
                }

                // currency
                if ((matches = /^\[\$(.+?)-\d+\]/.exec(formatCode))) {
                    pushLiteralToken(matches[1]);
                    continue;
                }

                // text
                if ((matches = /^@/.exec(formatCode))) {
                    pushLiteralToken('');
                    self.onlyText = true;
                    continue;
                }

                // ignore them
                // TODO: [RED] colors and *, what should we do with that?
                if ((matches = /^(\[.*\])/.exec(formatCode))) {
                    pushLiteralToken('');
                    continue;
                }

                // any non-matching token is considered a character literal
                matches = [formatCode[0]];
                pushLiteralToken(matches[0]);
            }
            // remove the current match from the formula string
            finally {
                formatCode = formatCode.slice(matches[0].length);
            }
        }

        // return the remaining format code
        return formatCode;
    };

    // class NumberFormatter ==================================================

    function NumberFormatter(app) {

        var // self reference
            self = this,

            // the font collection used to calculate text widths
            fontCollection = null,

            // cache for parsed code sections, mapped by format code
            parsedCodesCache = {},

            // all predefined number format codes, mapped by category
            categoryCodesDatabase = {},

            // all built-in number format codes, mapped by identifier
            builtInFormats = {},

            // all built-in number format codes, mapped by format code
            invBuiltInFormats = {},

            // the standard format code
            standardCode = 'General',

            // all format codes for in-place edit mode
            editCodesDatabase = { date: 'DD/MM/YYYY', time: 'hh:mm:ss' },

            // default currency symbol
            currencySymbol = '$',

            // localized decimal separator
            decimalSymbol = null,

            // localized group separator
            groupSymbol = null,

            // the localized Boolean literals
            falseLiteral = null,
            trueLiteral = null,

            // lazy initialized helper parsing the date
            unformatDate = null,

            // lazy initialized helper parsing the time
            unformatTime = null,

            // null date (corresponding to cell value zero) TODO: use null date from document settings
            nullDate = Date.UTC(1899, 11, 30),

            // whether to support negative date values (e.g. number -1 is one day before null date)
            negativeDates = false;

        // base constructor ---------------------------------------------------

        BaseObject.call(this);

        // private methods ----------------------------------------------------

        /**
         * Parses the specified format code, and returns a descriptor with
         * arrays of code tokens, separated into several sections.
         *
         * @param {String} formatCode
         *  The format code to be parsed.
         *
         * @returns {Array|Null}
         *   An array representing the parsed code sections contained in the
         *   passed format code. Each array element is an instance of the class
         *   ParsedSection.
         */
        function parseFormatCode(formatCode) {

            // first check whether the code has been parsed already
            if (formatCode in parsedCodesCache) {
                return parsedCodesCache[formatCode];
            }

            var // all format code sections, as array
                parsedCode = parsedCodesCache[formatCode] = [];

            while (formatCode.length > 0) {
                parsedCode.push(new ParsedSection());
                formatCode = _.last(parsedCode).parseSection(formatCode);
            }

            // return the parsed format code
            return parsedCode;
        }

        /**
         * Returns the decimal string representation of the passed number.
         *
         * @constructor
         *
         * @extends BaseObject
         *
         * @param {Number} number
         *  The number whose decimal string representation will be returned.
         *
         * @param {Number} digits
         *  The number of digits after the decimal separator the passed number
         *  will be rounded to.
         *
         * @returns {Object}
         *  A result object with the following properties:
         *  - {Number} number
         *      The resulting rounded number.
         *  - {String} text
         *      The complete decimal string representation of the number.
         *  - {String} abs
         *      The decimal string representation of the absolute number.
         *  - {Number} digits
         *      The number of significant decimal places in the integral and
         *      fractional part in the string representation of the rounded
         *      number.
         */
        function formatAsDecimal(number, digits) {

            var // the absolute number, rounded to the specified digit count after decimal separator
                round = Utils.round(Math.abs(number), Math.pow(10, -digits)),
                // the integral part of the absolute number
                int = Math.floor(round),
                // the fractional part of the absolute number
                frac = Math.round((round - int) * Math.pow(10, digits)),
                // the text representation of the integral part
                intText = String(int),
                // the text representation of the significant fractional part (without leading zeros)
                fracText = String(frac),
                // the zeros between decimal separator and significant fractional part
                fillText = null,
                // the result returned by this method
                result = { abs: intText, digits: (int === 0) ? 0 : intText.length };

            // add the rounded number to the result
            result.number = (number < 0) ? -round : round;

            // no fraction available: do not add decimal separator
            if (frac > 0) {

                // leading zeros for the fractional part
                fillText = Utils.repeatString('0', digits - fracText.length);
                // remove trailing zeros
                fracText = fracText.replace(/0+$/, '');
                // concatenate integer and fractional part, remove trailing zeros
                result.abs += decimalSymbol + fillText + fracText;
                // update number of significant digits
                if (int > 0) { result.digits += fillText.length; }
                result.digits += fracText.length;
            }

            // add final string representation (with sign)
            result.text = (number < 0) ? ('-' + result.abs) : result.abs;

            return result;
        }

        /**
         * Returns the scientific string representation of the passed number.
         *
         * @param {Number} mant
         *  The mantissa of the number whose scientific string representation
         *  will be returned. Must be in the half-open intervals +-[1, 10).
         *
         * @param {Number} exp
         *  The exponent of the number whose scientific string representation
         *  will be returned.
         *
         * @param {Number} digits
         *  The number of digits after the decimal separator the passed
         *  mantissa will be rounded to.
         *
         * @returns {Object}
         *  A result object with the following properties:
         *  - {Number} number
         *      The resulting rounded number.
         *  - {String} text
         *      The complete scientific string representation of the number.
         *  - {String} abs
         *      The scientific string representation of the absolute number.
         *  - {Number} digits
         *      The number of significant decimal places in the integral and
         *      fractional part in the string representation of the rounded
         *      mantissa.
         */
        function formatAsScientific(mant, exp, digits) {

            var // get the formatted mantissa
                result = formatAsDecimal(mant, digits);

            // if mantissa has been rounded up to 10, increase the exponent
            if (Math.abs(result.number) === 10) {
                result.number /= 10;
                result.abs = '1';
                exp += 1;
            }

            // update the rounded number in the result
            result.number *= Math.pow(10, exp);

            // add the exponent (at least two digits, but may be three digits)
            result.abs += EXP + ((exp < 0) ? '-' : '+');
            exp = Math.abs(exp);
            if (exp < 10) { result.abs += '0'; }
            result.abs += String(exp);

            // add final string representation (with sign)
            result.text = (mant < 0) ? ('-' + result.abs) : result.abs;

            return result;
        }

        /**
         * Formats a number with increasing number of digits following the
         * decimal separator, as long as the resulting number fits according to
         * the provided result validator.
         *
         * @param {Number} minDigits
         *  The minimum number of digits after the decimal separator.
         *
         * @param {Number} maxDigits
         *  The maximum number of digits after the decimal separator.
         *
         * @param {Function} formatCallback
         *  The callback function that receives the current number of digits
         *  following the decimal separator, and must return a result object
         *  containing the properties 'number', 'text', 'abs', and 'digits'.
         *  See methods formatAsDecimal() and formatAsScientific() above for a
         *  detailed description about this result object.
         *
         * @param {Function} validateCallback
         *  The callback function that checks the current result object it
         *  receives. Must return a Boolean value specifying whether the result
         *  object is valid.
         *
         * @returns {Object}
         *  The last matching result object returned by the callback function.
         */
        function findLongestFormat(minDigits, maxDigits, formatCallback, validateCallback) {

            var currDigits = minDigits,
                currResult = null,
                lastResult = null;

            do {
                currResult = formatCallback(currDigits);
                if (!validateCallback(currResult)) { break; }
                lastResult = currResult;
                currDigits += 1;
            } while (currDigits <= maxDigits);

            return lastResult;
        }

        /**
         * Reduce a fraction by finding the Greatest Common Divisor and dividing by it.
         * @see http://stackoverflow.com/questions/4652468/is-there-a-javascript-function-that-reduces-a-fraction
         */
        function cancelFract(frac){
            var num = frac.num;
            var deno = frac.deno;

            var gcd = function gcd(a,b){
              return b ? gcd(b, a%b) : a;
            };
            gcd = gcd(num,deno);

            frac.num = num / gcd;
            frac.deno = deno / gcd;
        }

        /**
         * fills up the "text" number with whitespace to the length of the assigned code
         * iterate the code and the text and does:
         *      code: ? text: '0' -> ' '
         *      code: # text: '0' -> delete
         *      code: 0 text: ' ' -> '0'
         *      code: 0-9 -> break iteration
         *
         * @param {Boolean} rightToLeft (default=false)
         *      if true the text is iterated from right to left (only used in significant part of the number)
         *
         */
        function setPlaceHolders(text, code, rightToLeft) {

            if(_.isNumber(text)){
                text = String(text);
            }
            if(text.length>code.length){
                return text;
            }

            var i, counter, end;
            if(rightToLeft){
                text = text + Utils.repeatString(' ', code.length - text.length);
                end=-1; counter=-1; i=text.length-1;
            }else{
                text = Utils.repeatString(' ', code.length - text.length) + text;
                i=0; counter=+1; end=text.length;
            }

            var arrT = [];
            for (var j = 0; j < text.length; j++) {
                arrT.push(text[j]);
            }

            for (; i !== end; i += counter) {
                var actT = arrT[i];
                var actC = code[i];

                if(actT !== '0' && actT !== ' '){
                    break;
                }
                if(actC === '0'){
                    arrT[i] = '0';
                }else{
                    if(actC === '?'){
                        arrT[i] = ' ';
                    }else if(actC === '#'){
                        arrT[i] = '';
                    }
                }
            }
            return arrT.join('');
        }

        /**
         * create helper object for converting in-place-formated values back to cell-format
         * takes the editCodesDatabase' date & time
         * parses the special chars between the alphanumercis ('mm.dd.yy' -> '.') and create a regexp for "x.x.x" & "x.x" formats
         * gets the index of the chars in the format ('mm.dd.yy -> m=0 d=1 y=2)
         */
        function initUnformatData() {
            function prepareUnformat(source/*, key0, key1, key2*/) {
                var args = arguments;
                var splitFinder = /\w+((?=[\S])\W)\w+/;
                var splitSymbol = splitFinder.exec(source);

                if(splitSymbol){
                    var result = {};

                    result.splitSymbol = splitSymbol[1];
                    result.finder = new RegExp('\\d+\\' + result.splitSymbol + '\\d+\\' + result.splitSymbol + '\\d+|\\d+\\' + result.splitSymbol + '\\d+');

                    var possKeys = source.split(new RegExp('\\' + result.splitSymbol, 'g'));
                    _.each(possKeys, function(value, index) {
                        for (var i = 1; i < args.length; i++) {
                            var argKey = args[i];
                            if(value[0].toLowerCase() === argKey){
                                result[argKey] = index;
                                break;
                            }
                        }
                    });
                    return result;
                }
            }

            unformatDate = prepareUnformat(editCodesDatabase.date, 'd', 'm', 'y');
            unformatTime = prepareUnformat(editCodesDatabase.time, 'h', 'm', 's');

        }

        // public methods -----------------------------------------------------

        /**
         * Returns all available number format codes for the specified format
         * category and current UI language.
         *
         * @param {String} category
         *  The identifier of the number format category.
         *
         * @returns {Array|Null}
         *  The format codes for the specified category, or null if no format
         *  codes have been defined for the category. Each element in the
         *  returned array is an object with the following properties:
         *  - {String} value
         *      The actual format code.
         *  - {String} label
         *      A caption label with a formatted example number, intended to be
         *      used in GUI form controls.
         *  - {Boolean} [red=false]
         *      If set to true, the format code contains a [RED] color modifier
         *      for negative numbers.
         */
        this.getCategoryCodes = function (category) {
            var categoryFormats = categoryCodesDatabase[category.toLocaleLowerCase()];
            return _.isArray(categoryFormats) ? _.copy(categoryFormats, true) : null;
        };

        /**
         * Returns the default number format code for the specified format
         * category and current UI language.
         *
         * @param {String} category
         *  The identifier of the number format category.
         *
         * @returns {String|Null}
         *  The default format code for the specified category, or null if no
         *  format code has been found.
         */
        this.getCategoryDefaultCode = function (category) {
            var categoryCodes = categoryCodesDatabase[category.toLocaleLowerCase()];
            return (_.isArray(categoryCodes) && _.isObject(categoryCodes[0])) ? categoryCodes[0].value : null;
        };

        /**
         * Returns the standard format code.
         *
         * @returns {String}
         *  The standard format code.
         */
        this.getStandardCode = function () {
            return standardCode;
        };

        /**
         * Returns the predefined number format code for the specified format
         * code identifier.
         *
         * @param {Number} formatId
         *  The format code identifier.
         *
         * @returns {String|Null}
         *  The number format code for the specified identifier if existing,
         *  otherwise null.
         */
        this.getBuiltInFormatCode = function (formatId) {
            return (formatId in builtInFormats) ? builtInFormats[formatId] : null;
        };

        /**
         * Returns the predefined number format code for the specified format
         * code identifier.
         *
         * @param {String} formatCode
         *  The number format code to be converted to a built-in identifier.
         *
         * @returns {Number|Null}
         *  The built-in identifier of the passed format code if available,
         *  otherwise null.
         */
        this.getBuiltInFormatId = function (formatCode) {
            return (formatCode in invBuiltInFormats) ? invBuiltInFormats[formatCode] : null;
        };

        /**
         * Returns the format code of the passed value of a 'numberFormat'
         * formatting attribute.
         *
         * @param {Object} numberFormat
         *  The value of a 'numberFormat' formatting attribute. Must contain at
         *  least one of the properties 'id' (numeric identifier of a built-in
         *  format code), and 'code' (explicit format code, as string). If an
         *  identifier and an explicit format code exists, the format code will
         *  be ignored, and the built-in format code of the identifier will be
         *  returned.
         *
         * @returns {String}
         *  The effective format code of the passed attribute value.
         */
        this.resolveFormatCode = function (numberFormat) {

            var // the built-in identifier
                id = Utils.getIntegerOption(numberFormat, 'id'),
                // explicit format code
                code = Utils.getStringOption(numberFormat, 'code');

            // prefer built-in identifier over format code
            if (_.isNumber(id) && (id in builtInFormats)) { return builtInFormats[id]; }

            // return existing explicit format code
            if (_.isString(code) && (code.length > 0)) { return code; }

            Utils.warn('NumberFormatter.resolveFormatCode(): invalid number format: ' + JSON.stringify(numberFormat));
            return standardCode;
        };

        /**
         * Converts the passed format code to a value that can be used as
         * 'numberFormat' formatting attribute. Tries to resolve the identifier
         * of a built-in number format if possible.
         *
         * @param {String} formatCode
         *  A format code.
         *
         * @returns {Object}
         *  A number format attribute value. The object will contain the
         *  property 'id', if the format code represents a built-in number
         *  format; otherwise the property 'code' with the passed format code.
         */
        this.createNumberFormat = function (formatCode) {
            // replace format codes of built-in number formats with identifier
            var id = this.getBuiltInFormatId(formatCode);
            return _.isNumber(id) ? { id: id } : { code: formatCode };
        };

        /**
         * Returns the default currency symbol for the current UI language.
         *
         * @returns {String}
         *  The default currency symbol for the current UI language.
         */
        this.getCurrencySymbol = function () {
            return currencySymbol;
        };

        /**
         * Returns whether the passed date object is valid to be processed by
         * the number formatter. The year must be in the range from 0 to 9999,
         * and must not be located before the null date, unless such dates are
         * supported by the document.
         *
         * @param {Date} date
         *  The date to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed date object is valid.
         */
        this.isValidDate = function (date) {

            var // milliseconds from 1970-01-01
                time = date.getTime(),
                // the year
                year = date.getUTCFullYear();

            // check the year, check for 'negative' dates
            return isFinite(time) && (negativeDates || (nullDate <= time)) && (0 <= year) && (year <= 9999);
        };

        /**
         * Returns a date object representing the passed floating-point number,
         * using the current null date of the document.
         *
         * @param {Number} number
         *  A floating point number. The integral part represents the number of
         *  days elapsed since the document's null date, the fractional part
         *  represents the time.
         *
         * @returns {Date|Null}
         *  An UTC date object representing the passed number; or null, if the
         *  passed number cannot be converted to a valid date (e.g. too large).
         */
        this.convertNumberToDate = function (number) {

            var // get correct date/time for current null date
                date = new Date(nullDate + Math.round(number * MSEC_PER_DAY));

            // return the date object if it is valid
            return this.isValidDate(date) ? date : null;
        };

        /**
         * Returns the floating-point number representing the passed date,
         * using the current null date of the document.
         *
         * @param {Date} date
         *  An UTC date object.
         *
         * @returns {Number|Null}
         *  The floating-point number representing the passed date; or null, if
         *  the passed date cannot be converted to a number.
         */
        this.convertDateToNumber = function (date) {

            // error, if date is invalid, or the year is not in the range 0 to 9999
            if (!this.isValidDate(date)) { return null; }

            // return the floating-point number
            return (date.getTime() - nullDate) / MSEC_PER_DAY;
        };

        /**
         * Tries to parse a floating-point number from the leading part of the
         * passed string.
         *
         * @param {String} text
         *  The text containing a floating-point number.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.sep]
         *      If specified, a custom decimal separator character. By default,
         *      the decimal separator of the current UI language will be used.
         *  @param {Boolean} [options.sign=false]
         *      If set to true, the passed string may contain a leading sign
         *      character ('-' or '+').
         *  @param {Boolean} [options.complete=false]
         *      If set to true, the entire passed string must represent a valid
         *      floating-point number. No remaining 'garbage' text will be
         *      accepted.
         *
         * @returns {Object|Null}
         *  A result descriptor, if the passed string starts with a valid
         *  floating-point number (according to the passed options); otherwise
         *  null. The result descriptor contains the following properties:
         *  - {Number} number
         *      The parsed floating-point number.
         *  - {String} text
         *      The original text from the passed string that has been parsed
         *      to the floating-point number.
         *  - {String} sign
         *      The sign character. Will be the empty string, if no sign exists
         *      in the passed text, or if the option 'sign' has not been set
         *      (see above).
         *  - {String} remaining
         *      The remaining characters from the passed string following the
         *      parsed floating-point number. Will be the empty string, if the
         *      option 'complete' has been set (see above).
         */
        this.parseNumberFromString = (function () {

            // TODO: parse other formats: date/time, percent, currency, etc.

            // returns the regular expression used to match a floating-point number
            // Group 1: the leading sign (may be an empty string)
            // Group 2: the floating-point number (decimal or scientific), without sign
            var getNumberRegExp = _.memoize(function (sep) {
                sep = _.escapeRegExp(sep);
                return new RegExp('^([-+]?)((?:\\d*' + sep + '\\d+|\\d+' + sep + '?)(?:E[-+]?\\d+)?)', 'i');
            });

            // the actual implementation of the parseNumberFromString() method
            return function parseNumberFromString(text, options) {

                var // the decimal separator used to parse the number
                    sep = Utils.getStringOption(options, 'sep', decimalSymbol),
                    // whether to accept a leading sign
                    sign = Utils.getBooleanOption(options, 'sign', false),
                    // whether to reject trailing garbage
                    complete = Utils.getBooleanOption(options, 'complete', false),
                    // try to parse the passed string for a leading floating-point number
                    matches = getNumberRegExp(sep).exec(text),
                    // the parsed floating-point number
                    number = null;

                // reject leading sign unless specified in the passed options
                if (!matches || (!sign && matches[1].length > 0)) { return null; }

                // reject trailing garbage if specified in the passed options
                if (complete && (matches[0].length < text.length)) { return null; }

                // parse the folating-point number
                number = parseFloat(matches[2].replace(decimalSymbol, '.'));
                if (!_.isFinite(number)) { return null; }
                if (matches[1] === '-') { number = -number; }

                // build the result descriptor
                return { number: number, text: matches[0], sign: matches[1], remaining: text.substr(matches[0].length) };
            };
        }());

        /**
         * Converts the passed text to a floating-point number, if possible.
         *
         * @param {String} text
         *  The string to be converted to a floating-point number.
         *
         * @returns {Number}
         *  The floating-point number represented by the passed string; or NaN,
         *  if the string cannot be parsed to a number.
         */
        this.convertStringToNumber = function (text) {
            var parseResult = this.parseNumberFromString(text, { sign: true, complete: true });
            return _.isObject(parseResult) ? parseResult.number : Number.NaN;
        };

        /**
         * Tries to convert the passed text to a number, Boolean, or error code
         * literal.
         *
         * @param {String} text
         *  The string to be converted to another data type.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.keepApos=false]
         *      If set to true, a leading apostrophe in the passed text (used
         *      to protect a string from conversion to other data types) will
         *      be kept in the result. by default, the result will be a string
         *      without the leading aposthrophe.
         *
         * @returns {Number|Boolean|String|ErrorCode}
         *  The resulting converted value (the original string, if the passed
         *  text cannot be converted to anyother data type).
         */
        this.parseString = function (text, options) {

            // try to parse Boolean literals
            if (text.toUpperCase() === falseLiteral) { return false; }
            if (text.toUpperCase() === trueLiteral) { return true; }

            // try to parse error code literals
            if (app.isLocalizedErrorCode(text)) {
                return app.convertStringToErrorCode(text);
            }

            // try to convert passed text to a floating-point number
            var number = this.convertStringToNumber(text);
            if (_.isFinite(number)) { return number; }

            // everything else is a regular string
            return Utils.getBooleanOption(options, 'keepApos', false) ? text : text.replace(/^'/, '');
        };

        /**
         * Returns the display string of the passed number formatted with the
         * 'General' number format, fitting into the specified text width.
         *
         * @param {Number} number
         *  The number whose string representation in 'General' number format
         *  will be returned.
         *
         * @param {Number} maxLength
         *  The maximum number of characters allowed for the absolute part of
         *  the passed number, including the decimal separator and the complete
         *  exponent in scientific notation.
         *
         * @param {Object} [attributes]
         *  Character formatting attributes influencing the text width. If
         *  omitted, the result object returned by this method will not contain
         *  the effective pixel width of the formatted number. MUST be
         *  specified, if the parameter 'maxWidth' (see below) is set.
         *  @param {String} attributes.fontName
         *      The name of the original font family (case-insensitive).
         *  @param {Number} attributes.fontSize
         *      The font size, in points.
         *  @param {Boolean} [attributes.bold=false]
         *      Whether the text will be rendered in bold characters.
         *  @param {Boolean} [attributes.italic=false]
         *      Whether the text will be rendered in italic characters.
         *
         * @param {Number} [maxWidth]
         *  The maximum width of the resulting string representation, in
         *  pixels. If omitted, the number will be formatted without a pixel
         *  width restriction (but still with the restriction to a maximum
         *  number of text characters specified in the 'maxLength' parameter).
         *
         * @returns {String|Null}
         *  The resulting display text of the number, formatted with the
         *  'General' number format. If a passed maximum pixel width has been
         *  passed to this method, and if it is too small to take the passed
         *  number in either decimal or scientific notation, the null value
         *  will be returned instead.
         */
        this.formatStandardNumber = function (number, maxLength, attributes, maxWidth) {

            // Bug 31125: This method uses a local cache containing the widths of all
            // characters that will appear in a number formatted with the 'General'
            // number format, and will add up these widths to calculate the total
            // width of a formatted number. This works twice as fast as simply using
            // the method FontCollection.getTextWidth() that inserts the text into a
            // DOM element and reads its resulting width. A very little drawback is
            // that the result might not be correct to the last pixel, as it misses
            // advanced font features such as kerning, but this should not be a
            // problem for numbers as hardly any font uses kerning to render digits.

            var // exponent (will be -INF for value zero)
                exp = 0,
                // mantissa in the half-open intervals +-[1, 10) (will be NaN for zero)
                mant = 0,
                // whether to calculate text widths in pixels
                hasAttributes = _.isObject(attributes),
                // whether to restrict the result to a pixel width
                useMaxWidth = hasAttributes && _.isNumber(maxWidth),
                // cached widths of all characters that can appear in the display string
                charWidths = hasAttributes ? getCharacterWidths() : null,
                // the maximum pixel width of a digit
                digitWidth = useMaxWidth ? Math.max(1, fontCollection.getDigitWidth(attributes)) : null,
                // the pixel width of the decimal separator character
                decWidth = useMaxWidth ? charWidths[decimalSymbol] : null,
                // the pixel width of the exponent with two digits
                exp2Width = useMaxWidth ? (charWidths[EXP] + charWidths['+'] + 2 * digitWidth) : null,
                // available, minimum, and maximum number of digits for the passed width
                availDigits = 0, minDigits = 0, maxDigits = 0,
                // the formatter callback function (decimal or scientific)
                formatterFunc = null,
                // the validator callback function (depending on pixel mode)
                validateFunc = null,
                // conversion result in decimal notation
                decResult = null,
                // conversion result in scientific notation
                scientResult = null,
                // the result to be used (either decimal or scientific)
                finalResult = null;

            if (number !== 0) {
                exp = Math.floor(Math.log(Math.abs(number)) / Math.LN10);
                mant = number / Math.pow(10, exp);
            }

            // returns an object with the widths of all characters that can appear in a result
            function getCharacterWidths() {
                return fontCollection.getCustomFontMetrics(attributes, function () {
                    return _.reduce('0123456789+-' + EXP + decimalSymbol, function (memo, char) {
                        memo[char] = fontCollection.getTextWidth(char, attributes);
                        return memo;
                    }, {});
                }, 'standard.widths');
            }

            // returns the width of the passed formatted number
            function getTextWidth(text) {
                var textWidth = _.reduce(text, function (memo, char) { return memo + charWidths[char]; }, 0);
                if (_.isNumber(textWidth)) { return textWidth; }
                Utils.warn('NumberFormatter.getTextWidth(): unexpected character in formatted number: ' + text);
                return fontCollection.getTextWidth(text, attributes);
            }

            // validates the passed result by character count only
            function validateLength(result) {
                return result.abs.length <= maxLength;
            }

            // validates the passed result by character count and pixel width
            function validateLengthAndWidth(result) {
                return (result.abs.length <= maxLength) && (getTextWidth(result.text) <= maxWidth);
            }

            // set the validator callback function, depending on pixel mode
            validateFunc = useMaxWidth ? validateLengthAndWidth : validateLength;

            // find the best decimal representation
            if (exp < 2 - maxLength) {
                // start with zero representation for numbers equal to or very close to zero
                decResult = { number: number, abs: '0', text: (number < 0) ? '-0' : '0', digits: 0 };
                if (!validateFunc(decResult)) { decResult = null; }
            } else if (exp <= maxLength - 1) {
                // try to find the best explicit representation in a specific interval of exponents
                availDigits = useMaxWidth ? Math.floor((maxWidth - decWidth) / digitWidth) : maxLength;
                minDigits = Math.max(0, -(exp + 2));
                maxDigits = Math.min(15, Math.max(0, 14 - exp), maxLength - Math.max(2, exp), availDigits - Math.max(0, exp) - 1);
                formatterFunc = _.partial(formatAsDecimal, number);
                decResult = findLongestFormat(minDigits, maxDigits, formatterFunc, validateFunc);
            }

            // find the best scientific representation (exclude zero)
            if (_.isFinite(exp) && ((exp <= -4) || (exp >= 4))) {
                availDigits = maxLength - 5;
                if (useMaxWidth) { availDigits = Math.min(availDigits, Math.floor((maxWidth - decWidth - exp2Width) / digitWidth)); }
                minDigits = Utils.minMax(availDigits - 2, 0, 13);
                maxDigits = Utils.minMax(availDigits - 1, 0, 14);
                formatterFunc = _.partial(formatAsScientific, mant, exp);
                scientResult = findLongestFormat(minDigits, maxDigits, formatterFunc, validateFunc);
            }

            // prefer decimal notation, if it shows at least as much significant digits as the scientific notation
            finalResult = (decResult && (!scientResult || (decResult.digits >= scientResult.digits))) ? decResult : scientResult;
            return finalResult ? finalResult.text : null;
        };

        /**
         * Returns the passed cell value, formatted with the specified number
         * format.
         *
         * @param {Number|String|Boolean|ErrorCode|Null} value
         *  The typed cell value to be formatted.
         *
         * @param {String} formatCode
         *  The format code used to format the passed cell value.
         *
         * @returns {String|Null}
         *  The formatted value; or null, if the passed format code is invalid,
         *  or if the value cannot be formatted with the format code (for
         *  example, a number is too large to be formatted as date/time).
         */
        this.formatValue = function (value, formatCode) {

            var // the parsed format code, either from cache or freshly parsed
                parsedCode = parseFormatCode(formatCode),
                // the section of the format code to be used for the passed value
                parsedSection = null,
                // the formatted string representation
                result = '',
                // the passed value, as Date object
                date = null,
                // default exponent is 0, so original value cant be changed, if not explicit wanted
                exp = 0;

            // if resulting text is shorter, fills with leading zeros
            function fillZeros(int, length) {
                return (ZEROS + int).substr(-Math.max(String(int).length, length));
            }

            // if resulting text is shorter, fills with leading zeros; if longer, truncates to length
            function fillZerosOrTruncate(int, length) {
                return (ZEROS + int).substr(-length);
            }

            // len 3,4 & 5 are localized names
            function getMonth(month, len) {
                switch (len) {
                    case 3: return CoreDate.locale.monthsShort[month].replace('.', '');
                    case 4: return CoreDate.locale.months[month];
                    case 5: return CoreDate.locale.monthsShort[month].substring(0,1);
                    default: return fillZeros(month + 1, len);
                }
            }

            // len 3 & 4 are localized names
            function getWeekDay(day, date, len) {
                switch (len) {
                    case 3: return CoreDate.locale.daysShort[day];
                    case 4: return CoreDate.locale.days[day];
                    default: {
                        return fillZeros(date, len);
                    }
                }
            }

            //format of the integer part of the number
            function formatNumber(token) {

                //standard group seperation! if the integer-format has a ',' inside every thee numbers comes the groupsymbol
                function handleGroupSeparator(nrInfo, format) {
                    if(format.indexOf(',')<0){return;}

                    var replacer = '$1' + groupSymbol + '$2';
                    var rgx = /(\d+)(\d{3})/;
                    while (rgx.test(nrInfo.text)) {
                        nrInfo.text = nrInfo.text.replace(rgx, replacer);
                    }
                    if (nrInfo.number < 0) {
                        nrInfo.text = '-' + nrInfo.text;
                    }
                }

                if(parsedSection.dateTime){
                    return token.text;
                }

                var result = {number: value, abs: String(Math.abs(value))};

                if(parsedSection.notInteger){
                    result.text = String(parseInt(value));
                }else{
                    result.text = String(Math.round(value));
                }

                handleGroupSeparator(result, token.text);

                result.text = setPlaceHolders(result.text, token.text.replace(/\,/g, ''));

                return result.text;
            }

            //format of the fraction part numerator & denominator
            //at the moment we only support untill 3signs for the denominator
            function formatFraction(token) {
                var parts = token.text.split('/');

                var denomLen = parts[1].length;
                if(denomLen>3){
                    parts[1] = parts[1].substring(0,3);
                    denomLen = 3;
                }
                var fractStart = parseInt('1' +  Utils.repeatString('0', denomLen - 1));
                var fractEnd = parseInt(Utils.repeatString('9', denomLen));

                var abs = Math.abs(value);
                var frac = abs - parseInt(abs);

                var closest = null;
                var near = Number.POSITIVE_INFINITY;

                if(frac===0){
                    closest = {num: 0, deno: fractEnd};
                }else{
                    for (var i = fractStart; i <= fractEnd; i++) {
                        var num = Math.round(frac * i);
                        var compare = num / i;
                        var dif = Math.abs(frac - compare);
                        if (dif < near) {
                            closest = {num: num, deno: i};
                            near = dif;
                            if(dif===0){break;}
                        }
                    }
                }

                if(closest){
                    if(parts[1].indexOf('?')>=0){
                        if(frac===0){
                            //placeholder complete fraction only white spaces!!!
                            return setPlaceHolders('', parts[0]) + ' ' + setPlaceHolders('', parts[1]);
                        }
                        cancelFract(closest);
                    }
                    closest.num = setPlaceHolders(closest.num, parts[0]);
                    closest.deno = setPlaceHolders(closest.deno, parts[1]);

                    return closest.num + '/' + closest.deno;
                }
                return decimalSymbol + frac;
            }

            //format the significant part of the number incl. decimal-symbol
            function formatSign(token) {
                var format = token.text.substring(1, token.text.length);
                var digits = format.length;

                var round = Utils.round(Math.abs(value), Math.pow(10, -digits));
                var abs = Math.abs(round);
                var frac = String(abs - parseInt(abs));

                var result = frac.substring(2, Math.min(digits + 2, frac.length));
                var res = setPlaceHolders(result, format, true);
                if(res.length === 0){
                    return '';
                }else if(res[0] === ' ') {
                    return ' ' + res;
                }else{
                    return decimalSymbol + res;
                }

            }

            function formatScience(token) {
                var res = setPlaceHolders(Math.abs(exp), token.text);
                return EXP + ((exp < 0) ? '-' : '+') + res;
            }

            // bail out if the passed format code is invalid
            if (!_.isArray(parsedCode) || (parsedCode.length === 0)) {
                return null;
            }

            // error codes will never be formatted
            if (SheetUtils.isErrorCode(value)) {
                return app.convertErrorCodeToString(value);
            }

            // TODO: currently, only numbers are supported
            if (!_.isNumber(value) || !isFinite(value)) {
                return null;
            }

            parsedSection = parsedCode[0];
            if(value === 0 && parsedCode.length > 2 && !parsedCode[2].onlyText){
                parsedSection = parsedCode[2];
            }else if(value < 0 && parsedCode.length > 1 && !parsedCode[1].onlyText){
                parsedSection = parsedCode[1];
                value = Math.abs(value);
            }

            // correct value for percentage formats
            if (parsedSection.percent > 0) {
                value *= Math.pow(100, parsedSection.percent);
                if (!isFinite(value)) { return null; }
            }

            // additional preparations for date/time formats
            if ((parsedSection.dateTime) && !(date = this.convertNumberToDate(value))) {
                return null;
            }

            if(parsedSection.expo){
                if (value !== 0) {
                    exp = Math.floor(Math.log(Math.abs(value)) / Math.LN10);
                    value = value / Math.pow(10, exp);
                    if (Math.abs(value) === 10) {
                        value /= 10;
                        exp += 1;
                    }
                }
            }

            // process all format code tokens
            _.each(parsedSection.tokens, function (token) {
                switch (token.type) {
                case 'general':
                    result += self.formatStandardNumber(value, SheetUtils.MAX_LENGTH_STANDARD_CELL);
                    break;
                case 'lit':
                    result += token.text;
                    break;
                case 'nr':
                    result += formatNumber(token);
                    break;
                case 'frac':
                    result += formatFraction(token);
                    break;
                case 'sign':
                    result += formatSign(token);
                    break;
                case 'scien':
                    result += formatScience(token);
                    break;
                case 'year':
                    result += fillZerosOrTruncate(date.getUTCFullYear(), token.length);
                    break;
                case 'month':
                    result += getMonth(date.getUTCMonth(), token.length);
                    break;
                case 'day':
                    result += getWeekDay(date.getUTCDay(), date.getUTCDate(), token.length);
                    break;
                case 'hour':
                    result += fillZeros(token.total ? Math.floor(value * 24) : parsedSection.isAmPm ? (((date.getUTCHours() + 11) % 12) + 1) : date.getUTCHours(), token.length);
                    break;
                case 'minute':
                    result += fillZeros(token.total ? Math.floor(value * 1440) : date.getUTCMinutes(), token.length);
                    break;
                case 'second':
                    result += fillZeros(token.total ? Math.round(value * 86400) : date.getUTCSeconds(), token.length);
                    break;
                case 'ampm':
                    result += CoreDate.locale.dayPeriods[(date.getUTCHours() < 12) ? 'am' : 'pm'];
                    break;
                default:
                    Utils.warn('NumberFormatter.formatValue(): unsupported token type "' + token.type + '"');
                }
            });
            return result;
        };

        /**
         * Returns the display string of the passed cell value, formatted with
         * the appropriate number format for the specified format code
         * category, intended for in-place cell edit mode.
         *
         * @param {Number|String|Boolean|ErrorCode|Null} value
         *  The typed cell value to be formatted for in-place cell edit mode.
         *
         * @param {String} category
         *  The number format category the passed value will be formatted to.
         *
         * @returns {String}
         *  The resulting display text of the cell value, formatted with the
         *  appropriate number format for the specified category.
         */
        this.formatValueForEditMode = function (value, category) {

            // strings: use plain unformatted string for editing (bug 34421: add an aposthrophe if necessary)
            if (_.isString(value)) {
                return ((/^['=]/).test(value) || app.isLocalizedErrorCode(value) || _.isFinite(this.convertStringToNumber(value))) ? ('\'' + value) : value;
            }

            // Booleans: use plain Boolean literal for editing
            if (_.isBoolean(value)) {
                return app.getBooleanLiteral(value);
            }

            // error codes: use plain error code literal for editing
            if (SheetUtils.isErrorCode(value)) {
                return app.convertErrorCodeToString(value);
            }

            // numbers: use appropriate number representation according to number format category
            if (_.isNumber(value) && isFinite(value)) {

                var // the resulting formatted value
                    result = null;

                switch (category) {

                // percent: multiply by 100, add percent sign without whitespace
                case 'percent':
                    value *= 100; // may become infinite
                    result = (isFinite(value) ? this.formatStandardNumber(value, SheetUtils.MAX_LENGTH_STANDARD_EDIT) : '') + '%';
                    break;

                // automatically show date and/or time, according to the number
                case 'date':
                case 'time':
                case 'datetime':

                    var // number of milliseconds
                        milliSecs = Math.round(value * MSEC_PER_DAY),
                        // number of days
                        date = Math.floor(milliSecs / MSEC_PER_DAY),
                        // number of milliseconds in the day
                        time = Math.floor(milliSecs % MSEC_PER_DAY),
                        // whether to add the date and time part to the result
                        showDate = (category !== 'time') || (date !== 0),
                        showTime = (category !== 'date') || (time !== 0),
                        // the resulting format code
                        formatCode = (showDate ? editCodesDatabase.date : '') + ((showDate && showTime) ? '\\ ' : '') + (showTime ? editCodesDatabase.time : '');

                    // the resulting formatted value (may be null for invalid dates)
                    result = this.formatValue(value, formatCode);
                    break;
                }

                // use 'General' number format for all other format codes
                return _.isString(result) ? result : this.formatStandardNumber(value, SheetUtils.MAX_LENGTH_STANDARD_EDIT);
            }

            // empty cells
            return '';
        };

        /**
         * converts the assigned formated numbers to the source-value
         * with help of the unformat-data & the assigned category
         */
        this.unformatEditString = function(value, category) {
            switch (category) {

            case 'percent':

                if(_.last(value) === '%'){
                    value = parseFloat(value) / 100;
                }else{
                    value = parseFloat(value);
                }
                break;
            case 'date':
            case 'time':
            case 'datetime':

                if(unformatDate){
                    var date = new Date();

                    var dateInfo = unformatDate.finder.exec(value);
                    if(dateInfo){
                        dateInfo = dateInfo[0].split(/\W/g);

                        date.setUTCDate     (parseInt(dateInfo[unformatDate.d]));
                        date.setUTCMonth    (parseInt(dateInfo[unformatDate.m]) - 1);
                        date.setUTCFullYear (parseInt(dateInfo[unformatDate.y]));
                    }

                    var timeInfo = unformatTime.finder.exec(value);
                    if(timeInfo){
                        timeInfo = timeInfo[0].split(/\W/g);

                        date.setUTCSeconds  (parseInt(timeInfo[unformatTime.s]));
                        date.setUTCMinutes  (parseInt(timeInfo[unformatTime.m]));
                        date.setUTCHours    (parseInt(timeInfo[unformatTime.h]));
                    }

                    value =  self.convertDateToNumber(date);
                }

                break;
            }
            return value;
        };

        /**
         * returns the display String for the cell, by assigned value & numberformat in the attributes
         */
        this.formatCellContent = function(value, attrs) {

            var cell = attrs.cell,
                formatCode = (cell && cell.numberFormat) ? this.resolveFormatCode(cell.numberFormat) : standardCode;

            var res = self.formatValue(value, formatCode);
            return _.isNull(res) ? String(value) : res;
        };

        // initialization -----------------------------------------------------

        // initialize class members
        app.onInit(function () {
            fontCollection = app.getModel().getFontCollection();
        });

        // initialize locale-dependent settings
        app.onInitLocale(function () {
            decimalSymbol = app.getDecimalSeparator();
            groupSymbol = app.getGroupSeparator();
            falseLiteral = app.getBooleanLiteral(false);
            trueLiteral = app.getBooleanLiteral(true);
        });

        // get localized predefined format codes
        this.listenTo(IO.loadResource('io.ox/office/spreadsheet/resource/numberformats'), 'done', function (data) {
            _.extend(categoryCodesDatabase, data.categories);
            _.extend(editCodesDatabase, data.edit);
            _.extend(builtInFormats, data.builtInFormats);
            invBuiltInFormats = Utils.mapProperties(_.invert(builtInFormats), function (id) { return parseInt(id, 10); });
            standardCode = self.getCategoryDefaultCode('standard') || standardCode;
            currencySymbol = Utils.getStringOption(data, 'currencySymbol', currencySymbol, true);
            initUnformatData();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            fontCollection = categoryCodesDatabase = editCodesDatabase = null;
        });

    } // class NumberFormatter

    // exports ================================================================

    return NumberFormatter;

});
