/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/operators', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/tokenutils',
    'io.ox/office/spreadsheet/model/formula/complex',
    'io.ox/office/spreadsheet/model/formula/matrix',
    'io.ox/office/spreadsheet/model/formula/reference',
    'io.ox/office/spreadsheet/model/formula/besselfuncs'
], function (Utils, SheetUtils, TokenUtils, Complex, Matrix, Reference, Bessel) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all unary and binary operators, and all functions
     * supported by the client-side formula interpreter.
     *
     * Each descriptor is mapped by the name of the native operator or function
     * name as used in document operations (not the translated function names).
     *
     * Each descriptor supports the following properties:
     *
     * @property {String} [formats]
     *  The identifiers of all file formats that support the operator or
     *  function, as space-separated list of string tokens. If omitted, all
     *  known file formats will support the respective operator or function.
     *
     * @property {String} [altNames]
     *  One or more alternative names for a function, separated by whitespace.
     *  Using one of these function names in a formula will invoke the same
     *  function resolver as it does for the original function name.
     *
     * @property {Number} minParams
     *  Minimum number of parameters supported by the operator or function.
     *  Must be 1 or 2 for operators.
     *
     * @property {Number} [maxParams]
     *  Maximum number of parameters supported by the operator or function.
     *  Must not be less than 'minParams'. Must be 1 or 2 for operators. For
     *  functions, can be omitted to specify that the function accepts the
     *  maximum number of function parameters supported by the current file
     *  format (last parameters can be repeated, see property 'repeatParams'
     *  for details).
     *
     * @property {Number} [repeatParams=1]
     *  If the function accepts as many parameters as supported by the current
     *  file format (property 'maxParams' has been omitted), this property
     *  specifies the length of a single sequence of parameters that will be
     *  repeated after the minimum number of supported parameters (see property
     *  'minParams'). Default value is 1 (the same parameter is repeated, e.g.
     *  function SUM). For example, if this property is set to 2, the function
     *  expects pairs of repeating parameters following the minimum number of
     *  parameters (e.g. the function SUMIFS which expects pairs of parameters
     *  following the first parameter).
     *
     * @property {String} type
     *  The default type of the return value of the operator or function. The
     *  following return types are supported:
     *  - 'val'
     *      A single value (number, string, Boolean value, error code, date, or
     *      complex number). May cause repeated invocation of the operator or
     *      function, if called in a matrix type context. Example: in the
     *      formula =MDETERM(ABS(A1:B2)), the 'val'-type function ABS will be
     *      called once for each cell in the range, because function MDETERM
     *      expects a matrix as parameter.
     *  - 'mat':
     *      A matrix constant, even if it has size 1x1.
     *  - 'ref':
     *      An unresolved cell reference (instance of class Reference). Error
     *      code literals will be accepted too.
     *  - 'any':
     *      The operator or function can return values of any of the previous
     *      type. The outer context of the operator or function will cause the
     *      appropriate type conversion.
     *
     * @property {String} [signature='']
     *  The type signature of the operator or function, as space-separated list
     *  of tokens specifying the expected data type of the respective operands.
     *  Can be omitted for functions without parameters. Will be ignored for
     *  unimplemented operators or functions (missing 'resolve' property, see
     *  below). The following data types are supported:
     *  - 'val':
     *      A single literal value of type number, string, or Boolean. The
     *      special value null represents an empty function parameter, e.g. in
     *      the formula =SUM(1,,2). If the parameter is a constant matrix, its
     *      top-left element will be used. If the parameter is a cell range
     *      reference, it will be resolved to the content value of the cell
     *      related to the reference cell of the formula. If the resulting
     *      value is an error code, it will be thrown as exception value,
     *      without resolving the operator or function.
     *  - 'val:num':
     *      Similar to type 'val', but converts the parameter to a floating-
     *      point number (see method FormulaContext.convertToNumber() for
     *      details). If this conversion fails, the operator or function will
     *      result in a #VALUE! error.
     *  - 'val:int':
     *      Similar to type 'val:num', but converts the parameter to an integer
     *      by rounding down after conversion to a floating-point number (see
     *      method FormulaContext.convertToNumber() for details). Negative
     *      numbers will be rounded down too (NOT towards zero). If this
     *      conversion fails, the operator or function will result in a #VALUE!
     *      error.
     *  - 'val:date':
     *      Similar to type 'val', but converts the parameter to a Date object
     *      (see method FormulaContext.convertToDate() for details). If this
     *      conversion fails, the operator or function will result in a #VALUE!
     *      error.
     *  - 'val:str':
     *      Similar to type 'val', but converts the parameter to a string (see
     *      method FormulaContext.convertToString() for details). If this
     *      conversion fails, the operator or function will result in a #VALUE!
     *      error.
     *  - 'val:bool':
     *      Similar to type 'val', but converts the parameter to a Boolean
     *      value (see method FormulaContext.convertToBoolean() for details).
     *      If this conversion fails, the operator or function will result in a
     *      #VALUE! error.
     *  - 'val:comp':
     *      Similar to type 'val', but converts the parameter to a complex
     *      number (see method FormulaContext.convertToComplex() for details).
     *      If this conversion fails, the operator or function will result in a
     *      #NUM! error.
     *  - 'val:any':
     *      Similar to type 'val', but accepts error codes, and passes them as
     *      value to the resolver function.
     *  - 'mat':
     *      A two-dimensional matrix of literal values available in formulas
     *      (numbers, strings, Boolean values), as instance of the class
     *      Matrix. If the matrix contains one or more error codes, the first
     *      will be thrown as exception value, without resolving the operator
     *      or function. The maximum size of a constant matrix is restricted.
     *      Too large matrixes will cause to throw an UNSUPPORTED_ERROR error
     *      code. If the parameter is a single literal value, it will be
     *      converted to a 1x1 matrix (except for an error code which will be
     *      thrown). If the parameter is a cell range reference, it will be
     *      resolved to a matrix with the content values of all cells (unless
     *      the matrix would be too large, or one of the cells contains an
     *      error code).
     *  - 'mat:num':
     *      Similar to type 'mat', but checks that all matrix elements are
     *      valid floating-point numbers. In difference to parameters of type
     *      'val', other data types will NOT be converted to numbers, but
     *      result in throwing the #VALUE! error code immediately.
     *  - 'mat:str':
     *      Similar to type 'mat', but checks that all matrix elements are
     *      strings. In difference to parameters of type 'val', other data
     *      types will NOT be converted to strings, but result in throwing the
     *      #VALUE! error code immediately.
     *  - 'mat:bool':
     *      Similar to type 'mat', but checks that all matrix elements are
     *      Boolean values. In difference to parameters of type 'val', other
     *      data types will NOT be converted to Boolean values, but result in
     *      throwing the #VALUE! error code immediately.
     *  - 'mat:any':
     *      Similar to type 'mat', but accepts error codes as matrix elements,
     *      and passes them to the resolver function.
     *  - 'ref':
     *      A cell reference structure, as instance of the class Reference. If
     *      the parameter is not a cell reference, a special error code will be
     *      thrown indicating that the structure of the formula is considered
     *      invalid.
     *  - 'any':
     *      The original operand (instance of class Operand): single value
     *      literals (including error codes), constant matrixes, or unresolved
     *      cell references.
     *
     * @property {Function} [resolve]
     *  The implementation of the operator or function.
     *  (1) Calling context:
     *      The resolver function will be called with an instance of the class
     *      FormulaContext as calling context. Thus, the symbol 'this' provides
     *      useful helper methods that can be used in the implementation of the
     *      operator or function.
     *  (2) Parameters:
     *      The resolver function receives the converted operands according to
     *      the type signature specified in the property 'signature'. If the
     *      operands cannot be resolved to the specified types, the operator or
     *      function will result in the respective error code according to the
     *      error, or a fatal exception will be thrown specifying that the
     *      formula structure is invalid. The resolver function will not be
     *      invoked in this case.
     *  (3) Return value:
     *      The resolver function must return either a literal value (number,
     *      string, Boolean value, UTC Date object, or error code as instance
     *      of ErrorCode), a constant two-dimensional matrix containing values
     *      of any of the types mentioned before, a cell reference literal
     *      (instance of the class Reference), or an operand as received by an
     *      'any' parameter. If the returned number is infinite or NaN, it will
     *      be converted to a #NUM! error. If the returned (absolute) number is
     *      less than the least supported normalized positive number (see
     *      constant TokenUtils.MIN_NUMBER), it will be converted to zero. All
     *      these rules apply to all numbers in a matrix too.
     *  (4) Exceptions:
     *      Error code literals can be thrown as exception value, instead of
     *      being returned. This may happen implicitly, e.g. when using the
     *      data type conversion methods provided by the calling context (see
     *      class FormulaContext). Additionally, the resolver function may
     *      throw specific exception codes:
     *      - 'fatal': An unrecoverable internal error occurred. Interpretation
     *          of the formula will be canceled.
     *  If omitted, the operator or function will result in an 'unsupported'
     *  exception.
     *
     * Every property mentioned above but the 'formats' property can be an
     * object mapping different settings per file format.
     *
     *************************************************************************/

    var // shortcut for the map of error code literals
        ErrorCodes = SheetUtils.ErrorCodes,

        // mathematical constants
        PI = Math.PI,
        PI_2 = PI / 2,
        PI_180 = PI / 180,
        LN2 = Math.LN2,
        LN10 = Math.LN10,

        // shortcuts to built-in Math functions
        floor = Math.floor,
        ceil = Math.ceil,
        round = Math.round,
        abs = Math.abs,
        pow = Math.pow,
        sqrt = Math.sqrt,
        exp = Math.exp,
        log = Math.log,
        sin = Math.sin,
        cos = Math.cos,
        tan = Math.tan,
        sinh = Math.sinh || function sinh(number) { return (exp(number) - exp(-number)) / 2; },
        cosh = Math.cosh || function cosh(number) { return (exp(number) + exp(-number)) / 2; },
        tanh = Math.tanh || function tanh(number) { var pexp = exp(number), nexp = exp(-number); return (pexp - nexp) / (pexp + nexp); },
        asin = Math.asin,
        acos = Math.acos,
        atan = Math.atan,
        atan2 = Math.atan2,
        asinh = Math.asinh || function asinh(number) { return log(number + sqrt(number * number + 1)); },
        acosh = Math.acosh || function acosh(number) { return log(number + sqrt(number * number - 1)); },
        atanh = Math.atanh || function atanh(number) { return log((1 + number) / (1 - number)) / 2; },

        // standard options for numeric parameter iterators (method FormulaContext.iterateNumbers())
        NUMBER_ITERATOR_OPTIONS = {

            SKIP_MAT_SKIP_REF: { // id: CSS5
                valMode: 'convert', // value operands: convert strings and Booleans to numbers
                matMode: 'skip', // matrix operands: skip strings and Booleans
                refMode: 'skip', // reference operands: skip strings and Booleans
                emptyParam: true, // empty parameters count as zero
                complexRef: true // accept multi-range and multi-sheet references
            },

            SKIP_MAT_SKIP_REF_SKIP_EMPTY: { // id: CSS4
                valMode: 'convert', // value operands: convert strings and Booleans to numbers
                matMode: 'skip', // matrix operands: skip strings and Booleans
                refMode: 'skip', // reference operands: skip strings and Booleans
                // emptyParam: false, // skip all empty parameters
                complexRef: true // accept multi-range and multi-sheet references
            },

            SKIP_MAT_ZERO_REF: { // id: CSZ5
                valMode: 'convert', // value operands: convert strings and Booleans to numbers
                matMode: 'skip', // matrix operands: skip strings and Booleans
                refMode: 'zero', // reference operands: replace all strings with zeros
                emptyParam: true, // empty parameters count as zero
                complexRef: true // accept multi-range and multi-sheet references
            },

            ZERO_MAT_ZERO_REF: { // id: CZZ5
                valMode: 'convert', // value operands: convert strings and Booleans to numbers
                matMode: 'zero', // matrix operands: replace all strings with zeros
                refMode: 'zero', // reference operands: replace all strings with zeros
                emptyParam: true, // empty parameters count as zero
                complexRef: true // accept multi-range and multi-sheet references
            },

            RCONVERT_ALL_SKIP_EMPTY: { // id: RRR0
                valMode: 'rconvert', // value operands: convert strings to numbers (but not Booleans)
                matMode: 'rconvert', // matrix operands: convert strings to numbers (but not Booleans)
                refMode: 'rconvert', // reference operands: convert strings to numbers (but not Booleans)
                // emptyParam: false, // skip all empty parameters
                // complexRef: false // do not accept multi-range and multi-sheet references
            }
        },

        // standard options for logical parameter iterators (method FormulaContext.iterateBooleans())
        BOOLEAN_ITERATOR_OPTIONS = {

            SKIP_MAT_SKIP_REF: { // id: CSS5
                valMode: 'convert', // value operands: convert strings and numbers to Booleans
                matMode: 'skip', // matrix operands: skip all strings
                refMode: 'skip', // reference operands: skip all strings
                emptyParam: true, // empty parameters count as FALSE
                complexRef: true // accept multi-range and multi-sheet references
            }
        };

    // private global functions ===============================================

    /**
     * Creates and returns a new function that invokes the specified method on
     * the object passed to its first parameter, and returns the result of that
     * method.
     *
     * @param {String} method
     *  The name of the method to be invoked on the object passed to the first
     *  parameter of the generated function.
     *
     * @returns {Function}
     *  A function that invokes the specified method on the object passed to
     *  its first parameter. Does not pass any parameters to the method.
     */
    function invokeMethod(method) {
        return function (obj) {
            return obj[method]();
        };
    }

    // basic mathematical helpers ---------------------------------------------

    /**
     * Returns the sum of the passed numbers, or the concatenation of the
     * passed strings (used for callbacks).
     */
    function add(number1, number2) {
        return number1 + number2;
    }

    /**
     * Returns the product of the passed numbers (used for callbacks).
     */
    function multiply(number1, number2) {
        return number1 * number2;
    }

    /**
     * Returns the quotient of the passed numbers. If the divisor is zero, the
     * error code #DIV/0! will be thrown instead.
     */
    function divide(number1, number2) {
        if (number2 === 0) { throw ErrorCodes.DIV0; }
        return number1 / number2;
    }

    /**
     * Returns the floating-point modulo of the passed numbers. If the divisor
     * is zero, the error code #DIV/0! will be thrown instead. If the passed
     * numbers have different signs, the result will be adjusted according to
     * the results of other spreadsheet applications.
     */
    function modulo(number1, number2) {
        if (number2 === 0) { throw ErrorCodes.DIV0; }
        var result = number1 % number2;
        return ((result !== 0) && (number1 * number2 < 0)) ? (result + number2) : result;
    }

    /**
     * Returns the power of the passed numbers. If both numbers are zero, the
     * error code #NUM! will be thrown instead of returning 1.
     */
    function power(number1, number2) {
        // Math.pow() returns 1 for 0^0; Excel returns #NUM! error
        if ((number1 === 0) && (number2 === 0)) { throw ErrorCodes.NUM; }
        return pow(number1, number2);
    }

    /**
     * Returns the square of the passed numbers (used for callbacks).
     */
    function square(number) {
        return number * number;
    }

    /**
     * Returns the two-parameter arctangent of the passed coordinates. If both
     * numbers are zero, the error code #DIV/0! will be thrown instead. The
     * order of the parameters 'x' and 'y' is switched compared to the native
     * atan2() method, according to usage in other spreadsheet applications.
     */
    function arctan(x, y) {
        if ((x === 0) && (y === 0)) { throw ErrorCodes.DIV0; }
        // Spreadsheet applications use (x,y); but Math.atan2() expects (y,x)
        return atan2(y, x);
    }

    /**
     * Returns the cotangent of the passed number.
     */
    function cot(number) {
        return divide(1, tan(number));
    }

    /**
     * Returns the hyperbolic cotangent of the passed number.
     */
    function coth(number) {
        var pexp = exp(number), nexp = exp(-number);
        return divide(pexp + nexp, pexp - nexp);
    }

    /**
     * Returns the arc cotangent of the passed number.
     */
    function acot(number) {
        return PI_2 - atan(number);
    }

    /**
     * Returns the hyperbolic arc cotangent of the passed number.
     */
    function acoth(number) {
        return log((number + 1) / (number - 1)) / 2;
    }

    /**
     * Rounds the passed number to the next integer towards zero (positive
     * numbers will be rounded down, negative numbers will be rounded up).
     */
    function trunc(number) {
        return (number < 0) ? ceil(number) : floor(number);
    }

    /**
     * Returns the factorial of the passed number. Throws the #NUM! error code,
     * if the passed number is negative or too large.
     */
    var factorial = (function () {
        var FACTS = [], fact = 1;
        while (isFinite(fact)) { FACTS.push(fact); fact *= FACTS.length; }
        return function factorial(number) {
            var fact = FACTS[floor(number)];
            if (!fact) { throw ErrorCodes.NUM; }
            return fact;
        };
    }());

    /**
     * Returns the greatest common divisor of the passed integer numbers.
     *
     * @param {Number} int1
     *  The first integer number.
     *
     * @param {Number} int2
     *  The second integer number.
     *
     * @returns {Number}
     *  The greatest common divisor of the passed numbers.
     */
    function gcd(int1, int2) {
        while (int2 > 0) {
            var tmp = int2;
            int2 = round(int1 % int2);
            int1 = tmp;
        }
        return int1;
    }

    /**
     * Returns the binomial coefficient of the passed input values.
     *
     * @param {Number} n
     *  The upper value of the binomial coefficient (size of the set to be
     *  chosen from).
     *
     * @param {Number} k
     *  The lower value of the binomial coefficient (number of choices from the
     *  set).
     *
     * @returns {Number}
     *  The binomial coefficient, if both values are non-negative, and 'n' is
     *  greater than or equal to 'k'.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed values are invalid, or if the
     *  resulting binomial coefficient is too large.
     */
    function binomial(n, k) {

        // check input values
        if ((n < 0) || (k < 0) || (n < k)) { throw ErrorCodes.NUM; }

        // special cases (n 0) and (n n)
        if ((k === 0) || (k === n)) { return 1; }

        // binomial coefficient is symmetric: (n k) = (n n-k); use lower k for performance
        if (k * 2 > n) { k = n - k; }

        var // the resulting binomial coefficient
            result = n - k + 1;

        // The function isFinite() is quite expensive, by using two nested loops (the inner
        // loop runs up to 10 iterations without checking the result) the entire algorithm
        // becomes about 30% to 40% faster.
        for (var i = 2; (i <= k) && isFinite(result); ) {
            for (var j = 0; (j < 10) && (i <= k); i += 1, j += 1) {
                result *= (n - k + i);
                result /= i;
            }
        }

        // check final result
        if (isFinite(result)) { return result; }
        throw ErrorCodes.NUM;
    }

    /**
     * @param {Number} value
     * the value to be rounded
     * @param {Number} mult
     * multiple to round to
     * @returns {Number} the value rounded to a multiple of mult
     * @throws {ErrorCode}
     *  if signs of value and mult are different
     */
    function mround(value, mult) {
        if (value * mult < 0) {
            throw ErrorCodes.NUM;
        }
        var mod = abs(value % mult),
            result = abs(value),
            amult = abs(mult);
        if (mod) {
            if (mod > amult / 2) {
                result += amult - mod;
            } else {
                result -= mod;
            }
        }
        return value < 0 ? -result : result;
    }

    // numeric aggregation ----------------------------------------------------

    /**
     * Creates and returns a resolver function for spreadsheet functions that
     * reduces all numbers (and other values convertible to numbers) of all
     * function parameters to the result of the function. See method
     * FormulaContext.aggregateNumbers() for more details about the parameters.
     *
     * @returns {Function}
     *  The resulting function implementation to be assigned to the 'resolve'
     *  property of a function descriptor. The spreadsheet function will pass
     *  all numbers of all its operands to the aggregation callback function.
     */
    function implementNumericAggregation(initial, aggregate, finalize, iteratorOptions) {

        // create and return the resolver function to be assigned to the 'resolve' property of a function descriptor
        return function numericAggregator() {
            return this.aggregateNumbers(arguments, initial, aggregate, finalize, iteratorOptions);
        };
    }

    /**
     * Creates and returns a resolver function for spreadsheet functions that
     * reduces all numbers (and other values convertible to numbers) of all
     * function parameters to the result of the function. Works similar to the
     * function implementNumericAggregation(), but collects all visited numbers
     * in an array, before invoking the 'finalize' callback function.
     *
     * @param {Function} finalize
     *  A callback function invoked at the end to convert the collected numbers
     *  to the actual function result. Receives the following parameters:
     *  (1) {Array} numbers
     *      The visited numbers, as plain JavaScript array.
     *  (2) {Number} sum
     *      The sum of all visited numbers, for convenience.
     *  Must return the final numeric result of the spreadsheet function, or an
     *  error code object. Alternatively, may throw an error code. Will be
     *  called in the context of the resolver function (the FormulaContext
     *  instance).
     *
     * @param {Object} iteratorOptions
     *  Parameters passed to the number iterator used internally (see method
     *  FormulaContext.iterateNumbers() for details).
     *
     * @returns {Function}
     *  The resulting function implementation to be assigned to the 'resolve'
     *  property of a function descriptor. The spreadsheet function will pass
     *  all numbers of all its operands to the aggregation callback function.
     */
    function implementNumericAggregationWithArray(finalize, iteratorOptions) {

        // create and return the resolver function to be assigned to the 'resolve' property of a function descriptor
        return function numericArrayAggregator() {

            var // all numbers collected in an array
                numbers = [];

            // stores the new number in the array, and updates the sum
            function aggregate(result, number) {
                numbers.push(number);
                return result + number;
            }

            // invokes the passed 'finalize' callback with the collected numbers
            function finalizeImpl(sum) {
                return finalize.call(this, numbers, sum);
            }

            return this.aggregateNumbers(arguments, 0, aggregate, finalizeImpl, iteratorOptions);
        };
    }

    /**
     * Creates and returns a resolver function that reduces all numbers of all
     * function parameters to their minimum or maximum.
     *
     * @param {String} method
     *  The aggregation method to be implemented, either 'min' or 'max'.
     *
     * @param {Object} iteratorOptions
     *  Parameters passed to the number iterator used internally.
     *
     * @returns {Function}
     *  The resulting function implementation to be assigned to the 'resolve'
     *  property of a function descriptor. The spreadsheet function will pass
     *  all numbers of all its operands to the aggregation callback function.
     */
    function implementMinMaxAggregation(method, iteratorOptions) {

        var // start with signed infinity according to the function type
            initial = (method === 'max') ? Number.NEGATIVE_INFINITY : Number.POSITIVE_INFINITY,
            // the aggregation function (Math.min or Math.max)
            aggregate = _.bind(Math[method], Math);

        // default result (no numbers found in parameters) is zero (no #NUM! error code from infinity)
        function finalize(result) { return isFinite(result) ? result : 0; }

        // create and return the resolver function to be assigned to the 'resolve' property of a function descriptor
        return implementNumericAggregation(initial, aggregate, finalize, iteratorOptions);
    }

    /**
     * Creates and returns a resolver function that reduces all numbers of all
     * function parameters to their standard deviation, or their variance.
     *
     * @param {String} method
     *  The aggregation method to be implemented, either 'dev' for the standard
     *  deviation, or 'var' for the variance.
     *
     * @param {Boolean} population
     *  If set to true, calculates the standard deviation or variance based on
     *  the entire population. If set to false, calculates the result based on
     *  a sample.
     *
     * @param {Object} iteratorOptions
     *  Parameters passed to the number iterator used internally.
     *
     * @returns {Function}
     *  The resulting function implementation.
     */
    function implementStdDevAggregation(method, population, iteratorOptions) {

        // default result (no numbers found in parameters) is the #DIV/0! error code thrown by divide()
        function finalize(numbers, sum) {
            var mean = divide(sum, numbers.length),
                result = Utils.getSum(numbers, function (number) { return square(number - mean); }),
                size = population ? numbers.length : (numbers.length - 1);
            result = divide(result, size);
            return (method === 'dev') ? sqrt(result) : result;
        }

        // create and return the resolver function to be assigned to the 'resolve' property of a function descriptor
        return implementNumericAggregationWithArray(finalize, iteratorOptions);
    }

    /**
     * Creates and returns a resolver function for spreadsheet functions that
     * reduces all values matching a filter criterion to the result of the
     * function. See method FormulaContext.aggregateFiltered() for more details
     * about the parameters.
     *
     * @returns {Function}
     *  The resulting function implementation to be assigned to the 'resolve'
     *  property of a function descriptor.
     */
    function implementFilterAggregation(initial, aggregate, finalize) {

        // create and return the resolver function to be assigned to the 'resolve' property of a function descriptor
        return function filterAggregator(sourceRef, criterion, dataRef) {
            return this.aggregateFiltered(sourceRef, criterion, dataRef, initial, aggregate, finalize);
        };
    }

    // logical operations -----------------------------------------------------

    /**
     * Returns the logical NOT of the passed Boolean (used for callbacks).
     */
    function logicalNot(bool) {
        return !bool;
    }

    /**
     * Returns the logical AND of the passed Booleans (used for callbacks).
     */
    function logicalAnd(bool1, bool2) {
        return bool1 && bool2;
    }

    /**
     * Returns the logical OR of the passed Booleans (used for callbacks).
     */
    function logicalOr(bool1, bool2) {
        return bool1 || bool2;
    }

    /**
     * Returns the logical XOR of the passed Booleans (used for callbacks).
     */
    function logicalXor(bool1, bool2) {
        return bool1 !== bool2;
    }

    /**
     * Compares the passed literal values. The following rules apply for
     * comparison:
     * (1) Two null values are equal.
     * (2) A null value is converted to 0, if compared to a number.
     * (3) A null value is converted to '', if compared to a string.
     * (4) A null value is converted to FALSE, if compared to a Boolean.
     * (5) Numbers are always less than strings and Boolean values.
     * (6) Strings are always less than Boolean values.
     * (7) Two numbers are simply compared by value.
     * (8) Two strings are compared lexicographically and case-insensitive.
     * (9) FALSE is less than TRUE.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value1
     *  The first value to be compared.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value2
     *  The second value to be compared.
     *
     * @returns {Number}
     *  - A negative value, if value1 is less than value2.
     *  - A positive value, if value1 is greater than value2.
     *  - Zero, if both values are of the same type and are equal.
     */
    function compareValues(value1, value2) {

        // converts special value null (empty cell) according to the other value type
        function convertNull(value) {
            if (_.isNumber(value)) { return 0; }
            if (_.isString(value)) { return ''; }
            if (_.isBoolean(value)) { return false; }
            throw 'fatal';
        }

        // returns a numeric identifier according to the value type
        function getTypeId(value) {
            if (_.isNumber(value)) { return 1; }
            if (_.isString(value)) { return 2; }
            if (_.isBoolean(value)) { return 3; }
            throw 'fatal';
        }

        // two empty cells are equal
        if (_.isNull(value1) && _.isNull(value2)) { return 0; }

        // convert null values to typed values according to the other value
        if (_.isNull(value1)) { value1 = convertNull(value2); }
        if (_.isNull(value2)) { value2 = convertNull(value1); }

        // get type identifiers, compare different types
        var type1 = getTypeId(value1), type2 = getTypeId(value2);
        if (type1 !== type2) { return type1 - type2; }

        // compare values of equal types
        switch (type1) {
        case 1:
            return TokenUtils.compareNumbers(value1, value2);
        case 2:
            return TokenUtils.compareStrings(value1, value2); // case-insensitive
        case 3:
            return TokenUtils.compareBooleans(value1, value2);
        }

        throw 'fatal';
    }

    /**
     * Creates and returns a resolver function for spreadsheet functions that
     * reduces all Boolean values (and other values convertible to Boolean
     * values) of all function parameters to the result of the function. See
     * method FormulaContext.aggregateBooleans() for more details about the
     * parameters.
     *
     * @returns {Function}
     *  The resulting function implementation to be assigned to the 'resolve'
     *  property of a function descriptor. The spreadsheet function will pass
     *  all Boolean values of all its operands to the aggregation callback
     *  function. If no Boolean values have been found at all, the spreadsheet
     *  function will result in the #VALUE! error code, instead of the initial
     *  result.
     */
    function implementLogicalAggregation(initial, aggregate, iteratorOptions) {

        // no Booleans available results in the #VALUE! error code (not the initial value)
        function finalize(result, count) {
            if (count === 0) { throw ErrorCodes.VALUE; }
            return result;
        }

        // create and return the resolver function to be assigned to the 'resolve' property of a function descriptor
        return function logicalAggregator() {
            return this.aggregateBooleans(arguments, initial, aggregate, finalize, iteratorOptions);
        };
    }

    // date and time ----------------------------------------------------------

    /**
     * Creates a UTC date object used inside the formula interpreter from the
     * specified date and time.
     *
     * @param {Number} year
     *  The full year.
     *
     * @param {Number} month
     *  The zero-based (!) month.
     *
     * @param {Number} day
     *  The one-based (!) day of the month.
     *
     * @param {Number} hours
     *  The hours of the time.
     *
     * @param {Number} minutes
     *  The minutes of the time.
     *
     * @param {Number} seconds
     *  The seconds of the time.
     *
     * @param {Number} millisecs
     *  The milliseconds of the time.
     *
     * @returns {Date}
     *  The UTC date object representing the specified date and time.
     */
    function makeDateTime(year, month, day, hours, minutes, seconds, millisecs) {
        return new Date(Date.UTC(year, month, day, hours, minutes, seconds, millisecs));
    }

    /**
     * Creates a UTC date object used inside the formula interpreter from the
     * specified date. The time will be set to midnight.
     *
     * @param {Number} year
     *  The full year.
     *
     * @param {Number} month
     *  The zero-based (!) month.
     *
     * @param {Number} day
     *  The one-based (!) day of the month.
     *
     * @returns {Date}
     *  The UTC date object representing the specified date and time.
     */
    function makeDate(year, month, day) {
        return makeDateTime(year, month, day, 0, 0, 0, 0);
    }

    /**
     * Returns whether the passed number is a leap year.
     *
     * @param {Number} year
     *  The year to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed number is a leap year.
     */
    function isLeapYear(year) {
        return (((year % 4) === 0) && ((year % 100) !== 0)) || ((year % 400) === 0);
    }

    /**
     * Returns the number of days in the specified month.
     *
     * @param {Number} month
     *  The month. MUST be a zero-based integer (0 to 11).
     *
     * @param {Number} year
     *  The year of the month.
     *
     * @returns {Number}
     *  The number of days in the specified month.
     */
    var getDaysInMonth = (function () {

        var // number of days per month (regular years)
            DAYS_PER_MONTH = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];

        return function getDaysInMonth(month, year) {
            // special handling for February in leap years
            return ((month === 1) && isLeapYear(year)) ? 29 : DAYS_PER_MONTH[month];
        };
    }());

    /**
     * Adds the specified number of days to the passed date, and returns the
     * resulting date.
     *
     * @param {Date} date
     *  The original date.
     *
     * @param {Number} days
     *  The number of days to add. If negative, a date in the past will be
     *  returned accordingly. The number will be truncated to an integer
     *  (always towards zero).
     */
    function addDaysToDate(date, days) {
        return new Date(date.getTime() + days * 86400000);
    }

    /**
     * Adds the specified number of months to the passed date, and returns the
     * resulting date. Implementation helper for the spreadsheet functions
     * EDATE and EOMONTH.
     *
     * @param {Date} date
     *  The original date.
     *
     * @param {Number} months
     *  The number of months to add. If negative, a date in the past will be
     *  returned accordingly. The number will be truncated to an integer
     *  (always towards zero).
     *
     * @param {Boolean} end
     *  If set to true, the returned date will be set to the last day of the
     *  resulting month. Otherwise, the day of the passed original date will be
     *  used (unless the resulting month is shorter than the passed day, in
     *  this case, the last valid day of the new month will be used).
     */
    function addMonthsToDate(date, months, end) {

        var // the year, month (zero-based), and day (one-based) of the passed date
            year = date.getUTCFullYear(),
            month = date.getUTCMonth(),
            day = date.getUTCDate();

        // get the new month and year
        month += trunc(months);
        year += floor(month / 12);
        month = Utils.mod(month, 12);

        // validate the day (new month may have less days than the day passed in the original date)
        day = end ? getDaysInMonth(month, year) : Math.min(day, getDaysInMonth(month, year));

        // build and return the new date (interpreter will throw #NUM! error, if new year is invalid)
        return makeDateTime(year, month, day, date.getUTCHours(), date.getUTCMinutes(), date.getUTCSeconds(), date.getUTCMilliseconds());
    }

    /**
     * Implementation helper for the sheet functions WEEKNUM and ISOWEEKNUM.
     *
     * @param {Date} date
     *  The UTC date the week number will be calculated for.
     *
     * @param {Number} firstDayOfWeek
     *  Determines the start day of the week: 0 for Sunday, 1 for Monday, ...,
     *  or 6 for Saturday.
     *
     * @param {Boolean} iso8601
     *  Whether to use ISO 8601 mode (week of January 1st is the first week, if
     *  at least 4 days are in the new year), or US mode (weeks are counted
     *  strictly inside the year, first and last week in the year may consist
     *  of less than 7 days).
     *
     * @returns {Number}
     *  The resulting week number for the passed date.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the date mode is not valid.
     */
    function getWeekNum(date, firstDayOfWeek, iso8601) {

        // returns the date of the first day of the first week, according to specified first day in week
        function getNullDateOfYear(year) {

            var // base date (that is always in the first week of the year; ISO: Jan-04, otherwise: Jan-01)
                baseDate = makeDate(year, 0, iso8601 ? 4 : 1),
                // week day of the base date
                baseDay = baseDate.getUTCDay();

            // first day of first week in the year, according to date mode
            return addDaysToDate(baseDate, -Utils.mod(baseDay - firstDayOfWeek, 7));
        }

        var // the year of the passed date
            currYear = date.getUTCFullYear(),
            // first day of first week in the year, according to date mode
            nullDate = getNullDateOfYear(currYear);

        // adjustment for ISO mode: passed date may be counted for the previous year
        // (for Jan-01 to Jan-03), or for the next year (for Dec-29 to Dec-31)
        if (iso8601) {
            if (date < nullDate) {
                // passed date is counted for previous year (e.g. Jan-01-2000 is in week 52 of 1999)
                nullDate = getNullDateOfYear(currYear - 1);
            } else {
                var nextNullDate = getNullDateOfYear(currYear + 1);
                if (nextNullDate <= date) {
                    // passed date is counted for next year (e.g. Dec-31-2001 is in week 1 of 2002)
                    nullDate = nextNullDate;
                }
            }
        }

        return floor((date - nullDate) / 604800000) + 1;
    }

    /**
     * Implementation helper for the sheet functions WEEKNUM and ISOWEEKNUM for
     * OOXML documents.
     *
     * @param {Date} date
     *  The UTC date the week number will be calculated for.
     *
     * @param {Number} [mode=1]
     *  Determines the start day of the week.
     *
     * @returns {Number}
     *  The resulting week number for the passed date.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the date mode is not valid.
     */
    var getWeekNumOOXML = (function () {

        var // maps date modes to first day in week (zero-based, starting from Sunday)
            FIRST_WEEKDAY = { 1: 0, 2: 1, 11: 1, 12: 2, 13: 3, 14: 4, 15: 5, 16: 6, 17: 0, 21: 1 };

        return function getWeekNumOOXML(date, mode) {

            var // first day in the week, according to passed date mode
                firstDayOfWeek = FIRST_WEEKDAY[_.isNumber(mode) ? mode : 1];

            // bail out if passed date mode is invalid
            if (!_.isNumber(firstDayOfWeek)) { throw ErrorCodes.NUM; }

            return getWeekNum(date, firstDayOfWeek, mode === 21);
        };
    }());

    /**
     * Implementation helper for the sheet functions WEEKNUM for ODF documents.
     * This function always uses ISO 8601 mode (week of January 1st is the
     * first week, if at least 4 days are in the new year).
     *
     * @param {Date} date
     *  The UTC date the week number will be calculated for.
     *
     * @param {Number} mode
     *  Determines the start day of the week.
     *
     * @returns {Number}
     *  The resulting week number for the passed date.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the date mode is not valid.
     */
    function getWeekNumODF(date, mode) {
        // date mode 1 for Sunday, every other value for Monday
        return getWeekNum(date, (mode === 1) ? 0 : 1, true);
    }

    // number systems ---------------------------------------------------------

    /**
     * Converts the string representing of a binary number to a decimal number.
     *
     * @param {String} value
     *  A binary number string to be converted to a number if it is not longer
     *  than 10 characters.
     *
     * @returns {Number}
     *  The resulting decimal number.
     *
     * @throws {ErrorCode}
     *  If the passed value is not a valid binary number, or if it is too long
     *  (more than 10 digits), a #NUM! error will be thrown.
     */
    function convertBinToDec(value) {

        // check input value, maximum length of the input string is 10 digits
        if (!/^[01]{0,10}$/.test(value)) {
            throw ErrorCodes.NUM;
        }

        var // the parsed result (empty string is interpreted as zero)
            result = (value.length > 0) ? parseInt(value, 2) : 0;

        // negative number, if 10th digit is 1
        if ((value.length === 10) && /^1/.test(value)) {
            result -= 0x400;
        }

        return result;
    }

    /**
     * Converts the string representing of an octal number to a decimal number.
     *
     * @param {String} value
     *  An octal number string to be converted to a number if it is not longer
     *  than 10 characters.
     *
     * @returns {Number}
     *  The resulting decimal number.
     *
     * @throws {ErrorCode}
     *  If the passed value is not a valid octal number, or if it is too long
     *  (more than 10 digits), a #NUM! error will be thrown.
     */
    function convertOctToDec(value) {

        // check input value, maximum length of the input string is 10 digits
        if (!/^[0-7]{0,10}$/.test(value)) {
            throw ErrorCodes.NUM;
        }

        var // the parsed result (empty string is interpreted as zero)
            result = (value.length > 0) ? parseInt(value, 8) : 0;

        // negative number, if 10th digit is greater than 3
        if ((value.length === 10) && /^[4-7]/.test(value)) {
            result -= 0x40000000;
        }

        return result;
    }

    /**
     * Converts the string representing of a hexadecimal number to a decimal
     * number.
     *
     * @param {String} value
     *  A hexadecimal number string to be converted to a number if it is not
     *  longer than 10 characters.
     *
     * @returns {Number}
     *  The resulting decimal number.
     *
     * @throws {ErrorCode}
     *  If the passed value is not a valid hexadecimal number, or if it is too
     *  long (more than 10 digits), a #NUM! error will be thrown.
     */
    function convertHexToDec(value) {

        // check input value, maximum length of the input string is 10 digits
        if (!/^[0-9a-f]{0,10}$/i.test(value)) {
            throw ErrorCodes.NUM;
        }

        var // the parsed result (empty string is interpreted as zero)
            result = (value.length > 0) ? parseInt(value, 16) : 0;

        // negative number, if 10th digit is greater than 7
        if ((value.length === 10) && /^[89a-f]/i.test(value)) {
            result -= 0x10000000000;
        }

        return result;
    }

    /**
     * Expands the passed string (binary, octal, or hexadecimal number) with
     * leading zero characters, according to the passed length.
     *
     * @param {String} number
     *  A number as string (binary, octal, or hexadecimal).
     *
     * @param {Number} [length]
     *  If specified, the target length of the passed number. Must be an
     *  integer between the length of the passed number and 10; otherwise this
     *  function throws a #NUM! error. If omitted, the passed number will be
     *  returned unmodified.
     *
     * @returns {String}
     *  The resulting number with leading zero characters.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed length is invalid.
     */
    function expandWithZeros(number, length) {
        if (_.isNumber(length)) {
            // length parameter must be valid
            if ((length < number.length) || (length > 10)) {
                throw ErrorCodes.NUM;
            }
            // expand, but do not truncate
            if (number.length < length) {
                number = ('0000000000' + number).slice(-length);
            }
        }
        return number;
    }

    /**
     * Converts the passed decimal number to its binary representation. Inserts
     * leading zero characters if specified.
     *
     * @param {Number} number
     *  The number to be converted to its binary representation.
     *
     * @param {Number} [length]
     *  If specified, the target length of the passed number. See function
     *  expandWithZeros() for details.
     *
     * @returns {String}
     *  The resulting binary number.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed number results in more than 10
     *  binary digits, or if the passed length is invalid.
     */
    function convertDecToBin(number, length) {
        // restrict passed number to signed 10-digit binary
        if ((-number > 0x200) || (number > 0x1FF)) { throw ErrorCodes.NUM; }
        // convert negative numbers to positive resulting in a 10-digit binary number
        if (number < 0) { number += 0x400; }
        // convert to hexadecimal number, expand to specified length
        return expandWithZeros(number.toString(2), length);
    }

    /**
     * Converts the passed decimal number to its octal representation. Inserts
     * leading zero characters if specified.
     *
     * @param {Number} number
     *  The number to be converted to its octal representation.
     *
     * @param {Number} [length]
     *  If specified, the target length of the passed number. See function
     *  expandWithZeros() for details.
     *
     * @returns {String}
     *  The resulting octal number.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed number results in more than 10
     *  octal digits, or if the passed length is invalid.
     */
    function convertDecToOct(number, length) {
        // restrict passed number to signed 10-digit octal
        if ((-number > 0x20000000) || (number > 0x1FFFFFFF)) { throw ErrorCodes.NUM; }
        // convert negative numbers to positive resulting in a 10-digit octal number
        if (number < 0) { number += 0x40000000; }
        // convert to octal number, expand to specified length
        return expandWithZeros(number.toString(8), length);
    }

    /**
     * Converts the passed decimal number to its hexadecimal representation.
     * Inserts leading zero characters if specified.
     *
     * @param {Number} number
     *  The number to be converted to its hexadecimal representation.
     *
     * @param {Number} [length]
     *  If specified, the target length of the passed number. See function
     *  expandWithZeros() for details.
     *
     * @returns {String}
     *  The resulting hexadecimal number.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed number results in more than 10
     *  hexadecimal digits, or if the passed length is invalid.
     */
    function convertDecToHex(number, length) {
        // restrict passed number to signed 10-digit hexadecimal
        if ((-number > 0x8000000000) || (number > 0x7FFFFFFFFF)) { throw ErrorCodes.NUM; }
        // convert negative numbers to positive resulting in a 10-digit hex number
        if (number < 0) { number += 0x10000000000; }
        // convert to hexadecimal number, expand to specified length
        return expandWithZeros(number.toString(16).toUpperCase(), length);
    }

    // complex numbers --------------------------------------------------------

    /**
     * Returns the absolute value of the passed complex number.
     */
    function complexAbs(complex) {
        return sqrt(square(complex.real) + square(complex.imag));
    }

    /**
     * Returns the argument of the passed complex number, as radiant. Throws a
     * #DIV/0! error for the complex number 0.
     */
    function complexArg(complex) {
        return arctan(complex.real, complex.imag);
    }

    /**
     * Returns the power of a complex number to the passed real exponent. If
     * both numbers are zero, the error code #NUM! will be thrown.
     *
     * @param {Complex} complex
     *  The complex number.
     *
     * @param {Number} exponent
     *  The exponent (real number).
     *
     * @returns {Complex}
     *  The complex power, using the imaginary unit of the passed complex
     *  number.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if both numbers are zero.
     */
    function complexPower(complex, exponent) {

        // 0^0 is undefined, 0^exp results in 0 (complexArg(0) would throw #DIV/0!)
        if ((complex.real === 0) && (complex.imag === 0)) {
            if (exponent === 0) { throw ErrorCodes.NUM; }
            return complex;
        }

        // calculate complex power via polar representation
        var cabs = pow(complexAbs(complex), exponent), carg = exponent * complexArg(complex);
        return new Complex(cabs * cos(carg), cabs * sin(carg), complex.unit);
    }

    /**
     * Returns a resolver function for IMSUM and IMPRODUCT that reduces all
     * complex numbers of all function parameters to their sum or product.
     *
     * @param {Function} callback
     *  The aggregation callback function. Receives two complex numbers, and
     *  must return the resulting complex number (imaginary unit can be omitted
     *  in the resulting complex number).
     *
     * @returns {Function}
     *  The resulting function implementation for IMSUM and IMPRODUCT.
     */
    function implementComplexAggregation(callback) {

        // create the resolver callback function, will be called with FormulaContext context
        return function complexAggregator() {

            var // the resulting complex product
                result = null,
                // whether any parameter was non-empty
                hasAny = false;

            this.iterateOperands(0, function (operand) {
                this.iterateValues(operand, function (value) {
                    var complex = this.convertToComplex(value);
                    if (result) {
                        this.checkComplexUnits(result, complex);
                        result = callback.call(this, result, complex);
                        result.unit = result.unit || complex.unit;
                    } else {
                        result = new Complex(complex.real, complex.imag, complex.unit);
                    }
                }, {
                    emptyParam: false, // empty parameters are skipped: IMSUM("1+i",) = "1+i"
                    complexRef: false // do not accept multi-range and multi-sheet references
                });
                if (!operand.isEmpty()) { hasAny = true; }
            });

            // only empty arguments: return #N/A instead of zero
            if (!hasAny) { throw ErrorCodes.NA; }
            // only references to empty cells: return 0
            return _.isObject(result) ? result : new Complex(0, 0);
        };
    }

    // operators ==============================================================

    var OPERATORS = {

        // arithmetic operators -----------------------------------------------

        // binary addition operator, or unary plus operator
        '+': {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val val:num', // unary plus operator passes all values (no conversion to number)
            resolve: function (value1, value2) {
                return _.isUndefined(value2) ? value1 : (this.convertToNumber(value1) + value2);
            }
        },

        // binary subtraction operator, or unary minus operator
        '-': {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num', // unary minus operator always converts to numbers
            resolve: function (value1, value2) {
                return _.isUndefined(value2) ? -value1: (value1 - value2);
            }
        },

        // binary multiplication operator
        '*': {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: multiply
        },

        // binary division operator
        '/': {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: divide
        },

        // binary power operator
        '^': {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: power
        },

        // unary percent operator
        '%': {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (value) { return value / 100; }
        },

        // string operators ---------------------------------------------------

        // binary string concatenation operator
        '&': {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:str',
            resolve: add
        },

        // comparison operators -----------------------------------------------

        // binary less-than operator
        '<': {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val val',
            resolve: function (value1, value2) {
                return compareValues(value1, value2) < 0;
            }
        },

        // binary less-than-or-equal operator
        '<=': {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val val',
            resolve: function (value1, value2) {
                return compareValues(value1, value2) <= 0;
            }
        },

        // binary greater-than operator
        '>': {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val val',
            resolve: function (value1, value2) {
                return compareValues(value1, value2) > 0;
            }
        },

        // binary greater-than-or-equal operator
        '>=': {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val val',
            resolve: function (value1, value2) {
                return compareValues(value1, value2) >= 0;
            }
        },

        // binary equals operator
        '=': {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val val',
            resolve: function (value1, value2) {
                return compareValues(value1, value2) === 0;
            }
        },

        // binary equals-not operator
        '<>': {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val val',
            resolve: function (value1, value2) {
                return compareValues(value1, value2) !== 0;
            }
        },

        // reference operators ------------------------------------------------

        // binary list reference operator
        ',': {
            minParams: 2,
            maxParams: 2,
            type: 'ref',
            signature: 'ref ref',
            resolve: function (ref1, ref2) {
                return new Reference(ref1.getRanges().concat(ref2.getRanges()));
            }
        },

        // binary intersection reference operator
        ' ': {
            minParams: 2,
            maxParams: 2,
            type: 'ref',
            signature: 'ref ref',
            resolve: function (ref1, ref2) {

                var // get all ranges in the passed operands
                    ranges1 = ref1.getRanges({ sameSheet: true }),
                    ranges2 = ref2.getRanges({ sameSheet: true }),
                    sheet = ranges1[0].sheet1,
                    intersectRanges = [];

                // ranges in both operands must refer to the same sheet
                if (sheet !== ranges2[0].sheet1) {
                    throw ErrorCodes.VALUE;
                }

                // calculate the intersection of each range pair from ranges1 and ranges2
                _.each(ranges1, function (range1) {
                    intersectRanges = intersectRanges.concat(SheetUtils.getIntersectionRanges(ranges2, range1));
                    // check resulting list size in loop, otherwise this loop may create a MAX^2 list
                    if (intersectRanges.length > TokenUtils.MAX_REF_LIST_SIZE) { throw TokenUtils.UNSUPPORTED_ERROR; }
                });

                // insert sheet index into all ranges
                _.each(intersectRanges, function (range) {
                    range.sheet1 = range.sheet2 = sheet;
                });

                return new Reference(intersectRanges);
            }
        },

        // binary range reference operator
        ':': {
            minParams: 2,
            maxParams: 2,
            type: 'ref',
            signature: 'ref ref',
            resolve: function (ref1, ref2) {

                var // get all ranges in the passed operands
                    ranges1 = ref1.getRanges({ sameSheet: true }),
                    ranges2 = ref2.getRanges({ sameSheet: true }),
                    boundRange = null;

                // ranges in both operands must refer to the same sheet
                if (ranges1[0].sheet1 !== ranges2[0].sheet1) {
                    throw ErrorCodes.VALUE;
                }

                // build the bounding range from all ranges
                boundRange = SheetUtils.getBoundingRange(ranges1, ranges2);
                boundRange.sheet1 = boundRange.sheet2 = ranges1[0].sheet1;
                return new Reference(boundRange);
            }
        }
    };

    // functions ==============================================================

    // ordered alphabetically to reduce risk of merge conflicts
    var FUNCTIONS = {

        ABS: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: abs
        },

        ACCRINT: {
            minParams: 6,
            maxParams: 8,
            type: 'val'
        },

        ACCRINTM: {
            minParams: 4,
            maxParams: 5,
            type: 'val'
        },

        ACOS: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: acos
        },

        ACOSH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: acosh
        },

        ACOT: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: acot
        },

        ACOTH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: acoth
        },

        ADDRESS: {
            minParams: 2,
            maxParams: 5,
            type: 'val',
            signature: 'val:int val:int val:int val:bool val:str',
            resolve: function (row, col, absMode, a1Style, sheetRef) {

                var // document model
                    docModel = this.getModel(),
                    // absolute columns or row index
                    absCol = false, absRow = false,
                    // the resulting address string
                    result = '';

                // returns the string representation of the passed column/row index for A1 notation
                function createA1Index(index, absRef, columns) {
                    var maxIndex = docModel.getMaxIndex(columns);
                    if ((1 <= index) && (index <= maxIndex + 1)) {
                        return (absRef ? '$' : '') + SheetUtils.getIndexName(index - 1, columns);
                    }
                    throw ErrorCodes.VALUE;
                }

                // returns the string representation of the passed column/row index for RC notation
                function createRCIndex(index, abs, columns) {
                    var maxIndex = docModel.getMaxIndex(columns);
                    if (abs) {
                        // absolute mode: always insert one-based column/row index (e.g. R1C4)
                        if ((1 <= index) && (index <= maxIndex + 1)) { return '' + index; }
                    } else {
                        // relative mode: leave out zero values (e.g. RC[-1])
                        if (index === 0) { return ''; }
                        if ((-maxIndex <= index) && (index <= maxIndex)) { return '[' + index + ']'; }
                    }
                    throw ErrorCodes.VALUE;
                }

                // default for missing or empty absolute mode (always defaults to 1)
                if (this.isMissingOrEmptyOperand(2)) {
                    absMode = 1;
                } else if ((absMode < 1) || (absMode > 4)) {
                    throw ErrorCodes.VALUE;
                }
                absCol = (absMode % 2) > 0;
                absRow = absMode <= 2;

                // default for missing or empty A1 style (always defaults to TRUE)
                if (this.isMissingOrEmptyOperand(3)) {
                    a1Style = true;
                }

                if (a1Style) {
                    result = createA1Index(col, absCol, true) + createA1Index(row, absRow, false);
                } else {
                    // TODO: localized R/C names
                    result = 'R' + createRCIndex(row, absRow, false) + 'C' + createRCIndex(col, absCol, true);
                }

                // add encoded sheet name (enclosed in apostrophes if needed)
                // (not for missing or empty parameter, but for explicit empty string)
                if (!this.isMissingOrEmptyOperand(4)) {
                    result = ((sheetRef.length > 0) ? this.getTokenizer().encodeExtendedSheetRef(sheetRef) : '') + '!' + result;
                }

                return result;
            }
        },

        AMORDEGRC: {
            minParams: 6,
            maxParams: 7,
            type: 'val'
        },

        AMORLINC: {
            minParams: 6,
            maxParams: 7,
            type: 'val'
        },

        AND: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as FALSE: AND(TRUE,) = FALSE
            resolve: implementLogicalAggregation(true, logicalAnd, BOOLEAN_ITERATOR_OPTIONS.SKIP_MAT_SKIP_REF)
        },

        ARABIC: {
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        AREAS: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'ref',
            resolve: invokeMethod('getLength')
        },

        ASC: {
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        ASIN: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: asin
        },

        ASINH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: asinh
        },

        ATAN: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: atan
        },

        ATAN2: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: arctan
        },

        ATANH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: atanh
        },

        AVEDEV: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // default result (no numbers found in parameters) is the #NUM! error code (not #DIV/0!)
                function finalize(numbers, sum) {
                    if (numbers.length === 0) { throw ErrorCodes.NUM; }
                    var mean = sum / numbers.length;
                    return Utils.getSum(numbers, function (number) { return abs(number - mean); }) / numbers.length;
                }

                // empty parameters count as zero: AVEDEV(1,) = 0.5
                return implementNumericAggregationWithArray(finalize, NUMBER_ITERATOR_OPTIONS.SKIP_MAT_SKIP_REF);
            }())
        },

        AVERAGE: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as zero: AVERAGE(1,) = 0.5
            // default result (no numbers found in parameters) is the #DIV/0! error code thrown by divide()
            resolve: implementNumericAggregation(0, add, divide, NUMBER_ITERATOR_OPTIONS.SKIP_MAT_SKIP_REF)
        },

        AVERAGEA: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as zero: AVERAGEA(1,) = 0.5
            // default result (no numbers found in parameters) is the #DIV/0! error code thrown by divide()
            resolve: implementNumericAggregation(0, add, divide, NUMBER_ITERATOR_OPTIONS.ZERO_MAT_ZERO_REF)
        },

        AVERAGEIF: {
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'ref val:any ref',
            resolve: implementFilterAggregation(0, add, divide)
        },

        AVERAGEIFS: {
            minParams: 3,
            repeatParams: 2,
            type: 'val'
        },

        B: {
            formats: 'odf', // OOXML name: BINOM.DIST.RANGE
            minParams: 3,
            maxParams: 4,
            type: 'val'
        },

        BAHTTEXT: {
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        BASE: {
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'val:int val:int val:int',
            resolve: function (number, radix, minLength) {
                var result;
                minLength = _.isUndefined(minLength) ? 0 : minLength;
                if (number < 0 || number > 0x1FFFFFFFFFFFFF || radix < 2 || radix > 36 || minLength < 0 || minLength > 255) {
                    throw ErrorCodes.NUM;
                }
                result = number.toString(radix);
                while (minLength > result.length) {
                    result = '0' + result;
                }
                return result.toUpperCase();
            }
        },

        BESSELI: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:int',
            resolve: Bessel.I
        },

        BESSELJ: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:int',
            resolve: Bessel.J
        },

        BESSELK: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:int',
            resolve: Bessel.K
        },

        BESSELY: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:int',
            resolve: Bessel.Y
        },

        BETADIST: {
            minParams: 3,
            maxParams: 5,
            type: 'val'
        },

        BETAINV: {
            minParams: 3,
            maxParams: 5,
            type: 'val'
        },

        BIN2DEC: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: convertBinToDec
        },

        BIN2HEX: {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: function (number, length) {
                return convertDecToHex(convertBinToDec(number), length);
            }
        },

        BIN2OCT: {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: function (number, length) {
                return convertDecToOct(convertBinToDec(number), length);
            }
        },

        BINOMDIST: {
            // OOXML: new version BINOM.DIST
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        CEILING: {
            minParams: 2,
            maxParams: { def: 2, odf: 3 },
            type: 'val',
            signature: 'val:num val:num val:int',
            resolve: function (number, significance, mode) {
                var result, modulo, minBorder = 1e-16;
                if (_.isUndefined(mode)) {
                    mode = this.isODF() ? 0 : 1;
                }
                if (significance === 0) {
                    throw ErrorCodes.VALUE;
                } else if(significance < 0 && number > 0) {
                    throw ErrorCodes.NUM;
                }
                modulo = abs(significance - (number % significance));
                if (number === 0 || modulo < minBorder) {
                    result = number;
                } else {
                    result = (floor(number / significance) + 1) * significance;
                }
                return result;
            }
        },

        CELL: {
            minParams: 1,
            maxParams: 2,
            type: 'val'
        },

        CHAR: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:int',
            resolve: function (value) {
                if ((value < 1) || (value > 255)) { throw ErrorCodes.VALUE; }
                // TODO: not exact, needs codepage conversion
                return String.fromCharCode(value);
            }
        },

        CHIDIST: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        CHIINV: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        CHISQDIST: {
            minParams: 2,
            maxParams: 3,
            type: 'val'
        },

        CHISQINV: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        CHITEST: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        CHOOSE: {
            minParams: 2,
            type: 'any',
            signature: 'val:int any',
            resolve: function (index) {
                return this.getOperand(index - 1);
            }
        },

        CLEAN: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: function (value) {
                // TODO: other Unicode characters?
                return value.replace(/[\x00-\x1f\x81\x8d\x8f\x90\x9d]/g, '');
            }
        },

        CODE: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: function (value) {
                // TODO: not exact, needs codepage conversion
                return (value.length === 0) ? ErrorCodes.VALUE : value.charCodeAt(0);
            }
        },

        COLUMN: {
            minParams: 0,
            maxParams: 1,
            type: 'val',
            signature: 'ref',
            resolve: function (ref) {
                // use reference address, if parameter is missing
                var address = _.isUndefined(ref) ? this.getRefAddress() : ref.getSingleRange().start;
                // function returns one-based column index
                return address[0] + 1;
            }
        },

        COLUMNS: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'any',
            resolve: function (operand) {
                switch (operand.getType()) {
                case 'val':
                    var value = operand.getValue();
                    return SheetUtils.isErrorCode(value) ? value : 1;
                case 'mat':
                    return operand.getMatrix().getColCount();
                case 'ref':
                    return SheetUtils.getColCount(operand.getReference().getSingleRange());
                default:
                    throw 'fatal';
                }
            }
        },

        COMBIN: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:int val:int',
            resolve: binomial
        },

        COMBINA: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:int val:int',
            resolve: function (number1, number2) {
                return binomial(number1 + number2 - 1, number2);
            }
        },

        COMPLEX: {
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'val:num val:num val:str',
            resolve: function (real, imag, unit) {
                // use 'i', if imaginary unit is missing (undefined), empty parameter (null), or empty string
                if (!unit) { unit = 'i'; }
                // unit must be lower-case 'i' or 'j'
                if ((unit !== 'i') && (unit !== 'j')) { throw ErrorCodes.VALUE; }
                // create a complex number (will be converted to a string)
                return new Complex(real, imag, unit);
            }
        },

        CONCATENATE: {
            minParams: 1,
            type: 'val',
            // each parameter is a single value (function does not concatenate cell ranges or matrixes!)
            signature: 'val:str',
            resolve: function () {
                return _.reduce(arguments, add, '');
            }
        },

        CONFIDENCE: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        CONVERT: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        CONVERT_ADD: {
            formats: 'odf',
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        CORREL: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        COS: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: cos
        },

        COSH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: cosh
        },

        COT: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: cot
        },

        COTH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: coth
        },

        COUNT: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: function () {
                var count = 0;
                this.iterateOperands(0, function (operand) {
                    if (operand.isValue()) {
                        // count Booleans and strings that are convertible to numbers
                        // (simple values only, not counted in matrixes and references)
                        try {
                            this.convertToNumber(operand.getValue());
                            count += 1;
                        } catch (ex) {}
                    } else {
                        // matrixes and references: count real numbers only (no strings, no Booleans)
                        this.iterateValues(operand, function (value) {
                            if (_.isNumber(value)) { count += 1; }
                        }, {
                            acceptErrors: true, // do not fail on error codes: =COUNT({1,#VALUE!}) = 1
                            complexRef: true // accept multi-range and multi-sheet references
                        });
                    }
                });
                return count;
            }
        },

        COUNTA: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: function () {
                var count = 0;
                this.iterateOperands(0, function (operand) {
                    switch (operand.getType()) {
                    case 'val':
                        // all values are counted (also zeros, empty strings, FALSE,
                        // error codes, empty parameters): COUNTA(1,,#VALUE!) = 3
                        count += 1;
                        break;
                    case 'mat':
                        // all matrix elements are counted (also zeros, empty strings, FALSE, error codes)
                        count += operand.getMatrix().getElementCount();
                        break;
                    case 'ref':
                        // all filled cells are counted (also error codes)
                        this.iterateValues(operand, function () { count += 1; }, {
                            acceptErrors: true, // count all error codes
                            complexRef: true // accept multi-range and multi-sheet references
                        });
                    }
                });
                return count;
            }
        },

        COUNTBLANK: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'ref',
            resolve: function (ref) {

                var // number of non-blank cells
                    count = 0,
                    // whether empty strings (formula results) are blank (Excel only)
                    emptyStr = this.isOOXML();

                // count all non-blank cells in the reference
                this.iterateValues(ref, function (value) {
                    if (!emptyStr || (value !== '')) { count += 1; }
                }, {
                    acceptErrors: true, // count all error codes
                    complexRef: false // do not accept multi-range and multi-sheet references
                });

                // return number of remaining blank cells
                return ref.getCellCount() - count;
            }
        },

        COUNTIF: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'ref val:any',
            // COUNTIF returns the count of all matching cells in the source range (last parameter of the finalizer)
            resolve: implementFilterAggregation(0, _.identity, function (result, count1, count2) { return count2; })
        },

        COUNTIFS: {
            minParams: 2,
            repeatParams: 2,
            type: 'val'
        },

        COUPDAYBS: {
            minParams: 3,
            maxParams: 4,
            type: 'val'
        },

        COUPDAYS: {
            minParams: 3,
            maxParams: 4,
            type: 'val'
        },

        COUPDAYSNC: {
            minParams: 3,
            maxParams: 4,
            type: 'val'
        },

        COUPNCD: {
            minParams: 3,
            maxParams: 4,
            type: 'val'
        },

        COUPNUM: {
            minParams: 3,
            maxParams: 4,
            type: 'val'
        },

        COUPPCD: {
            minParams: 3,
            maxParams: 4,
            type: 'val'
        },

        COVAR: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        CRITBINOM: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        CSC: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return divide(1, sin(number));
            }
        },

        CSCH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return divide(2 * exp(-number), 1 - exp(-2 * number));
            }
        },

        CUMIPMT: {
            minParams: 6,
            maxParams: 6,
            type: 'val'
        },

        CUMPRINC: {
            minParams: 6,
            maxParams: 6,
            type: 'val'
        },

        CURRENT: {
            formats: 'odf',
            minParams: 0,
            maxParams: 0,
            type: 'any'
        },

        DATE: {
            minParams: 3,
            maxParams: 3,
            type: 'val',
            signature: 'val:int val:int val:int',
            resolve: function (year, month, day) {
                // Excel adds 1900 to all years less than 1900; e.g. 1899 becomes 3799 (TODO: different behavior for ODF?)
                if (year < 1900) { year += 1900; }
                // create a UTC date object (Date object expects zero-based month)
                return makeDate(year, month - 1, day);
            }
        },

        DATEVALUE: {
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        DAVERAGE: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        DAY: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:date',
            resolve: invokeMethod('getUTCDate')
        },

        DAYS: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        DAYS360: {
            minParams: 2,
            maxParams: 3,
            type: 'val'
        },

        DAYSINMONTH: {
            formats: 'odf',
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        DAYSINYEAR: {
            formats: 'odf',
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        DB: {
            minParams: 4,
            maxParams: 5,
            type: 'val'
        },

        DCOUNT: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        DCOUNTA: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        DDB: {
            minParams: 4,
            maxParams: 5,
            type: 'val'
        },

        DDE: {
            minParams: 3,
            maxParams: 4,
            type: 'val'
        },

        DEC2BIN: {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:int val:int',
            resolve: convertDecToBin
        },

        DEC2HEX: {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:int val:int',
            resolve: convertDecToHex
        },

        DEC2OCT: {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:int val:int',
            resolve: convertDecToOct
        },

        DECIMAL: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        DEGREES: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return number / PI_180;
            }
        },

        DELTA: {
            minParams: 1,
            maxParams: 2,
            type: 'val'
        },

        DEVSQ: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // default result (no numbers found in parameters) is the #NUM! error code (not #DIV/0!)
                function finalize(numbers, sum) {
                    if (numbers.length === 0) { throw ErrorCodes.NUM; }
                    var mean = sum / numbers.length;
                    return Utils.getSum(numbers, function (number) { return square(number - mean); });
                }

                // empty parameters count as zero: DEVSQ(1,) = 0.5
                return implementNumericAggregationWithArray(finalize, NUMBER_ITERATOR_OPTIONS.SKIP_MAT_SKIP_REF);
            }())
        },

        DGET: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        DISC: {
            minParams: 4,
            maxParams: 5,
            type: 'val'
        },

        DMAX: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        DMIN: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        DOLLAR: {
            minParams: 1,
            maxParams: 2,
            type: 'val'
        },

        DOLLARDE: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        DOLLARFR: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        DPRODUCT: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        DSTDEV: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        DSTDEVP: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        DSUM: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        DURATION: {
            minParams: 5,
            maxParams: 6,
            type: 'val'
        },

        DURATION_ADD: {
            formats: 'odf',
            minParams: 5,
            maxParams: 6,
            type: 'val'
        },

        DVAR: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        DVARP: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        EASTERSUNDAY: {
            formats: 'odf',
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        EDATE: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:date val:num',
            resolve: function (date, months) {
                // 2nd parameter needs truncation (negative toward zero), 'val:int' in signature would use floor
                // set day in result to the day of the passed original date
                return addMonthsToDate(date, trunc(months), false);
            }
        },

        EFFECT: {
            formats: 'ooxml', // called 'EFFECTIVE' in ODF
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        EFFECTIVE: {
            formats: 'odf', // called 'EFFECT' in OOXML
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        EOMONTH: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:date val:num',
            resolve: function (date, months) {
                // 2nd parameter needs truncation (negative toward zero), 'val:int' in signature would use floor
                // set day in result to the last day in the resulting month
                return addMonthsToDate(date, trunc(months), true);
            }
        },

        ERF: {
            minParams: 1,
            maxParams: 2,
            type: 'val'
        },

        ERFC: {
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        'ERROR.TYPE': {
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        EUROCONVERT: {
            minParams: 3,
            maxParams: 5,
            type: 'val'
        },

        EVEN: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                var result = 2 * ceil(abs(number) / 2);
                return (number < 0) ? -result : result;
            }
        },

        EXACT: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:str',
            resolve: function (value1, value2) {
                return value1 === value2; // exact case-sensitive comparison
            }
        },

        EXP: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: exp
        },

        EXPONDIST: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        FACT: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:int',
            resolve: factorial
        },

        FACTDOUBLE: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:int',
            resolve: (function () {
                var FACTS = [1, 1];
                while (isFinite(_.last(FACTS))) { FACTS.push(FACTS[FACTS.length - 2] * FACTS.length); }
                return function (number) {
                    return (number in FACTS) ? FACTS[number] : ErrorCodes.NUM;
                };
            }())
        },

        FALSE: {
            minParams: 0,
            maxParams: 0,
            type: 'val',
            resolve: _.constant(false)
        },

        FDIST: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        FIND: {
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'val:str val:str val:int',
            resolve: function (searchText, text, index) {
                if (_.isUndefined(index)) { index = 1; }
                this.checkStringIndex(index);
                index -= 1;
                // do not search, if index is too large (needed if searchText is empty)
                if (index > text.length - searchText.length) { throw ErrorCodes.VALUE; }
                // FIND searches case-sensitive
                index = text.indexOf(searchText, index);
                if (index < 0) { throw ErrorCodes.VALUE; }
                return index + 1;
            }
        },

        FINV: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        FISHER: {
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        FISHERINV: {
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        FIXED: {
            minParams: 1,
            maxParams: 3,
            type: 'val'
        },

        FLOOR: {
            minParams: 2,
            maxParams: { def: 2, odf: 3 },
            type: 'val',
            signature: 'val:num val:num val:int',
            resolve: function (number, significance, mode) {
                var result, modulo, minBorder = 1e-16;
                if (_.isUndefined(mode)) {
                    mode = this.isODF() ? 0 : 1;
                }
                if (significance === 0 || (number > 0 && significance < 0)) {
                    throw ErrorCodes.VALUE;
                }
                modulo = abs(significance - (number % significance));
                if (number === 0 || modulo < minBorder) {
                    result = number;
                } else {
                    if (mode === 0 && number < 0) {
                        result = (floor(number / significance) + 1) * significance;
                    } else {
                        result = (floor(number / significance)) * significance;
                    }
                }
                return result;
            }
        },

        FORECAST: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        FORMULA: {
            // real OOXML name: FORMULATEXT
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        FREQUENCY: {
            minParams: 2,
            maxParams: 2,
            type: 'mat'
        },

        FTEST: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        FV: {
            minParams: 3,
            maxParams: 5,
            type: 'val'
        },

        FVSCHEDULE: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        GAMMA: {
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        GAMMADIST: {
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        GAMMAINV: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        GAMMALN: {
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        GAUSS: {
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        GCD: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // returns the GCD of the result and the new number
                function aggregate(result, number) {
                    // round down to integers, fail on negative numbers
                    number = floor(number);
                    if (number < 0) { throw ErrorCodes.NUM; }
                    return gcd(result, number);
                }

                // throw #VALUE! without any number (reference to empty cells)
                function finalize(result, count) {
                    if (count === 0) { throw ErrorCodes.VALUE; }
                    return result;
                }

                // start with 0, needed to force calling aggregate() for all function
                // parameters for integer conversion and check for negative numbers
                // Booleans lead to error: GCD(6,TRUE) = #VALUE!
                // empty parameters are ignored: GCD(6,) = 6
                return implementNumericAggregation(0, aggregate, finalize, NUMBER_ITERATOR_OPTIONS.RCONVERT_ALL_SKIP_EMPTY);
            }())
        },

        GEOMEAN: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // default result (no numbers found in parameters) is the #NUM! error code
                // a single 0 in the parameters results in the #NUM! error code
                function finalize(result, count) {
                    if ((result === 0) || (count === 0)) { throw ErrorCodes.NUM; }
                    return pow(result, 1 / count);
                }

                // empty parameters count as zero: GEOMEAN(1,) = #NUM!
                return implementNumericAggregation(1, multiply, finalize, NUMBER_ITERATOR_OPTIONS.SKIP_MAT_SKIP_REF);
            }())
        },

        GESTEP: {
            minParams: 1,
            maxParams: 2,
            type: 'val'
        },

        GETPIVOTDATA: {
            minParams: 2,
            repeatParams: 2,
            type: 'val'
        },

        GROWTH: {
            minParams: 1,
            maxParams: 4,
            type: 'mat'
        },

        HARMEAN: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // adds reciprocal of the passed number to the intermediate result
                function aggregate(result, number) {
                    // a single 0 in the parameters results in the #NUM! error code
                    if (number === 0) { throw ErrorCodes.NUM; }
                    return result + 1 / number;
                }

                // default result (no numbers found in parameters) is the #N/A error code
                function finalize(result, count) {
                    if (count === 0) { throw ErrorCodes.NA; }
                    return count / result;
                }

                // empty parameters count as zero: GEOMEAN(1,) = #NUM!
                return implementNumericAggregation(0, aggregate, finalize, NUMBER_ITERATOR_OPTIONS.SKIP_MAT_SKIP_REF);
            }())
        },

        HEX2BIN: {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: function (number, length) {
                return convertDecToBin(convertHexToDec(number), length);
            }
        },

        HEX2DEC: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: convertHexToDec
        },

        HEX2OCT: {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: function (number, length) {
                return convertDecToOct(convertHexToDec(number), length);
            }
        },

        HLOOKUP: {
            minParams: 3,
            maxParams: 4,
            type: 'any'
        },

        HOUR: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:date',
            resolve: invokeMethod('getUTCHours')
        },

        HYPERLINK: {
            minParams: 1,
            maxParams: 2,
            type: 'val'
        },

        HYPGEOMDIST: {
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        IF: {
            minParams: 2,
            maxParams: 3,
            type: 'any',
            signature: 'val:bool any any',
            resolve: function (cond, operand1, operand2) {
                return cond ? (_.isUndefined(operand1) ? true : operand1) : (_.isUndefined(operand2) ? false : operand2);
            }
        },

        IFERROR: {
            minParams: 2,
            maxParams: 2,
            type: 'any',
            signature: 'val:any val:any', // always return values (in difference to IF)
            resolve: function (value1, value2) {
                return SheetUtils.isErrorCode(value1) ? value2 : value1;
            }
        },

        IFNA: {
            minParams: 2,
            maxParams: 2,
            type: 'any',
            signature: 'val:any val:any', // always return values (in difference to IF)
            resolve: function (value1, value2) {
                return ErrorCodes.NA.equals(value1) ? value2 : value1;
            }
        },

        IMABS: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: complexAbs
        },

        IMAGINARY: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: _.property('imag')
        },

        IMARGUMENT: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: complexArg
        },

        IMCONJUGATE: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                return new Complex(c.real, -c.imag, c.unit);
            }
        },

        IMCOS: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                return new Complex(cos(c.real) * cosh(c.imag), -sin(c.real) * sinh(c.imag), c.unit);
            }
        },

        IMCOSH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                return new Complex(cosh(c.real) * cos(c.imag), sinh(c.real) * sin(c.imag), c.unit);
            }
        },

        IMCOT: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                var quot = cos(2 * c.real) - cosh(2 * c.imag);
                return new Complex(-sin(2 * c.real) / quot, sinh(2 * c.imag) / quot, c.unit);
            }
        },

        IMCSC: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                var quot = cos(2 * c.real) - cosh(2 * c.imag);
                return new Complex(-2 * sin(c.real) * cosh(c.imag) / quot, 2 * cos(c.real) * sinh(c.imag) / quot, c.unit);
            }
        },

        IMCSCH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                var quot = cosh(2 * c.real) - cos(2 * c.imag);
                return new Complex(2 * sinh(c.real) * cos(c.imag) / quot, -2 * cosh(c.real) * sin(c.imag) / quot, c.unit);
            }
        },

        IMDIV: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:comp val:comp',
            resolve: function (c1, c2) {
                this.checkComplexUnits(c1, c2);
                var quot = square(c2.real) + square(c2.imag);
                // division of complex number by zero results in #NUM! instead of #DIV/0!
                // -> do not use the divide() helper function here, invalid coefficients
                // in the returned complex number will be resolved to #NUM! automatically
                return new Complex(
                    (c1.real * c2.real + c1.imag * c2.imag) / quot,
                    (c1.imag * c2.real - c1.real * c2.imag) / quot,
                    c1.unit
                );
            }
        },

        IMEXP: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                var cabs = exp(c.real);
                return new Complex(cabs * cos(c.imag), cabs * sin(c.imag), c.unit);
            }
        },

        IMLN: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                return new Complex(log(complexAbs(c)), complexArg(c), c.unit);
            }
        },

        IMLOG10: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                return new Complex(log(complexAbs(c)) / LN10, complexArg(c) / LN10, c.unit);
            }
        },

        IMLOG2: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                return new Complex(log(complexAbs(c)) / LN2, complexArg(c) / LN2, c.unit);
            }
        },

        IMPOWER: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:comp val:num',
            resolve: complexPower
        },

        IMPRODUCT: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: implementComplexAggregation(function (c1, c2) {
                return new Complex(c1.real * c2.real - c1.imag * c2.imag, c1.real * c2.imag + c1.imag * c2.real);
            })
        },

        IMREAL: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: _.property('real')
        },

        IMSEC: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                var quot = cos(2 * c.real) + cosh(2 * c.imag);
                return new Complex(2 * cos(c.real) * cosh(c.imag) / quot, 2 * sin(c.real) * sinh(c.imag) / quot, c.unit);
            }
        },

        IMSECH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                var quot = cosh(2 * c.real) + cos(2 * c.imag);
                return new Complex(2 * cosh(c.real) * cos(c.imag) / quot, -2 * sinh(c.real) * sin(c.imag) / quot, c.unit);
            }
        },

        IMSIN: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                return new Complex(sin(c.real) * cosh(c.imag), cos(c.real) * sinh(c.imag), c.unit);
            }
        },

        IMSINH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                return new Complex(sinh(c.real) * cos(c.imag), cosh(c.real) * sin(c.imag), c.unit);
            }
        },

        IMSQRT: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                return complexPower(c, 0.5);
            }
        },

        IMSUB: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:comp val:comp',
            resolve: function (c1, c2) {
                this.checkComplexUnits(c1, c2);
                return new Complex(c1.real - c2.real, c1.imag - c2.imag, c1.unit);
            }
        },

        IMSUM: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: implementComplexAggregation(function (c1, c2) {
                return new Complex(c1.real + c2.real, c1.imag + c2.imag);
            })
        },

        IMTAN: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                var quot = cos(2 * c.real) + cosh(2 * c.imag);
                return new Complex(sin(2 * c.real) / quot, sinh(2 * c.imag) / quot, c.unit);
            }
        },

        INDEX: {
            minParams: 2,
            maxParams: 4,
            type: 'any'
        },

        INDIRECT: {
            minParams: 1,
            maxParams: 2,
            type: 'ref'
        },

        INFO: {
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        INT: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:int',
            resolve: _.identity // always rounds down, e.g. INT(-1.5) = -2
        },

        INTERCEPT: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        INTRATE: {
            minParams: 4,
            maxParams: 5,
            type: 'val'
        },

        IPMT: {
            minParams: 4,
            maxParams: 6,
            type: 'val'
        },

        IRR: {
            minParams: 1,
            maxParams: 2,
            type: 'val'
        },

        ISBLANK: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: _.isNull // error codes result in FALSE
        },

        ISERR: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: function (value) {
                return SheetUtils.isErrorCode(value) && !ErrorCodes.NA.equals(value);
            }
        },

        ISERROR: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: SheetUtils.isErrorCode
        },

        ISEVEN: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return (floor(abs(number)) % 2) === 0;
            }
        },

        ISEVEN_ADD: {
            formats: 'odf',
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        ISFORMULA: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'ref',
            resolve: function (ref) {

                var // parameter must be a single range
                    range = ref.getSingleRange(),
                    // use the first cell in the range (TODO: matrix context)
                    address = range.start;

                // accept reference to own cell (no circular reference error)
                if ((range.sheet1 === this.getRefSheet()) && _.isEqual(address, this.getRefAddress())) {
                    return true; // do not check the cell, a new formula has not been inserted yet
                }

                // getCellFormula() returns null for value cells
                return _.isString(this.getCellFormula(range.sheet1, address));
            }
        },

        ISLEAPYEAR: {
            formats: 'odf',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:date',
            resolve: function (date) {
                return isLeapYear(date.getUTCFullYear());
            }
        },

        ISLOGICAL: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: _.isBoolean // error codes and empty cells result in FALSE
        },

        ISNA: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: _.bind(ErrorCodes.NA.equals, ErrorCodes.NA)
        },

        ISNONTEXT: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: function (value) {
                return !_.isString(value); // error codes and empty cells result in TRUE
            }
        },

        ISNUMBER: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: _.isNumber // error codes and empty cells result in FALSE
        },

        ISODD: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return (floor(abs(number)) % 2) !== 0;
            }
        },

        ISODD_ADD: {
            formats: 'odf',
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        ISOWEEKNUM: {
            formats: 'ooxml',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:date',
            resolve: function (date) {
                return getWeekNumOOXML(date, 21);
            }
        },

        ISPMT: {
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        ISREF: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'any',
            resolve: invokeMethod('isReference') // error codes result in FALSE
        },

        ISTEXT: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: _.isString // error codes and empty cells result in FALSE
        },

        JIS: {
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        KURT: {
            minParams: 1,
            type: 'val'
        },

        LARGE: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        LCM: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // updates the LCM in the passed result according to the passed number
                function aggregate(result, number) {
                    // round down to integers, fail on negative numbers
                    number = floor(number);
                    if (number < 0) { throw ErrorCodes.NUM; }
                    return ((result === 0) || (number === 0)) ? 0 : (result * number / gcd(number, result));
                }

                // throw #VALUE! without any number (reference to empty cells)
                function finalize(result, count) {
                    if (count === 0) { throw ErrorCodes.VALUE; }
                    return result;
                }

                // start with 1, needed to force calling aggregate() for all function
                // parameters for integer conversion and check for negative numbers
                // Booleans lead to error: LCM(6,TRUE) = #VALUE!
                // empty parameters are ignored: LCM(6,) = 6 while LCM(6,0) = 0
                return implementNumericAggregation(1, aggregate, finalize, NUMBER_ITERATOR_OPTIONS.RCONVERT_ALL_SKIP_EMPTY);
            }())
        },

        LEFT: {
            altNames: 'LEFTB', // different behavior for multi-byte character sets
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: function (text, count) {
                if (_.isUndefined(count)) { count = 1; }
                this.checkStringLength(count);
                return (count === 0) ? text : text.substr(0, count);
            }
        },

        LEN: {
            altNames: 'LENB', // different behavior for multi-byte character sets
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: _.property('length')
        },

        LINEST: {
            minParams: 1,
            maxParams: 4,
            type: 'mat'
        },

        LN: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: log
        },

        LOG: {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: function (number, base) {
                if (_.isUndefined(base)) { base = 10; }
                return log(number) / log(base);
            }
        },

        LOG10: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return log(number) / LN10;
            }
        },

        LOGEST: {
            minParams: 1,
            maxParams: 4,
            type: 'mat'
        },

        LOGINV: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        LOGNORMDIST: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        LOOKUP: {
            minParams: 2,
            maxParams: 3,
            type: 'any'
        },

        LOWER: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: invokeMethod('toLowerCase')
        },

        MATCH: {
            minParams: 2,
            maxParams: 3,
            type: 'val'
        },

        MAX: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as zero: MAX(-1,) = 0
            resolve: implementMinMaxAggregation('max', NUMBER_ITERATOR_OPTIONS.SKIP_MAT_SKIP_REF)
        },

        MAXA: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as zero: MAXA(-1,) = 0
            resolve: implementMinMaxAggregation('max', NUMBER_ITERATOR_OPTIONS.SKIP_MAT_ZERO_REF)
        },

        MDETERM: {
            minParams: 1,
            maxParams: 1,
            type: 'mat'
        },

        MDURATION: {
            minParams: 5,
            maxParams: 6,
            type: 'val'
        },

        MEDIAN: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // default result (no numbers found in parameters) is the #NUM! error code
                function finalize(numbers) {
                    var count = numbers.length;
                    if (count === 0) { throw ErrorCodes.NUM; }
                    numbers = _.sortBy(numbers);
                    // even array length: return arithmetic mean of both numbers in the middle of the array
                    return (count % 2 === 0) ? ((numbers[count / 2 - 1] + numbers[count / 2]) / 2) : numbers[(count - 1) / 2];
                }

                // create and return the resolver function to be assigned to the 'resolve' option of a function descriptor
                return implementNumericAggregationWithArray(finalize, NUMBER_ITERATOR_OPTIONS.SKIP_MAT_SKIP_REF);
            }())
        },

        MID: {
            altNames: 'MIDB', // different behavior for multi-byte character sets
            minParams: 3,
            maxParams: 3,
            type: 'val',
            signature: 'val:str val:int val:int',
            resolve: function (text, start, count) {
                this.checkStringIndex(start);
                this.checkStringLength(count);
                return text.substr(start - 1, count);
            }
        },

        MIN: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as zero: MIN(1,) = 0
            resolve: implementMinMaxAggregation('min', NUMBER_ITERATOR_OPTIONS.SKIP_MAT_SKIP_REF)
        },

        MINA: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as zero: MINA(1,) = 0
            resolve: implementMinMaxAggregation('min', NUMBER_ITERATOR_OPTIONS.SKIP_MAT_ZERO_REF)
        },

        MINUTE: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:date',
            resolve: invokeMethod('getUTCMinutes')
        },

        MINVERSE: {
            minParams: 1,
            maxParams: 1,
            type: 'mat'
        },

        MIRR: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        MMULT: {
            minParams: 2,
            maxParams: 2,
            type: 'mat',
            signature: 'mat:num mat:num',
            resolve: function (matrix1, matrix2) {

                // width of first matrix must match height of second matrix
                if (matrix1.getColCount() !== matrix2.getRowCount()) { throw ErrorCodes.VALUE; }

                // calculate all elements of the result matrix
                return new Matrix(matrix1.getRowCount(), matrix2.getColCount(), function (row, col) {
                    var elem = 0;
                    for (var i = 0, l = matrix1.getColCount(); i < l; i += 1) {
                        elem += (matrix1.getElement(row, i) * matrix2.getElement(i, col));
                    }
                    return elem;
                });
            }
        },

        MOD: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: modulo
        },

        MODE: {
            altNames: { ooxml: 'MODE.SNGL' },
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // finds the most used number in all collected numbers
                function finalize (numbers) {

                    // default result (all parameters skipped) is the #N/A error code
                    if (numbers.length === 0) { throw ErrorCodes.NA; }

                    var // reduce numbers to their counts (maps each distinct number to its count)
                        counts = _.countBy(numbers),
                        // find the highest count
                        maxCount = _.reduce(counts, function (max, num) { return Math.max(max, num); }, 0),
                        // a map that contains all numbers occuring the most as key
                        map = {};

                    // no duplicate numbers: throw the #N/A error code
                    if (maxCount === 1) { throw ErrorCodes.NA; }

                    // insert each number into the map that occurs the most
                    _.each(counts, function (count, number) {
                        if (count === maxCount) { map[number] = true; }
                    });

                    // find the first number in the original array that occurs the most
                    // e.g.: =MODE(3,3,4,4) results in 3; but: =MODE(4,4,3,3) results in 4
                    return _.find(numbers, function (number) { return number in map; });
                }

                // create and return the resolver function to be assigned to the 'resolve' option of a function descriptor
                return implementNumericAggregationWithArray(finalize, {
                    valMode: 'exact', // value operands: exact match for numbers (no strings, no Booleans)
                    matMode: 'skip', // matrix operands: skip strings and Booleans
                    refMode: 'exact', // reference operands: exact match for numbers (no strings, no Booleans)
                    emptyParam: true, // empty parameters result in #VALUE! error
                    complexRef: false // do not accept multi-range and multi-sheet references
                });
            }())
        },

        MONTH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:date',
            resolve: function (date) {
                return date.getUTCMonth() + 1; // Date class returns zero-based month
            }
        },

        MONTHS: {
            formats: 'odf',
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        MROUND: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: mround
        },

        MULTINOMIAL: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // use the same simple implementation as Excel: if the sum of all numbers exceeds
                // the value 170, getting factorial in the numerator will fail with a #NUM! error,
                // although the final result of the function would be small enough
                function finalize(numbers) {
                    if (numbers.length === 0) { throw ErrorCodes.VALUE; }
                    var sum = 0, denom = 1;
                    _.each(numbers, function (number) {
                        sum += floor(number);
                        denom *= factorial(number);
                    });
                    return factorial(sum) / denom;
                }

                // create and return the resolver function to be assigned to the 'resolve' option of a function descriptor
                return implementNumericAggregationWithArray(finalize, NUMBER_ITERATOR_OPTIONS.RCONVERT_ALL_SKIP_EMPTY);
            }())
        },

        MUNIT: {
            minParams: 1,
            maxParams: 1,
            type: 'mat',
            signature: 'val:int',
            resolve: function (dim) {
                if (dim < 1) { throw ErrorCodes.VALUE; }
                if (dim > Math.min(TokenUtils.MAX_MATRIX_ROW_COUNT, TokenUtils.MAX_MATRIX_COL_COUNT)) { throw TokenUtils.UNSUPPORTED_ERROR; }
                return new Matrix(dim, dim, function (row, col) {
                    return (row === col) ? 1 : 0;
                });
            }
        },

        N: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val',
            resolve: function (value) {
                return _.isNumber(value) ? value : (value === true) ? 1 : 0;
            }
        },

        NA: {
            minParams: 0,
            maxParams: 0,
            type: 'val',
            resolve: _.constant(ErrorCodes.NA)
        },

        NEG: {
            formats: 'odf',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) { return -number; }
        },

        NEGBINOMDIST: {
            // OOXML: new version NEGBINOM.DIST
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        NETWORKDAYS: {
            minParams: 2,
            maxParams: 3,
            type: 'val'
        },

        NOMINAL: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        NORMDIST: {
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        NORMINV: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        NORMSDIST: {
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        NORMSINV: {
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        NOT: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:bool',
            resolve: logicalNot
        },

        NOW: {
            minParams: 0,
            maxParams: 0,
            type: 'val',
            resolve: function() {
                var now = new Date();
                // create a UTC date object representing the current local date/time
                return makeDateTime(now.getFullYear(), now.getMonth(), now.getDate(), now.getHours(), now.getMinutes(), now.getSeconds(), now.getMilliseconds());
            }
        },

        NPER: {
            minParams: 3,
            maxParams: 5,
            type: 'val'
        },

        NPV: {
            minParams: 2,
            type: 'val'
        },

        NUMBERVALUE: {
            minParams: 1,
            maxParams: 3,
            type: 'val'
        },

        OCT2BIN: {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: function (number, length) {
                return convertDecToBin(convertOctToDec(number), length);
            }
        },

        OCT2DEC: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: convertOctToDec
        },

        OCT2HEX: {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: function (number, length) {
                return convertDecToHex(convertOctToDec(number), length);
            }
        },

        ODD: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                var result = 2 * ceil((abs(number) + 1) / 2) - 1;
                return (number < 0) ? -result : result;
            }
        },

        ODDFPRICE: {
            minParams: 8,
            maxParams: 9,
            type: 'val'
        },

        ODDFYIELD: {
            minParams: 8,
            maxParams: 9,
            type: 'val'
        },

        ODDLPRICE: {
            minParams: 7,
            maxParams: 8,
            type: 'val'
        },

        ODDLYIELD: {
            minParams: 7,
            maxParams: 8,
            type: 'val'
        },

        OFFSET: {
            minParams: 3,
            maxParams: 5,
            type: 'ref',
            signature: 'ref val:int val:int val:int val:int',
            resolve: function (ref, rows, cols, height, width) {

                var // reference must contain a single range address
                    range = ref.getSingleRange({ valueError: true });

                // apply offset
                range.start[0] += cols;
                range.start[1] += rows;
                range.end[0] += cols;
                range.end[1] += rows;

                // modify size (but not if 'height' or 'width' parameters exist but are empty)
                if (!this.isMissingOrEmptyOperand(4)) { range.end[0] = range.start[0] + width - 1; }
                if (!this.isMissingOrEmptyOperand(3)) { range.end[1] = range.start[1] + height - 1; }

                // check that the range does not left the sheet boundaries
                if (this.getModel().isValidRange(range)) { return new Reference(range); }
                throw ErrorCodes.REF;
            }
        },

        OR: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as FALSE (but do not skip to catch empty-only parameters)
            resolve: implementLogicalAggregation(false, logicalOr, BOOLEAN_ITERATOR_OPTIONS.SKIP_MAT_SKIP_REF)
        },

        PEARSON: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        PERCENTRANK: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        PERCENTILE: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        PERMUT: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        PERMUTATIONA: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        PHI: {
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        PI: {
            minParams: 0,
            maxParams: 0,
            type: 'val',
            resolve: _.constant(PI)
        },

        PMT: {
            minParams: 3,
            maxParams: 5,
            type: 'val'
        },

        POISSON: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        POWER: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: power
        },

        PPMT: {
            minParams: 4,
            maxParams: 6,
            type: 'val'
        },

        PRICE: {
            minParams: 6,
            maxParams: 7,
            type: 'val'
        },

        PRICEDISC: {
            minParams: 4,
            maxParams: 5,
            type: 'val'
        },

        PRICEMAT: {
            minParams: 5,
            maxParams: 6,
            type: 'val'
        },

        PROB: {
            minParams: 3,
            maxParams: 4,
            type: 'val'
        },

        PRODUCT: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // default result is 0, if no numbers have been found, e.g. =PRODUCT(,)
                function finalize(result, count) { return (count === 0) ? 0 : result; }

                // empty parameters are ignored: PRODUCT(1,) = 1
                return implementNumericAggregation(1, multiply, finalize, NUMBER_ITERATOR_OPTIONS.SKIP_MAT_SKIP_REF_SKIP_EMPTY);
            }())
        },

        PROPER: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: function (text) {
                // TODO: the reg-exp does not include all characters that count as letters
                return text.replace(/[A-Za-z\xAA\xB5\xBA\xC0-\xD6\xD8-\xF6\xF8-\u02B8\u02BB-\u02C1\u02D0-\u02D1\u02E0-\u02E4\u02EE\u0370-\u0373\u0376-\u0377\u037A-\u037D\u0386\u0388-\u038A\u038C\u038E-\u03A1\u03A3-\u03F5\u03F7-\u0481\u048A-\u0523]+/g, Utils.capitalize);
            }
        },

        PV: {
            minParams: 3,
            maxParams: 5,
            type: 'val'
        },

        QUARTILE: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        QUOTIENT: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: function (number1, number2) {
                return trunc(divide(number1, number2));
            }
        },

        RADIANS: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return number * PI_180;
            }
        },

        RAND: {
            minParams: 0,
            maxParams: 0,
            type: 'val',
            resolve: Math.random
        },

        RANDBETWEEN: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: function (min, max) {
                min = ceil(min);
                max = floor(max);
                if (max < min) { throw ErrorCodes.NUM; }
                return floor(Math.random() * (max - min + 1)) + min;
            }
        },

        RANK: {
            minParams: 2,
            maxParams: 3,
            type: 'val'
        },

        RATE: {
            minParams: 3,
            maxParams: 6,
            type: 'val'
        },

        RECEIVED: {
            minParams: 4,
            maxParams: 5,
            type: 'val'
        },

        REPLACE: {
            minParams: 4,
            maxParams: 4,
            type: 'val',
            signature: 'val:str val:int val:int val:str',
            resolve: function (oldText, start, count, newText) {
                this.checkStringIndex(start);
                this.checkStringLength(count);
                start -= 1;
                return Utils.replaceSubString(oldText, start, start + count, newText);
            }
        },

        REPT: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: function (text, repeat) {
                this.checkStringLength(text.length * repeat);
                return (text.length === 0) ? '' : Utils.repeatString(text, repeat);
            }
        },

        RIGHT: {
            altNames: 'RIGHTB', // different behavior for multi-byte character sets
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: function (text, count) {
                if (_.isUndefined(count)) { count = 1; }
                this.checkStringLength(count);
                return (count === 0) ? text : text.substr(-count);
            }
        },

        ROMAN: {
            minParams: 1,
            maxParams: 2,
            type: 'val'
        },

        ROT13: {
            formats: 'odf',
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        ROUND: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:int',
            resolve: function (number, digits) {
                var mult = pow(10, -digits);
                return mround(number, number < 0 ? -mult : mult);
            }
        },

        ROUNDDOWN: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:int',
            resolve: function (number, digits) {
                var mult = pow(10, -digits),
                    smult = (number < 0) ? -mult : mult,
                    result = mround(number, smult);
                return ((result !== number) && (abs(result) > abs(number))) ? (result - smult) : result;
            }
        },

        ROUNDUP: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:int',
            resolve: function (number, digits) {
                var mult = pow(10, -digits),
                    smult = (number < 0) ? -mult : mult,
                    result = mround(number, smult);
                return ((result !== number) && (abs(result) < abs(number))) ? (result + smult) : result;
            }
        },

        ROW: {
            minParams: 0,
            maxParams: 1,
            type: 'val',
            signature: 'ref',
            resolve: function (ref) {
                // use reference address, if parameter is missing
                var address = _.isUndefined(ref) ? this.getRefAddress() : ref.getSingleRange().start;
                // function returns one-based row index
                return address[1] + 1;
            }
        },

        ROWS: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'any',
            resolve: function (operand) {
                switch (operand.getType()) {
                case 'val':
                    var value = operand.getValue();
                    return SheetUtils.isErrorCode(value) ? value : 1;
                case 'mat':
                    return operand.getMatrix().getRowCount();
                case 'ref':
                    return SheetUtils.getRowCount(operand.getReference().getSingleRange());
                default:
                    throw 'fatal';
                }
            }
        },

        RRI: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        RSQ: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        SEARCH: {
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'val:str val:str val:int',
            resolve: function (searchText, text, index) {

                // prepare and check the passed index
                if (_.isUndefined(index)) { index = 1; }
                this.checkStringIndex(index);
                index -= 1;

                // return index, if searchText is empty or a 'match all' selector
                if (/^\**$/.test(searchText)) {
                    if (text.length <= index) { throw ErrorCodes.VALUE; }
                    return index + 1;
                }

                // shorten the passed text according to the start index
                text = text.substr(index);

                var // create the regular expression
                    regExp = this.convertPatternToRegExp(searchText),
                    // find first occurrence of the pattern
                    matches = regExp.exec(text);

                // nothing found: throw #VALUE! error code
                if (!_.isArray(matches)) { throw ErrorCodes.VALUE; }

                // return the correct index of the search text
                return matches.index + index + 1;
            }
        },

        SECOND: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:date',
            resolve: invokeMethod('getUTCSeconds')
        },

        SEC: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return divide(1, cos(number));
            }
        },

        SECH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return divide(2 * exp(-number), 1 + exp(-2 * number));
            }
        },

        SERIESSUM: {
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        SHEET: {
            minParams: 0,
            maxParams: 1,
            type: 'val',
            signature: 'any', // string (sheet name) or reference
            resolve: function (operand) {
                // use reference sheet, if parameter is missing
                if (_.isUndefined(operand)) { return this.getRefSheet() + 1; }
                // use sheet index of a reference
                if (operand.isReference()) { return operand.getReference().getSingleRange().sheet1 + 1; }
                // convert values to strings (sheet name may look like a number or Boolean)
                if (operand.isValue()) {
                    var sheet = this.getModel().getSheetIndex(this.convertToString(operand.getValue()));
                    return (sheet < 0) ? ErrorCodes.NA : (sheet + 1);
                }
                // always fail for matrix operands, regardless of context
                return ErrorCodes.NA;
            }
        },

        SHEETS: {
            minParams: 0,
            maxParams: 1,
            type: 'val',
            signature: 'ref',
            resolve: function (ref) {
                // return total number of sheets in document, if parameter is missing
                if (_.isUndefined(ref)) { return this.getModel().getSheetCount(); }
                // parameter must be a single range, but may point to multiple sheets
                var range = ref.getSingleRange({ multiSheet: true });
                return range.sheet2 - range.sheet1 + 1;
            }
        },

        SIGN: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return (number < 0) ? -1 : (number > 0) ? 1 : 0;
            }
        },

        SIN: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: sin
        },

        SINH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: sinh
        },

        SKEW: {
            minParams: 1,
            type: 'val'
        },

        SLN: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        SLOPE: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        SMALL: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        SQRT: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: sqrt
        },

        SQRTPI: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return sqrt(PI * number);
            }
        },

        STANDARDIZE: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        STDEV: {
            altNames: { ooxml: 'STDEV.S' },
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as 0: STDEV(1,) = 0.707
            resolve: implementStdDevAggregation('dev', false, NUMBER_ITERATOR_OPTIONS.SKIP_MAT_SKIP_REF)
        },

        STDEVA: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as 0: STDEVA(1,) = 0.707
            resolve: implementStdDevAggregation('dev', false, NUMBER_ITERATOR_OPTIONS.SKIP_MAT_ZERO_REF)
        },

        STDEVP: {
            altNames: { ooxml: 'STDEV.P' },
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as 0: STDEVP(1,) = 0.5
            resolve: implementStdDevAggregation('dev', true, NUMBER_ITERATOR_OPTIONS.SKIP_MAT_SKIP_REF)
        },

        STDEVPA: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as 0: STDEVPA(1,) = 0.5
            resolve: implementStdDevAggregation('dev', true, NUMBER_ITERATOR_OPTIONS.SKIP_MAT_ZERO_REF)
        },

        STEYX: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        STYLE: {
            minParams: 1,
            maxParams: 3,
            type: 'val'
        },

        SUBSTITUTE: {
            minParams: 3,
            maxParams: 4,
            type: 'val',
            signature: 'val:str val:str val:str val:int',
            resolve: function (text, replaceText, newText, index) {

                // check passed index first (one-based index)
                if (_.isNumber(index) && (index <= 0)) { throw ErrorCodes.VALUE; }

                // do nothing, if the text to be replaced is empty
                if (replaceText.length === 0) { return text; }

                var // the regular expression that will find the text(s) to be replaced (SUBSTITUTE works case-sensitive)
                    regExp = new RegExp(_.escapeRegExp(replaceText), 'g');

                // replace all occurences
                if (_.isUndefined(index)) {
                    return text.replace(regExp, newText);
                }

                // replace the specified occurence of the text
                var splits = text.split(regExp);
                if (index < splits.length) {
                    splits[index - 1] += newText + splits[index];
                    splits.splice(index, 1);
                    return splits.join(replaceText);
                }

                // index too large: return original text
                return text;
            }
        },

        SUBTOTAL: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        SUM: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters are zero and can be ignored
            resolve: implementNumericAggregation(0, add, _.identity, NUMBER_ITERATOR_OPTIONS.SKIP_MAT_SKIP_REF_SKIP_EMPTY)
        },

        SUMIF: {
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'ref val:any ref',
            resolve: implementFilterAggregation(0, add, _.identity)
        },

        SUMIFS: {
            minParams: 3,
            repeatParams: 2,
            type: 'val'
        },

        SUMPRODUCT: {
            minParams: 1,
            type: 'val'
        },

        SUMSQ: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // adds the square of the number to the sum
                function aggregate(result, number) { return result + square(number); }

                // empty parameters are zero and can be ignored
                return implementNumericAggregation(0, aggregate, _.identity, NUMBER_ITERATOR_OPTIONS.SKIP_MAT_SKIP_REF_SKIP_EMPTY);
            }())
        },

        SUMX2MY2: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        SUMX2PY2: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        SUMXMY2: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        SYD: {
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        T: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val',
            resolve: function (value) {
                return _.isString(value) ? value : '';
            }
        },

        TAN: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: tan
        },

        TANH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: tanh
        },

        TBILLEQ: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        TBILLPRICE: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        TBILLYIELD: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        TDIST: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        TEXT: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        TIME: {
            minParams: 3,
            maxParams: 3,
            type: 'val',
            signature: 'val:int val:int val:int',
            resolve: function (hour, minute, second) {

                var // the time value, as fraction of a day
                    time = (hour / 24 + minute / 1440 + second / 86400) % 1;

                // must not result in a negative value
                if (time < 0) { throw ErrorCodes.NUM; }

                // convert to a date object, using the document's current null date
                return this.convertToDate(time);
            }
        },

        TIMEVALUE: {
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        TINV: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        TODAY: {
            minParams: 0,
            maxParams: 0,
            type: 'val',
            resolve: function() {
                var now = new Date();
                // create a UTC date object representing the current local date (but not the time)
                return makeDate(now.getFullYear(), now.getMonth(), now.getDate());
            }
        },

        TRANSPOSE: {
            minParams: 1,
            maxParams: 1,
            type: 'mat',
            signature: 'mat:any',
            resolve: function (matrix) {
                return new Matrix(matrix.getColCount(), matrix.getRowCount(), function (row, col) {
                    return matrix.getElement(col, row);
                });
            }
        },

        TREND: {
            minParams: 1,
            maxParams: 4,
            type: 'mat'
        },

        TRIM: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: function (text) {
                // TRIM removes space characters (U+0020) only, no other white-space or control characters
                return text.replace(/^ +| +$/, '').replace(/  +/g, ' ');
            }
        },

        TRIMMEAN: {
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        TRUE: {
            minParams: 0,
            maxParams: 0,
            type: 'val',
            resolve: _.constant(true)
        },

        TRUNC: {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:int',
            resolve: function (number, digits) {

                // nothing to do for zeros
                if (number === 0) { return 0; }

                // default value for missing digits
                digits = digits || 0;
                // simpler implementation for integer truncation
                if (digits === 0) { return trunc(number); }

                var // absolute number
                    absNum = abs(number),
                    // exponent of the passed number
                    exp10 = floor(log(absNum) / LN10),
                    // mantissa of the passed number
                    mant = absNum / pow(10, exp10);

                // ignore the exponent when truncating the mantissa to the specified digits
                digits += exp10;
                // truncating mantissa (which is always less than 10) to entire tens results in zero
                if (digits < 0) { return 0; }
                // truncating to more than 14 decimal places does not change the passed number
                if (digits >= 14) { return number; }
                // truncate and construct the result
                return ((number < 0) ? -1 : 1) * floor(mant * pow(10, digits)) * pow(10, exp10 - digits);
            }
        },

        TTEST: {
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        TYPE: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any', // resolve cell references to values, accept error codes
            resolve: function (value) {
                // array literals always result in 64 regardless of their contents
                if (this.getOperand(0).isMatrix()) { return 64; }
                // determine type of value parameters and cell contents
                if (_.isNumber(value)) { return 1; }
                if (_.isString(value)) { return 2; }
                if (_.isBoolean(value)) { return 4; }
                if (SheetUtils.isErrorCode(value)) { return 16; }
                // default (reference to empty cell, or cell not fetched from server)
                return 1;
            }
        },

        UNICHAR: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:int',
            resolve: function (value) {
                if ((value < 1) || (value > 65535)) { throw ErrorCodes.VALUE; }
                var char = String.fromCharCode(value);
                if (/[\ud800-\udfff\ufdd0-\ufdef\ufffe\uffff]/.test(char)) { throw ErrorCodes.NA; }
                return char;
            }
        },

        UNICODE: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: function (value) {
                return (value.length === 0) ? ErrorCodes.VALUE : value.charCodeAt(0);
            }
        },

        UPPER: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: invokeMethod('toUpperCase')
        },

        VALUE: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: function (text) {
                // Do not use 'val:num' as signature, this would convert Booleans to numbers automatically.
                // The VALUE function does not accept Booleans (the text "TRUE" cannot be converted to a number).
                // Therefore, use convertToNumber() explicitly on a text argument which will throw with Booleans.
                return this.convertToNumber(text);
            }
        },

        VAR: {
            altNames: { ooxml: 'VAR.S' },
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as 0: VAR(1,) = 0.5
            resolve: implementStdDevAggregation('var', false, NUMBER_ITERATOR_OPTIONS.SKIP_MAT_SKIP_REF)
        },

        VARA: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as 0: VARA(1,) = 0.5
            resolve: implementStdDevAggregation('var', false, NUMBER_ITERATOR_OPTIONS.SKIP_MAT_ZERO_REF)
        },

        VARP: {
            altNames: { ooxml: 'VAR.P' },
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as 0: VARP(1,) = 0.25
            resolve: implementStdDevAggregation('var', true, NUMBER_ITERATOR_OPTIONS.SKIP_MAT_SKIP_REF)
        },

        VARPA: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as 0: VARPA(1,) = 0.25
            resolve: implementStdDevAggregation('var', true, NUMBER_ITERATOR_OPTIONS.SKIP_MAT_ZERO_REF)
        },

        VDB: {
            minParams: 5,
            maxParams: 7,
            type: 'val'
        },

        VLOOKUP: {
            minParams: 3,
            maxParams: 4,
            type: 'any'
        },

        WEEKDAY: {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:date val:int',
            resolve: function (date, mode) {

                var // weekday index, from 0 to 6, starting at Sunday
                    day = date.getUTCDay();

                // missing parameter defaults to 1 (but: empty parameter defaults to 0)
                switch (_.isUndefined(mode) ? 1 : mode) {
                case 1:
                    return day + 1; // 1 to 7, starting at Sunday
                case 2:
                    return (day === 0) ? 7 : day; // 1 to 7, starting at Monday
                case 3:
                    return (day + 6) % 7; // 0 to 6, starting at Monday
                }

                // other modes result in #NUM! error
                throw ErrorCodes.NUM;
            }
        },

        WEEKNUM: {
            minParams: { def: 1, odf: 2 },
            maxParams: 2,
            type: 'val',
            signature: 'val:date val:int',
            resolve: { def: getWeekNumOOXML, odf: getWeekNumODF }
        },

        WEEKNUM_ADD: {
            formats: 'odf', // ODF's emulation of OOXML's WEEKNUM
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:date val:int',
            resolve: getWeekNumOOXML
        },

        WEEKS: {
            formats: 'odf',
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        WEEKSINYEAR: {
            formats: 'odf',
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        WEIBULL: {
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        WORKDAY: {
            minParams: 2,
            maxParams: 3,
            type: 'val'
        },

        XIRR: {
            minParams: 2,
            maxParams: 3,
            type: 'val'
        },

        XNPV: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        XOR: {
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as FALSE (but do not skip to catch empty-only parameters)
            resolve: implementLogicalAggregation(false, logicalXor, BOOLEAN_ITERATOR_OPTIONS.SKIP_MAT_SKIP_REF)
        },

        YEAR: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:date',
            resolve: invokeMethod('getUTCFullYear')
        },

        YEARFRAC: {
            minParams: 2,
            maxParams: 3,
            type: 'val'
        },

        YEARS: {
            formats: 'odf',
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        YIELD: {
            minParams: 6,
            maxParams: 7,
            type: 'val'
        },

        YIELDDISC: {
            minParams: 4,
            maxParams: 5,
            type: 'val'
        },

        YIELDMAT: {
            minParams: 5,
            maxParams: 6,
            type: 'val'
        },

        ZTEST: {
            minParams: 3,
            maxParams: 3,
            type: 'val'
        }
    };

    // static class Operators =================================================

    var Operators = { OPERATORS: OPERATORS, FUNCTIONS: FUNCTIONS };

    /**
     * Returns a cached map with descriptors for all supported operators and
     * functions. Specific settings in the original descriptors will already be
     * resolved for the specified file format.
     *
     * @param {String} fileFormat
     *  The identifier of the file format.
     *
     * @returns {Object}
     *  A map containing descriptors with all supported operators (map key
     *  'OPERATORS') and functions (map key 'FUNCTIONS').
     */
    Operators.getDescriptors = _.memoize(function (fileFormat) {

        var // the resulting operator/function descriptors
            result = {};

        // process all raw descriptors depending on the file format
        _.each([OPERATORS, FUNCTIONS], function (DESCRIPTORS, functions) {

            var // the map for fucntions/operators
                descriptors = result[functions ? 'FUNCTIONS' : 'OPERATORS'] = {};

            // process all raw descriptors depending on the file format
            _.each(DESCRIPTORS, function (DESCRIPTOR, opName) {

                // skip descriptor, if not supported by the current file format
                var formats = Utils.getTokenListOption(DESCRIPTOR, 'formats');
                if (_.isArray(formats) && !_.contains(formats, fileFormat)) { return; }

                // create a new entry in the returned map
                var descriptor = descriptors[opName] = {};

                // copy all properties of the raw descriptor (pick format specific values)
                _.each(DESCRIPTOR, function (value, propName) {
                    descriptor[propName] = (!_.isObject(value) || _.isArray(value) || _.isFunction(value)) ? value : Utils.getOption(value, fileFormat, value.def);
                });

                // convert signature strings to arrays
                descriptor.signature = Utils.getTokenListOption(descriptor, 'signature', []);

                // insert map entries for alternative names
                descriptor.altNames = Utils.getTokenListOption(descriptor, 'altNames', []);
                _.each(descriptor.altNames, function (altName) {
                    descriptors[altName] = _.clone(descriptor);
                    descriptors[altName].origName = opName;
                });
            });
        });

        return result;
    });

    // exports ================================================================

    return Operators;

});
