/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/documentstyles', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/model/documentstyles',
    'io.ox/office/drawinglayer/model/drawingstylecollection',
    'io.ox/office/spreadsheet/model/cellstylecollection',
    'io.ox/office/spreadsheet/model/tablestylecollection'
], function (Utils, Color, Border, DocumentStyles, DrawingStyleCollection, CellStyleCollection, TableStyleCollection) {

    'use strict';

    // definitions for spreadsheet document attributes ------------------------

    var SPREADSHEET_DOCUMENT_ATTRIBUTES = {

            /**
             * Number of columns in a sheet.
             */
            cols: 1024,

            /**
             * Number of rows in a sheet.
             */
            rows: 1048576,

            // view attributes

            /**
             * Index of the active (displayed) sheet.
             */
            activeSheet: 0
        };

    // definitions for character attributes in cells --------------------------

    var CHARACTER_ATTRIBUTE_DEFINITIONS = {
            fontName: { def: 'Arial', apply: 'font' },
            fontSize: { def: 11, apply: 'font' },
            bold: { def: false, apply: 'font' },
            italic: { def: false, apply: 'font' },
            underline: { def: false, apply: 'font' },
            strike: { def: 'none', apply: 'font' },
            color: { def: Color.AUTO, apply: 'font' },
            url: { def: '', scope: 'element' }
        };

    // definitions for cell attributes ----------------------------------------

    var CELL_ATTRIBUTE_DEFINITIONS = {
            alignHor: { def: 'auto', apply: 'align' },
            alignVert: { def: 'bottom', apply: 'align' },
            wrapText: { def: false, apply: 'align' },
            numberFormat: { def: { id: 0 }, apply: 'number' },
            fillColor: { def: Color.AUTO, apply: 'fill' },
            borderLeft: { def: Border.NONE, apply: 'border' },
            borderRight: { def: Border.NONE, apply: 'border' },
            borderTop: { def: Border.NONE, apply: 'border' },
            borderBottom: { def: Border.NONE, apply: 'border' },
            borderInsideHor: { def: Border.NONE, scope: 'element' },
            borderInsideVert: { def: Border.NONE, scope: 'element' },
            unlocked: { def: false, apply: 'protect' },
            hidden: { def: false, apply: 'protect' }
        };

    // definitions for apply group flags --------------------------------------

    var APPLY_ATTRIBUTE_DEFINITIONS = {
            font: { def: true },
            fill: { def: true },
            border: { def: true },
            align: { def: true },
            number: { def: true },
            protect: { def: true }
        };

    // definitions for column attributes --------------------------------------

    var COLUMN_ATTRIBUTE_DEFINITIONS = {
            visible: { def: true },
            width: { def: 2000 },
            autoFit: { def: true },
            customWidth: { def: false }
        };

    // definitions for row attributes -----------------------------------------

    var ROW_ATTRIBUTE_DEFINITIONS = {
            visible: { def: true },
            height: { def: 500 },
            customHeight: { def: false },
            customFormat: { def: false }
        };

    // definitions for sheet attributes ---------------------------------------

    var SHEET_ATTRIBUTE_DEFINITIONS = {
            visible: { def: true },
            locked: { def: false },
            baseColWidth: { def: 8 },

            // view attributes
            zoom: { def: 1 },
            selectedRanges: { def: [{ start: [0, 0], end: [0, 0] }] },
            activeIndex: { def: 0 },
            activeCell: { def: [0, 0] },
            selectedDrawings: { def: [] },
            showGrid: { def: true },
            gridColor: { def: Color.AUTO },
            splitMode: { def: 'split' },
            splitWidth: { def: 0 },
            splitHeight: { def: 0 },
            activePane: { def: 'topLeft' },
            scrollLeft: { def: 0 },
            scrollRight: { def: 0 },
            scrollTop: { def: 0 },
            scrollBottom: { def: 0 }
        };

    // definitions for table attributes ---------------------------------------

    var TABLE_ATTRIBUTE_DEFINITIONS = {

            /**
             * Whether the first row of the table is a header row. Header rows
             * will not be filtered or sorted.
             */
            headerRow: { def: false, scope: 'element' },

            /**
             * Whether the last row of the table is a footer row. Footer rows
             * will not be filtered or sorted.
             */
            footerRow: { def: false, scope: 'element' },

            /**
             * Whether the table can be filtered (this flag does not mean that
             * the table has actual filter rules applied, but the GUI may
             * provide visual feedback and control elements for filtering).
             */
            filtered: { def: false, scope: 'element' }
        };

    // definitions for filter attributes in table columns ---------------------

    var TABLE_FILTER_ATTRIBUTE_DEFINITIONS = {

            /**
             * The type of the filter rule in this column. The following values
             * are supported:
             * - 'none': No filter rule is active.
             * - 'discrete': Discrete filter. The attribute 'entries' contains
             *      all cell entries to be shown for the table column.
             */
            type: { def: 'none' },

            /**
             * All entries (cell display strings) that will be shown for this
             * table column. Only used, if the filter type is 'discrete'.
             */
            entries: { def: [], parent: { name: 'type', value: 'discrete' } },

            /**
             * Whether the drop-down button of the filter column is currently
             * hidden by a merged cell range.
             */
            hideMergedButton: { def: false }
        };

    // definitions for sorting attributes in table columns --------------------

    var TABLE_SORT_ATTRIBUTE_DEFINITIONS = {

            /**
             * Whether to sort the column entries in descending order.
             */
            descending: { def: false },
        };

    // definitions for data validation attributes -----------------------------

    var VALIDATION_ATTRIBUTE_DEFINITIONS = {
            type: { def: 'all' },
            compare: { def: '', parent: { name: 'type', value: function (value) { return (/^(integer|number|date|time|length)$/).test(value); } } },
            value1: { def: '', parent: { name: 'type', value: function (value) { return value !== 'all'; } } },
            value2: { def: '', parent: { name: 'type', value: function (value) { return value !== 'all'; } } },
            showInfo: { def: true },
            infoTitle: { def: '' },
            infoText: { def: '' },
            showError: { def: true },
            errorTitle: { def: '' },
            errorText: { def: '' },
            errorType: { def: 'error' },
            showDropDown: { def: true },
            ignoreEmpty: { def: true }
        };

    // additional definitions for drawing attributes --------------------------

    var DRAWING_ATTRIBUTE_DEFINITIONS = {

            /**
             * Type of the drawing anchor (specifies which anchor attributes
             * have to be used), and runtime behavior of the drawing while
             * inserting, removing, or changing columns and rows. Supported
             * values are 'twoCell', 'oneCell', 'absolute', 'twoCellAsOneCell',
             * and 'twoCellAsAbsolute'.
             */
            anchorType: { def: 'twoCell', scope: 'element' },

            /**
             * Zero-based index of the column containing the leading border of
             * the drawing object.
             */
            startCol: { def: 0, scope: 'element' },

            /**
             * The exact position of the leading border in the column, as ratio
             * of the column width (in the range [0,1]).
             */
            startColOffset: { def: 0, scope: 'element' },

            /**
             * Zero-based index of the row containing the top border of the
             * drawing object.
             */
            startRow: { def: 0, scope: 'element' },

            /**
             * The exact position of the top border in the row, as ratio of the
             * row height (in the range [0,1]).
             */
            startRowOffset: { def: 0, scope: 'element' },

            /**
             * Zero-based index of the column containing the trailing border of
             * the drawing object.
             */
            endCol: { def: 0, scope: 'element' },

            /**
             * The exact position of the trailing border in the column, as
             * ratio of the column width (in the range [0,1]).
             */
            endColOffset: { def: 0, scope: 'element' },

            /**
             * Zero-based index of the row containing the bottom border of the
             * drawing object.
             */
            endRow: { def: 0, scope: 'element' },

            /**
             * The exact position of the bottom border in the row, as ratio of
             * the row height (in the range [0,1]).
             */
            endRowOffset: { def: 0, scope: 'element' }
        };

    // class SpreadsheetDocumentStyles ========================================

    /**
     * Provides the style sheet containers for all attribute families used in a
     * spreadsheet document, and other containers collecting special formatting
     * information for the spreadsheet document.
     *
     * @constructor
     *
     * @extends DocumentStyles
     *
     * @param {SpreadsheetApplication} app
     *  The root application instance.
     */
    function SpreadsheetDocumentStyles(app) {

        var // the font collection of the document
            fontCollection = null,

            // the style sheet collection for cells, columns, and rows
            cellStyles = null,

            // the style sheet collection for table ranges
            tableStyles = null,

            // the style sheet collection for drawing objects
            drawingStyles = null;

        // base constructor ---------------------------------------------------

        DocumentStyles.call(this, app, SPREADSHEET_DOCUMENT_ATTRIBUTES);

        // private methods ----------------------------------------------------

        function getDefaultCharAttributes() {
            return cellStyles.getStyleSheetAttributes(cellStyles.getDefaultStyleId()).character;
        }

        // methods ------------------------------------------------------------

        /**
         * Calculates the text line height in 1/100 of millimeters to be used
         * for cell text and row heights, for the passed character attributes.
         *
         * @param {Object} charAttributes
         *  Character formatting attributes influencing the line height.
         *  @param {String} charAttributes.fontName
         *      The name of the original font family (case-insensitive).
         *  @param {Number} charAttributes.fontSize
         *      The original font size, in points.
         *  @param {Boolean} [charAttributes.bold=false]
         *      Whether the text will be rendered in bold characters.
         *  @param {Boolean} [charAttributes.italic=false]
         *      Whether the text will be rendered in italic characters.
         *
         * @returns {Number}
         *  The line height for the passed character attributes, in 1/100 mm.
         */
        this.getLineHeight = function (charAttributes) {

            var // the normal line height according to the font settings
                lineHeight = fontCollection.getNormalLineHeight(charAttributes);

            // use at least 120% of the font size, to compensate for browsers
            // reporting very small line heights (especially Chrome)
            lineHeight = Math.max(lineHeight, Utils.convertLength(charAttributes.fontSize, 'pt', 'px') * 1.2);

            // enlarge the resulting 'normal' line height by another 10% to add top/bottom cell padding
            return Utils.convertLengthToHmm(Math.floor(lineHeight * 1.1), 'px');
        };

        /**
         * Returns the default column width for all undefined columns in a
         * sheet, according to the font settings in the current default cell
         * style, and the passed base column width.
         *
         * @param {Number} baseWidth
         *  The base column width, as number of zero digits.
         *
         * @returns {Number}
         *  The default column width for all undefined columns, in 1/100 mm.
         */
        this.getDefaultColWidth = function (baseWidth) {

            var charAttributes = getDefaultCharAttributes(),
                digitWidth = fontCollection.getDigitWidth(charAttributes),
                colWidth = digitWidth * baseWidth + 5;

            return Utils.convertLengthToHmm(colWidth, 'px');
        };

        /**
         * Returns the default row height for all undefined rows in a sheet,
         * according to the font settings in the current default cell style.
         *
         * @returns {Number}
         *  The default row height for all undefined rows, in 1/100 mm.
         */
        this.getDefaultRowHeight = function () {
            return this.getLineHeight(getDefaultCharAttributes());
        };

        // initialization -----------------------------------------------------

        fontCollection = this.getFontCollection();
        cellStyles = new CellStyleCollection(app, this);
        tableStyles = new TableStyleCollection(app, this);
        drawingStyles = new DrawingStyleCollection(app, this);

        this.registerAttributeDefinitions('character', CHARACTER_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('cell', CELL_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('apply', APPLY_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('column', COLUMN_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('row', ROW_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('sheet', SHEET_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('table', TABLE_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('filter', TABLE_FILTER_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('sort', TABLE_SORT_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('validation', VALIDATION_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('drawing', DRAWING_ATTRIBUTE_DEFINITIONS);

        this.addStyleCollection(cellStyles)
            .addStyleCollection(tableStyles)
            .addStyleCollection(drawingStyles);

        this.registerDestructor(function () {
           app = fontCollection = cellStyles = tableStyles = drawingStyles = null;
        });

    } // class SpreadsheetDocumentStyles

    // exports ================================================================

    // derive this class from class DocumentStyles
    return DocumentStyles.extend({ constructor: SpreadsheetDocumentStyles });

});
