/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/app/application', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/io',
    'io.ox/office/tk/errorcode',
    'io.ox/office/editframework/app/editapplication',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/model',
    'io.ox/office/spreadsheet/model/formula/tokenutils',
    'io.ox/office/spreadsheet/model/formula/operators',
    'io.ox/office/spreadsheet/view/view',
    'io.ox/office/spreadsheet/view/labels',
    'io.ox/office/spreadsheet/app/controller',
    'gettext!io.ox/office/spreadsheet'
], function (Utils, IO, ErrorCode, EditApplication, Config, SheetUtils, SpreadsheetModel, TokenUtils, Operators, SpreadsheetView, Labels, SpreadsheetController, gt) {

    'use strict';

    var // default settings for locale data
        DEFAULT_LOCALE_DATA = {
            functions: {},
            errorCodes: _.chain(SheetUtils.ErrorCodes).map(function (code) { return [code.code, code.code]; }).object().value(),
            decSeparator: '.',
            groupSeparator: ',',
            listSeparator: ',',
            trueLiteral: 'TRUE',
            falseLiteral: 'FALSE'
        },

        // RE pattern for a valid error code literal
        RE_ERROR_CODE = /^#[^ !?]+[!?]?$/,

        // global cache for locale data, mapped by locale identifier and file format
        localeDataCache = {},

        // global cache for function help texts, mapped by locale identifier and file format
        functionHelpCache = {};

    // private static functions ===============================================

    /**
     * Inserts additional information into the passed locale data object.
     */
    function completeLocaleData(localeData) {

        // build inverted function name map (translated to native), and arrays of function names
        localeData.invFunctions = _.invert(localeData.functions);
        localeData.nativeFunctionNames = _.chain(localeData.functions).keys().sort().value();
        localeData.localizedFunctionNames = _.chain(localeData.invFunctions).keys().sort().value();

        // build inverted error code map (translated to native), and arrays of error codes
        localeData.invErrorCodes = _.invert(localeData.errorCodes);
        localeData.nativeErrorCodes = _.chain(localeData.errorCodes).keys().sort().value();
        localeData.localizedErrorCodes = _.chain(localeData.invErrorCodes).keys().sort().value();
    }

    // class SpreadsheetApplication ===========================================

    /**
     * The OX Spreadsheet application.
     *
     * Triggers the following additional events:
     * - 'docs:update:cells'
     *      After the application has received a 'docs:update' event which
     *      contains the contents and/or addresses of changed cells in the
     *      document. Event handlers receive the following parameter:
     *      (1) {Object} changedData
     *          A map of change descriptors for each changed sheet, mapped by
     *          the zero-based sheet index. Each change descriptor object
     *          contains the following properties:
     *          - {Number} sheet
     *              The zero-based sheet index. Equal to the map key of this
     *              object, but as number for convenience.
     *          - {Array} [rangeContents]
     *              An array with cell range descriptors. Each descriptor
     *              object contains the same properties as also sent in the
     *              response data of view update requests. If omitted, no cell
     *              content data is available for the sheet. If omitted, no
     *              cells have been changed in the sheet, or no explicit cell
     *              content data is available yet (see property 'dirtyRanges').
     *          - {Array} [dirtyRanges]
     *              An array with range addresses of additional cells that are
     *              not included in the 'rangeContents' property, but whose
     *              contents have been changed in the sheet. These ranges need
     *              to be updated with further server requests. If omitted, the
     *              sheet does not contain additional dirty cell ranges.
     *          - {Array} [changedRanges]
     *              An array containing the unified range addresses from the
     *              properties 'rangeContents' and 'dirtyRanges'. Will exist,
     *              if either of these properties exists too.
     *          - {Number} [usedCols]
     *              The number of used columns in the sheet. If omitted, the
     *              number of used columns has not changed.
     *          - {Number} [usedRows]
     *              The number of used rows in the sheet. If omitted, the
     *              number of used rows has not changed.
     *      (2) {Array} allChangedRanges
     *          An array containing the range addresses of all changed ranges
     *          in all sheets. This is the concatenation of all cell ranges
     *          contained in the 'changedRanges' properties for all sheets in
     *          the 'changedData' event parameter. Each range object in this
     *          array contains the additional property 'sheet' with the
     *          zero-based sheet index of the range.
     *      (3) {String} [errorCode]
     *          A specific error code associated to the changed cells. The
     *          following error codes are supported:
     *          - 'circular': The changed cells contain at least one formula
     *              cell with a circular reference.
     *
     * @constructor
     *
     * @extends EditApplication
     *
     * @param {Object} launchOptions
     *  All options passed to the core launcher (the ox.launch() method).
     *
     * @param {Object} [appOptions]
     *  Static application options that have been passed to the static method
     *  BaseApplication.createLauncher().
     */
    var SpreadsheetApplication = EditApplication.extend({ constructor: function (launchOptions, appOptions) {

        var // self reference
            self = this,

            // the spreadsheet document model and view
            docModel = null,
            docView = null,

            // the locale data (translated function names etc.)
            localeData = _.copy(DEFAULT_LOCALE_DATA, true),

            // the Deferred object that will be resolved when locale data is available
            localeDef = $.Deferred(),

            // the function help resource
            functionHelp = {},

            // the map key for the global caches for locale data and function help
            cacheKey = null;

        // base constructor ---------------------------------------------------

        EditApplication.call(this, SpreadsheetModel, SpreadsheetView, SpreadsheetController, launchOptions, appOptions, {
            newDocumentParams: { initial_sheetname: Labels.getSheetName(0) },
            preProcessHandler: preProcessDocument,
            postProcessHandler: postProcessDocument,
            previewHandler: previewHandler,
            importFailedHandler: importFailedHandler,
            prepareFlushHandler: prepareFlushDocument,
            prepareLoseEditRightsHandler: prepareLoseEditRights,
            realTimeDelay: 10
        });

        // private methods ----------------------------------------------------

        /**
         * Reformats and completes the locale data received from the server,
         * and stores it in the internal instance variable 'localeData'.
         */
        function processLocaleData(data) {

            Utils.log('SpreadsheetApplication.preProcessDocument(): locale data response:', data);

            // print console warnings for missing properties
            if (Config.DEBUG) {
                if (!data.decSeparator) { Utils.error('\xa0 missing decimal separator'); }
                if (!data.groupSeparator) { Utils.error('\xa0 missing group separator'); }
                if (!data.listSeparator) { Utils.error('\xa0 missing list separator'); }
                if (!data.trueLiteral) { Utils.error('\xa0 missing literal for TRUE'); }
                if (!data.falseLiteral) { Utils.error('\xa0 missing literal for FALSE'); }
            }

            // the map of all function names
            localeData = { functions: Utils.getObjectOption(data, 'functions', {}) };

            // error code literals
            localeData.errorCodes = _.extend({}, DEFAULT_LOCALE_DATA.errorCodes, Utils.getObjectOption(data, 'errorCodes'));

            // all separator characters
            _.each(['decSeparator', 'groupSeparator', 'listSeparator'], function (propertyName) {
                localeData[propertyName] = Utils.getStringOption(data, propertyName, DEFAULT_LOCALE_DATA[propertyName], true)[0];
            });

            // Boolean literals
            localeData.trueLiteral = Utils.getStringOption(data, 'trueLiteral', DEFAULT_LOCALE_DATA.trueLiteral, true).toUpperCase();
            localeData.falseLiteral = Utils.getStringOption(data, 'falseLiteral', DEFAULT_LOCALE_DATA.falseLiteral, true).toUpperCase();

            // other settings
            localeData.engineVersion = Utils.getStringOption(data, 'engineVersion', '');

            // cache complete locale data object in the global cache
            completeLocaleData(localeData);
            Utils.info('SpreadsheetApplication.preProcessDocument(): new locale data:', localeData);
            localeDataCache[cacheKey] = localeData;

            // print error messages to browser console
            Utils.withLogging(function () {

                // check for duplicates in translated function names
                if (localeData.nativeFunctionNames.length !== localeData.localizedFunctionNames.length) {
                    Utils.error('\xa0 list of translated function names contains duplicates');
                }

                // show differences between passed function names and implemented function names
                var implementedFunctions = _.keys(Operators.getDescriptors(self.getFileFormat()).FUNCTIONS),
                    unknownFunctions = _.difference(localeData.nativeFunctionNames, implementedFunctions),
                    missingFunctions = _.difference(implementedFunctions, localeData.nativeFunctionNames);
                if (unknownFunctions.length > 0) {
                    Utils.warn('\xa0 unknown functions in locale data: ' + unknownFunctions.join(', '));
                }
                if (missingFunctions.length > 0) {
                    Utils.warn('\xa0 missing functions in locale data: ' + missingFunctions.join(', '));
                }

                // check for duplicates in translated error code names
                if (localeData.nativeErrorCodes.length !== localeData.localizedErrorCodes.length) {
                    Utils.error('\xa0 list of translated error codes contains duplicates');
                }

                // check native error codes
                _.each(localeData.nativeErrorCodes, function (errorCode) {
                    if (!RE_ERROR_CODE.test(errorCode)) {
                        Utils.error('\xa0 invalid native error code: "' + errorCode + '"');
                    }
                });

                // check translated error codes
                _.each(localeData.localizedErrorCodes, function (errorCode) {
                    if (!RE_ERROR_CODE.test(errorCode)) {
                        Utils.error('\xa0 invalid translated error code: "' + errorCode + '"');
                    }
                });
            });
        }

        /**
         * Reformats and completes the function help JSON data, and stores it
         * in the internal instance variable 'functionHelp'.
         */
        function processFunctionHelp(data) {

            var // internal raw function descriptors
                rawFunctionDescriptors = Operators.FUNCTIONS,
                // function descriptors for current file format
                functionDescriptors = Operators.getDescriptors(self.getFileFormat()).FUNCTIONS,
                // names of functions without help that will not be alerted in the console
                SILENT_UNSUPPORTED_FUNCTIONS = _.values(_.pick(localeData.functions, 'CURRENT', 'IFERROR', 'IFNA'));

            // store function help data in internal instance member, and in the global cache
            // bug 36743: store a clone of the object (do not modify the object cached by IO.loadResource())
            functionHelp = functionHelpCache[cacheKey] = _.copy(data, true);

            // process each entry in the function help data object (delete invalid entries)
            _.each(functionHelp, function (funcData, funcName) {

                var // the native function name of the current function help object
                    nativeFuncName = localeData.invFunctions[funcName],
                    // the internal raw function descriptor (for all file formats)
                    rawFunctionDesc = rawFunctionDescriptors[nativeFuncName],
                    // the function descriptor resolved for the current file format
                    functionDesc = functionDescriptors[nativeFuncName];

                // writes a warning or error message for the current function
                function log(type, msg) {
                    Utils[type]('\xa0 ' + msg + ' function ' + funcName);
                }

                // deletes the function help entry of the current function
                function deleteFunctionHelp(type, msg) {
                    if (type) { log(type, msg); }
                    delete functionHelp[funcName];
                }

                // returns the maximum of a file-format-dependent numeric function descriptor property
                function resolveMaxNumber(value) {
                    return _.isNumber(value) ? value : _.isObject(value) ? _.reduce(value, function (a, b) { return Math.max(a, b); }, 0) : value;
                }

                // function name must exist in locale data (do not warn for help entries of ODF add-in functions)
                if (!nativeFuncName) {
                    return (/_ADD$/).test(funcName) ? deleteFunctionHelp() : deleteFunctionHelp('warn', 'help entry found for unknown');
                }

                // get internal raw descriptor for alternative function names
                if (!rawFunctionDesc && functionDesc && _.isString(functionDesc.origName)) {
                    rawFunctionDesc = rawFunctionDescriptors[functionDesc.origName];
                }

                // function data entry must correspond to a supported function
                if (!rawFunctionDesc) {
                    return deleteFunctionHelp('warn', 'help entry found for unimplemented');
                }

                // silently delete function help for functions not supported by the current file format
                if (!functionDesc) {
                    return deleteFunctionHelp();
                }

                // function data entry must be an object
                if (!_.isObject(funcData)) {
                    return deleteFunctionHelp('error', 'invalid map entry for');
                }

                // function data entry must contain a description
                if (_.isString(funcData.description)) {
                    funcData.description = Utils.trimString(funcData.description);
                } else {
                    log('error', 'missing description for');
                    delete functionHelp[funcName].description;
                }

                // function data entry must contain arrays for parameter names
                if (!_.isArray(funcData.params)) {
                    log('error', 'missing parameter names for');
                    funcData.params = [];
                } else if (!_.all(funcData.params, _.isString)) {
                    log('error', 'invalid parameter names for');
                }

                // function data entry must contain arrays for parameter descriptions
                if (!_.isArray(funcData.paramshelp)) {
                    log('error', 'missing parameter descriptions for');
                    funcData.paramshelp = [];
                } else if (!_.all(funcData.paramshelp, _.isString)) {
                    log('error', 'invalid parameter descriptions for');
                }

                // arrays for parameter names and descriptions must have equal length
                if (funcData.params.length !== funcData.paramshelp.length) {
                    log('error', 'count mismatch in parameter names and descriptions for');
                }

                // check parameter help availability of required parameters
                var rawMinParams = resolveMaxNumber(rawFunctionDesc.minParams);
                if (funcData.params.length < rawMinParams) {
                    log('error', 'missing help for required parameters for');
                    while (funcData.params.length < rawMinParams) { funcData.params.push(''); }
                }

                // check parameter help availability of fixed number of optional parameters
                var rawMaxParams = resolveMaxNumber(rawFunctionDesc.maxParams);
                if (_.isNumber(rawMaxParams)) {
                    if (funcData.params.length < rawMaxParams) {
                        log('error', 'missing help for optional parameters for');
                    } else if (rawMaxParams < funcData.params.length) {
                        log('warn', 'superfluous optional parameters for');
                    }
                }

                // validate number of array elements of function parameter help
                if (_.isNumber(functionDesc.maxParams)) {
                    while (funcData.params.length < functionDesc.maxParams) { funcData.params.push(''); }
                    funcData.params.splice(functionDesc.maxParams);
                }

                // check parameter help availability of variable number of optional parameters
                var repeatParams = _.isNumber(functionDesc.maxParams) ? null : (functionDesc.repeatParams || 1);
                if (_.isNumber(repeatParams)) {
                    var optParams = funcData.params.length - functionDesc.minParams;
                    // remove help texts of superfluous optional parameters
                    if ((optParams !== 0) && (optParams !== repeatParams)) {
                        log('warn', 'superfluous variable parameters for');
                        funcData.params.splice(functionDesc.minParams + ((optParams > repeatParams) ? repeatParams : 0));
                    }
                    funcData.repeat = repeatParams;
                }

                // create a single parameter array with name, description, and additional information
                funcData.params = _.map(funcData.params, function (name, index) {
                    var description = funcData.paramshelp[index];
                    return {
                        name: _.isString(name) ? Utils.trimString(name) : '',
                        description: _.isString(description) ? Utils.trimString(description) : '',
                        optional: functionDesc.minParams <= index
                    };
                });

                var // whether the function has repeating parameters
                    isRepeating = _.isNumber(funcData.repeat),
                    // absolute index of first repeated parameter
                    repeatStart = isRepeating ? (funcData.params.length - funcData.repeat) : -1;

                // convert to the function descriptor provided by the method SpreadsheetApplication.getFunctionHelp()
                functionHelp[funcName] = {
                    name: funcName,
                    description: funcData.description,
                    params: funcData.params,
                    repeatStart: repeatStart,
                    cycleLength: isRepeating ? funcData.repeat : 0,
                    cycleCount: isRepeating ? Math.floor((TokenUtils.MAX_PARAM_COUNT - repeatStart) / funcData.repeat) : 0
                };

                // duplicate help entries for alternative function names
                _.each(functionDesc.altNames, function (altFuncName) {
                    var altLocalName = localeData.functions[altFuncName];
                    if (_.isString(altLocalName) && !(altLocalName in functionHelp)) {
                        functionHelp[altLocalName] = _.clone(functionHelp[funcName]);
                        functionHelp[altLocalName].name = altLocalName;
                    }
                });
            });

            // print all function names without help texts
            var missingHelpFunctions = _.difference(_.values(localeData.functions), _.keys(functionHelp));
            missingHelpFunctions = _.difference(missingHelpFunctions, SILENT_UNSUPPORTED_FUNCTIONS);
            _.each(missingHelpFunctions, function (funcName) {
                Utils.error('\xa0 missing help for function ' + funcName);
            });
        }

        /**
         * Preparation of the document, before its operations will be applied.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  document has been prepared successfully; or rejected on any error.
         */
        function preProcessDocument() {

            var // the server request for the locale data
                localeRequest = null;

            // map key for the caches (locale and file format)
            cacheKey = JSON.stringify({ locale: Config.LOCALE, format: self.getFileFormat() });

            // try to receive cached function help texts already downloaded before
            if (cacheKey in functionHelpCache) {
                functionHelp = functionHelpCache[cacheKey];
            } else {
                // download function help data, but ignore the success/failure result (application may run even without function help)
                $.when(IO.loadResource('io.ox/office/spreadsheet/resource/functions'), localeDef).done(processFunctionHelp);
            }

            // try to receive cached locale data already downloaded before
            if (cacheKey in localeDataCache) {
                localeData = localeDataCache[cacheKey];
                localeDef.resolve();
                return $.when();
            }

            // request locale data from server, process the raw data received from server
            localeRequest = self.sendQueryRequest('localeData', { locale: Config.LOCALE }).done(processLocaleData);

            // fail with appropriate error message
            localeRequest = localeRequest.then(null, function () {
                Utils.error('SpreadsheetApplication.preProcessDocument(): could not download locale data for "' + Config.LOCALE + '"');
                return { headline: gt('Server Error'), message: gt('The engine used for calculating formulas is not available.') };
            });

            // resolve the internal Deferred object used for the onInitLocale() method
            return localeRequest.always(function () { localeDef.resolve(); });
        }

        /**
         * Post-processing of the document, after all its import operations
         * have been applied successfully.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  document has been post-processed successfully; or rejected when the
         *  document is invalid, or an error has occurred.
         */
        function postProcessDocument() {

            // document must contain at least one sheet (returning a rejected Deferred
            // object will cause invocation of the prepareInvalidDocument() method)
            if (docModel.getSheetCount() === 0) { return $.Deferred().reject(); }

            // create all built-in cell styles not imported from the file
            docModel.getStyleCollection('cell').createMissingStyles();

            // nothing to wait for
            return $.when();
        }

        /**
         * Shows an early preview of the document. Activates the sheet that is
         * specified in the passed preview data, and starts querying cell
         * contents for the active sheet.
         *
         * @param {Object} previewData
         *  The preview data containing the index of the active sheet.
         *
         * @returns {Boolean}
         *  Whether a sheet has been activated successfully.
         */
        function previewHandler(previewData) {
            var activeSheet = Utils.getIntegerOption(previewData, 'activeSheet', 0);
            return self.getView().activatePreviewSheet(activeSheet);
        }

        /**
         * Handler will be called by base class if importing the document
         * failed. The functions handles Spreadsheet specific errors.
         */
        function importFailedHandler(response) {

            var // specific error code sent by the server
                error = new ErrorCode(response);

            switch (error.getCodeAsConstant()) {
            case 'LOADDOCUMENT_COMPLEXITY_TOO_HIGH_ERROR':
                response.message = gt('This document exceeds the spreadsheet size and complexity limits.');
                break;
            case 'LOADDOCUMENT_CALCENGINE_NOT_WORKING_ERROR':
                response.headline = gt('Server Error');
                response.message = gt('The engine used for calculating formulas is not available.');
                break;
            }

            // document must contain at least one sheet, to prevent if-else
            // statements for empty documents at thousand places in the view code
            if (!self.isInQuit()) {
                docModel.prepareInvalidDocument();
            }
        }

        /**
         * Preprocessing before the document will be flushed for downloading,
         * printing, sending as mail, or closing.
         */
        function prepareFlushDocument() {
            // commit current text of the cell in-place edit mode (without validation)
            docView.leaveCellEditMode('auto');
            // send all changed view settings before flushing editable document
            docView.sendChangedViewAttributes();
        }

        /**
         * Preparations before the edit mode will be switched off.
         */
        function prepareLoseEditRights() {
            // commit current text of the cell in-place edit mode (without validation)
            docView.leaveCellEditMode('auto');
        }

        /**
         * Handles 'docs:update' notifications. If the message data contains
         * information about changed cells in the document, this method
         * triggers a 'docs:update:cells' event with preprocessed user data.
         *
         * @param {Object} data
         *  The message data of the 'docs:update' application event.
         */
        function updateNotificationHandler(data) {

            var // complete changed data
                changedData = Utils.getObjectOption(data, 'changed'),
                // the locale used to format the cell contents
                locale = Utils.getStringOption(data, 'locale'),
                // a special error code associated to the changed cells
                errorCode = Utils.getStringOption(changedData, 'error'),
                // all changed cell information for the own 'docs:update:cells' event
                eventData = null,
                // all changed range addresses, with sheet index
                allChangedRanges = null;

            // update message may not contain any data about changed cells
            if (!_.isObject(changedData)) { return; }
            Utils.info('SpreadsheetApplication.updateNotificationHandler(): notification received:', changedData);

            // extract the map of changed data per sheet
            changedData = Utils.getObjectOption(changedData, 'sheets');
            if (!changedData) {
                Utils.warn('SpreadsheetApplication.updateNotificationHandler(): no sheets defined');
                return;
            }

            // process all changed sheets
            eventData = {};
            allChangedRanges = [];
            _.each(changedData, function (changedSheetData, sheetKey) {

                var sheet = parseInt(sheetKey, 10),
                    rangeContents = Utils.getArrayOption(changedSheetData, 'contents', []),
                    dirtyRanges = Utils.getArrayOption(changedSheetData, 'cells', []),
                    colIntervals = Utils.getArrayOption(changedSheetData, 'columns'),
                    rowIntervals = Utils.getArrayOption(changedSheetData, 'rows'),
                    usedCols = Utils.getIntegerOption(changedSheetData, 'usedCols'),
                    usedRows = Utils.getIntegerOption(changedSheetData, 'usedRows'),
                    eventSheetData = {},
                    changedRanges = null;

                if (!isFinite(sheet) || (sheet < 0) || (sheet >= docModel.getSheetCount())) {
                    Utils.warn('SpreadsheetApplication.updateNotificationHandler(): invalid sheet index: "' + sheetKey + '"');
                    return;
                }

                // insert cell contents
                if (_.isArray(rangeContents) && (rangeContents.length > 0)) {
                    _.each(rangeContents, function (rangeData) {
                        if (!rangeData.end) { rangeData.end = _.clone(rangeData.start); }
                    });
                    // bug 32624: use content data only if formatted with the own locale; otherwise store as 'dirty ranges'
                    if (locale === Config.LOCALE) {
                        eventSheetData.rangeContents = rangeContents;
                    } else {
                        dirtyRanges = dirtyRanges.concat(rangeContents);
                    }
                }

                // insert addresses of dirty ranges
                _.each(dirtyRanges, function (range) {
                    if (!range.end) { range.end = _.clone(range.start); }
                });
                _.each(colIntervals, function (interval) {
                    dirtyRanges.push(docModel.makeColRange(('last' in interval) ? interval : interval.first));
                });
                _.each(rowIntervals, function (interval) {
                    dirtyRanges.push(docModel.makeRowRange(('last' in interval) ? interval : interval.first));
                });
                if (dirtyRanges.length > 0) {
                    eventSheetData.dirtyRanges = SheetUtils.getUnifiedRanges(dirtyRanges);
                }

                // insert number of used columns/rows
                if (_.isNumber(usedCols)) { eventSheetData.usedCols = usedCols; }
                if (_.isNumber(usedRows)) { eventSheetData.usedRows = usedRows; }

                // insert sheet entry into the own event data
                if (!_.isEmpty(eventSheetData)) {
                    eventSheetData.sheet = sheet;
                    eventData[sheetKey] = eventSheetData;
                }

                // extend array of all changed ranges
                if (eventSheetData.rangeContents && eventSheetData.dirtyRanges) {
                    changedRanges = SheetUtils.getUnifiedRanges(eventSheetData.rangeContents.concat(eventSheetData.dirtyRanges));
                } else if (eventSheetData.rangeContents) {
                    changedRanges = SheetUtils.getUnifiedRanges(eventSheetData.rangeContents);
                } else if (eventSheetData.dirtyRanges) {
                    changedRanges = _.copy(dirtyRanges, true);
                }
                _.each(changedRanges, function (range) { range.sheet = sheet; });
                eventSheetData.changedRanges = changedRanges;
                allChangedRanges = allChangedRanges.concat(changedRanges);
            });

            // trigger the event, if the event data object contains any useful information
            if (!_.isEmpty(eventData)) {
                self.trigger('docs:update:cells', eventData, allChangedRanges, errorCode);
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Registers a callback function that will be invoked when the locale
         * settings (number format settings, translated function names, etc.)
         * for this document have been initialized. In the early initialization
         * phase of the document, the callback function will be invoked
         * deferred, when the server request for the locale data has returned.
         * Later, the callback function will be invoked immediately when
         * calling this method.
         *
         * @param {Function} callback
         *  The callback function invoked after the locale data has been
         *  initialized.
         *
         * @param {Object} [context]
         *  The context bound to the invoked callback function.
         *
         * @returns {SpreadsheetApplication}
         *  A reference to this instance.
         */
        this.onInitLocale = function (callback, context) {
            localeDef.done(_.bind(callback, context));
            return this;
        };

        /**
         * Returns the sorted list with the names of all supported functions
         * used in formulas, translated for the current UI language.
         *
         * @returns {Array}
         *  The sorted list with the names of all supported functions.
         */
        this.getLocalizedFunctionNames = function () {
            return localeData.localizedFunctionNames;
        };

        /**
         * Returns whether the passed string is the name of a known built-in
         * function in the current UI language.
         *
         * @param {String} funcName
         *  The string to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed string is the name of a known function in the
         *  current UI language.
         */
        this.isLocalizedFunctionName = function (funcName) {
            return funcName.toUpperCase() in localeData.invFunctions;
        };

        /**
         * Returns the name of the specified function, translated for the
         * current UI language.
         *
         * @param {String} nativeFuncName
         *  The English name of the function.
         *
         * @returns {String}
         *  The translated name of the specified function if available,
         *  otherwise the passed function name.
         */
        this.getLocalizedFunctionName = function (nativeFuncName) {
            nativeFuncName = nativeFuncName.toUpperCase();
            return (nativeFuncName in localeData.functions) ? localeData.functions[nativeFuncName] : nativeFuncName;
        };

        /**
         * Returns the sorted list with the native (English) names of all
         * supported functions used in formulas.
         *
         * @returns {Array}
         *  The sorted list with the native names of all supported functions.
         */
        this.getNativeFunctionNames = function () {
            return localeData.nativeFunctionNames;
        };

        /**
         * Returns whether the passed string is the native (English) name of a
         * known built-in function.
         *
         * @param {String} funcName
         *  The string to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed string is the native name of a known function.
         */
        this.isNativeFunctionName = function (funcName) {
            return funcName.toUpperCase() in localeData.functions;
        };

        /**
         * Returns the native (English) name of the specified function name in
         * the current UI language.
         *
         * @param {String} localFuncName
         *  The translated name of the function.
         *
         * @returns {String}
         *  The native name of the specified function if available, otherwise
         *  the passed function name.
         */
        this.getNativeFunctionName = function (localFuncName) {
            localFuncName = localFuncName.toUpperCase();
            return (localFuncName in localeData.invFunctions) ? localeData.invFunctions[localFuncName] : localFuncName;
        };

        /**
         * Returns a help text descriptor for the specified translated function
         * name.
         *
         * @param {String} localFuncName
         *  The translated name of the function (case-insensitive).
         *
         * @returns {Object|Null}
         *  A help descriptor if the specified function is available, with the
         *  following properties:
         *  - {String} name
         *      The upper-case name of the function.
         *  - {String} description
         *      A short description for the function itself.
         *  - {Array} params
         *      An array with the names and descriptions of all supported
         *      function parameters. Each object in the array represents one
         *      function parameter, and contains the following properties:
         *      - {String} param.name
         *          The name of the parameter.
         *      - {String} param.description
         *          The description text of the parameter.
         *      - {Boolean} param.optional
         *          Whether the parameter is required or optional.
         *  - {Number} repeatStart
         *      The zero-based index of the first trailing parameter that can
         *      be used repeatedly in the function; or -1, if the function does
         *      not support parameter repetition.
         *  - {Number} cycleLength
         *      The number of trailing repeated parameters; or 0, if the
         *      function does not support parameter repetition.
         *  - {Number} cycleCount
         *      The maximum number of repetition cycles supported by the
         *      function, according to the index of first repeated parameter,
         *      and the size of the repetition cycles. Will be 0, if the
         *      function does not support parameter repetition.
         */
        this.getFunctionHelp = function (localFuncName) {
            localFuncName = localFuncName.toUpperCase();
            // bug 33438: do not show help for unsupported functions (e.g. due to current file format)
            return ((localFuncName in functionHelp) && (localFuncName in localeData.invFunctions)) ? functionHelp[localFuncName] : null;
        };

        /**
         * Returns the sorted list with the names of all supported error codes
         * used in formulas, translated for the current UI language.
         *
         * @returns {Array}
         *  The sorted list with the names of all supported error codes.
         */
        this.getLocalizedErrorCodes = function () {
            return localeData.localizedErrorCodes;
        };

        /**
         * Returns whether the passed string is a known error code literal in
         * the current UI language.
         *
         * @param {String} errorCode
         *  The string to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed string is a known error code literal in the
         *  current UI language.
         */
        this.isLocalizedErrorCode = function (errorCode) {
            return errorCode.toUpperCase() in localeData.invErrorCodes;
        };

        /**
         * Returns the sorted list with the internal values of all supported
         * error codes used in formulas.
         *
         * @returns {Array}
         *  The sorted list with the internal values of all supported error
         *  codes.
         */
        this.getNativeErrorCodes = function () {
            return localeData.nativeErrorCodes;
        };

        /**
         * Returns whether the passed string is a known internal error code
         * literal.
         *
         * @param {String} errorCode
         *  The string to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed string is a known internal error code literal.
         */
        this.isNativeErrorCode = function (errorCode) {
            return errorCode.toUpperCase() in localeData.errorCodes;
        };

        /**
         * Returns the decimal separator for the current UI language.
         *
         * @returns {String}
         *  The decimal separator for the current UI language.
         */
        this.getDecimalSeparator = function () {
            return localeData.decSeparator;
        };

        /**
         * Returns the group separator (usually the character that separates
         * groups of 3 digits in the integral part of a number, but different
         * rules may apply in specific locales) for the current UI language.
         *
         * @returns {String}
         *  The group separator for the current UI language.
         */
        this.getGroupSeparator = function () {
            return localeData.groupSeparator;
        };

        /**
         * Returns the list separator for the current UI language. The list
         * separator is used to separate parameters of functions, and for the
         * range list operator.
         *
         * @returns {String}
         *  The list separator for the current UI language.
         */
        this.getListSeparator = function () {
            return localeData.listSeparator;
        };

        /**
         * Returns the string literal of the specified Boolean value for the
         * current UI language.
         *
         * @param {Boolean} value
         *  The Boolean value.
         *
         * @returns {String}
         *  The string literal of the specified Boolean value.
         */
        this.getBooleanLiteral = function (value) {
            return localeData[value ? 'trueLiteral' : 'falseLiteral'];
        };

        /**
         * Returns the name of the specified error code, translated for the
         * current UI language.
         *
         * @param {ErrorCode} errorCode
         *  The error code object.
         *
         * @returns {String}
         *  The translated name of the specified error code if available,
         *  otherwise the internal error code (the 'code' property) of the
         *  passed error code object.
         */
        this.convertErrorCodeToString = function (errorCode) {
            return (errorCode.code in localeData.errorCodes) ? localeData.errorCodes[errorCode.code] : errorCode.code;
        };

        /**
         * Returns the error code object representing the passed translated
         * error code literal.
         *
         * @param {String} errorCode
         *  The translated name of the error code.
         *
         * @returns {ErrorCode}
         *  The error code object representing the specified translated error
         *  code.
         */
        this.convertStringToErrorCode = function (errorCode) {
            errorCode = errorCode.toUpperCase();
            return SheetUtils.makeErrorCode((errorCode in localeData.invErrorCodes) ? localeData.invErrorCodes[errorCode] : errorCode);
        };

        /**
         * Returns the version of the server-side Calc Engine.
         *
         * @returns {String}
         *  The version of the Calc Engine.
         */
        this.getEngineVersion = function () {
            return localeData.engineVersion;
        };

        /**
         * Sends an action POST request in a specific format to the server.
         *
         * @param {String} action
         *  The action identifier, inserted as 'action' property into the
         *  request POST data.
         *
         * @param {Object} data
         *  The data object, inserted as 'requestdata' property into the
         *  request POST data.
         *
         * @param {Object} [options]
         *  Optional parameters. See method IO.sendRequest() for details.
         *
         * @returns {jQuery.Promise}
         *  The abortable promise representing the action request.
         */
        this.sendActionRequest = function (action, data, options) {
            options = _.extend({ method: 'POST' }, options); // default to POST
            return this.sendFileRequest(IO.FILTER_MODULE_NAME, { action: action, requestdata: JSON.stringify(data) }, options);
        };

        /**
         * Sends a query POST request (an action request with the 'action'
         * property set to the value 'query') in a specific format to the
         * server.
         *
         * @param {String} type
         *  The type of the query, inserted as 'type' property into the request
         *  POST data.
         *
         * @param {Object} data
         *  The data object, inserted as 'requestdata' property into the
         *  request POST data.
         *
         * @param {Object} [options]
         *  Optional parameters. See method IO.sendRequest() for details.
         *
         * @returns {jQuery.Promise}
         *  The abortable promise representing the query request.
         */
        this.sendQueryRequest = function (type, data, options) {
            return this.sendActionRequest('query', _.extend({}, data, { type: type }), options);
        };

        // initialization -----------------------------------------------------

        // initialization of class members
        this.onInit(function () {
            docModel = self.getModel();
            docView = self.getView();
            // initialize default locale data
            completeLocaleData(localeData);
        });

        // application notifies changed contents/results of cells create an own 'docs:update:cells' event
        this.on('docs:update', updateNotificationHandler);

        // destroy all class members
        this.registerDestructor(function () {
            launchOptions = appOptions = null;
            docModel = docView = localeData = functionHelp = null;
        });

    }}); // class SpreadsheetApplication

    // static methods ---------------------------------------------------------

    /**
     * Replacement for the generic method EditApplication.createLauncher()
     * without parameters, to launch spreadsheet applications.
     */
    SpreadsheetApplication.createLauncher = function () {
        return EditApplication.createLauncher('io.ox/office/spreadsheet', SpreadsheetApplication, { icon: 'fa-table' });
    };

    // exports ================================================================

    return SpreadsheetApplication;

});
