/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/settings/templatefolders', [
    'io.ox/core/folder/picker',
    'io.ox/core/folder/api',
    'io.ox/mail/detail/links',
    'io.ox/office/tk/utils',
    'gettext!io.ox/office/main'
], function (FolderPicker, FolderAPI, DeepLinks, Utils, gt) {

    'use strict';

    // class TemplateFolders ====================================================

    /**
     * @param {TemplateModel} templateModel
     *  storage
     *
     * @param {Object} options
     *  @param {Boolean} [options.useMyFiles=false]
     *  @param {Boolean} [options.readOnly=false]
     *  @param {String} [options.title]
     *  @param {String} [options.folderPicker]
     *  @param {String} [options.newFolder]
     */
    function TemplateFolders(templateModel, options) {
        var readOnly = Utils.getBooleanOption(options, 'readOnly', false);

        this.refresh = function (settings) {
            settings.appendTitle(Utils.getStringOption(options, 'title', gt('My template folders')));

            var folders = $('<ul class="list-unstyled publications list-group widget-list"></ul>');

            settings.append(folders);


            /**
             * calls FolderApi.path(id), but checks if the folder is deleted,
             * if not callback is called
             */
            function path(id, callback) {
                FolderAPI.path(id).done(function(path) {
                    for (var i = 1; i < path.length; i++) {
                        if (FolderAPI.is('trash', path[i])) {
                            templateModel.remove(id);
                            return;
                        }
                    }
                    callback(path);
                });
            }

            /**
             * removes folder from the Model & from the Gui
             */
            function remove(path, li) {
                var id = _.last(path).id;
                if (!templateModel.remove(id)) {
                    if (ox.debug) { console.warn('folder not in list', id); }
                } else {
                    li.remove();
                }
            }

            /**
             * add new li-element with:
             *
             * folder-title
             * folder-link (opens Drive)
             * delete button
             */
            function add(path, noRemove, oldLi) {
                var folder = _.last(path);
                var id = folder.id;


                var li = $('<li class="widget-settings-view">');

                var ctrl = $('<div class="widget-controls">');
                li.append(ctrl);

                if (!noRemove && !readOnly) {
//                    var edit = $('<a href="#" role="button" tabindex="1" data-action="edit" class="action" style="position: absolute; right: 45px; top: 20px;"></a>');
//                    edit.text(gt('Edit'));
//                    edit.click(function(evt) {
//                        evt.preventDefault();
//                        clickAdd(evt, li, id);
//                    });



                    var rem = $('<a class="remove" href="#" tabindex="1" role="button" data-action="remove">');
                    //FIXME: aria-label

                    rem.append('<i class="fa fa-trash-o">');
                    rem.attr('title', gt('Delete'));
                    rem.click(function(evt) {
                        evt.preventDefault();
                        remove(path, li);
                    });
//                    ctrl.append(edit);
                    ctrl.append(rem);
                }

                var cnt = $('<span class="content">');
                li.append(cnt);

                var title = $('<span data-property="displayName" class="list-title pull-left">');
                title.html('<b>' + Utils.escapeHTML(folder.title) + '</b>');
                cnt.append(title);

                var link = $('<a tabindex="1" class="sectiontitle" target="_blank" style="text-align: left; display: block; max-width: 80%;" href="#"></a>');

                link.data({folder: id});
                link.addClass('deep-link-files');

                var displayPath = '';
                for (var i = 1; i < path.length-1; i++) {
                    displayPath += FolderAPI.getFolderTitle(path[i].title, 30) + ' / ';
                }
                displayPath += FolderAPI.getFolderTitle(folder.title, 30);

                link.text(gt.noI18n(displayPath));
                link.attr('title', gt.noI18n(displayPath));

                cnt.append(link);

                if (oldLi) {
                    oldLi.replaceWith(li);
                } else {
                    folders.append(li);
                }


            }

            if (Utils.getBooleanOption(options, 'useMyFiles', false)) {
                var myFiles = String(FolderAPI.getDefaultFolder('infostore'));

                path(myFiles, function(path) {
                    add(path, true);
                });
            }

            //load all folders saved on account
            //generates its gui elements
            _.each(templateModel.getAll(), function(id) {
                try {
                    delete FolderAPI.pool.models[id];
                    //on this way i force a refresh of the folder, if its deleted
                } catch (e) {
                    if (ox.debug) { console.warn('cannot delete FolderAPI.pool.models', e); }
                }
                path(id, function(path) {
                    add(path);
                });
            });

            var busy = false;

            function clickAdd(evt, oldLi, oldId) {
                evt.preventDefault();

                if (busy) {
                    return;
                }
                busy = true;

                var theTree = null;
                var dia = null;

                new FolderPicker({
                    addClass: '',
                    button: gt('OK'),
                    height: 350,
                    module: 'infostore',
                    persistent: 'folderpopup/filepicker',
                    root: '9',
                    title: Utils.getStringOption(options, 'folderPicker', gt('Select template folder')),

                    initialize: function(dialog, tree) {
                        theTree = tree;
                        dia = dialog;
                    },

                    filter: function (parent, model) {
                        if (FolderAPI.is('trash', model.toJSON())) { return false; }
                    }
                }).done(function (reason) {
                    busy = false;
                    if (reason !== 'ok') {
                        return;
                    }
                    var folder_id = theTree.selection.get();
                    if (folder_id !== undefined && folder_id !== myFiles) {
                        path(folder_id, function(path) {
                            var id = _.last(path).id;
                            if (!templateModel.add(id)) {
                                if (ox.debug) { console.warn('folder already in list', id); }
                                return;
                            }
                            if (oldId) {
                                templateModel.remove(oldId);
                            }
                            add(path, false, oldLi);
                        });
                    }
                });
                dia.getPopup().focus();
            }

            if (!readOnly) {
                /**
                 * standard button for opening FolderPicker
                 * if selected folder is new, it will be added to account & to gui
                 */
                settings.appendButton(Utils.getStringOption(options, 'newFolder', gt('Add new template folder')), clickAdd);
            }

            settings.append($('<div style="display: block; height: 20px">'));

        };
    }

    // exports ================================================================

    return TemplateFolders;

});
