/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Kai Ahrens <kai.ahrens@open-xchange.com>
 */

/* global blankshield */

define('io.ox/office/preview/view/printview', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/io',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/baseframework/app/baseapplication',
    'io.ox/office/baseframework/app/extensionregistry',
    'io.ox/office/preview/view/pageloader',
    'gettext!io.ox/office/preview'
], function (Utils, IO, BaseObject, BaseApp, ExtensionRegistry, PageLoader, gt) {

    'use strict';

    // class PrintView ========================================================

    /**
     * The print view of the OX Preview application. Preloads pages of the previewed
     * document to be rendered on a printer and initiates the final rendering process.
     *
     * @constructor
     *
     * @extends BaseObject
     *
     * @param {PreviewApplication} app
     *  The OX Preview application that has created this view instance.
     */
    function PrintView(app) {
        var app = app,

            model = null,

            // the root node containing all page nodes
            pageContainerNode = $('<div>').addClass('page-container'),

            // all page nodes as permanent jQuery collection, for performance
            pageNodes = $(),

            // the object to load single pages
            pageLoader = new PageLoader(app),

            model = app.getModel(),

            pageCount = model.getPageCount(),

            loadedPageNodes = {},


            pageMarkup = '';

        // base constructors --------------------------------------------------

        BaseObject.call(this);

        // ---------------------------------------------------------------------

        function loadAllPrintPagesIntoNodes(pageNodes) {
            var defs = [];

            // create mark-up for all page nodes
            _.times(pageCount, function (pagePos) {
                var // the deferreds for all pages to be rendered
                    innerDef = $.Deferred(),
                    // the page node of the specified page
                    pageNode = pageNodes.eq(pagePos);

                defs.push(innerDef);

                pageNode.attr('data-rendertype', 'pdf');

                // preload all PDF pages into nodes
                pageLoader.loadPDFPageIntoNode(pageNode, pagePos + 1, {printing: true}).then( function () {
                    loadedPageNodes[pagePos] = pageNode;
                    innerDef.resolve();
                }, function () {
                    pageNode.append($('<div>').addClass('error-message').text(gt('Sorry, this page is not available at the moment.')));
                    innerDef.resolve();
                });
            });

            return $.when.apply($, defs);
        }

        /**
         * Renders the PDF page with given page number and zoom
         * into the given canvas node, that already has the
         * appropriate size and style size attributes set
         * @param {jQuery} canvasNode
         *  The target canvas node, as jQuery object.
         *
         * @param {Number} page
         *  The one-based index of the page to be rendered.
         *
         * @param {Number} pageZoom
         *  The zoom of the page for rendering.
         *
         * @returns {jquery promise}
         *  The jQuery promise that will be resolved when the
         *  page and the page text have been rendered successfully
         *  or rejected on error.
         */
        function renderPrintPage(pageNode, pageNumber) {
            var def = $.Deferred();

            return model.getPDFDocument().getPDFJSPagePromise(pageNumber).then( function (pdfPage) {
                var canvas = pageNode.children('canvas'),
                    printZoom = 1.0,
                    pdfViewport = pdfPage.getViewport(printZoom),
                    pdfPageSize = { width: pdfViewport.width, height: pdfViewport.height };

                pageNode.data('page-zoom', printZoom);
                canvas.attr(pdfPageSize).css(pdfPageSize);

                return pdfPage.render({
                    canvasContext: canvas[0].getContext('2d'),
                    viewport: pdfViewport
                }).then( function() {
                    return def.resolve();
                }, function () {
                    return def.reject();
                });
            });
        }


        // methods ------------------------------------------------------------

        this.doPrint = function () {
            var def = $.Deferred(),
                printWin = blankshield.open('', '_blank', 'toolbar=no, status=no, menubar=no, scrollbars=no, resizable=yes, visible=no, width=0, height=0');

            // close window when done in every case
            def.always( function() {
                printWin.close();
            });

            $(printWin.document).ready( function() {
                pageContainerNode = $('<div>').addClass('page-container');

                $(printWin.document.body).append(pageContainerNode);

                if (pageCount >= 1) {
                    // create mark-up for all page nodes
                    _.times(pageCount, function (pagePos) {
                        pageMarkup += '<div class="page" data-page="' + (pagePos + 1) + '" style="visibility:hidden"></div>';
                    });

                    // insert the mark-up into the container node and initialize the
                    // variable 'pageNodes' BEFORE inserting everything into the living
                    // DOM (variable is used when refreshing the layout)
                    pageContainerNode[0].innerHTML = pageMarkup;
                    pageNodes = pageContainerNode.children();

                    return loadAllPrintPagesIntoNodes(pageNodes).then( function () {
                        var defs = [];

                        // create mark-up for all page nodes
                        _.times(pageCount, function (pagePos) {
                            var // the deferreds for all pages to be rendered
                                innerDef = $.Deferred(),
                                // the page node of the specified page
                                pageNode = pageNodes.eq(pagePos);

                            defs.push(innerDef);

                            renderPrintPage(pageNode, pagePos + 1).then( function() {
                                var canvasNode = pageNode.children('canvas'),
                                    imgNode = $(new Image());

                                imgNode.attr('src', canvasNode[0].toDataURL());
                                canvasNode.replaceWith(imgNode);
                                pageNode.css('visibility', 'visible');

                                innerDef.resolve();
                            }, function() {
                                innerDef.resolve();
                            });
                        });

                        return $.when.apply($, defs).then ( function () {
                            printWin.print();
                            def.resolve();
                        }, function () {
                            $.reject();
                        });

                    });
                }

                def.reject();
            });

            return def.promise();
        };

        // methods ------------------------------------------------------------

        this.doPrintAsPDF = function () {
            var file = app.getFileDescriptor();

            return blankshield.open(app.getServerModuleUrl(IO.CONVERTER_MODULE_NAME, {
                action: 'getdocument',
                documentformat: 'pdf',
                priority: 'instant',
                mimetype: file.file_mimetype ? encodeURIComponent(file.file_mimetype) : '',
                nocache: _.uniqueId() // needed to trick the browser cache (is not evaluated by the backend)
            }), '_blank');
        };
}

    // exports ================================================================

    // derive this class from class BaseObject
    return BaseObject.extend({ constructor: PrintView });
});
