/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/preview/app/controller',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/app/basecontroller'
    ], function (Utils, BaseController) {

    'use strict';

    // class PreviewController ================================================

    /**
     * The controller of the OX Preview application.
     *
     * @constructor
     *
     * @extends BaseController
     *
     * @param {PreviewApplication} app
     *  The OX Preview application that has created this controller instance.
     */
    function PreviewController(app) {

        var // the model instance
            model = null,

            // the view instance
            view = null,

            // all the little controller items
            items = {

                'document/valid': {
                    enable: function () { return model.getPageCount() > 0; }
                },

                // start editing the document (disabled in stand-alone mode)
                'document/edit': {
                    parent: ['document/valid', 'app/bundled'],
                    enable: function () { return app.isDocumentEditable(); },
                    set: function () { app.editDocument(); }
                },

                // view -------------------------------------------------------

                // toggle the main side pane
                'view/sidepane': {
                    parent: 'document/valid',
                    enable: function() { return !Modernizr.touch&&model.getPageCount() > 1; },
                    get: function() { return app.getView().isSidePaneVisible(); },
                    set: function (state) { app.getView().toggleSidePane(state); },
                    shortcut: { keyCode: 'F3', ctrlOrMeta: true, value: function (state) { return !state; } }
                },

                // toggle the search pane
//                'view/searchpane': {
//                    get: function () {
//                        return app.getWindow().search.active;
//                    },
//                    set: function (state) {
//                        var win = app.getWindow();
//                        state = _.isBoolean(state) ? state : !this.getValue();
//                        if (state) {
//                            if (win.search.active) {
//                                win.nodes.searchField.focus();
//                            } else {
//                                win.search.open();
//                            }
//                        } else {
//                            win.search.close();
//                            app.getView().grabFocus();
//                        }
//                    },
//                    // shortcuts always enable the search pane (no toggling)
//                    shortcut: [{ keyCode: 'F', ctrlOrMeta: true, value: true }, { keyCode: 'F3', value: true }],
//                    preserveFocus: true // do not return focus to document
//                },

                // pages ------------------------------------------------------

                'pages/first': {
                    parent: 'document/valid',
                    enable: function () { return view.getPage() > 1; },
                    set: function () { view.showPage('first'); }
                },

                'pages/previous': {
                    parent: 'document/valid',
                    enable: function () { return view.getPage() > 1; },
                    set: function () { view.showPage('previous'); },
                    shortcut: { keyCode: 'PAGE_UP', altOrMeta: true }
                },

                'pages/next': {
                    parent: 'document/valid',
                    enable: function () { return view.getPage() < model.getPageCount(); },
                    set: function () { view.showPage('next'); },
                    shortcut: { keyCode: 'PAGE_DOWN', altOrMeta: true }
                },

                'pages/last': {
                    parent: 'document/valid',
                    enable: function () { return view.getPage() < model.getPageCount(); },
                    set: function () { view.showPage('last'); }
                },

                'pages/current': {
                    parent: 'document/valid',
                    get: function () { return view.getPage(); },
                    set: function (page) { view.showPage(page); }
                },

                // zoom -------------------------------------------------------


                'zoom/dec': {
                    parent: 'document/valid',
                    enable: function () { return view.getZoomFactor() > view.getMinZoomFactor(); },
                    set: function () { view.decreaseZoomLevel(); },
                    shortcut: { charCode: '-' }
                },

                'zoom/inc': {
                    parent: 'document/valid',
                    enable: function () { return view.getZoomFactor() < view.getMaxZoomFactor(); },
                    set: function () { view.increaseZoomLevel(); },
                    shortcut: { charCode: '+' }
                },

                'zoom/percentage': {
                    parent: 'document/valid',
                    get: function () { return view.getZoomFactor() / 100; },
                    set: function (zoom) { view.setZoomType(zoom * 100); }
                },

                'zoom/page': {
                    parent: 'document/valid',
                    get: function() { return view.getZoomType()==='page'; },
                    set: function () { view.setZoomType('page'); }
                },

                'zoom/width': {
                    parent: 'document/valid',
                    get: function() { return view.getZoomType()==='width'; },
                    set: function () { view.setZoomType('width'); }
                }
            };

        // base constructor ---------------------------------------------------

        BaseController.call(this, app);

        // initialization -----------------------------------------------------

        // register item definitions
        this.registerDefinitions(items);

        // initialization after construction
        app.onInit(function () {
            // model and view are not available at construction time
            model = app.getModel();
            view = app.getView();
        });

    } // class PreviewController

    // exports ================================================================

    // derive this class from class BaseController
    return BaseController.extend({ constructor: PreviewController });

});
