/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Kai Ahrens <kai.ahrens@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/preview/app/application',
    ['io.ox/core/collection',
     'io.ox/core/api/folder',
     'io.ox/office/tk/utils',
     'io.ox/office/tk/io',
     'io.ox/office/baseframework/app/baseapplication',
     'io.ox/office/baseframework/app/extensionregistry',
     'io.ox/office/preview/model/model',
     'io.ox/office/preview/view/view',
     'io.ox/office/preview/view/printview',
     'io.ox/office/preview/app/controller',
     'io.ox/office/baseframework/pdf/pdfdocument',
     'gettext!io.ox/office/preview'
    ], function (Collection, FolderAPI, Utils, IO, BaseApplication, ExtensionRegistry, PreviewModel, PreviewView, PrintView, PreviewController, PDFDocument, gt) {

    'use strict';

    var // the module name of this application
        MODULE_NAME = 'io.ox/office/preview';

    // class PreviewApplication ===============================================

    /**
     * The OX Preview application used to view a wide range of document types.
     *
     * @constructor
     *
     * @extends BaseApplication
     *
     * @param {Object} launchOptions
     *  All options passed to the core launcher (the ox.launch() method).
     *
     * @param {Object} [appOptions]
     *  Static application options that have been passed to the static method
     *  BaseApplication.createLauncher().
     */
    var PreviewApplication = BaseApplication.extend({ constructor: function (launchOptions, appOptions) {

        var // self reference
            self = this,

            // the unique job identifier to be used for page requests
            jobId = null,

            // user may edit the document in an OX Documents edit application
            isEditable = false;

        // base constructor ---------------------------------------------------

        BaseApplication.call(this, PreviewModel, PreviewView, PreviewController, importDocument, launchOptions, appOptions);

        // private methods ----------------------------------------------------

        /**
         * Loads the first page of the document described in the current file
         * descriptor.
         *
         * @param {Object} [point]
         *  The save point, if called from fail-restore.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  initial data of the preview document has been loaded; or rejected
         *  when an error has occurred.
         */
        function importDocument() {

            var // the PDF document
                pdfDocument,
                // promise of the initial image server request
                documentImagesPromise = null,
                // promise of a Deferred that returns the collection properties
                collectionPromise = null,
                // promise of a Deferred that returns the folder properties
                folderPromise = null,
                // the file descriptor
                file = self.getFileDescriptor();

            //disable user-select on body, that only the "user-select-text" classes can really select the text
            self.registerVisibleStateHandler(function(state) {
                $('body').toggleClass('user-select-none', state);
            });

            // disable drop events
            self.getWindowNode().on('drop dragstart dragover', false);

            // immediately show the busy blocker element
            self.getView().enterBusy({ cancelHandler: _.bind(self.quit, self), showFileName: true, immediate: true });

            pdfDocument = new PDFDocument(self.getServerModuleUrl(IO.CONVERTER_MODULE_NAME, {
                action: 'getdocument',
                documentformat: 'pdf',
                priority: 'instant',
                mimetype: file.file_mimetype ? encodeURIComponent(file.file_mimetype) : '',
                nocache: _.uniqueId() // needed to trick the browser cache (is not evaluated by the backend)
            }));

            // begin stateful image conversion for the document on the server
            documentImagesPromise = self.sendFileRequest(IO.CONVERTER_MODULE_NAME, {
                action: 'convertdocument',
                convert_action: 'beginconvert',
                convert_format: 'image'
                // convert_format: 'html'
            });

            // check edit rights of user, check whether file is located in the trash folder
            if ('source' in file) {
                // TODO: file is an attachment, e.g. of a mail or task, editable?
                collectionPromise = null;
                folderPromise = null;
            } else {
                collectionPromise = new Collection(file).getProperties();
                folderPromise = FolderAPI.get({ folder: file.folder_id });
            }

            // wait for all promises and check the response objects
            return $.when(pdfDocument.getLoadPromise(), documentImagesPromise, collectionPromise, folderPromise).then(function (pageCount, imageConvertData, collectionData, folderData) {
                // check that no error has occurred while opening the document
                if (_.isString(imageConvertData.JobID) && (imageConvertData.JobID.length > 0) && _.isNumber(pageCount) && (pageCount > 0)) {

                    // store whether the document can really be edited
                    isEditable = _.isObject(collectionData) && collectionData.one && collectionData.read && collectionData.modify && _.isObject(folderData) && !FolderAPI.is('trash', folderData);

                    // further initialization after all imports
                    jobId = imageConvertData.JobID;

                    // set the PDF.js document promise with page count,
                    // if available, or just the page count at the model
                    self.getModel().setPDFDocument(pdfDocument);

                    // resolve the promise with the response data for the beginConvert request
                    return imageConvertData;
                }

                // special text for password protected files
                var errorCause = Utils.getStringOption(imageConvertData, 'cause');

                if (errorCause === 'passwordProtected') {
                    imageConvertData.message = gt('The document is protected by a password.');
                } else if (errorCause === 'maxSourceSizeExceeded') {
                    //#. %1$s is the maximum allowed source file size in MB
                    imageConvertData.message = gt('The document exceeds the maximum permitted size of %1$s MB.', imageConvertData.MaxSourceSizeMB);
                }

                // convert result to a rejected Deferred object
                return $.Deferred().reject(imageConvertData);
            });
        }

        /**
         * Sends a close notification to the server, before the application
         * will be really closed.
         */
        function quitHandler() {
            if (!self.isImportFailed()) {
                return self.sendPreviewRequest({ convert_action: 'endconvert' });
            }
        }

        // methods ------------------------------------------------------------

        this.sendPreviewRequest = function (params, options) {

            if (!_.isString(jobId)) { return $.Deferred().reject(); }

            // add correct action and job identifier to the request parameters
            params = _.extend({}, params, { action: 'convertdocument', job_id: jobId });

            // send the server request
            return this.sendFileRequest(IO.CONVERTER_MODULE_NAME, params, options);
        };

        this.getPreviewModuleUrl = function (options) {
            return _.isString(jobId) ? this.getServerModuleUrl(IO.CONVERTER_MODULE_NAME, Utils.extendOptions({
                action: 'convertdocument',
                job_id: jobId
            }, options)) : undefined;
        };

        /**
         * Returns whether the current document can be edited with one of the
         * OX Document edit applications.
         */
        this.isDocumentEditable = function () {
            return isEditable && _.isString(jobId) && ExtensionRegistry.isEditable(this.getFullFileName());
        };

        /**
         * Launches the appropriate OX Document edit applications for the
         * current document.
         */
        this.editDocument = function () {

            var // the file descriptor of the current document
                file = this.getFileDescriptor(),
                // the edit application module identifier
                editModule = this.isDocumentEditable() ? ExtensionRegistry.getEditModule(this.getFullFileName()) : '';

            // TODO: edit mail attachments and task attachments
            if (editModule.length > 0) {

                // launch the correct edit application
                if (ExtensionRegistry.isNative(this.getFullFileName())) {
                    ox.launch(editModule + '/main', { action: 'load', file: file });
                } else if (ExtensionRegistry.isConvertible(this.getFullFileName())) {
                    ox.launch(editModule + '/main', { action: 'convert', folderId: file.folder_id, templateFile: file, preserveFileName: true });
                } else {
                    Utils.error('PreviewApplication.editDocument(): unknown document type');
                    return;
                }

                // close this application
                this.quit();
            }
        };

        /**
         * Currently using old print as pdf functionality.
         */
        this.print = function () {
            new PrintView(this).doPrintAsPDF();
        };

        // initialization -----------------------------------------------------

        this.registerQuitHandler(quitHandler);

        // destroy all class members
        this.registerDestructor(function () {
            launchOptions = appOptions = null;
        });

    }}); // class PreviewApplication

    // static methods ---------------------------------------------------------

    /**
     * Replacement for the generic method BaseApplication.createLauncher()
     * without parameters, to launch viewer applications.
     */
    PreviewApplication.createLauncher = function () {
        return BaseApplication.createLauncher(MODULE_NAME, PreviewApplication, { chromeless: true });
    };

    // exports ================================================================

    return PreviewApplication;

});
