/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/editview',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/baseframework/app/extensionregistry',
     'io.ox/office/baseframework/view/baseview',
     'io.ox/office/editframework/utils/editconfig',
     'io.ox/office/editframework/view/editlabels',
     'io.ox/office/editframework/view/toolbartabcollection',
     'io.ox/office/editframework/view/toppane',
     'io.ox/office/editframework/view/edittoolpane',
     'io.ox/office/editframework/view/searchpane',
     'io.ox/office/editframework/view/operationspane',
     'io.ox/office/editframework/view/clipboardpane',
     'io.ox/office/editframework/view/editlabels',
     'io.ox/office/editframework/view/editcontrols',
     'io.ox/office/editframework/view/editdialogs',
     'io.ox/office/editframework/view/popup/userslayermenu',
     'gettext!io.ox/office/editframework',
     'less!io.ox/office/editframework/view/editstyle'
    ], function (Utils, KeyCodes, ExtensionRegistry, BaseView, Config, EditLabels, ToolBarTabCollection, TopPane, EditToolPane, SearchPane, OperationsPane, ClipboardPane, Labels, Controls, Dialogs, UsersLayerMenu, gt) {

    'use strict';

    // class EditView =========================================================

    /**
     * @constructor
     *
     * @extends BaseView
     *
     * @param {EditApplication} app
     *  The application containing this view instance.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options that are supported by the
     *  constructor of the base class BaseView, with the following additions
     *  and differences:
     *  @param {Function} [initOptions.initDebugHandler]
     *      A callback handler function called for additional initialization of
     *      debug settings. Will be called in debug mode ONLY. Receives the
     *      following parameters:
     *      (1) {OperationsPane} operationsPane
     *          The view pane containing the operations log, and additional
     *          debugging information to be displayed.
     *      (2) {ClipboardPane} clipboardPane
     *          The view pane with the HTML mark-up contained in the browser
     *          clipboard.
     *      May return a promise to be able to run asynchronous code during
     *      initialization.
     *  @param {Function} [initOptions.initGuiHandler]
     *      A callback handler function called to initialize the graphical
     *      elements of this view instance, after the document has been
     *      imported. Receives the following parameters:
     *      (1) {CompoundButton} viewMenuGroup
     *          The 'View' drop-down menu shown in the tool pane, containing
     *          all view settings for the document.
     *      May return a promise to be able to run asynchronous code during
     *      initialization.
     *  @param {Function} [initOptions.initDebugGuiHandler]
     *      A callback handler function called to add graphical elements for
     *      debugging purposes to this view instance. Will be called in debug
     *      mode ONLY. Will be called after the GUI initialization handler (see
     *      option 'initGuiHandler'). Receives the following parameters:
     *      (1) {CompoundButton} viewMenuGroup
     *          The 'View' drop-down menu shown in the tool pane, containing
     *          all view settings for the document.
     *      (2) {CompoundButton} actionMenuGroup
     *          The 'Actions' drop-down menu shown in the debug tool bar,
     *          containing action commands for debugging.
     *      May return a promise to be able to run asynchronous code during
     *      initialization.
     */
    function EditView(app, initOptions) {

        var // self reference
            self = this,

            // beneath this min height, the tool-/tabbars should be combined
            // Nexus 7 = min resolution 601
            minHeight = 610,

            // show combined Panes
            panesCombined = false,

            // the tool bar tab collection
            toolBarTabs = new ToolBarTabCollection(app),

            // the top-level view pane containing global controls and the tool bar tabs
            topPane = null,

            // the main tool pane below the top-level view pane containing the tool bars
            toolPane = null,

            // the search/replace pane below the tool pane
            searchPane = null,

            // collaborator list
            collaboratorPopup = null;

        // base constructor ---------------------------------------------------

        BaseView.call(this, app, Utils.extendOptions(initOptions, {
            initHandler: initHandler,
            initGuiHandler: initGuiHandler,
            classes: 'io-ox-office-edit-main'
        }));

        // private methods ----------------------------------------------------

        /**
         * Builds generic contents of this edit view instance: the side pane,
         * the overlay panes, and the debug view panes. After that, calls the
         * passed callback handler to build the application-specific view
         * elements.
         */
        function initHandler() {

            var // the operations pane (debug mode only)
                operationsPane = null,
                // the clipboard pane (debug mode only)
                clipboardPane = null;

            // determine whether the panes are combined or not
            if ((Utils.TOUCHDEVICE && Math.min(Utils.getScreenWidth(), Utils.getScreenHeight()) <= minHeight) || (Config.DEBUG && Config.getUrlFlag('office:panes-combined'))) {
                panesCombined = true;
            }

            // disable internal browser handles on tables and drawings after edit mode has changed
            self.listenTo(app.getModel(), 'change:editmode', function () { self.disableBrowserEditControls(); });

            // create the tool bars (initialize themselves internally)
            self.addPane(topPane = new TopPane(app, toolBarTabs));
            self.addPane(toolPane = new EditToolPane(app, toolBarTabs, { landmark: false }));
            self.addPane(searchPane = new SearchPane(app).hide());

            // deactivate tool bar tabs while tool pane is hidden
            toolPane.on('pane:hide', function () { toolBarTabs.activateTab(null); });

            // hide tool bars after double click on tab buttons
            topPane.getNode().find('.group.tab-group').on('dblclick', function () {
                toolPane.hide();
            });

            // update controller immediately when toggling the collaborator pop-up menu
            this.listenTo(collaboratorPopup, 'popup:show popup:hide', function () {
                app.getController().update();
            });

            // initialize debug panes before other custom panes of the application
            if (Config.DEBUG) {

                // create and insert the clipboard pane, hide it initially
                clipboardPane = new ClipboardPane(app);
                clipboardPane.hide();
                self.addPane(clipboardPane);

                // create and insert the operations pane, hide it initially
                operationsPane = new OperationsPane(app);
                operationsPane.hide();
                self.addPane(operationsPane);

                // always show the operations pane when an internal application error occurs
                self.listenTo(app, 'docs:state:error', function () { operationsPane.show(); });

                // create controller items for debug functionality
                app.getController().registerDefinitions({
                    'debug/operationspane': {
                        parent: 'debug/enabled',
                        get: function () { return operationsPane.isVisible(); },
                        set: function (state) { operationsPane.toggle(state); }
                    },
                    'debug/clipboardpane': {
                        parent: 'debug/enabled',
                        get: function () { return clipboardPane.isVisible(); },
                        set: function (state) { clipboardPane.toggle(state); }
                    },
                    'debug/highlight': {
                        parent: 'debug/enabled',
                        get: function () { return app.getWindowNode().hasClass('debug-highlight'); },
                        set: function (state) { app.getWindowNode().toggleClass('debug-highlight', state); }
                    },
                    'debug/toolbars/white': {
                        parent: 'debug/enabled',
                        get: function () { return app.getWindowNode().hasClass('debug-white'); },
                        set: function (state) { app.getWindowNode().toggleClass('debug-white', state); }
                    },
                    'debug/realtime/trigger': {
                        parent: 'debug/enabled',
                        set: function (type) { app.debugTriggerRealtimeEvent(type); }
                    },
                    'debug/realtime/sendinvalidoperation': {
                        parent: ['debug/enabled', 'document/editable'],
                        set: function (options) { app.debugSendInvalidOperation(options); }
                    },
                    'debug/operations/apply': {
                        parent: ['debug/enabled', 'document/editable'],
                        set: function (operation) { app.getModel().applyOperations(operation); }
                    },
                    'debug/quit': {
                        parent: 'debug/enabled',
                        set: function (type) { app.debugQuit(type); }
                    }
                });
            }

            // call initialization handler passed by sub class
            var result = Utils.getFunctionOption(initOptions, 'initHandler', $.noop).call(self);

            // call debug initialization handler passed by sub class
            if (Config.DEBUG) {
                result = $.when(result).then(_.bind(Utils.getFunctionOption(initOptions, 'initDebugHandler', $.noop), self, operationsPane, clipboardPane));
            }

            return result;
        }

        /**
         * Initialization after importing the document. Creates all tool boxes
         * in the side pane and overlay pane. Needed to be executed after
         * import, to be able to hide specific GUI elements depending on the
         * file type.
         */
        function initGuiHandler() {

            var
                renameToolbar   = null,
                saveToolbar     = null,
                commonToolbar   = null,
                actionToolbar   = null,

                btnSaveAs           = new Controls.Button({ label: gt('Save as'), value: 'file' }),
                btnSaveAsTemplate   = new Controls.Button({ label: gt('Save as template'), value: 'template' }),
                btnDownload         = new Controls.Button({ icon: 'fa-cloud-download', tooltip: gt('Download'), classes: 'link-style', dropDownVersion: { label: gt('Download') } }),
                btnPrint            = new Controls.Button({ icon: 'fa-print', tooltip: gt('Print as PDF'), classes: 'link-style', dropDownVersion: { label: gt('Print as PDF') } }),
                btnSendAsMail       = new Controls.Button({ icon: 'fa-envelope-o', tooltip: gt('Send as mail'), classes: 'link-style', dropDownVersion: { label: gt('Send as mail') } }),
                btnReload           = new Controls.Button({ icon: 'fa-repeat', label: gt('Reload'), tooltip: gt('Reload document'), classes: 'link-style' }),
                btnAcquireEdit      = new Controls.AcquireEditButton(app, { classes: 'link-style' }),

                cBoxAutosave        = new Controls.CheckBox({ label: gt('AutoSave') }),

                fileNameField       = new Controls.FileNameField(app);


            // -----------------------------------------------------
            // TABS
            //      prepare all tabs (for normal or combined panes)
            // -----------------------------------------------------
            self.createToolBarTab('file', { label: Labels.FILE_HEADER_LABEL, priority: 99 });

            // -----------------------------------------------------
            // TOOLBARS
            //      prepare all toolbars (for normal or combined panes)
            // -----------------------------------------------------
            if (!self.panesCombined()) {
                renameToolbar   = self.createToolBar('file');
                saveToolbar     = self.createToolBar('file', {priority: 2, visibleKey: 'app/bundled'});
            }
            commonToolbar       = self.createToolBar('file', {priority: 3, label: gt('Actions'), classes: 'link-style', prepareShrink: true });
            actionToolbar       = self.createToolBar('file', {priority: 1, tooltip: gt('actions'), classes: 'link-style' });

            // -----------------------------------------------------
            // CONTROLS
            //      add all controls
            // -----------------------------------------------------
            if (!self.panesCombined()) {
                renameToolbar
                    .addGroup('document/rename', fileNameField);

                saveToolbar
                    .addGroup(null, new Controls.CompoundButton(app, { label: gt('Save in Drive'), classes: 'link-style', smallerVersion:{ icon: 'fa-save', hideLabel: true } })
                        .addGroup('document/saveas/dialog', btnSaveAs)
                        .addGroup('document/saveas/dialog', btnSaveAsTemplate)
                        .addSeparator()
                        .addGroup('document/autosave', cBoxAutosave)
                    );
            }
            commonToolbar
                .addGroup('document/download', btnDownload)
                .addGap()
                .addGroup('document/print', btnPrint)
                .addGap()
                .addGroup('document/sendmail', btnSendAsMail, { visibleKey: 'app/bundled' });

            actionToolbar
                .addGroup('document/acquireedit', btnAcquireEdit, { visibleKey: 'document/acquireedit' })
                .addGroup('document/reload', btnReload, { visibleKey: 'document/reload' });


            // register double-click handler for the application launcher
            self.listenTo(app.getImportPromise(), 'done', function () {
                $('#io-ox-topbar .launcher[data-app-guid="' + app.guid + '"] a.apptitle').on('dblclick', function () {
                    if (fileNameField.isEnabled()) {
                        app.getController().executeItem('view/toolbars/tab', 'file', { focusTarget: fileNameField });
                    } else {
                        app.getView().grabFocus();
                    }
                });
            });

            // call initialization handler passed by sub class
            var result = Utils.getFunctionOption(initOptions, 'initGuiHandler', $.noop).call(self, topPane.getViewMenuGroup());

            // debug initialization
            if (Config.DEBUG) {
                result = $.when(result).then(initDebugGuiHandler);
            }

            // immediately activate the first available tool bar tab
            return $.when(result).done(function () { toolBarTabs.activateFirstTab(); });
        }

        /**
         * Additional initialization of debug GUI after importing the document.
         */
        function initDebugGuiHandler() {

            var // the 'View' drop-down menu
                viewMenuGroup = topPane.getViewMenuGroup(),
                // the drop-down menu with debug actions
                actionMenuGroup = null;

            // add view options
            viewMenuGroup
                .addSectionLabel(_.noI18n('Debug'))
                .addGroup('debug/operationspane', new Controls.CheckBox({ label: _.noI18n('Show operations panel') }))
                .addGroup('debug/clipboardpane',  new Controls.CheckBox({ label: _.noI18n('Show clipboard panel') }))
                .addGroup('debug/highlight',      new Controls.CheckBox({ label: _.noI18n('Highlight DOM elements') }))
                .addGroup('debug/toolbars/white', new Controls.CheckBox({ label: _.noI18n('White toolbar panels') }));

            // create the drop-down list for the debug actions
            actionMenuGroup = new Controls.CompoundButton(app, { icon: 'fa-wrench', label: EditLabels.ACTIONS_LABEL, tooltip: _.noI18n('Additional debug actions'), classes: 'link-style' })
                // register realtime debug actions
                .addSectionLabel(_.noI18n('Offline mode'))
                .addGroup('debug/realtime/trigger', new Controls.Button({ label: _.noI18n('Switch to offline mode'),      value: 'offline' }))
                .addGroup('debug/realtime/trigger', new Controls.Button({ label: _.noI18n('Switch to online mode'),       value: 'online' }))
                .addSectionLabel(_.noI18n('Realtime connection'))
                .addGroup('debug/realtime/trigger', new Controls.Button({ label: _.noI18n('Simulate connection reset'),   value: 'reset' }))
                .addGroup('debug/realtime/trigger', new Controls.Button({ label: _.noI18n('Simulate connection timeout'), value: 'timeout' }))
                .addGroup('debug/realtime/trigger', new Controls.Button({ label: _.noI18n('Simulate connection hangup'),  value: 'hangup' }))
                // register debug actions for operations
                .addSectionLabel(_.noI18n('Operations'))
                .addGroup('debug/operations/apply',              new Controls.Button({ label: _.noI18n('Apply operation with invalid name'), value: { name: '_invalid_operation_' } }))
                .addGroup('debug/operations/apply',              new Controls.Button({ label: _.noI18n('Apply operation with invalid position'), value: { name: 'insertText', start: [987654321, 0], text: 'test' } }))
                .addGroup('debug/realtime/sendinvalidoperation', new Controls.Button({ label: _.noI18n('Send operation with invalid OSN'), value: { operation: { name: 'delete', start: [987654321, 0]}, badOsn: true }}))
                .addGroup('debug/realtime/sendinvalidoperation', new Controls.Button({ label: _.noI18n('Send invalid operation'), value: { operation: { name: 'createError', start: [987654321, 0]}}}))
                // register debug actions for application quit
                .addSectionLabel(_.noI18n('Quit'))
                .addGroup('debug/quit', new Controls.Button({ label: _.noI18n('Close document with unsaved changes'), value: 'unsaved' }))
                .addGroup('debug/quit', new Controls.Button({ label: _.noI18n('Close document with server error'),    value: 'error' }));

            // create a new tool bar tab for debugging
            self.createToolBarTab('debug', { label: _.noI18n('Debug'), visibleKey: 'debug/enabled', priority: 9999 });
            self.createToolBar('debug').addGroup(null, actionMenuGroup);

            // call initialization handler passed by sub class
            return Utils.getFunctionOption(initOptions, 'initDebugGuiHandler', $.noop).call(self, viewMenuGroup, actionMenuGroup);
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the top pane containing the tool bar tab buttons, and other
         * global control elements.
         *
         * @returns {TopPane}
         *  The top pane of this application.
         */
        this.getTopPane = function () {
            return topPane;
        };

        /**
         * Returns the tool pane containing all existing tool bars.
         *
         * @returns {EditToolPane}
         *  The tool pane of this application.
         */
        this.getToolPane = function () {
            return toolPane;
        };

        /**
         * Returns the search pane.
         *
         * @returns {Pane}
         *  The search pane of this application.
         */
        this.getSearchPane = function () {
            return searchPane;
        };

        /**
         * Returns whether the search pane is currently visible.
         *
         * @returns {Boolean}
         *  Whether the search pane is currently visible.
         */
        this.isSearchActive = function () {
            return searchPane.isVisible();
        };

        /**
         * Returns the collection containing data for all registered tool bar
         * tabs.
         *
         * @returns {ToolBarTabCollection}
         *  The collection containing data for all registered tool bar tabs.
         */
        this.getToolBarTabs = function () {
            return toolBarTabs;
        };

        /**
         * Creates a new tab button in the top view pane used to control the
         * visibility of one or more tool bar components in the tool pane of
         * the application view.
         *
         * @param {String} tabId
         *  The unique identifier for the new tab button.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options supported by the method
         *  ToolBarTabCollection.createTab() used to control the behavior and
         *  visibility of the tab, and RadioGroup.createOptionButton() used to
         *  create the tab button control (especially label and icon settings).
         *
         * @returns {EditView}
         *  A reference to this instance.
         */
        this.createToolBarTab = function (tabId, options) {
            toolBarTabs.createTab(tabId, options);
            return this;
        };

        /**
         * Creates a new tool bar in the tool pane of the application view.
         *
         * @param {String} tabId
         *  The identifier of the tab button the tool bar will be associated
         *  with.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options supported by the method
         *  EditToolPane.createToolBar().
         *
         * @returns {ToolBar}
         *  The new tool bar instance.
         */
        this.createToolBar = function (tabId, options) {
            return toolPane.createToolBar(tabId, options);
        };

        /**
         * Creates a new editable container node for clipboard functionality,
         * and inserts it into the hidden container node of the application.
         *
         * @returns {jQuery}
         *  The new empty clipboard container node.
         */
        this.createClipboardNode = function () {
            // clipboard node must be content-editable to allow system copy-to-clipboard
            // clipboard node must contain 'user-select-text' to be focusable in Chrome
            var clipboardNode = $('<div>', { contenteditable: true, 'data-focus-role': 'clipboard' }).addClass('clipboard user-select-text');
            this.insertHiddenNodes(clipboardNode);
            return clipboardNode;
        };

        /**
         * Disables the internal resize handles shown for of tables, drawing
         * objects, etc. provided by the browser edit mode (content-editable
         * elements).
         *
         * @returns {EditView}
         *  A reference to this instance.
         */
        this.disableBrowserEditControls = function () {
            // execute delayed to prevent conflicts with ongoing event handling
            this.executeDelayed(function () {
                try {
                    // disable FireFox table manipulation handlers
                    window.document.execCommand('enableObjectResizing', false, false);
                    window.document.execCommand('enableInlineTableEditing', false, false);
                } catch (ex) {
                }
            });
            return this;
        };

        /**
         * Closes the passed dialog automatically, if the document switches to
         * read-only mode.
         *
         * @constructor
         *
         * @param {ModalDialog} dialog
         *  The dialog instance that will be closed on read-only mode.
         */
        this.closeDialogOnReadOnlyMode = function (dialog) {

            var // the document model
                model = app.getModel();

            // close the dialog if document switches to read-only mode
            function changeEditModeHandler(event, editMode) {
                if (!editMode) { dialog.close(); }
            }

            // listen to read-only mode, close dialog automatically
            model.on('change:editmode', changeEditModeHandler);

            // unregister listener after closing the dialog
            dialog.on('close', function () {
                model.off('change:editmode', changeEditModeHandler);
            });
        };

        /**
         * Toggles the visibility of the collaborator popup
         *
         * @param {Boolean} visibility
         */
        this.toggleCollaboratorPopup = function (visibility) {
            collaboratorPopup.toggle(visibility);
        };

        /**
         * Gets the instance of the collaborator popup
         *
         * @returns {UsersLayerMenu}
         */
        this.getCollaboratorPopup = function () {
            return collaboratorPopup;
        };

        /**
         * Shows a text input dialog that allows to enter a new file name for
         * this document.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the document has been renamed
         *  successfully, or rejected when the dialog has been canceled, or
         *  renaming the document has failed.
         */
        this.showRenameDialog = function () {

            var // the dialog instance
                dialog = new Dialogs.ModalInputDialog({
                    title: gt('Rename document'),
                    okLabel: gt('Rename'),
                    value: this.getShortFileName(),
                    placeholder: gt('Document name')
                });

            // close dialog automatically after losing edit rights
            this.closeDialogOnReadOnlyMode(dialog);

            // show the dialog, and try to rename the document
            return dialog.show().then(function () {
                return app.rename(dialog.getText());
            });
        };

        /**
         * Saves the current document to the specified folder using the
         * current file name without extension and prefixing it with
         * the phrase 'Copy of'.
         *
         * @param {String} [type='file']
         *  The target file type. The following file types are supported:
         *  - 'file' (default): Save with the current extension.
         *  - 'template': Save as template file.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  file has been copied successfully; or that will be rejected, if
         *  copying the file has failed.
         */
        this.showSaveAsDialog = function (type) {

            var // the current file name
                oldFileName = app.getShortFileName(),
                // the new file name
                newFileName = null,
                // template extension
                templateExt = null,
                // title of the dialog
                title = null,
                // the dialog instance
                dialog = null;

            // prepare file extension and dialog title
            if (type === 'template') {
                newFileName = oldFileName;
                // use original extension as fallback - possible if a template
                // document should be stored via save as as template.
                templateExt = app.getFileExtension();
                templateExt = ExtensionRegistry.getTemplateExtensionForFileName(app.getFullFileName()) || templateExt;
                //#. %1$s is the new file extension used for the template file
                //#, c-format
                title = gt('Save as template (%1$s)', _.noI18n(templateExt));
            } else {
                //#. %1$s is the original file name that should be prefixed by 'Copy of' to create a new file that is a copy of the original file
                //#, c-format
                newFileName = gt('Copy of %1$s', _.noI18n(oldFileName));
                title = gt('Save as');
            }

            // create the dialog
            dialog = new Dialogs.ModalSaveAsDialog({
                title: title,
                value: newFileName,
                folderPicker: type !== 'template'
            });

            // show the dialog, and perform the save action
            return dialog.show().then(function () {
                return app.saveDocumentAs(dialog.getText(), dialog.getSelectedFolderId(), type);
            });
        };

        /**
         * return whether tool-/tabbars should be combined or not
         *
         * @return {Boolean}
         *  'true' for combining the bars
         */
        this.panesCombined = function(){
            return panesCombined;
        };

        // undo manager -------------------------------------------------------

        /**
         * Returns whether at least one undo action is available on the undo
         * stack.
         *
         * @returns {Boolean}
         *  Whether at least one undo action is available on the stack.
         */
        this.isUndoAvailable = function () {
            return app.getModel().getUndoManager().getUndoCount() > 0;
        };

        /**
         * Applies the topmost undo action on the undo stack.
         *
         * @returns {jQuery.Promise}
         *  A Promise that will be resolved after the undo action has been
         *  applied.
         */
        this.undo = function () {
            return app.getModel().getUndoManager().undo(1);
        };

        /**
         * Returns whether at least one redo action is available on the redo
         * stack.
         *
         * @returns {Boolean}
         *  Whether at least one redo action is available on the stack.
         */
        this.isRedoAvailable = function () {
            return app.getModel().getUndoManager().getRedoCount() > 0;
        };

        /**
         * Applies the topmost redo action on the redo stack.
         *
         * @returns {jQuery.Promise}
         *  A Promise that will be resolved after the redo action has been
         *  applied.
         */
        this.redo = function () {
            return app.getModel().getUndoManager().redo(1);
        };

        // initialization -----------------------------------------------------

        // early initialization after MVC construction
        app.onInit(function () {
            collaboratorPopup = new UsersLayerMenu(app);
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            toolBarTabs.destroy();
            collaboratorPopup.destroy();
            app = initOptions = self = toolBarTabs = topPane = toolPane = collaboratorPopup = null;
        });

    } // class EditView

    // exports ================================================================

    // derive this class from class BaseView
    return BaseView.extend({ constructor: EditView });

});
