/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/edittoolpane',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/forms',
     'io.ox/office/baseframework/view/toolpane',
     'io.ox/office/baseframework/view/toolbar',
     'io.ox/office/editframework/view/editlabels',
     'io.ox/office/editframework/view/editcontrols',
     'gettext!io.ox/office/editframework'
    ], function (Utils, Forms, ToolPane, ToolBar, Labels, Controls, gt) {

    'use strict';

    // class EditToolPane =====================================================

    /**
     * Represents the main tool pane in OX Document applications shown below
     * the top-level view pane, containing the currently active tool bar, and
     * the 'View' drop-down menu.
     *
     * @constructor
     *
     * @extends ToolPane
     *
     * @param {EditApplication} app
     *  The application containing this tool pane.
     *
     * @param {ToolBarTabCollection} toolBarTabs
     *  The collection of all registered tool bar tabs.
     */
    function EditToolPane(app, toolBarTabs, initOptions) {

        var // self reference
            self = this,

            // view reference
            view = app.getView(),

            // the tool bars, mapped by tab identifier
            toolBarsMap = {},

            // the wrapper nodes for the center tool bar area
            centerWrapperNode = null,
            centerAreaNode = null,

            // state of "paneLayoutHandler" (to prevent multiple calls)
            reLayout = false,
            // store for last known pane width
            paneWidthStore = 0,
            // possibility to force invoking the paneLayoutHandler
            forceLayoutHandler = false,

            // the new menu including the tab buttons to activate the different edit tool bars, as drop-down list
            flatTabView = null,
            // new document close button for flat-pane
            closeBtn = null,
            // new search button for the new flat-pane-menu
            searchBtn = null,

            // reference to the topPane
            topPane = view.getTopPane(),
            // reference to the view-menu-group
            viewMenuGroup = topPane.getViewMenuGroup(),
            // reference to the view-menu
            viewMenu = viewMenuGroup.getMenu();

        // base constructor ---------------------------------------------------

        ToolPane.call(this, app, Utils.extendOptions({
            position: 'top',
            classes: 'edit-pane standard-design',
            cursorNavigate: true
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * Shows or hides the tool bars associated to the specified tab
         * identifier, according to the own visibility of the tool bars if
         * bound to a controller item.
         */
        function toggleToolBars(tabId, state) {
            if (_.isString(tabId)) {
                _.each(toolBarsMap[tabId], function (toolBarInfo) {
                    if (_.isObject(toolBarInfo.shrinkedToolBar)) {
                        toolBarInfo.toolBar.toggle(state && toolBarInfo.visible);

                        if (toolBarInfo.shrinked && toolBarInfo.shrinkedMenu.hasVisibleGroups()) {
                            toolBarInfo.shrinkedToolBar.toggle(state && toolBarInfo.visible);
                        }else {
                            toolBarInfo.shrinkedToolBar.hide();
                        }

                    } else {
                        toolBarInfo.toolBar.toggle(state && toolBarInfo.visible);
                    }
                });
            }
        }

        /**
         * Updates the layout of tool bars according to available space in the
         * tool pane.
         */
        var paneLayoutHandler = this.createDebouncedMethod(Utils.NOOP, function () {
            // check if another process/trigger is running this already.
            // cancel in this case, to prevent flashing controls
            if (reLayout) { return; }
            reLayout = true;

            var // pane width
                paneWidth = centerWrapperNode.width(),
                // tool bars width
                toolbarsWidth = centerAreaNode.outerWidth(),
                // overflow size of the center tool bar area
                overflowWidth = toolbarsWidth - paneWidth,
                // all tool bars of the active tab
                toolbars = _.sortBy(toolBarsMap[toolBarTabs.getActiveTabId()], _.property('priority')),
                // shrinkable tool bars
                shrinkableToolbars = _.filter(toolbars, function (bar) { return _.has(bar, 'shrinkedToolBar'); }),
                // get the number of shrunken tool bars
                countShrinked = _.filter(toolbars, function (bar) { return bar.shrinked; }).length,
                // hideable tool bars
                hideableToolbars = _.filter(toolbars, function (bar) { return (bar.hideable === true); }),
                // get the number of hidden tool bars
                countHidden = _.filter(toolbars, function (bar) { return bar.hidden; }).length,
                // stop acting
                forceStop = false;

            // if the width hasn't changed and no control is out of the visible range
            // and no force is requested - get out here. Do not repaint the panes
            if (paneWidthStore === paneWidth && overflowWidth < 0 && !forceLayoutHandler) {
                reLayout = false;
                return;
            } else {
                paneWidthStore = paneWidth;
            }

            // catch new widths
            function resetDimensions() {
                // go on, if the node isn't visible
                if (centerAreaNode.outerWidth() === 0) { return false; }

                // pane width
                paneWidth = centerWrapperNode.width();
                // tool bars width
                toolbarsWidth = centerAreaNode.outerWidth();
                // overflow size of the center tool bar area
                return toolbarsWidth - paneWidth;
            }

            /**
             * add/remove given class to/from toolbar
             *
             * @param {String} addClass
             *  class which should be add/remove
             *
             * @param {Boolean} visible
             *  whether add or remove the classes
             */
            function toolbarClass(addClass, visible) {
                var node = view.getToolPane().getNode();
                    node.toggleClass(addClass, visible);
            }

            /**
             * add/remove class "vertical-list" to/from pop-up
             *
             * @param {Integer} i
             *  which of the shrinkable tool bars
             *
             * @param {Boolean} state
             *  whether add or remove the classes
             */
            function makeVerticalList(i, state) {
                var popupWrapper = shrinkableToolbars[i].shrinkedMenu.getNode();
                    popupWrapper.toggleClass('vertical-list drop-down', state);
            }

            /**
             * moves the toolbar from pane to drop-down or reversed
             *
             * @param {Integer} i
             *  which of the shrinkable tool bars
             *
             * @param {String} type
             *  whether shrink or expand = move to default pane or to small drop-down
             */
            function moveToolBar(i, type) {

                if (type === 'shrink' && self.hasViewComponent(shrinkableToolbars[i].toolBar)) {
                    // remove component from pane
                    self.removeViewComponent(shrinkableToolbars[i].toolBar);
                    // add component to shrunken (drop-down) menu
                    shrinkableToolbars[i].shrinkedMenu.replaceComponent(shrinkableToolbars[i].toolBar);

                    // go through all groups
                    shrinkableToolbars[i].toolBar.iterateGroups(function (group) {
                        // and, if exists a menu
                        if (_.isFunction(group.getMenu)) {
                            // add the components to the focusable nodes (to prevent closing the pop-up)
                            shrinkableToolbars[i].shrinkedMenu.registerFocusableNodes(group.getMenu().getContentNode());
                        }
                        // show controls which should only be shown in drop-down version
                        group.toggleDropDownMode(true);
                    });

                    // maximize all groups
                    shrinkableToolbars[i].toolBar.iterateGroups(function (group) {
                        group.deactivateSmallVersion();
                    });

                    // add class for vertical design
                    makeVerticalList(i, true);
                    return true;



                } else if (type === 'unshrink' && !self.hasViewComponent(shrinkableToolbars[i].toolBar)) {

                    // remove component from shrunken drop-down menu
                    var component = shrinkableToolbars[i].shrinkedMenu.releaseComponent();
                    // add view component to default pane
                    self.addViewComponent(component, { insertBefore: shrinkableToolbars[i].shrinkedToolBar });

                    // minimize all groups
                    if (shrinkableToolbars[i].shrinked === true) {
                        shrinkableToolbars[i].toolBar.iterateGroups(function (group) {
                            group.activateSmallVersion();
                            // hide controls which should only be shown in drop-down version
                            group.toggleDropDownMode(false);
                        });
                    }

                    // remove vertical-design
                    makeVerticalList(i, false);
                    return true;

                } else {
                    return false;
                }
            }


            view.lockPaneLayout(function() {
                // returns whether the toolbar(s) are too big for the pane
                function tooBig() {
                    overflowWidth = resetDimensions();
                    if (overflowWidth > 0) {
                        // reset force-option to stop invoking paneLayoutHandler here
                        forceLayoutHandler = false;
                        return true;
                    } else {
                        return false;
                    }
                }
                // returns whether the toolbar(s) fit in the pane
                function fitIn() {
                    overflowWidth = resetDimensions();
                    return (overflowWidth < 0);
                }
                // returns the sum of all shrinked toolbars of this pane
                function getCountShrinked() {
                    countShrinked = _.filter(toolBarsMap[toolBarTabs.getActiveTabId()], function(bar) { return bar.shrinked; }).length;
                    return countShrinked;
                }
                // returns the sum of all hidden toolbars of this pane
                function getCountHidden() {
                    countHidden = _.filter(toolBarsMap[toolBarTabs.getActiveTabId()], function(bar) { return bar.hidden; }).length;
                    return countHidden;
                }

                // Controls not fit in the pane
                if (overflowWidth > 0) {
                    // shrink the space between the controls
                    toolbarClass('small-distance', true);

                    // activate small versions of groups
                    if ( tooBig() ) {
                        Utils.iterateArray(toolbars, function(toolbar) {
                            // only activate the small version of the groups, if their were not shrinked
                            if (!toolbar.shrinked) {
                                toolbar.toolBar.iterateGroups(function(group) {
                                    group.activateSmallVersion();
                                });
                                if (fitIn()) { return Utils.BREAK; }
                            }
                        }, {reverse: true});

                        // shrink complete toolbar to dropdown-menu
                        if (tooBig()) {
                            // iterate over all toolbars (reversed)
                            Utils.iterateArray(shrinkableToolbars, function(toolbar, i) {
                                if (toolbar.shrinked) { return; }

                                if (i===(shrinkableToolbars.length-1) || (i < (shrinkableToolbars.length-1) && shrinkableToolbars[(i+1)].shrinked)) {
                                    if (moveToolBar(i, 'shrink')) {
                                        toolbar.shrinked = true;
                                        if (fitIn()) { return Utils.BREAK; }
                                    }
                                }
                            }, {reverse: true});

                            // hide toolbars (last step to fit in the given area)
                            if (tooBig()) {
                                // iterate backwards through all hideable toolbars
                                Utils.iterateArray(hideableToolbars, function(toolbar, i) {
                                    if (toolbar.hidden) { return; }

                                    if (i===(hideableToolbars.length-1) || (i < (hideableToolbars.length-1) && hideableToolbars[(i+1)].hidden)) {
                                        var toolBarNode = (_.has(toolbar, 'shrinkedToolBar'))? toolbar.shrinkedToolBar.getNode():toolbar.toolBar.getNode();
                                            toolBarNode.hide();

                                        toolbar.hidden = true;
                                        if (fitIn()) { return Utils.BREAK; }
                                    }
                                }, {reverse: true});
                            }
                        }
                    }


                // Controls have free space in the pane
                } else {

                    // show hidden toolbars
                    Utils.iterateArray(hideableToolbars, function(toolbar, i) {
                        if (!toolbar.hidden) { return; }

                        if (i===0 || (i > 0 && !hideableToolbars[(i-1)].hidden)) {
                            var toolbarNode = (_.has(toolbar, 'shrinkedToolBar'))?toolbar.shrinkedToolBar.getNode():toolbar.toolBar.getNode();

                            toolbarNode.css('display','');
                            toolbar.hidden = false;

                            // if it was too much, revert to last state
                            if (tooBig()) {
                                toolbarNode.hide();
                                toolbar.hidden = true;
                                forceStop = true;
                                return Utils.BREAK;
                            }
                        }
                    });

                    if (fitIn() && forceStop === false && getCountHidden() === 0) {

                        // unshrink toolbars
                        Utils.iterateArray(shrinkableToolbars, function(toolbar, i) {
                            if (!toolbar.shrinked) { return; }

                            // toolbar have to be shrinked
                            // and the toolbar before have to be unshrinked (or do not exists "===0")
                            if (i===0 || (i > 0 && !shrinkableToolbars[(i-1)].shrinked)) {
                                if (moveToolBar(i, 'unshrink')) {
                                    toolbar.shrinked = false;
                                    overflowWidth = resetDimensions();
                                    forceStop = true;

                                    // if "overflow" minus "shrinked-toolbar" positive
                                    // reshrink the toolbar
                                    if ((overflowWidth - toolbar.shrinkedToolBar.getNode().outerWidth()) > 0) {
                                        if (moveToolBar(i, 'shrink')) {
                                            toolbar.shrinked = true;
                                        }
                                        overflowWidth = resetDimensions();
                                        return Utils.BREAK;
                                    }
                                }
                            }
                        });

                        // maximize groups
                        if (fitIn() && forceStop === false && getCountShrinked() === 0) {
                            // iterate through all groups of all toolbars
                            Utils.iterateArray(toolbars, function(toolbar) {
                                toolbar.toolBar.iterateGroups(function(group) {
                                    group.deactivateSmallVersion();
                                });

                                // if it was too much, revert to last state
                                if (tooBig()) {
                                    toolbar.toolBar.iterateGroups(function(group) {
                                        group.activateSmallVersion();
                                    });
                                    forceStop = true;
                                    return Utils.BREAK;
                                }
                            });
                        }
                    }
                }
            });

            // check visibility of toolbars
            toggleToolBars(toolBarTabs.getActiveTabId(), true);

            // reset status (actually not in layout-progess)
            reLayout = false;
        }, { delay: 500 });

        /**
         *  Checks the height of the window-node and (de-)activates the flat-pane
         */
        function setPaneHeight() {
            var isStandAlone    = app.isStandalone();

            self.getNode().toggleClass('flat-pane', true);  // add special flat-pane css-class

            viewMenu.iterateGroups(function (group) {
                // add sub-menus to focusable nodes, so that the parent-menu don't close automatically
                if (_.isFunction(group.getMenu)) {
                    flatTabView.getMenu().registerFocusableNodes(group.getMenu().getNode());
                }

                // check whether something should happen with this view-menu-group if its moved to the flatPane
                var options = group.getOptions();
                if (_.has(options, 'flatPane') && options.flatPane === 'hide') {
                    group.hide();
                }
            });

            topPane.hide();                                 // hide top pane (tab bar)
            flatTabView.show();                             // show new tab menu (in leading container of tool pane)
            closeBtn.toggle(!isStandAlone);                 // show new close button (in trailing container of tool pane) if standAloneMode is off
            self.show();                                    // make tool pane visible (maybe the user has made the pane invisible)
        }

        function tabCreateHandler(event, tabId, options) {

            var tabButton = new Controls.Button({
                    label: Utils.getStringOption(options, 'label', ''),
                    value: tabId,
                    highlight: function (value) { return value === tabId; }
                });

            flatTabView.addGroup('view/toolbars/tab', tabButton, {
                visibleKey: options.visibleKey,
                insertBefore: searchBtn.getNode(),
                saveSeparator: true
            });
        }

        /**
         *  Sets the correct label to the dynamic filled "drawing-tab"
         */
        function tabLabelHandler(event, tabId, label) {
            flatTabView.getMenu().iterateGroups(function (group) {
                var options = group.getOptions();
                if (options.value === tabId) {
                    group.setLabel(label);
                    return Utils.BREAK;
                }
            });
        }

        /**
         * Updates the visibility of the tool bars according to the active tab.
         */
        function tabActivateHandler(event, newTabId, oldTabId) {
            view.lockPaneLayout(function () {
                toggleToolBars(oldTabId, false);
                if (_.isString(newTabId)) { self.show(); }
                toggleToolBars(newTabId, true);
                // the changing of the tab invokes paneLayoutHandler by force
                forceLayoutHandler = true;
                paneLayoutHandler();
            });
        }

        /**
         * Updates the visibility of the active tool bars that are bound to the
         * enabled state of a controller item.
         */
        function controllerChangeHandler(event, changedItems) {
            _.each(toolBarsMap, function (toolBarInfos) {
                _.each(toolBarInfos, function (toolBarInfo) {
                    var key = toolBarInfo.visibleKey,
                        enabled = (_.isString(key) && (key in changedItems)) ? changedItems[key].enabled : null;
                    if (_.isBoolean(enabled)) {
                        toolBarInfo.visible = enabled;
                    }
                });
            });
            toggleToolBars(toolBarTabs.getActiveTabId(), true);
        }

        /**
          * Prepare shrunken version of the toolbar
          */
        function addShrinkedVersion(options) {

            var // use SplitButton
                splitBtn = Utils.getOption(options, 'splitBtn', false),
                // controller key for split-button
                controllerKey = null,
                // shrunken menu
                smenu = null,
                // shrunken object (includes the toolbar and the menu)
                shrinkObj = {};

            // without split button in toolbar
            if (splitBtn === false) {
                smenu = new Controls.CompoundButton(app, {
                    tooltip:    Utils.getStringOption(options, 'tooltip', ''),
                    label:      Utils.getStringOption(options, 'label', ''),
                    icon:       Utils.getStringOption(options, 'icon', ''),
                    classes:    Utils.getStringOption(options, 'classes', '')
                });

            // with split button in toolbar
            } else {
                smenu = new Controls.CompoundSplitButton(app, Utils.extendOptions({
                    tooltip:        Utils.getStringOption(options, 'tooltip', ''),
                    caretTooltip:   Utils.getStringOption(options, 'caretTooltip', ''),
                    label:          Utils.getStringOption(options, 'label', ''),
                    icon:           Utils.getStringOption(options, 'icon', ''),
                    classes:        Utils.getStringOption(options, 'classes', '')
                }, Utils.getOption(options.splitBtn, 'options')));

                controllerKey = Utils.getStringOption(options.splitBtn, 'key');
            }

            shrinkObj.shrinkedMenu = smenu.getMenu();
            shrinkObj.shrinkedToolBar = new ToolBar(app).addGroup(controllerKey, smenu).hide();

            self.addViewComponent(shrinkObj.shrinkedToolBar);

            return shrinkObj;
        }

        // methods ------------------------------------------------------------

        /**
         * Creates a new tool bar in this tool pane.
         *
         * @param {String} tabId
         *  The identifier of the tab button the tool bar will be associated
         *  with.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options supported by the ToolBar
         *  class constructor. Additionally, the following options are
         *  supported:
         *  @param {String} [options.visibleKey]
         *      The key of the controller item that controls the visibility of
         *      the tool bar while it is active according to the specified tab
         *      identifier. The visibility will be bound to the 'enabled' state
         *      of the respective controller item. If omitted, the tool bar
         *      will always be visible.
         *  @param {Boolean} [options.withShrink=false]
         *      Prepare a (smaller) drop-down menu where all the controls will
         *      be moved to, if there isn't enough free space for the normal
         *      toolbar. Special logic for mobile UI.
         *
         * @returns {ToolBar}
         *  The new tool bar instance.
         */
        this.createToolBar = function (tabId, options) {

            var // the new tool bar instance
                toolBar = new ToolBar(app, options),
                // prepare shrunken toolbar
                shrinkObj = (Utils.getBooleanOption(options, 'prepareShrink', false)) ? addShrinkedVersion(options) : {};

            // set to hidden state initially
            toolBar.hide();

            // insert the tool bar into this pane
            (toolBarsMap[tabId] || (toolBarsMap[tabId] = [])).push(_.extend({
                toolBar: toolBar,                                                   // normal toolbar
                visibleKey: Utils.getStringOption(options, 'visibleKey'),           // toolbar visible at
                visible: true,                                                      // toolbar visibility
                priority: Utils.getIntegerOption(options, 'priority', null),        // toolbar priority (1: high, 10: low)
                hideable: Utils.getBooleanOption(options, 'hideable', null),        // is toolbar hideable
                hidden: false,                                                      // is toolbar hidden
                shrinked: false                                                     // is toolbar shrunken
            }, shrinkObj));
            this.addViewComponent(toolBar);

            return toolBar;
        };

        // initialization -----------------------------------------------------

        // new menu for flat-pane (in leading-container of toolbar)
        flatTabView = new Controls.CompoundButton(app, { icon: 'fa-bars', caret: false, anchorPadding: -3 });

        // add static controls to new flat-pane-menu
        flatTabView
            .addSeparator()
            .addGroup('view/searchpane', searchBtn = new Controls.Button({ icon: 'fa-search', label: gt('Search'), tooltip: gt('Toggle search'), toggle: true }), { visibleKey: 'app/valid' })
            .addSeparator()
            .addGroup('document/undo', new Controls.Button({ icon: 'docs-undo', tooltip: gt('Revert last operation') }), { visibleKey: 'document/editable', inline: true })
            .addGroup('document/redo', new Controls.Button({ icon: 'docs-redo', tooltip: gt('Restore last operation') }), { visibleKey: 'document/editable', inline: true });

        // create the tool bar containing the tab buttons and add it to the leading-container of the toolBar
        this.addViewComponent(new ToolBar(app)
            .addGroup(null, flatTabView.hide()),
        { targetArea: 'leading' });

        // create the toolbar containing the close button and add it to the trailing-container of the toolBar
        this.addViewComponent(new ToolBar(app)
            .addGroup('app/quit', closeBtn = new Controls.Button(Labels.QUIT_BUTTON_OPTIONS).hide()),
        { targetArea: 'trailing' });

        // update layout of tool bars according to available space
        this.on('pane:layout', paneLayoutHandler);

        // listen to tab events, update visibility of the tool bars
        this.listenTo(toolBarTabs, 'tab:create', tabCreateHandler);
        this.listenTo(toolBarTabs, 'tab:label', tabLabelHandler);
        this.listenTo(toolBarTabs, 'tab:activate', tabActivateHandler);
        this.listenTo(app.getController(), 'change:items', controllerChangeHandler);

        // activate a tool bar tab, if the tool pane becomes visible
        this.on('pane:show', function () {
            if (!toolBarTabs.getActiveTabId()) {
                toolBarTabs.activateFirstTab();
            }
        });

        // get the container nodes for the center tool bar area
        centerWrapperNode = this.getAreaWrapperNode('center');
        centerAreaNode = centerWrapperNode.find('>.area-container');

        // decide whether to show the combined tool/tab bar
        if (view.panesCombined()){
            setPaneHeight();
        }
        app.registerWindowResizeHandler(function(){
            // only remove the 'small-distance' marker, if the width has changed
            if (centerWrapperNode.width() !== paneWidthStore) {
                app.getView().getToolPane().getNode().removeClass('small-distance');
            }
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = toolBarTabs = self = toolBarsMap = centerWrapperNode = centerAreaNode = null;
        });

    } // class ToolPane

    // exports ================================================================

    // derive this class from class ToolPane
    return ToolPane.extend({ constructor: EditToolPane });

});
