/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/editdialogs',
    ['io.ox/core/cache',
     'io.ox/core/folder/api',
     'io.ox/core/tk/folderviews',
     'io.ox/files/filepicker',
     'io.ox/office/tk/utils',
     'io.ox/office/tk/dialogs',
     'settings!io.ox/office',
     'gettext!io.ox/office/editframework'
    ], function (CoreCache, FolderAPI, FolderViews, FilePicker, Utils, Dialogs, Settings, gt) {

    'use strict';

    var // key in user settings for the recent 'save as' folder
        RECENT_FOLDER_KEY = 'recentSaveAsFolder';

    // static class EditDialogs ===============================================

    /**
     * Additional classes defining specialized modal dialogs for the OX editor
     * applications.
     *
     * @extends Dialogs
     */
    var EditDialogs = _.clone(Dialogs);

    // class ModalSaveAsDialog ================================================

    /**
     * A custom dialog to save a file in a Drive folder. This dialog provides a
     * folder picker and a simple text input.
     *
     * TODO This is a temporary solution by using FolderViews directly which is
     * now deprecated. A better solution using the official FilePicker with
     * text input feature is planned.
     *
     * @constructor
     *
     * @extends ModalDialog
     *
     * @param {Object} [initOptions]
     *  Optional parameters that control the appearance and behavior of the
     *  dialog. Supports all options also supported by the base class
     *  ModalDialog. Additionally, the following options are supported:
     *  @param {String} [initOptions.value='']
     *      The initial value of the text field.
     *  @param {String} [initOptions.placeholder='']
     *      The place-holder text that will be shown in the empty text field.
     *  @param {Boolean} [initOptions.allowEmpty=false]
     *      If set to true, empty text will be considered valid. Otherwise, the
     *      primary button will be disabled as long as the text input field is
     *      empty.
     *  @param {Boolean} [initOptions.folderPicker=false]
     *      Whether to show the Drive folder picker.
     */
    EditDialogs.ModalSaveAsDialog = Dialogs.ModalDialog.extend({ constructor: function (initOptions) {

        var // self reference
            self = this,

            // container of the Drive folder picker
            treeNode = null,

            // create Drive folder picker if requested
            folderPicker = null,

            // selected folder id in the folder picker
            selectedFolderId = null,

            // label for the save as input node
            filenameLabelNode = $('<label for="save-as-filename">').text(gt('File name')),

            // the text field control
            inputNode = $('<input type="text" id="save-as-filename" class="form-control" tabindex="1">').attr({
                placeholder: Utils.getStringOption(initOptions, 'placeholder', ''),
                value: Utils.getStringOption(initOptions, 'value', '')
            }),

            // whether to allow empty text
            allowEmpty = Utils.getBooleanOption(initOptions, 'allowEmpty', false);

        // base constructor ---------------------------------------------------

        Dialogs.ModalDialog.call(this, Utils.extendOptions({
            okLabel: gt('Save')
        }, initOptions));

        // private functions --------------------------------------------------

        /**
         * Creates an initializes an infostore folder picker.
         *
         * @returns {FolderViews.ApplicationFolderTree}
         */
        function initFolderPicker() {

            var folderCache = new CoreCache.SimpleCache('folder-all'),
                subFolderCache = new CoreCache.SimpleCache('subfolder-all'),
                onChange = function (event, folderId) {
                    selectedFolderId = folderId;
                    // remember user's last folder location, to be saved later while leaving dialog
                    treeNode.data('recent-folder', folderId);
                    // do ok button activation check
                    updateOkButton();
                };

            // init container node
            treeNode = $('<div>').addClass('save-as-folder-list');

            folderPicker = new FolderViews.ApplicationFolderTree(treeNode, {
                type: 'infostore',
                tabindex: 0,
                rootFolderId: '9',
                all: true,
                storage: {
                    folderCache: folderCache,
                    subFolderCache: subFolderCache
                },
                filter: function (folder) {
                    return !FolderAPI.is('trash', folder);
                }
            });

            // paint infostore folder tree and preselect the first folder (My Files folder)
            folderPicker.paint().done(function () {
                // preselect user's recent image folder if exists
                var recentSaveAsFolder = Settings.get(RECENT_FOLDER_KEY);
                if (recentSaveAsFolder) {
                    folderPicker.select(recentSaveAsFolder);
                    folderPicker.selection.select(recentSaveAsFolder);

                    onChange(null, recentSaveAsFolder);
                }
                else {
                    folderPicker.selection.selectFirst();
                }
            });

            // on foldertree select, update selected folder id
            folderPicker.selection.on('select', onChange);
        }

        /**
         * Enables or disables the primary button according to the contents of
         * the text field, and the presence of folder id if folder picker is activated.
         */
        function updateOkButton() {
            var state = allowEmpty || (inputNode.val().length > 0);
            if (folderPicker) { state = state && !!selectedFolderId; }
            self.enableOkButton(state);
        }

        // public methods -----------------------------------------------------

        /**
         * Gets the currently selected folder ID from the folder tree
         *
         * @returns {Number}
         *  the infostore folder ID as a number
         */
        this.getSelectedFolderId = function () {
            return selectedFolderId;
        };

        /**
         * Returns the text that is contained in the text input element.
         *
         * @returns {String}
         *  The text that is contained in the text input element.
         */
        this.getText = function () {
            return inputNode.val();
        };

        /**
         * Changes the text contained in the text input element.
         *
         * @param {String} text
         *  The text to be inserted into the text input element.
         *
         * @returns {ModalInputDialog}
         *  A reference to this instance.
         */
        this.setText = function (text) {
            inputNode.val(text);
            return this;
        };

        // initialization -----------------------------------------------------

        // create the Drive folder picker instance
        if (Utils.getBooleanOption(initOptions, 'folderPicker', true)) {
            initFolderPicker();
        }

        // give this dialog a good class name
        this.getPopup().addClass('save-as-dialog');

        // add the folder tree and text field
        this.append(treeNode, filenameLabelNode, inputNode);

        // enable/disable the primary button
        inputNode.on('input', updateOkButton);
        updateOkButton();

        // focus and select the text in the input field
        this.on({
            show: function () { inputNode.focus().select(); },
            close: function () {
                inputNode.remove();
                inputNode = null;
                if (folderPicker) {
                    // save user's last selected folder
                    var userRecentFolder = treeNode.data('recent-folder');
                    if (userRecentFolder) { Settings.set(RECENT_FOLDER_KEY, userRecentFolder).save(); }
                    folderPicker.destroy();
                    treeNode.remove();
                    folderPicker = treeNode = null;
                }
            }
        });

    }}); // class ModalSaveAsDialog

    // exports ================================================================

    return EditDialogs;

});
