/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/control/fontfamilypicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/control/combofield',
    'gettext!io.ox/office/editframework'
], function (Utils, ComboField, gt) {

    'use strict';

    // class FontFamilyPicker =================================================

    /**
     * A combo-box control used to select a font family.
     *
     * @constructor
     *
     * @extends ComboField
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the base class ComboField.
     */
    function FontFamilyPicker(app, initOptions) {

        var // self reference
            self = this,

            // the collection of fonts of the edited document
            fontCollection = null;

        // base constructor ---------------------------------------------------

        ComboField.call(this, Utils.extendOptions({
            width: 95,
            tooltip: gt('Font name'),
            typeAhead: true
        }, initOptions));

        // methods ------------------------------------------------------------

        /**
         * Fills the drop-down list with all known font names.
         */
        var fillList = this.createDebouncedMethod($.noop, function () {

            var // all font names, sorted by name
                fontNames = _.sortBy(fontCollection.getFontNames(), function (fontName) {
                    return fontName.toLowerCase();
                });

            self.getMenu().clearContents().createSectionNode('values');
            _.each(fontNames, function (fontName) {
                self.createListEntry(fontName, {
                    section: 'values',
                    label: _.noI18n(fontName),
                    labelStyle: { fontFamily: fontCollection.getCssFontFamily(fontName), fontSize: '115%' }
                });
            });
        });

        // initialization -----------------------------------------------------

        app.onInit(function () {
            fontCollection = app.getModel().getFontCollection();
        });

        // add all known fonts after import and changed fonts
        this.listenTo(app.getImportPromise(), 'done', function () {
            fillList();
            self.listenTo(fontCollection, 'triggered', fillList);
        });

    } // class FontFamilyPicker

    // exports ================================================================

    // derive this class from class ComboField
    return ComboField.extend({ constructor: FontFamilyPicker });

});
