/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/operationsgenerator', [
    'io.ox/office/editframework/utils/operations'
], function (Operations) {

    'use strict';

    // class OperationsGenerator ==============================================

    /**
     * An instance of this class contains an array of operations and provides
     * methods to generate new operations.
     *
     * @constructor
     *
     * @param {EditApplication} app
     *  The application instance containing this generator.
     */
    function OperationsGenerator(app) {

        var // the document styles (style sheet containers etc.)
            documentStyles = app.getModel().getDocumentStyles(),

            // the operations buffer
            operations = [];

        // methods ------------------------------------------------------------

        /**
         * Returns the array with all operations that have been generated so
         * far. Does not remove the operations from this generator instance.
         *
         * @returns {Array}
         *  The array with all generated operations.
         */
        this.getOperations = function () {
            return operations;
        };

        /**
         * Reverses the entire operations array.
         *
         * @returns {OperationsGenerator}
         *  A reference to this instance.
         */
        this.reverseOperations = function () {
            operations.reverse();
            return this;
        };

        /**
         * Creates and appends a new operation to the operations array.
         *
         * @param {String} name
         *  The name of the operation.
         *
         * @param {Object} [operationOptions]
         *  Additional options that will be stored in the operation.
         *
         * @returns {Object}
         *  The created operation object.
         */
        this.generateOperation = function (name, operationOptions) {
            var operation = _.extend({ name: name }, operationOptions);
            operations.push(operation);
            return operation;
        };

        /**
         * Adding operations to this operations generator.
         *
         * @param {String[]} ops
         *  The additional operations.
         *
         * @returns {Number}
         *  The number of operations in the operations generator.
         */
        this.addOperations = function (ops) {
            operations = operations.concat(ops);
            return operations.length;
        };

        /**
         * Generates an 'insertStyleSheet' operation, if the specified style
         * sheet exists in the style sheet collection, and is marked dirty.
         *
         * @param {String} styleFamily
         *  The style family that specifies which style sheet collection will
         *  be checked for a dirty style sheet.
         *
         * @param {String} styleId
         *  The identifier of the style sheet.
         *
         * @returns {OperationsGenerator}
         *  A reference to this instance.
         */
        this.generateMissingStyleSheetOperation = function (styleFamily, styleId) {

            var // the style sheet container
                styleSheets = documentStyles.getStyleCollection(styleFamily),
                // additional properties for the operation
                operationOptions = null;

            // do not generate an operation for the default style
            if (styleSheets && (styleId !== styleSheets.getDefaultStyleId()) && styleSheets.isDirty(styleId)) {

                // create the properties for the 'insertStyleSheet' operation
                operationOptions = {
                    attrs: styleSheets.getStyleSheetAttributeMap(styleId),
                    type: styleFamily,
                    styleId: styleId,
                    styleName: styleSheets.getName(styleId),
                    parent: styleSheets.getParentId(styleId),
                    uiPriority: styleSheets.getUIPriority(styleId)
                };
                if (!_.isString(operationOptions.parent)) { delete operationOptions.parent; }
                if (!_.isNumber(operationOptions.uiPriority)) { delete operationOptions.uiPriority; }

                // generate and store the 'insertStyleSheet' operation
                this.generateOperation(Operations.INSERT_STYLESHEET, operationOptions);

                // remove the dirty flag from the style sheet
                styleSheets.setDirty(styleId, false);
            }

            return this;
        };

    } // class OperationsGenerator

    // exports ================================================================

    return _.makeExtendable(OperationsGenerator);

});
