/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Malte Timmermann <malte.timmermann@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/editframework/app/editapplication',
    ['io.ox/files/api',
     'io.ox/office/tk/utils',
     'io.ox/office/tk/forms',
     'io.ox/office/tk/errorcode',
     'io.ox/office/tk/io',
     'io.ox/office/baseframework/app/baseapplication',
     'io.ox/office/baseframework/app/extensionregistry',
     'io.ox/office/editframework/utils/editconfig',
     'io.ox/office/editframework/app/rtconnection',
     'io.ox/office/editframework/view/editlabels',
     'io.ox/office/editframework/view/editdialogs',
     'gettext!io.ox/office/editframework'
    ], function (FilesAPI, Utils, Forms, ErrorCode, IO, BaseApplication, ExtensionRegistry, Config, RTConnection, Labels, Dialogs, gt) {

    'use strict';

    var // maximal time for synchronization
        MAX_SECONDS_FOR_SYNC = 15;

    // private static functions ===============================================

    /**
     * Extracts and validates the actions array from the passed data object of
     * a server response.
     *
     * @param {Object} data
     *  The response data object of the server request.
     *
     * @returns {Array|Undefined}
     *  The actions array, if existing, otherwise undefined.
     */
    function getActionsFromData(data) {

        var // the operations array (may be missing completely)
            actions = Utils.getArrayOption(data, 'actions', []);

        // check that all array elements are objects
        return _.all(actions, _.isObject) ? actions : undefined;
    }

    // class EditApplication ==================================================

    /**
     * The base class for all OX Documents applications allowing to edit a
     * document.
     *
     * Triggers the events supported by the base class BaseApplication, and the
     * following additional events:
     *  - 'docs:state':
     *      When the state of the application has changed. The event handlers
     *      receive the current application state, as also returned by the
     *      method 'EditApplication.getState()'.
     *  - 'docs:state:<STATE_ID>':
     *      When the state of the application has changed to the specified
     *      identifier <STATE_ID>. Will be triggered after the generic
     *      'docs:state' event. All identifiers returned by the method
     *      'EditApplication.getState()' are supported. Example: when the state
     *      'offline' is reached, the application will trigger a 'docs:state'
     *      event, and a 'docs:state:offline' event.
     *  - 'docs:update':
     *      When the application receives an update push message from the
     *      real-time framework. The event handlers receive the complete data
     *      object of the update message.
     *  - 'docs:editrights':
     *      An event related to acquiring the edit rights while the document is
     *      in read-only mode. The event handlers receive the actual state:
     *      - 'acquire': After sending the server request to acquire edit
     *          rights for the document.
     *      - 'accept': The server has accepted the request to acquire edit
     *          right for the document. The document will switch to editing
     *          mode soon.
     *      - 'decline': The server has rejected the request to acquire edit
     *          right for the document. The document will remain in read-only
     *          mode.
     *  - 'docs:editrights:<STATE_ID>':
     *      Will be triggered after a generic 'docs:editrights' event, where
     *      <STATE_ID> is one of the states described above.
     *  - 'docs:users':
     *      After the list of editor clients for this document has changed.
     *      Event handlers receive an array with all active editor clients. See
     *      method EditApplication.getActiveClients() for details.
     *
     * @constructor
     *
     * @extends BaseApplication
     *
     * @param {Function} ModelClass
     *  The constructor function of the document model class. See the
     *  BaseApplication base class constructor for details.
     *
     * @param {Function} ViewClass
     *  The constructor function of the view class. See the BaseApplication
     *  base class constructor for details.
     *
     * @param {Function} ControllerClass
     *  The constructor function of the controller class. See the
     *  BaseApplication base class constructor for details.
     *
     * @param {Object} launchOptions
     *  All options passed to the core launcher (the ox.launch() method):
     *  @param {String} launchOptions.action
     *      Controls how to connect the application to a document file. The
     *      following actions are supported:
     *      - 'load': Tries to load the file described in the launch option
     *          'launchOptions.file'.
     *      - 'new': Tries to create a new document. Optionally, the new file
     *          will be a clone of an existing file described in the launch
     *          option 'launchOptions.templateFile'.
     *      - 'convert': Tries to convert the file described in the launch
     *          option 'launchOptions.templateFile' to a native file format.
     *  @param {Object} [launchOptions.file]
     *      The descriptor of the file to be loaded. Used by the action 'load'
     *      only.
     *  @param {String} [launchOptions.folderId]
     *      The identifier of the folder a new file will be created in. Used by
     *      the actions 'new' and 'convert'.
     *  @param {Object} [launchOptions.templateFile]
     *      The descriptor of an existing file used as template for the new
     *      file. Used by the actions 'new' and 'convert'.
     *  @param {Boolean} [launchOptions.preserveFileName=false]
     *      If set to true, the name of the template file will be preserved
     *      when converting a file to a native file format. Otherwise, a
     *      default file name will be generated. Used by the action 'convert'
     *      only.
     *
     * @param {Object} [appOptions]
     *  Static application options that have been passed to the static method
     *  BaseApplication.createLauncher().
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options supported by the base class
     *  BaseApplication. Additionally, the following options are supported:
     *  @param {Object} [initOptions.newDocumentParams]
     *      Additional parameters that will be inserted into the AJAX request
     *      sent to create a new empty document on the server.
     *  @param {Boolean} [initOptions.applyActionsDetached=false]
     *      If set to true, the application pane will be detached from the DOM
     *      while the import actions will be applied. This may improve the
     *      import performance of large documents.
     *  @param {Function} [initOptions.preProcessHandler]
     *      A function that will be called before the document operations will
     *      be downloaded and applied, with the progress bar already being
     *      visible. Will be called in the context of this application. Must
     *      return a Deferred object that will be resolved or rejected after
     *      the document has been prepared.
     *  @param {Function} [initOptions.postProcessHandler]
     *      A function that will be called after the document operations have
     *      been downloaded and applied successfully, and that post-processes
     *      the document contents while the progress bar is still visible. Will
     *      be called in the context of this application. Must return a
     *      Deferred object that will be resolved or rejected after the
     *      document has been post-processed.
     *  @param {Function} [initOptions.postProcessHandlerStorage]
     *      Load performance: Similar to postProcessHandler. A function that
     *      will be called after the document has been loaded from the local
     *      storage. Will be called in the context of this application. Must
     *      return a Deferred object that will be resolved or rejected after
     *      the document has been post-processed.
     *  @param {Function} [initOptions.fastEmptyLoadHandler]
     *      A function that will be called when importing an empty document in
     *      a special fast way is supported. Will be called in the context of
     *      this application. Must return a Deferred object that will be
     *      resolved or rejected after the document has been loaded quickly.
     *  @param {Function} [initOptions.previewHandler]
     *      A function that can be used to show an early preview of the
     *      imported document, while the import is still running. Receives the
     *      preview data object sent from the server as first parameter. Must
     *      return a Boolean value stating whether displaying the preview
     *      succeeded. On success, the view will leave the busy state. Will be
     *      called in the context of this application instance.
     *  @param {Function} [initOptions.importFailedHandler]
     *      A function that will be called when importing the document has
     *      failed for any reason (either connection problems, or while
     *      applying the operations). Will be called in the context of this
     *      application.
     *  @param {Function} [initOptions.optimizeOperationsHandler]
     *      A function that can be used to optimize the operations before they
     *      are sent to the server. Receives an operations array as first
     *      parameter, and must return an operations array. Will be called in
     *      the context of this application instance.
     *  @param {Function} [initOptions.operationsFilter]
     *      A function that can be used to filter the operations that are
     *      executed during importing the document. Will be called in the
     *      context of the model instance.
     *  @param {Function} [initOptions.prepareFlushHandler]
     *      A function that will be called before sending all pending actions,
     *      and flushing the document to the source file. Will be called in the
     *      context of this application. Receives a string identifier as first
     *      parameter specifying the cause of the flush request:
     *      - 'download': The document will be prepared for downloading or
     *          printing.
     *      - 'email': The document will be prepared to be sent as e-mail.
     *      - 'quit': The document is about to be closed.
     *      May return a Deferred object that will be resolved/rejected as soon
     *      as the application has finished all preparations for flushing the
     *      document.
     *  @param {Function} [initOptions.prepareLoseEditRightsHandler]
     *      A function that will be called if the server sends a request to
     *      leave the edit mode. Will be called in the context of this
     *      application. May return a Deferred object that will be resolved or
     *      rejected as soon as the application can safely switch to read-only
     *      mode.
     *  @param {Number} [initOptions.sendActionsDelay=0]
     *      The duration in milliseconds used to debounce passing new pending
     *      actions to the realtime connection.
     *  @param {Number} [initOptions.realTimeDelay=700]
     *      The duration in milliseconds the realtime framework will collect
     *      messages before sending them to the server.
     *  @param {Boolean} [initOptions.useStorage=false]
     *      Load performance: Whether the local storage shall be used for
     *      storing the document.
     *  @param {Boolean} [initOptions.supportsOnlineSync=false]
     *      Whether the application supports synchronization to enable seamless
     *      editing of a document after a offline/online transition.
     */
    var EditApplication = BaseApplication.extend({ constructor: function (ModelClass, ViewClass, ControllerClass, launchOptions, appOptions, initOptions) {

        var // self reference
            self = this,

            // the document model instance
            model = null,

            // application view: contains panes, tool bars, etc.
            view = null,

            // the unique identifier of this application
            clientId = null,

            // the display name of the user loading the document
            clientDisplayName = null,

            // connection to the realtime framework
            rtConnection = null,

            // buffer for actions not yet sent to the server
            actionsBuffer = [],

            // new action collecting multiple operations in an undo group
            pendingAction = null,

            // whether actions are currently sent to the server
            sendingActions = false,

            // additional parameters for the AJAX request to create a new document
            newDocumentParams = Utils.getObjectOption(initOptions, 'newDocumentParams'),

            // preprocessing of the document before applying import operations
            preProcessHandler = Utils.getFunctionOption(initOptions, 'preProcessHandler'),

            // post-processing of the document after successfully applying import operations
            postProcessHandler = Utils.getFunctionOption(initOptions, 'postProcessHandler'),

            // Load performance: A simplified post process handler, that is called after using document from local storage
            postProcessHandlerStorage = Utils.getFunctionOption(initOptions, 'postProcessHandlerStorage'),

            // fast load empty documents: display page after first server request
            fastEmptyLoadHandler = Utils.getFunctionOption(initOptions, 'fastEmptyLoadHandler'),

            // callback for displaying an early preview while import is still running
            previewHandler = Utils.getFunctionOption(initOptions, 'previewHandler'),

            // post-processing of the document when import has failed
            importFailedHandler = Utils.getFunctionOption(initOptions, 'importFailedHandler', $.noop),

            // callback to flush the document before accessing its source
            prepareFlushHandler = Utils.getFunctionOption(initOptions, 'prepareFlushHandler', $.noop),

            // callback to prepare the document before it goes into read-only mode
            prepareLoseEditRightsHandler = Utils.getFunctionOption(initOptions, 'prepareLoseEditRightsHandler', $.noop),

            // callback function that implements merging of actions to reduce data transfer to the server
            optimizeOperationsHandler = Utils.getFunctionOption(initOptions, 'optimizeOperationsHandler'),

            // Load Performance: Only selected operations need to be executed, if document can be loaded from local storage
            operationsFilter = Utils.getFunctionOption(initOptions, 'operationFilter'),

            // application specific delay in milliseconds, until operations are sent to the server
            sendActionsDelay = Utils.getNumberOption(initOptions, 'sendActionsDelay', 0),

            // current application state (see method EditApplication.getState())
            state = null,

            // whether the application is currently connected to the server
            connected = false,

            // cached name of the user currently editing the document
            oldEditUser = null,

            // if any operations where generated by the own document model and sent to the server
            locallyModified = false,

            // if any operations where generated by a remote application
            remotelyModified = false,

            // current server request to rename the edited document
            renamePromise = null,

            // whether the file has been renamed locally
            locallyRenamed = false,

            // whether the file has been move to other folder or to trash
            movedAway = false,

            // whether application is locked globally (no file permissions)
            locked = false,

            // document file is locked by another user (lock set via OX Drive)
            documentFileIsLocked = false,

            // document file is locked by the user
            documentFileLockedByUser = null,

            // whether application is locked globally (internal unrecoverable error)
            internalError = false,

            // whether the user had edit rights before the offline handler was called
            onlineEditRights = false,

            // whether the connection was at least once offline
            offlineHandlerTriggered = false,

            // the update attributes used by server responses
            updateDataAttributes = ['editUserId', 'editUser', 'hasErrors', 'writeProtected', 'activeUsers', 'locked', 'lockedByUser', 'failSafeSaveDone', 'wantsToEditUser', 'wantsToEditUserId'],

            // old edit user id
            oldEditClientId = null,

            // whether we are in a prepare losing edit rights state
            inPrepareLosingEditRights = false,

            // the user who gets the edit rights
            wantsToEditUser = null,

            // whether the documents of the application can be saved in the local storage
            localStorageApp = Utils.getBooleanOption(initOptions, 'localStorageApp', false),

            // the required version of the local storage
            requiredStorageVersion = Utils.getIntegerOption(initOptions, 'requiredStorageVersion', 0),

            // the supported version of the local storage
            supportedStorageVersion = Utils.getIntegerOption(initOptions, 'supportedStorageVersion', 0),

            // reading configuration whether using local browser storage is configured
            // Additionally checking, if the 'Stay signed in' check box is checked or not
            saveFileInLocalStorage = Config.USE_LOCAL_STORAGE && (ox.secretCookie === true),

            // whether the local storage can be used to save the file
            isLocalStorageSupported = Utils.isLocalStorageSupported(),

            // whether the document was loaded from the local storage
            loadedFromStorage = false,

            // maps client identifiers to unique and constant integral indexes
            clientIndexes = {},

            // next free unique client index (0 is used for this client)
            nextClientIndex = 1,

            // all clients working on this document (including the local client, including inactive clients)
            editClients = [],

            // all active clients working on this document (less than 60 seconds inactivity)
            activeClients = [],

            // pending remote updates queue
            pendingRemoteUpdates = [],

            // object for logging several time stamps
            performanceLogger = {},

            // start time of preview phase (import still running but with enabled view)
            previewStartTime = null,

            // time stamp when import finishes
            importFinishedTime = 0,

            // asynchronous loading finished
            asyncLoadingFinished = false,

            // server sent us a fast empty doc response with document data
            fastEmpty = false,

            // Synchronization support
            // supports online sync or not
            supportsOnlineSync = Utils.getBooleanOption(initOptions, 'supportsOnlineSync', false),

            // must sync after online again, with mode ('editSync', 'viewSync')
            mustSyncNowMode = null;

        // base constructor ---------------------------------------------------

        BaseApplication.call(this, ModelClass, ViewClass, ControllerClass, importHandler, launchOptions, appOptions, {
            flushHandler: flushDocumentHandler
        });

        // private methods ----------------------------------------------------

        /**
         * Extracts the update data from the passed data object of a
         * server response.
         *
         * @param {Object} data
         *  The response data object of a server request.
         *
         * @returns {Object}
         *  An object which contains only the update attributes, if
         *  existing, otherwise undefined.
         */
        function getUpdateAttributesFromData(data) {
            var // the return object
                updateData = {},
                // the found attributes
                attrs = _.intersection(_.keys(data), updateDataAttributes);

            // copy only the known update attributes
            _.each(attrs, function (key) {
                updateData[key] = data[key];
            });

            return updateData;
        }

        /**
         * Recalculates the application state. Triggers a 'docs:state' event,
         * if the application state has been changed.
         */
        function updateState() {

            // calculate the new application state
            var newState = internalError ? 'error' :
                    _.isNumber(previewStartTime) ? 'preview' :
                    !connected ? 'offline' :
                    (locked || !model.getEditMode()) ? 'readonly' :
                    self.hasUnsavedChanges() ? 'sending' :
                    locallyModified ? 'ready' :
                    'initial';

            // trigger event if state has been changed
            if (state !== newState) {
                state = newState;
                self.getWindow().nodes.outer.attr('data-app-state', state);
                self.trigger('docs:state', state);
                self.trigger('docs:state:' + state);
            }
        }

        /**
         * Sets this application into the internal error mode.
         */
        function setInternalError() {
            internalError = true;
            connected = false;
            locked = true;
            model.setEditMode(false);
            updateState();
        }

        /**
         * Shows a read-only alert banner according to the current state of the
         * application. If the document is locked due to missing editing
         * permissions, shows an appropriate alert, otherwise shows the name of
         * the current editing user.
         *
         * @param {String} [message]
         *  A custom message to be shown in the alert banner. If omitted, a
         *  default message will be generated according to the document state.
         */
        function showReadOnlyAlert(message) {

            var // alert type
                type = 'info',
                // the alert title
                headline = gt('Read-Only Mode');

            // build message text
            if (!message) {
                if (locked) {
                    if (offlineHandlerTriggered) {
                        // Messages due to offline detection
                        if (connected) {
                            if (supportsOnlineSync) {
                                if (onlineEditRights) {
                                    message = gt('Connection to server was lost. Please wait, trying to synchronize to enable edit mode.');
                                } else {
                                    message = gt('Connection to server was lost. Enable client to acquire edit rights.');
                                }
                            } else {
                                message = gt('Connection to server was lost. Please reopen the document to acquire edit rights.');
                            }
                        } else {
                            type = 'warning';
                            message = gt('Connection to server was lost. Switched to read-only mode.');
                        }
                    } else if (documentFileIsLocked) {
                        // Messages due to file locking
                        if (documentFileLockedByUser && documentFileLockedByUser.length > 0) {
                            //#. %1$s is the name of the user who has locked the document
                            //#, c-format
                            message = gt('The document has been locked by %1$s.', _.noI18n(documentFileLockedByUser));
                        } else {
                            message = gt('The document has been locked by another user.');
                        }
                    }  else {
                        // General message due to missing permissions
                        message = gt('You do not have permissions to change this document.');
                    }
                } else {
                    if (inPrepareLosingEditRights) {
                        // message if we are in the process to lose the edit rights, the editor is in read-only
                        // mode to prevent endless editing
                        if (wantsToEditUser) {
                            message =
                                //#. %1$s is the name of the user who will get the edit rights
                                //#, c-format
                                gt('Transferring edit rights to %1$s. Your latest changes will be saved now.', _.noI18n(wantsToEditUser));
                        } else {
                            message = gt('Transferring edit rights to another user. Your latest changes will be saved now.');
                        }
                    } else if (oldEditUser) {
                        // message if we lost the edit rights
                        message =
                            //#. %1$s is the name of the user who is currently editing the document
                            //#, c-format
                            gt('%1$s is currently editing this document.', _.noI18n(oldEditUser));
                    } else if (self.isImportSucceeded() || asyncLoadingFinished) {
                        message = gt('Another user is currently editing this document.');
                    } else {
                        message = gt('The document could not be loaded correctly. Editing is not allowed.');
                    }
                }
            }

            // show the alert banner
            view.yell({
                type: type,
                headline: headline,
                message: message,
                action: { itemKey: 'document/acquireedit', icon: Labels.EDIT_ICON, label: Labels.EDIT_LABEL }
            });
        }

        /**
         * Shows a read-only alert banner according to the current state of the
         * application. If the document is locked due to missing editing
         * permissions, shows an appropriate alert, otherwise shows the name of
         * the current editing user.
         */
        function showReloadOnErrorAlert(yellOptions) {
            view.yell(_.extend({
                type: 'error',
                action: { itemKey: 'document/reload', icon: Labels.RELOAD_ICON, label: Labels.RELOAD_LABEL }
            }, yellOptions));
        }

        /**
         * In debug mode, the global configuration value for 'useLocalStorage' can be overwritten
         * with the local configuration value of 'debugUseLocalStorage'.
         */
        var checkDebugUseLocalStorage = Config.DEBUG ? function () {
            saveFileInLocalStorage = self.getUserSettingsValue('debugUseLocalStorage', saveFileInLocalStorage);
        } : $.noop;

        /**
         * Adding two specific key-value pairs to the performance logger object.
         * For a given key name the absolute time and the duration compared
         * to the launch time is saved.
         *
         * @param {String} key
         *  The key for the performance logger object.
         *
         * @param {Number} [time]
         *  The logged time for the performance logger object.
         */
        var addTimePerformance = Config.LOG_PERFORMANCE_DATA ? function (key, time) {
            var localTime = time || _.now(),
                absoluteKey = key + 'Absolute',
                durationKey = key + 'Duration';
            addToPerformanceLogger(absoluteKey, localTime);
            addToPerformanceLogger(durationKey, localTime - self.getLaunchStartTime());
        } : $.noop;

        /**
         * Handles a lost network connection. Switches the document model to
         * read-only mode.
         */
        function connectionOfflineHandler() {
            offlineHandlerTriggered = true;
            if (connected && !internalError && !self.isInQuit()) {
                onlineEditRights = model.getEditMode();
                connected = false;

                // Sets the client lock mode to ensure that the user is not able
                // to do anything via the UI while we try to synchronize.
                if (!mustSyncNowMode) {
                    locked = true;
                    model.setEditMode(false);
                    updateState();
                    showReadOnlyAlert();
                } else {
                    // a off-line notification while synchronizing means
                    // that we give up and set internal error to true
                    setInternalError();
                    updateState();
                    showReloadOnErrorAlert({
                        headline: gt('Synchronization Error'),
                        message: gt('Cannot synchronize while in offline-mode - giving up. Please reopen the document for further usage.')
                    });
                }
                // more logging for RT to better see who has the edit rights
                RTConnection.log('Connection offline handler called by Realtime!');
            }
        }

        /**
         * Handles if we get a network connection again. Sets the internal
         * connected flag to true if our internal error state is not set.
         */
        function connectionOnlineHandler() {
            // RT-Library sometimes calls online although the state has not changed,
            // therefore we also check the offlineHandlerTriggered flag. See
            // #27831 for more information.
            if (offlineHandlerTriggered && !connected && !internalError && !self.isInQuit()) {
                // more logging for RT to better see who has the edit rights
                RTConnection.log('Connection online handler called by Realtime!');

                connected = true;
                // check for synchronization
                if (supportsOnlineSync) {
                    RTConnection.log('Client supports synchronization - starting to sync with server');
                    // if application supports synchronization we have
                    // to check what to do next - the editor needs to
                    // synchronize while the other clients just reset
                    // locked.
                    if (onlineEditRights) {
                        // edit synchronization must be done
                        RTConnection.log('Client is editor - synchronizing server with current client state');
                        mustSyncNowMode = 'editSync';
                    } else {
                        // view synchronization must be done
                        RTConnection.log('Client is viewer - waiting for server messages to unlock view');
                        mustSyncNowMode = 'viewSync';
                    }

                    // setup timeout to prevent waiting forever
                    self.executeDelayed(function () {
                        checkSynchronizationState(null, null, null, {cause: 'timeout'});
                    }, MAX_SECONDS_FOR_SYNC * 1000);
                }

                updateState();
                showReadOnlyAlert();
            }
        }

        /**
         * Handles if a connection has been timed out and was reset. The
         * document is set to read-only and the user has to reopen it to
         * edit it again.
         */
        function connectionResetHandler() {
            // more logging for RT to better see who has the edit rights
            RTConnection.log('Reset for connection called by Realtime!');

            if (!self.isLocked() && !self.isInQuit()) {
                setInternalError();
                showReloadOnErrorAlert({
                    headline: gt('Connection Error'),
                    message: gt('Due to a server connection error the editor switched to read-only mode. Please close and reopen the document.')
                });
            }
            // check reset message while we are synchronizing
            if (mustSyncNowMode) {
                checkSynchronizationState(null, null, null, {cause: 'reset'});
            }
        }

        /**
         * Handles communication with server failed state.
         */
        function connectionTimeoutHandler() {
            // more logging for RT to better see who has the edit rights
            RTConnection.log('Connection time out called by Realtime!');

            if (!self.isLocked() && !self.isInQuit()) {
                setInternalError();
                showReloadOnErrorAlert({
                    headline: gt('Connection Error'),
                    message: gt('Connection to the server has been lost. Please close and reopen the document.')
                });
            }
        }

        /**
         * Handles problems with the connection instance on the server side.
         * The handler is called if the connection instance was closed  due
         * to the fact that the server node has been shutdown. The document
         * must be reopened to be accessible again. Can also happen if the
         * server-side runs into a timeout and removes our instance from the
         * document connection.
         */
        function connectionNotMemberHandler() {
            // more logging for RT
            RTConnection.log('NotMember for connection called by Realtime!');

            if (!self.isLocked() && !self.isInQuit()) {
                setInternalError();
                showReloadOnErrorAlert({
                    headline: gt('Connection Error'),
                    message: gt('Connection to the server hosting the document has been lost. Please close and reopen the document.')
                });
            }

            // check not member message while we are synchronizing
            if (mustSyncNowMode) {
                checkSynchronizationState(null, null, null, {cause: 'notMember'});
            }
        }

        /**
         * Handles the situation that we have sent actions to the server but
         * we are not the editor registered at the server. Therefore the server
         * doesn't accept our actions and sends us a hangup notification for
         * reloading. Our document contains changes that are not part of the
         * shared document on the server!
         *
         * @param {jQuery.Event} event
         *  The jQuery event object.
         *
         * @param {Object} data
         *  The data sent with the notification.
         */
        function connectionHangupHandler(event, data) {

            var // the error message
                message = null,
                // the extracted error code from the server response
                errorCode = null;

            if (!self.isInQuit()) {

                message = gt('Your document changed on the server. Please close and reopen the document.');
                errorCode = new ErrorCode(data);

                setInternalError();
                switch (errorCode.getCodeAsConstant()) {
                case 'HANGUP_INVALID_OPERATIONS_SENT_ERROR':
                case 'HANGUP_NO_EDIT_RIGHTS_ERROR':
                    message = gt('The server detected a synchronization problem with your client. Please close and reopen the doument.');
                    break;
                case 'HANGUP_CALCENGINE_NOT_RESPONDING_ERROR':
                    message = gt('The server detected a synchronization problem with all clients of this document. Please close and reopen the document.');
                    break;
                case 'HANGUP_INVALID_OSN_DETECTED_ERROR':
                    break;
                default:
                    break;
                }
                showReloadOnErrorAlert({ headline: gt('Synchronization Error'), message: message });
            }
            // more logging for RT to better see who has the edit rights
            RTConnection.log('Server sent a hangup message due to inconsistency between client/server, client osn = ' + model.getOperationStateNumber());
        }

        /**
         * Handles the situation that the server notifies us that another
         * client wants to receive the edit rights. Therefore we are forced
         * to switch to read-only mode and flush our buffer. If that happened
         * we have to notify the server that the edit rights switch can be
         * accomplished.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object.
         *
         * @param {Object} data
         *  The data sent with the notification.
         */
        function handlePrepareLosingEditRights(event, data) {

            function checkForPendingOperations() {

                if (!self.hasUnsavedChanges()) {

                    // just to be sure that nothing is left
                    sendActions().then(function () {
                        // After sending the actions we can acknowledge that
                        // we are ready to lose our edit rights.
                        rtConnection.canLoseEditRights();
                        oldEditUser = wantsToEditUser || oldEditUser;
                        wantsToEditUser = '';
                        inPrepareLosingEditRights = false;
                        self.getController().update();
                        // more logging for RT to better see who has the edit rights
                        RTConnection.log('sent canLoseEditRights');
                    });
                } else {
                    // wait again some time and try again
                    self.executeDelayed(checkForPendingOperations, 2000);
                }
            }

            if (self.isInQuit()) { return; }

            inPrepareLosingEditRights = true;
            self.getController().update();
            wantsToEditUser = Utils.getStringOption(data, 'wantsToEditUser', '');

            // invoke callback for document specific preparations
            $.when(prepareLoseEditRightsHandler.call(self))
            .always(function () {
                model.setEditMode(false);
                updateState();

                RTConnection.log('handle PrepareLosingEditRights');
                showReadOnlyAlert((wantsToEditUser.length > 0) ?
                    //#. %1$s is the user name who has received the edit rights
                    //#, c-format
                    gt('Transferring edit rights to %1$s. Your latest changes will be saved now.', _.noI18n(wantsToEditUser)) :
                    gt('Transferring edit rights to another user. Your latest changes will be saved now.')
                );

                self.executeDelayed(checkForPendingOperations, 1000);
            });
        }

        /**
         * Triggers an event after the edit rights for the document have been
         * tried to acquire.
         */
        function triggerEditRightsEvent(editState) {
            self.trigger('docs:editrights', editState).trigger('docs:editrights:' + editState);
        }

        /**
         * Handles the situation that this client will receive the
         * edit rights as soon as the transfer from the editor
         * has been completed.
         */
        function handleAcceptedAcquireEditRights() {
            if (!self.isInQuit()) {
                view.yell({ type: 'info', message: gt('Edit rights are being transferred to you. Please wait a moment.') });
            }
            // more logging for RT to better see who has the edit rights
            RTConnection.log('handle AccepteAcquireEditRights');
            triggerEditRightsEvent('accept');
        }

        /**
         * Handles the situation that this client tries to acquire the
         * edit rights but someone else already did and the process is
         * still in progress.
         */
        function handleDeclinedAcquireEditRights() {
            if (!self.isInQuit()) {
                view.yell({ type: 'warning', message: gt('Someone else already wants to receive the edit rights for this document. Please try again later.') });
            }
            // more logging for RT to better see who has the edit rights
            RTConnection.log('handle DeclinedAcquireEditRights');
            triggerEditRightsEvent('decline');
        }

        /**
         * Handles the situation that this client detects that no update
         * has been received since a certain amount of actions were sent
         * to the server.
         * This can happen if the server side code is not able to reply
         * on our requests. Currently we just show a warning to the
         * user.
         */
        function handleNoUpdateFromServer() {
            if (!self.isInQuit()) {
                view.yell({ type: 'warning', message: gt('The server does not answer our action requests. May be the connection is unstable or the server side has problems.') });
            }
            // more logging for RT to better see that the server doesn't send updates
            RTConnection.log('handle NoUpdateFromServer');
        }

        /**
         * Handles the situation if the server detects problems while storing the latest
         * changes to the document. Normally this means that a _ox file was created. It's
         * also possible that just a warning has to be shown (in case the old document
         * content cannot be preserved in a backup file). This function provides feedback
         * to the user to notify her that there is a problem.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object.
         *
         * @param {Object} data
         *  The data sent with the notification.
         */
        function handleFlushErrorInfo(event, data) {

            var // extract error information
                errorCode = new ErrorCode(data),
                // message
                message = null,
                // type of message,
                msgType = 'error',
                // the headline text
                headline = gt('Server Error');

            if (!self.isInQuit()) {
                if (errorCode.isError()) {
                    setInternalError();
                }

                switch (errorCode.getCodeAsConstant()) {
                case 'GENERAL_QUOTA_REACHED_ERROR':
                    message = gt('The document could not be saved. Your quota limit has been reached. Please free up storage space.');
                    break;
                case 'GENERAL_FILE_NOT_FOUND_ERROR':
                    message = gt('The document could not be written, because the document could not be found. To save your work, please copy the complete content into the clipboard and paste it into a new document.');
                    break;
                case 'GENERAL_PERMISSION_READ_MISSING_ERROR':
                    message = gt('The document could not be read, because you lost read permissions. To save your work, please copy the complete content into the clipboard and paste it into a new document.');
                    break;
                case 'GENERAL_PERMISSION_WRITE_MISSING_ERROR':
                    message = gt('The document could not be written, because you lost write permissions. To save your work, please copy the complete content into the clipboard and paste it into a new document.');
                    break;
                case 'SAVEDOCUMENT_FAILED_FILTER_OPERATION_ERROR':
                    message = gt('The document is in an inconsistent state. To save your work, please copy the complete content into the clipboard and paste it into a new document. Furthermore the last consistent document was restored.');
                    break;
                case 'GENERAL_MEMORY_TOO_LOW_ERROR':
                    message = gt('The document could not be saved. The server has not enough memory. To save your last changes, please copy the complete content into the clipboard and paste it into a new document. If you close the document, the last consistent version will be restored.');
                    break;
                case 'SAVEDOCUMENT_BACKUPFILE_CREATE_PERMISSION_MISSING_ERROR':
                    headline = gt('Server Warning');
                    msgType = 'warning';
                    message = gt('The document was saved successfully, but a backup copy could not be created. You do not have the permission to create a file in the folder.');
                    break;
                case 'SAVEDOCUMENT_BACKUPFILE_READ_OR_WRITE_PERMISSION_MISSING_ERROR':
                    headline = gt('Server Warning');
                    msgType = 'warning';
                    message = gt('The document was saved successfully, but a backup copy could not be created. You do not have the correct permissions in the folder.');
                    break;
                case 'SAVEDOCUMENT_BACKUPFILE_QUOTA_REACHED_ERROR':
                    headline = gt('Server Warning');
                    msgType = 'warning';
                    message = gt('The document was saved successfully, but a backup copy could not be created. Your quota limit has been reached. Please free up storage space. ');
                    break;
                case 'SAVEDOCUMENT_BACKUPFILE_CREATE_FAILED_ERROR':
                    headline = gt('Server Warning');
                    message = gt('The document was saved successfully, but a backup copy could not be created.');
                    msgType = 'warning';
                    break;
                default:
                    message = gt('The document is in an inconsistent state. To save your work, please copy the complete content into the clipboard and paste it into a new document.');
                }

                view.yell({ type: msgType, headline: headline, message: message });
            }

            // more logging for RT to better see that the server doesn't send updates
            RTConnection.log('handle flushErrorInfo from server');
        }

        /**
         * Updates the edit state of the application according to the passed data of
         * a server notification message.
         *
         * @param {Object} data
         *  The data of a server notification message.
         */
        function applyEditState(data) {

            var // unique identifier of current user with edit rights
                editClientId = Utils.getStringOption(data, 'editUserId', ''),
                // display name of current user with edit rights
                editUser = Utils.getStringOption(data, 'editUser', ''),
                // current state of document edit mode
                oldEditMode = model.getEditMode(),
                // new state of document edit mode
                newEditMode = oldEditMode,
                // the file descriptor of this application
                file = self.getFileDescriptor(),
                // client OSN
                clientOSN = model.getOperationStateNumber(),
                // server OSN
                serverOSN = Utils.getIntegerOption(data, 'serverOSN', -1);

            // no state information or internal error may have been set while request was running
            if (internalError || (_.intersection(_.keys(data), updateDataAttributes).length === 0)) {
                return;
            }

            if (editClientId !== oldEditClientId) {
                RTConnection.log('Edit rights changed, new editing user id: ' + editClientId);
                // Bug 32043: Set edit mode only if the edit user has changed, otherwise it's
                // possible that we switch back to read/write mode although we switched
                // programmatically to read-only mode due to 'preparelosingeditrights'
                newEditMode = (clientId === editClientId);
            }

            // check if we have to update the files view (fail safe save done on server)
            if (Utils.getBooleanOption(data, 'failSafeSaveDone', false)) {
                FilesAPI.propagate('change', file);
            }

            // internal server error: stop editing completely, show permanent error banner
            if (Utils.getBooleanOption(data, 'hasErrors', false)) {
                setInternalError();
                view.yell({ type: 'error', headline: gt('Server Error'), message: gt('Synchronization to the server has been lost.') });
                RTConnection.error('synchronization to the server has been lost.');
                return;
            }

            if (!locked) {
                // check if file lacks write permissions on server
                if (Utils.getBooleanOption(data, 'writeProtected', false)) {
                    locked = true;
                    model.setEditMode(false);
                    showReadOnlyAlert();
                } else if (Utils.getBooleanOption(data, 'locked', false)) {
                    documentFileIsLocked = true;
                    documentFileLockedByUser = Utils.getStringOption(data, 'lockedByUser', '');
                    locked = true;
                    model.setEditMode(false);
                    showReadOnlyAlert();
                }
            }

            // update edit mode (not if document is locked due to missing write permissions)
            if (!locked) {

                // set edit mode at document model
                RTConnection.log('Edit mode: ' + newEditMode);
                model.setEditMode(newEditMode);

                // Remember old edit user id to check for edit rights instead of using editMode
                // which can be misleading if we in the process of switching the edit rights.
                // There is a time where the client still has edit rights but is in read-only
                // mode to prevent the user from changing the document further. All changes which
                // have been done should be sent to the server in time!
                if (connected && editClientId && (editClientId !== oldEditClientId)) {
                    oldEditClientId = editClientId;
                    oldEditUser = editUser;
                }

                // Show success banner, if edit rights gained after read-only
                // mode, or after connection re-established, but not directly
                // after loading the document (oldEditMode === null).
                if (newEditMode && (oldEditMode === false)) {
                    view.yell({ type: 'success', message: gt('You have edit rights.') });
                }

                // Make consistency check if we just have lost the edit rights
                // and our OSN is higher than the server OSN. The old editor
                // must never be in front of operation, as that would mean that
                // operations have not been sent to the server side. E.g. a
                // timeout scenario occurred.
                if (connected && !newEditMode && oldEditMode && (serverOSN !== -1) && (clientOSN > serverOSN)) {
                    setInternalError();
                    view.yell({
                        type: 'error',
                        headline: gt('Client Error'),
                        message: gt('Synchronization between client and server is broken. Switching editor into read-only mode.')
                    });
                    RTConnection.error('synchronization between client and server is broken. Switching editor into read-only mode. New edit mode: ' + newEditMode + 'Old edit mode: ' + oldEditMode + ', Server OSN: ' + serverOSN + ', Client OSN: ' + clientOSN);
                }

                // Show warning if edit rights have been lost, or are still missing
                // after connection has been re-established, or if the edit user has changed.
                if (!newEditMode && (!connected || oldEditMode || ((editUser !== oldEditUser) && editUser.length))) {
                    oldEditUser = editUser;
                    showReadOnlyAlert();
                }
            }

            // check for a possible synchronization
            checkSynchronizationState(editClientId, clientOSN, serverOSN, null);

            // application is now in connected state
            connected = true;
            updateState();
        }

        /**
         * Manages the current synchronization state after a off-/online
         * transition.
         *
         * @param {String} editClientId
         *  The current editClientId received from the latest update message.
         *
         * @param {Number} clientOSN
         *  The client osn, can be null.
         *
         * @param {Number} serverOSN
         *  The server osn, can be null.
         *
         * @param {Object} failedData
         *  An optional object describing what went wrong, if other handlers
         *  detected a bad synchronization state. Can be null.
         *  @param {String} [failedData.cause]
         *      A string describing the root cause why the synchronization
         *      has failed. Must be set, if failedData is provided.
         */
        function checkSynchronizationState(editClientId, clientOSN, serverOSN, failedData) {
            // Start synchronization after we received an update message which
            // is an indication that we have a clean connection. We also must be
            // sure that all pending operations stored in the RT queue were sent
            // to the backend.
            if (mustSyncNowMode) {

                RTConnection.log('Checking synchronization state with server');

                // Handle a detected failed synchronization state here - other
                // bad dynamic states are detected within this function.
                if (failedData) {
                    switch (failedData.cause) {
                    case 'timeout':
                        // in case of time out show error message and give up!
                        RTConnection.log('Synchronization not possible - maybe connection instance is gone');
                        setInternalError();
                        updateState();
                        mustSyncNowMode = null;
                        showReloadOnErrorAlert({
                            headline: gt('Synchronization Error'),
                            message: gt('Synchronization not possible, because the server was not able to process the synchronization request. Please reload the document.')
                        });
                        break;

                    case 'notMember':
                    case 'reset':
                        // real-time is not able to recover easily - for now give up!
                        RTConnection.log('Synchronization not possible - connection instance is gone or rt-state broken');
                        setInternalError();
                        updateState();
                        mustSyncNowMode = null;
                        showReloadOnErrorAlert({
                            headline: gt('Synchronization Error'),
                            message: gt('Synchronization not possible, because the server is not able to synchronize this document. Please reload the document for further usage.')
                        });
                        break;
                    }
                }

                switch (mustSyncNowMode) {
                case 'editSync':
                    // Ignore updates which are not up to date with the server osn
                    // until we reached the timeout.
                    if ((editClientId === oldEditClientId) && (clientOSN !== serverOSN)) {
                        RTConnection.log('Synchronization in progess server osn != client osn');
                        return;
                    }

                    if (editClientId === oldEditClientId) {
                        if (clientOSN === serverOSN) {
                            RTConnection.log('Synchronization successfully completed');
                            locked = false;
                            model.setEditMode(true);
                            view.yell({
                                type: 'info',
                                headline: gt('Synchronization'),
                                message: gt('Synchronization between client and server was successful. You can edit the document again.')
                            });
                        } else {
                            // We are not editor anymore - therefore no easy sync is
                            // possible. Just give up!
                            RTConnection.log('Synchronization not possible - editor changed');
                            setInternalError();
                            showReloadOnErrorAlert({
                                headline: gt('Synchronization Error'),
                                message: gt('Synchronization is not possible, because the document has been changed. Please reload the document.')
                            });
                        }
                    } else {
                        // We are not editor anymore - therefore no easy sync is
                        // possible. Just give up!
                        RTConnection.log('Synchronization not possible - editor changed');
                        setInternalError();
                        showReloadOnErrorAlert({
                            headline: gt('Synchronization Error'),
                            message: gt('You are not the editor of the document anymore. Synchronization is not possible, please reload the document.')
                        });
                    }
                    break;
                case 'viewSync':
                    RTConnection.log('Synchronization not possible - editor changed');
                    locked = false;
                    break;
                }

                mustSyncNowMode = null;
            }
        }

        /**
         * Updates the file state of the application according to the passed
         * data of a server notification message.
         *
         * @param {Object} data
         *  The data of a server notification message.
         */
        function applyFileState(data) {

            var // current file descriptor
                file = self.getFileDescriptor(),
                // the new file name
                newFileName = Utils.getStringOption(data, 'fileName', ''),
                // the new file version
                newFileVersion = Utils.getStringOption(data, 'fileVersion', ''),
                // the changed components of the file descriptor
                changedFile = {};

            // file name changed after a 'rename' operation
            if ((newFileName.length > 0) && (newFileName !== Utils.getStringOption(file, 'filename'))) {
                changedFile.filename = newFileName;
            }

            // file version changed after remote client closes the modified file
            if ((newFileVersion.length > 0) && (newFileVersion !== Utils.getStringOption(file, 'version'))) {
                changedFile.version = newFileVersion;
            }

            if (!internalError && !_.isEmpty(changedFile)) {
                self.updateFileDescriptor(changedFile);
                self.getController().update();
                updateState();
            }
        }

        /**
         * Updates the user data of the application according to the passed
         * data of a server notification message.
         *
         * @param {Object} data
         *  The data of a server notification message.
         */
        function applyUserData(data) {

            var // reference to the old array of active clients, for comparison
                oldActiveClients = activeClients;

            // trigger a 'docs:users' event only if server data contains a user list
            if (!_.isArray(data.activeUsers)) { return; }

            // update the list of edit clients (all active and inactive)
            editClients = _.filter(data.activeUsers, function (client) {
                var valid = _.isObject(client) && _.isString(client.userId) && _.isNumber(client.id) && _.isString(client.userDisplayName);
                if (!valid) { Utils.warn('EditApplication.applyUserData(): missing identifier or display name'); }
                return valid;
            });

            // translate raw server data to internal representation of the client
            editClients = _.map(editClients, function (client) {

                var // the new internal client descriptor
                    clientDesc = {};

                // add identifiers and user name to the descriptor
                clientDesc.clientId = client.userId; // server sends real-time client identifier as 'userId'
                clientDesc.userId = client.id; // server sends user identifier as 'id'
                clientDesc.userName = client.userDisplayName;

                // add other client properties
                clientDesc.active = (client.active === true) || (_.isNumber(client.durationOfInactivity) && (client.durationOfInactivity < 60));
                clientDesc.remote = clientId !== clientDesc.clientId;
                clientDesc.editor = oldEditClientId === clientDesc.clientId;

                // add user data if existing
                if (_.isObject(client.userData)) {
                    clientDesc.userData = client.userData;
                }

                // store unknown clients into the index map
                if (!(clientDesc.clientId in clientIndexes)) {
                    clientIndexes[clientDesc.clientId] = nextClientIndex;
                    nextClientIndex += 1;
                }

                // add client index and color scheme index into the descriptor
                clientDesc.clientIndex = clientIndexes[clientDesc.clientId];
                clientDesc.colorIndex = (clientDesc.clientIndex % Utils.SCHEME_COLOR_COUNT) + 1;

                return clientDesc;
            });

            // create the list of active edit clients (less than 60 seconds inactivity)
            activeClients = _.where(editClients, { active: true });

            // sort by unique client index for comparison
            activeClients = _.sortBy(activeClients, 'clientIndex');

            // notify listeners with all active clients
            if (!_.isEqual(oldActiveClients, activeClients)) {
                Utils.info('EditApplication.applyUserData(): activeClients=', activeClients);
                self.trigger('docs:users', activeClients);
            }
        }

        /**
         * Applies actions and their operations passed with an 'update' message
         * received from a remote editor of the current document in a
         * background task.
         *
         * @param {Object} data
         *  The data of a server notification message.
         *
         * @return {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved after all
         *  actions have been applied.
         */
        function applyRemoteActions(data) {

            var // all actions contained in the passed update message data
                actions = getActionsFromData(data);

            if (internalError) {
                RTConnection.error('applyRemoteActions failed due to internal error!');
                return $.Deferred().reject();
            }

            // do nothing, if the update message does not contain any actions
            if (!_.isArray(actions) || (actions.length === 0)) {
                return $.when();
            }

            // more rt debug logging to better find problems
            RTConnection.log('Last remote action for call model.applyActions: ' + JSON.stringify(_.last(actions)));

            return model.applyActions(actions, { external: true, async: true }).done(function () {
                // remember that actions have been received from the server
                remotelyModified = true;
            }).fail(function () {
                //TODO: We have to set internalError to true. Our DOM is not in sync anymore!
                RTConnection.error('model.applyActions failed!');
            });
        }

        /**
         * Applies the passed application state and actions passed with an
         * 'update' push message received from the real-time framework. Actions
         * will be applied in a background loop to prevent unresponsive browser
         * and warning messages from the browser. If actions from older
         * 'update' messages are still being applied, the new message data
         * passed to this method will be cached, and will be processed after
         * the background task for the old actions is finished.
         *
         * @param {Object} data
         *  The message data received from 'update' messages sent by the
         *  real-time framework.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.notify=false]
         *      If set to true, the application will trigger a 'docs:update'
         *      event with the message data, after the actions have been
         *      applied.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved or rejected
         *  after the message data has been processed.
         */
        var applyUpdateMessageData = this.createSynchronizedMethod(function (data, options) {

            Utils.log('EditApplication.applyUpdateMessageData(): received update message:', data);

            // apply all actions asynchronously
            return applyRemoteActions(data).done(function () {

                // update application state (read-only mode, etc.)
                applyEditState(data);

                // update application state (fileName, fileVersion)
                applyFileState(data);

                // update user data (user's current selection, etc.)
                applyUserData(data);

                // remove message data from pending queue
                pendingRemoteUpdates.shift();
                RTConnection.log('pendingUpdates: update processed, pending = ' + pendingRemoteUpdates.length);

                // notify all listeners if specified
                if (Utils.getBooleanOption(options, 'notify', false)) {
                    self.trigger('docs:update', data);
                }
            }).fail(function () {
                RTConnection.error('applyRemoteActions failed!');
            }).always(function () {
                RTConnection.log('applyRemoteActions finished!');
            });
        });

        /**
         * Tries to create a new document in the Files application according to
         * the launch options passed to the constructor.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.convert=false]
         *      Whether to convert the template file to a native file format.
         *  @param {Object} [options.params]
         *      Additional parameters to be inserted into the AJAX request.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the new
         *  document has been created successfully. Otherwise, the Promise will
         *  be rejected.
         */
        function createNewDocument(options) {

            var // parameters passed to the server request
                requestParams = Utils.getObjectOption(options, 'params', {}),
                // whether to convert the file to a native format
                convert = Utils.getBooleanOption(options, 'convert', false),

                // file options
                folderId = Utils.getStringOption(launchOptions, 'folderId', ''),
                templateFile = Utils.getObjectOption(launchOptions, 'templateFile'),
                preserveFileName = Utils.getBooleanOption(launchOptions, 'preserveFileName', false),
                initial_filename = Utils.getStringOption(launchOptions, 'initial_filename', /*#. default base file name (without extension) for new OX Documents files (will be extended to e.g. 'unnamed(1).docx') */ gt('unnamed')),
                initial_folderId = Utils.getStringOption(launchOptions, 'initial_folderId', ''),

                // origin of a file attachment stored as template file in Drive (e.g. a mail)
                origin = Utils.getObjectOption(templateFile, 'origin'),

                // the server request to create the document
                request = null,

                // error messages, mapped by server error codes
                ERROR_MESSAGES = {
                    CREATEDOCUMENT_CONVERSION_FAILED_ERROR:               gt('Could not create document because the necessary conversion of the file failed.'),
                    CREATEDOCUMENT_CANNOT_READ_TEMPLATEFILE_ERROR:        gt('Could not create document because the necessary template file could not be read.'),
                    CREATEDOCUMENT_CANNOT_READ_DEFAULTTEMPLATEFILE_ERROR: gt('Could not create document because the default template file could not be read.'),
                    GENERAL_PERMISSION_CREATE_MISSING_ERROR:              gt('Could not create document due to missing folder permissions.'),
                    GENERAL_QUOTA_REACHED_ERROR:                          gt('Could not create document because the allowed quota is reached. Please delete some items in order to create new ones.'),
                    GENERAL_FILE_NOT_FOUND_ERROR:                         gt('Could not create document because the template file could not be found.')
                };

            // sets document into internal error state, returns a rejected promise with an appropriate error descriptor
            function createErrorResponse(errorCode) {

                var // error message according to passed error code, with default message according to conversion mode
                    message = (_.isString(errorCode) && (errorCode in ERROR_MESSAGES)) ? ERROR_MESSAGES[errorCode] :
                        convert ? gt('Could not convert document.') : gt('Could not create document.');

                setInternalError();
                return $.Deferred().reject({ cause: convert ? 'convert' : 'create', headline: gt('Server Error'), message: message });
            }

            addTimePerformance('createNewDocStart');

            // folder identifier must exist in launch options
            if (folderId.length === 0) {
                Utils.error('EditApplication.createNewDocument(): cannot create new document without folder identifier');
                return $.Deferred().reject();
            }


            // create the URL options
            _.extend(requestParams, {
                action: 'createdefaultdocument',
                folder_id: folderId,
                document_type: self.getDocumentType(),
                initial_filename: initial_filename,
                initial_folderId: initial_folderId,
                preserve_filename: preserveFileName,
                convert: convert
            });

            if (launchOptions.template) {
                _.extend(requestParams, {
                    action: 'createfromtemplate',
                    folder_id: folderId,
                    id: templateFile.id
                });
            }

            // additional parameters according to creation mode
            if (_.isObject(templateFile)) {
                _.extend(requestParams, {
                    template_id: templateFile.id,
                    version: templateFile.version
                });
            }

            // show alert banners after document has been imported successfully
            self.onImportSuccess(function () {

                var // the message text of the alert banner
                    message = null,
                    // the target folder, may differ from the folder in the launch options
                    targetFolderId = Utils.getStringOption(self.getFileDescriptor(), 'folder_id'),
                    // Whether the file has been created in another folder. We
                    // have to check the template option which always use the
                    // user folder as target. In that case no message should be shown.
                    copied = (launchOptions.template) ? false : (folderId !== targetFolderId);

                // For a file created from a template we have to check the
                // converted attribute in the file descriptor to be sure
                // that it has been converted to another file format or not.
                convert = (launchOptions.template) ? self.getFileDescriptor().converted : convert;
                if (convert) {
                    message = copied ?
                        //#. %1$s is the new file name
                        //#, c-format
                        gt('Document was converted and stored in your default folder as "%1$s".', _.noI18n(self.getFullFileName())) :
                        //#. %1$s is the new file name
                        //#, c-format
                        gt('Document was converted and stored as "%1$s".', _.noI18n(self.getFullFileName()));
                } else if (copied) {
                    message =
                        //#. %1$s is the file name
                        //#, c-format
                        gt('Document "%1$s" was created in your default folder to allow editing.', _.noI18n(self.getFullFileName()));
                }

                if (message) {
                    view.yell({ type: 'info', message: message });
                }
            });

            //optional values for changing "insertField" operation by url-hashs assigned as "fields" or "sourcefields" if it is a csv file
            var fields = _.url.hash('fields');
            var sourcefields = _.url.hash('fields-source');
            if(fields || sourcefields){
                _.url.hash('fields', null);
                _.url.hash('fields-source', null);
                requestParams.fields = fields;
                requestParams.sourcefields = sourcefields;
            }

            // send the server request to create the new document
            request = self.sendRequest(IO.FILTER_MODULE_NAME, requestParams);

            // check response whether it contains an error code
            request = request.then(function (response) {
                addTimePerformance('createNewDocAfterRequest');
                // forward response data if no error occured, otherwise create a rejected promise with a GUI message
                var error = new ErrorCode(response);
                return error.isError() ? createErrorResponse(error.getCodeAsConstant()) : response;
            }, function () {
                // server request failed completely, create generic response data
                return createErrorResponse();
            });

            // process data descriptor returned by the server
            request = request.then(function (data) {

                // callback for promise piping: leaves busy mode, forwards original state and result
                function leaveBusyEarly(response) {
                    var resolved = this.state() === 'resolved';
                    return self.leaveBusyDuringImport().then(
                        // leaving busy state succeeded: return original state and response
                        function () {
                            addTimePerformance('afterLeaveBusyEarly');
                            return resolved ? response : $.Deferred().reject(response);
                        },
                        // leaving busy state failed: always reject the promise chain
                        function () { return resolved ? arguments[0] : response; }
                    );
                }

                if (!_.isObject(data.file)) {
                    Utils.error('EditApplication.createNewDocument(): missing file descriptor');
                    return $.Deferred().reject();
                }

                // add the origin to the file descriptor (e.g. mail descriptor whose attachment has been copied to Drive)
                if (origin) { data.file.origin = origin; }
                // set and propagate file descriptor
                self.registerFileDescriptor(data.file);

                // determine if server sent an extended response with initial document contents
                fastEmpty = _.isString(data.htmlDoc) && _.isArray(data.operations) && (data.operations.length > 0);
                if (fastEmpty && fastEmptyLoadHandler) {
                    // invoke fast-load handler provided by the application
                    return fastEmptyLoadHandler.call(self, data.htmlDoc, { operations: data.operations })
                        // leave busy mode early (also if import handler fails)
                        .then(leaveBusyEarly, leaveBusyEarly)
                        // forward response data to resulting piped promise
                        .then(function () { return data; });
                }

                // forward returned data to piped promise
                return data;
            });

            // return the final piped promise
            return request;
        }

        /**
         * Loads the document described in the current file descriptor.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  document has been loaded, or rejected when an error has occurred.
         */
        function importDocument() {

            var // time for initializing document
                initializationTime = 0,
                // Performance: Whether stored data can be used for loading the document
                useStorageData = false,
                // connect update messages
                connectUpdateMessages = [],
                // update with document operations
                updateWithDocumentOperation = $.Deferred();

            // shows a log message with a duration in the browser console
            function logDuration(duration, message1, message2) {
                Utils.info('EditApplication.importDocument(): ' + message1 + ' in ' + duration + 'ms' + (_.isString(message2) ? (' (' + message2 + ')') : ''));
            }

            // invokes the passed callback function that must return a Deferred, and updates the progress bar during invocation
            function invokeCallbackWithSpinner(callback, message, timerLogKey, count) {

                var // profiling: start time of the callbck invocation
                    lapTime = _.now();

                addTimePerformance(timerLogKey + 'Start', lapTime);

                return (_.isFunction(callback) ? callback.call(self) : $.when())
                    .then(function (response) {
                        var endTime = _.now(),
                            totalTime = endTime - lapTime,
                            timePerElem = (count > 0) ? Utils.round(totalTime / count, 0.001) : 0;

                        logDuration(totalTime, message, (count > 0) ? (count + ' elements, ' + timePerElem + 'ms per element') : null);
                        addTimePerformance(timerLogKey + 'End', endTime);
                        // bug 32520: reject the chain of deferreds when closing document during import
                        return self.isInQuit() ? $.Deferred().reject({ cause: 'closing' }) : response;
                    });
            }

            // Function to collect update data while connecting to/importing document.
            // Must handle error messages, too in case of asynchronous loading.
            function collectUpdateEvent(event, data) {
                if (data && data.finalLoadUpdate) {
                    if (data.hasErrors) {
                        updateWithDocumentOperation.reject(data);
                    } else {
                        updateWithDocumentOperation.resolve(data);
                        RTConnection.log('Final load update received for asynchronous load');
                    }
                } else {
                    connectUpdateMessages.push(data);
                    RTConnection.log('Update received during document import');
                }
            }

            addTimePerformance('importDocStart');

            // during opened undo group, collect all operations in the same action
            model.getUndoManager().on('undogroup:open', function () {
                pendingAction = createAction();
            });

            // undo group closed: send the pending action to the server
            model.getUndoManager().on('undogroup:close', function () {
                if (pendingAction.operations.length > 0) {
                    registerAction(pendingAction);
                }
                pendingAction = null;
            });

            // collect all operations generated by the document model, send them to the server
            model.on('operations:success', function (event, operations, external) {
                // ignore external operations
                if (!external) {
                    if (_.isObject(pendingAction)) {
                        // if inside an undo group, push all operations into the pending action
                        pendingAction.operations = pendingAction.operations.concat(operations);
                    } else {
                        // otherwise, generate and send a self-contained action
                        registerAction(createAction(operations));
                    }
                }
            });

            // error occurred while applying operations
            model.on('operations:error', function () {
                setInternalError();
                if (self.isImportFinished()) {
                    view.yell({ type: 'error', headline: gt('Internal Error'), message: gt('An unrecoverable error occurred while modifying the document.') });
                    RTConnection.error('an unrecoverable error occurred while modifying the document');
                }
            });

            // disable dropping of images onto the application background area
            self.getWindowNode().on('drop', false);

            // immediately show the busy blocker screen
            if (!fastEmpty) {
                view.enterBusy({
                    cancelHandler: _.bind(self.quit, self),
                    immediate: true,
                    showFileName: true,
                    warningLabel: gt('Sorry, your document is very large. It will take some time to be loaded.'),
                    warningDelay: 10000
                });
            }

            // create the real-time connection to the server, register event listeners
            rtConnection = new RTConnection(self, initOptions);
            rtConnection.on({
                online: connectionOnlineHandler,
                offline: connectionOfflineHandler,
                reset: connectionResetHandler,
                timeout: connectionTimeoutHandler,
                'error:notMember': connectionNotMemberHandler
            });

            // Start to listen for updates before we connect, it's possible
            // that we receive updates before the connect promise is resolved.
            rtConnection.on('update', collectUpdateEvent);

            // log the initalization time needed before import actually starts
            logDuration(initializationTime, 'initialization finished');

            // called for normal doc loading, and for fast load of empty doc
            // after postprocessing is done. Applies edit state and update message data
            function finalizeImport(data, docUpdateMessages) {
                asyncLoadingFinished = (updateWithDocumentOperation.state() !== 'pending');

                // register handlers for hangup and switching edit rights notifications
                rtConnection.on({
                    hangup: connectionHangupHandler,
                    preparelosingeditrights: handlePrepareLosingEditRights,
                    declinedacquireeditrights: handleDeclinedAcquireEditRights,
                    acceptedacquireeditrights: handleAcceptedAcquireEditRights,
                    noUpdateFromServer: handleNoUpdateFromServer,
                    flusherrorinfo: handleFlushErrorInfo
                });

                // update application state (read-only mode, etc.)
                applyEditState(data);
                addTimePerformance('editMode');

                // apply all 'update' messages collected during import (wait for the import promise, otherwise
                // methods that work with the method BaseApplication.isImportFinished() will not run correctly)
                self.onImportSuccess(function () {
                    // first apply actions received on connect() answer
                    RTConnection.log('Applying document update messages, count = ' + docUpdateMessages.length);
                    _.each(docUpdateMessages, function (updateData) { applyUpdateMessageData(updateData); });
                    // second apply actions received during connect() and document import
                    RTConnection.log('Applying update messages collected during connect/import, count = ' + connectUpdateMessages.length);
                    _.each(connectUpdateMessages, function (updateData) { applyUpdateMessageData(updateData); });
                    // reconnect 'update' events for direct message processing, forward event
                    rtConnection.off('update', collectUpdateEvent).on('update', function (event, updateData) {
                        // public method triggers 'docs:update' event
                        self.applyUpdateMessageData(updateData);
                    });
                });
            }

            function loadAndProcessDocumentActions(data) {
                // successfully connected to real-time framework, actions downloaded

                var // the initial action from the server response
                    actions = getActionsFromData(data),
                    // 'update' messages received by answer from connect
                    docUpdateMessages = [],
                    // update data retrieved from connect message
                    updateData = getUpdateAttributesFromData(data),
                    // the error data sent with the server response (no indication of an error)
                    error = new ErrorCode(data),
                    // the result passed with a rejected Deferred object
                    errorResult = null,
                    // the key under which the data might be stored in browser cache
                    loadStorageKey = null,
                    // the key containing the version of the saved string in the storage
                    storageVersionKey = null,
                    // the saved storage version
                    storageVersion = 0,
                    // the key under which the OSN and the file version for the file are stored in browser cache
                    loadStorageKeyOSN = null, storageFileVersionKey = null,
                    // the supported extensions for the local storage
                    supportedExtensions = null,
                    // a counter for the supported extensions
                    counter = 0;

                // helper function to clear content in local storage
                function clearLocalStorage() {
                    _.each(supportedExtensions, function (oneExtensionDefintion) { localStorage.removeItem(loadStorageKey + oneExtensionDefintion.extension); });
                    localStorage.removeItem(loadStorageKeyOSN);
                    localStorage.removeItem(storageVersionKey);
                    localStorage.removeItem(storageFileVersionKey);
                }

                // saveFileInLocalStorage can be overwritten with user setting in debug mode
                checkDebugUseLocalStorage();

                // checking, if the file can be loaded from storage (or another source)
                if ((typeof(Storage) !== 'undefined') && (localStorageApp) && (saveFileInLocalStorage) && (isLocalStorageSupported) && (data.syncInfo) &&
                    (data.syncInfo.fileVersion) && (data.syncInfo['document-osn']) && (_.isFinite(data.syncInfo['document-osn'])) &&
                    (data.syncInfo.fileId) && (data.syncInfo.folderId) &&
                    (_.isFunction(model.setFullModelNode)) && _.isFunction(model.getSupportedStorageExtensions)) {

                    loadStorageKey = data.syncInfo.fileId + '_' + data.syncInfo.folderId;
                    storageFileVersionKey = loadStorageKey + '_FILEVERSION';
                    storageVersionKey = loadStorageKey + '_STORAGEVERSION';
                    loadStorageKeyOSN = loadStorageKey + '_OSN';

                    supportedExtensions = model.getSupportedStorageExtensions();

                    // checking the supported storage version
                    storageVersion = (localStorage.getItem(storageVersionKey) && parseInt(localStorage.getItem(storageVersionKey), 10)) || 1;

                    if (storageVersion >= requiredStorageVersion) {

                        Utils.info('importDocument, document-osn=' + data.syncInfo['document-osn'] + ', document-rev=' + data.syncInfo.fileVersion);

                        // Support for localStorage -> comparing OSN and file version
                        if ((localStorage.getItem(storageFileVersionKey) && localStorage.getItem(storageFileVersionKey) === data.syncInfo.fileVersion) &&
                            ((localStorage.getItem(loadStorageKeyOSN)) && parseInt(localStorage.getItem(loadStorageKeyOSN), 10) === parseInt(data.syncInfo['document-osn'], 10))) {

                            Utils.iterateArray(supportedExtensions, function (oneExtensionDefintion) {

                                var // one specific local storage key
                                    oneLocalStorageKey = loadStorageKey + oneExtensionDefintion.extension,
                                    // the options for setFullModelNode
                                    options = { usedLocalStorage: true };

                                if (_.isObject(oneExtensionDefintion.additionalOptions)) { _.extend(options, oneExtensionDefintion.additionalOptions); }

                                if (localStorage.getItem(oneLocalStorageKey)) {
                                    Utils.info('importDocument, loading document from local storage. Key: ' + oneLocalStorageKey + ' Length: ' + localStorage.getItem(oneLocalStorageKey).length);
                                    loadedFromStorage = true;
                                    useStorageData = true;
                                    if (counter === 0) { model.setLoadDocumentOsn(parseInt(data.syncInfo['document-osn'], 10)); }
                                    model.setFullModelNode(localStorage.getItem(oneLocalStorageKey), options);
                                    counter++;
                                } else {
                                    Utils.info('importDocument, optional local storage value not found. Key: ' + oneLocalStorageKey);
                                    if (!oneExtensionDefintion.optional) {
                                        Utils.info('importDocument, file in local storage not found. Key: ' + loadStorageKey + '. But found OSN key: ' + loadStorageKeyOSN);
                                        clearLocalStorage();
                                        return Utils.BREAK;
                                    }
                                }
                            });

                        } else {
                            Utils.info('importDocument, file in local storage not found. Key: ' + loadStorageKey);
                            clearLocalStorage();
                        }
                    } else {
                        Utils.info('importDocument, version of file in local storage not supported. Key: ' + loadStorageKey);
                        clearLocalStorage();
                    }
                }

                if (!loadedFromStorage) {
                    if (data.htmlDoc && _.isFunction(model.setFullModelNode) && !fastEmpty) { // if doc is empty, this was already called
                        Utils.info('importDocument, load complete HTML document, osn: ', data.syncInfo['document-osn']);
                        model.setLoadDocumentOsn(parseInt(data.syncInfo['document-osn'], 10));
                        //model.setFullModelNode(data.htmlDoc);

                        // markup is in form of object, parse and get data from mainDocument
                        var htmlDoc = JSON.parse(data.htmlDoc);

                        if (htmlDoc.mainDocument) {
                            model.setFullModelNode(htmlDoc.mainDocument);
                        }
                        // if there is header/footer data, restore that, too
                        if (htmlDoc.headerFooter) {
                            var unionString = '';
                            _.each(htmlDoc.headerFooter, function (element) {
                                unionString += element;
                            });
                            model.setFullModelNode(unionString, { headerFooter: true });
                            model.getPageLayout().headerFooterCollectionUpdate();
                        }
                    }
                }

                // setting the full display name of the client
                clientDisplayName = setClientDisplayName(data);

                // connect sends us the first update event together with the action data
                docUpdateMessages.push(updateData);

                // get the own unique client identifier
                clientId = Utils.getStringOption(data, 'clientId', '');
                clientIndexes[clientId] = 0;

                // check error conditions
                if (Utils.getBooleanOption(data, 'hasErrors', false)) {
                    Utils.error('EditApplication.importDocument(): server side error');
                    errorResult = { cause: 'server'};
                } else if (Utils.getBooleanOption(data, 'operationError', false)) {
                    Utils.error('EditApplication.importDocument(): operation failure');
                    errorResult = { cause: data.message };
                } else if (error.isError()) {
                    Utils.error('EditApplication.importDocument(): server side error, error code = ' + error.getCodeAsConstant());
                    errorResult = { cause: 'server'};
                } else if (clientId.length === 0) {
                    Utils.error('EditApplication.importDocument(): missing client identifier');
                    errorResult = { cause: 'noclientid' };
                } else if (!_.isArray(actions) || (actions.length === 0)) {
                    Utils.error('EditApplication.importDocument(): missing actions');
                    errorResult = { cause: 'noactions' };
                }

                if (_.isObject(errorResult)) {
                    // return rejected Deferred object with error information
                    errorResult = _.extend(errorResult, genericImportFailedHandler(data));
                    return $.Deferred().reject(errorResult);
                }

                if (fastEmpty) { // we already have preprocessing, actions and formatting applied, so we finalize doc importing
                    finalizeImport(data, docUpdateMessages);
                } else {
                    // invoke pre-process callback function passed to the constructor of this application
                    return invokeCallbackWithSpinner(preProcessHandler, 'preprocessing finished', 'preProcessing')
                        .then(function () {

                            var actionOptions = { external: true, async: true, useStorageData: useStorageData };

                            // Load performance: Execute only specified operations during load of document, if the document can be loaded from local storage
                            if (useStorageData && operationsFilter) { _.extend(actionOptions, {filter: operationsFilter}); }

                            // callback function passed to the method invokeCallbackWithProgress()
                            function applyActions() {
                                var detachAppPane = Utils.getBooleanOption(initOptions, 'applyActionsDetached', false);
                                if (detachAppPane) { view.detachAppPane(); }
                                return model.applyActions(actions, actionOptions).always(function () {
                                    if (detachAppPane) { view.attachAppPane(); }
                                });
                            }

                            var // total number of operations, for logging
                                operationCount = Utils.DEBUG ? Utils.getSum(actions, function (action) {
                                    return Utils.getArrayOption(action, 'operations', []).length;
                                }) : 0;

                            // apply actions at document model asynchronously, update progress bar
                            return invokeCallbackWithSpinner(applyActions, 'operations applied', 'applyActions', operationCount)
                                .then(function () {
                                    // successfully applied all actions

                                    // After successfully applied all actions we must have a valid operation state number.
                                    // Otherwise there is a synchronization error and we have to stop processing the document
                                    // further showing an error message.
                                    if (model.getOperationStateNumber() < 0) {
                                        return $.Deferred().reject({ cause: 'applyactions', headline: gt('Synchronization Error') });
                                    }

                                    // invoke post-process callback function passed to the constructor of this application
                                    return invokeCallbackWithSpinner(useStorageData ? postProcessHandlerStorage : postProcessHandler, 'postprocessing finished', 'postProcessing')
                                        .then(function () {
                                            finalizeImport(data, docUpdateMessages);
                                        }, function (response) {
                                            // failure: post-processing failed
                                            return _.extend({ cause: 'postprocess' }, response);
                                        });

                                }, function (response) {
                                    // failure: applying actions failed
                                    // RT connection already destroyed when quitting while loading
                                    if (rtConnection) {
                                        rtConnection.off('update', collectUpdateEvent);
                                    }
                                    return _.extend({ cause: 'applyactions' }, response);
                                });

                        }, function (response) {
                            // failure (pre-processing failed): adjust error response
                            return _.extend({ cause: 'preprocess' }, response);
                        });
                }
            }

            // synchronizes with a finalLoad update that is required to finalize
            // a asynchronous loading.
            function waitUntilUpdateWithOperations() {
                return updateWithDocumentOperation.then(function (data) {
                    return loadAndProcessDocumentActions(data);
                }, function (response) {
                    return genericImportFailedHandler(_.extend({ cause: 'asyncload' }, response));
                });
            }

            // connect to server, receive initial operations of the document
            return invokeCallbackWithSpinner(function () {
                    var // parameters for connect
                        connectOptions = {
                            fastEmpty: fastEmpty,
                            useLocalStorage: saveFileInLocalStorage,
                            app: self.getDocumentType(),
                            newDocument: (Utils.getStringOption(launchOptions, 'action') === 'new')
                        };

                    return rtConnection.connect(connectOptions);
                }, 'actions downloaded', 'connect')
                .then(function (data) {

                    var // extract error object from the server response
                        error = new ErrorCode(data),
                        // extract optional syncLoad property from the server response
                        syncLoad = Utils.getBooleanOption(data, 'syncLoad', false),
                        // data for an early preview of the imported document (must contain some operations)
                        previewData = Utils.getObjectOption(data, 'preview');

                    // helper function to avoid code duplication in then-handler
                    function postPreprocessHelper() {
                        if (fastEmpty) {
                            // osn is -1 and we need to set it to correct value, server osn is used
                            model.setOperationStateNumber(parseInt(data.serverOSN, 10));
                        }

                        if (syncLoad || error.isError()) {
                            return loadAndProcessDocumentActions(data);
                        }

                        return invokeCallbackWithSpinner(waitUntilUpdateWithOperations, 'asynchronous import', 'asyncLoad');
                    }

                    // Special synchronous handling to show document preview
                    if (previewData && _.isArray(previewData.operations)) {

                        // Forcing to call the preprocess handler, before operations are applied
                        return invokeCallbackWithSpinner(preProcessHandler, 'preprocessing finished', 'preProcessing')
                        .then(function () {

                            var promise = null;

                            // avoid that the preprocess handler is called again
                            preProcessHandler = null;

                            // apply the operations synchronously
                            promise = model.applyActions({ operations: data.preview.operations }, { external: true });

                            // synchronous promise can be evaluated immediately
                            if (promise.state() === 'rejected') {
                                // handling error during applying operations synchronously (36639)
                                data.operationError = true;
                                data.message = gt('An unrecoverable error occurred while modifying the document.');
                                syncLoad = true; // misusing for error handling
                                return postPreprocessHelper();
                            }

                            // invoke the preview handler after the operations have been applied
                            // TODO: treat false as error, or continue silently in busy mode?
                            promise = previewHandler.call(self, previewData) ? self.leaveBusyDuringImport() : $.when();

                            return promise.then(postPreprocessHelper);
                        });
                    }

                    return postPreprocessHelper();
                }, function (response) {
                    return genericImportFailedHandler(response);
                })
                .always(function () {
                    // TODO: show a success message if the preview was active too long?
                    previewStartTime = null;
                    updateState();
                    addTimePerformance('leaveBusy');
                });
        }

        /**
         * Generic import failed handler which handles general errors that are
         * not application dependent. Application specific errors will be
         * handled by an optional importFailedHandler which can be set using
         * [initOptions.importFailedHandler.
         *
         * @param {Object} response
         *  The response of the server-side which contains internal error info
         *  that can be used to create UI specific strings.
         *
         * @returns {Object}
         *  The result object extended with user information to be presented to
         *  the user describing what is the root cause of the error.
         */
        function genericImportFailedHandler(response) {

            if (self.isInQuit()) {
                response = { cause: 'quit' };
            }

            var // specific error code sent by the server
                error = new ErrorCode(response),
                constant = error.getCodeAsConstant();

            internalError = true;
            importFailedHandler.call(self, response);

            response = _.clone(response);
            switch (constant) {
            case 'LOADDOCUMENT_FAILED_ERROR':
                delete response.message; // fall-back to default error message
                break;
            case 'LOADDOCUMENT_CANNOT_RETRIEVE_OPERATIONS_ERROR':
                delete response.message; // fall-back to default error message
                break;
            case 'LOADDOCUMENT_CANNOT_READ_PASSWORD_PROTECTED_ERROR':
                response.message = gt('The document is protected with a password.');
                break;
            case 'GENERAL_PERMISSION_READ_MISSING_ERROR':
                response.message = gt('The document could not be read, because you do not have read permissions.');
                break;
            case 'GENERAL_FILE_NOT_FOUND_ERROR':
                movedAway = true;
                response.message = gt('The document cannot be found. Please check if the document has been removed.');
                break;
            case 'GENERAL_MEMORY_TOO_LOW_ERROR':
                response.message = gt('The document could not be loaded. The server is currently too busy. Please try again later.');
                break;
            case 'LOADDOCUMENT_TIMEOUT_RETRIEVING_STREAM_ERROR':
                response.message = gt('The document could not be loaded, because the document data could not be read in time. Please try again later.');
                break;
            case 'NO_ERROR':
                // in case we haven't received a office backend error we have to
                // check for a possible real-time error
                error = ErrorCode.extractRealtimeError(response);
                if (error && error.code) {
                    switch (error.code) {
                    case 'RT_STANZA-0006':
                        response.message = gt('The document could not be loaded. The server is currently too busy. Please try again later.');
                        break;
                    }
                }
                break;
            }

            return response;
        }

        /**
         * Callback executed by the base class BaseApplication to import the
         * document. Creates a new document or converts an existing document,
         * if specified in the launch options.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  document has been loaded, or rejected when an error has occurred.
         */
        function importHandler() {

            var // the Deferred object for creation of new document or conversion of existing document
                def = null;

            addTimePerformance('importHandlerStart');

            // create new document in OX Files if required
            switch (Utils.getStringOption(launchOptions, 'action', 'load')) {
            case 'load':
                def = $.when();
                break;
            case 'new':
                def = createNewDocument({ params: newDocumentParams });
                break;
            case 'convert':
                def = createNewDocument({ convert: true });
                break;
            default:
                Utils.error('EditApplication.importHandler(): unsupported launch action');
                def = $.Deferred().reject();
            }

            // import the document, if creation/conversion has succeeded
            return def.then(importDocument);
        }

        /**
         * Creates and returns a new action containing operations and
         * additional data.
         *
         * @param {Array} [operations]
         *  The operations contained by the new action. If omitted, an empty
         *  array will be inserted into the action.
         *
         * @returns {Object}
         *  The new action.
         */
        function createAction(operations) {
            return { operations: operations || [] };
        }

        /**
         * Sends all actions that are currently stored in the actions buffer.
         * If called repeatedly while still sending actions, the new actions
         * buffered in the meantime will be stored in another internal buffer,
         * and will be sent afterwards, until no new actions have been inserted
         * in the actions buffer anymore.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  last real-time message containing new actions has been
         *  acknowledged, or rejected if any real-time message has failed.
         */
        var sendActions = (function () {

            var // the Deferred object that will be returned to the caller
                resultDef = null;

            // sends the current actions buffer (calls itself recursively)
            function sendActionsBuffer() {

                if (RTConnection.debug && inPrepareLosingEditRights && (actionsBuffer && actionsBuffer.length > 0)) {
                    // we need to debug operations while we are in the process of switching edit rights
                    var tmp = actionsBuffer[actionsBuffer.length - 1].operations;
                    try {
                        if (tmp && tmp.length > 0) {
                            Utils.log('EditApplication.sendActionsBuffer(): last action: ' + JSON.stringify(JSON.stringify(tmp[tmp.length - 1])));
                        }
                    } catch (ex) {
                        // do nothing
                    }
                }

                // try to reduce the number of actions and operations that need to be sent to the server
                // testing optimizations because of problems with second undo/redo
                if (optimizeOperationsHandler) {
                    actionsBuffer = optimizeOperationsHandler.call(self, actionsBuffer);
                }

                // send all existing actions
                rtConnection.sendActions(actionsBuffer)
                .done(function () {
                    if (internalError) {
                        RTConnection.error('EditApplications.sendActionsBuffer(): sent successful, but internal error detected!');
                        resultDef.reject();
                    } else if (actionsBuffer.length === 0) {
                        resultDef.resolve();
                    } else {
                        sendActionsBuffer();
                    }
                })
                .fail(function (response) {
                    // TODO: error handling
                    RTConnection.error('EditApplications.sendActionsBuffer(): rtConnection.sendActions promise failed!');
                    resultDef.reject(response);
                });

                actionsBuffer = [];
            }

            return function () {

                RTConnection.log('EditApplication.sendActions(): entered');

                // a real-time message is currently running: return the current
                // result Deferred object that will be resolved after all messages
                if (sendingActions) {
                    RTConnection.log('EditApplication.sendActions(): sending in progress - using current promise');
                    return $.when(resultDef, renamePromise);
                }

                // internal error: immediately return with rejected Deferred
                if (internalError) {
                    RTConnection.error('EditApplication.sendActions(): internal error detected - promise rejected!');
                    return $.Deferred().reject();
                }

                // no new actions: return immediately
                if (actionsBuffer.length === 0) {
                    RTConnection.log('EditApplication.sendActions(): no need to send action buffer empty');
                    return $.when(renamePromise);
                }

                // prepare the result Deferred object (will be kept unresolved until
                // all actions have been sent, also including subsequent iterations
                resultDef = $.Deferred().always(function () {
                    RTConnection.log('EditApplication.sendActions(): current send promise resolved - sendingActions = false');
                    sendingActions = false;
                    updateState();
                });
                sendingActions = true;
                updateState();

                // start sending the actions (will call itself recursively)
                sendActionsBuffer();
                return $.when(resultDef, renamePromise);
            };

        }()); // end of local scope for sendActions() method

        /**
         * Registers the passed action to be sent to the server. All registered
         * actions will be collected and sent automatically in a debounced
         * call to the sendActions() method.
         *
         * @param {Object} action
         *  A new action to be sent to the server.
         */
        var registerAction = (function () {

            // direct callback: called every time registerAction() is called, push action to array
            function storeAction(action) {
                if (RTConnection.debug && (action.operations.length > 0)) {
                    // for debug purposes we log actions
                    RTConnection.log('EditApplication.storeAction(): last action: ' + JSON.stringify(_.last(action.operations)));
                }

                if (!internalError && (action.operations.length > 0)) {
                    actionsBuffer.push(action);
                    locallyModified = true;
                    updateState();
                }
            }

            // create and return the debounced registerAction() method
            return self.createDebouncedMethod(storeAction, sendActions, { delay: sendActionsDelay, maxDelay: 5 * sendActionsDelay });

        }()); // registerAction()

        /**
         * Sends all pending actions and flushes the document.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  real-time message has been acknowledged.
         */
        function flushDocumentHandler(reason) {

            // do nothing if application is already in internal error state
            if (internalError) { return $.Deferred().reject(); }

            // call the flush preparation handler passed to the constructor, then force sending all pending actions, then do the flush
            return $.when(prepareFlushHandler.call(self, reason)).then(sendActions).then(_.bind(rtConnection.flushDocument, rtConnection));
        }

        /**
         * The handler function that will be called when the application is
         * asked to be closed. If the edited document has unsaved changes, a
         * dialog will be shown asking whether to save or drop the changes, or
         * to continue editing.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved if the
         *  application can be closed (either if it is unchanged, or the user
         *  has chosen to save or lose the changes), or will be rejected if the
         *  application will continue to run (user has cancelled the dialog,
         *  save operation failed).
         */
        function beforeQuitHandler() {

            var // the deferred returned to the framework
                def = $.Deferred();

            // don't call sendActions when we are offline or in an error state
            if ((state !== 'offline') && (state !== 'error')) {
                // send pending actions to the server
                $.when(prepareFlushHandler.call(self, 'quit'))
                .then(function () {

                    var // maximum time for sendActions (in ms)
                        maxDelay = 5000,
                        // a timer for controlling sendActions
                        sendActionsTimer = null,
                        // the deferred returned by sendActions
                        sendActionsDef = sendActions();

                    if (sendActionsDef.state() === 'pending') {

                        // prevent endless waiting for sendActions to complete
                        sendActionsTimer = self.executeDelayed(function () {
                            if (sendActionsDef.state() === 'pending') {
                                def.resolve({ cause: 'timeout' });
                            }
                        }, maxDelay);

                        // This must not take more than specified in maxDelay
                        sendActionsDef.always(function () {
                            sendActionsTimer.abort();
                        });
                    }

                    return sendActionsDef;
                })
                .done(function () {

                    var // dialog to ask the user whether to close the application
                        dialog = null;

                    // still pending actions: ask user whether to close the application
                    if (self.hasUnsavedChanges() && (clientId === oldEditClientId)) {
                        dialog = new Dialogs.ModalQueryDialog(gt('This document contains unsaved changes. Do you really want to close?'));
                        dialog.show().done(function () { def.resolve(); }).fail(function () { def.reject(); });
                    } else {
                        // all actions saved, close application without dialog
                        def.resolve();
                    }
                })
                .fail(function () {
                    def.resolve();  // reject would keep application alive
                });

            } else {
                // offline/error state: rejected promise would keep application alive
                def.resolve();
            }

            return def.promise();
        }

        /**
         * Called before the application will be really closed.
         *
         * @param {String} reason
         *  The reason for quitting the application, as passed to the method
         *  BaseApplication.quit().
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected if the application
         *  can be savely closed.
         */
        function quitHandler(reason) {

            var // the result deferred
                def = null,
                // the application window
                win = self.getWindow(),
                // the file descriptor of this application (bug 34464: may be
                // null, e.g. when quitting during creation of a new document)
                file = self.getFileDescriptor(),
                // whether to show a progress bar (enabled after a short delay)
                showProgressBar = false,
                // recent file data
                recentFileData = null,
                // data, that will be sent to the server
                dataToSend = null;

            // evaluates the result of the 'closedocument' call
            function checkCloseDocumentResult(data) {

                var // the result Deferred object
                    def = $.Deferred(),
                    // the error code in the response
                    errorCode = new ErrorCode(data),
                    // cause of a rejected call
                    cause = Utils.getStringOption(data, 'cause', 'unknown'),
                    // the message text
                    message = null,
                    // the headline text
                    headline = gt('Server Error'),
                    // message type
                    msgType = 'error';

                // no error/warning, forward response data to piped callbacks
                if (!errorCode.isError() && !errorCode.isWarning()) {
                    return data;
                }

                // create error message depending on error code
                switch (errorCode.getCodeAsConstant()) {
                case 'SAVEDOCUMENT_FAILED_NOBACKUP_ERROR':
                    message = gt('The document is inconsistent and the server was not able to create a backup file. To save your work, please copy the complete content into the clipboard and paste it into a new document.');
                    break;
                case 'SAVEDOCUMENT_FAILED_FILTER_OPERATION_ERROR':
                    message = gt('The document is in an inconsistent state. To save your work, please copy the complete content into the clipboard and paste it into a new document. Furthermore the last consistent document was restored.');
                    break;
                case 'GENERAL_FILE_NOT_FOUND_ERROR':
                    message = gt('The document cannot be found. Saving the document is not possible. To save your work, please copy the complete content into the clipboard and paste it into a new document.');
                    break;
                case 'GENERAL_QUOTA_REACHED_ERROR':
                    message = gt('The document could not be saved. Your quota limit has been reached. Please free up storage space.');
                    break;
                case 'GENERAL_PERMISSION_READ_MISSING_ERROR':
                    message = gt('The document could not be saved, because you lost read permissions. To save your work, please copy the complete content into the clipboard and paste it into a new document.');
                    break;
                case 'GENERAL_PERMISSION_WRITE_MISSING_ERROR':
                    message = gt('The document could not be saved, because you lost write permissions. To save your work, please copy the complete content into the clipboard and paste it into a new document.');
                    break;
                case 'GENERAL_MEMORY_TOO_LOW_ERROR':
                    message = gt('The document could not be saved. The server has not enough memory. To save your last changes, please copy the complete content into the clipboard and paste it into a new document. If you close the document, the last consistent version will be restored.');
                    break;
                case 'SAVEDOCUMENT_BACKUPFILE_CREATE_PERMISSION_MISSING_ERROR':
                    headline = gt('Server Warning');
                    msgType = 'warning';
                    message = gt('The document was saved successfully, but a backup copy could not be created. You do not have the permission to create a file in the folder.');
                    break;
                case 'SAVEDOCUMENT_BACKUPFILE_READ_OR_WRITE_PERMISSION_MISSING_ERROR':
                    headline = gt('Server Warning');
                    msgType = 'warning';
                    message = gt('The document was saved successfully, but a backup copy could not be created. You do not have the correct permissions in the folder.');
                    break;
                case 'SAVEDOCUMENT_BACKUPFILE_QUOTA_REACHED_ERROR':
                    headline = gt('Server Warning');
                    msgType = 'warning';
                    message = gt('The document was saved successfully, but a backup copy could not be created. Your quota limit has been reached. Please free up storage space. ');
                    break;
                case 'SAVEDOCUMENT_BACKUPFILE_CREATE_FAILED_ERROR':
                    headline = gt('Server Warning');
                    message = gt('The document was saved successfully, but a backup copy could not be created.');
                    msgType = 'warning';
                    break;
                default:
                    if (cause === 'timeout') {
                        headline = gt('Connection Error');
                        message = gt('Connection to the server has been lost.');
                    } else {
                        message = gt('An unknown error occurred on the server.');
                    }
                }

                if (errorCode.isError()) {
                    // Switch to internal error state, show the alert banner, and return to
                    // idle state to allow the user to rescue the document contents to the
                    // clipboard. After that, the user has to click the Quit button again.
                    internalError = true;
                    locked = true;
                    updateState();
                }

                if (model && view) {
                    model.setEditMode(false);
                    view.yell({ type: msgType, headline: headline, message: message });

                    // Bug 32989: Register a new quit handler that will reject the pending
                    // Deferred object (the base application has deleted all old quit handlers
                    // from its internal list, including this handler currently running). This
                    // covers execution of app.immediateQuit(), e.g. when logging out while
                    // this document is still waiting for the user to click the Quit button.
                    // Simply reject the Deferred but do not return it, otherwise the logout
                    // process will be aborted.
                    self.registerQuitHandler(function () { def.reject(); });

                    // The original app.quit() method has been modified to ensure that the
                    // quit code does not run multiple times, e.g. by double-clicking the Quit
                    // button. Here, the application will return to idle state, and the user
                    // has to click the Quit button again to really close the application. Thus,
                    // calling app.quit() has to reject the Deferred object in order to ensure
                    // that the application really shuts down.
                    var origMethod = self.quit;
                    self.quit = function () {
                        def.reject();
                        return origMethod.call(self);
                    };

                    // back to idle to allow final copy&paste
                    win.idle();
                    def.always(function () { win.busy(); });
                } else {
                    def.reject();
                }

                if (errorCode.getDescription()) {
                    Utils.log(errorCode.getDescription());
                }

                return def.promise();
            }

            // called in case the 'closedocument' call has been rejected
            function closeDocumentResultFailed(data) {
                return checkCloseDocumentResult(data);
            }

            // updating the progress bar during saving file in storage
            function progressHandler(progress) {
                if (showProgressBar) { view.updateBusyProgress(progress); }
            }

            // trying to save the file in the local storage
            function saveFileInStorage(data) {

                var // a timer for measuring conversion time
                    startTime = null,
                    // the key used in the local storage
                    saveKey = null,
                    // the key used in the local storage to save the OSN, file version and storage version
                    saveKeyOSN = null, versionKey = null, storageVersionKey = null,
                    // the result deferred object
                    def = null,
                    // whether the document was modified
                    newVersion = (locallyModified || locallyRenamed || remotelyModified),
                    // the operation state number of this local client
                    clientOSN = model.getOperationStateNumber(),
                    // a counter for the different strings in the local storage
                    counter = 0,
                    // the maximum number of previous versions, that are checked to be deleted
                    maxOldVersions = 20,
                    // a collector for all extensions used in the local storage
                    allExtensions = ['_OSN', '_FILEVERSION', '_STORAGEVERSION'];

                // helper function to clear content in local storage
                function clearLocalStorage(options) {

                    var // whether old registry entries shall be removed
                        migration = Utils.getBooleanOption(options, 'migration', false);

                    // deleting all currently used registry entries
                    _.each(allExtensions, function (oneExtension) { localStorage.removeItem(saveKey + oneExtension); });

                    // Migration code: Removing old registry values, where the file version was part of the base key (these are no longer used)
                    if (migration) {
                        if (_.isFinite(data.syncInfo.fileVersion)) {
                            _.each(_.range((data.syncInfo.fileVersion > maxOldVersions) ? (data.syncInfo.fileVersion - maxOldVersions) : 0, data.syncInfo.fileVersion), function (number) {
                                _.each(allExtensions, function (oneExtension) { localStorage.removeItem(saveKey + '_' + number + oneExtension); });
                            });
                        } else {
                            localStorage.removeItem(saveKey + '_' + data.syncInfo.fileVersion);
                            localStorage.removeItem(saveKey + '_CURRENTVERSION');
                        }
                    }
                }

                // saveFileInLocalStorage can be overwritten with user setting in debug mode
                checkDebugUseLocalStorage();

                // saving file in local storage, using the file version returned from the server (if storage can be used)
                if ((newVersion || !loadedFromStorage) && data && data.syncInfo && (typeof(Storage) !== 'undefined') &&
                        localStorageApp && saveFileInLocalStorage && isLocalStorageSupported && _.isFunction(model.getFullModelDescription) &&
                        data.syncInfo.fileVersion && data.syncInfo.fileId && data.syncInfo.folderId && data.syncInfo['document-osn'] &&
                        _.isFinite(data.syncInfo['document-osn'], 10) && (clientOSN === parseInt(data.syncInfo['document-osn'], 10))) {

                    def = $.Deferred();

                    // show the progress bar, if saving takes more than 1.5 seconds
                    self.executeDelayed(function () { showProgressBar = true; }, 1500);

                    // create the string describing the document to save it in local storage
                    self.executeDelayed(function () {

                        def.notify(0.1);

                        // creating the converted document to be stored in local storage
                        startTime = _.now();

                        model.getFullModelDescription(def, 0.1, 0.9)
                        .always(function () {

                            // defining the base key in the local storage
                            saveKey = data.syncInfo.fileId + '_' + data.syncInfo.folderId;

                            // expanding this list, some layers might no longer be used
                            if (_.isFunction(model.getSupportedStorageExtensions)) {
                                _.each(model.getSupportedStorageExtensions(), function (oneExtension) {
                                    if (oneExtension && _.isString(oneExtension.extension)) { allExtensions.push(oneExtension.extension); }
                                });
                            }

                            // removing old content in the local storage (all possible layers need to be removed, even if they are no longer used)
                            clearLocalStorage({ migration: true });
                        })
                        .done(function (docStringList) {

                            // and finally writing all strings from each layer into the registry
                            Utils.iterateArray(docStringList, function (oneStringObject) {

                                var // the html string for one layer
                                    docString = oneStringObject.htmlString,
                                    // the save key for this one layer
                                    localSaveKey = saveKey + oneStringObject.extension;

                                Utils.info('quitHandler, created document string with size: ' + docString.length + '. Time for conversion: ' + (_.now() - startTime) + ' ms.');

                                try {
                                    if (counter === 0) {
                                        // defining keys used in the local storage
                                        saveKeyOSN = saveKey + '_OSN';
                                        versionKey = saveKey + '_FILEVERSION';
                                        storageVersionKey = saveKey + '_STORAGEVERSION';
                                        // saving the keys in the local storage
                                        localStorage.setItem(saveKeyOSN, data.syncInfo['document-osn']);
                                        localStorage.setItem(storageVersionKey, supportedStorageVersion);
                                        localStorage.setItem(versionKey, data.syncInfo.fileVersion);
                                    }
                                    // saving string in the cache, if there is sufficient space
                                    localStorage.setItem(localSaveKey, docString);

                                    // checking, if saving was successful
                                    if (localStorage.getItem(localSaveKey) && (localStorage.getItem(localSaveKey).length === docString.length)) {
                                        Utils.info('quitHandler, successfully saved document in local storage with key: ' + localSaveKey);
                                    } else {
                                        clearLocalStorage();
                                        Utils.info('quitHandler, failed to save document in local storage (1).');
                                        return Utils.BREAK;
                                    }

                                    if (counter === 0) {
                                        if ((localStorage.getItem(saveKeyOSN)) && (parseInt(localStorage.getItem(saveKeyOSN), 10) === data.syncInfo['document-osn'])) {
                                            Utils.info('quitHandler, successfully saved OSN for document in local storage with key: ' + saveKeyOSN + ' : ' + data.syncInfo['document-osn']);
                                        } else {
                                            clearLocalStorage();
                                            Utils.info('quitHandler, failed to save OSN for document in local storage (1).');
                                            return Utils.BREAK;
                                        }
                                    }

                                    counter++;

                                } catch (ex) {
                                    // do nothing, simply not using local storage
                                    clearLocalStorage();
                                    Utils.info('quitHandler: failed to save document and/or document OSN in local storage.');
                                    return Utils.BREAK;
                                }
                            });
                        })
                        .always(function () {
                            def.notify(1.0);
                            def.resolve(data);
                        });

                    });

                }

                return (def === null) ? data : def.promise();
            }

            function propagateChangedFile() {
                return FilesAPI.propagate('change', file);
            }

            win.busy();

            if (file && (locallyModified || locallyRenamed || remotelyModified || movedAway || Config.LOG_PERFORMANCE_DATA) || (!((Utils.getStringOption(launchOptions, 'action') === 'new') || launchOptions.template) && (editClients.length <= 1))) {
                dataToSend = { file: _.extend({ last_opened: Date.now() }, file), app: self.getDocumentType() };
                if (Config.LOG_PERFORMANCE_DATA) {
                    addGlobalPerformanceInfo();
                    dataToSend.performanceData = performanceLogger;
                }
                recentFileData = dataToSend;
            }

            // Don't call closeDocument when we are offline, in an error state or
            // the document has not been modified.
            if (file && rtConnection && (state !== 'offline') && (state !== 'error')) {
                // notify server to save/close the document file
                def = rtConnection.closeDocument(recentFileData).then(checkCloseDocumentResult, closeDocumentResultFailed).then(saveFileInStorage).progress(progressHandler);
            } else {
                // error/offline: it's not safe to make remote calls therefore simply close rtConnection
                def = $.when();
            }

            // disconnect real-time connection after document is closed
            // ATTENTION: it be MUST ensured that rtConnection.destroy() is called
            // before model, view and controller are destroyed
            // rtConnection implementation relies on this fact, breaking this will
            // result in problems especially on shutdown/closing documents
            if (rtConnection) {
                def.always(function () {
                    rtConnection.destroy();
                    rtConnection = null;
                });
            }

            // bug 34464: do nothing else, if the new document has not been created yet (no file descriptor)
            if (file) {
                if (locallyModified || locallyRenamed || remotelyModified || movedAway) {
                    // propagate changed document file to Files application (also in case of an error)
                    def = def.then(propagateChangedFile, propagateChangedFile);
                }
                // bug 35797: do not delete an empty file, if user has requested to reload the document
                else if ((reason !== 'reload') && ((Utils.getStringOption(launchOptions, 'action') === 'new') || launchOptions.template) && (editClients.length <= 1) && !launchOptions.keepFile) {
                    // Delete new document in Files application if no changes have been made at all or
                    // the file has been created from a template and not via edit as new.
                    // Shared documents should never be deleted and also a quit due to reload
                    // should preserve a new file.
                    def = def.then(function () { return FilesAPI.remove([file]); });
                }
            }

            return def;
        }

        /**
         * Adds the provided message data to the pending update queue and sets
         * the new editUser as the current editor for all pending updates as
         * long as the editor is not this client.
         * This prevents that pending message data lead to a wild changing
         * of editors, if the browser catches up processing the pending updates.
         * Therefore we manipulate the pending updates and set the current editor.
         * This must NOT be done, if the current editor is this client, otherwise
         * the edit rights are enabled although the osn is not up-to-date!!
         *
         * @param {Object} data
         *  The message data to be applied. Must be in the exact same format as
         *  also received from 'update' messages sent by the real-time
         *  framework.
         */
        function setAndUpdatePendingRemoteUpdates(data) {

            RTConnection.log('pendingUpdates: updates queued, pending = ' + (pendingRemoteUpdates.length + 1));

            if ((pendingRemoteUpdates.length > 1) && (clientId !== data.editUserId)) {
                RTConnection.log('pendingUpdates: pending updates must be changed to user: ' + data.editUserId + ' now.');
                var extendData = { editUser: data.editUser, editUserId: data.editUserId };
                _.each(pendingRemoteUpdates, function (entryData) {
                    _.extend(entryData, extendData);
                });
            }

            // add the latest message data to the pending update queue
            pendingRemoteUpdates.push(data);
        }

        /**
         * Adding one key-value pair to the performance logger object.
         *
         * @param {String} key
         *  The key for the performance logger object.
         *
         * @param {any} value
         *  The value for the performance logger object.
         */
        function addToPerformanceLogger(key, value) {
            performanceLogger[key] = value;
        }

        /**
         * Adding global performance information that will be sent
         * to the server.
         *
         */
        function addGlobalPerformanceInfo() {
            // logging some additional performance information
            _.each(self.getTimerLogCollector(), function (time, key) {
                addTimePerformance(key, time);
            });

            addTimePerformance('launchStart', self.getLaunchStartTime());
            addTimePerformance('triggerImportSuccess', importFinishedTime);
            addToPerformanceLogger('LocalStorage', loadedFromStorage);
            addToPerformanceLogger('FastEmptyLoad', fastEmpty);
            addToPerformanceLogger('user-agent', navigator.userAgent);
            addToPerformanceLogger('platform', navigator.platform);
            addToPerformanceLogger('user', ox.user);
            addToPerformanceLogger('version', ox.version);
            addToPerformanceLogger('server', ox.abs);
        }

        /**
         * Setting the full client display name, that can be used for user
         * interactions. This is the display name of the user, who opens
         * the application.
         *
         * @param {Object} data
         *  The data object received from the 'welcome' message. It must contain
         *  the properties 'clientId' and 'activeUsers' to determine the client
         *  display name.
         *
         * @returns {String}
         *  The client display name. Set to 'unknown', if it cannot be determined.
         */
        function setClientDisplayName(data) {

            var // the display name of the client, defaulting to 'unknown'
                clientDisplayName = 'unknown',
                // the current user object from the active users collection
                currentUser = null;

            if (data && data.clientId && data.activeUsers) {

                currentUser = _.find(data.activeUsers, function (user) {
                    return data.clientId === user.userId;
                });

                if (currentUser && currentUser.userDisplayName) {
                    clientDisplayName = currentUser.userDisplayName;
                }
            }

            return clientDisplayName;
        }

        // protected methods --------------------------------------------------

        /**
         * Leaves the busy mode while importing the document has not been
         * finished.
         *
         * @internal
         *  Must not be called after importing the document is finished.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the GUI
         *  initialization handler has finished.
         */
        this.leaveBusyDuringImport = _.once(function () {

            // remember start time of preview mode
            previewStartTime = _.now();
            addTimePerformance('previewStart');
            updateState();

            // leave busy mode early
            return view.showAfterImport();
        });

        // public methods -----------------------------------------------------

        /**
         * Creates a method that debounces multiple invocations, if the 'action
         * processing mode' of the document model is currently active, i.e. the
         * model currently applies operation actions (needed e.g. in complex
         * undo or redo operations). Otherwise, the callback function will be
         * invoked immediately.
         *
         * @param {BaseObject} object
         *  The instance used as calling context for the passed callback
         *  function. Additionally, the lifetime of the object will be taken
         *  into account. If the object will be destroyed, before the operation
         *  actions have been applied, the callback function will not be called
         *  anymore.
         *
         * @param {Function} callback
         *  The callback function to be called debounced, after the document
         *  model has applied the current operation actions.
         *
         * @returns {Function}
         *  A debounced version of the passed callback function, bound to the
         *  lifetime of the passed context object.
         */
        this.createDebouncedActionsMethod = function (object, callback) {

            var // whether the created method already waits for document operations
                waiting = false;

            // the actual waitForDocumentActions() method returned from the local scope
            return function debouncedActionsMethod() {

                if (!model.isProcessingActions()) {
                    // invoke callback directly, if no operations are applied currently
                    callback.call(object);
                } else if (!waiting) {
                    // nothing to do, if the method is already waiting for the operation promise
                    waiting = true;
                    object.listenTo(model.getActionsPromise(), 'always', function () {
                        waiting = false;
                        callback.call(object);
                    });
                }
            };
        };

        /**
         * Returns the file format of the edited document.
         */
        this.getFileFormat = function () {
            return this.hasFileDescriptor() && ExtensionRegistry.getFileFormat(this.getFullFileName());
        };

        /**
         * Returns whether the edited document is an Office-Open-XML file.
         */
        this.isOOXML = function () {
            return this.hasFileDescriptor() && ExtensionRegistry.isOfficeOpenXML(this.getFullFileName());
        };

        /**
         * Returns whether the edited document is an OpenDocument file.
         */
        this.isODF = function () {
            return this.hasFileDescriptor() && ExtensionRegistry.isOpenDocument(this.getFullFileName());
        };

        /**
         * Returns whether the edited document is a template document file.
         */
        this.isTemplateFile = function () {
            return this.hasFileDescriptor() && ExtensionRegistry.isTemplate(this.getFullFileName());
        };

        /**
         * Returns whether the edited document may contain macro scripts.
         */
        this.isScriptable = function () {
            return this.hasFileDescriptor() && ExtensionRegistry.isScriptable(this.getFullFileName());
        };

        /**
         * Returns the own unique real-time client identifier.
         *
         * @returns {String}
         *  The own unique real-time client identifier.
         */
        this.getClientId = function () {
            return clientId;
        };

        /**
         * Returns the unique real-time identifier of the client with edit
         * rights.
         *
         * @returns {String}
         *  The unique real-time identifier of the client with edit rights.
         */
        this.getEditClientId = function () {
            return oldEditClientId;
        };

        /**
         * Returns the own client display name.
         *
         * @returns {String}
         *  The own client display name.
         */
        this.getClientDisplayName = function () {
            return clientDisplayName;
        };

        /**
         * Returns the array of all active clients currently editing this
         * document. Inactive clients (clients without activity for more than
         * 60 seconds) will not be included in this list.
         *
         * @returns {Array}
         *  The active clients. Each array element is a client descriptor
         *  object with the following properties:
         *  - {String} client.clientId
         *      The unique real-time session identifier of the client.
         *  - {Number} client.clientIndex
         *      A unique integral zero-based index for the client that will
         *      never change during the lifetime of this application.
         *  - {Number} client.userId
         *      The identifier of the user (teh same user will have the same
         *      identifier on different remote clients).
         *  - {String} client.userName
         *      The name of the user intended to be displayed in the UI.
         *  - {Boolean} client.remote
         *      Whether the entry represents a remote client (true), or the
         *      local client (false).
         *  - {Boolean} client.editor
         *      Whether the respective client has edit rights.
         *  - {Number} client.colorIndex
         *      A one-based index into the color scheme used to visualize the
         *      client (e.g. user list, or remote selections).
         *  - {Object} [client.userData]
         *      Optional data for the user that has been set by the respective
         *      client, e.g. the current selection.
         */
        this.getActiveClients = function () {
            return activeClients;
        };

        /**
         * Returns whether this document has been modified locally (true after
         * any operations where generated and applied by the own document model
         * and sent to the server).
         *
         * @returns {Boolean}
         *  Whether this document has been modified locally.
         */
        this.isLocallyModified = function () {
            return locallyModified;
        };

        /**
         * Returns whether we are in the process of switching edit rights.
         */
        this.isPrepareLosingEditRights = function () {
            return inPrepareLosingEditRights;
        };

        /**
         * Sends a real-time request to the server and waits for a response.
         *
         * @param {String} action
         *  The identifier of the action sent to the server.
         *
         * @param {Number} [timeout=20000]
         *  The delay time in milliseconds, after the Promise returned by this
         *  method will be rejected, if the query is still pending.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved with the
         *  answer of the server request, or rejected on error or timeout.
         */
        this.sendRealTimeQuery = function (action, timeout) {
            if (!_.isNumber(timeout)) { timeout = 20000; }
            return rtConnection.sendQuery(action, timeout);
        };

        /**
         * Applies the passed application state, actions, and other application
         * data. Actions will be applied in a background loop to prevent
         * unresponsive browser and warning messages from the browser. If
         * actions from older 'update' messages are still being applied, the
         * new data passed to this method will be cached, and will be processed
         * after the background task for the old actions is finished. The
         * application will trigger a 'docs:update' event with the passed data,
         * after the actions have been applied.
         *
         * @param {Object} data
         *  The message data to be applied. Must be in the exact same format as
         *  also received from 'update' messages sent by the real-time
         *  framework.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved or rejected
         *  after the application data has been processed.
         */
        this.applyUpdateMessageData = function (data) {
            setAndUpdatePendingRemoteUpdates(data);
            return applyUpdateMessageData(data, { notify: true });
        };

        /**
         * Sends a log message via the real-time connection if logging is
         * enabled.
         *
         * @param {String} message
         *  The message to be logged on the server-side.
         *
         * @returns {EditApplication}
         *  A reference to this instance.
         */
        this.sendLogMessage = Config.LOG_ERROR_DATA ? function (message) {
            if (rtConnection) { rtConnection.sendLogMessage(message); }
            return this;
        } : Utils.NOOP;

        /**
         * Renames the file currently edited by this application.
         *
         * @param {String} shortName
         *  The new short file name (without extension).
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  file has been renamed successfully; or that will be rejected, if
         *  renaming the file has failed.
         */
        this.rename = function (shortName) {

            var // the original file name
                oldFileName = this.getFullFileName(),
                // the new file name (trim NPCs and spaces at beginning and end, replace embedded NPCs)
                newFileName = Utils.trimAndCleanString(shortName),
                // we fetch the currentfocus in IE because it could get lost
                currentFocus = document.activeElement;

            // return running request (should not happen anyway!)
            if (renamePromise) {
                Utils.warn('EditDocument.rename(): multiple calls, ignoring subsequent call');
                return renamePromise;
            }

            // empty name (after trimming)
            if (newFileName.length === 0) {
                return $.Deferred().reject();
            }

            // name does not change
            if (newFileName === this.getShortFileName()) {
                return $.when();
            }

            // Bug 27724: immediately update file descriptor
            newFileName += '.' + this.getFileExtension();
            this.updateFileDescriptor({ filename: newFileName });

            // send the server request
            renamePromise = this.sendFileRequest(IO.FILTER_MODULE_NAME, { action: 'renamedocument', filename: newFileName });

            // filter response by the 'error' property containing an error code
            renamePromise = renamePromise.then(function (data) {

                var // the result of the rename can be extract from the error propety
                    error = new ErrorCode(data);

                // forward result data on success, otherwise reject with error code
                return (!error.isError()) ? data : $.Deferred().reject({ errorCode: error.getCodeAsConstant() });
            });

            // renaming succeeded: prevent deletion of new empty file, update file descriptor
            renamePromise.done(function (data) {
                locallyRenamed = true;
                self.updateFileDescriptor({ filename: Utils.getStringOption(data, 'fileName', oldFileName) });
            });

            // renaming failed: show an appropriate warning message
            renamePromise.fail(function (response) {

                var error = Utils.getStringOption(response, 'errorCode', ErrorCode.CONSTANT_NO_ERROR),
                    type = 'warning',
                    headline = null,
                    message = null;

                if (response !== 'abort') {
                    switch (error) {
                    case 'RENAMEDOCUMENT_FAILED_ERROR':
                        message = gt('Renaming the document failed.');
                        break;
                    case 'RENAMEDOCUMENT_VALIDATION_FAILED_CHARACTERS_ERROR':
                        message = gt('Renaming the document failed. The file name contains invalid characters.');
                        break;
                    case 'RENAMEDOCUMENT_NOT_SUPPORTED_ERROR':
                        message = gt('Renaming the document failed. The storage containing the file does not support renaming a file.');
                        break;
                    default:
                        Utils.warn('EditApplication.rename(): unknown error code "' + error.getCodeAsConstant() + '"');
                        type = 'error';
                        headline = gt('Server Error');
                        message = gt('Renaming the document failed.');
                    }
                    view.yell({ type: type, headline: headline, message: message });
                }

                // back to old file name
                self.updateFileDescriptor({ filename: oldFileName });
            });

            // clean up after renaming has finished
            renamePromise.always(function () {
                renamePromise = null;

                //after renaming the file, the appsuite spinner fetches the focus
                //so we have to get it back
                if (_.browser.IE && currentFocus) { currentFocus.focus(); }
            });
            return renamePromise;
        };

        /**
         * Returns whether there are actions left that have not been sent
         * successfully to the server.
         *
         * @attention
         *  Do not rename this method. The OX core uses it to decide whether to
         *  show a warning before the browser refreshes or closes the page.
         *
         * @returns {Boolean}
         *  Whether the application currently caches unsaved actions.
         */
        this.hasUnsavedChanges = function () {
            return sendingActions || !!renamePromise || (actionsBuffer.length > 0) || _.isObject(pendingAction);
        };

        /**
         * Returns the Promise of a Deferred object that waits until all
         * pending actions with their operations have seen sent to the server.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved after all
         *  pending actions have been saved. If no actions are currently
         *  pending, the returned Promise will be resolved already.
         */
        this.saveChanges = function () {
            return sendActions();
        };

        /**
         * Returns the current application state.
         *
         * @returns {String}
         *  A string representing the application state. Will be one of the
         *  following values:
         *  - 'error': internal error, application locked permanently.
         *  - 'offline': network connection missing.
         *  - 'readonly': the document cannot be changed.
         *  - 'initial': the document has not been changed yet.
         *  - 'ready': all local changes have been sent to the server.
         *  - 'sending': local changes are currently being sent to the server.
         */
        this.getState = function () {
            return state;
        };

        /**
         * Returns whether the application is locked for editing. Reasons for
         * the lock may be missing edit permissions for the document file, or
         * an unrecoverable internal application error.
         *
         * @returns {Boolean}
         *  Whether editing the document is locked.
         */
        this.isLocked = function () {
            return internalError || locked;
        };

        /**
         * Returns whether acquiring edit rights is currently possible (the
         * document must be in read-only mode, and no internal application
         * error has occurred).
         *
         * @returns {Boolean}
         *  Whether acquiring edit rights is currently possible.
         */
        this.isAcquireEditRightsEnabled = function () {
            return _.isObject(rtConnection) && !this.isLocked() && (this.getState() !== 'offline') && !model.getEditMode();
        };

        /**
         * Tries to acquire edit rights for the document. Edit rights will be
         * updated automatically while synchronizing operations with the
         * server. Multiple calls of this function will be throttled with a
         * delay of 3 seconds (first call will be executed immediately).
         */
        this.acquireEditRights = _.throttle(function () {
            if (this.isAcquireEditRightsEnabled()) {
                rtConnection.acquireEditRights();
                triggerEditRightsEvent('acquire');
            }
        }, 3000);

        /**
         * Returns whether the auto-save feature is currently enabled.
         *
         * @returns {Boolean}
         *  Whether the auto-save feature is currently enabled.
         */
        this.isAutoSaveEnabled = function () {
            // auto-save is always enabled, trying to toggle it results in an alert box (see below)
            return true;
        };

        /**
         * Enables or disables the auto-save feature.
         *
         * @param {Boolean} state
         *  Whether to enable or disable the auto-save feature.
         *
         * @returns {EditApplication}
         *  A reference to this instance.
         */
        this.toggleAutoSave = function (/*state*/) {
            // auto-save is always enabled, trying to toggle it results in an alert box
            view.yell({
                type: 'info',
                headline: gt('What is AutoSave?'),
                message: gt('In OX Documents all your changes are saved automatically through AutoSave. It is not necessary to save your work manually.')
            });
            return this;
        };

        /**
         * Saves the current document to a specified folder using the provided
         * file name (without extension). If saving the current document
         * to the new folder is successful the function automatically
         * loads the copied document, which restarts the application.
         *
         * @param {String} shortName
         *  The new short file name (without extension).
         *
         * @param {Number} targetFolderId
         *  The target folder id where the document or template should be saved
         *
         * @param {String} [type='file']
         *  The target file type. The following file types are supported:
         *  - 'file' (default): Save with the current extension.
         *  - 'template': Save as template file.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  file has been saved successfully; or that will be rejected, if
         *  saving the file has failed.
         */
        this.saveDocumentAs = function (shortName, targetFolderId, type) {

            var // the result Deferred object
                def = null,
                // the new file name (trim NPCs and spaces at beginning and end, replace embedded NPCs)
                newFileName = _.isString(shortName) ? Utils.trimAndCleanString(shortName) : '';

            // empty name (after trimming)
            if (newFileName.length === 0) {
                return $.Deferred().reject();
            }

            // send the server request
            def = this.sendFileRequest(IO.FILTER_MODULE_NAME, {
                action: 'copydocument',
                initial_filename: newFileName,
                asTemplate: type === 'template',
                outFolder_id: targetFolderId
            });

            // filter response by the 'error' property containing an error code
            def = def.then(function (data) {

                var // error state of the request
                    error = new ErrorCode(data);

                // forward result data on success, otherwise reject with error code
                return (!error.isError()) ? data : $.Deferred().reject(error);
            });

            // renaming succeeded: prevent deletion of new empty file, update file descriptor
            def.done(function (data) {

                var // target folder of the copied document
                    folderId = Utils.getStringOption(data, 'folder_id', null),
                    // new file name of the copied document
                    fileName = Utils.getStringOption(data, 'filename', newFileName);

                model.setEditMode(false);

                // Templates are always saved in the user directory. Notify users about this
                if (type === 'template') {
                    view.yell({
                        type: 'info',
                        message:
                            //#. %1$s is the file name of the copied document
                            //#, c-format
                            gt('The document was saved as a template "%1$s" in your default user folder.', _.noI18n(fileName))
                    });
                }
                self.updateFileDescriptor({
                    id: Utils.getStringOption(data, 'id', null),
                    folder_id: targetFolderId || folderId,
                    filename: fileName,
                    version: Utils.getStringOption(data, 'version', '1'),
                    title: ExtensionRegistry.getBaseName(fileName)
                });

                // Adapt launch options to prevent removing the file as it
                // could be empty and new or a template file.
                // See #32719 "Save as with an empty document: => Load Error"
                launchOptions.action = 'load';
                delete launchOptions.template;

                // close the current document and restart with new one
                self.reloadDocument();
            });

            // copy the document failed: show an appropriate warning message
            def.fail(function (error) {

                var type = 'error',
                    headline = null,
                    message = gt('The new document could not be saved.');

                switch (error.getCodeAsConstant()) {
                case 'GENERAL_FILE_NOT_FOUND_ERROR':
                    message = gt('Saving the document to a new file failed, because the source document could not be found.');
                    break;
                case 'COPYDOCUMENT_FAILED_ERROR':
                case 'COPYDOCUMENT_FILENAME_UNKNOWN_ERROR':
                    // use the default error message
                    headline = gt('Server Error');
                    break;
                case 'COPYDOCUMENT_USERFOLDER_UNKOWN_ERROR':
                    headline = gt('Server Error');
                    message = gt('The new document could not be saved, because your default folder is unknown.');
                    break;
                case 'GENERAL_QUOTA_REACHED_ERROR':
                    message = gt('The new document could not be saved, because your quota has been reached.');
                    break;
                case 'GENERAL_PERMISSION_CREATE_MISSING_ERROR':
                    message = gt('The new document could not be saved, because you do not have permissions to create a file in the folder.');
                    break;
                default:
                    headline = gt('Server Error');
                    Utils.warn('EditApplication.saveDocumentAs(): unknown error code "' + error.getCodeAsConstant() + '"');
                }
                view.yell({ type: type, headline: headline, message: message });
            });

            return def.promise();
        };

        /**
         * Closes this application instance, and launches a new application
         * instance with the document that is currently edited.
         */
        this.reloadDocument = function () {

            var // the file descriptor of the current document
                file = this.getFileDescriptor(),
                // full filename
                fullFileName = this.getFullFileName(),
                // the edit application module identifier
                editModule = ExtensionRegistry.getEditModule(fullFileName);

            if (editModule.length === 0) {
                Utils.error('EditApplication.reloadDocument(): unknown application type');
                return this;
            }

            if (!ExtensionRegistry.isNative(fullFileName)) {
                Utils.error('EditApplication.reloadDocument(): unsupported document type');
                return this;
            }

            // Close this application, and wait for quit to finish. This is required
            // to prevent that the framework detects a running application for the
            // edit document and rejects launching the new application instance.
            this.quit('reload').done(function () {
                // additional delay before launching, for safety
                _.delay(function () {
                    ox.launch(editModule + '/main', { action: 'load', file: file });
                }, 1000);
            });
        };

        /**
         * Rejects an attempt to edit the document (e.g. due to received
         * keyboard events in read-only mode). The application will show an
         * appropriate warning alert banner to the user, and will update the
         * internal state according to the cause of the rejection.
         *
         * @param {String} [cause='edit']
         *  The cause of the rejection. If omitted, a simple edit attempt has
         *  been rejected, without any further influence on the application
         *  state.
         *  'image':
         *    Inserting an image into the document has failed.
         *  'siri':
         *    A siri event has been detected and the application is
         *    set to interal error mode = true. The user has to reload the
         *    document to make changes.
         *  'android-ime':
         *    A ime notification has been detected and the application is set
         *    to internal error mode = true. The user has to reload the
         *    document to make changes.
         *
         * @returns {EditApplication}
         *  A reference to this instance.
         */
        this.rejectEditAttempt = function (cause) {

            switch (cause) {
            case 'image':
                view.yell({ type: 'warning', message: gt('The image could not be inserted.') });
                break;

            case 'siri':
                setInternalError();
                showReloadOnErrorAlert({ message: gt('Siri is not supported as input method. Please do not use to input text.') });
                break;

            case 'clipboardTimeout':
                setInternalError();
                showReadOnlyAlert(gt('The server detected a sychronization problem with your client. Please close and reopen the doument.'));
                break;

            case 'loadingInProgress':
                showReadOnlyAlert(gt('Please wait until document is loaded completely.'));
                break;

            case 'android-ime':
                setInternalError();
                showReloadOnErrorAlert({
                    message: gt('Document editing is not supported with your current soft/virtual keyboard settings. Please disable automatic correction, gesture typing and suggestions.'),
                    hyperlink: {
                        url: 'http://oxpedia.org/wiki/index.php?title=OXText_on_Android',
                        label: gt('Find further information here.')
                    }
                });
                break;

            default:
                showReadOnlyAlert();
            }

            return this;
        };

        /**
         * Calls the realtime update user data function debounced.
         *
         * @param {Object} userData
         *  the user data, containing e.g. user selection etc.
         *
         * @returns {EditApplication}
         *  A reference to this instance.
         */
        this.updateUserData = (function () {

            var // the cached user data from the last call of this method
                cachedData = null;

            // direct callback: store user data of last invocation
            function storeUserData(userData) {
                cachedData = userData;
                return this;
            }

            // deferred callback: send last cached user data to server
            function sendUserData() {
                // bug 32520: do not send anything when closing document during import
                if (rtConnection && !internalError && !self.isInQuit()) {
                    rtConnection.updateUserData('updateuserdata', cachedData);
                }
            }

            return self.createDebouncedMethod(storeUserData, sendUserData, { delay: 500, maxDelay: 1000 });
        })();

        // debug --------------------------------------------------------------

        /**
         * Triggers the specified event at the realtime connection instance.
         *
         * @attention
         *  Only available for debug purposes.
         */
        this.debugTriggerRealtimeEvent = Config.DEBUG ? function (type) {
            rtConnection.trigger(type);
            return this;
        } : $.noop;

        /**
         * Sends a invalid operation to the server side. Depending on the
         * options different invalid operations can be created and sent
         * to the server side.
         *
         * @param {Object} options
         *  options data to control what kind of invalid operation is sent
         *  to the server side.
         *  @param {Object} options.operation
         *   An operation object to be sent to the server side.
         *  @param {Boolean} options.badOsn [optional]
         *   Use a bad osn for the operation.
         *
         * @attention
         *  Only available for debug purposes.
         */
        this.debugSendInvalidOperation = Config.DEBUG ? function (options) {
            var // bad osn option
                badOsn = Utils.getBooleanOption(options, 'badOsn', false),
                // operations array
                opsArray = [],
                // osn to be used for the operation
                osn = badOsn ? model.getOperationStateNumber() + 1000 : model.getOperationStateNumber(),
                // operation to be sent
                operation = Utils.getObjectOption(options, 'operation', null);

            if (operation) {
                operation = _.extend(operation, { osn: osn, opl: 1 });
                opsArray.push(operation);
                if (!badOsn) {
                    model.setOperationStateNumber(osn + 1);
                }
                rtConnection.sendActions([{operations: opsArray }]);
            }
            return this;
        } : $.noop;

        /**
         * Quits the application with additional debug actions.
         *
         * @attention
         *  Only available for debug purposes.
         *
         * @param {String} type
         *  The debug action type to be performed. The following actions are
         *  supported:
         *  - 'unsaved': Shows the 'unsaved changes' dialog.
         *  - 'error': Simulates a server error while trying to shut down.
         */
        this.debugQuit = Config.DEBUG ? function (type) {
            switch (type) {
            case 'unsaved':
                var originalMethod = self.hasUnsavedChanges;
                this.hasUnsavedChanges = _.constant(true);
                this.quit().fail(function () { self.hasUnsavedChanges = originalMethod; });
                break;
            case 'error':
                // simulate server error after 2 seconds
                rtConnection.closeDocument = function () {
                    var def = $.Deferred();
                    _.delay(function () { def.reject(); }, 2000);
                    return def;
                };
                this.quit();
                break;
            }
        } : $.noop;

        // initialization -----------------------------------------------------

        // initialization after construction
        this.onInit(function () {

            // get references to MVC instances after construction
            model = self.getModel();
            view = self.getView();

            // store exact time stamp when import finishes (regardless of the result)
            self.onImport(function () {
                importFinishedTime = _.now();
            });

            // calculate initial application state
            updateState();

            // update application state when model changes read-only state
            // (this includes all kinds of internal errors)
            model.on('readonly', updateState);

            // check whether editing is globally disabled for this application type
            if (!ExtensionRegistry.supportsEditMode(self.getDocumentType())) {
                locked = true;
                model.setEditMode(false);
                if (_.browser.Android) {
                    showReadOnlyAlert(gt('Editing on Android not supported yet.'));
                } else {
                    showReadOnlyAlert(gt('This browser is not supported on your current platform.'));
                }
            } else if (!(self.attributes && self.attributes.name === 'io.ox/office/spreadsheet') && _.browser.MacOS && _.browser.Chrome && !Utils.supportedChromeVersionOnMacForText()) {
                locked = true;
                model.setEditMode(false);
                // Bug 48777 & Bug 48784
                showReadOnlyAlert(gt('This browser is not supported on your current platform.'));
            }

            if (localStorageApp && saveFileInLocalStorage) {
                Utils.info('EditApplication.onInit(): saving document to local storage is active');
                if (isLocalStorageSupported) {
                    Utils.info('EditApplication.onInit(): current browser supports local storage');
                } else {
                    Utils.warn('EditApplication.onInit(): current browser does not support local storage');
                }
            }
        });

        // set quit handlers
        this.registerBeforeQuitHandler(beforeQuitHandler)
            .registerQuitHandler(quitHandler);

    }}); // class EditApplication

    // static methods ---------------------------------------------------------

    // create and return the application launcher
    EditApplication.createLauncher = function (moduleName, ApplicationClass, appOptions) {
        return BaseApplication.createLauncher(moduleName, ApplicationClass, _.extend({ chromeless: true, search: true }, appOptions));
    };

    // exports ================================================================

    return EditApplication;

});
