/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/view/toolbox',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/control/button',
     'io.ox/office/baseframework/view/component'
    ], function (Utils, Button, Component) {

    'use strict';

    var // CSS class for tool boxes currently collapsed
        COLLAPSED_CLASS = 'collapsed';

    // class ToolBox ==========================================================

    /**
     * Represents a view component with a fixed width and a vertically oriented
     * layout. Optionally, a heading label will be shown that allows to
     * collapse and expand the tool box.
     *
     * @constructor
     *
     * @extends Component
     *
     * @param {BaseApplication} app
     *  The application containing this tool box instance.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the Component base class.
     *  Additionally, the following options are supported:
     *  @param {String} [initOptions.label]
     *      If specified, a heading label will be shown at the top border of
     *      the tool box. The heading label can be clicked to collapse (hide
     *      all its contents but the heading label) and expand (show all its
     *      contents) the tool box.
     */
    function ToolBox(app, initOptions) {

        var // self reference
            self = this,

            // the label for the heading button
            headingLabel = Utils.getStringOption(initOptions, 'label'),

            // the heading button
            headingButton = null,

            // the current line node as target for new container nodes
            lineNode = null,

            // whether next groups will be inserted into the right tab node
            rightTab = false,

            // the current container node as target for new groups
            containerNode = null;

        // base constructor ---------------------------------------------------

        Component.call(this, app, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Inserts the root DOM node of the passed group into this tool box.
         */
        function groupInserter(group) {

            var // the current tab node (for left/right alignment)
                tabNode = null,
                // whether the group takes the entire width of the tool box
                fullWidth = Utils.getBooleanOption(group.getOptions(), 'fullWidth', false);

            // get or create the line node
            if (fullWidth || !lineNode) {
                lineNode = $('<div>').addClass('group-line');
                self.getNode().append(lineNode);
            }

            // full-width: insert group node directly into the line
            if (fullWidth) {
                // class 'auto-width' expands width of group child node (button, text field) to group width
                lineNode.append(group.getNode().addClass('auto-width').css('width', '100%'));
                self.newLine();
                return;
            }

            // get or create tab node
            tabNode = lineNode.children(rightTab ? '.tab-right' : '.tab-left');
            if (tabNode.length === 0) {
                tabNode = $('<div>').addClass(rightTab ? 'tab-right' : 'tab-left');
                lineNode.append(tabNode);
            }

            // get or create group container node, insert the group
            if (!containerNode) {
                containerNode = $('<div>').addClass('group-container');
                tabNode.append(containerNode);
            }
            containerNode.append(group.getNode());
        }

        // methods ------------------------------------------------------------

        /**
         * A gap will be inserted before the next inserted group in the current
         * line.
         *
         * @param {Number} [width=9]
         *  The width of the gap, in pixels.
         *
         * @returns {ToolBox}
         *  A reference to this instance.
         */
        this.addGap = function (width) {
            if (containerNode) {
                containerNode.css('margin-right', (_.isNumber(width) ? width : 9) + 'px');
            }
            containerNode = null;
            return this;
        };

        /**
         * A gap will be inserted before the next inserted group in the current
         * line. The following groups inserted into the current line will be
         * aligned to the right border.
         *
         * @returns {ToolBox}
         *  A reference to this instance.
         */
        this.addRightTab = function () {
            if (!rightTab) {
                containerNode = null;
                rightTab = true;
            }
            return this;
        };

        /**
         * New groups will be inserted into a new line in this tool box.
         *
         * @returns {ToolBox}
         *  A reference to this instance.
         */
        this.newLine = function () {
            lineNode = containerNode = null;
            rightTab = false;
            return this;
        };

        // initialization -----------------------------------------------------

        this.getNode().addClass('toolbox');

        // tool box with heading (button that collapses/expands the tool box)
        if (_.isString(headingLabel)) {

            // add a special marker CSS class
            this.getNode().addClass('collapsing-box');

            // create the heading button
            headingButton = new Button({ classes: 'heading', icon: 'fa-angle-down', label: headingLabel, iconPos: 'right' });
            this.addGroup(null, headingButton);

            // collapse/expand the tool box when clicking the button
            headingButton.on('group:change', function (event) {
                var collapse = !self.getNode().hasClass(COLLAPSED_CLASS);
                self.getNode().toggleClass(COLLAPSED_CLASS, collapse);
                headingButton.setIcon(collapse ? 'fa-angle-left' : 'fa-angle-down');
                self.trigger('expand', !collapse);
                if (!event.preserveFocus) {
                    app.getView().grabFocus();
                }
            });
        }

        // insert new groups into the current container node
        this.registerGroupInserter(groupInserter);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = initOptions = self = headingButton = lineNode = containerNode = null;
        });

    } // class ToolBox

    // exports ================================================================

    // derive this class from class Component
    return Component.extend({ constructor: ToolBox });

});
