/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.documentconverter;

import java.io.BufferedOutputStream;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import org.apache.commons.io.IOUtils;

/**
 * {@link Measurement}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.6.0
 */
public class Measurement {

    /**
     * {@link Group}
     *
     * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
     * @since v7.6.0
     */
    public class Group {

        /**
         * Initializes a new {@link Group}.
         *
         * @param parentGroup
         * @param level
         * @param title
         * @param additionalData
         */
        public Group(Group parentGroup, int level, Enum<?> title, Object... additionalData) {
            super();
            m_parentGroup = parentGroup;
            m_level = level;
            m_title = title;
            m_additionalData = additionalData;
        }

        /**
         *
         */
        public void startMeasurement() {
            m_date = new Date();
            m_startTime = System.nanoTime();
        }

        /**
         *
         */
        public void stopMeasurement() {
            m_stopTime = System.nanoTime();
        }

        /**
         * @return The elapsed time from calling {@link #startMeasurement()} to {@link #stopMeasurement()}
         */
        public long getStartTimeNanoSeconds() {
            return m_startTime;
        }

        /**
         * @return The elapsed time from calling {@link #startMeasurement()} to {@link #stopMeasurement()}
         */
        public long getEndTimeNanoSeconds() {
            return m_stopTime;
        }

        /**
         * @return The elapsed time from calling {@link #startMeasurement()} to {@link #stopMeasurement()}
         */
        public long getElapsedTimeNanoSeconds() {
            return (m_stopTime - m_startTime);
        }

        /**
         * @return The title of this subgroup
         */
        public Enum<?> getTitle() {
            return m_title;
        }

        /**
         * @return The title of this subgroup
         */
        /**
         * @return The optional, additional data of this subgroup
         */
        public Object[] getAdditionalData() {
            return m_additionalData;
        }

        /**
         * @return The actual level of this group within the measurement hierarchy
         */
        public int getLevel() {
            return m_level;
        }

        /**
         * @return The {@link Group} parent of this group
         */
        public Group getParentGroup() {
            return m_parentGroup;
        }

        /**
         * @return The {@link Date} when starting the measurement
         */
        public Date getDate() {
            return m_date;
        }

        // ---------------------------------------------------------------------

        private Enum<?> m_title = null;

        private Object[] m_additionalData = null;

        private Group m_parentGroup = null;

        private Date m_date = null;

        private int m_level = 0;

        private long m_startTime = 0;

        private long m_stopTime = 0;
    }

    /**
     * Initializes a new {@link Measurement}.
     *
     * @param perfId
     * @param title
     * @param outputFile
     */
    public Measurement(Object perfId, Enum<?> measureTitle, HashMap<String, Object> jobProperties, File outputFile) {
        super();

        m_perfId = perfId;
        m_measureTitle = measureTitle;
        (m_jobProperties = new HashMap<String, Object>(jobProperties.size())).putAll(jobProperties);
        m_outputFile = outputFile;
    }

    /*
     * (non-Javadoc)
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((m_perfId == null) ? 0 : m_perfId.hashCode());
        return result;
    }

    /*
     * (non-Javadoc)
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        Measurement other = (Measurement) obj;
        if (m_perfId == null) {
            if (other.m_perfId != null)
                return false;
        } else if (!m_perfId.equals(other.m_perfId))
            return false;
        return true;
    }

    /**
     * @param title
     */
    public synchronized Group beginGroup(Enum<?> title, Object... additionalData) {
        // increase group level only, if the parent group type differs from the given group type
        if ((null == m_curActiveGroup) || (m_curActiveGroup.getTitle() != title)) {
            ++m_curGroupLevel;
        }

        final Group newGroup = new Group(m_curActiveGroup, m_curGroupLevel, title, additionalData);
        m_curActiveGroup = newGroup;
        m_groups.add(newGroup);
        newGroup.startMeasurement();

        return newGroup;
    }

    /**
     *
     */
    public synchronized void endGroup() {
        if (null != m_curActiveGroup) {
            m_curActiveGroup.stopMeasurement();
            m_curActiveGroup = m_curActiveGroup.getParentGroup();
            --m_curGroupLevel;
        }
    }

    /**
     *
     */
    public synchronized void finishMeasurement() {
        while (null != m_curActiveGroup) {
            endGroup();
        }

        if (null != m_outputFile) {
            BufferedOutputStream outputStm = null;

            try {
                writeOut(outputStm = new BufferedOutputStream(new FileOutputStream(m_outputFile)));
            } catch (FileNotFoundException e) {
                e.printStackTrace();
            } finally {
                IOUtils.closeQuietly(outputStm);
            }
        }
    }

    /**
     * @param outputFile The {@link File}, where to write the output to
     */
    public void write(OutputStream outputStm) {
        writeOut(outputStm);
    }

    // - Implementation

    protected void writeOut(OutputStream outputStm) {
        if (null != outputStm) {
            Writer writer = null;

            try {
                writer = new BufferedWriter(new PrintWriter(outputStm));

                // writer title of measurement
                writer.write((null != m_measureTitle) ? m_measureTitle.toString() : ("Measurement #" + m_perfId.toString()));
                writer.write(CRLF);
                writer.write("JobProperties: ");
                writer.write(m_jobProperties.toString());
                writer.write(CRLF);

                for (final Group curGroup : m_groups) {
                    if (null != curGroup) {
                        final int groupOffset = curGroup.getLevel();
                        final String linePredicate = getLinePredicate("\u251c", "\u2500\u2500", groupOffset).toString();
                        final Date groupDate = curGroup.getDate();
                        final long groupTimeNanoSeconds = curGroup.getElapsedTimeNanoSeconds();
                        final StringBuilder groupString = new StringBuilder(" [");

                        // write group date, time and title
                        groupString.append(groupDate.toString()).append("] ").append(getScaledTimeString(groupTimeNanoSeconds)).append(": ").append(curGroup.getTitle().toString());

                        // write possible group data
                        final Object[] groupData = curGroup.getAdditionalData();

                        if ((null != groupData) && (groupData.length > 0)) {
                            groupString.append(" (");

                            for (int i = 0, len = groupData.length; i < len; ++i) {
                                final Object curGroupData = groupData[i];

                                if (null != curGroupData) {
                                    if (i > 0) {
                                        groupString.append("; ");
                                    }

                                    groupString.append(curGroupData.toString());
                                }
                            }

                            groupString.append(")");
                        }

                        writer.write(linePredicate);
                        writer.write(groupString.toString());
                        writer.write(CRLF);
                    }
                }

                writer.write(CRLF);
                writer.write(SEPARATOR_LINE);
                writer.write(CRLF);
                writer.flush();

            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * @param timeNanoSeconds
     * @return The nanosecond value scaled to milliseconds and appended with the correct unit
     */
    public static String getScaledTimeString(long timeNanoSeconds) {
        final long scalingMilliSeconds = 1000000;
        final String milliSecondsString = Long.valueOf(timeNanoSeconds / scalingMilliSeconds).toString();
        final String unitString = "s";
        String secondsString;

        if (milliSecondsString.length() < 2) {
            secondsString = "0.00" + milliSecondsString;
        } else if (milliSecondsString.length() < 3) {
            secondsString = "0.0" + milliSecondsString;
        } else if (milliSecondsString.length() < 4) {
            secondsString = "0." + milliSecondsString;
        } else {
            final int sepPos = milliSecondsString.length() - 3;
            secondsString = milliSecondsString.substring(0, sepPos) + "." + milliSecondsString.substring(sepPos);
        }

        return (secondsString + unitString);
    }

    /**
     * @param firstChar
     * @param secondChar
     * @param secondCharRepeat
     * @return
     */
    static private String getLinePredicate(String firstStr, String secondStr, int secondStrRepeat) {
        final StringBuilder ret = new StringBuilder(Math.max(2, secondStrRepeat * ((null != secondStr) ? secondStr.length() : 0)));

        if (null != firstStr) {
            ret.append(firstStr);
        }

        if (null != secondStr) {
            for (int i = 0; i < secondStrRepeat; ++i) {
                ret.append(secondStr);
            }
        }

        ret.trimToSize();

        return ret.toString();
    }

    // - Static Members --------------------------------------------------------

    final protected static String CRLF = "\r\n";

    final protected static String SEPARATOR_LINE = "********************************************************************************\r\n";

    final protected static String SUMMARY_SECTION = "Summary\r\n=======\r\n";

    final protected static String TOTALTIME_ITEM = "Total:         ";

    final protected static String PROCESSINGTIME_ITEM = "Processing:    ";

    final protected static String WAITINGTIME_ITEM = "Waiting:       ";

    // - Members ---------------------------------------------------------------

    private File m_outputFile = null;

    private Object m_perfId = null;

    private Enum<?> m_measureTitle = null;

    private HashMap<String, Object> m_jobProperties = null;

    private List<Group> m_groups = new ArrayList<Group>();

    private Group m_curActiveGroup = null;

    private int m_curGroupLevel = 0;
}
