/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.obs.ant.tasks;

import java.util.HashSet;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.Task;
import com.openexchange.obs.api.BuildServiceClient;
import com.openexchange.obs.api.BuildServiceException;
import com.openexchange.obs.api.ObsPackage;

/**
 * Checks the package names already on OBS and compares them to a given list. Will delete not needed packages.
 * Separator of given list is ",".
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public final class DeleteObsoletePackages extends Task {

    private String url;
    private String login;
    private String password;
    private String projectName;
    private String packageNames;

    public DeleteObsoletePackages() {
        super();
    }

    public void setUrl(String url) {
        this.url = url;
    }

    public void setLogin(String login) {
        this.login = login;
    }

    public void setPassword(String password) {
        this.password = password;
    }

    public void setProjectName(String projectName) {
        this.projectName = projectName;
    }

    public void setPackageNames(String packageNames) {
        this.packageNames = packageNames;
    }

    @Override
    public void execute() throws BuildException {
        if (url == null) {
            throw new BuildException("Please define URL of OBS.");
        }
        if (login == null) {
            throw new BuildException("Please define login for OBS.");
        }
        if (password == null) {
            throw new BuildException("Please define password for OBS.");
        }
        if (projectName == null) {
            throw new BuildException("Please define project.");
        }
        if (packageNames == null) {
            throw new BuildException("Please define packageNames.");
        }
        final BuildServiceClient client;
        try {
            client = new BuildServiceClient(url, login, password);

            // Fetch current list of packages from OBS and list them if debug log level is enabled.
            ObsPackage[] packages = client.getPackageList(projectName);
            HashSet<String> packagesToDelete = new HashSet<String>();
            getProject().log("The following packages are currently existing on OBS in project " + projectName + ':', Project.MSG_DEBUG);
            for (ObsPackage obsPackage : packages) {
                String packageName = obsPackage.getPackageName();
                getProject().log("    " + packageName, Project.MSG_DEBUG);
                packagesToDelete.add(packageName);
            }

            // List packages on the current packages list if debug log level is enabled and remove them from the set of packages already
            // uploaded to OBS.
            getProject().log("The following packages are currently defined in the packages list:", Project.MSG_DEBUG);
            for (String packageName : packageNames.split(",")) {
                getProject().log("    " + packageName, Project.MSG_DEBUG);
                packagesToDelete.remove(packageName);
            }

            // Delete the leftovers from OBS.
            for (String packageName : packagesToDelete) {
                getProject().log("Deleting package " + packageName + " on OBS.", Project.MSG_INFO);
                client.deletePackage(projectName, packageName);
            }
        } catch (BuildServiceException e) {
            throw new BuildException(e.getMessage(), e);
        }
    }
}
